% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apa_list.R
\name{apa_list}
\alias{apa_list}
\title{Calculating APA-maps}
\usage{
apa_list(
  plot_dat,
  tree_dat,
  plot_id_column,
  tree_id_column,
  weight_column,
  agg_class_column = NULL,
  core_column = attr(plot_dat, "sf_column"),
  buffer_column = core_column,
  res = 1,
  subplot_dat = NULL,
  subplot_id_column = NULL,
  radius = 10,
  apa_properties = NA,
  edge_correction = "none",
  dis_trait_column = NULL,
  dis_method = "gowdis",
  dis_transform = sqrt,
  scope = "global",
  apa_polygon = TRUE
)
}
\arguments{
\item{plot_dat}{An \code{sf-data.frame} with plot-level data. \code{plot.dat}
must contain a column that specifies the unique id of the plot. the column
name of the plot id is specified with the \code{plot_id_column}-argument.
The plot id is used to relate plot- and tree-data. A column with
\code{POLYGON}-data that specify the outline of the plots has to be
provided.}

\item{tree_dat}{An \code{sf}-\code{data.frame} with tree data.
\code{tree.dat} must contain a unique id of each tree. The column name of
the tree id is specified with \code{tree_id_column}. Plot and tree-data are
related by the id of the plot, which has to be a column in both datasets.
The geometry type of \code{tree_dat} has to be \code{POINT}-data that
specifies tree coordinates.}

\item{plot_id_column}{Column name of \code{plot_dat} specifying unique plot
id's.}

\item{tree_id_column}{Column name of \code{tree_dat} specifying unique tree
id's.}

\item{weight_column}{Column name of \code{tree_dat} that specifies a variable
to be used as weighting factor for the calculation of APA-maps.}

\item{agg_class_column}{An optional vector of column names of \code{tree_dat}
that specify variables that are used as additional grouping variable (for
example tree species) to aggregate apa-properties (see \link{apa_add_agg_class}
for more details).}

\item{core_column}{Column name of \code{plot_dat} specifying a
\code{\link[sf:sfc]{sfc}}-column with \code{POLYGON}-data of plot boundaries.
Default is the active geometry of \code{plot_dat}.}

\item{buffer_column}{Column name of \code{plot_dat} specifying a
\code{\link[sf:sfc]{sfc}}-column with \code{POLYGON}s of boundaries of a buffer-zone
around the core plot. To specify a buffer is only necessary if trees
outside of the core area of a plot were sampled (plus-sampling) and the
\code{edge-correction} method is \code{"critical"}. Defaults to
\code{core_column}.}

\item{res}{single number specifying the resolution of the APA-maps (identical
in x- and y-direction).}

\item{subplot_dat}{A named list of \code{sf-data.frames} with additional data
of subplots. If the active geometry column of \code{plot_dat} is
\code{POLYGON}-data, calculation of apa-properties will be done for the
area within these \code{POLYGON}s. For \code{POINT}-data, circular subplots
are created around the point coordinates.}

\item{subplot_id_column}{A named vector specifying the id columns of
\code{subplot_dat}.}

\item{radius}{The radius of circular subplots that should be used if a
dataset in in \code{subplot_dat} contains \code{POINT}-data.}

\item{apa_properties}{A vector specifying the apa-properties that will be
calculated. May be any combination of \code{"border_tree"},
\code{"apa_size"}, \code{"ndiv"} and \code{"pdiv"}.}

\item{edge_correction}{which of the implemented edge correction method should
be applied when calculating apa-properties (\code{"none"},
\code{"critical"} or \code{"border_tree_exclusion"}.}

\item{dis_trait_column}{A list containing combinations of traits that will
be used to estimate dissimilarity between trees when calculating
apa-properties. Refers to column names of \code{tree_dat}.}

\item{dis_method}{Which method should be used to estimate dissimilarity
between trees. If \code{dis_method} is \code{gowdis}, \code{\link[FD:gowdis]{FD::gowdis()}} will
be used to calculate Gower dissimilarity. Alternatively, \code{dis_method}
may be any \code{function} that calculates a dissimilarity matrix out of a
\code{data.frame} with trait values.}

\item{dis_transform}{A \code{function} to transform dissimilarities. Defaults
to \code{sqrt}.}

\item{scope}{Should scaling of the dissimilarity be done at \code{"global"}
or at \code{"local"} level? If \code{dis_method} scales dissimilarity
between trees according to the range of occurring values in the dataset (as
done by \code{"gowdis"}), a \code{"global"} \code{scope} will use the range
of all values in \code{tree_dat}. Any other \code{scope} will use the range
of values at plot-level to scale dissimilarity.}

\item{apa_polygon}{logical, specifies if \code{POLYGON}s of APA-patches be
should be added to the datasets. Defaults to \code{TRUE}.}
}
\value{
\code{apa_list} returns an object of class \code{"apa_list"}. An
\code{apa_list} is a \code{list} of \code{data.frame}s with at least two
elements:
\itemize{
\item \code{apa_list$plot_dat}: A \code{data.frame} with the original
plot-level data is stored with an additional column \code{apa_map} that is
a list containing all APA-maps. APA-maps of individual plots are
represented by \code{RasterStack} objects. If additional APA-properties
are calculated at stand-level, they are appended to this \code{data.frame}
as well.
\item \code{apa_list$tree_dat}: A \code{data.frame} with the original
tree-level data Additional tree-level APA-characteristics are added to
\code{tree_dat} as separate columns.
\item aggregation classes (optional): if one or multiple aggregation classes
(e.g., tree species) were added to the \code{apa_list}, additional
\code{data.frame}s at class-level will be added..
\item \code{apa_list$subplot_dat} (optional): If subplots were specified, all
data at subplot-level are stored here.
}
}
\description{
Calculate maps of the area potentially available to trees (APA-maps) out of
tree inventory data. All vector data that is used or provided by the
\code{APAtree}-package is stored in \code{data.frame}s of the additional
\code{sf} class of the \code{\link[sf:sf]{sf}}-package (referred to as
\code{sf-data.frame}). APA-maps are raster data that are stored in objects of
the \code{\link[raster:raster-package]{raster}}-package.
}
\details{
APA-maps are derived by calculating a rasterized version of weighted
Voronoi-Diagrams of tree coordinates. To each tree a weight may be assigned
that reflects the competitive ability of the tree.
}
\examples{
library(APAtree)
data(tree_enrico, package = "APAtree")
data(plot_enrico, package = "APAtree")

tree_enrico$height_class <- tree_enrico$height > 20

# only calculate an apa_list for two plots and with a coarse resolution of 1 m
# to save time.
apa_list_enrico <- 
  apa_list(plot_dat = subset(plot_enrico, id_plot \%in\% c("5.2", "8.2")), 
           tree_dat = tree_enrico,
           plot_id_column = "id_plot",
           tree_id_column = "id_tree",
           weight_column = "crown_radius_95",
           agg_class_column = c("species", "height_class"),
           res = 1,
           apa_polygon = TRUE)

apa_list_enrico
}
\references{
{ Glatthorn, Jonas (2021). A spatially explicit index for tree
species or trait diversity at neighborhood and stand level. Ecological
Indicators, 130, 108073. https://doi.org/10.1016/j.ecolind.2021.108073.}

{ Römisch,K. (1995) Durchmesserwachstum und ebene
Bestandesstruktur am Beispiel der Kiefernversuchsfläche Markersbach. In
Deutscher Verband forstl. Forschungsanstalten, Sektion Biometrie und
Informatik. Gottfried Hempel (ed.) Tagung Tharanth/Grillenburg, Vol. 8, pp.
84–103.}

{ Gspaltl, M., Sterba, H., & O’hara, K. L. (2012). The relationship
between available area efficiency and area exploitation index in an even-aged
coast redwood (Sequoia sempervirens) stand. Forestry, 85(5), 567-577.}
}
