



use core::{
    mem::ManuallyDrop,
    sync::atomic::{self, AtomicUsize, Ordering},
};

use super::utils::Backoff;


pub(super) type AtomicChunk = AtomicUsize;
pub(super) type Chunk = usize;





pub(super) struct SeqLock {

    state_hi: AtomicUsize,





    state_lo: AtomicUsize,
}

impl SeqLock {
    #[inline]
    pub(super) const fn new() -> Self {
        Self { state_hi: AtomicUsize::new(0), state_lo: AtomicUsize::new(0) }
    }




    #[inline]
    pub(super) fn optimistic_read(&self) -> Option<(usize, usize)> {






        let state_hi = self.state_hi.load(Ordering::Acquire);
        let state_lo = self.state_lo.load(Ordering::Acquire);
        if state_lo == 1 { None } else { Some((state_hi, state_lo)) }
    }





    #[inline]
    pub(super) fn validate_read(&self, stamp: (usize, usize)) -> bool {



        atomic::fence(Ordering::Acquire);






        let state_lo = self.state_lo.load(Ordering::Acquire);



        let state_hi = self.state_hi.load(Ordering::Relaxed);




        (state_hi, state_lo) == stamp
    }


    #[inline]
    pub(super) fn write(&self) -> SeqLockWriteGuard<'_> {
        let mut backoff = Backoff::new();
        loop {
            let previous = self.state_lo.swap(1, Ordering::Acquire);

            if previous != 1 {


                atomic::fence(Ordering::Release);

                return SeqLockWriteGuard { lock: self, state_lo: previous };
            }

            while self.state_lo.load(Ordering::Relaxed) == 1 {
                backoff.snooze();
            }
        }
    }
}


#[must_use]
pub(super) struct SeqLockWriteGuard<'a> {

    lock: &'a SeqLock,


    state_lo: usize,
}

impl SeqLockWriteGuard<'_> {

    #[inline]
    pub(super) fn abort(self) {


        let this = ManuallyDrop::new(self);




        this.lock.state_lo.store(this.state_lo, Ordering::Release);
    }
}

impl Drop for SeqLockWriteGuard<'_> {
    #[inline]
    fn drop(&mut self) {
        let state_lo = self.state_lo.wrapping_add(2);




        if state_lo == 0 {
            let state_hi = self.lock.state_hi.load(Ordering::Relaxed);
            self.lock.state_hi.store(state_hi.wrapping_add(1), Ordering::Release);
        }




        self.lock.state_lo.store(state_lo, Ordering::Release);
    }
}

#[cfg(test)]
mod tests {
    use super::SeqLock;

    #[test]
    fn smoke() {
        let lock = SeqLock::new();
        let before = lock.optimistic_read().unwrap();
        assert!(lock.validate_read(before));
        {
            let _guard = lock.write();
        }
        assert!(!lock.validate_read(before));
        let after = lock.optimistic_read().unwrap();
        assert_ne!(before, after);
    }

    #[test]
    fn test_abort() {
        let lock = SeqLock::new();
        let before = lock.optimistic_read().unwrap();
        {
            let guard = lock.write();
            guard.abort();
        }
        let after = lock.optimistic_read().unwrap();
        assert_eq!(before, after, "aborted write does not update the stamp");
    }
}
