#' Azure DevOps Project Release Definition Information
#'
#' @description
#' These functions will allow you to scrape release definition information from Azure DevOps.
#'
#' @details
#' For more information about release definition API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/release/releases}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param quiet logical whether want general running information from printing. Any
#' issue with the API call will still show up if set to \code{TRUE}
#'
#' @examples
#' \dontrun{
#' # Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key("<username>", "<password>")
#' vsts_get_release_defs("domain", "project", auth_key)
#' }
#'
#' @rdname vsts_release_def
#' @export
vsts_get_release_defs <- function(domain, project, auth_key, quiet = FALSE) {
  uri <- file.path(AZURE_VSRM_URL, domain, project, "_apis/release/definitions?api-version=6.0")
  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))

  if (httr::status_code(response) != 200) {
    send_failure_message(response, "get release definition list")
    return(NULL)
  }

  content <- httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)$value
  if (!quiet) cat("Available release definitions:", paste(content$name, collapse = ", "), "\n")

  content
}

#' Azure DevOps Project Release Information
#'
#' @description
#' These functions will allow you to create releases from Azure DevOps.
#'
#' @details
#' The \code{artifacts} object within the body contains two items:
#' \itemize{
#' \item{alias}{[character] Sets alias of artifact.}
#' \item{instanceReference}{[list] Sets instance reference of artifact. e.g. for build artifact it is build number.}
#' }
#'
#' For more information about release  API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/release/releases}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param body a list of extra parameters that can need to be sent to the API call (* mandatory):
#' \describe{
#' \item{\code{artifacts *}}{[list] Sets list of artifact to create a release. Check \code{Details} for more information.}
#' \item{\code{definitionId *}}{[integer] Sets definition Id to create a release.}
#' \item{\code{description *}}{[character] Sets description to create a release.}
#' \item{\code{isDraft}}{[logical] Sets 'true' to create release in draft mode, 'false' otherwise.}
#' \item{\code{manualEnvironments}}{[character] Sets list of environments to manual as condition.}
#' \item{\code{properties}}{[list] The class represents a property bag as a collection of key-value pairs.}
#' \item{\code{reason}}{[character] Sets reason to create a release.}
#' }
#'
#' @examples
#' \dontrun{
#' # Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key("<username>", "<password>")
#' art_list <- list(
#'   list(alias = "Art1", instanceReference = list(id = 1)),
#'   list(alias = "Art2", instanceReference = list(id = 2))
#' )
#' body <- list(
#'   definitionId = 1, description = "R API Release",
#'   artifacts = I(art_list)
#' )
#' vsts_create_release("domain", "project", auth_key, body)
#' }
#' @export
vsts_create_release <- function(domain, project, auth_key, body) {
  uri <- file.path(AZURE_VSRM_URL, domain, project, "_apis/release/releases?api-version=6.0")
  request_body <- jsonlite::toJSON(body, auto_unbox = TRUE)

  response <- httr::POST(uri, httr::add_headers(Authorization = auth_key),
    httr::content_type("application/json"),
    body = request_body
  )

  if (httr::status_code(response) >= 300) {
    send_failure_message(response, paste0("create release definition #", body[["definitionId"]]))
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}

#' Azure DevOps Project Release Information
#'
#' @description
#' These functions will allow you to scrape releases from Azure DevOps.
#'
#' @details
#' For more information about release API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/release/Releases}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param query a list of extra parameters that can be sent to the API call
#'
#' @examples
#' \dontrun{
#' # Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key("<username>", "<password>")
#' vsts_get_releases("domain", "project", auth_key)
#' }
#'
#' @rdname vsts_get_release
#' @export
vsts_get_releases <- function(domain, project, auth_key, query = NULL) {
  uri <- file.path(AZURE_VSRM_URL, domain, project, "_apis/release/releases?api-version=6.0")
  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))

  if (response$status_code != 200) {
    send_failure_message(response, "get releases list")
    return(NULL)
  }

  content <- httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)$value
  content
}

#' @param release Release Definition ID
#'
#' @rdname vsts_get_release
#' @export
vsts_get_release <- function(domain, project, release, auth_key) {
  uri <- file.path(AZURE_VSRM_URL, domain, project, "_apis/release/releases", paste0(release, "?api-version=6.0"))
  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), httr::content_type("application/json"))

  if (httr::status_code(response) != 200) {
    send_failure_message(response, "get release")
    return(NULL)
  }

  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}

#' Azure DevOps Project Release Environment Information
#'
#' @description
#' These functions will allow you to run release environment tasks from Azure DevOps.
#'
#' @details
#' For more information about release environment API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/azure/devops/release/releases/update-release-environment}.
#'
#' @param domain The name of the Azure DevOps organization.
#' @param project the name of the project in \code{domain} to look at
#' @param release the release ID of the release
#' @param env the release environment ID to release on
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#'
#' @examples
#' \dontrun{
#' # Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key("<username>", "<password>")
#' vsts_deploy_release("domain", "project", auth_key, 1, 1)
#' }
#'
#' @rdname vsts_release_env
#' @export
vsts_deploy_release <- function(domain, project, release, env, auth_key) {
  uri <- file.path(
    AZURE_VSRM_URL, domain, project, "_apis/Release/releases", release, "environments",
    paste0(env, "?api-version=5.0-preview.6")
  )
  request_body <- jsonlite::toJSON(list(status = "inProgress"), auto_unbox = TRUE)

  response <- httr::PATCH(
    uri,
    httr::add_headers(Authorization = auth_key),
    httr::content_type("application/json"),
    body = request_body
  )

  if (httr::status_code(response) != 200) {
    send_failure_message(response, "send release")
    return(NULL)
  }

  cat("Deployment of release has started.\n")
  httr::content(response, encoding = "UTF-8", simplifyDataFrame = TRUE)
}
