#' Creating an S3 object for computing the product moment 
#' 
#' \code{mtrunmnt} creates an S3 object designed to compute the product moment 
#'     for a truncated multivariate normal distribution, utilizing the algorithm 
#'     by Lee (2020). Key attributes of this object are the nodes and weights of 
#'     the multivariate Gaussian quadrature and the probability of the 
#'     truncation interval
#' 
#' Assume the parent multivariate normal distribution comes from a mixed-effects 
#'   linear model:
#'   \deqn{\mathbf{Y} = \mathbf{X}\boldsymbol{\beta} + \mathbf{Zu} + 
#'   \boldsymbol{\epsilon}}
#'   where \eqn{\mathbf{X}} and \eqn{\mathbf{Z}} are design matrices 
#'   corresponding to \eqn{\boldsymbol{\beta}} and \eqn{\mathbf{u}} representing 
#'   fixed and random effects, respectively, and \eqn{\boldsymbol{\epsilon}} is 
#'   the vector of errors. It is assumed that the random effects 
#'   \eqn{\mathbf{u}} follows a multivariate normal distribution with mean 
#'   \eqn{\mathbf{0}}, and symmetric positive definite variance-covariance 
#'   matrix \eqn{\mathbf{D}}. As usual, the distribution of 
#'   \eqn{\boldsymbol{\epsilon}} is assumed to be a multivariate normal with 
#'   mean \eqn{\mathbf{0}} and variance-covariance matrix 
#'   \eqn{\sigma^2_{\epsilon} \mathbf{I}}, but for more flexibility, it can be 
#'   assumed that the error terms are independent, but do not have equal 
#'   variance. That is, **we assume** \eqn{\boldsymbol{\epsilon} \sim 
#'   N(\mathbf{0}, \mathbf{E})} where \eqn{\mathbf{E}} is a diagonal matrix.
#'   Then, \deqn{\mathbf{Y} \sim N(\boldsymbol{\mu}, \mathbf{\Sigma})}
#'   where \eqn{\mathbf{\Sigma} = \mathbf{Z}\mathbf{D}\mathbf{Z}' + \mathbf{E}}.
#'   The variance-covariance structure in \code{mtrunmnt} can thus be specified 
#'   either by providing the individual components \eqn{\mathbf{D}, \mathbf{Z}}, 
#'   and \eqn{\mathbf{E}}, or by directly supplying the resulting overall 
#'   variance-covariance matrix \eqn{boldsymbol{\Sigma}}. 
#' 
#' @references Lee, S.-C. (2020). Moments calculation for truncated multivariate 
#'     normal in nonlinear generalized mixed models. \emph{Communications for 
#'     Statistical Applications and Methods}, Vol. 27, No. 3, 377–383.
#'
#' @param mu **(Required)** Mean vector of the parent multivariate normal 
#'     distribution. 
#' @param lower vector of lower limits. If the lower limits are the same, a 
#'     scalar value can be given. Defaults to -Inf.  
#' @param upper Vector of upper limits. If the upper limits are the same, a 
#'     scalar value can be given. Defaults to Inf.  
#' @param Sigma The variance-covariance matrix of the parent multivariate normal 
#'     distribution. It must be given a symmetric positive definite matrix, if 
#'     Sigmae, D and Z are not specified.
#' @param Sigmae Vector of variances of error terms. If the variances are the
#'     same, a scalar value can be given. Defaults to 1.  
#' @param Z Design matrix for the random components. Defaults to 
#'     \eqn{n \times 1} matrix of 1's where \eqn{n} is the dimension of mu. 
#'     It must be specified carefully with the argument D. \code{ncol(Z)} 
#'     determines the dimension of D.   
#' @param D Variance-covariance matrix  of \eqn{u}. See Details. If the random 
#'     components are independent, you can specify either a vector of variances 
#'     or a scalar value. A scalar value means that the random components have 
#'     the same variance. Defaults to 1.
#' @param nGH Number of quadrature points. Defaults to 35. 
#' @export
#' @importFrom fastGHQuad gaussHermiteData
#' @return A mtrunmnt object
#' @examples
#' ### Create a mtrunmnt objective ###
#' 
#' set.seed(123)
#' sigma2e <- 1
#' sigma2a <- 2
#' n <- 5
#' mu <- seq(-1,1, length.out = n)
#' y <- mu + rnorm(1, sd = sqrt(sigma2a)) + rnorm(n, sd = sqrt(sigma2e))
#' S <- matrix(sigma2a, ncol = n, nrow = n) + diag(sigma2e, n)
#' a  <- rep(-Inf, n)
#' b  <- rep(Inf, n)
#' a[y >= 0] <- 0
#' b[y <  0] <- 0
#' obj1 <- mtrunmnt(mu, lower = a, upper = b, Sigmae = sigma2e, D = sigma2a) 
#' obj2 <- mtrunmnt(mu, lower = a, upper = b, Sigma = S) 
#' probntrun(obj1)
#' probntrun(obj2)
#' prodmnt(obj1, c(2,2,0,0,0))
#' prodmnt(obj2, c(2,2,0,0,0))
#' meanvar(obj1)
#' meanvar(obj2)

mtrunmnt <- function(mu, lower = -Inf, upper = Inf, Sigma = 1, Sigmae = 1, 
                     Z = matrix(rep(1, length(mu)), ncol = 1), 
                     D = matrix(1, ncol = 1, nrow = 1), nGH = 35){       
  m <- length(mu)
  is.identical = F
  if(is.matrix(Sigma)){    # The variance matrix is given by Sigma = S
    if(!isSymmetric(Sigma)) stop("The variance-covariance matrix is not symmetric")
    SPD <- eigen(Sigma, symmetric = TRUE, only.values = FALSE)
    sigma2e  <- SPD$values[m]
    if(sigma2e <= 0) stop("The variance-covariance matrix is not positive-definite")
    SPD$values <- SPD$values - sigma2e
    SPD$values[abs(SPD$values) < 1.e-12] <- 0
    D <- SPD$values[SPD$values > 0]
    k <- length(D)
    if(k == 0) {
      is.identical = T
      D <- matrix(0, nrow = 1, ncol = 1)
      Z <- matrix(0, nrow = m, ncol = 1)
    } else {
      D <- diag(D,k)
      Z <- as.matrix(SPD$vectors[,1:k])
    }
    Sigmae <- rep(sigma2e, m)
  }
  else{                                                 # The variance is given by Sigma2, Z and D 
    if(length(Sigmae) == 1) Sigmae <- rep(Sigmae, m)
    if(length(Sigmae) != m) stop("The dimension of Sigmae does not match with the dimension of mu")
    
    if(is.matrix(D)){   # D is given by a matrix
      if(!isSymmetric(D)) stop("The D matrix is not symmetric")
    } else{             # D is given by a vector or a scalar
        if(length(D) == 1) D <- diag(D, ncol(Z))
        else D <- diag(D)
    }
    if(nrow(D) != ncol(Z)) stop("The dimension of Z does not match with the dimension of D")
  }
  sigmae <- sqrt(Sigmae)    # vector of standard deviations of errors
  
  if(length(lower) == 1) lower <- rep(lower, m)
  if(length(upper) == 1) upper <- rep(upper, m)
  
  dimcheck <- c(length(lower), length(upper), nrow(Z), length(mu))
  if(any(dimcheck != m)) stop("Check the dimensions of the mu, lower, bound, and Z arguments.")
  
  if(is.identical){
    GH <- list(weights = matrix(0, ncol = 1, nrow = 1), nodes = matrix(0, ncol = 1, nrow = 1))
  } else GH <- mgauss_hermite(nGH, rep(0, ncol(D)), D)
  probab <- getprobab(mu, sigmae, lower, upper, Z, GH)
  trunObj <- structure(list(mu = mu, sigmae = sigmae, lower = lower, upper = upper,  
                            Z = Z, D = D, nGH = nGH, weights = GH$weights,
                            nodes = GH$nodes, probab = probab), class = "mtrunmnt")
  return(trunObj)            
}




