

is_initialized <- function() {
    isTRUE(stanza_options("initialized"))
}

is_python_initialized <- function() {
    tryCatch(getNamespace("reticulate")$is_python_initialized(), 
             error = function(e) FALSE)
}


get_model_dir <- function() {
    getNamespace("stanza")$stanza$resources$common$DEFAULT_MODEL_DIR
}


is_valid_python_path <- function(python_path) {
    version <- try(system2(python_path, "--version", stdout = TRUE, stderr = FALSE), silent = TRUE)
    !inherits(version, "try-error") && any(grepl("python", version, ignore.case = TRUE))
}


find_stanza_env <- function(env_variable) {
    python_path <- Sys.getenv(env_variable)
    if (nchar(python_path) > 0L && is_valid_python_path(python_path)) {
        return(python_path)
    }
}


py_discover_stanza <- function() {
    config <- reticulate::py_discover_config("stanza", "stanza")
    if (!is.null(config)) {
        python_path <- unlist(strsplit(config$pythonpath, ":", fixed = TRUE))
        files <- unlist(lapply(python_path, dir, include.dirs = TRUE))
        if (isTRUE("stanza" %in% files)) {
            return(config$python)
        }
    }
}


conda_find_stanza <- function() {
    root_path <- reticulate::miniconda_path()
    if (is.character(root_path) && dir.exists(root_path)) {
        if (isTRUE(Sys.info()["sysname"] == "Windows")) {

        } else {
            python_path <- file.path(root_path, "envs/stanza/bin/python")
        }
        if (file.exists(python_path) && is_valid_python_path(python_path)) {
            return(python_path)
        }
    }
}


virtualenv_find_stanza <- function() {
    root_path <- normalizePath(reticulate::virtualenv_root(), mustWork = FALSE)
    if (is.character(root_path) && dir.exists(root_path)) {
        if (isTRUE(Sys.info()["sysname"] == "Windows")) {
            python_path <- file.path(root_path, "stanza\\Scripts\\python.exe")
        } else {
            python_path <- file.path(root_path, "stanza/bin/python")
        }
        if (file.exists(python_path) && is_valid_python_path(python_path)) {
            return(python_path)
        }
    }
}


conda_find_stanza <- function() {
    root_path <- reticulate::miniconda_path()
    if (is.character(root_path) && dir.exists(root_path)) {
        if (isTRUE(Sys.info()["sysname"] == "Windows")) {
            python_path <- file.path(root_path, "envs\\stanza\\python.exe")
            dir(python_path)
        } else {
            python_path <- file.path(root_path, "envs/stanza/bin/python")
        }
        if (file.exists(python_path) && is_valid_python_path(python_path)) {
            return(python_path)
        }
    }
}


stanza_find_python <- function() {
    python_path <- find_stanza_env("STANZA_PYTHON")
    if (!is.null(python_path)) return(python_path)
    python_path <- find_stanza_env("RETICULATE_PYTHON")
    if (!is.null(python_path)) return(python_path)
    python_path <- py_discover_stanza()
    if (!is.null(python_path)) return(python_path)
    python_path <- virtualenv_find_stanza()
    if (!is.null(python_path)) return(python_path)
    python_path <- conda_find_stanza()
    if (!is.null(python_path)) return(python_path)
    return(NULL)
}


set_python_version <- function(python = NULL, virtualenv = NULL, condaenv = NULL) {
    if (!is.null(python)) {
        reticulate::use_python(python, required = TRUE)
    } else if (!is.null(virtualenv)) {
        reticulate::use_virtualenv(virtualenv, required = TRUE)
    } else if (!is.null(condaenv)) {
        reticulate::use_condaenv(condaenv, required = TRUE)
    } else {
        python_path <- stanza_find_python()
        if (is.null(python_path)) {
            msg <- "could not find module 'stanza', please set the environment variable 'STANZA_PYTHON'!"
            stop(msg)
        }
        reticulate::use_python(python_path, required = TRUE)
    }
    reticulate::py_config()
}


# variable <- "DEFAULT_MODEL_DIR"
py_get_stanza_common <- function(variable, default = NULL) {
    tryCatch(stanza$resources$common[[variable]], error = function(e) default)
}


stanza_available_download_methods <- function() {
    code <- "{item.name: item.value for item in r['stanza'].pipeline.core.DownloadMethod}"
    methods <- py_eval(code)
    unlist(methods)
}


#' Initialize Stanza
#'
#' Initialize the \verb{Python} binding to stanza.
#'
#' @param python a character string giving the path to the \verb{Python}
#'               binary (executeable) to be used.
#'               The variable \code{python} is passed to \code{reticulate::use_python}.
#' @param virtualenv a character string giving the name of the virtual environment,
#'               or the path to the virtual environment, to be used.
#'               The variable \code{virtualenv} is passed to \code{reticulate::use_virtualenv}.
#' @param condaenv a character string giving the name of the \verb{Conda} environment to be used.
#'               The variable \code{condaenv} is passed to \code{reticulate::use_condaenv}.
#' @param model_dir a character sting giving the path to the directory storing the \code{Stanza} models.
#' @param resources_url a character string giving the url to the \code{Stanza} model resources.
#' @param model_url a character string giving the model url.
#' 
#' @returns \code{NULL}
#' 
#' @examples
#' if (stanza_options("testing_level") >= 3L) {
#' stanza_initialize()
#' }
#' @export
stanza_initialize <- function(python = NULL,
                              virtualenv = NULL,
                              condaenv = NULL, 
                              model_dir = NULL,
                              resources_url = NULL,
                              model_url = NULL) {
    assert(check_character(python, len = 1L, any.missing = FALSE, null.ok = TRUE),
           check_character(virtualenv, len = 1L, any.missing = FALSE, null.ok = TRUE),
           check_character(condaenv, len = 1L, any.missing = FALSE, null.ok = TRUE),
           check_character(model_dir, len = 1L, any.missing = FALSE, null.ok = TRUE),
           check_character(resources_url, len = 1L, any.missing = FALSE, null.ok = TRUE),
           check_character(model_url, len = 1L, any.missing = FALSE, null.ok = TRUE),
           combine = "and")

    if (is_initialized()) {
        writeLines("stanza is already initialized!")
        return(NULL)
    }

    python_config <- set_python_version(python, virtualenv, condaenv)
    reticulate::py_run_string("import sys", convert = FALSE)
    state <- try(rpy("stanza", reticulate::import("stanza")), silent = TRUE)
    if (inherits(state, "try-error")) {
        msg <- c("could not import module 'stanza', with ",
                 sprintf("python '%s'. ", python_config$python),
                 "Please install stanza and/or set the environment variable 'STANZA_PYTHON'.")
        stop(msg)
    }
    code <- sprintf("sys.path.append('%s')", system.file("python", package = "stanza"))
    reticulate::py_eval(code, convert = TRUE)
    rpy("rstanza", reticulate::import("rstanza"))
    stanza_options("initialized", TRUE)

    model_dir <- as.character(py_get_stanza_common("DEFAULT_MODEL_DIR", ""))
    if (!file.exists(model_dir)) {
        model_dir <- tools::R_user_dir("stanza", which = "data")
    }
    resources_url <- py_get_stanza_common("DEFAULT_RESOURCES_URL")
    resources_version <- py_get_stanza_common("DEFAULT_RESOURCES_VERSION")
    model_url <- py_get_stanza_common("DEFAULT_MODEL_URL")
    stanza_options("model_dir", model_dir, update_python_defaults = TRUE)
    stanza_options("resources_url", resources_url)
    stanza_options("resources_version", resources_version)
    stanza_options("model_url", model_url)
    stanza_options("stanza_version", stanza_version())

    rpy("download_methods", unlist(rstanza$available_download_methods(stanza)))
    return(NULL)
}


#' Check if Stanza is Initialized
#' 
#' Checks if Stanza is initialized.
#'
#' @returns \code{TRUE} if Stanza is initialized, otherwise \code{FALSE}
#' @examples
#' is_stanza_initialized()
#' @export
is_stanza_initialized <- function() {
    isTRUE(stanza_options("initialized"))
}
