% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group_lines.R
\name{group_lines}
\alias{group_lines}
\title{Group Lines}
\usage{
group_lines(
  DT = NULL,
  threshold = NULL,
  projection = NULL,
  id = NULL,
  coords = NULL,
  timegroup = NULL,
  sortBy = NULL,
  splitBy = NULL,
  sfLines = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{threshold}{The width of the buffer around the lines in the units of the
projection. Use \code{threshold = 0} to compare intersection without buffering.}

\item{projection}{numeric or character defining the coordinate reference
system to be passed to \link[sf:st_crs]{sf::st_crs}. For example, either
\code{projection = "EPSG:32736"} or \code{projection = 32736}.}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{timegroup}{timegroup field in the DT within which the grouping will be
calculated}

\item{sortBy}{Character string of date time column(s) to sort rows by. Must
be a POSIXct.}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}

\item{sfLines}{Alternatively to providing a DT, provide a simple feature
LINESTRING object generated with the sf package. The id argument is
required to provide the identifier matching each LINESTRING.
If an sfLines object is provided, groups cannot be calculated by timegroup
or splitBy.}
}
\value{
\code{group_lines} returns the input \code{DT} appended with a "group"
column.

This column represents the spatial (and if \code{timegroup} was provided -
spatiotemporal) group calculated by intersecting lines. As with the other
grouping functions,  the actual value of group is arbitrary and represents
the identity of a given group where 1 or more individuals are assigned to a
group. If the data was reordered, the group may change, but the contents of
each group would not.

A message is returned when a column named "group" already exists in the
input \code{DT}, because it will be overwritten.
}
\description{
\code{group_lines} groups rows into spatial groups by generating LINESTRINGs and
grouping based on spatial intersection. The function accepts a \code{data.table}
with relocation data, individual identifiers and a distance threshold. The
relocation data is transformed into sf LINESTRINGs using \link{build_lines} and
intersecting LINESTRINGs are grouped. The threshold argument is used to
specify the distance criteria for grouping. Relocation data should be in two
columns representing the X and Y coordinates.
}
\details{
\subsection{R-spatial evolution}{

Please note, spatsoc has followed updates from R spatial, GDAL and PROJ for
handling projections, see more at
\url{https://r-spatial.org/r/2020/03/17/wkt.html}.

In addition, \code{group_lines} (and \link{build_lines}) previously used
\link[sp:SpatialLines]{sp::SpatialLines}, rgeos::gIntersects, rgeos::gBuffer but have been
updated to use \link[sf:st_as_sf]{sf::st_as_sf}, \link[sf:st]{sf::st_linestring}, \link[sf:geos_binary_pred]{sf::st_intersects}, and
\link[sf:geos_unary]{sf::st_buffer} according to the R-spatial evolution, see more at
\url{https://r-spatial.org/r/2022/04/12/evolution.html}.
}

\subsection{Notes on arguments}{

The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using \link[data.table:setDT]{data.table::setDT}.

The \code{id}, \code{coords}, \code{sortBy} (and optional \code{timegroup}
and \code{splitBy}) arguments expect the names of respective columns in
\code{DT} which correspond to the individual identifier, X and Y coordinates,
sorting, timegroup (generated by \link{group_times}) and additional grouping
columns.

The \code{projection} argument expects a numeric or character defining the
coordinate reference system. For example, for UTM zone 36N (EPSG 32736), the
projection argument is either \code{projection = 'EPSG:32736'} or \code{projection = 32736}. See details in \code{\link[sf:st_crs]{sf::st_crs()}} and
\url{https://spatialreference.org} for a list of EPSG codes.

The \code{sortBy} argument is used to order the input \code{DT} when creating sf
LINESTRINGs. It must a column in the input \code{DT} of type POSIXct to ensure the
rows are sorted by date time.

The \code{threshold} must be provided in the units of the coordinates. The
\code{threshold} can be equal to 0 if strict overlap is intended, otherwise it
should be some value greater than 0. The coordinates must be planar
coordinates (e.g.: UTM). In the case of UTM, a \code{threshold = 50} would
indicate a 50m distance threshold.

The \code{timegroup} argument is optional, but recommended to pair with
\link{group_times}. The intended framework is to group rows temporally with
\link{group_times} then spatially with \link{group_lines} (or \link{group_pts},
\link{group_polys}). With \link{group_lines}, pick a relevant \link{group_times} threshold
such as \code{'1 day'} or \code{'7 days'} which is informed by your study species,
system or question.

The \code{splitBy} argument offers further control building LINESTRINGs. If in
your input \code{DT}, you have multiple temporal groups (e.g.: years) for example,
you can provide the name of the column which identifies them and build
LINESTRINGs for each individual in each year. The grouping performed by
\link{group_lines} will only consider rows within each \code{splitBy} subgroup.
}
}
\examples{
# Load data.table
library(data.table)
\dontshow{data.table::setDTthreads(1)}

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Subset only individuals A, B, and C
DT <- DT[ID \%in\% c('A', 'B', 'C')]

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# EPSG code for example data
utm <- 32736

group_lines(DT, threshold = 50, projection = utm, sortBy = 'datetime',
            id = 'ID', coords = c('X', 'Y'))

## Daily movement tracks
# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '1 day')

# Subset only first 50 days
DT <- DT[timegroup < 25]

# Spatial grouping
group_lines(DT, threshold = 50, projection = utm,
            id = 'ID', coords = c('X', 'Y'),
            timegroup = 'timegroup', sortBy = 'datetime')

## Daily movement tracks by population
group_lines(DT, threshold = 50, projection = utm,
            id = 'ID', coords = c('X', 'Y'),
            timegroup = 'timegroup', sortBy = 'datetime',
            splitBy = 'population')
}
\seealso{
\link{build_lines} \link{group_times}

Other Spatial grouping: 
\code{\link{group_polys}()},
\code{\link{group_pts}()}
}
\concept{Spatial grouping}
