#' Fill gaps of first three dates of MOD13Q1
#' 
#' Since MOD13Q1 was released on 18-02-2000 and its temporal resolution
#' is 16 days, there are no measurements available for the first three acquisition dates of 2000. 
#' This function allows to fill these three dates using historic data.
#' 
#' @param   m matrix with \code{nrow} equal to the number of periods (seasons or years) 
#'            studied, and \code{ncol} equal to the number of observations per period.
#' @param fun a function employed to impute missing values. Default, \code{stats::median}.
#'          
#' @export
#' 
#' @details The missing values of \code{m} are \code{m[1,1]}, \code{m[1,2]} and \code{m[1,3]}. For instance,
#' to fill \code{m[1,1]} the values of \code{m[2:nrow(m),1]} are used, and consequently, it 
#' is expected that the larger the numeric vector, the smaller the variability of the imputed
#' value for \code{m[1,1]}.
#' 
#' @note It is recommended to use \code{\link[sephora]{vecToMatrix}} to transfer
#' the values of a numeric vector of MOD13Q1 measurements into a matrix.
#' 
#' @examples 
#' data("deciduous_polygon")
#' str(deciduous_polygon, vec.len = 1)
#' x <- deciduous_polygon[1,] # check x[1:3]
#' x_asMatrix <- vecToMatrix(x, lenPeriod = 23) # check str(x_asMatrix)
#' x_asMat_complete <- fill_initialgap_MOD13Q1(m=x_asMatrix)
#' 
#' #filled first three values of x
#' x[1:3] <- x_asMat_complete
#' 
#' @return A numeric vector of length 3
#' 
#' @seealso \code{\link[sephora]{vecToMatrix}}, \code{\link[sephora]{vecFromData}}
#' 
fill_initialgap_MOD13Q1 <- function(m, fun=stats::median){
  apply( m[-1,1:3], MARGIN=2, FUN=fun)
}

