% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/st_app_enriched.R
\name{st_app_enriched}
\alias{st_app_enriched}
\title{Fetch Enriched Metrics for Specific Apps}
\usage{
st_app_enriched(
  unified_app_ids,
  os = "unified",
  regions = "WW",
  auth_token = NULL
)
}
\arguments{
\item{unified_app_ids}{Character vector. One or more unified app IDs
(24-character hex strings). Required. Use `st_app_info()` to find these.}

\item{os}{Character string. Operating system context for the request.
Must be "unified" (default), "ios", or "android".}

\item{regions}{Character vector. Region codes for data filtering.
Defaults to "WW" (worldwide).}

\item{auth_token}{Optional. Character string. Your Sensor Tower API token.
Defaults to environment variable SENSORTOWER_AUTH_TOKEN.}
}
\value{
A [tibble][tibble::tibble] with enriched metrics including:
  - **Identification**: `unified_app_id`, `unified_app_name`
  - **Active Users**: `dau_30d_us`, `dau_30d_ww`, `wau_4w_us`, `wau_4w_ww`,
    `mau_month_us`, `mau_month_ww`
  - **Retention**: `retention_1d_us/ww`, `retention_7d_us/ww`,
    `retention_14d_us/ww`, `retention_30d_us/ww`, `retention_60d_us/ww`
  - **Demographics**: `genders_us`, `genders_ww`, `age_us`, `age_ww`,
    `male_share_us`, `female_share_us`
  - **Revenue/Downloads**: `revenue_30d_ww`, `revenue_90d_ww`, `revenue_alltime_us/ww`,
    `downloads_30d_ww`, `downloads_alltime_us/ww`
  - **Monetization**: `rpd_alltime_us/ww`, `arpu_90d_us/ww`
  - **Launch**: `release_date_us/ww`, `earliest_release_date`
}
\description{
Retrieves comprehensive metrics including retention, MAU, DAU, demographics,
and other aggregate tags for specific apps by their unified app IDs.
}
\details{
This function is designed for the common workflow of:
1. Search for apps by name using `st_app_info()`
2. Get their unified IDs
3. Fetch enriched metrics for those specific apps using this function
}
\section{Recommended Workflow}{

```
# Step 1: Search for apps by name
apps <- st_app_info("Royal Match")

# Step 2: Get unified IDs
app_ids <- apps$unified_app_id

# Step 3: Fetch enriched metrics
metrics <- st_app_enriched(app_ids)
```
}

\section{Data Availability Notes}{

- **IMPORTANT: Geographic Limitations** - All enriched metrics are only available
  for **US market (`_us` suffix)** and **Worldwide aggregates (`_ww` suffix)**.
  Per-country data (e.g., GB, DE, FR, JP) is NOT available through this endpoint.
  For per-country data, use [st_sales_report()] for revenue/downloads or
  [st_batch_metrics()] for MAU/DAU time-series.
- Retention data (D1, D7, D14, D30, D60) is aggregated for the "last quarter" -
  not time-series data. D90 retention is NOT available through the API.
- Demographics (age/gender) are primarily available for US market only.
- Not all metrics are available for all apps - smaller apps may have NA values.
- This returns **snapshot data**, not time-series. For historical trends, use
  [st_batch_metrics()] or [st_sales_report()].
}

\examples{
\dontrun{
# Get enriched data for specific apps
royal_match <- st_app_info("Royal Match")
enriched <- st_app_enriched(royal_match$unified_app_id)

# Get data for multiple apps at once
game_ids <- c("5f16a8019f7b275235017614", "660af7c66237390ce7c829fc")
multi_enriched <- st_app_enriched(game_ids)

# View retention metrics
multi_enriched \%>\%
  select(unified_app_name, starts_with("retention"))
}

}
\seealso{
[st_app_info()] for searching apps by name,
  [st_app_lookup()] for resolving app IDs,
  [st_sales_report()] for time-series revenue/download data,
  [st_batch_metrics()] for time-series DAU/WAU/MAU data
}
