
#' Process an MRM file in total
#'
#' This is a convenience function that performs an end to end analysis of an
#' Sciex MRM run; it can be used when the intermediate data structures (total
#' ion chromatograms, shot lists, .wiff file readers) are not required.
#'
#' @param wiffFile A valid file path to the Sciex data file (extension .wiff)
#' containing the metadata for one or more EchoMS runs
#' @param wiffScanFile An valid file path the the Sciex scan file (extension
#' .wiff.scan) containing the full raw EchoMS data
#' @param removeBaseline If TRUE, baseline intensity for each mass transition
#' will be subtracted from the intensities in each ejection, to account for
#' baseline intensities measured near the lower limit of quantitation
#' @param ... Additional optional parameters to be passed to [measureEjections()]
#'
#' @return A list containing two data frames: `ejections`, with all extracted
#' TIC peaks for all runs, as output by [measureEjections()], and `massAreas`,
#' containing the measured areas for all mass transitoins and all peaks in all
#' runs, as output by [getMRMAreas()].  Both data frames contain an addtional
#' `wiffSample` column indicating which run within the file the values were
#' extracted from.
#' @export
processAllMRMAreas <- function(wiffFile,wiffScanFile,removeBaseline=TRUE,...) {
	wiff <- newWiffReader(wiffFile,wiffScanFile)
	shots <- getWiffShots(wiff)
	tics <- getAllTIC(wiff)

	ejectiondf <- data.frame()
	massdf <- data.frame()
	for (si in seq_along(wiff$samples)) {
		cursample <- paste0("Sample",si)
		ejections <- measureEjections(tics[[si]],shots=shots,...)
		ejections$wiffSample <- cursample
		ejectiondf <- rbind(ejectiondf,ejections)

		massarea <- getAllMRMAreas(wiffScanFile,tics[[si]],wiff$samples[[si]],ejections,removeBaseline)
		massarea$wiffSample <- cursample
		massdf <- rbind(massdf,massarea)
	}

	list(ejections=ejectiondf,massAreas=massdf)
}

#' Extract the peak areas for MRM mass transitions for all wells
#'
#' @param scanfile A file path to a Sciex raw scan file (extension .wiff.scan)
#' containing the raw data referenced by the .wiff file from which the other
#' parameters were extracted
#' @param tic A total ion chromatogram data.frame as extracted by [getTIC()] or
#' [getAllTIC()]
#' @param sample A "sample" object representing a run in a .wiff file, from the
#' `samples` field of an `rtmsWiffReader` object, containing information about
#' the mass transitions measured in that run
#' @param ejections An ejection table listing the timing and boundaries of the
#' total ion chromatogram peaks for all shots in the run, as returned by
#' [measureEjections()]
#' @param removeBaseline If TRUE, baseline intensity for each mass transition
#' will be subtracted from the intensities in each ejection, to account for
#' baseline intensities measured near the lower limit of quantitation
#'
#' @return A data frame containing the total intensity for each mass transition
#' and each ejection; see Details for column specifics.
#'
#' @details
#' The table return includes a measurement of total area for each of the mass
#' transitions listed in `sample`.  It contains one row for each measured shot
#' each mass transition, with the following columns:
#' * `shotorder`: The order of the peak within the shots fired during the run
#' * `well`: The alphanumeric well name of the well from which the shot was
#' fired
#' * `time`: The time (in seconds) after the beginning of the run at which the
#' intensity from the shot was at its peak
#' * `massindex`: The index of the measured mass transition in the set of masses
#' in the given sample
#' * `mass`: The name of the mass transition measured (often a compound name or
#' id)
#' * `area`: The intensity area (in counts) for that particular mass transition
#' from that ejection
#'
#' @export
getAllMRMAreas <- function(scanfile,tic,sample,ejections,removeBaseline=TRUE) {
	# numMasses <- length(sample$masses)
	# chroma <- getMRMChromaArray(scanfile,tic,sample)
	#
	# meanstep <- mean(diff(tic$time))
	# tic$step <- c(diff(tic$time),meanstep)
	#
	# areadf <- data.frame()
	# for (miter in seq_len(numMasses)) {
	# 	ejectionvec <- rep(NA,nrow(ejections))
	# 	for (eiter in seq_len(nrow(ejections))) {
	# 		rel <- tic$time >= ejections$minTime[[eiter]] & tic$time<= ejections$maxTime[[eiter]]
	# 		ejectionvec[[eiter]] <- sum(tic$step[rel]*chroma[rel,miter])
	# 	}
	# 	areadf <- rbind(areadf,data.frame(shotorder=ejections$shotorder,
	# 									  well=ejections$well,
	# 									  time=ejections$time,
	# 									  massindex = miter,
	# 									  mass=sample$masses[[miter]]$name,
	# 									  area=ejectionvec))
	# }
	# areadf <- areadf[order(areadf$shotorder,areadf$massindex),]
	# areadf
	struct <- getMRMChromaStructure(scanfile,tic,sample,ejections,removeBaseline)
	getMRMAreas_internal(tic,sample,struct,ejections)
}

#' Extract the peak areas for MRM mass transitions for a given well
#'
#' @inheritParams getAllMRMAreas
#' @param well The well (a string) or well index (an integer) to be measured
#'
#' @return A data frame containing the total intensity for each mass transition
#' and each ejection; see Details for column specifics.
#'
#' @details
#' The table return includes a measurement of total area for each of the mass
#' transitions listed in `sample`.  It contains one row for each measured shot
#' each mass transition, with the following columns:
#' * `shotorder`: The order of the peak within the shots fired during the run
#' * `well`: The alphanumeric well name of the well from which the shot was
#' fired
#' * `time`: The time (in seconds) after the beginning of the run at which the
#' intensity from the shot was at its peak
#' * `massindex`: The index of the measured mass transition in the set of masses
#' in the given sample
#' * `mass`: The name of the mass transition measured (often a compound name or
#' id)
#' * `area`: The intensity area (in counts) for that particular mass transition
#' from that shot's peak
#'
#' @export
getMRMAreas <- function(scanfile,tic,sample,ejections,well,removeBaseline=TRUE) {
	if (length(well)!=1) {
		stop("Parameter 'well' should be of length 1.")
	}
	if (is.character(well)) {
		well <- which(ejections$well==well)
		if (length(well)==0) {
			stop("Unable to locate well in ejection table.")
		}
	}
	rel_ejections <- ejections[well,]

	# getAllMRMAreas(scanfile,tic,sample,rel_ejections)
	struct <- getMRMChromaStructure(scanfile,tic,sample,ejections,removeBaseline)
	getMRMAreas_internal(tic,sample,struct,rel_ejections)
}


getMRMChromaStructure <- function(scanfile,tic,sample,ejections,removeBaseline=TRUE) {
	chroma <- getMRMChromaArray(scanfile,tic,sample)
	struct <- list(chroma=chroma)
	if (removeBaseline) {
		minTime <- min(ejections$minTime)
		maxTime <- max(ejections$maxTime)
		rel <- (tic$time>=(minTime-2) & tic$time<(minTime-1)) |
			(tic$time>=(maxTime+1) & tic$time<(maxTime+2))
		baseline <- rep(0,ncol(chroma))
		for (miter in seq_len(ncol(chroma))) {
			baseline[[miter]] <- mean(chroma[rel,miter])
		}
		struct$baseline <- baseline*0.67
	}
	struct
}

getMRMAreas_internal <- function(tic,sample,structure,ejections) {
	chroma <- structure$chroma

	meanstep <- mean(diff(tic$time))
	tic$step <- c(diff(tic$time),meanstep)

	numMasses <- length(sample$masses)
	areadf <- data.frame()
	for (miter in seq_len(numMasses)) {
		ejectionvec <- rep(NA,nrow(ejections))
		for (eiter in seq_len(nrow(ejections))) {
			rel <- tic$time >= ejections$minTime[[eiter]] & tic$time<= ejections$maxTime[[eiter]]
			ejectIntensity <- chroma[rel,miter]
			if (!is.null(structure$baseline)) {
				ejectIntensity <- ejectIntensity-structure$baseline[[miter]]
				ejectIntensity[ejectIntensity<0] <- 0
			}
			ejectionvec[[eiter]] <- sum(tic$step[rel]*ejectIntensity)
		}
		areadf <- rbind(areadf,data.frame(shotorder=ejections$shotorder,
										  well=ejections$well,
										  time=ejections$time,
										  massindex = miter,
										  mass=sample$masses[[miter]]$name,
										  area=ejectionvec))
	}
	areadf <- areadf[order(areadf$shotorder,areadf$massindex),]
	areadf
}

#' Extract full chromatograms for each mass transition
#'
#' @param scanfile A file path to a Sciex raw scan file (extension .wiff.scan)
#' containing the raw data referenced by the .wiff file from which the other
#' parameters were extracted
#' @param tic A total ion chromatogram data.frame as extracted by [getTIC()] or
#' [getAllTIC()]
#' @param sample A "sample" object representing a run in a .wiff file, from the
#' `samples` field of an `rtmsWiffReader` object, containing information about
#' the mass transitions measured in that run
#'
#' @return A single data frame expressing all chromatograms in a "long" format,
#' with the following columns:
#'
#' * `massindex`: The index of the measured mass transition in the set of masses
#' in the given sample
#' * `mass`: The name of the mass transition measured (often a compound name or
#' id)
#' * `time`: The time (in seconds) at which the intensity was measured
#' * `intensity`: The intensity for the given mass transition at the given time
#'
#' @export
getMRMChromatograms <- function(scanfile,tic,sample) {
	chroma <- getMRMChromaArray(scanfile,tic,sample)
	massNames <- vapply(sample$masses, function(m) m$name, "")
	chromatograms <- data.frame(
		massindex = rep(seq_along(massNames),each=nrow(tic)),
		mass = rep(massNames,each=nrow(tic)),
		time = rep(tic$time,times=length(massNames)),
		intensity = as.numeric(chroma)
	)
	chromatograms
}

getMRMChromaArray <- function(scanfile,tic,sample) {
	numMasses <- length(sample$masses)
	tic <- tic[order(tic$offset),]
	if (any(tic$size != 4*numMasses)) {
		stop("All MRM scans should be represented as blocks of 4-byte floating point numbers.")
	}
	numPoints <- nrow(tic)
	blockSize <- tic$offset[[numPoints]] + tic$size[[numPoints]]

	scon <- file(scanfile,"rb")
	on.exit({ close(scon) }, add=TRUE)
	seek(scon,sample$offset)
	seek(scon,12,origin="current")
	blockSizeCheck <- readBin(scon,"integer",1,size=4,endian="little")
	if (blockSize!=blockSizeCheck) {
		stop("Scan file block size does not match wiff file data.")
	}
	seek(scon,8,origin="current")
	dataBlock <- t(array(readBin(scon,"double",numPoints*numMasses,size=4,endian="little"),
						 dim=c(numMasses,numPoints)))
	dataBlock
}
