% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rptProportion.R
\name{rptProportion}
\alias{rptProportion}
\title{GLMM-based Repeatability Estimation for Proportion Data}
\usage{
rptProportion(
  formula,
  grname,
  data,
  link = c("logit", "probit"),
  CI = 0.95,
  nboot = 1000,
  npermut = 0,
  parallel = FALSE,
  ncores = NULL,
  ratio = TRUE,
  adjusted = TRUE,
  expect = "meanobs",
  rptObj = NULL,
  update = FALSE,
  ...
)
}
\arguments{
\item{formula}{Formula as used e.g. by \link[lme4]{lmer}. The grouping factor(s) of
interest needs to be included as a random effect, e.g. '(1|groups)'.
Covariates and additional random effects can be included to estimate adjusted 
repeatabilities.}

\item{grname}{A character string or vector of character strings giving the
name(s) of the grouping factor(s), for which the repeatability should
be estimated. Spelling needs to match the random effect names as given in \code{formula} 
and terms have to be set in quotation marks. The reseved terms "Residual", 
"Overdispersion" and "Fixed" allow the estimation of overdispersion variance, residual 
variance and variance explained by fixed effects, respectively.}

\item{data}{A dataframe that contains the variables included in the \code{formula}
and \code{grname} arguments.}

\item{link}{Link function. \code{logit} and \code{probit} are allowed, defaults to \code{logit}.}

\item{CI}{Width of the required confidence interval between 0 and 1 (defaults to 
0.95).}

\item{nboot}{Number of parametric bootstraps for interval estimation 
(defaults to 1000). Larger numbers of bootstraps give a better
asymtotic CI, but may be time-consuming. Bootstrapping can be switch off by setting 
\code{nboot = 0}. See also \strong{Details} below.}

\item{npermut}{Number of permutations used when calculating asymptotic p-values 
(defaults to 0). Larger numbers of permutations give a better
asymtotic p-values, but may be time-consuming (in particular when multiple grouping factors
are specified). Permutaton tests can be switch off by setting \code{npermut = 0}. 
See also \strong{Details} below.}

\item{parallel}{Boolean to express if parallel computing should be applied (defaults to FALSE). 
If TRUE, bootstraps and permutations will be distributed across multiple cores.}

\item{ncores}{Specifying the number of cores to use for parallelization. On default,
all but one of the available cores are used.}

\item{ratio}{Boolean to express if variances or ratios of variance should be estimated. 
If FALSE, the variance(s) are returned without forming ratios. If TRUE (the default) ratios 
of variances (i.e. repeatabilities) are estimated.}

\item{adjusted}{Boolean to express if adjusted or unadjusted repeatabilities should be estimated. 
If TRUE (the default), the variances explained by fixed effects (if any) will not
be part of the denominator, i.e. repeatabilities are calculated after controlling for 
variation due to covariates. If FALSE, the varianced explained by fixed effects (if any) will
be added to the denominator.}

\item{expect}{A character string specifying the method for estimating the expectation in Poisson models
with log link and in Binomial models with logit link (in all other cases the agrument is ignored). 
The only valid terms are 'meanobs' and 'latent' (and 'liability for binary and proportion data). 
With the default 'meanobs', the expectation is 
estimated as the mean of the observations in the sample. With 'latent', the expectation is
estimated from estiamtes of the intercept and variances on the link scale. While this is a 
preferred solution, it is susceptible to the distribution of fixed effect covariates and gives 
appropriate results typically only when all covariances are centered to zero. With 'liability' 
estimates follow formulae as presented in Nakagawa & Schielzeth (2010). Liability estimates tend 
to be slightly higher.}

\item{rptObj}{The output of a rptR function. Can be specified in combination with update = TRUE
to update bootstraps and permutations}

\item{update}{If TRUE, the rpt object to be updated has to be inputted with the rptObj argument.
The function just updates the permutations and bootstraps, so make sure to specify all other
arguments excactly like for the rpt object specified in rptObj.}

\item{...}{Other parameters for the lmer or glmer call, such as optimizers.}
}
\value{
Returns an object of class \code{rpt} that is a a list with the following elements: 
\item{call}{Function call}
\item{datatype}{Response distribution (here: 'Proportion').}
\item{CI}{Width of the confidence interval.}
\item{R}{\code{data.frame} with point estimates for repeatabilities. Columns
     represent grouping factors of interest. Rows show original and link scale repeatabilites 
     (in this order).}
\item{se}{\code{data.frame} with approximate standard errors (\emph{se}) for repeatabilities. Columns
     are groups of interest. Rows are original and link scale (in this order).
     Note that the distribution might not be symmetrical, in which case the \emph{se} is less informative.}
\item{CI_emp}{\code{list} of two elements containing the confidence intervals for repeatabilities 
     on the link and original scale, respectively. Within each list element, lower and upper CI
     are columns and each row for each grouping factor of interest.}
\item{P}{\code{data.frame} with p-values from a significance test based on likelihood-ratios
     in the first column and significance test based on permutation of residuals for 
     both the original and link scale in the second and third column. Each row represents a grouping
     factor of interest.}
\item{R_boot_link}{Parametric bootstrap samples for \emph{R} on the link scale. Each \code{list}
      element is a grouping factor.}
\item{R_boot_org}{Parametric bootstrap samples for \emph{R} on the original scale. Each \code{list}
      element is a grouping factor.}
\item{R_permut_link}{Permutation samples for \emph{R} on the link scale. Each \code{list}
      element is a grouping factor.}
\item{R_permut_org}{Permutation samples for \emph{R} on the original scale. Each \code{list}
      element is a grouping factor.}
\item{LRT}{List of two elements. \emph{LRT_mod} is the likelihood for the full model and (2) \emph{LRT_table} is a data.frame 
     for the reduced model(s) including columns for the likelihood \emph{logl_red}, the likelihood ratio(s) \emph{LR_D}, 
     p-value(s)\emph{LR_P} and degrees of freedom for the likelihood-ratio test(s) \emph{LR_df}.} 
\item{ngroups}{Number of groups for each grouping level.}
\item{nobs}{Number of observations.}
\item{overdisp}{Overdispersion parameter. Equals the variance in the observational factor random effect}
\item{mod}{Fitted model.}
\item{ratio}{Boolean. TRUE, if ratios have been estimated, FALSE, if variances have been estimated}
\item{adjusted}{Boolean. TRUE, if estimates are adjusted}
\item{all_warnings}{\code{list} with two elements. 'warnings_boot' and 'warnings_permut' contain
     warnings from the lme4 model fitting of bootstrap and permutation samples, respectively.}
}
\description{
Estimates repeatability from a generalized linear mixed-effects models fitted by restricted maximum likelihood (REML).
}
\details{
see details section of \code{\link{rpt}} for details on parametric bootstrapping,
permutation and likelihood-ratio tests.
}
\examples{
data(BeetlesMale)

# prepare proportion data
BeetlesMale$Dark <- BeetlesMale$Colour
BeetlesMale$Reddish <- (BeetlesMale$Colour-1)*-1
BeetlesColour <- aggregate(cbind(Dark, Reddish) ~ Treatment + Population + Container, 
     data=BeetlesMale, FUN=sum)

# Note: nboot and npermut are set to 0 for speed reasons. 

# repeatability with one grouping level
rptProportion(cbind(Dark, Reddish) ~ (1|Population), 
     grname=c("Population"), data=BeetlesColour, nboot=3, npermut=3)

# unadjusted repeatabilities with  fixed effects and 
# estimation of the fixed effect variance
rptProportion(cbind(Dark, Reddish) ~ Treatment + (1|Container) + (1|Population), 
     grname=c("Population", "Fixed"), 
     data=BeetlesColour, nboot=0, npermut=0, adjusted=FALSE)
                   
# variance estimation of random effects, residual and overdispersion 
rptProportion(cbind(Dark, Reddish) ~ Treatment + (1|Container) + (1|Population), 
     grname=c("Container","Population","Residual", "Overdispersion"), 
     data = BeetlesColour, nboot=0, npermut=0, ratio = FALSE)

}
\references{
Carrasco, J. L. & Jover, L.  (2003) \emph{Estimating the generalized 
concordance correlation coefficient through variance components}. Biometrics 59: 849-858.

Faraway, J. J. (2006) \emph{Extending the linear model with R}. Boca Raton, FL, Chapman & Hall/CRC.

Nakagawa, S. & Schielzeth, H. (2010) \emph{Repeatability for Gaussian and 
non-Gaussian data: a practical guide for biologists}. Biological Reviews 85: 935-956
}
\seealso{
\link{rpt}
}
\author{
Holger Schielzeth  (holger.schielzeth@uni-jena.de),
        Shinichi Nakagawa (s.nakagawa@unsw.edu.au) &
        Martin Stoffel (martin.adam.stoffel@gmail.com)
}
