% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diagnose.R
\name{model_lineup}
\alias{model_lineup}
\title{Produce a lineup for a fitted model}
\usage{
model_lineup(fit, fn = augment, nsim = 20, ...)
}
\arguments{
\item{fit}{A model fit to data, such as by \code{lm()} or \code{glm()}}

\item{fn}{A diagnostic function. The function's first argument should be the
fitted model, and it must return a data frame. Defaults to
\code{broom::augment()}, which produces a data frame containing the original
data and additional columns \code{.fitted}, \code{.resid}, and so on. To see a list
of model types supported by \code{broom::augment()}, and to find documentation
on the columns reported for each type of model, load the \code{broom} package
and use \code{methods(augment)}.}

\item{nsim}{Number of total diagnostics. For example, if \code{nsim = 20}, the
diagnostics for \code{fit} are hidden among 19 null diagnostics.}

\item{...}{Additional arguments passed to \code{fn} each time it is called.}
}
\value{
A data frame (tibble) with columns corresponding to the columns
returned by \code{fn}. The additional column \code{.sample} indicates which set of
diagnostics each row is from. For instance, if the true data is in position
5, selecting rows with \code{.sample == 5} will retrieve the diagnostics from
the original model fit.
}
\description{
A lineup hides diagnostics among "null" diagnostics, i.e. the same
diagnostics calculated using models fit to data where all model assumptions
are correct. For each null diagnostic, \code{model_lineup()} simulates new
responses from the model using the fitted covariate values and the model's
error distribution, link function, and so on. Hence the new response values
are generated under ideal conditions: the fitted model is true and all
assumptions hold. \code{decrypt()} reveals which diagnostics are the true
diagnostics.
}
\details{
To generate different kinds of diagnostics, the user can provide a custom
\code{fn}. The \code{fn} should take a model fit as its argument and return a data
frame. For instance, the data frame might contain one row per observation and
include the residuals and fitted values for each observation; or it might be
a single row containing a summary statistic or test statistic.

\code{fn} will be called on the original \code{fit} provided. Then
\code{parametric_boot_distribution()} will be used to simulate data from the model
fit \code{nsim - 1} times, refit the model to each simulated dataset, and run \code{fn}
on each refit model. The null distribution is conditional on X, i.e. the
covariates used will be identical, and only the response values will be
simulated. The data frames are concatenated with an additional \code{.sample}
column identifying which fit each row came from.

When called, this function will print a message such as
\code{decrypt("sD0f gCdC En JP2EdEPn ZY")}. This is how to get the location of the
true diagnostics among the null diagnostics: evaluating this in the R console
will produce a string such as \code{"True data in position 5"}.
}
\section{Model limitations}{
Because this function uses S3 generic methods such as \code{simulate()} and
\code{update()}, it can be used with any model fit for which methods are provided.
In base R, this includes \code{lm()} and \code{glm()}.

The model provided as \code{fit} must be fit using the \code{data} argument to provide
a data frame. For example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{fit <- lm(dist ~ speed, data = cars)
}\if{html}{\out{</div>}}

When simulating new data, this function provides the simulated data as the
\code{data} argument and re-fits the model. If you instead refer directly to local
variables in the model formula, this will not work. For example, if you fit a
model this way:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# will not work
fit <- lm(cars$dist ~ cars$speed)
}\if{html}{\out{</div>}}

It will not be possible to refit the model using simulated datasets, as that
would require modifying your environment to edit \code{cars}.
}

\examples{
fit <- lm(dist ~ speed, data = cars)
model_lineup(fit, nsim = 5)

resids_vs_speed <- function(f) {
  data.frame(resid = residuals(f),
             speed = model.frame(f)$speed)
}
model_lineup(fit, fn = resids_vs_speed, nsim = 5)

}
\references{
Buja et al. (2009). Statistical inference for exploratory data
analysis and model diagnostics. \emph{Philosophical Transactions of the Royal
Society A}, 367 (1906), pp. 4361-4383. \doi{10.1098/rsta.2009.0120}

Wickham et al. (2010). Graphical inference for infovis. \emph{IEEE Transactions on
Visualization and Computer Graphics}, 16 (6), pp. 973-979.
\doi{10.1109/TVCG.2010.161}
}
\seealso{
\code{\link[=parametric_boot_distribution]{parametric_boot_distribution()}} to simulate draws by using the
fitted model to draw new response values; \code{\link[=sampling_distribution]{sampling_distribution()}} to
simulate draws from the population distribution, rather than from the model
}
