% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dim_values.R
\name{dim_values}
\alias{dim_values}
\title{Dim values of a dimension based on the distance to an n-dimensional origin}
\usage{
dim_values(
  data,
  cols,
  dimming_fn = create_dimming_fn(numerator = 1, exponent = 2, add_to_distance = 1),
  origin = NULL,
  origin_fn = NULL,
  dim_col = tail(cols, 1),
  suffix = "_dimmed",
  keep_original = TRUE,
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{cols}{Names of columns in \code{`data`} to calculate distances from.
The dimming column (\code{`dim_col`}) is dimmed based on all the columns.
Each column is considered a dimension.

\strong{N.B.} when the dimming dimension is included in \code{`cols`},
it is used in the distance calculation as well.}

\item{dimming_fn}{\code{Function} for calculating the dimmed values.

\strong{Input}: Two (2) input arguments:
\enumerate{
\item A \code{numeric vector} with the values in the dimming dimension.
\item A \code{numeric vector} with corresponding distances to the origin.
}

\strong{Output}: A \code{numeric vector} with the same length as the input vectors.

E.g.:

\code{function(x, d)\{}

\verb{  }\code{x * (1 / ((1 + d) ^ 2))}

\code{\}}

This kind of dimming function can be created with
\code{\link[rearrr:create_dimming_fn]{create_dimming_fn()}},
which for instance makes it easy to change the exponent (the \code{2} above).}

\item{origin}{Coordinates of the origin to dim around.
A scalar to use in all dimensions
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{dim_col}{Name of column to dim. Default is the last column in \code{`cols`}.

When the \code{`dim_col`} is not present in \code{`cols`}, it is not used in the distance calculation.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the dimmed column,
along with the origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Dims the values in the dimming dimension (last by default)
based on the data point's distance to the origin.

Distance is calculated as:
\deqn{d(P1, P2) = sqrt( (x2 - x1)^2 + (y2 - y1)^2 + (z2 - z1)^2 + ... )}

The default \code{`dimming_fn`} multiplies by the inverse-square of
\eqn{1 + distance} and is calculated as:
\deqn{dimming_fn(x, d) = x * (1 / (1 + d) ^ 2)}

Where \eqn{x} is the value in the dimming dimension. The \eqn{+1} is added
to ensure that values are dimmed even when the distance is below \code{1}. The quickest
way to change the exponent or the \eqn{+1} is with
\code{\link[rearrr:create_dimming_fn]{create_dimming_fn()}}.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and dimming around e.g. the centroid
of each group.
}
\details{
\itemize{
\item Calculates distances to origin with: \deqn{d(P1, P2) = sqrt( (x2 - x1)^2 + (y2 - y1)^2 + (z2 - z1)^2 + ... )}
\item Applies the \code{`dimming_fn`} to the \code{`dim_col`} based on the distances.
}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(purrr)
has_ggplot <- require(ggplot2)  # Attach if installed

# Set seed
set.seed(7)

# Create a data frame with clusters
df <- generate_clusters(
  num_rows = 70,
  num_cols = 3,
  num_clusters = 5,
  compactness = 1.6
) \%>\%
  dplyr::rename(x = D1, y = D2, z = D3) \%>\%
  dplyr::mutate(o = 1)

# Dim the values in the z column
dim_values(
  data = df,
  cols = c("x", "y", "z"),
  origin = c(0.5, 0.5, 0.5)
)

# Dim the values in the `o` column (all 1s)
# around the centroid
dim_values(
  data = df,
  cols = c("x", "y"),
  dim_col = "o",
  origin_fn = centroid
)

# Specify dimming_fn
# around the centroid
dim_values(
  data = df,
  cols = c("x", "y"),
  dim_col = "o",
  origin_fn = centroid,
  dimming_fn = function(x, d) {
    x * 1 / (2^(1 + d))
  }
)

#
# Dim cluster-wise
#

# Group-wise dimming
df_dimmed <- df \%>\%
  dplyr::group_by(.cluster) \%>\%
  dim_values(
    cols = c("x", "y"),
    dim_col = "o",
    origin_fn = centroid
  )

# Plot the dimmed data such that the alpha (opacity) is
# controlled by the dimming
# (Note: This works because the `o` column is 1 for all values)
if (has_ggplot){
  ggplot(
    data = df_dimmed,
    aes(x = x, y = y, alpha = o_dimmed, color = .cluster)
  ) +
    geom_point() +
    theme_minimal() +
    labs(x = "x", y = "y", color = "Cluster", alpha = "o_dimmed")
}
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{expand_distances}()},
\code{\link{expand_distances_each}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}

Other distance functions: 
\code{\link{closest_to}()},
\code{\link{distance}()},
\code{\link{expand_distances}()},
\code{\link{expand_distances_each}()},
\code{\link{furthest_from}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{distance functions}
\concept{mutate functions}
