% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aem.R, R/plot.R
\name{aem}
\alias{aem}
\alias{solve.aem}
\alias{plot.element}
\alias{plot.aem}
\title{Create an analytic element model}
\usage{
aem(
  k,
  top,
  base,
  n,
  ...,
  type = c("variable", "confined"),
  verbose = FALSE,
  maxiter = 10
)

\method{solve}{aem}(a, b, maxiter = 10, verbose = FALSE, ...)

\method{plot}{element}(
  x,
  y = NULL,
  add = FALSE,
  pch = 16,
  cex = 0.75,
  use.widths = TRUE,
  col = "black",
  xlim,
  ylim,
  ...
)

\method{plot}{aem}(x, y = NULL, add = FALSE, xlim, ylim, ...)
}
\arguments{
\item{k}{numeric, hydraulic conductivity of the aquifer.}

\item{top}{numeric, top elevation of the aquifer.}

\item{base}{numeric, bottom elevation of the aquifer.}

\item{n}{numeric, effective porosity of the aquifer as a fraction of total unit volume. Used for determining flow velocities with \code{\link[=velocity]{velocity()}}.}

\item{...}{for \code{aem()}, objects of class \code{element}, or a single (named) list with \code{element} objects. Otherwise, ignored.}

\item{type}{character specifying the type of flow in the aquifer, either \code{variable} (default) or \code{confined}. See details.}

\item{verbose}{logical indicating if information during the solving process should be printed. Defaults to \code{FALSE}.}

\item{maxiter}{integer specifying the maximum allowed iterations for a non-linear solution. Defaults to 10. See details.}

\item{a}{\code{aem} object.}

\item{b}{ignored}

\item{x}{\code{aem} object, or analytic element of class \code{element} to plot. If not a point or line geometry, nothing is plotted.}

\item{y}{ignored}

\item{add}{logical, should the plot be added to the existing plot? Defaults to \code{FALSE}.}

\item{pch}{numeric point symbol value, defaults to \code{16}. For a reference point, a value of \code{4} is used.}

\item{cex}{numeric symbol size value, defaults to \code{0.75}.}

\item{use.widths}{logical, if line elements with non-zero width are plotted, should they be plotted as polygons including
the width (\code{TRUE}; default) or as infinitesimally thin lines (\code{FALSE})?}

\item{col}{color of element. Defaults to \code{'black'}.}

\item{xlim}{numeric, plot limits along the x-axis. Required if \code{add = FALSE}.}

\item{ylim}{numeric, plot limits along the y-axis. Required if \code{add = FALSE}.}
}
\value{
\code{\link[=aem]{aem()}} returns an object of class \code{aem} which is a list consisting of \code{k}, \code{top}, \code{base}, \code{n},
a list containing all elements with the names of the objects specified in \code{...}, and a logical \code{solved}
indicating if the model is solved.

\code{\link[=solve.aem]{solve.aem()}} returns the solved \code{aem} object, i.e. after finding the solution
to the system of equations as constructed by the contained elements.
}
\description{
\code{\link[=aem]{aem()}} creates an analytic element model to which elements can be added

\code{\link[=solve.aem]{solve.aem()}} solves the system of equations as constructed by the elements in the \code{aem} model

\code{\link[=plot.element]{plot.element()}} plots the location of an analytic element with point or line geometry.

\code{\link[=plot.aem]{plot.aem()}} plots the locations of all analytic elements with a point or line geometry
in an \code{aem} object by calling \code{\link[=plot.element]{plot.element()}} on them, or adds them to an existing plot.
}
\details{
The default \code{type = 'variable'} allows for unconfined/confined flow, i.e. flow with variable saturated thickness. If \code{type = 'confined'},
the saturated thickness is always constant and equal to the aquifer thickness. This results in a linear model when head-specified elements with
a resistance are used, whereas \code{type = 'variable'} would create a non-linear model in that case.

\code{\link[=solve.aem]{solve.aem()}} is called on the \code{aem} object before it is returned by \code{aem()}, which solves the system of equations.

\subsection{Solving}{

\code{\link[=solve.aem]{solve.aem()}} sets up the system of equations, and calls \code{\link[=solve]{solve()}} to
solve. If head-specified elements are supplied, an element of class \code{constant} as
created by \code{\link[=constant]{constant()}} (also called the reference point), should be supplied as well.
Constructing an \code{aem} object by a call to \code{\link[=aem]{aem()}} automatically calls \code{\link[=solve.aem]{solve.aem()}}.

If the system of equations is non-linear, i.e. when the flow system is unconfined (variable
saturated thickness) and elements with hydraulic resistance are specified, a Picard iteration is entered.
During each Picard iteration step (outer iteration), the previously solved model parameters are used to set up and
solve a linear system of equations. The model parameters are then updated and the next outer iteration step is
entered, until \code{maxiter} iterations are reached. For an linear model, \code{maxiter} is ignored.
}

\subsection{Plotting}{

If the analytic element has a point geometry and has a collocation point
(e.g. \code{\link[=headwell]{headwell()}}), that point is also plotted with \code{pch = 1}.
}

A reference point (as created by \code{\link[=constant]{constant()}}) is never plotted when plotting the model
as it is not a hydraulic feature. Area-sinks (as created by \code{\link[=areasink]{areasink()}} or \code{\link[=headareasink]{headareasink()}})
are also never plotted as they would clutter the plot. These elements can be plotted by
calling \code{plot()} on them directly.
}
\examples{
k <- 10
top <- 10
base <- 0
n <- 0.2
TR <- k * (top - base)

w <- well(xw = 50, yw = 0, Q = 200)
rf <- constant(xc = -500, yc = 0, h = 20)
uf <- uniformflow(gradient = 0.002, angle = -45, TR = TR)
hdw <- headwell(xw = 0, yw = 100, rw = 0.3, hc = 8)
ls <- linesink(x0 = -200, y0 = -150, x1 = 200, y1 = 150, sigma = 1)

# Creating aem ----
m <- aem(k, top, base, n, w, rf, uf, hdw, ls)

# or with elements in named list
m <- aem(k, top, base, n,
         list('well' = w, 'constant' = rf, 'flow' = uf, 'headwell' = hdw, 'river' = ls),
         type = 'confined')

# Solving ----
m <- solve(m)

# solving requires a reference point (constant) element if head-specified elements are supplied
try(
  m <- aem(k = k, top = top, base = base, n = n, w, uf, hdw)
)

# Plotting ----
plot(ls)
plot(w, add = TRUE)
plot(uf) # empty

plot(m, xlim = c(-500, 500), ylim = c(-250, 250))

xg <- seq(-500, 500, length = 200)
yg <- seq(-250, 250, length = 100)

contours(m, x = xg, y = yg, col = 'dodgerblue', nlevels = 20)
plot(m, add = TRUE)

}
\seealso{
\code{\link[=add_element]{add_element()}} \code{\link[=contours]{contours()}}
}
