/*

 Copyright (c) 2021, Dominic Szablewski - https://phoboslab.org
 SPDX-License-Identifier: MIT


 QOI - The "Quite OK Image" format for fast, lossless image compression

 -- About

 QOI encodes and decodes images in a lossless format. Compared to stb_image and
 stb_image_write QOI offers 20x-50x faster encoding, 3x-4x faster decoding and
 20% better compression.


 -- Synopsis

 // Define `QOI_IMPLEMENTATION` in *one* C/C++ file before including this
 // library to create the implementation.

#define QOI_IMPLEMENTATION
#include "qoi.h"

 // Encode and store an RGBA buffer to the file system. The qoi_desc describes
 // the input pixel data.
 qoi_write("image_new.qoi", rgba_pixels, &(qoi_desc){
                     .width = 1920,
                     .height = 1080,
                     .channels = 4,
                     .colorspace = QOI_SRGB
 });

 // Load and decode a QOI image from the file system into a 32bbp RGBA buffer.
 // The qoi_desc struct will be filled with the width, height, number of channels
 // and colorspace read from the file header.
 qoi_desc desc;
 void *rgba_pixels = qoi_read("image.qoi", &desc, 4);



 -- Documentation

 This library provides the following functions;
 - qoi_read    -- read and decode a QOI file
 - qoi_decode  -- decode the raw bytes of a QOI image from memory
 - qoi_write   -- encode and write a QOI file
 - qoi_encode  -- encode an rgba buffer into a QOI image in memory

 See the function declaration below for the signature and more information.

 If you don't want/need the qoi_read and qoi_write functions, you can define
 QOI_NO_STDIO before including this library.

 This library uses malloc() and free(). To supply your own malloc implementation
 you can define QOI_MALLOC and QOI_FREE before including this library.

 This library uses memset() to zero-initialize the index. To supply your own
 implementation you can define QOI_ZEROARR before including this library.


 -- Data Format

 A QOI file has a 14 byte header, followed by any number of data "chunks" and an
 8-byte end marker.

 struct qoi_header_t {
 char     magic[4];   // magic bytes "qoif"
 uint32_t width;      // image width in pixels (BE)
 uint32_t height;     // image height in pixels (BE)
 uint8_t  channels;   // 3 = RGB, 4 = RGBA
 uint8_t  colorspace; // 0 = sRGB with linear alpha, 1 = all channels linear
 };

 Images are encoded row by row, left to right, top to bottom. The decoder and
 encoder start with {r: 0, g: 0, b: 0, a: 255} as the previous pixel value. An
 image is complete when all pixels specified by width * height have been covered.

 Pixels are encoded as
 - a run of the previous pixel
 - an index into an array of previously seen pixels
 - a difference to the previous pixel value in r,g,b
 - full r,g,b or r,g,b,a values

 The color channels are assumed to not be premultiplied with the alpha channel
 ("un-premultiplied alpha").

 A running array[64] (zero-initialized) of previously seen pixel values is
 maintained by the encoder and decoder. Each pixel that is seen by the encoder
 and decoder is put into this array at the position formed by a hash function of
 the color value. In the encoder, if the pixel value at the index matches the
 current pixel, this index position is written to the stream as QOI_OP_INDEX.
 The hash function for the index is:

 index_position = (r * 3 + g * 5 + b * 7 + a * 11) % 64

 Each chunk starts with a 2- or 8-bit tag, followed by a number of data bits. The
 bit length of chunks is divisible by 8 - i.e. all chunks are byte aligned. All
 values encoded in these data bits have the most significant bit on the left.

 The 8-bit tags have precedence over the 2-bit tags. A decoder must check for the
 presence of an 8-bit tag first.

 The byte stream's end is marked with 7 0x00 bytes followed a single 0x01 byte.


 The possible chunks are:


 .- QOI_OP_INDEX ----------.
 |         Byte[0]         |
 |  7  6  5  4  3  2  1  0 |
 |-------+-----------------|
 |  0  0 |     index       |
 `-------------------------`
 2-bit tag b00
 6-bit index into the color index array: 0..63

 A valid encoder must not issue 2 or more consecutive QOI_OP_INDEX chunks to the
 same index. QOI_OP_RUN should be used instead.


 .- QOI_OP_DIFF -----------.
 |         Byte[0]         |
 |  7  6  5  4  3  2  1  0 |
 |-------+-----+-----+-----|
 |  0  1 |  dr |  dg |  db |
 `-------------------------`
 2-bit tag b01
 2-bit   red channel difference from the previous pixel between -2..1
 2-bit green channel difference from the previous pixel between -2..1
 2-bit  blue channel difference from the previous pixel between -2..1

 The difference to the current channel values are using a wraparound operation,
 so "1 - 2" will result in 255, while "255 + 1" will result in 0.

 Values are stored as unsigned integers with a bias of 2. E.g. -2 is stored as
 0 (b00). 1 is stored as 3 (b11).

 The alpha value remains unchanged from the previous pixel.


 .- QOI_OP_LUMA -------------------------------------.
 |         Byte[0]         |         Byte[1]         |
 |  7  6  5  4  3  2  1  0 |  7  6  5  4  3  2  1  0 |
 |-------+-----------------+-------------+-----------|
 |  1  0 |  green diff     |   dr - dg   |  db - dg  |
 `---------------------------------------------------`
 2-bit tag b10
 6-bit green channel difference from the previous pixel -32..31
 4-bit   red channel difference minus green channel difference -8..7
 4-bit  blue channel difference minus green channel difference -8..7

 The green channel is used to indicate the general direction of change and is
 encoded in 6 bits. The red and blue channels (dr and db) base their diffs off
 of the green channel difference and are encoded in 4 bits. I.e.:
 dr_dg = (cur_px.r - prev_px.r) - (cur_px.g - prev_px.g)
 db_dg = (cur_px.b - prev_px.b) - (cur_px.g - prev_px.g)

 The difference to the current channel values are using a wraparound operation,
 so "10 - 13" will result in 253, while "250 + 7" will result in 1.

 Values are stored as unsigned integers with a bias of 32 for the green channel
 and a bias of 8 for the red and blue channel.

 The alpha value remains unchanged from the previous pixel.


 .- QOI_OP_RUN ------------.
 |         Byte[0]         |
 |  7  6  5  4  3  2  1  0 |
 |-------+-----------------|
 |  1  1 |       run       |
 `-------------------------`
 2-bit tag b11
 6-bit run-length repeating the previous pixel: 1..62

 The run-length is stored with a bias of -1. Note that the run-lengths 63 and 64
 (b111110 and b111111) are illegal as they are occupied by the QOI_OP_RGB and
 QOI_OP_RGBA tags.


 .- QOI_OP_RGB ------------------------------------------.
 |         Byte[0]         | Byte[1] | Byte[2] | Byte[3] |
 |  7  6  5  4  3  2  1  0 | 7 .. 0  | 7 .. 0  | 7 .. 0  |
 |-------------------------+---------+---------+---------|
 |  1  1  1  1  1  1  1  0 |   red   |  green  |  blue   |
 `-------------------------------------------------------`
 8-bit tag b11111110
 8-bit   red channel value
 8-bit green channel value
 8-bit  blue channel value

 The alpha value remains unchanged from the previous pixel.


 .- QOI_OP_RGBA ---------------------------------------------------.
 |         Byte[0]         | Byte[1] | Byte[2] | Byte[3] | Byte[4] |
 |  7  6  5  4  3  2  1  0 | 7 .. 0  | 7 .. 0  | 7 .. 0  | 7 .. 0  |
 |-------------------------+---------+---------+---------+---------|
 |  1  1  1  1  1  1  1  1 |   red   |  green  |  blue   |  alpha  |
 `-----------------------------------------------------------------`
 8-bit tag b11111111
 8-bit   red channel value
 8-bit green channel value
 8-bit  blue channel value
 8-bit alpha channel value

 */


/* -----------------------------------------------------------------------------
 Header - Public functions */

#ifndef QOI_H
#define QOI_H

#define QOI_SRGB   0
#define QOI_LINEAR 1

typedef struct {
  unsigned int width;
  unsigned int height;
  unsigned char channels;
  unsigned char colorspace;
} qoi_desc;

#ifndef QOI_MALLOC
#define QOI_MALLOC(sz) malloc(sz)
#define QOI_FREE(p)    free(p)
#endif
#ifndef QOI_ZEROARR
#define QOI_ZEROARR(a) memset((a),0,sizeof(a))
#endif

#define QOI_OP_INDEX  0x00 /* 00xxxxxx */
#define QOI_OP_DIFF   0x40 /* 01xxxxxx */
#define QOI_OP_LUMA   0x80 /* 10xxxxxx */
#define QOI_OP_RUN    0xc0 /* 11xxxxxx */
#define QOI_OP_RGB    0xfe /* 11111110 */
#define QOI_OP_RGBA   0xff /* 11111111 */

#define QOI_MASK_2    0xc0 /* 11000000 */
//
#define QOI_COLOR_HASH(C) (C.rgba.r*3 + C.rgba.g*5 + C.rgba.b*7 + C.rgba.a*11)
#define QOI_MAGIC                                        \
(((unsigned int)'q') << 24 | ((unsigned int)'o') << 16 | \
((unsigned int)'i') <<  8 | ((unsigned int)'f'))
#define QOI_HEADER_SIZE 14

/* 2GB is the max file size that this implementation can safely handle. We guard
 against anything larger than that, assuming the worst case with 5 bytes per
 pixel, rounded down to a nice clean value. 400 million pixels ought to be
 enough for anybody. */
#define QOI_PIXELS_MAX ((unsigned int)400000000)

typedef union {
  struct { unsigned char r, g, b, a; } rgba;
  unsigned int v;
} qoi_rgba_t;

static const unsigned char qoi_padding[8] = {0,0,0,0,0,0,0,1};

static void qoi_write_32(unsigned char *bytes, int *p, unsigned int v) {
  bytes[(*p)++] = (0xff000000 & v) >> 24;
  bytes[(*p)++] = (0x00ff0000 & v) >> 16;
  bytes[(*p)++] = (0x0000ff00 & v) >> 8;
  bytes[(*p)++] = (0x000000ff & v);
}

static unsigned int qoi_read_32(const unsigned char *bytes, int *p) {
  unsigned int a = bytes[(*p)++];
  unsigned int b = bytes[(*p)++];
  unsigned int c = bytes[(*p)++];
  unsigned int d = bytes[(*p)++];
  return a << 24 | b << 16 | c << 8 | d;
}

#endif // QOI_H
