% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logitBvs.R
\name{logitBvs}
\alias{logitBvs}
\title{Bayesian variable selection for the binomial logit model}
\usage{
logitBvs(
  y,
  N,
  X,
  model = list(),
  prior = list(),
  mcmc = list(),
  start = NULL,
  BVS = TRUE
)
}
\arguments{
\item{y}{an integer vector of binomial counts}

\item{N}{an integer vector containing the number of trials}

\item{X}{a design matrix (including an intercept term)}

\item{model}{an (optional) list specifying the structure of the model 
(see details)}

\item{prior}{an (optional) list of prior settings and hyper-parameters 
controlling the priors (see details)}

\item{mcmc}{an (optional) list of MCMC sampling options (see details)}

\item{start}{an (optional), numeric vector containing starting values for the 
regression effects (including an intercept term); defaults to \code{NULL}
(i.e. a vector of zeros is used).}

\item{BVS}{if \code{TRUE} (default), Bayesian variable selection is performed
to identify regressors with a non-zero effect; otherwise, an unrestricted 
model is estimated (without variable selection).}
}
\value{
The function returns an object of class "\code{pogit}" with methods
 \code{\link{print.pogit}}, \code{\link{summary.pogit}} and 
 \code{\link{plot.pogit}}. 

 The returned object is a list containing the following elements: 

 \item{\code{samplesL}}{a named list containing the samples from the posterior
   distribution of the parameters in the binomial logit model 
   (see also \code{msave}): 
   \describe{
   \item{\code{alpha, thetaAlpha}}{regression coefficients \eqn{\alpha} and
   \eqn{\theta_\alpha}}
   \item{\code{pdeltaAlpha}}{P(\eqn{\delta_\alpha}=1)}
   \item{\code{psiAlpha}}{scale parameter \eqn{\psi_\alpha} of the slab component}
   \item{\code{pgammaAlpha}}{P(\eqn{\gamma_\alpha}=1)}
   \item{\code{ai}}{cluster-specific random intercept}
   }}
 \item{\code{data}}{a list containing the data \code{y}, \code{N} and \code{X}}
 \item{\code{model.logit}}{a list containing details on the model specification, 
 see details for \code{model}}
 \item{\code{mcmc}}{see details for \code{mcmc}}
 \item{\code{prior.logit}}{see details for \code{prior}}
 \item{\code{dur}}{a list containing the total runtime (\code{total}) 
   and the runtime after burn-in (\code{durM}), in seconds}
 \item{\code{BVS}}{see arguments}
 \item{\code{start}}{a list containing starting values, see arguments}
 \item{\code{family}}{"logit"}
 \item{\code{call}}{function call}
}
\description{
This function performs Bayesian variable selection for binomial logit regression
models via spike and slab priors. A cluster-specific random intercept can be included 
in the model to account for within-cluster dependence with variance selection of 
the random intercept to determine whether there is between-cluster variation 
in the model. For posterior inference, a MCMC sampling algorithm is used which 
is based on data augmentation.
}
\details{
The method provides Bayesian variable selection for binomial logit models 
using mixture priors with a spike and a slab component to identify regressors 
with a non-zero effect. More specifically, a Dirac spike is used, i.e. a 
point mass at zero and (by default), the slab component is specified as a scale 
mixture of normal distributions, resulting in a Student-t distribution with 
2\code{psi.nu} degrees of freedom. 
In the more general random intercept model, variance selection of the random
intercept is based on the non-centered parameterization of the model, where
the signed standard deviation \eqn{\theta_\alpha} of the random intercept term 
appears as a further regression effect in the model equation. 
For details, see Wagner and Duller (2012). 

The implementation of Bayesian variable selection further relies on the 
representation of the binomial logit model as a Gaussian regression model 
in auxiliary variables. Data augmentation is based on Fussl et 
al. (2013), who show that the binomial logit model can be represented as a 
linear regression model in the latent variable, which has an interpretation as 
the difference of aggregated utilities. The error distribution in the auxiliary 
model is approximated by a finite scale mixture of normal distributions, where
the mixture parameters are taken from the R package \code{binomlogit}. 
See Fussl (2014) for details. 

For details concerning the sampling algorithm see Dvorzak and Wagner (2016) 
and Wagner and Duller (2012). 


Details for model specification (see arguments): 
\describe{ 
 \item{\code{model}:}{
 \describe{
   \item{\code{deltafix}}{an indicator vector of length \code{ncol(X)-1} 
   specifying which regression effects are subject to selection (i.e., 0 =
   subject to selection, 1 = fix in the model); defaults to a vector of zeros.} 
   \item{\code{gammafix}}{an indicator for variance selection of the random 
   intercept term (i.e., 0 = with variance selection (default), 1 = no
   variance selection); only used if a random intercept is includued in the 
   model (see \code{ri}).} 
   \item{\code{ri}}{logical. If \code{TRUE}, a cluster-specific
   random intercept is included in the model; defaults to \code{FALSE}.} 
   \item{\code{clusterID}}{a numeric vector of length equal to the number
   of observations containing the cluster ID c = 1,...,C for each observation 
   (required if \code{ri=TRUE}).} 
 }}
 \item{\code{prior}:}{
 \describe{
  \item{\code{slab}}{distribution of the slab component, i.e. "\code{Student}" 
   (default) or "\code{Normal}".}
   \item{\code{psi.nu}}{hyper-parameter of the Student-t slab (used for a 
   "\code{Student}" slab); defaults to 5.}
   \item{\code{m0}}{prior mean for the intercept parameter; defaults to 0.}
   \item{\code{M0}}{prior variance for the intercept parameter; defaults to 100.}
   \item{\code{aj0}}{a vector of prior means for the regression effects (which
   is encoded in a normal distribution, see notes); defaults to vector of zeros.}
   \item{\code{V}}{variance of the slab; defaults to 5.}
   \item{\code{w}}{hyper-parameters of the Beta-prior for the mixture weight
   \eqn{\omega}; defaults to \code{c(wa0=1, wb0=1)}, i.e. a uniform
   distribution.}
   \item{\code{pi}}{hyper-parameters of the Beta-prior for the mixture weight 
   \eqn{\pi}; defaults to \code{c(pa0=1, pb0=1)}, i.e. a uniform 
   distribution.}
 }}
 \item{\code{mcmc}:}{
 \describe{
   \item{\code{M}}{number of MCMC iterations after the burn-in phase; defaults 
   to 8000.} 
   \item{\code{burnin}}{number of MCMC iterations discarded as burn-in; 
   defaults to 2000.} 
   \item{\code{thin}}{thinning parameter; defaults to 1.} 
   \item{\code{startsel}}{number of MCMC iterations drawn from the unrestricted
   model (e.g., \code{burnin/2}); defaults to 1000.} 
   \item{\code{verbose}}{MCMC progress report in each \code{verbose}-th 
   iteration step; defaults to 500. If \code{verbose=0}, no output is 
   generated.} 
   \item{\code{msave}}{returns additional output with variable 
   selection details (i.e. posterior samples for \eqn{\omega}, 
   \eqn{\delta}, \eqn{\pi}, \eqn{\gamma}); defaults to \code{FALSE}.} 
}}}
}
\note{
If prior information on the regression parameters is available, this
 information is encoded in a normal distribution instead of the 
 spike and slab prior (\code{BVS} is set to \code{FALSE}). 
 
 For binary observations, a vector of ones for the number of trials \code{N}
 is required.
}
\examples{
\dontrun{
## Examples below should take about 1-2 minutes.

# load simulated data set 'simul_binomial'
data(simul_binomial)
y <- simul_binomial$y
N <- simul_binomial$N
X <- as.matrix(simul_binomial[, -c(1, 2)])

# Bayesian variable selection for simulated data set
m1 <- logitBvs(y = y, N = N, X = X)

# print, summarize and plot results
print(m1)
summary(m1)
plot(m1)

# MCMC sampling without BVS with specific MCMC and prior settings
m2 <- logitBvs(y = y, N = N, X = X, prior = list(slab = "Normal"), 
               mcmc = list(M = 4000, burnin = 1000, thin = 5),
               BVS = FALSE)
print(m2)    
summary(m2)
plot(m2, maxPlots = 4) 

# BVS with specification of regression effects subject to selection
m3 <- logitBvs(y = y, N = N, X = X, mcmc = list(M = 4000, burnin = 1000), 
               model = list(deltafix = c(1, 1, 1, 0, 0, 0, 1, 0, 0)))   
print(m3)   
summary(m3)
plot(m3, burnin = FALSE, maxPlots = 4)
plot(m3, type = "acf", maxPlots = 4)       
}
}
\references{
Dvorzak, M. and Wagner, H. (2016). Sparse Bayesian modelling
 of underreported count data. \emph{Statistical Modelling}, \strong{16}(1),
 24 - 46, \doi{10.1177/1471082x15588398}.

Fussl, A., Fruehwirth-Schnatter, S. and Fruehwirth, R. (2013). 
  Efficient MCMC for Binomial Logit Models. \emph{ACM Transactions on 
  Modeling and Computer Simulation}, 23, \strong{1}, Article 3, 1-21.

Fussl, A. (2014). \code{binomlogit}: Efficient MCMC for Binomial 
  Logit Models. R package version 1.2, 
  \url{https://CRAN.R-project.org/package=binomlogit}.

Wagner, H. and Duller, C. (2012). Bayesian model selection for 
  logistic regression models with random intercept. \emph{Computational 
  Statistics and Data Analysis}, \strong{56}, 1256-1274.
}
\seealso{
\code{\link{pogitBvs}}
}
\author{
Michaela Dvorzak <m.dvorzak@gmx.at>, Helga Wagner
}
\keyword{models}
