% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lsirmicc.R
\name{lsirmicc}
\alias{lsirmicc}
\title{Latent-Space Item Characteristic Curve (ICC)}
\usage{
lsirmicc(
  item_id,
  posterior = NULL,
  beta = NULL,
  gamma = NULL,
  w_pos = NULL,
  z_pos = NULL,
  alpha_grid = seq(-4, 4, length.out = 201),
  person_id = NULL,
  compare = TRUE,
  credibleRibbon = FALSE,
  cred_level = 0.95,
  reference = c("item", "person-global", "origin"),
  ref_col = "grey40",
  person_cols = NULL
)
}
\arguments{
\item{item_id}{Scalar index of the item to plot.}

\item{posterior}{Optional list of draws with components
\describe{
\item{\code{beta}}{\eqn{M \times I}{M x I} matrix of item intercepts.}
\item{\code{gamma}}{Length-\code{M} vector of distance weights.}
\item{\code{w}}{Either an \eqn{M \times I \times D}{M x I x D} array or a length-\code{M}
list of \eqn{I \times D}{I x D} matrices of item coordinates.}
\item{\code{z}}{Optional array/list of person coordinates (same format as
\code{w}). Only needed if you request \code{person_id} or
\code{reference = "person-global"}.}
}}

\item{beta, gamma}{Numeric point estimates used \strong{only} when
\code{posterior = NULL}. \code{beta} may be scalar or length-\code{I}.}

\item{w_pos, z_pos}{Matrices of point estimates for item (\eqn{I \times D}{I x D}) and
person (\eqn{N \times D}{N x D}) coordinates, used only when \code{posterior = NULL}.}

\item{alpha_grid}{Numeric vector of ability values (default
\code{seq(-4, 4, length.out = 201)}).}

\item{person_id}{\code{NULL} (no person curves) or integer vector of
respondent indices to overlay.}

\item{compare}{Logical. If \code{TRUE} (default) the reference curve is
drawn in addition to any person curves; if \code{FALSE} only person curves
appear.}

\item{credibleRibbon}{Logical. Draw the credible ribbon for draws-based
inputs? Ignored (forced \code{FALSE}) when \code{posterior = NULL}.
Default \code{FALSE} to keep plots uncluttered.}

\item{cred_level}{Width of the credible ribbon (e.g., \code{0.95}).}

\item{reference}{One of \code{"item"}, \code{"origin"}, or
\code{"person-global"}; see Details.}

\item{ref_col}{Colour for the reference curve.}

\item{person_cols}{Optional vector of colours for person curves; recycled or
auto-generated as needed.}
}
\value{
(Invisibly) a \pkg{ggplot2} object; the plot is displayed as a
side-effect.
}
\description{
Plots the LSIRM ICC for \strong{one item} on a user-defined grid of ability
values (\code{alpha_grid}). The function supports two ways of supplying inputs:
}
\details{
\itemize{
\item \strong{Point-estimate inputs (default)} - leave \code{posterior = NULL} and
supply deterministic \code{beta}, \code{gamma}, \code{w_pos}, and (if needed) \code{z_pos}.
A single curve per requested group is drawn (no ribbon).
\item \strong{Draws-based inputs} - supply a \code{posterior} list with draws of
\code{beta}, \code{gamma}, \code{w}, and optionally \code{z}. The plotted curve is the
\strong{posterior-predictive mean probability} at each \eqn{\theta}{theta}
(i.e., average over draws). Optionally add a credible ribbon via
\code{credibleRibbon = TRUE} with width \code{cred_level}.
}

The probability model is

\deqn{P(Y_{ij}=1 \mid \theta_{j}, d_{ij})
      = \operatorname{logit}^{-1}\!\bigl(\theta_{j} + \beta_{i} - \gamma\,d_{ij}\bigr)}

where \eqn{d_{ij} = \lVert z_{j} - w_{i} \rVert}. Choice of the
reference position (\code{reference = "item"}, \code{"origin"}, or \code{"person-global"})
determines how \eqn{d_{ij}} is computed for the \emph{baseline} (grey) curve.
}
\section{Curve types}{

\itemize{
\item \strong{Reference curve} - distance is computed from the chosen reference
position to the item for every posterior draw (or once with point-estimate
inputs). Shown unless \code{compare = FALSE}.
\item \strong{Person curve(s)} - distance is computed from the latent position(s) of
respondent(s) listed in \code{person_id}. Requires \code{z} (posterior draws or
point estimates).
}
}

\examples{
## ---- reproducible demonstration ------------------------------------
set.seed(1)
I <- 6; N <- 40; D <- 2; M <- 300           # toy dimensions

## 1. Point-estimate inputs (default) ---------------------------------
beta_hat  <- 0.3
gamma_hat <- 1.2
w_hat     <- matrix(rnorm(I * D), I, D)
z_hat     <- matrix(rnorm(N * D), N, D)

# population curve + one person (no ribbon in point-estimate usage)
lsirmicc(item_id  = 4,
         beta     = beta_hat,
         gamma    = gamma_hat,
         w_pos    = w_hat,
         z_pos    = z_hat,
         person_id = 7)

## 2. Draws-based inputs (posterior list) ------------------------------
w_base <- matrix(0, I, D); w_base[, 1] <- seq(-1.2, 1.2, length.out = I)
z_base <- matrix(0, N, D); z_base[, 1] <- rep(c(-0.6, 0.6), length.out = N)

posterior <- list(
  beta  = matrix(rnorm(M * I, 0, 0.25), M, I),
  gamma = rgamma(M, shape = 300, rate = 300),
  w     = array(rep(w_base, each = M), c(M, I, D)) +
          array(rnorm(M * I * D, sd = 0.12), c(M, I, D)),
  z     = array(rep(z_base, each = M), c(M, N, D)) +
          array(rnorm(M * N * D, sd = 0.12), c(M, N, D))
)

# posterior-predictive mean curve with ribbon and two people
lsirmicc(item_id   = 2,
         posterior = posterior,
         person_id = c(22, 31),
         credibleRibbon = TRUE,
         cred_level = 0.95,
         person_cols = c("red", "blue"))

}
