% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anthropic_batch_api.R
\name{run_anthropic_batch_pipeline}
\alias{run_anthropic_batch_pipeline}
\title{Run an Anthropic batch pipeline for pairwise comparisons}
\usage{
run_anthropic_batch_pipeline(
  pairs,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  reasoning = c("none", "enabled"),
  include_thoughts = FALSE,
  batch_input_path = NULL,
  batch_output_path = NULL,
  poll = TRUE,
  interval_seconds = 60,
  timeout_seconds = 86400,
  api_key = Sys.getenv("ANTHROPIC_API_KEY"),
  anthropic_version = "2023-06-01",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{pairs}{Tibble or data frame with at least columns \code{ID1},
\code{text1}, \code{ID2}, \code{text2}.}

\item{model}{Anthropic model name (for example \code{"claude-sonnet-4-5"}).}

\item{trait_name}{Trait name to pass to
\code{\link{build_anthropic_batch_requests}}.}

\item{trait_description}{Trait description to pass to
\code{\link{build_anthropic_batch_requests}}.}

\item{prompt_template}{Prompt template string, typically from
\code{\link{set_prompt_template}}.}

\item{reasoning}{Character scalar; one of \code{"none"} or \code{"enabled"}.
See details above for how \code{include_thoughts} influences this value and
how temperature defaults are derived.}

\item{include_thoughts}{Logical; if \code{TRUE}, requests extended thinking
from Claude (by setting \code{reasoning = "enabled"} when necessary) and
parses any thinking blocks into a \code{thoughts} column in the batch
results.}

\item{batch_input_path}{Path to write the JSON file containing the
\code{requests} object. Defaults to a temporary file with suffix
\code{".json"}.}

\item{batch_output_path}{Path to write the downloaded \code{.jsonl} results
if \code{poll = TRUE}. Defaults to a temporary file with suffix
\code{".jsonl"}.}

\item{poll}{Logical; if \code{TRUE}, the function will poll the batch until
it reaches \code{processing_status = "ended"} using
\code{\link{anthropic_poll_batch_until_complete}} and then download and
parse the output. If \code{FALSE}, it stops after creating the batch and
returns without polling or parsing.}

\item{interval_seconds}{Polling interval in seconds (used when
\code{poll = TRUE}).}

\item{timeout_seconds}{Maximum total time in seconds for polling before
giving up (used when \code{poll = TRUE}).}

\item{api_key}{Optional Anthropic API key. Defaults to
\code{Sys.getenv("ANTHROPIC_API_KEY")}.}

\item{anthropic_version}{Anthropic API version string passed as the
\code{anthropic-version} HTTP header. Defaults to \code{"2023-06-01"}.}

\item{verbose}{Logical; if \code{TRUE}, prints progress messages while
polling.}

\item{...}{Additional Anthropic parameters forwarded to
\code{\link{build_anthropic_batch_requests}} (for example
\code{max_tokens}, \code{temperature}, \code{top_p},
\code{thinking_budget_tokens}).}
}
\value{
A list with elements (aligned with
\code{\link{run_openai_batch_pipeline}}):
\describe{
\item{batch_input_path}{Path to the JSON file containing the batch
\code{requests} object.}
\item{batch_output_path}{Path to the downloaded \code{.jsonl} results file
if \code{poll = TRUE}, otherwise \code{NULL}.}
\item{file}{Always \code{NULL} for Anthropic batches (OpenAI uses a File
object here). Included for structural compatibility.}
\item{batch}{Message Batch object; if \code{poll = TRUE}, this is the final
batch after polling, otherwise the initial batch returned by
\code{\link{anthropic_create_batch}}.}
\item{results}{Parsed tibble from
\code{\link{parse_anthropic_batch_output}} if \code{poll = TRUE},
otherwise \code{NULL}.}
}
}
\description{
This high-level helper mirrors \code{\link{run_openai_batch_pipeline}} but
targets Anthropic's \emph{Message Batches API}. It:
}
\details{
\enumerate{
\item Builds Anthropic batch requests from a tibble of pairs using
\code{\link{build_anthropic_batch_requests}}.
\item Writes a JSON file containing the \code{requests} object for
reproducibility.
\item Creates a Message Batch via \code{\link{anthropic_create_batch}}.
\item Optionally polls until the batch reaches \code{processing_status =
    "ended"} using \code{\link{anthropic_poll_batch_until_complete}}.
\item If polling is enabled, downloads the \code{.jsonl} result file with
\code{\link{anthropic_download_batch_results}} and parses it via
\code{\link{parse_anthropic_batch_output}}.
}

It is the Anthropic analogue of \code{\link{run_openai_batch_pipeline}} and
returns a list with the same overall structure so that downstream code can
treat the two backends uniformly.

When \code{include_thoughts = TRUE} and \code{reasoning} is left at its
default of \code{"none"}, this function automatically upgrades
\code{reasoning} to \code{"enabled"} so that Claude's extended thinking
blocks are returned and parsed into the \code{thoughts} column by
\code{\link{parse_anthropic_batch_output}}.

\strong{Temperature and reasoning defaults}

Temperature and thinking-mode behaviour are controlled by
\code{\link{build_anthropic_batch_requests}}:
\itemize{
\item When \code{reasoning = "none"} (no extended thinking):
\itemize{
\item The default \code{temperature} is \code{0} (deterministic),
unless you explicitly supply a \code{temperature} argument via
\code{...}.
\item The default \code{max_tokens} is \code{768}, unless you
override it via \code{max_tokens} in \code{...}.
}
\item When \code{reasoning = "enabled"} (extended thinking enabled):
\itemize{
\item \code{temperature} \strong{must} be \code{1}. If you supply a
different value in \code{...},
\code{build_anthropic_batch_requests()} will throw an error.
\item By default, \code{max_tokens = 2048} and
\code{thinking_budget_tokens = 1024}, subject to the constraint
\code{1024 <= thinking_budget_tokens < max_tokens}. Violations of
this constraint also produce an error.
}
}

Therefore, when you run batches without extended thinking (the usual case),
the effective default is a temperature of \code{0}. When you explicitly use
extended thinking (either by setting \code{reasoning = "enabled"} or by
using \code{include_thoughts = TRUE}), Anthropic's requirement of
\code{temperature = 1} is enforced.
}
\examples{
\dontrun{
# Requires ANTHROPIC_API_KEY and network access.
library(pairwiseLLM)

data("example_writing_samples", package = "pairwiseLLM")

pairs <- example_writing_samples |>
  make_pairs() |>
  sample_pairs(n_pairs = 5, seed = 123) |>
  randomize_pair_order(seed = 456)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Standard batch without extended thinking
pipeline_none <- run_anthropic_batch_pipeline(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "none",
  include_thoughts  = FALSE,
  interval_seconds  = 60,
  timeout_seconds   = 3600,
  verbose           = TRUE
)

pipeline_none$batch$processing_status
head(pipeline_none$results)

# Batch with extended thinking and thoughts column
pipeline_thoughts <- run_anthropic_batch_pipeline(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  include_thoughts  = TRUE,
  interval_seconds  = 60,
  timeout_seconds   = 3600,
  verbose           = TRUE
)

pipeline_thoughts$batch$processing_status
head(pipeline_thoughts$results)
}

}
