% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulator.R
\name{Simulator}
\alias{Simulator}
\title{Simulator R6 Object}
\description{
Creates a simulation setup, where a function \code{f} is evaluated \code{runs} times,
optionally at each combination of input values.

Provides some convenience (see below for more details):
\itemize{
\item Simulation results can be restored from a backup if the R session crashes.
\item More simulation runs can be conducted after the initial simulation, failed
simulation cases can be re-run.
\item Parallel computation and progress updates are supported.
}
}
\details{
\subsection{Backup}{

Simulation results can be saved to disk, allowing you to restore the results
if the R session is interrupted or crashes before the simulation completes.
To enable backup, set \code{backup = TRUE} in the \verb{$go()} method, which will
create a backup directory at the location specified by \code{path}.
To restore, use \code{Simulator$initialize(use_backup = path)}.
}

\subsection{More runs and re-run}{

If additional simulation runs are needed, simply call the \verb{$go()} method
again. Any cases that were not successfully completed in previous runs will
be attempted again.
}

\subsection{Parallel computation}{

By default, simulations run sequentially. But since they are independent,
they can be parallelized to decrease computation time. To enable parallel
computation, use the \href{https://future.futureverse.org/}{\code{{future}} framework}.
For example, run
\preformatted{future::plan(future::multisession, workers = 4)}
in advance for computation in 4 parallel R sessions.
}

\subsection{Progress updates}{

Use the \href{https://progressr.futureverse.org/}{\code{{progressr}} framework} to
get progress updates. For example, run the following in advance:
\preformatted{progressr::handlers(global = TRUE)
progressr::handlers(
  progressr::handler_progress(format = ">> :percent, :eta to go :message")
)
}
}
}
\examples{
# 1. Initialize a new simulation setup:
object <- Simulator$new(verbose = TRUE)

# 2. Define function `f` and arguments (if any):
f <- function(x, y = 1) {
  Sys.sleep(runif(1)) # to see progress updates
  x + y
}
x_args <- list(1, 2)
object$define(f = f, x = x_args)
print(object)

# 3. Define 'future' and 'progress' (optional):
\dontrun{
future::plan(future::sequential)
progressr::handlers(global = TRUE)}

# 4. Evaluate `f` `runs` times at each parameter combination (backup is optional):
path <- file.path(tempdir(), paste0("backup_", format(Sys.time(), "\%Y-\%m-\%d-\%H-\%M-\%S")))
object$go(runs = 2, backup = TRUE, path = path)

# 5. Access the results:
object$results

# 6. Check if cases are pending or if an error occurred:
object$cases

# 7. Restore simulation results from backup:
object_restored <- Simulator$new(use_backup = path)
print(object_restored)
\dontrun{all.equal(object, object_restored)}

# 8. Run more simulations and pending simulations (if any):
object_restored$go(runs = 2)
}
\seealso{
Other simulation helpers: 
\code{\link{correlated_regressors}()},
\code{\link{ddirichlet_cpp}()},
\code{\link{dmixnorm_cpp}()},
\code{\link{dmvnorm_cpp}()},
\code{\link{dtnorm_cpp}()},
\code{\link{dwishart_cpp}()},
\code{\link{gaussian_tv}()},
\code{\link{simulate_markov_chain}()}
}
\concept{simulation helpers}
\keyword{simulation}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{results}}{[\code{tibble}, read-only]\cr
The simulation results.}

\item{\code{cases}}{[\code{tibble}, read-only]\cr
The simulation cases.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Simulator-new}{\code{Simulator$new()}}
\item \href{#method-Simulator-print}{\code{Simulator$print()}}
\item \href{#method-Simulator-define}{\code{Simulator$define()}}
\item \href{#method-Simulator-go}{\code{Simulator$go()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Simulator-new"></a>}}
\if{latex}{\out{\hypertarget{method-Simulator-new}{}}}
\subsection{Method \code{new()}}{
Initialize a \code{Simulator} object, either a new one or from backup.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Simulator$new(
  use_backup = NULL,
  verbose = getOption("verbose", default = FALSE)
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{use_backup}}{[\code{NULL} | \code{character(1)}]\cr
Optionally a path to a backup folder previously used in \verb{$go()}.}

\item{\code{verbose}}{[\code{logical(1)}]\cr
Provide info? Does not include progress updates. For that, see details.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Simulator-print"></a>}}
\if{latex}{\out{\hypertarget{method-Simulator-print}{}}}
\subsection{Method \code{print()}}{
Print method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Simulator$print()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Simulator-define"></a>}}
\if{latex}{\out{\hypertarget{method-Simulator-define}{}}}
\subsection{Method \code{define()}}{
Define function and arguments for a new \code{Simulator} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Simulator$define(f, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{f}}{[\code{function}]\cr
A \code{function} to evaluate.}

\item{\code{...}}{Arguments for \code{f}. Each value must be
\enumerate{
\item named after an argument of \code{f}, and
\item a \code{list}, where each element is a variant of that argument for \code{f}.
}}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Simulator-go"></a>}}
\if{latex}{\out{\hypertarget{method-Simulator-go}{}}}
\subsection{Method \code{go()}}{
Run simulations.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Simulator$go(
  runs = 0,
  backup = FALSE,
  path = paste0("backup_", format(Sys.time(), "\%Y-\%m-\%d-\%H-\%M-\%S"))
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{runs}}{[\code{integer(1)}]\cr
The number of (additional) simulation runs.

If \code{runs = 0}, only pending cases (if any) are solved.}

\item{\code{backup}}{[\code{logical(1)}]\cr
Create a backup under \code{path}?}

\item{\code{path}}{[\code{character(1)}]\cr
Only relevant, if \code{backup = TRUE}.

In this case, a path for a new folder, which does not yet exist and
allows reading and writing.}
}
\if{html}{\out{</div>}}
}
}
}
