#' City Record Online
#'
#' Downloads City Record Online data from NYC Open Data.
#'
#' @param limit Number of rows to retrieve (default = 10,000).
#' @param filters Optional list of field-value pairs to filter results.
#' @param timeout_sec Request timeout in seconds (default = 30).
#' @return A tibble containing City Record Online data.
#'
#' @details
#' The City Record Online (CROL) is a fully searchable database of notices
#' published in the City Record newspaper, including, but not limited to:
#' public hearings and meetings, public auctions and sales, solicitations and
#' awards and official rules proposed and adopted by city agencies.
#'
#' @source NYC Open Data: <https://data.cityofnewyork.us/City-Government/City-Record-Online/dg92-zbpx/about_data>
#'
#' @examples
#' # Examples that hit the live NYC Open Data API are wrapped so CRAN checks
#' # do not fail when the network is unavailable or slow.
#' \donttest{
#' if (curl::has_internet()) {
#'   # Quick example (fetch 10 rows)
#'   small_sample <- nyc_city_record(limit = 10)
#'   head(small_sample)
#'
#'   nyc_city_record(limit = 5000)
#'   nyc_city_record(filters = list(short_title = "APPOINTED"))
#' }
#' }
#' @export
nyc_city_record <- function(limit = 10000, filters = list(), timeout_sec = 30) {
  endpoint <- "https://data.cityofnewyork.us/resource/dg92-zbpx.json"

  query_list <- list(
    "$limit" = limit,
    "$order" = "start_date DESC"
  )

  if (length(filters) > 0) {
    where_clauses <- paste0(names(filters), " = '", unlist(filters), "'")
    query_list[["$where"]] <- paste(where_clauses, collapse = " AND ")
  }

  data <- .nyc_get_json(endpoint, query_list, timeout_sec = timeout_sec)
  tibble::as_tibble(data)
}
