% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/npcs.R
\name{npcs}
\alias{npcs}
\title{Fit a multi-class Neyman-Pearson classifier with error controls via cost-sensitive learning.}
\usage{
npcs(
  x,
  y,
  algorithm = c("CX", "ER"),
  classifier,
  seed = 1,
  w,
  alpha,
  trControl = list(),
  tuneGrid = list(),
  split.ratio = 0.5,
  split.mode = c("by-class", "merged"),
  tol = 1e-06,
  refit = TRUE,
  protect = TRUE,
  opt.alg = c("Hooke-Jeeves", "Nelder-Mead")
)
}
\arguments{
\item{x}{the predictor matrix of training data, where each row and column represents an observation and predictor, respectively.}

\item{y}{the response vector of training data. Must be integers from 1 to K for some K >= 2. Can be either a numerical or factor vector.}

\item{algorithm}{the NPMC algorithm to use. String only. Can be either "CX" or "ER", which implements NPMC-CX or NPMC-ER in Tian, Y. & Feng, Y. (2021).}

\item{classifier}{which model to use for estimating the posterior distribution P(Y|X = x). String only.}

\item{seed}{random seed}

\item{w}{the weights in objective function. Should be a vector of length K, where K is the number of classes.}

\item{alpha}{the levels we want to control for error rates of each class. Should be a vector of length K, where K is the number of classes. Use NA if if no error control is imposed for specific classes.}

\item{trControl}{list; resampling method}

\item{tuneGrid}{list; for hyperparameters tuning or setting}

\item{split.ratio}{the proportion of data to be used in searching lambda (cost parameters). Should be between 0 and 1. Default = 0.5. Only useful when \code{algorithm} = "ER".}

\item{split.mode}{two different modes to split the data for NPMC-ER. String only. Can be either "per-class" or "merged". Default = "per-class". Only useful when \code{algorithm} = "ER".
\itemize{
\item per-class: split the data by class.
\item merged: split the data as a whole.
}}

\item{tol}{the convergence tolerance. Default = 1e-06. Used in the lambda-searching step. The optimization is terminated when the step length of the main loop becomes smaller than \code{tol}. See pages of \code{\link[dfoptim]{hjkb}} and \code{\link[dfoptim]{nmkb}} for more details.}

\item{refit}{whether to refit the classifier using all data after finding lambda or not. Boolean value. Default = TRUE. Only useful when \code{algorithm} = "ER".}

\item{protect}{whether to threshold the close-zero lambda or not. Boolean value. Default = TRUE. This parameter is set to avoid extreme cases that some lambdas are set equal to zero due to computation accuracy limit. When \code{protect} = TRUE, all lambdas smaller than 1e-03 will be set equal to 1e-03.}

\item{opt.alg}{optimization method to use when searching lambdas. String only. Can be either "Hooke-Jeeves" or "Nelder-Mead". Default = "Hooke-Jeeves".}
}
\value{
An object with S3 class \code{"npcs"}.
\item{lambda}{the estimated lambda vector, which consists of Lagrangian multipliers. It is related to the cost. See Section 2 of Tian, Y. & Feng, Y. (2021) for details.}
\item{fit}{the fitted classifier.}
\item{classifier}{which classifier to use for estimating the posterior distribution P(Y|X = x).}
\item{algorithm}{the NPMC algorithm to use.}
\item{alpha}{the levels we want to control for error rates of each class.}
\item{w}{the weights in objective function.}
\item{pik}{the estimated marginal probability for each class.}
}
\description{
Fit a multi-class Neyman-Pearson classifier with error controls via cost-sensitive learning. This function implements two algorithms proposed in Tian, Y. & Feng, Y. (2021). The problem is minimize a linear combination of P(hat(Y)(X) != k| Y=k) for some classes k while controlling P(hat(Y)(X) != k| Y=k) for some classes k. See Tian, Y. & Feng, Y. (2021) for more details.
}
\examples{
# data generation: case 1 in Tian, Y., & Feng, Y. (2021) with n = 1000
set.seed(123, kind = "L'Ecuyer-CMRG")
train.set <- generate_data(n = 1000, model.no = 1)
x <- train.set$x
y <- train.set$y

test.set <- generate_data(n = 1000, model.no = 1)
x.test <- test.set$x
y.test <- test.set$y

# contruct the multi-class NP problem: case 1 in Tian, Y., & Feng, Y. (2021)
alpha <- c(0.05, NA, 0.01)
w <- c(0, 1, 0)
\donttest{
# try NPMC-CX, NPMC-ER, and vanilla multinomial logistic regression
fit.vanilla <- nnet::multinom(y~., data = data.frame(x = x, y = factor(y)), trace = FALSE)
fit.npmc.CX <- try(npcs(x, y, algorithm = "CX", classifier = "multinom", 
w = w, alpha = alpha))
fit.npmc.ER <- try(npcs(x, y, algorithm = "ER", classifier = "multinom", 
w = w, alpha = alpha, refit = TRUE))
# test error of vanilla multinomial logistic regression
y.pred.vanilla <- predict(fit.vanilla, newdata = data.frame(x = x.test))
error_rate(y.pred.vanilla, y.test)
# test error of NPMC-CX
y.pred.CX <- predict(fit.npmc.CX, x.test)
error_rate(y.pred.CX, y.test)
# test error of NPMC-ER
y.pred.ER <- predict(fit.npmc.ER, x.test)
error_rate(y.pred.ER, y.test)
}
}
\references{
Tian, Y., & Feng, Y. (2021). Neyman-Pearson Multi-class Classification via Cost-sensitive Learning. Submitted. Available soon on arXiv.
}
\seealso{
\code{\link{predict.npcs}}, \code{\link{error_rate}}, \code{\link{generate_data}}, \code{\link{gamma_smote}}.
}
