#' Draws from a Dirichlet distribution and shows the clusters
#' that were generated by this draw. Varying alpha, will
#' put more or less mass in the first clusters compared to
#' higher clusters (rhos).
#'
#' @param a Parameter that will be passed in to a Gamma distribution
#' in order to draw from the Dirichlet distribution.
#' @param K Number of clusters to draw
#' @return No return value
#' @examples
#' generate_dirichlet_clusters(10, 10)
#' generate_dirichlet_clusters(0.5, 30)
#' @export
generate_dirichlet_clusters <- function(a, K) {
    dev.new(width = 8, height = 3, unit = "in")
    rho <- rdirichlet(1, rep(a, K))
    rmatrix <- matrix(as.numeric(rho), ncol = 1)
    bar_colors <- rep("grey", K)

    barplot(
        rmatrix,
        horiz = TRUE,
        xlim = c(0, 1),
        col = bar_colors,
        beside = FALSE,
        main = bquote(rho~"~Dirichlet"~"("~.(a)~",...,"~.(a)~")K="~.(K))
    )
}


#' Draws from a Dirichlet distribution and shows the clusters
#' that were generated by this draw. Additionally, adds points to
#' these clusters and shows which clusters are occupied
#'
#' Each point is generated one at a time, need to hit enter to generate
#' a new point. Typing "x" will stop the clustering and the function
#' will return. 
#' @param n Number of points to be drawn in the clusters
#' @param a Parameter that will be passed in to a Gamma distribution
#' in order to draw from the Dirichlet distribution.
#' @param K Number of clusters to draw
#' @return No return value
#' @examples
#' generate_dirichlet_clusters_with_sampled_points(15, 0.5, 20)
#' @export
generate_dirichlet_clusters_with_sampled_points <- function(n, a, K) {
    dev.new(width = 8, height = 3, unit = "in")
    rho <- rdirichlet(1, rep(a, K))
    rmatrix <- matrix(as.numeric(rho), ncol = 1)
    bar_colors <- rep("grey", K)
    rho_cum_sum <- c(0, cumsum(rho))

    print("Newline to sample one point at a time")
    for (i in 1:n) {
        u <- runif(n = 1)
        index <- max(which(rho_cum_sum < u))
        bar_colors[index] <- "blue"

        barplot(
            rmatrix,
            horiz = TRUE,
            xlim = c(0, 1),
            col = bar_colors,
            beside = FALSE,
            main = bquote(
                rho~"~Dirichlet"~"("~.(a)~",...,"~.(a)~") K="~.(K)~", N="~.(i)
            )
        )

        points(u, 1.4, pch = 25, col = "red", bg = "red", cex = 3.5)
        line <- readline()
        if (line == "x") {
            return("done")
        }
    }

}