
#' Plot NMR spectra
#'
#' @description Plot NMR spectra, with line-brackets denoting binned regions.
#' Uses spectra data processed in MestreNova or TopSpin.
#' @param dat Processed spectral data, output from (a) \code{\link{nmr_import_spectra}}
#' and \code{\link{nmr_assign_bins}}; or (b) \code{\link{nmr_import_peaks}}
#' @param binset A binset; e.g. \code{\link{bins_Clemente2012}},
#' \code{\link{bins_Hertkorn2013}}, etc., or a similarly-structured data frame
#' @param label_position y-axis position for bin labels
#' @param mapping An aesthetic mapping generated by \code{\link[ggplot2]{aes}}:
#' \code{aes(x = ..., y = ...)}
#' @param stagger How much to stagger the labels, numeric;
#' same units as \code{label_position}
#'
#' @return A \code{\link[ggplot2]{ggplot}} object.
#'
#' @import ggplot2
#' @importFrom utils head
#' @export
#' @author Kaizad Patel
#' @examples
#' sdir <- system.file("extdata", "kfp_hysteresis", "spectra_mnova", package = "nmrrr")
#' spec <- nmr_import_spectra(path = sdir, method = "mnova")
#' library(ggplot2)
#' p_aes <- aes(x = ppm, y = intensity)
#' p <- nmr_plot_spectra(spec, bins_Clemente2012, 5, p_aes, stagger = 0.5)
#' p + ylim(0, 6)
#'
nmr_plot_spectra <- function(dat, binset, label_position = 100, mapping = aes(x = ppm, y = intensity), stagger = 10) {
  if (inherits(class(mapping), "uneval")) {
    stop("'mapping' must be a ggplot2::aes() output")
  }

  # Quiet R CMD CHECK notes
  start <- number <- sampleID <- newsource <- NULL

  # create spectra-base plot ----
  odds <- binset[seq(1, nrow(binset), by = 2), ]
  evens <- binset[seq(2, nrow(binset), by = 2), ]

  label_stagger <- label_position / 50
  spectra_base <-
    ggplot() +
    # stagger bracketing lines for odd vs. even rows
    geom_segment(
      data = evens,
      aes(x = start, xend = stop, y = label_position, yend = label_position),
      color = "black"
    ) +
    geom_segment(
      data = odds,
      aes(x = start, xend = stop, y = label_position - (2 * label_stagger), yend = label_position - (2 * label_stagger)),
      color = "black"
    ) +
    # stagger numbering like the lines
    geom_text(
      data = evens,
      aes(x = (start + stop) / 2, y = label_position + label_stagger, label = number)
    ) +
    geom_text(
      data = odds,
      aes(x = (start + stop) / 2, y = label_position - label_stagger, label = number)
    ) +
    scale_x_reverse() +
    xlab("shift, ppm") +
    ylab("intensity")

  # add staggering factor ----

  stagger_factor <- 1 / stagger
  dat_y_stagger <- weak_tibble(
    sampleID = unique(dat$sampleID),
    y_factor = (seq_along(sampleID) - 1) / stagger_factor
  )

  spectra_new <- merge(dat, dat_y_stagger, by = "sampleID")
  spectra_new$intensity <- spectra_new$intensity + spectra_new$y_factor

  # combined plot ----

  spectra_base +
    geom_path(data = spectra_new, mapping)
}
