\name{mvout}
\alias{mvout}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Robust Multivariate Outlier Detection
}
\description{
Detection of multivariate outliers using robust estimates of location and scale.
}
\usage{
mvout(x, method = c("none", "princomp", "factanal"), standardize = TRUE,
      robust = TRUE, direction = rep("two.sided", ncol(x)), thresh = 0.01, 
      keepx = TRUE, factors = 2, scores = c("regression", "Bartlett"), 
      rotation = c("none", "varimax", "promax"), ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{
  Data matrix (n x p)
  }
  \item{method}{
  Character specifying the factorization method used to define the covariance matrix: "none" uses the unfactorized (robust) covariance matrix, "princomp" uses the (robust) principal components analysis (PCA) implied covariance matrix, and "factanal" uses the (robust) factor analysis (FA) implied covariance matrix.
  }
  \item{standardize}{
  Logical specifying whether to apply PCA to the correlation (default) or covariance matrix. Ignored if \code{method = "none"} or \code{method = "factanal"}.
  }
  \item{robust}{
  If \code{TRUE} (default), robust estimates of the mean vector and covariance matrix are obtained using the \code{\link[robustbase]{covMcd}} function. Otherwise standard estimators are obtained using the \code{\link{colMeans}} and \code{\link{cov}} functions.
  }
  \item{direction}{
  Direction defining "outlier" for each variable (character). Three options are available: "two.sided" considers large postive and negative deviations from the mean as outliers, "less" only considers large negative deviations as outliers, and "greater" only considers large positve deviations as outliers. Accepts a single character giving the common direction for each variable, or a character vector of length p.
  }
  \item{thresh}{
  Scalar specifying the threshold for flagging outliers (0 < thresh < 1). See Note.
  }
  \item{keepx}{
  Logical indicating if input \code{x} should be saved and returned as part of the output. 
  }
  \item{factors}{
  Integer giving the number of factors for PCA or FA model. Ignored if \code{method = "none"}.
  }
  \item{scores}{
  Method used to compute factor scores (only used if \code{method = "factanal"}).
  }
  \item{rotation}{
  Factor rotation method aapplied to PCA or FA loadings. Ignored if \code{method = "none"}.
  }
  \item{...}{
  Additional arguments passed to the \code{\link[robustbase]{covMcd}} function, e.g., \code{alpha}, \code{nsamp}, etc. Note that the \code{cor} argument should not be used, as this is controlled by the \code{standardize} argument.
  }
}
\details{
Outliers are determined using a (squared) Mahalanobis distance calculated using either the Minimum Covariance Determinant (MCD) estimator for the mean vector and covariance matrix (default) or the standard unbiased sample estimators. The MCD is computed using the \code{\link[robustbase]{covMcd}} function. Includes options for specifying the direction of interest for outlier detection, as well as options for using bilinear models (PCA and FA) to define the covariance matrix used for the Mahalanobis distance.
}
\value{
An object of class \code{mvout} which is a list with the following components:
\item{distance }{Numeric vector of (squared) Mahalanobis distances for the n observations.}
\item{outlier }{Logical vector indicating whether or not each of the n observations is an outlier.}
\item{mcd }{Object of class \code{mcd} that is output from the \code{\link[robustbase]{covMcd}} function.}
\item{args }{List of input arguments (e.g., x, method, standardize, etc.)}
\item{scores }{Factor or principal component scores (will be \code{NULL} if \code{method = "none"}).}
\item{loadings }{Factor or principal component loadings (will be \code{NULL} if \code{method = "none"}).}
\item{uniquenesses }{Variables uniquenesses (will be \code{NULL} if \code{method = "none"}).}
\item{invrot }{Inverse of the matrix that was used to rotate the loadings (will be \code{NULL} if \code{method = "none"}).}
\item{cormat }{Factor or principal component score correlation matrix (will be \code{NULL} if \code{method = "none"}).}
}
\references{
%Delgado, J. E., Elison, J. T., & Helwig, N. E. (2025). Robust Detection of Signed Outliers in Multivariate Data With Applications to
%Early Identification of Risk for Autism. Psychological Methods. (conditionally accepted)

Todorov, V., & Filzmoser, F. (2009). An Object-Oriented Framework for Robust Multivariate Analysis. Journal of Statistical Software, 32(3), 1-47.
}
\author{
Jesus E. Delgado <delga220@umn.edu>
Nathaniel E. Helwig <helwig@umn.edu>
}
\note{
For observations included in the (robust) covariance calculation, the critical value that designates an observation as an outlier is defined as \code{qchisq(1 - thresh, df = p)}. 

For the excluded observations, the critical value is defined as \code{qf(1 - thresh, df1 = p, df2 = n - p) * ((n - 1) * p / (n - p))}.
}

\section{Warning }{
The default behavior of the \code{\link[robustbase]{covMcd}} function (and, consequently, the \code{\link{mvout}} function) is for the MCD estimator to be computed from a random sample of 500 observations. The \code{nsamp} argument of the \code{\link[robustbase]{covMcd}} function can be used to control the number of samples or request a different method (e.g., nsamp = "deterministic").
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{predict.mvout}} for obtaining predictions from \code{mvout} objects.
}
\examples{
# generate some data
n <- 200
p <- 2
set.seed(0)
x <- matrix(rnorm(n * p), n, p)

# thresh = 0.01
set.seed(1)    # for reproducible MCD estimate
out1 <- mvout(x)
plot(out1)

# thresh = 0.05
set.seed(1)    # for reproducible MCD estimate
out5 <- mvout(x, thresh = 0.05)
plot(out5)

# direction = "greater"
set.seed(1)    # for reproducible MCD estimate
out <- mvout(x, direction = "greater", thresh = 0.05)
plot(out)

# direction = c("greater", "less")
set.seed(1)    # for reproducible MCD estimate
out <- mvout(x, direction = c("greater", "less"), thresh = 0.05)
plot(out)

}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory (show via RShowDoc("KEYWORDS")):
\keyword{ multivariate }
\keyword{ robust }
% Use only one keyword per line.
% For non-standard keywords, use \concept instead of \keyword:
% \concept{ ~cpt1 }
% \concept{ ~cpt2 }
% Use only one concept per line.
