% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mland.R
\name{mland}
\alias{mland}
\title{Generates object of class 'MultiLand'}
\usage{
mland(
  points_layer,
  rast_layer = NULL,
  radii,
  class_names = NULL,
  site_ref = NULL,
  bufftype = "round",
  segs = 20,
  ext_rast_layer = NULL,
  rast_names = NULL,
  on_the_fly = FALSE,
  progress = TRUE
)
}
\arguments{
\item{points_layer}{An object of class 'SpatVector', 'SpatialPoints', 'SpatialPointsDataFrame' or 'sf', or a string with the
path to a vector file.}

\item{rast_layer, ext_rast_layer}{An object of class 'SpatRaster', 'RasterLayer', 'RasterStack', 'RasterBrick',
or a list of raster objects (any of 'RasterLayer' or 'SpatRaster').}

\item{radii}{A numeric vector with the radii (in meters) from which buffers will be created.}

\item{class_names}{A list matching each raster value with a class name. See Details.}

\item{site_ref}{A string with the name of the column containing the identity of the sites in
points layer data (argument \code{points_layer}). See Details.}

\item{bufftype}{Type of buffer to be created: "round" for circular buffers (default) or "square".}

\item{segs}{Number of line segments to use to approximate a quarter circle during buffer generation. Only valid when
\code{bufftype = "round"}. Default is 20.}

\item{rast_names}{A character vector with the names of the raster layers provided in \code{rast_layer}
and \code{ext_rast_layer}. See Details.}

\item{on_the_fly}{Logical. If FALSE (default) intersections between buffers and raster layers will
be calculated. If TRUE, only buffers will be generated. See Details.}

\item{progress}{Logical. If TRUE (default), progress of the analysis will be printed.}
}
\value{
An object of class 'MultiLand'. This object can be used to generate useful plots with
\code{\link[=mland_plot]{mland_plot()}}, calculate metrics with \code{\link[=mland_metrics]{mland_metrics()}} and calculate buffer's overlapping with
\code{\link[=mland_overlap]{mland_overlap()}}. See ?MultiLand for more details on the content of this object.
}
\description{
Creates an object of class 'MultiLand', which is the
main object to be used by other functions
of the package to generate plots, calculate landscape metrics and perform other relevant
analyses.
}
\details{
\code{mland()} is the primary function of the package. It creates an object of class
'MultiLand' that holds relevant objects and information about points, buffers and intersections
between buffers and raster layers.

The function firstly creates buffers with center in the sites defined in \code{points_layer}, and size defined by
the values of \code{radii}. If each point defined in \code{points_layer} has an associated name or id for
ulterior identification, the user should provide the name of the attribute inside \code{points_layer}
containing this information, by passing this string through the argument \code{site_ref}.

Argument \code{rast_layer} must be provided with raster layers with discrete values from which different landscape metrics (provided by package \code{landscapemetrics})
could be calculated. Extra raster layers can be provided in \code{ext_rast_layer}, from which other metrics can be
calculated. For instance, an extra raster layer could be one depicting continuous values of slope,
from which a mean value per landscape may be calculated. Raster layers should be in a coordinate reference
system with meters as the unit, and all raster and vector layers must be in the same coordinate reference system.
The user may check the validity of the raster layer with \code{\link[=check_raster]{check_raster()}}.

The extent of the provided points layer should not exceed the limits of the extent of the provided raster layers. In addition,
the difference between the outer borders of provided points layer and raster layers should not be less than the maximum
provided radius in \code{radii}. The purpose is to avoid generating total or partially "empty landscapes" due to the existence of
non-overlapping regions between the buffers (generated around each point given a provided radius) and the raster layers.
If any of this happens, a warning will be returned.

If \code{on_the_fly = FALSE} (default), intersections between buffers and raster layers
defined in \code{rast_layer} and/or \code{ext_rast_layer} will be generated. Otherwise, if \code{on_the_fly = TRUE}, only buffers will be generated. The
latter approach may be particularly useful for 'MultiLand' objects with numerous points
(hundreds or thousands), in order to avoid returning an object excessively heavy for memory. If
this is the case, intersections between buffers and raster layers will be generated when
required ("on the fly"). For instance, to calculate metrics with \code{\link[=mland_metrics]{mland_metrics()}}.

The names of the provided raster layers can be defined in \code{rast_names}. If so, this must be
a character vector with as many names (strings) as provided raster layers in arguments \code{rast_layer} and
\code{ext_rast_layer}, in the mentioned order. If there is no need of a name for a particular raster layer, the given element in the vector
should be \code{NA}. Definition of these names could be useful when applying other functions of
the package to the object generated here. For instance, to get the name of the raster layer
of a particular row of the data.frame with landscape metrics exported by \code{\link[=mland_metrics]{mland_metrics()}}.

Classes names can be associated with each value of the raster layers defined in \code{rast_layer}, for a
easier future identification. If so, the user must provide a list with as many elements as
raster layers defined in \code{rast_layer}. If a 'SpatRaster' with multiple layers, a 'RasterStack' or a
'RasterBrick' is provided, the
number of raster layers are extracted from these objects. Each element of the list must be a
vector built from concatenated pairs of values, with the value of the raster (the class) in the
first place and the associated class name in the second place. For example, in the case only
one rasterlayer is provided (with four unique values: 1, 2, 3 and 4), a plausible definition
for the argument \code{class_names} could be the following:

\preformatted{
 list(c(1, "Forest", 2, "Crops", 3, "Urban", 4, "Grassland"))
}

If, for instance, two raster layers are provided (with four unique values for the first layer,
and two unique values for the second one), a plausible definition would be:

\preformatted{
 list(c(1, "Forest", 2, "Crops", 3, "Urban", 4, "Grassland"),
      c(1, "Burnt Areas", 2, "Non-burnt Areas"))
}
}
\examples{
# Loads main raster with land covers
elchaco <- terra::rast(system.file("extdata", "elchaco.tif", package = "multilandr"))

# Main raster should have discrete values (e.g. land covers). This can be
# checked with the function check_raster():

check_raster(elchaco)

# Loads extra raster with NDVI values
elchaco_ndvi <- terra::rast(system.file("extdata", "elchaco_ndvi.tif", package = "multilandr"))

# Classes names
cl_names <- c(1, "Forest",
              2, "Grassland",
              3, "Crops",
              4, "Pastures",
              5, "Water",
              6, "Urban")

# Loads points
elchaco_sites <- terra::vect(system.file("extdata", "elchaco_sites.gpkg", package = "multilandr"))

# Creates 'MultiLand' object by loading main raster, an extra raster and points.
ernesdesign <- mland(points_layer = elchaco_sites,
                     rast_layer = elchaco,
                     radii = seq(1000, 5000, 1000),
                     class_names = list(cl_names),
                     site_ref = "name",
                     ext_rast_layer = elchaco_ndvi,
                     rast_names = c("landcover", "NDVI"),
                     segs = 20)

# Returns basic information about the object
ernesdesign

# Returns the classes of each rasterlayer and its names, if initially provided
ernesdesign@classes

\donttest{
# Loads another main raster. Same classes as "elchaco", but a different year.
elchaco2 <- terra::rast(system.file("extdata", "elchaco2.tif", package = "multilandr"))

# Creates 'MultiLand' with two raster layers.
ernesdesign2 <- mland(points_layer = elchaco_sites,
                      rast_layer = list(elchaco, elchaco2),
                      radii = seq(1000, 5000, 1000),
                      class_names = list(cl_names, cl_names),
                      site_ref = "name")

# Creates the same object but with "on_the_fly = TRUE". Intersections between
# buffers and rasters will not be generated in this step
ernesdesign3 <- mland(points_layer = elchaco_sites,
                      rast_layer = list(elchaco, elchaco2),
                      radii = seq(1000, 5000, 1000),
                      class_names = list(cl_names, cl_names),
                      site_ref = "name",
                      on_the_fly = TRUE)

# Creates a MultiLand object with hundreds of points. In this case, these
# points were generated with generate_points(), another function from this
# package. Also, "on_the_fly = TRUE" assures that no intersections between buffers
# and the raster are created in this step.

# Loads points
otf_sites <- terra::vect(system.file("extdata", "otf_sites.gpkg", package = "multilandr"))

# Creates MultiLand object
otf_design <- mland(points_layer = otf_sites,
                    rast_layer = elchaco,
                    radii = 2000,
                    class_names = list(c(1, "Forest",
                                         2, "Grassland",
                                         3, "Crops",
                                         4, "Pastures",
                                         5, "Water",
                                         6, "Urban")),
                    on_the_fly = TRUE)
}
}
\seealso{
\code{\link[=mland_plot]{mland_plot()}}, \code{\link[=mland_metrics]{mland_metrics()}}, \code{\link[=mland_overlap]{mland_overlap()}}, \code{\link[=generate_points]{generate_points()}}
}
