% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_mme.data.frame.R
\name{calculate_mme.data.frame}
\alias{calculate_mme.data.frame}
\alias{calculate_mme.tbl_df}
\title{Calculate morphine milligram equivalents (MME) with a \code{data.frame} or tibble}
\usage{
\method{calculate_mme}{data.frame}(
  x,
  id_col = "patient_id",
  medication_col = "medication_name",
  dose_col = "dose",
  doses_per_day_col = "doses_per_24_hours",
  days_col = "days_of_medication",
  therapy_days_col = "therapy_days",
  observation_days_col = "observation_window_days",
  therapy_days_without_col = NULL,
  observation_days_without_col = NULL,
  use_api = FALSE,
  ...
)

\method{calculate_mme}{tbl_df}(
  x,
  id_col = "patient_id",
  medication_col = "medication_name",
  dose_col = "dose",
  doses_per_day_col = "doses_per_24_hours",
  days_col = "days_of_medication",
  therapy_days_col = "therapy_days",
  observation_days_col = "observation_window_days",
  therapy_days_without_col = NULL,
  observation_days_without_col = NULL,
  use_api = FALSE,
  ...
)
}
\arguments{
\item{x}{(\code{data.frame} or \code{tbl_df})\cr
Object with input data - either a \code{data.frame} or tibble with data in
long format, with one row per medication per patient or participant
(\code{id_col}) and including all necessary data for MME calculations
(see \link{opioid_trial} data) and/or other function arguments}

\item{id_col}{(\code{charcter})\cr
Name of the column containing patient identifier; default is
\code{"patient_id"}}

\item{medication_col}{(\code{charcter})\cr
Name of the column containing medication names; default is
\code{"medication_name"}}

\item{dose_col}{(\code{charcter})\cr
Name of the column containing dose values; default is \code{"dose"}}

\item{doses_per_day_col}{(\code{charcter})\cr
Name of the column containing doses per 24 hours; default is
\code{"doses_per_24_hours"}}

\item{days_col}{(\code{charcter})\cr
Name of the column containing days of medication; default is
\code{"days_of_medication"}}

\item{therapy_days_col}{(\code{charcter})\cr
Name of the column containing therapy days with buprenorphine (up to one
unique value per patient); default is \code{"therapy_days"}}

\item{observation_days_col}{(\code{charcter})\cr
Name of the column containing observation window days with buprenorphine
(up to one unique value per patient); default is
\code{"observation_window_days"}}

\item{therapy_days_without_col}{(\code{charcter})\cr
Name of the column containing therapy days without buprenorphine (up to
one unique value per patient). If \code{NULL} (default), uses the value from
\code{therapy_days_col}.}

\item{observation_days_without_col}{(\code{charcter})\cr
Name of the column containing observation window days without
buprenorphine (up to one unique value per patient). If \code{NULL} (default),
uses the value from \code{observation_days_col}.}

\item{use_api}{(\code{logical})\cr
Indicates whether to use the NIH HEAL Online MME Calculator API to
perform calculations or perform them locally instead. For
\code{calculate_mme.data.frame()} and \code{calculate_mme.tbl_df()}, the default is
\code{FALSE}, as the functions assume the user needs to perform the MME
calculations without being restricted by the API rate limit of 50
patient-level calculations per 15 minutes. This also allows the user to
perform the calculations without relying on internet access.}

\item{...}{These dots are for future extensions and must be empty.}
}
\value{
A list containing three \code{data.frame} elements:
\itemize{
\item \code{medications}: The original data with added prescription-level MME columns
\item \code{patient_summary_with_buprenorphine}: Patient-level MME summary including
buprenorphine
\item \code{patient_summary_without_buprenorphine}: Patient-level MME summary
excluding buprenorphine
}
}
\description{
Calculates the single-day MME and total MME for each individual prescription
opioid medication submitted for calculation. Also calculates total MME,
total days of supply, and four distinct Total MME/Day calculations from
the NIH HEAL Online MME Calculator across all prescription medications
for two different medication groupings: 1) opioids without buprenorphine
and 2) opioids with buprenorphine.
}
\details{
The function will provide the same results regardless of whether the user has
specified they want calculation done using the API (\code{use_api}). Specifying
\code{use_api == FALSE} helps overcome the online calculator API rate limit of 50
(patient-level) requests per 15  minutes. In addition to returning
user-specified arguments, \code{calculate_mme()} also returns several  other
variables mentioned in the \strong{Description} section. Output variable
description details are below; see
\href{https://www.doi.org/10.1097/j.pain.0000000000003529}{Adams, \emph{et al}. (2025)}
for a comprehensive overview.
}
\section{Prescription-Level}{
\strong{Conversion Factor for <\code{medication_name}>} (\code{factor}): the conversion
factor used for calculating total MME/day.

\strong{MME for <\code{medication_name}>} (\code{mme}): Morphine milligram equivalent
for the whole prescription specified in \code{medication_name}, calculated as
\code{(dose) * (doses_per_24_hours) * (factor) * (days_of_medication)}.

\strong{24h MME for <\code{medication_name}>} (\code{single_day_mme}): Morphine milligram
equivalent for the prescription specified in \code{medication_name} for a
single day, calculated as \code{(dose) * (doses_per_24_hours) * (factor)}.

\strong{One day}: Typically, the day with highest opioid exposure is entered, and
the sum of 24-hour MME across the drugs that apply to this day is
calculated. Highest MME in one day is definition 4.
}

\section{Summary-Level:}{
\strong{On-therapy Days} (\code{therapy_days}): The sum of prescription duration
(\code{days_of_medication}) for each medication, but \emph{with each calendar day
counted only ONCE}. User-supplied; this is the denominator for MME/Day
definition 2.
\itemize{
\item If there is only one prescription, or if there is no calendar overlap
(no days on which more than one prescription is active), this will be
the same as the total days supply.
\item If there are overlapping prescriptions, this is the number of unique
calendar days.
}

\strong{Total MME} (\code{total_mme}): The MME for each medication, summed across all
prescriptions. This is the numerator for MME/Day definitions 1, 2, and 3.

\strong{Total Days Supply} (\code{total_days}): The sum of the entered prescription
duration (\code{days_of_medication}) for each of the \code{medications} (Med 1
duration + med 2 duration...). Automatically calculated. This is the
denominator for MME/Day definition 1.
\subsection{MME/Day}{

MME/Day is an aggregate measure, calculating the total MME divided by a
specified time window (a number of days). The MME/Day definitions specify
the number of days:

\strong{MME/Day Definition 1 (\code{mme1}): Total Days Supply}

MME Definition 1 = Total MME / Total Days Supply time window (sum of entered
prescription durations).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme1 = total_mme / total_days
}\if{html}{\out{</div>}}
\itemize{
\item Note that the same calendar day may contribute multiple times, if
overlapping prescriptions.
\item Reason to select this definition: This is the least complicated
calculation; appears best suited when immediate-release opioids are
prescribed for short discrete times.
\item Identified challenge with this definition: It consistently underestimated
MME per day when overlapping prescriptions were present or when
immediate-release and extended release opioids were prescribed
concurrently.
}

\strong{MME/Day Definition 2 (\code{mme2}): On-therapy Days}

MME Definition 2 = Total MME / On-therapy Days time window (sum of entered
prescription durations except each calendar day is counted only ONCE).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme2 = total_mme / therapy_days
}\if{html}{\out{</div>}}
\itemize{
\item Note - On-therapy Days unique calendar days.
\item Reason to select this definition: Provides a smoothed measure useful in
studies of dose-dependent adverse effects, including opioid-induced
constipation or overdose in patients with opioid tolerance or who have
been stable on opioids.
\item Identified challenge with this definition: The metric is time-varying and
affords the greatest flexibility to define medication gap periods and
leftover/unused medications to improve pharmacoepidemiologic studies.
}

\strong{MME/Day Definition 3 (\code{mme3}): Fixed Observation Window}

Uses the Total MME study-specified fixed observation window. MME Definition
3 = Total MME / Number of days in observation window:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme3 = total_mme / observation_window_days
}\if{html}{\out{</div>}}
\itemize{
\item If this definition is selected, it is important to report on the duration
of the fixed window.
\item Reason to select this definition: Most suitable for studies with a known
or suspected duration of risk during which adverse events are expected
to occur, such as incidence of opioid use disorder. This definition may
be useful when prescriptions are filled at irregular time intervals on a
as needed basis (\emph{pro re nata}, PRN).
\item Identified challenge with this definition: The definition consistently had
the lowest milligrams per day for immediate-release opioids. It is the
most robust to misspecification, amenable to transformations, and has
the least noise when constructing continuous functions. However, since
it assumes uniform exposure/risk within a window, there is less scope
for time-varying adjustment.
\item This is the definition recommended by the Department of Health and Human
Services Office of the Inspector General.
}

\strong{MME/Day Definition 4 (\code{mme4}): Maximum Daily Dose}

Uses the sum of 24-hour MME for the day with highest opioid exposure.

MME Definition 4 = Drug 1 (dose (mg) x # of doses per day) x conversion
factor + Drug 2 (dose (mg) x # of doses per day) x conversion factor + ...

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme4 = sum(dose * doses_per_24_hours * factor)
}\if{html}{\out{</div>}}
\itemize{
\item Report the highest single-day exposure.
\item Reason to select this definition: A toxicological perspective may be
appropriate for patients with no opioid tolerance and in the presence
of comorbidities for respiratory depression. It appears to be best
suited for immediate dose-dependent toxic effects, such as respiratory
depression.
\item Identified challenged with this definition: This definition may have
limited use if it includes opioids where fatal toxicity does not involve
respiratory depression (e.g., tramadol) or have atypical \emph{mu}-opioid
receptor agonism (e.g., tapentadol, buprenorphine).
\item The definition assumes uniform risk of adverse outcomes regardless of time
on-therapy. More so than the others, this definition is prone to
influence from early refills, unused medication, and how the 90 MME
threshold is operationalized.
\item This definition underlies the algorithm embedded in the CDC Opioid
Guideline mobile app. There may be difficulty reconciling findings with
studies using the other definitions because it returns a MME per day
that is significantly higher.
\item This calculator sums the 24-hour MME for every prescription, without
considering calendar dates.
}
}
}

\examples{
library(dplyr)
# Calculate MME using long-format data
# Subset of opioid_trial data used for speedier example
mme <- calculate_mme(
  x = opioid_trial |> dplyr::filter(patient_id \%in\% sprintf("P\%03d", 1:100)),
  therapy_days_without_col = "therapy_days_without",
  observation_days_without_col = "observation_window_days_without"
  )

head(mme$medications)

head(mme$patient_summary_with_buprenorphine)

head(mme$patient_summary_without_buprenorphine)

# Cleanup
rm(mme)

}
\seealso{
\code{\link[=calculate_mme.list]{calculate_mme.list()}}
}
