% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mplus.lca.summa.R
\name{mplus.lca.summa}
\alias{mplus.lca.summa}
\title{Summary Result Tables and Grouped Bar Charts for Latent Class Analysis in Mplus}
\usage{
mplus.lca.summa(folder = getwd(), exclude = NULL, sort.n = TRUE, sort.p = FALSE,
                digits = 0, p.digits = 3, bf.trunc = TRUE, conf.level = 0.95,
                plot = FALSE, group.ind = TRUE, ci = TRUE,
                axis.title = 9, axis.text = 9, levels = NULL, labels = NULL,
                ylim = NULL, ylab = c("Mean Value", "Item Response Probability"),
                breaks = ggplot2::waiver(), errorbar.width = 0.1,
                legend.title = 9, legend.text = 9, legend.key.size = 0.5,
                gray = FALSE, start = 0.15, end = 0.85, dpi = 600,
                width.ind = NULL, width.nclass = NULL, height.categ = NULL,
                height = NA, write = NULL, append = TRUE, check = TRUE,
                output = TRUE)
}
\arguments{
\item{folder}{a character string indicating the path of the folder
containing subfolders with the Mplus output files. By
default Mplus outputs in the subfolders of the current
working directory are read. Note that if there are no
subfolders available, Mplus outputs from the folder
specified in the argument \code{folder} are extracted.}

\item{exclude}{a character vector indicating the name of the subfolders
excluded from the result tables.}

\item{sort.n}{logical: if \code{TRUE} (default), result table is sorted
according to the number of classes within each folder
in increasing order.}

\item{sort.p}{logical: if \code{TRUE}, model-estimated class counts,
proportions, average posterior class probabilities,
and odds of correct classification ratio in the
classification diagnostics are sorted in increasing order.}

\item{digits}{an integer value indicating the number of decimal places
to be used for displaying LL, AIC, CAIC, BIC, SABIC,
AWE, OCC, and approximate Bayes factors (aBF).}

\item{p.digits}{an integer value indicating the number of decimal places
to be used for displaying cmP, \emph{p}-values, relative entropy
values, class proportions, and confidence intervals. Note that the scaling
correction factor is displayed with \code{p.digits} minus 1 decimal places.}

\item{bf.trunc}{logical: if \code{TRUE} (default), approximate Bayes
factors (aBF) greater than 1000 are truncated.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence
level of the intervals in the result table with means
and variances for each latent class separately. Note
that only \code{0.9}, \code{0.95}, or \code{0.95} are
allowed when extracting confidence intervals from Mplus
outputs.}

\item{plot}{logical: if \code{TRUE}, bar charts with error bars for
confidence intervals for LCA with continuous, count
variables are saved in the folder \code{_Plots} within
subfolders.}

\item{group.ind}{logical: if \code{TRUE} (default), latent class indicators
are represented by separate bars when saving plots of
an LCA based on continuous or count indicator variables,
while latent class indicators are represented on the
x-axis when saving plots of an LCA based on ordered or
unordered categorical indicator, if \code{FALSE} latent
classes are represented by separate bars when saving plots
of an LCA based on continuous or count indicator variables,
while latent classes are represented on the x-axis when
saving plots of an LCA based on ordered or unordered
categorical indicator.}

\item{ci}{logical: if \code{TRUE} (default), confidence intervals
are added to the bar charts for LCA with continuous or
count indicator variables.}

\item{axis.title}{a numeric value specifying the size of the axis title.}

\item{axis.text}{a numeric value specifying the size of the axis text}

\item{levels}{a character string specifying the order of the indicator
variables shown on the x-axis.}

\item{labels}{a character string specifying the labels of the indicator
variables shown on the x-axis.}

\item{ylim}{a numeric vector of length two specifying limits of the
y-axis.}

\item{ylab}{a character string specifying the label of the y-axis.}

\item{breaks}{a numeric vector specifying the points at which tick-marks
are drawn at the y-axis.}

\item{errorbar.width}{a numeric vector specifying the width of the error bars.
By default, the width of the error bars is 0.1 plus
number of classes divided by 30.}

\item{legend.title}{a numeric value specifying the size of the legend title.}

\item{legend.text}{a numeric value specifying the size of the legend text.}

\item{legend.key.size}{a numeric value specifying the size of the legend keys.}

\item{gray}{logical: if \code{TRUE}, bar charts are drawn in gray
scale.}

\item{start}{a numeric value between 0 and 1 specifying the gray value
at the low end of the palette.}

\item{end}{a numeric value between 0 and 1 specifying the gray value
at the high end of the palette.}

\item{dpi}{a numeric value specifying the plot resolution when saving
the bar chart.}

\item{width.ind}{a numeric value specifying the width of the plot as a
factor depending on the number of indicator variables.
By default, the factor is 1.5.}

\item{width.nclass}{a numeric value specifying the width of the plot as a
factor depending on the number of classes. By default, the factor is 0.5 when saving plots of an
LCA based on continuous or count indicator variables,
while the factor is 1.5 when saving plots of an LCA
based on ordered or unordered categorical indicator
variables.}

\item{height.categ}{a numeric value specifying the height of the plot as a
factor depending on the number of response categories.
By default, the factor is 0.6. Note that this argument
is used only when saving plots of an LCA based on ordered
or unordered categorical indicator variables.}

\item{height}{a numeric value specifying the height of the plot when
saving the bar chart. Note that this argument is used
only when saving plots of an LCA based on continuous
or count indicator variables.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:

\item{\code{call}}{function call}
\item{\code{type}}{type of analysis}
\item{\code{output}}{list with all Mplus outputs}
\item{\code{args}}{specification of function arguments}
\item{\code{result}}{list with result tables, i.e., \code{summary} for the
                     model summaries, \code{bf} for approximate Bayes factors,
                     \code{classif} classification diagnostics,
                     \code{mean_var} for class-specific means and variances
                     of the indicator variables, \code{prob} for class-specific
                     item response probabilities of the indicator variables
                     and \code{d} for Cohen's d standardized mean difference
                     between latent class \emph{j} and latent class \emph{k}}}

\description{
This function reads all Mplus output files from latent class analysis in
subfolders to create result tables with model summaries (e.g., AIC, CAIC, BIC,
SABIC, AWE and cmP), approximate Bayes factors, classification diagnostics
(e.g., relative Entropy, AvePP, and OCC), class-specific means and variances
or class-specific item response probabilities of
the indicator variables, and Cohen's \emph{d}s to quantify class separation
between latent class \emph{j} and latent class \emph{k}. By default, the
function reads output files in all subfolders of the current working directory
or output files in the current working directory and prints a table with model
summaries on the console. Bar charts including confidence intervals for each
latent class solution can be requested by setting the argument \code{plot} to
\code{TRUE}. Note that result tables with Bayes factors, classification diagnostics,
class-specific means and variances, class-specific item response probabilities,
and Cohen's \emph{d}s will not be printed on the console, but are only available
in the exported Excel file when specifying the \code{write} argument (e.g.,
\code{write = "Results_LCA.xlsx"}).
}
\details{
The Excel file exported by the function for reading Mplus output files from
latent class analysis with continuous or count indicator variables by
specifying the \code{write} argument (e.g., \code{write = "Results_LCA.xlsx"})
contains five sheets.

\strong{(1) Summary: Model Summaries}

\itemize{
   \item{\code{"Folder"}}: Subfolder from which the group of Mplus outputs files
                           were summarized
   \item{\code{"#Class"}}: Number of latent classes, i.e., \code{CLASSES ARE c(#Class)}
   \item{\code{"Conv"}}: Model converged, \code{TRUE} or \code{FALSE}, i.e.,
                         \code{THE MODEL ESTIMATION TERMINATED NORMALLY}
   \item{\code{"#Param"}}: Number of estimated parameters, i.e., \code{Number of Free Parameters}
   \item{\code{"logLik"}}: Log-likelihood of the estimated model, i.e., \code{H0 Value}
   \item{\code{"Scale"}}: Scaling correction factor, i.e., \code{H0 Scaling Correction Factor for},
                          available only when \code{ESTIMATOR IS MLR}
   \item{\code{"LLRep"}}: Best log-likelihood replicated, \code{TRUE} or \code{FALSE},
                          i.e., \code{THE BEST LOGLIKELIHOOD VALUE HAS BEEN REPLICATED}
   \item{\code{"AIC"}}: Akaike information criterion, i.e., \code{Akaike (AIC)}
   \item{\code{"CAIC"}}: Consistent AIC, not reported in the Mplus output, but
                         simply \code{BIC + #Param}
   \item{\code{"BIC"}}: Bayesian information criterion, i.e., \code{Bayesian (BIC)}
   \item{\code{"SABIC"}}: Sample-size adjusted BIC, i.e., \code{Sample-Size Adjusted BIC}
   \item{\code{"AWE"}}: Approximate weight of evidence criterion (Banfield & Raftery, 1993)
   \item{\code{"cmP"}}: Approximate correct model probability (Schwarz, 1978)
                        across estimated models in all Mplus output files in
                        the subfolders to create result tables
   \item{\code{"Chi-Pear"}}: Pearson chi-square test of model fit, i.e., \code{Pearson Chi-Square},
                             available only when indicators are count or ordered categorical
   \item{\code{"Chi-LRT"}}: Likelihood ratio chi-square test of model fit, i.e., \code{Likelihood Ratio Chi-Square},
                            available only when indicators are count or ordered categorical
   \item{\code{"LMR-LRT"}}: Significance value (\emph{p}-value) of the Vuong-Lo-Mendell-Rubin test,
                            i.e., \code{VUONG-LO-MENDELL-RUBIN LIKELIHOOD RATIO TEST},
                            available only when \code{OUTPUT: TECH11}
   \item{\code{"A-LRT"}}: Significance value (\emph{p}-value) of the Adjusted Lo-Mendell-Rubin Test,
                          i.e., \code{LO-MENDELL-RUBIN ADJUSTED LRT TEST},
                          available only when \code{OUTPUT: TECH11}
   \item{\code{"BLRT"}}: Significance value (\emph{p}-value) of the bootstrapped
                         likelihood ratio test, available only when \code{OUTPUT: TECH14}
   \item{\code{"Entropy"}}: Summary of the class probabilities across classes
                            and individuals in the sample, i.e., \code{Entropy}
   \item{\code{"aPPMin"}}: Minimum average posterior class probability (AvePP)
                           for the latent classes
   \item{\code{"OCCMin"}}: Minimum odds of correct classification ratio (OCC)
   \item{\code{"nMin"}}: Minimum class count for the latent classes based on
                         the estimated model
   \item{\code{"pMin"}}: Minimum class proportion for the latent classes based
                         on the estimated model
}

\strong{(2) aBF: Approximate Bayes Factors}

\itemize{
   \item{\code{"A-Folder"}}: Subfolder from which the group of Mplus outputs files
                             for Model A were summarized
   \item{\code{"A-#Class"}}: Number of latent classes for Model A, i.e., \code{CLASSES ARE c(#Class)}
   \item{\code{"A-BIC"}}: Bayesian information criterion for Model A, i.e., \code{Bayesian (BIC)}
   \item{\code{"B-Folder"}}: Subfolder from which the group of Mplus outputs files
                             for Model B were summarized
   \item{\code{"B-#Class"}}: Number of latent classes for Model B, i.e., \code{CLASSES ARE c(#Class)}
   \item{\code{"B-BIC"}}: Bayesian information criterion for Model B, i.e., \code{Bayesian (BIC)}
   \item{\code{"aBF"}}: Approximate Bayes Factor for pairwise comparison of relative fit
                        between Model A and Model B, i.e., ratio of the probability of
                        Model A being correct model to Model B being the correct model
}

\strong{(3) Classif: Classification Diagnostics}

\itemize{
   \item{\code{"Folder"}}: Subfolder from which the group of Mplus outputs files
                           were summarized
   \item{\code{"#Class"}}: Number of latent classes, i.e., \code{CLASSES ARE c(#Class)}.
   \item{\code{"Conv"}}: Model converged, \code{TRUE} or \code{FALSE}, i.e.,
                         \code{THE MODEL ESTIMATION TERMINATED NORMALLY}.
   \item{\code{"#Param"}}: Number of estimated parameters, i.e.,
                           \code{Number of Free Parameters}
   \item{\code{"LLRep"}}: Best log-likelihood replicated, \code{TRUE} or \code{FALSE},
                          i.e., \code{THE BEST LOGLIKELIHOOD VALUE HAS BEEN REPLICATED}
   \item{\code{"n1"}}: Class count for the first latent class based on the estimated model,
                       i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}
   \item{\code{"n2"}}: Class count for the second latent class based on the estimated model,
                       i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}
   \item{\code{"p1"}}: Class proportion of the first class based on the estimated
                       posterior probabilities, i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}
   \item{\code{"p2"}}: Class proportion of the second class based on the estimated
                       posterior probabilities, i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}
   \item{\code{"Entropy"}}: Summary of the class probabilities across classes
                            and individuals in the sample, i.e., \code{Entropy}
   \item{\code{"aPP1"}}: Average posterior class probability (AvePP) of the
                         first latent class for the latent classes
   \item{\code{"aPP2"}}: Average posterior class probability (AvePP) of the
                         second latent class for the latent classes
   \item{\code{"OCC1"}}: Odds of correct classification ratio (OCC) of the
                         first latent class
   \item{\code{"OCC2"}}: Odds of correct classification ratio (OCC) of the
                         second latent class
}

\strong{(4) Mean_Var: Means and Variances for each Latent Class Separately}

\itemize{
   \item{\code{"Folder"}}: Subfolder from which the group of Mplus outputs files
                           were summarized
   \item{\code{"#Class"}}: Number of latent classes, i.e., \code{CLASSES ARE c(#Class)}
   \item{\code{"n"}}: Class counts based on the estimated model,
                      i.e., \code{FINAL CLASS COUNTS AND PROPORTIONS}
   \item{\code{"Param"}}: Parameter, i.e., mean or variance
   \item{\code{"Ind"}}: Latent class indicator variable
   \item{\code{"Est."}}: Parameter estimate.
   \item{\code{"SE"}}: Standard error
   \item{\code{"z"}}: Test statistic
   \item{\code{"pval"}}: Significance value
   \item{\code{"Low"}}: Lower bound of the confidence interval
   \item{\code{"Upp"}}: Upper bound of the confidence interval
}

\strong{(5) d: Cohen's d}

\itemize{
   \item{\code{"Folder"}}: Subfolder from which the group of Mplus outputs files
                           were summarized
   \item{\code{"#Class"}}: Number of latent classes, i.e., \code{CLASSES ARE c(#Class)}
   \item{\code{"Ind"}}: Latent class indicator variable
   \item{\code{"Class.j"}}: Number of classes for model \emph{j}
   \item{\code{"Class.k"}}: Number of classes for model \emph{k}
   \item{\code{"n.j"}}: Latent classes \emph{j}
   \item{\code{"M.j"}}: Class-specific mean of the indicator for the latent class \emph{j}
   \item{\code{"SD.j"}}: Class-specific standard deviation of the indicator for the latent class \emph{j}
   \item{\code{"n.k"}}: Latent classes \emph{k}
   \item{\code{"M.k"}}: Class-specific mean of the indicator for the latent class \emph{k}
   \item{\code{"SD.k"}}: Class-specific standard deviation of the indicator for the latent class \emph{k}
   \item{\code{"d"}}: Cohen's d, Standardized mean difference
}

For more info on fit indices, classification diagnostics, and evaluating class
separation see Masyn (2013) and Sorgente et al. (2025).
}
\examples{
\dontrun{

# Load data set "HolzingerSwineford1939" in the lavaan package
data("HolzingerSwineford1939", package = "lavaan")

# Run LCA with k = 1 to k = 6 classes
mplus.lca(HolzingerSwineford1939, ind = c("x1", "x2", "x3", "x4"),
          mplus.run = TRUE)

# Example 1a: Read Mplus output files, create result table, write table, and save plots
mplus.lca.summa(write = "Results_LCA.xlsx", plot = TRUE)

# Example 1b: Write results into a text file
mplus.lca.summa(write = "Results_LCA.txt")

#-------------------------------------------------------------------------------
# Example 2: Draw bar chart manually

library(ggplot2)

# Collect LCA results
lca.result <- mplus.lca.summa()

# Result table with means
means <- lca.result$result$mean

# Extract results from variance-covariance structure A with 4 latent classes
plotdat <- means[means$folder == "A_Invariant-Theta_Diagonal-Sigma" &  means$nclass == 4, ]

# Draw bar chart
ggplot(plotdat, aes(ind, est, group = class, fill = class)) +
  geom_bar(stat = "identity", position = "dodge", color = "black",
           linewidth = 0.1) +
  geom_errorbar(aes(ymin = low, ymax = upp), width = 0.23,
                linewidth = 0.2, position = position_dodge(0.9)) +
  scale_x_discrete("") +
  scale_y_continuous("Mean Value", limits = c(0, 9),
                     breaks = seq(0, 9, by = 1)) +
  labs(fill = "Latent Class") +
  guides(fill = guide_legend(nrow = 1L)) +
  theme(axis.title = element_text(size = 11),
        axis.text = element_text(size = 11),
        legend.position = "bottom",
        legend.key.size = unit(0.5 , 'cm'),
        legend.title = element_text(size = 11),
        legend.text = element_text(size = 11),
        legend.box.spacing = unit(-9L, "pt"))

# Save bar chart
ggsave("LCA_4-Class.png", dpi = 600, width = 6, height = 4)
}
}
\references{
Banfield, J. D., & Raftery, A E. (1993). Model-based Gaussian and non-Gaussian
clustering. \emph{Biometrics, 49}, 803-821.

Masyn, K. E. (2013). Latent class analysis and finite mixture modeling. In T. D.
Little (Ed.), \emph{The Oxford handbook of quantitative methods: Statistical analysis}
(pp. 551–611). Oxford University Press.

Muthen, L. K., & Muthen, B. O. (1998-2017). \emph{Mplus User's Guide} (8th ed.).
Muthen & Muthen.

Schwartz, G. (1978). Estimating the dimension of a model. \emph{The Annals of Statistics, 6},
461-464.

Sorgente, A., Caliciuri, R., Robba, M., Lanz, M., & Zumbo, B. D. (2025) A systematic
review of latent class analysis in psychology: Examining the gap between guidelines
and research practice. \emph{Behavior Research Methods, 57}(11), 301.
https://doi.org/10.3758/s13428-025-02812-1
}
\seealso{
\code{\link{mplus.lca}}, \code{\link{mplus.run}}, \code{\link{read.mplus}},
\code{\link{write.mplus}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
