% file man/bootMtrPairwise.Rd
% (c) 20018 S Ellison
\name{ boot.mtr.pairwise }
\Rdversion{1.1}
\alias{ boot.mtr.pairwise }

\title{ Parametric bootstrap for pairwise comparison statistics. }
\description{
  Generates a parametric bootstrap for the pair-difference chi-squared statistic or
  MSD.  
  
  \code{boot.mtr.pairwise} is primarily intended to be called from other functions,
  notably \code{bootPDchisq}.
}
\usage{
	boot.mtr.pairwise(x, s=mad , B=3000, probs=c(0.95, 0.99), 
		cov=NULL, cor = NULL, stat=c("MSD", "PDchisq"), 
		method=c("rnorm", "lhs"), keep=FALSE, labels=names(x), ...) 
	
}

\arguments{
   \item{x}{Vector of observations}
   \item{s}{Either a function returning an estimate of scale for \code{x} or a vector of 
  		length \code{length(x)} of standard errors or standard uncertainties in \code{x}.}
   \item{B}{
		Scalar number of bootstrap replicates.
	}
   \item{probs}{
		Vector of probabilities at which to calculate upper quantiles. Passed to
		\code{\link{quantile}}.
	}
   \item{cov, cor}{
		Covariance or correlation matrix, respectively, describing the covariance structure 
		across \code{x}. Passed to \code{pdchisq}.
	}
   \item{stat}{The statistic to be bootstrapped. Either \code{"MSD"} or \code{"PDchisq"} 
               for \code{\link{msd}} or \code{\link{pdchisq}}, respectively. } 
   \item{method}{
		Character value describing the simulation method. 
	}
   \item{keep}{
		If \code{keep == TRUE} the individual bootstrap replicates are retained.
	}
   \item{labels}{
		Character vector of labels for individual values.
	}
   \item{\dots}{Parameters passed to other methods. }
   
}
\details{
	\code{boot.mtr.pairwise} calculates a parametric bootstrap simulation 
	(or Monte carlo simulation) of the results of \code{\link{msd}} or \code{\link{pdchisq}}
	applied to data. This allows individual case-specific quantiles and \emph{p}-values
	to be estimated that allow for different standard errors (or standard uncertainties) \code{s}. 
	
	The sampling method is currently sampling from \code{rnorm}.
	
	Individual upper quantiles for probabilities \code{probs} and \emph{p}-values 
	are estimated 	directly from the bootstrap replicates. Quantiles use 
	\code{\link{quantile}}. \emph{p}-values are estimated from the proportion 
	of replicates that exceed the observed values calculated by 
	\code{\link{msd}} or \code{\link{pdchisq}}. Note that the \code{print} 
	method for the \code{summary} object does not report zero proportions 
	as identically zero. 	
}
\value{
	An object of class \dQuote{bootMtrPairs}, consisting of a list containing:
	of length \code{length(x)} of median scaled absolute deviations for each 
	observation, with attributes:

	\item{t0}{vector of values calculated by \code{\link{msd}} or \code{\link{pdchisq}} }
	\item{labels}{character vector of labels, by default taken from \code{x}}
	\item{probs}{vector of probabilities supplied and used for quantiles}
	\item{critical.values}{matrix of quantiles. Each row corresponds to a probability 
		  in \code{probs} and each column to an individual data point.}
	\item{pvals}{\emph{p}-values estimated as the observed proportion of
		simulated values exceeding the calculated values \code{t0}.}
	\item{B}{Number of bootstrap replicates used.}
	\item{method}{The sampling method used by the parametric bootstrap.}
	\item{stat}{The statistic bootstrapped. Either \code{"MSD"} or \code{"PDchisq"} 
		for \code{\link{msd}} or \code{\link{pdchisq}}, respectively. } 
	\item{t}{If \code{keep=TRUE}, the individual bootstrap replicates 
		generated by \code{\link{msd}} or \code{\link{pdchisq}}.  
		\code{t} is set to \code{NA} if \code{keep=FALSE}}.
        
        Summary, print and plot methods are provided for the class; see \code{\link{bootMtrPairs-class}}. 
}

\seealso{
  \code{\link{msd}}, \code{\link{bootMSD-class}}, 
  \code{\link{pdchisq}}, \code{\link{PDchisq-class}},
}
\author{
S. L. R. Ellison \email{s.ellison@lgcgroup.com}
}
\examples{

  data(Pb)
  \dontrun{
  #Default method:
  set.seed(1023)
  boot.Pb.default <- boot.mtr.pairwise(Pb$value, Pb$u)  # Uses individual standard uncertainties
  summary(boot.Pb.default)
  
  
  #Method for pair-difference chi-squared object:
  pwch.Pb<-pdchisq(Pb$value, Pb$u)  # Uses individual standard uncertainties
  boot.Pb <- boot.mtr.pairwise(pwch.Pb, B=5000)
  	#Increased replication compared to default
  summary(boot.Pb)
  
  # NOTE: The default summary gives individual observation p-values. 
  # To correct for multiple comparisons, apply 
  # a suitable p-value adjustment; for example:
  summary(boot.Pb, p.adjust="holm")

  }


}


\keyword{ univar }
