% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cliHoldridge.R
\name{cliHoldridgePoints}
\alias{cliHoldridgePoints}
\title{Vegetation Classifier Using the HLZ System}
\usage{
cliHoldridgePoints(temp, prec, verbose = FALSE)
}
\arguments{
\item{temp}{'numeric' R object with one-year time series of monthly mean air temperature (in °C)}

\item{prec}{'numeric' R object with one-year time series of monthly precipitation sum (in mm)}

\item{verbose}{'logical' scalar that indicates whether or not values of the bioclimatic indices used should be
added to the output.}
}
\value{
Depending on the setting, a data frame with one or more columns where the HLZ types are stored in the last
    (character) column, while the additional columns contain the values of bioclimatic indices used. The
    abbreviations of HLZ types can be found in the data frame \code{\link[macroBiome]{vegClsNumCodes}}. If
    \code{verbose = FALSE}, the return object is a one-column data frame with the HLZ types.
}
\description{
Calculates the values of bioclimatic indices used in the Holdridge life zone (HLZ) system (Holdridge
    1947, 1967), and designates the HLZ type using these values, by using the monthly time series of temperature
    and precipitation.
}
\details{
To classify vegetation, the HLZ system developed by Holdridge (1947, 1967) uses the values of the
    following 3 bioclimatic indices:

    \itemize{
      \item{\code{abt}: Mean Annual Biotemperature (Eq 1 in Szelepcsényi et al. (2014); in °C)}
      \item{\code{tap}: Total Annual Precipitation (in mm)}
      \item{\code{per}: Potential Evapotranspiration Ratio (Eq 4 in Szelepcsényi et al. (2014); dimensionless)}
    }

    For details about calculating bioclimatic indices, see the function
    \code{\link[macroBiome]{cliBioCliIdxPoints}}. \cr
    The HLZ system classifies the vegetation type based on the distance from the ideal (theoretical) point in the
    3-dimensional space of bioclimatic indices. Numerous variants of the HLZ system are known (e.g.,
    Henderson-Sellers 1994; Yates et al. 2000). Here, one of its most widely used versions ('version with no
    altitudinal belts') is implemented, in accordance with works of Szelepcsényi et al. (2014, 2018). In this
    version, a total of 39 HLZ types are distinguished (see \code{\link[macroBiome]{vegClsNumCodes}}).
}
\note{
As with any function with a point mode, a set of basic input data is defined here. In this case, they are as
    follows: \code{'temp'} (one-year time series of monthly mean air temperature), and \code{'prec'} (one-year
    time series of monthly precipitation sum). The objects \code{'temp'} and \code{'pre'} must be either vectors
    of length 12 or 12-column matrices. The first dimensions of these matrices have to be the same length. The
    function automatically converts vectors into single-row matrices during the error handling, and then uses
    these matrices. The first dimensions of these matrices determines the number of rows in the result matrix.
}
\examples{
# Loading mandatory data for the Example 'Points'
data(inp_exPoints)
data(vegClsNumCodes)

# Designate the HLZ type (using the related bioclimatic indices),
# at a grid cell near Szeged, Hungary (46.3N, 20.2E) (for the normal period 1981-2010)
with(inp_exPoints, {
HLZ <- cliHoldridgePoints(colMeans(temp), colMeans(prec), verbose = TRUE)
numCode <- which(sapply(vegClsNumCodes$Code.HLZ, identical, HLZ[, "vegCls"]))
cbind(HLZ[,-c(4)], vegClsNumCodes[numCode, c("Name.HLZ", "Code.HLZ")])
})

}
\references{
\cite{Henderson-Sellers A (1994) Global terrestrial vegetation ‘prediction’: the use and abuse of climate and
    application models. Prog Phys Geogr 18(2):209–246. \doi{10.1177/030913339401800203}}

\cite{Holdridge LR (1947) Determination of World Plant Formations From Simple Climatic Data. Science
    105(2727):367–368. \doi{10.1126/science.105.2727.367}}

\cite{Holdridge LR (1967) Life zone ecology. Tropical Science Center, San Jose, Costa Rica}

\cite{Szelepcsényi Z, Breuer H, Sümegi P (2014) The climate of Carpathian Region in the 20th century based on the
    original and modified Holdridge life zone system. Cent Eur J Geosci 6(3):293–307.
    \doi{10.2478/s13533-012-0189-5}}

\cite{Szelepcsényi Z, Breuer H, Kis A, Pongrácz R, Sümegi P (2018) Assessment of projected climate change in the
    Carpathian Region using the Holdridge life zone system. Theor Appl Climatol 131(1–2):593–610.
    \doi{10.1007/s00704-016-1987-3}}

\cite{Yates DN, Kittel TGF, Cannon RF (2000) Comparing the Correlative Holdridge Model to Mechanistic
    Biogeographical Models for Assessing Vegetation Distribution Response to Climatic Change. Clim Chang
    44(1–2):59–87. \doi{10.1023/A:1005495908758}}
}
