#' Select a book, passage or portion of a book from the old or new testament
#'
#' Retrieves a whole book(s), passage from a book or a portion of a book or chapter, based on specified criteria such as book, chapter,
#' verse, or alternative selection using author, section, or date.
#'
#' @param book A character vector specifying the book name(s) from which to select the passage.
#'   Ignored if the \code{by} parameter is provided.
#' @param chapter A numeric vector indicating the chapter(s) of the book. If selection by section, author or date is used, the chapter must be set to NULL.
#' @param verse A numeric vector indicating the verse(s) within the chapter.
#' @param fraction A numeric value (default \code{1}) indicating how many equal parts to divide the chapter into.
#' @param part A numeric value specifying which part to return (must be between 1 and `fraction`).
#' @param by An optional character string for alternative selection criteria. It should be one of
#'   \code{"author"}, \code{"section"}, or \code{"date"}. When provided, the \code{book}, \code{chapter},
#'   and \code{verse} parameters must be \code{NULL}.
#' @param divider A character vector specifying the author, section or date range to be selected.
#' @param language A character string indicating the language of the passage. Must be one of
#'   \code{"English"}, \code{"Hebrew"}, or \code{"Greek"}. Default is \code{"English"}.
#' @param testament A character string specifying the testament. Must be one of
#'   \code{"Old"}, \code{"New"}, or \code{"Both"} (case-insensitive). This parameter is required.
#'
#' @details
#' This function validates the input parameters and determines the book(s) to be used based on the provided criteria.
#' If the \code{by} argument is given, the function leverages helper functions such as \code{by_author()},
#' \code{by_section()}, or \code{by_date()} to select the appropriate book(s) based on the alternative criterion.
#' The passage is then retrieved via \code{retrieve_chapter()}, applying verse filtering and partitioning based on
#' the \code{fraction} and \code{part} arguments.
#'
#' @return Returns the selected passage as generated by the \code{retrieve_chapter()} function.
#'
#' @examples
#' # Example 1: Select a passage by specifying book, chapter, and verse.
#' select_passage(book = "Genesis", chapter = 1, verse = 1, testament = "Old")
#' select_passage(book = "Mat", chapter = 1, verse = 1:10, testament = "new")
#' @export
select_passage <- function(book = NULL,
                           chapter = NULL,
                           verse = NULL,
                           fraction = 1,
                           part = 1,
                           by = NULL,
                           divider = NULL,
                           language = "English",
                           testament = NULL) {

  # We definitely need a language and testament
  stopifnot(is.character(language),
            is.character(testament))

  language <- match.arg(language,
                        c("English", "Hebrew", "Greek"),
                        several.ok = FALSE)

  testament <- match.arg(testament,
                         c("Old", "New", "Both", "old", "new", "both"),
                         several.ok = FALSE)

  # Validate 'by' argument (if provided)
  if (!is.null(by)) {
    by <- match.arg(by, c("author", "section", "date"))
  }

  # Validate book
  if (!is.null(book)) {
    books <- book
  }

  # Validate chapter
  if (!is.null(chapter) && !is.numeric(chapter)) {
    stop("chapter must be a numeric vector.")
  }

  # Validate verse
  if (!is.null(verse) && !is.numeric(verse)) {
    stop("verse must be a numeric vector.")
  }

  # Handle 'by' argument logic
  if (!is.null(by)) {
    if (!is.null(book) || !is.null(chapter) || !is.null(verse)) {
      stop("If selecting by author, section, or date, the 'book', 'chapter', and 'verse' arguments must be NULL.")
    }

    # Ensure divider is provided when using 'by'
    if (is.null(divider)) {
      stop("A 'divider' must be provided when selecting by criteria.")
    }

    # Process based on the type of 'by'
    if (by == "author") {
      if (divider %in% author_data$author) {
        books <- by_author(divider, testament)
      } else {
        stop("Invalid author provided in 'divider'. Please check 'author_data' for valid options.")
      }
    } else if (by == "section") {
      if (divider %in% author_data$section) {
        books <- by_section(divider, testament)
      } else {
        stop("Invalid section provided in 'divider'. Please check 'author_data' for valid options.")
      }
    } else if (by == "date") {
      if (divider %in% author_data$date) {
        books <- by_date(divider, testament)
      } else {
        stop("Invalid date provided in 'divider'. Please check 'author_data' for valid options.")
      }
    } else {
      stop("Invalid selection for 'by'. Must be 'author', 'section', or 'date'.")
    }
  }

  # Retrieve passage with verse filtering applied before partitioning
  passage <- retrieve_chapter(book = books,
                              chapter = chapter,
                              verse = verse,
                              fraction = fraction,
                              part = part,
                              language = language,
                              testament = testament)

  return(passage)
}
