% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kissmig.R
\name{kissmig}
\alias{kissmig}
\title{Run a simple dynamic species distribution model}
\usage{
kissmig(
  O,
  S = NULL,
  it,
  type = "FOC",
  signed = FALSE,
  pext = 1,
  pcor = 0.2,
  seed = NULL,
  n_threads = 1,
  n_random = 1e+05
)
}
\arguments{
\item{O}{SpatRaster with a single layer of the initial species distribution as the geographic origin.}

\item{S}{SpatRaster with a single or multiple suitability layers. For the default setting \code{NULL} the suitability of all cells is set to 1.0 (maximum).}

\item{it}{integer of the number of iteration steps representing the species' migration ability and applied to each suitability layer of S.}

\item{type}{string defining the type of resulting map. Default setting is \code{"FOC"}.
\itemize{
\item \code{"DIS"} final species distribution after the last iteration step
\item \code{"FOC"} number of the iteration step of the first occurrence
\item \code{"LOC"} number of the iteration step of the last occurrence
\item \code{"NOC"} number of the iteration steps with occurrence
}}

\item{signed}{bool. If \code{TRUE}, the sign indicates whether the cells was colonized (positive) or uncolonized (negative) after the last iteration step.
Used in combination with \code{"FOC"}, \code{"LOC"}, or \code{"NOC"}. Default setting is \code{FALSE}.}

\item{pext}{probability [0,1] that a colonized cell becomes uncolonized between iteration steps, i.e., without recolonization the species gets locally extinct.
Default setting is \code{1.0}.}

\item{pcor}{probability [0,1] that corner cells are considered in the 3x3 cell neighborhood.
Default setting is \code{0.2}.}

\item{seed}{integer used to set the seed of the random number generator.
Default setting is \code{NULL}.}

\item{n_threads}{integer of the number of threads for parallel computing.
Default setting is \code{1} (i.e., no parallel computing).}

\item{n_random}{integer defining the amount of random numbers for the simulation.
Default setting is \code{100000}.}
}
\description{
\command{kissmig} runs a simple, raster-based, stochastic model to simulate species
distribution dynamics driven by environmental suitability and the migration ability of the species.
}
\details{
Starting from an initial species distribution \code{O} as the geographic origin, \command{kissmig} simulates species distributions
in an environment characterized by a single or multiple (time series) suitability layers \code{S}.
The simulation uses a simple 3x3 cell algorithm, applied \code{it} iteration steps. Colonized cells
have value \code{1}, uncolonized cells value \code{0}, and environmental suitability values vary binary or
continuously between \code{0} (unsuitable) and \code{1} (suitability maximum). In case \code{S} consists of
multiple suitability layers, \code{it} is applied to each layer.

Between iteration steps, colonized cells become extinct with the probability \code{pext}, and for a recolonization
or new colonization event corner cells within the 3x3 neighborhood are considered with probability \code{pcor}
(\code{pcor}=\code{0.2} produces more realistic circular spread patterns in homogeneous environments - see Nobis and Normand 2014, \doi{10.1111/ecog.00930}).
If required, the random numbers are reused as a ring, and small values of \code{n_random} may produce regular structures in spread pattern.

To get reproducible results, the seed of the R random number generator can be set using the \code{seed} parameter.
In addition, for reproducibility the value of \code{n_random} needs to stay the same between simulations.
Changing one of these parameters will generate stochastically different results - even with binary suitability due to the probability \code{pcor}
of the corner events.

Signed results with \code{signed}=\code{TRUE} can be generated to get in addition to the result type
\code{"FOC"}, \code{"LCO"}, or \code{"NOC"} with the same call the final distribution (\code{"DIS"}) with positive values being colonized and negative values being
previously colonized but uncolonized after the last iteration step.

Runtime optimization by parallel computing can be initialized by increasing \code{n_threads}. Depending on the hardware used, it is advantageous to set
\code{n_threads} significantly higher than the number of available CPU cores. To figure out optimal settings compare execution
time with \code{n_threads} set to 2^x (e.g. 4, 8, 16, 32, ...). The implementation uses the OpenMP specification for parallel programming,
\url{https://www.openmp.org/}.
}
\examples{
\donttest{
# generate some SpatRaster data to run kissmig

s <- kissmigDummyS(mean = 12, sd = 3) # a simple climate suitability map
o <- kissmigOrigin(s, 8.0, 44.5, 0.5) # geographic origin
l <- s >= 0 # land mask used for visualization
plot(s, asp = 1.0, main = "Climate suitability & origin (in red)")
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin

# run kissmig with different types of output

sb <- s>0.5 # binary suitability for an arbitrary threshold of 0.5
k1 <- kissmig(o, sb, it = 150, type = "DIS")
plot(k1*l, asp = 1.0, main = "Final distribution (DIS) using binary suitability")
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin

k2 <- kissmig(o, s, it = 150, type = "DIS")
plot(k2*l, asp = 1.0, main = "Final distribution (DIS) using quantitative suitability")
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin

k3 <- kissmig(o, s, it = 150, type = "FOC")
plot(k3*l, asp = 1.0, main = "First iteration step of occurrence (FOC)",
     col = c("lightgrey", map.pal("viridis", n = max(values(k3))+1)))
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin

a <- kissmigAccess(k3, rel = TRUE)
plot(a*l, asp = 1.0, main = "Accessibility based on 'FOC', relative values",
     col = c("lightgrey", map.pal("viridis", n = max(values(k3))+1)))
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin

k4 <- kissmig(o, s, it = 150, type = "NOC")
plot(k4*l, asp = 1.0, main = "Number of iteration steps with occurrences (NOC)",
     col = c("lightgrey", map.pal("viridis", n = max(values(k4))+1)))
plot(o, col = c(rgb(0,0,0,0), "red"), legend = FALSE, add = TRUE) # add origin
}
}
\references{
\itemize{
\item{Nobis, M.P. and Normand, S. 2014. KISSMig - a simple model for R to account for limited migration in
analyses of species distributions. \cite{Ecography} 37: 1282-1287. \doi{10.1111/ecog.00930}}
\item{KISSMig homepage \url{https://purl.oclc.org/wsl/kissmig}.}
}
}
\seealso{
\code{\link{kissmigAccess}, \link{kissmigOrigin}}
}
