% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kre_adaptive.R
\name{nadwat}
\alias{nadwat}
\title{The Classical Nadaraya-Watson Regression Estimator}
\usage{
nadwat(x, dataX, dataY, K, h)
}
\arguments{
\item{x}{Numeric vector with the location(s) at which the Nadaraya-Watson
regression estimator is to be computed.}

\item{dataX}{Numeric vector \eqn{(X_1, \ldots, X_n)} of the x-values from
which (together with the pertaining y-values) the estimate is
to be computed.}

\item{dataY}{Numeric vector \eqn{(Y_1, \ldots, Y_n)} of the y-values from
which (together with the pertaining x-values) the estimate is
to be computed.}

\item{K}{A kernel function (with vectorized in- & output) to be used for the
estimator.}

\item{h}{Numeric scalar for bandwidth \eqn{h}.}
}
\value{
A numeric vector of the same length as \code{x}.
}
\description{
In its arguments \code{x} and \code{dataX} vectorized function to compute
the classical Nadaraya-Watson estimator (as it is \eqn{m_n} in eq. (1.1)
in Eichner & Stute (2012)).
}
\details{
Implementation of the classical Nadaraya-Watson estimator as in eq. (1.1) in
Eichner & Stute (2012) at given location(s) in \code{x} for data \eqn{(X_1,
Y_1), \ldots, (X_n, Y_n)}, a kernel function \eqn{K} and a bandwidth \eqn{h}.
}
\examples{
require(stats)

 # Regression function: a polynomial of degree 4 with one maximum (or
 # minimum), one point of inflection, and one saddle point.
 # Memo: for p(x) = a * (x - x1) * (x - x2)^3 + b the max. (or min.)
 # is at x = (3*x1 + x2)/4, the point of inflection is at x =
 # (x1 + x2)/2, and the saddle point at x = x2.
m <- function(x, x1 = 0, x2 = 8, a = 0.01, b = 0) {
 a * (x - x1) * (x - x2)^3 + b
 }
 # Note: for m()'s default values a minimum is at x = 2, a point
 # of inflection at x = 4, and a saddle point at x = 8.

n <- 100       # Sample size.
set.seed(42)   # To guarantee reproducibility.
X <- runif(n, min = -3, max = 15)      # X_1, ..., X_n
Y <- m(X) + rnorm(length(X), sd = 5)   # Y_1, ..., Y_n

x <- seq(-3, 15, length = 51)   # Where the Nadaraya-Watson estimator
                                # mn of m shall be computed.
mn <- nadwat(x = x, dataX = X, dataY = Y, K = dnorm, h = n^(-1/5))

plot(x = X, y = Y);   rug(X)
lines(x = x, y = mn, col = "blue")  # The estimator.
curve(m, add = TRUE, col = "red")   # The "truth".

}
