#' Find the partial derivative of the error rate with respect to the correlation coefficient rho when k and rho are given
#'
#' @param rho the correlation coefficient between two test statistics
#' @param k a pre-specified constant in the improved trimmed weighted Hochberg procedure
#' @param alpha the significance level
#' @param alphavec a numeric vector of two values representing the weighted significance levels assigned to the two hypotheses
#'
#' @returns the partial derivative of the error rate with respect to the correlation coefficient rho
#' @export
#' @importFrom mvtnorm dmvnorm
#' @importFrom stats qnorm
#' @author Jiangtao Gou
#' @references
#' Gou, J., Chang, Y., Li, T., and Zhang, F. (2025). Improved trimmed weighted Hochberg procedures with two endpoints and sample size optimization. Technical Report.
find_rho_target_mvtnorm <- function (rho, k, alpha, alphavec = c(alpha/2, alpha/2)) {
  # 20241109a
  # Rfun_find_rho_target_mvtnorm
  varcov <- matrix(c(1, rho, rho, 1), nrow = 2)
  x <- rep(qnorm(alpha, lower.tail = F), times = 2)
  PartA <- mvtnorm::dmvnorm(x = x, mean = c(0,0), sigma = varcov)
  x <- c(-qnorm(alphavec[1], lower.tail = F), qnorm(alphavec[2]*(1 + k*alphavec[1]), lower.tail = F) )
  PartB1 <- mvtnorm::dmvnorm(x = x, mean = c(0,0), sigma = varcov)
  x <- c(qnorm(alpha, lower.tail = F), qnorm(alphavec[2]*(1 + k*alphavec[1]), lower.tail = F) )
  PartB2 <- mvtnorm::dmvnorm(x = x, mean = c(0,0), sigma = varcov)
  #
  x <- c(-qnorm(alphavec[2], lower.tail = F), qnorm(alphavec[1]*(1 + k*alphavec[2]), lower.tail = F) )
  PartC1 <- mvtnorm::dmvnorm(x = x, mean = c(0,0), sigma = varcov)
  x <- c(qnorm(alpha, lower.tail = F), qnorm(alphavec[1]*(1 + k*alphavec[2]), lower.tail = F) )
  PartC2 <- mvtnorm::dmvnorm(x = x, mean = c(0,0), sigma = varcov)
  #
  Total <- PartA + PartB1 - PartB2 + PartC1 - PartC2
  #
  return (Total)
}
