% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dindexm.R
\name{dindexm}
\alias{dindexm}
\title{Distributional index model (DIM)}
\usage{
dindexm(
  formula,
  indexfit,
  data,
  response,
  pars = osqpSettings(verbose = FALSE, eps_abs = 1e-05, eps_rel = 1e-05, max_iter =
    10000L),
  progress = TRUE,
  ...
)
}
\arguments{
\item{formula}{object of class \code{formula} that describes the index 
model}

\item{indexfit}{function that fits the index model to training data. Should
accept arguments \code{formula} and \code{data} and admit a \code{predict}
method. Further arguments in \code{...} are passed to indexfit.
See examples.}

\item{data}{\code{data.frame} containing the covariates of the index model
and the response variable.}

\item{response}{name of the response variable in \code{data}.}

\item{pars}{parameters for quadratic programming optimization (only relevant
for multivariate index functions), set using
\code{\link[osqp]{osqpSettings}}.}

\item{progress}{display progressbar for fitting idr?}

\item{...}{further arguments passed to \code{indexfit}.}
}
\value{
Object of class \code{dindexm}: A list containing the index model (first
component) and the IDR fit on the pseudo-data with the index as covariate
(second component).
}
\description{
Fits distributional index model with user-specified index
function to training dataset. See the examples at the bottom to learn
how to specify a distributional single index model.
}
\details{
This function fits a distributional index model (DIM) to training data. The
DIM assumes that the response is more likely to attain higher values when the
values of the index function increases. The index function can be
estimated by parametric methods like \code{\link[stats]{lm}} or 
 \code{\link[stats]{glm}} or also nonparametrically.
 
The formal mathematical assumption of the DIM is that the conditional CDFs
\eqn{F_{y | g(X) = g(x)}(z)} at each fixed threshold \emph{z} decreases, as
\emph{g(x)} increases. Here \code{y} denotes the response, \code{x}, \code{X}
are the covariates in \code{data} and \code{g} is the index function
estimated by \code{indexfit}.

Estimation is performed in two steps: \code{indexfit} is applied to 
\code{data} to estimate the function \code{g}. With this estimate,
\code{\link{idr}} is applied with the pseudo-covariates \code{g(x)} and
response \code{y}.
}
\examples{
n <- 1000
X <- data.frame(x1 = rnorm(n), x2 = rnorm(n), x3 = rnorm(n))
y <- rnorm(n, 1 - X[, 1] + X[, 2]^2 / 3 - (1 - X[, 3]) * (1 + X[, 3]) / 2)
data <- cbind(y = y, as.data.frame(X))

## data for out-of-sample prediction
newX <- data.frame(x1 = rnorm(10), x2 = rnorm(10), x3 = rnorm(10))

## linear regression model for index
model <- dindexm(
  formula = y ~ poly(x1, degree = 2) + poly(x2, degree = 2) + 
    poly(x3, degree = 2),
  indexfit = lm,
  response = "y",
  data = data
)
pred <- predict(model, data = newX)

## plot
plot(pred, 1, main = "LM based DIM")
grd <- pred[[1]]$points
trueCdf <- pnorm(
  grd,
  1 - newX[1, 1] + newX[1, 2]^2 / 3 - (1 - newX[1, 3]) * (1 + newX[1, 3]) / 2
)
points(grd, trueCdf, type = "l", col = 2)
}
\references{
Henzi, A., Kleger, G. R., & Ziegel, J. F. (2020). Distributional (Single)
Index Models. arXiv preprint arXiv:2006.09219.
}
\seealso{
\code{\link{idr}} for more information on IDR,
\code{\link{predict.dindexfit}} for (out-of-sample) predictions based on a
model with with \code{dindexm}.
}
