
<!-- README.md is generated from README.Rmd. Please edit that file -->

<!-- badges: start -->

[![R-CMD-check](https://github.com/beanb2/intkrige/actions/workflows/R-CMD-check.yaml/badge.svg)](https://github.com/beanb2/intkrige/actions/workflows/R-CMD-check.yaml)
<!-- badges: end --> d

# intkrige

This package implements the interval-valued kriging models described in
Bean et. al. (2022).

## Installation

The package can be most conveniently installed using the command

``` r
install.packages("intkrige")
```

For the most up to date version, please use the command

``` r
devtools::install_github("beanb2/intkrige")
```

Note that the github installation makes use of the devtools package
(Wickham, Hester, and Chang 2019). The most current version of this
package can be found at .

## Package Contents

The package revolves around its main function named *intkrige*.
Information about this function can be obtained by entering

``` r
?intkrige::intkrige
```

in the console.

The package also contains an interval-valued design ground snow load
dataset named *utsnow* and an interval-valued temperature dataset named
*ohtemp*. These data are available by installing the intkrige package
and invoking the command

``` r
data(utsnow)
```

with dataset documentation obtained by

``` r
?utsnow
```

## Intro

Once installed, this package runs simple and ordinary interval-valued
kriging models. The following example comes from the design ground snow
load application described in Bean et. al. (Bean, Sun, and Maguire
2022). The snow load problem requires a proper consideration of the
effect of elevation on ground snow loads prior to input into kriging.
The current interval-valued kriging models allow for linear
considerations of secondary variables in the model predictions. Such
models are akin to “kriging with an external drift” (Goovaerts 1997) or
“simple kriging with varying local means” (Goovaerts 2000).

The example shows how to make an interval valued kriging design ground
snow load prediction (after removing the effect of elevation) for Logan,
Utah. Many of the functions in the following examples are wrappers to
functions in the gstat and sp packages (Pebesma 2004; Gräler, Pebesma,
and Heuvelink 2016; Bivand et al. 2013) designed to make the
interval-valued spatial analysis more convenient. Additional details are
provided in the vignette associated with this package. This vignette can
be accessed with the command

``` r
vignette("intkriging-workflow")
```

## Defining Classes and Variograms

The intkrige package relies on interval-valued spatial objects. These
objects are named *intsp* and *intgrd* respectively. *intsp* inherits
directly from *SpatialPointsDataFrame* objects in the sp class, while
*intgrd* inherits from *SpatialPixelsDataFrame*. These class extensions
are created using the *interval()* function as demonstrated below.

``` r
library(intkrige)
#> Loading required package: sp
#> Loading required package: gstat
#> Loading required package: raster

# First, define the location and elevation of interest. 
# (In this case we pick coordinates of Utah State University)
templocs <- data.frame(lat = 41.745, long = -111.810, ELEVATION = 1456)
sp::coordinates(templocs) <- c("long", "lat")
sp::proj4string(templocs) <- "+proj=longlat +ellps=WGS84
                              +datum=WGS84 +no_defs +towgs84=0,0,0"

# Load the Utah Snow Load Data
data(utsnow)
utsnow.sp <- utsnow

# Convert to an 'intsp' object that inherits a SpatialPointsDataFrame
sp::coordinates(utsnow.sp) <- c("LONGITUDE", "LATITUDE")
sp::proj4string(utsnow.sp) <- sp::proj4string(templocs)
interval(utsnow.sp) <- c("minDL", "maxDL")
```

The objects are defined by the interval slot. Intervals must be defined
according to their upper and lower endpoints. The program will not
accept values in the interval slot that do not meet this criteria. The
interval-valued kriging models in the package rely on characterizations
of intervals by their center and radius. The center/radius values are
calculated automatically as needed by the various functions. The
rationale is behind this approach is that intervals are easier to read
using lower/upper endpoint convention, but easier to *visualize* and
*analyze* using center/radius convention. One consequence of this
approach is that variable transformations are most conveniently
performed using the lower/upper endpoint characterization.

``` r
# analyze interval on log scale
interval(utsnow.sp) <- log(interval(utsnow.sp))
```

## Fitting variograms

Next, we demonstrate how to create and fit empirical variograms. The
functions that accomplish this task are fairly shallow wrappers to the
functions in the gstat package. These function in the intkrige package
simplify the workflow by fitting the center, radius, and center/radius
variograms using the information contained in the interval slot. The
*intvariogram* function defines the empirical variograms, while the
*fit.intvariogram* function automatically fits variograms to the output
of *intvariogram*. Note that intvariogram defines an *intvariogram*
class which is the only class that fit.intvariogram will accept. This
new class designation is intended to ensure that *intvariogram* and
*fit.intvariogram* are always used in tandem. Note that the cross
variogram for the center/radius interaction has a degenerate fit in this
case.

``` r
# Define the formulas we will use to define the intervals. 
temp_formulas <- center ~ ELEVATION

# Define, fit and check the variogram fits. 
varios <- intvariogram(utsnow.sp,
                       centerFormula = temp_formulas)
varioFit <- fit.intvariogram(varios, models = gstat::vgm(c("Sph", "Sph", "Gau")))
#> Warning in gstat::fit.variogram(x_sub, model = models[[i]], ...): No
#> convergence after 200 iterations: try different initial values?
intvCheck(varios, varioFit)
```

<img src="man/figures/README-varioCheck2-1.png" width="100%" />

## Interval-valued kriging models

With variograms defined for each component of the interval, we are now
ready to make interval-valued predictions using our interval-valued
kriging models. If you explore the function documentation, you will
notice that many of the function arguments relate to the Newton-Raphson
optimization technique that is used to calculate the kriging model
weights. These arguments should not need to be changed from the defaults
for most user purposes. The following demonstrates an ordinary kriging
model after accounting for elevation in the prediction of the centers.

``` r
preds <- intkrige::intkrige(locations = utsnow.sp,
                            newdata = templocs, 
                            models = varioFit,
                            centerFormula = temp_formulas)
#> [generalized least squares trend estimation]
#> [generalized least squares trend estimation]

# The final results are predicted intervals after removing the effect of elevation.  
preds
#>         coordinates               interval ELEVATION    center    radius
#> 1 (-111.81, 41.745) [-0.1769736, 1.030301]      1456 0.6757244 0.6036374
#>   kriging_variance warn
#> 1        0.1651463    0
```

The interval prediction (including the elevation effect) is included in
the interval slot. The raw predictions are appended to the data frame
under the following variable names:

-center: the value of the interval centers returned directly from the
kriging model,

-radius: raw radii predictions,

-variance: the kriging variance according the generalized L2 distance,

-warn: a boolean that is 0 for a convergent optimization and 1 for a
non-convergent optimization. This quiet approach to warnings ensures a
non-disruptive experience for the user when predicting a large number of
locations.

# References

<div id="refs" class="references csl-bib-body hanging-indent"
entry-spacing="0">

<div id="ref-Bean2019-int" class="csl-entry">

Bean, Brennan, Yan Sun, and Marc Maguire. 2022. “Interval-Valued Kriging
for Geostatistical Mapping with Imprecise Inputs.” *International
Journal of Approximate Reasoning* 140: 31–51.
<https://doi.org/10.1016/j.ijar.2021.10.003>.

</div>

<div id="ref-Bivand2008" class="csl-entry">

Bivand, Roger S, Edzer J Pebesma, Virgilio Gomez-Rubio, and Edzer Jan
Pebesma. 2013. *Applied Spatial Data Analysis with r*. 2nd ed. Vol.
747248717. Springer.

</div>

<div id="ref-Goovaerts1997" class="csl-entry">

Goovaerts, Pierre. 1997. *Geostatistics for Natural Resources
Evaluation*. Oxford University Press.

</div>

<div id="ref-Goovaerts2000" class="csl-entry">

———. 2000. “Geostatistical Approaches for Incorporating Elevation into
the Spatial Interpolation of Rainfall.” *Journal of Hydrology* 228 (1):
113–29.

</div>

<div id="ref-gstat2" class="csl-entry">

Gräler, Benedikt, Edzer Pebesma, and Gerard Heuvelink. 2016.
“Spatio-Temporal Interpolation Using Gstat.” *The R Journal* 8: 204–18.
<https://journal.r-project.org/archive/2016-1/na-pebesma-heuvelink.pdf>.

</div>

<div id="ref-gstat1" class="csl-entry">

Pebesma, Edzer J. 2004. “Multivariable Geostatistics in S: The Gstat
Package.” *Computers & Geosciences* 30: 683–91.

</div>

<div id="ref-devtools2019" class="csl-entry">

Wickham, Hadley, Jim Hester, and Winston Chang. 2019. *Devtools: Tools
to Make Developing r Packages Easier*.
<https://github.com/r-lib/devtools>.

</div>

</div>
