nnlocalsearch <- function(rsim, l, u, guess, data, ctrl) {
    n <- data$n
    N <- ctrl$Ntotal
    nlocal <- ctrl$NFitlocal
    nfit <- max(ctrl$Nelite, nlocal - (N - n))
    nextreport <- trace <- ctrl$trace
    p <- data$p
    q <- data$q
    add <- ctrl$NAddlocal
    rhobar <- ctrl$Rhomax
    rho <- rhobar / 10
    lambda <- ctrl$Lambda
    modok <- q * ctrl$Tolmodel
    tol <- ctrl$Tollocal
    eps <- sqrt(.Machine$double.eps)
    Ip <- diag(p)
    zp <- rep(0, p)
    Zp <- matrix(0, p, p)
    V <- B <- NULL
    while (1) {
        ## first, we fit a local linear model using the `nfit` nearest points
        ## to the current guess
        lmod <- nnlm(data, nfit, guess, TRUE)
        ## then update the estimates of the jacobian and score vcov
        if (is.null(B)) {
            B <- lmod$B
            V <- lmod$V
        } else {
            B <- B + lambda * (lmod$B - B)
            V <- V + lambda * (lmod$V - V)
        }
        ## and, finally, we propose a new guess
        sc <- isqm(V)
        sj <- sc %*% B
        sa <- sc %*% lmod$a
        grad <- -crossprod(sj, sa)
        JVJ <- crossprod(sj)
        g <- rho * lmod$scale
        sol <- lpSolve::lp(
            objective.in = c(double(2 * p), rep(1, 2 * p)),
            const.mat = rbind(
                cbind(JVJ, -JVJ, -Ip, Ip),
                cbind(Ip, Zp, Zp, Zp),
                cbind(Zp, Ip, Zp, Zp)
            ),
            const.dir = c(rep("=", p), rep("<", 2 * p)),
            const.rhs = c(grad, pmin(g, u - eps - guess), pmin(g, guess - l - eps))
        )
        old <- guess
        if (!sol$status) {
            ## paranoid if? the linear optimization problem should alwyas be feasible
            guess <- guess + sol$solution[seq.int(1, p)] - sol$solution[seq.int(p + 1, 2 * p)]
        }
        ## Convergence? And report
        ia <- seq(1, q * (p + 1), by = p + 1)
        Vg <- crossprod(sj, sc %*% lmod$Va %*% sc %*% sj)
        grad.se <- sqrt(diag(Vg))
        free <- which(((guess > l + 1.1 * eps) | (grad > 0)) & ((guess < u - 1.1 * eps) | (grad < 0)))
        lfree <- length(free)
        grnorm2 <- if (lfree) sum((isqm(Vg[free, free]) %*% grad[free])^2) else 0
        done <- (n == N) || ((nfit >= nlocal) && (grnorm2 < tol * lfree))
        if ((trace > 0) && (done || (n >= nextreport))) {
            cat(
                "After", n, "simulations (local search)\n",
                " guess:               ", guess, "\n",
                " estimating fun.:     ", grad, "\n",
                " estimating fun. s.e.:", grad.se, "\n",
                " squared estimating fun. norm:", grnorm2,
                " rho:", rho, "\n"
            )
            nextreport <- n + trace
        }
        if (done) break
        ## Simulations around the new guess
        mn <- min(add, N - n)
        idx <- seq.int(n + 1, n + mn)
        S <- matrix(stats::rnorm(mn * p), p)
        data$par[, idx] <-
            pmax(l, pmin(u, guess + isqm(JVJ) %*%
                scale(S, center = FALSE, scale = sqrt(colSums(S * S)) / stats::runif(mn, eps, 1))))
        data$stat[, idx] <-
            if (is.null(data$cl)) {
                apply(data$par[, idx, drop = FALSE], 2, rsim)
            } else {
                parallel::parCapply(data$cl, data$par[, idx, drop = FALSE], rsim)
            }
        ## checking the prediction errors to decide if we have to
        ## go back to the old guess and how to update the size of the
        ## trust region
        err <- data$stat[, idx, drop = FALSE] - lmod$a - B %*% (data$par[, idx, drop = FALSE] - guess)
        err <- sum((sc %*% err)^2)
        if (err < mn * modok) {
            rho <- min(2 * rho, rhobar)
        } else {
            rho <- max(eps, rho / 4)
            guess <- old
        }
        data$n <- n <- n + mn
        nfit <- min(nfit + mn, nlocal)
    }
    U <- isqm(crossprod(sj))
    list(
        guess = guess, cov = tcrossprod(U), a = lmod$a, B = B, V = V, isqmV = sc,
        rho = rho, grad = as.numeric(grad), grad.se = grad.se, grnorm2 = grnorm2, lfree = lfree
    )
}
