#' Plot ALOS - Average Length of Stay
#'
#' @family Plotting Functions
#'
#' @author Steven P. Sanderson II, MPH
#'
#' @description
#' Plot ALOS - Average Length of Stay
#'
#' @details
#' - Expects a tibble with a date time column and a value column
#' - Uses `timetk` for underlying sumarization and plot
#' - If .by_grouping is missing it will default to "day"
#' - A static ggplot2 object is return if the .interactive function is FALSE
#' otherwise a `plotly` plot is returned.
#'
#' @param .data The time series data you need to pass
#' @param .date_col The date column
#' @param .value_col The value column
#' @param .by_grouping How you want the data summarized - "sec", "min", "hour",
#' "day", "week", "month", "quarter" or "year"
#' @param .interactive TRUE or FALSE. TRUE returns a `plotly` plot and FALSE
#' returns a static `ggplot2` plot
#'
#' @examples
#' library(healthyR)
#' library(healthyR.data)
#' library(timetk)
#' library(dplyr)
#' library(purrr)
#'
#' # Make A Series of Dates ----
#' data_tbl <- healthyR_data
#'
#' df_tbl <- data_tbl %>%
#'     filter(ip_op_flag == "I") %>%
#'     select(visit_end_date_time, length_of_stay) %>%
#'     summarise_by_time(
#'         .date_var = visit_end_date_time
#'         , .by     = "day"
#'         , visits  = mean(length_of_stay, na.rm = TRUE)
#'     ) %>%
#'     filter_by_time(
#'         .date_var     = visit_end_date_time
#'         , .start_date = "2012"
#'         , .end_date   = "2019"
#'     ) %>%
#'     set_names("Date","Values")
#'
#' ts_alos_plt(
#'   .data = df_tbl
#'   , .date_col = Date
#'   , .value_col = Values
#'   , .by = "month"
#'   , .interactive = FALSE
#' )
#'
#' @return
#' A timetk time series plot
#'
#' @export
#'

ts_alos_plt <- function(.data, .date_col, .value_col, .by_grouping, .interactive) {

    # * Tidyeval ----
    date_var_expr        <- rlang::enquo(.date_col)
    value_var_expr       <- rlang::enquo(.value_col)
    by_var_expr          <- .by_grouping
    interactive_var_expr <- .interactive

    # * Checks ----
    if(!is.data.frame(.data)) {
        stop(call. = FALSE, "(data) is not a data-frame or tibble. Please supply.")
    }

    if (rlang::quo_is_missing(date_var_expr)) {
        stop(call. = FALSE, "(date_var_expr) is missing. Please supply.")
    }

    if(rlang::quo_is_missing(value_var_expr)) {
        stop(call. = FALSE, "(value_var_expr) is missing. Please supply.")
    }

    # * Manipulate ----
    df_tbl <- tibble::as_tibble(.data) %>%
        dplyr::select( {{date_var_expr}}, {{value_var_expr}} ) %>%
        purrr::set_names("date", "value")

    # If .by is missing then manipulate
    if(by_var_expr == "") {
        df_summarised_tbl <- timetk::summarise_by_time(
            .data = df_tbl
            , .date_var = date
            , value = mean(value)
        )
    } else {
        df_summarised_tbl <- timetk::summarise_by_time(
            .data = df_tbl
            , .date_var = date
            , .by = by_var_expr
            , value = mean(value)
        )
    }

    # * Plot ----
    if(!interactive_var_expr) {
        g <- df_summarised_tbl %>%
            timetk::plot_time_series(
                .date_var = date
                , .value = value
                , .title = "Average Length of Stay Plot"
                , .interactive = FALSE
            ) +
            ggplot2::theme_minimal()
    } else {
        g <- df_summarised_tbl %>%
            timetk::plot_time_series(
                .date_var = date
                , .value = value
                , .title = "Average Length of Stay Plot"
                , .interactive = FALSE
            ) +
            ggplot2::theme_minimal()

        g <- plotly::ggplotly(g)
    }

    # * Return ----
    return(g)

}
