/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5G_FRIEND     
#include "H5Lmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Lpkg.h"      
#include "H5MMprivate.h" 
#include "H5Opublic.h"   
#include "H5Pprivate.h"  
#include "H5VLprivate.h" 

#define H5L_EXT_TRAVERSE_BUF_SIZE 256

static hid_t   H5L__extern_traverse(const char *link_name, hid_t cur_group, const void *udata,
                                    size_t udata_size, hid_t lapl_id, hid_t dxpl_id);
static ssize_t H5L__extern_query(const char *link_name, const void *udata, size_t udata_size,
                                 void *buf , size_t buf_size);

static const H5L_class_t H5L_EXTERN_LINK_CLASS[1] = {{
    H5L_LINK_CLASS_T_VERS, 
    H5L_TYPE_EXTERNAL,     
    "external",            
    NULL,                  
    NULL,                  
    NULL,                  
    H5L__extern_traverse,  
    NULL,                  
    H5L__extern_query      
}};

static hid_t
H5L__extern_traverse(const char H5_ATTR_UNUSED *link_name, hid_t cur_group, const void *_udata,
                     size_t H5_ATTR_UNUSED udata_size, hid_t lapl_id, hid_t H5_ATTR_UNUSED dxpl_id)
{
    H5P_genplist_t    *plist;                              
    H5G_loc_t          root_loc;                           
    H5G_loc_t          loc;                                
    H5F_t             *ext_file = NULL;                    
    const uint8_t     *p        = (const uint8_t *)_udata; 
    const char        *file_name;                    
    const char        *obj_name;                     
    size_t             fname_len;                    
    unsigned           intent;                       
    H5L_elink_cb_t     cb_info;                      
    hid_t              fapl_id    = H5I_INVALID_HID; 
    void              *ext_obj    = NULL;            
    hid_t              ext_obj_id = H5I_INVALID_HID; 
    H5I_type_t         opened_type;                  
    char              *parent_group_name = NULL;     
    char               local_group_name[H5L_EXT_TRAVERSE_BUF_SIZE]; 
    H5P_genplist_t    *fa_plist;                                    
    H5F_close_degree_t fc_degree    = H5F_CLOSE_WEAK;               
    char              *elink_prefix = NULL;                         
    hid_t              ret_value    = H5I_INVALID_HID;              

    FUNC_ENTER_PACKAGE

    
    assert(p);

    
    if (((*p >> 4) & 0x0F) > H5L_EXT_VERSION)
        HGOTO_ERROR(H5E_LINK, H5E_CANTDECODE, H5I_INVALID_HID, "bad version number for external link");
    if ((*p & 0x0F) & ~H5L_EXT_FLAGS_ALL)
        HGOTO_ERROR(H5E_LINK, H5E_CANTDECODE, H5I_INVALID_HID, "bad flags for external link");
    p++;

    
    file_name = (const char *)p;
    fname_len = strlen(file_name);
    obj_name  = (const char *)p + fname_len + 1;

    
    if (NULL == (plist = H5P_object_verify(lapl_id, H5P_LINK_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, H5I_INVALID_HID, "can't find object for ID");

    
    if (H5P_get(plist, H5L_ACS_ELINK_FAPL_NAME, &fapl_id) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5I_INVALID_HID, "can't get fapl for links");

    
    if (H5G_loc(cur_group, &loc) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_CANTGET, H5I_INVALID_HID, "can't get object location");

    
    if (H5P_get(plist, H5L_ACS_ELINK_FLAGS_NAME, &intent) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5I_INVALID_HID, "can't get elink file access flags");

    
    if (intent == H5F_ACC_DEFAULT)
        intent = H5F_INTENT(loc.oloc->file);

    if ((fapl_id == H5P_DEFAULT) && ((fapl_id = H5F_get_access_plist(loc.oloc->file, false)) < 0))
        HGOTO_ERROR(H5E_LINK, H5E_CANTGET, H5I_INVALID_HID, "can't get parent's file access property list");

    
    if (H5P_get(plist, H5L_ACS_ELINK_CB_NAME, &cb_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5I_INVALID_HID, "can't get elink callback info");

    
    if (NULL == (fa_plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, H5I_INVALID_HID, "can't find object for ID");

    
    if (cb_info.func) {
        const char *parent_file_name;   
        size_t      group_name_len = 0; 

        
        parent_file_name = H5F_OPEN_NAME(loc.oloc->file);

        
        if (H5G_get_name(&loc, NULL, (size_t)0, &group_name_len, NULL) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTGET, H5I_INVALID_HID, "unable to retrieve length of group name");

        
        group_name_len++;

        
        if (group_name_len > sizeof(local_group_name)) {
            if (NULL == (parent_group_name = (char *)H5MM_malloc(group_name_len)))
                HGOTO_ERROR(H5E_LINK, H5E_CANTALLOC, H5I_INVALID_HID,
                            "can't allocate buffer to hold group name, group_name_len = %llu", (unsigned long long)group_name_len);
        } 
        else
            parent_group_name = local_group_name;

        
        if (H5G_get_name(&loc, parent_group_name, group_name_len, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTGET, H5I_INVALID_HID, "unable to retrieve group name");

        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = (cb_info.func)(parent_file_name, parent_group_name, file_name, obj_name, &intent,
                                           fapl_id, cb_info.user_data);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CALLBACK, H5I_INVALID_HID, "traversal operator failed");

        
        if ((intent & H5F_ACC_TRUNC) || (intent & H5F_ACC_EXCL))
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid file open flags");
    } 

    
    if (H5P_set(fa_plist, H5F_ACS_CLOSE_DEGREE_NAME, &fc_degree) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, H5I_INVALID_HID, "can't set file close degree");

    
    if (H5P_peek(plist, H5L_ACS_ELINK_PREFIX_NAME, &elink_prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, H5I_INVALID_HID, "can't get external link prefix");

    
    if (H5F_prefix_open_file(false, &ext_file, loc.oloc->file, H5F_PREFIX_ELINK, elink_prefix, file_name,
                             intent, fapl_id) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_CANTOPENFILE, H5I_INVALID_HID,
                    "unable to open external file, external link file name = '%s'", file_name);

    
    if (H5G_root_loc(ext_file, &root_loc) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_BADVALUE, H5I_INVALID_HID, "unable to create location for file");

    
    if (NULL == (ext_obj = H5O_open_name(&root_loc, obj_name, &opened_type)))
        HGOTO_ERROR(H5E_LINK, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open object");

    
    if ((ext_obj_id = H5VL_wrap_register(opened_type, ext_obj, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register external link object");

    
    ret_value = ext_obj_id;

done:
    
    
    if (fapl_id > 0 && H5I_dec_ref(fapl_id) < 0)
        HDONE_ERROR(H5E_ID, H5E_CANTRELEASE, H5I_INVALID_HID,
                    "unable to close ID for file access property list");
    if (ext_file && H5F_efc_close(loc.oloc->file, ext_file) < 0)
        HDONE_ERROR(H5E_LINK, H5E_CANTCLOSEFILE, H5I_INVALID_HID, "problem closing external file");
    if (parent_group_name && parent_group_name != local_group_name)
        parent_group_name = (char *)H5MM_xfree(parent_group_name);
    if (ret_value < 0) {
        
        if (ext_obj_id >= 0 && H5I_dec_ref(ext_obj_id) < 0)
            HDONE_ERROR(H5E_ID, H5E_CANTRELEASE, H5I_INVALID_HID, "unable to close ID for external object");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static ssize_t
H5L__extern_query(const char H5_ATTR_UNUSED *link_name, const void *_udata, size_t udata_size,
                  void *buf , size_t buf_size)
{
    const uint8_t *udata     = (const uint8_t *)_udata; 
    ssize_t        ret_value = SUCCEED;                 

    FUNC_ENTER_PACKAGE

    
    if (((*udata >> 4) & 0x0F) != H5L_EXT_VERSION)
        HGOTO_ERROR(H5E_LINK, H5E_CANTDECODE, FAIL, "bad version number for external link");
    if ((*udata & 0x0F) & ~H5L_EXT_FLAGS_ALL)
        HGOTO_ERROR(H5E_LINK, H5E_CANTDECODE, FAIL, "bad flags for external link");

    
    if (buf) {
        if (udata_size < buf_size)
            buf_size = udata_size;

        
        H5MM_memcpy(buf, udata, buf_size);
    } 

    
    ret_value = (ssize_t)udata_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5L_register_external(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5L_register(H5L_EXTERN_LINK_CLASS) < 0)
        HGOTO_ERROR(H5E_LINK, H5E_NOTREGISTERED, FAIL, "unable to register external link class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
