/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ACmodule.h" 
#define H5C_FRIEND      
#define H5F_FRIEND      

#include "H5private.h"   
#include "H5ACpkg.h"     
#include "H5Clog.h"      
#include "H5Cpkg.h"      
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Pprivate.h"  
#include "H5SLprivate.h" 

static herr_t H5AC__check_if_write_permitted(const H5F_t *f, bool *write_permitted_ptr);
static herr_t H5AC__ext_config_2_int_config(const H5AC_cache_config_t *ext_conf_ptr,
                                            H5C_auto_size_ctl_t       *int_conf_ptr);
#if H5AC_DO_TAGGING_SANITY_CHECKS
static herr_t H5AC__verify_tag(const H5AC_class_t *type);
#endif 

bool H5_PKG_INIT_VAR = false;

#ifdef H5_HAVE_PARALLEL

bool H5_coll_api_sanity_check_g = false;
#endif 

static const H5AC_class_t *const H5AC_class_s[] = {
    H5AC_BT,               
    H5AC_SNODE,            
    H5AC_LHEAP_PRFX,       
    H5AC_LHEAP_DBLK,       
    H5AC_GHEAP,            
    H5AC_OHDR,             
    H5AC_OHDR_CHK,         
    H5AC_BT2_HDR,          
    H5AC_BT2_INT,          
    H5AC_BT2_LEAF,         
    H5AC_FHEAP_HDR,        
    H5AC_FHEAP_DBLOCK,     
    H5AC_FHEAP_IBLOCK,     
    H5AC_FSPACE_HDR,       
    H5AC_FSPACE_SINFO,     
    H5AC_SOHM_TABLE,       
    H5AC_SOHM_LIST,        
    H5AC_EARRAY_HDR,       
    H5AC_EARRAY_IBLOCK,    
    H5AC_EARRAY_SBLOCK,    
    H5AC_EARRAY_DBLOCK,    
    H5AC_EARRAY_DBLK_PAGE, 
    H5AC_FARRAY_HDR,       
    H5AC_FARRAY_DBLOCK,    
    H5AC_FARRAY_DBLK_PAGE, 
    H5AC_SUPERBLOCK,       
    H5AC_DRVRINFO,         
    H5AC_EPOCH_MARKER,     
    H5AC_PROXY_ENTRY,      
    H5AC_PREFETCHED_ENTRY  
};

herr_t
H5AC_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC__init_package(void)
{
    FUNC_ENTER_PACKAGE_NOERR

#ifdef H5_HAVE_PARALLEL
    
    {
        const char *s; 

        s = getenv("H5_COLL_API_SANITY_CHECK");
        if (s && isdigit(*s)) {
            long env_val               = strtol(s, NULL, 0);
            H5_coll_api_sanity_check_g = (0 == env_val) ? false : true;
        }
    }
#endif 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5AC_term_package(void)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR)
        
        H5_PKG_INIT_VAR = false;

    FUNC_LEAVE_NOAPI(0)
} 

bool
H5AC_cache_image_pending(const H5F_t *f)
{
    H5C_t *cache_ptr;
    bool   ret_value = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;

    ret_value = H5C_cache_image_pending(cache_ptr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_create(const H5F_t *f, H5AC_cache_config_t *config_ptr, H5AC_cache_image_config_t *image_config_ptr)
{
#ifdef H5_HAVE_PARALLEL
    char        prefix[H5C__PREFIX_LEN] = "";
    H5AC_aux_t *aux_ptr                 = NULL;
#endif 
    struct H5C_cache_image_ctl_t int_ci_config = H5C__DEFAULT_CACHE_IMAGE_CTL;
    herr_t                       ret_value     = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(NULL == f->shared->cache);
    assert(config_ptr != NULL);
    assert(image_config_ptr != NULL);
    assert(image_config_ptr->version == H5AC__CURR_CACHE_IMAGE_CONFIG_VERSION);
    HDcompile_assert(NELMTS(H5AC_class_s) == H5AC_NTYPES);
    HDcompile_assert(H5C__MAX_NUM_TYPE_IDS == H5AC_NTYPES);

    
    if (H5AC_validate_config(config_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Bad cache configuration");
    if (H5AC_validate_cache_image_config(image_config_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Bad cache image configuration");

#ifdef H5_HAVE_PARALLEL
    if (H5F_HAS_FEATURE(f, H5FD_FEAT_HAS_MPI)) {
        MPI_Comm mpi_comm;
        int      mpi_rank;
        int      mpi_size;

        if (MPI_COMM_NULL == (mpi_comm = H5F_mpi_get_comm(f)))
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get MPI communicator");

        if ((mpi_rank = H5F_mpi_get_rank(f)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get mpi rank");

        if ((mpi_size = H5F_mpi_get_size(f)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "can't get mpi size");

        if (NULL == (aux_ptr = H5FL_CALLOC(H5AC_aux_t)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "Can't allocate H5AC auxiliary structure");

        aux_ptr->mpi_comm                = mpi_comm;
        aux_ptr->mpi_rank                = mpi_rank;
        aux_ptr->mpi_size                = mpi_size;
        aux_ptr->write_permitted         = false;
        aux_ptr->dirty_bytes_threshold   = H5AC__DEFAULT_DIRTY_BYTES_THRESHOLD;
        aux_ptr->dirty_bytes             = 0;
        aux_ptr->metadata_write_strategy = H5AC__DEFAULT_METADATA_WRITE_STRATEGY;
#ifdef H5AC_DEBUG_DIRTY_BYTES_CREATION
        aux_ptr->dirty_bytes_propagations      = 0;
        aux_ptr->unprotect_dirty_bytes         = 0;
        aux_ptr->unprotect_dirty_bytes_updates = 0;
        aux_ptr->insert_dirty_bytes            = 0;
        aux_ptr->insert_dirty_bytes_updates    = 0;
        aux_ptr->move_dirty_bytes              = 0;
        aux_ptr->move_dirty_bytes_updates      = 0;
#endif 
        aux_ptr->d_slist_ptr         = NULL;
        aux_ptr->c_slist_ptr         = NULL;
        aux_ptr->candidate_slist_ptr = NULL;
        aux_ptr->write_done          = NULL;
        aux_ptr->sync_point_done     = NULL;
        aux_ptr->p0_image_len        = 0;

        snprintf(prefix, sizeof(prefix), "%d:", mpi_rank);

        if (mpi_rank == 0) {
            if (NULL == (aux_ptr->d_slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTCREATE, FAIL, "can't create dirtied entry list");

            if (NULL == (aux_ptr->c_slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTCREATE, FAIL, "can't create cleaned entry list");
        } 

        
        if (NULL == (aux_ptr->candidate_slist_ptr = H5SL_create(H5SL_TYPE_HADDR, NULL)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTCREATE, FAIL, "can't create candidate entry list");

        if (aux_ptr->mpi_rank == 0)
            f->shared->cache = H5C_create(H5AC__DEFAULT_MAX_CACHE_SIZE, H5AC__DEFAULT_MIN_CLEAN_SIZE,
                                          (H5AC_NTYPES - 1), H5AC_class_s, H5AC__check_if_write_permitted,
                                          true, H5AC__log_flushed_entry, (void *)aux_ptr);
        else
            f->shared->cache =
                H5C_create(H5AC__DEFAULT_MAX_CACHE_SIZE, H5AC__DEFAULT_MIN_CLEAN_SIZE, (H5AC_NTYPES - 1),
                           H5AC_class_s, H5AC__check_if_write_permitted, true, NULL, (void *)aux_ptr);
    } 
    else {
#endif 
        
        f->shared->cache =
            H5C_create(H5AC__DEFAULT_MAX_CACHE_SIZE, H5AC__DEFAULT_MIN_CLEAN_SIZE, (H5AC_NTYPES - 1),
                       H5AC_class_s, H5AC__check_if_write_permitted, true, NULL, NULL);
#ifdef H5_HAVE_PARALLEL
    }  
#endif 

    if (NULL == f->shared->cache)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "memory allocation failed");

#ifdef H5_HAVE_PARALLEL
    if (aux_ptr != NULL)
        if (H5C_set_prefix(f->shared->cache, prefix) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL, "H5C_set_prefix() failed");
#endif 

    
    if (H5F_USE_MDC_LOGGING(f))
        if (H5C_log_set_up(f->shared->cache, H5F_MDC_LOG_LOCATION(f), H5C_LOG_STYLE_JSON,
                           H5F_START_MDC_LOG_ON_ACCESS(f)) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "mdc logging setup failed");

    
    if (H5AC_set_cache_auto_resize_config(f->shared->cache, config_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL, "auto resize configuration failed");

    
    int_ci_config.version            = image_config_ptr->version;
    int_ci_config.generate_image     = image_config_ptr->generate_image;
    int_ci_config.save_resize_status = image_config_ptr->save_resize_status;
    int_ci_config.entry_ageout       = image_config_ptr->entry_ageout;
    if (H5C_set_cache_image_config(f, f->shared->cache, &int_ci_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL, "auto resize configuration failed");

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_create_cache_msg(f->shared->cache, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

#ifdef H5_HAVE_PARALLEL
    
    if (ret_value < 0) {
        if (aux_ptr != NULL) {
            if (aux_ptr->d_slist_ptr != NULL)
                H5SL_close(aux_ptr->d_slist_ptr);
            if (aux_ptr->c_slist_ptr != NULL)
                H5SL_close(aux_ptr->c_slist_ptr);
            if (aux_ptr->candidate_slist_ptr != NULL)
                H5SL_close(aux_ptr->candidate_slist_ptr);
            aux_ptr = H5FL_FREE(H5AC_aux_t, aux_ptr);
        } 
    }     
#endif    

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_dest(H5F_t *f)
{
    bool log_enabled;  
    bool curr_logging; 
#ifdef H5_HAVE_PARALLEL
    H5AC_aux_t *aux_ptr = NULL;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

#if H5AC_DUMP_STATS_ON_CLOSE
    
    H5AC_stats(f);
#endif 

    
    if (H5C_get_logging_status(f->shared->cache, &log_enabled, &curr_logging) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to get logging status");

    
    if (log_enabled) {
        if (curr_logging)
            if (H5C_log_write_destroy_cache_msg(f->shared->cache) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

        if (H5C_log_tear_down(f->shared->cache) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "metadata cache logging tear-down failed");
    } 

#ifdef H5_HAVE_PARALLEL
    
    if (H5C_clear_coll_entries(f->shared->cache, false) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL, "can't clear collective entries");

    aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(f->shared->cache);
    if (aux_ptr) {
        
        if (H5F_ACC_RDWR & H5F_INTENT(f)) {
            
            if (H5C_set_slist_enabled(f->shared->cache, true, true) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't enable skip list");

            if (H5AC__flush_entries(f) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't flush");

            
            if (H5C_set_slist_enabled(f->shared->cache, false, false) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't disable skip list");
        } 
    }     
#endif    

    
    if (H5C_dest(f) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "can't destroy cache");

    f->shared->cache = NULL;

#ifdef H5_HAVE_PARALLEL
    if (aux_ptr != NULL) {
        if (aux_ptr->d_slist_ptr != NULL) {
            assert(H5SL_count(aux_ptr->d_slist_ptr) == 0);
            H5SL_close(aux_ptr->d_slist_ptr);
        } 

        if (aux_ptr->c_slist_ptr != NULL) {
            assert(H5SL_count(aux_ptr->c_slist_ptr) == 0);
            H5SL_close(aux_ptr->c_slist_ptr);
        } 

        if (aux_ptr->candidate_slist_ptr != NULL) {
            assert(H5SL_count(aux_ptr->candidate_slist_ptr) == 0);
            H5SL_close(aux_ptr->candidate_slist_ptr);
        } 

        aux_ptr = H5FL_FREE(H5AC_aux_t, aux_ptr);
    }  
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_evict(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    
    if (H5C_evict(f) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "can't evict cache");

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_evict_cache_msg(f->shared->cache, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_expunge_entry(H5F_t *f, const H5AC_class_t *type, haddr_t addr, unsigned flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(type);
    assert(type->serialize);
    assert(H5_addr_defined(addr));

    if (H5C_expunge_entry(f, type, addr, flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTEXPUNGE, FAIL, "H5C_expunge_entry() failed");

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_expunge_entry_msg(f->shared->cache, addr, type->id, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_flush(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

#ifdef H5_HAVE_PARALLEL
    
    if (H5C_clear_coll_entries(f->shared->cache, false) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "H5C_clear_coll_entries() failed");

    
    if (H5AC__flush_entries(f) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't flush");
#endif 

    
    
    if (H5C_flush_cache(f, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't flush cache");

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_flush_cache_msg(f->shared->cache, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_entry_status(const H5F_t *f, haddr_t addr, unsigned *status)
{
    bool   in_cache;     
    bool   is_dirty;     
    bool   is_protected; 
    bool   is_pinned;    
    bool   is_corked;
    bool   is_flush_dep_child;  
    bool   is_flush_dep_parent; 
    bool   image_is_up_to_date; 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if ((f == NULL) || (!H5_addr_defined(addr)) || (status == NULL))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad param(s) on entry");

    if (H5C_get_entry_status(f, addr, NULL, &in_cache, &is_dirty, &is_protected, &is_pinned, &is_corked,
                             &is_flush_dep_parent, &is_flush_dep_child, &image_is_up_to_date) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_entry_status() failed");

    if (in_cache) {
        *status |= H5AC_ES__IN_CACHE;
        if (is_dirty)
            *status |= H5AC_ES__IS_DIRTY;
        if (is_protected)
            *status |= H5AC_ES__IS_PROTECTED;
        if (is_pinned)
            *status |= H5AC_ES__IS_PINNED;
        if (is_corked)
            *status |= H5AC_ES__IS_CORKED;
        if (is_flush_dep_parent)
            *status |= H5AC_ES__IS_FLUSH_DEP_PARENT;
        if (is_flush_dep_child)
            *status |= H5AC_ES__IS_FLUSH_DEP_CHILD;
        if (image_is_up_to_date)
            *status |= H5AC_ES__IMAGE_IS_UP_TO_DATE;
    } 
    else
        *status = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_insert_entry(H5F_t *f, const H5AC_class_t *type, haddr_t addr, void *thing, unsigned int flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(type);
    assert(type->serialize);
    assert(H5_addr_defined(addr));
    assert(thing);

    
    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "no write intent on file");

#if H5AC_DO_TAGGING_SANITY_CHECKS
    if (!H5C_get_ignore_tags(f->shared->cache) && H5AC__verify_tag(type) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "Bad tag value");
#endif 

    
    if (H5C_insert_entry(f, type, addr, thing, flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "H5C_insert_entry() failed");

#ifdef H5_HAVE_PARALLEL
    {
        H5AC_aux_t *aux_ptr;

        if (NULL != (aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(f->shared->cache))) {
            
            if (H5AC__log_inserted_entry((H5AC_info_t *)thing) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTINS, FAIL, "H5AC__log_inserted_entry() failed");

            
            if (aux_ptr->dirty_bytes >= aux_ptr->dirty_bytes_threshold)
                if (H5AC__run_sync_point(f, H5AC_SYNC_POINT_OP__FLUSH_TO_MIN_CLEAN) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't run sync point");
        } 
    }
#endif 

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_insert_entry_msg(f->shared->cache, addr, type->id, flags,
                                           ((H5C_cache_entry_t *)thing)->size, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_load_cache_image_on_next_protect(H5F_t *f, haddr_t addr, hsize_t len, bool rw)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    if (H5C_load_cache_image_on_next_protect(f, addr, len, rw) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTLOAD, FAIL, "call to H5C_load_cache_image_on_next_protect failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_mark_entry_dirty(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    
    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;

#ifdef H5_HAVE_PARALLEL
    {
        H5AC_aux_t *aux_ptr;

        aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(cache_ptr);
        if ((!entry_ptr->is_dirty) && (!entry_ptr->is_protected) && (entry_ptr->is_pinned) &&
            (NULL != aux_ptr))
            if (H5AC__log_dirtied_entry(entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "can't log dirtied entry");
    }
#endif 

    if (H5C_mark_entry_dirty(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "can't mark pinned or protected entry dirty");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_mark_entry_dirty_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_mark_entry_clean(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;

#ifdef H5_HAVE_PARALLEL
    {
        H5AC_aux_t *aux_ptr;

        aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(cache_ptr);
        if ((!entry_ptr->is_dirty) && (!entry_ptr->is_protected) && (entry_ptr->is_pinned) &&
            (NULL != aux_ptr))
            if (H5AC__log_cleaned_entry(entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "can't log cleaned entry");
    }
#endif 

    if (H5C_mark_entry_clean(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKCLEAN, FAIL, "can't mark pinned or protected entry clean");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_mark_entry_clean_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_mark_entry_unserialized(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    
    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;

    if (H5C_mark_entry_unserialized(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKUNSERIALIZED, FAIL, "can't mark entry unserialized");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_mark_unserialized_entry_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_mark_entry_serialized(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;

    if (H5C_mark_entry_serialized(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKSERIALIZED, FAIL, "can't mark entry serialized");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_mark_serialized_entry_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_move_entry(H5F_t *f, const H5AC_class_t *type, haddr_t old_addr, haddr_t new_addr)
{
#ifdef H5_HAVE_PARALLEL
    H5AC_aux_t *aux_ptr;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared->cache);
    assert(type);
    assert(H5_addr_defined(old_addr));
    assert(H5_addr_defined(new_addr));
    assert(H5_addr_ne(old_addr, new_addr));

#ifdef H5_HAVE_PARALLEL
    
    if (NULL != (aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(f->shared->cache)))
        if (H5AC__log_moved_entry(f, old_addr, new_addr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "can't log moved entry");
#endif 

    if (H5C_move_entry(f->shared->cache, type, old_addr, new_addr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTMOVE, FAIL, "H5C_move_entry() failed");

#ifdef H5_HAVE_PARALLEL
    
    if (NULL != aux_ptr && aux_ptr->dirty_bytes >= aux_ptr->dirty_bytes_threshold)
        if (H5AC__run_sync_point(f, H5AC_SYNC_POINT_OP__FLUSH_TO_MIN_CLEAN) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't run sync point");
#endif 

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_move_entry_msg(f->shared->cache, old_addr, new_addr, type->id, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_pin_protected_entry(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (H5C_pin_protected_entry(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTPIN, FAIL, "can't pin entry");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_pin_entry_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_prep_for_file_close(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    if (H5C_prep_for_file_close(f) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "cache prep for file close failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_prep_for_file_flush(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    if (H5C_set_slist_enabled(f->shared->cache, true, true) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't enable skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_secure_from_file_flush(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    if (H5C_set_slist_enabled(f->shared->cache, false, false) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "can't disable skip list");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_create_flush_dependency(void *parent_thing, void *child_thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(parent_thing);
    assert(child_thing);

    entry_ptr = (H5AC_info_t *)parent_thing;
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (H5C_create_flush_dependency(parent_thing, child_thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "H5C_create_flush_dependency() failed");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_create_fd_msg(cache_ptr, (H5AC_info_t *)parent_thing,
                                            (H5AC_info_t *)child_thing, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5AC_protect(H5F_t *f, const H5AC_class_t *type, haddr_t addr, void *udata, unsigned flags)
{
    void *thing     = NULL; 
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(type);
    assert(type->serialize);
    assert(H5_addr_defined(addr));

    
#ifdef H5_HAVE_PARALLEL
    assert(0 == (flags &
                 (unsigned)(~(H5C__READ_ONLY_FLAG | H5C__FLUSH_LAST_FLAG | H5C__FLUSH_COLLECTIVELY_FLAG))));
#else  
    assert(0 == (flags & (unsigned)(~(H5C__READ_ONLY_FLAG | H5C__FLUSH_LAST_FLAG))));
#endif 

    
    if ((0 == (H5F_INTENT(f) & H5F_ACC_RDWR)) && (0 == (flags & H5C__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, NULL, "no write intent on file");

#if H5AC_DO_TAGGING_SANITY_CHECKS
    if (!H5C_get_ignore_tags(f->shared->cache) && H5AC__verify_tag(type) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, NULL, "Bad tag value");
#endif 

    if (NULL == (thing = H5C_protect(f, type, addr, udata, flags)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTPROTECT, NULL, "H5C_protect() failed");

    
    ret_value = thing;

done:
    
    {
        herr_t fake_ret_value = (NULL == ret_value) ? FAIL : SUCCEED;

        if (f->shared->cache->log_info->logging)
            if (H5C_log_write_protect_entry_msg(f->shared->cache, (H5AC_info_t *)thing, type->id, flags,
                                                fake_ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, NULL, "unable to emit log message");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_resize_entry(void *thing, size_t new_size)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

#ifdef H5_HAVE_PARALLEL
    

    if ((!entry_ptr->is_dirty) && (entry_ptr->size != new_size)) {

        
        H5AC_aux_t *aux_ptr;

        aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(cache_ptr);

        if (NULL != aux_ptr) {

            if (H5AC__log_dirtied_entry(entry_ptr) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTMARKDIRTY, FAIL, "can't log dirtied entry");
        }
    }
#endif 

    
    if (H5C_resize_entry(thing, new_size) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTRESIZE, FAIL, "can't resize entry");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_resize_entry_msg(cache_ptr, entry_ptr, new_size, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_unpin_entry(void *thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);

    entry_ptr = (H5AC_info_t *)thing;
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (H5C_unpin_entry(thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "can't unpin entry");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_unpin_entry_msg(cache_ptr, entry_ptr, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_destroy_flush_dependency(void *parent_thing, void *child_thing)
{
    H5AC_info_t *entry_ptr = NULL;    
    H5C_t       *cache_ptr = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(parent_thing);
    assert(child_thing);

    entry_ptr = (H5AC_info_t *)parent_thing;
    cache_ptr = entry_ptr->cache_ptr;
    assert(cache_ptr);

    
    if (H5C_destroy_flush_dependency(parent_thing, child_thing) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL, "H5C_destroy_flush_dependency() failed");

done:
    
    if (cache_ptr != NULL && cache_ptr->log_info != NULL)
        if (cache_ptr->log_info->logging)
            if (H5C_log_write_destroy_fd_msg(cache_ptr, (H5AC_info_t *)parent_thing,
                                             (H5AC_info_t *)child_thing, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_unprotect(H5F_t *f, const H5AC_class_t *type, haddr_t addr, void *thing, unsigned flags)
{
    bool dirtied;
    bool deleted;
#ifdef H5_HAVE_PARALLEL
    H5AC_aux_t *aux_ptr = NULL;
#endif                          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(type);
    assert(type->deserialize);
    assert(type->image_len);
    assert(H5_addr_defined(addr));
    assert(thing);
    assert(((H5AC_info_t *)thing)->addr == addr);
    assert(((H5AC_info_t *)thing)->type == type);

    dirtied =
        (bool)(((flags & H5AC__DIRTIED_FLAG) == H5AC__DIRTIED_FLAG) || (((H5AC_info_t *)thing)->dirtied));
    deleted = (bool)((flags & H5C__DELETED_FLAG) == H5C__DELETED_FLAG);

    
    if (dirtied && !deleted) {
        size_t curr_size = 0;

        if ((type->image_len)(thing, &curr_size) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTGETSIZE, FAIL, "Can't get size of thing");

        if (((H5AC_info_t *)thing)->size != curr_size)
            HGOTO_ERROR(H5E_CACHE, H5E_BADSIZE, FAIL, "size of entry changed");
    } 

#ifdef H5_HAVE_PARALLEL
    if (NULL != (aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(f->shared->cache))) {
        if (dirtied && ((H5AC_info_t *)thing)->is_dirty == false)
            if (H5AC__log_dirtied_entry((H5AC_info_t *)thing) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "can't log dirtied entry");

        if (deleted && aux_ptr->mpi_rank == 0) {
            if (H5AC__log_deleted_entry((H5AC_info_t *)thing) < 0) {
                
                HDONE_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "H5AC__log_deleted_entry() failed");
            }
        }
    }  
#endif 

    if (H5C_unprotect(f, addr, thing, flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPROTECT, FAIL, "H5C_unprotect() failed");

#ifdef H5_HAVE_PARALLEL
    
    if ((aux_ptr != NULL) && (aux_ptr->dirty_bytes >= aux_ptr->dirty_bytes_threshold))
        if (H5AC__run_sync_point(f, H5AC_SYNC_POINT_OP__FLUSH_TO_MIN_CLEAN) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Can't run sync point");
#endif 

done:
    
    if (f->shared->cache->log_info->logging)
        if (H5C_log_write_unprotect_entry_msg(f->shared->cache, addr, type->id, flags, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_cache_auto_resize_config(const H5AC_t *cache_ptr, H5AC_cache_config_t *config_ptr)
{
    H5C_auto_size_ctl_t internal_config;
    bool                evictions_enabled;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if ((cache_ptr == NULL) || (config_ptr == NULL) ||
        (config_ptr->version != H5AC__CURR_CACHE_CONFIG_VERSION))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr or config_ptr on entry");

    
    if (H5C_get_cache_auto_resize_config((const H5C_t *)cache_ptr, &internal_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_cache_auto_resize_config() failed");
    if (H5C_get_evictions_enabled((const H5C_t *)cache_ptr, &evictions_enabled) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_resize_enabled() failed");

    
    if (internal_config.rpt_fcn == NULL)
        config_ptr->rpt_fcn_enabled = false;
    else
        config_ptr->rpt_fcn_enabled = true;
    config_ptr->open_trace_file        = false;
    config_ptr->close_trace_file       = false;
    config_ptr->trace_file_name[0]     = '\0';
    config_ptr->evictions_enabled      = evictions_enabled;
    config_ptr->set_initial_size       = internal_config.set_initial_size;
    config_ptr->initial_size           = internal_config.initial_size;
    config_ptr->min_clean_fraction     = internal_config.min_clean_fraction;
    config_ptr->max_size               = internal_config.max_size;
    config_ptr->min_size               = internal_config.min_size;
    config_ptr->epoch_length           = (long)(internal_config.epoch_length);
    config_ptr->incr_mode              = internal_config.incr_mode;
    config_ptr->lower_hr_threshold     = internal_config.lower_hr_threshold;
    config_ptr->increment              = internal_config.increment;
    config_ptr->apply_max_increment    = internal_config.apply_max_increment;
    config_ptr->max_increment          = internal_config.max_increment;
    config_ptr->decr_mode              = internal_config.decr_mode;
    config_ptr->upper_hr_threshold     = internal_config.upper_hr_threshold;
    config_ptr->flash_incr_mode        = internal_config.flash_incr_mode;
    config_ptr->flash_multiple         = internal_config.flash_multiple;
    config_ptr->flash_threshold        = internal_config.flash_threshold;
    config_ptr->decrement              = internal_config.decrement;
    config_ptr->apply_max_decrement    = internal_config.apply_max_decrement;
    config_ptr->max_decrement          = internal_config.max_decrement;
    config_ptr->epochs_before_eviction = (int)(internal_config.epochs_before_eviction);
    config_ptr->apply_empty_reserve    = internal_config.apply_empty_reserve;
    config_ptr->empty_reserve          = internal_config.empty_reserve;
#ifdef H5_HAVE_PARALLEL
    {
        H5AC_aux_t *aux_ptr;

        if (NULL != (aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(cache_ptr))) {
            config_ptr->dirty_bytes_threshold   = aux_ptr->dirty_bytes_threshold;
            config_ptr->metadata_write_strategy = aux_ptr->metadata_write_strategy;
        } 
        else {
#endif 
            config_ptr->dirty_bytes_threshold   = H5AC__DEFAULT_DIRTY_BYTES_THRESHOLD;
            config_ptr->metadata_write_strategy = H5AC__DEFAULT_METADATA_WRITE_STRATEGY;
#ifdef H5_HAVE_PARALLEL
        } 
    }
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_cache_size(const H5AC_t *cache_ptr, size_t *max_size_ptr, size_t *min_clean_size_ptr,
                    size_t *cur_size_ptr, uint32_t *cur_num_entries_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5C_get_cache_size((const H5C_t *)cache_ptr, max_size_ptr, min_clean_size_ptr, cur_size_ptr,
                           cur_num_entries_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_cache_size() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_cache_flush_in_progress(H5AC_t *cache_ptr, bool *flush_in_progress_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5C_get_cache_flush_in_progress((H5C_t *)cache_ptr, flush_in_progress_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_cache_flush_in_progress() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_cache_hit_rate(const H5AC_t *cache_ptr, double *hit_rate_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5C_get_cache_hit_rate((const H5C_t *)cache_ptr, hit_rate_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_get_cache_hit_rate() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_reset_cache_hit_rate_stats(H5AC_t *cache_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5C_reset_cache_hit_rate_stats((H5C_t *)cache_ptr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_reset_cache_hit_rate_stats() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_set_cache_auto_resize_config(H5AC_t *cache_ptr, const H5AC_cache_config_t *config_ptr)
{
    H5C_auto_size_ctl_t internal_config;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cache_ptr);

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "bad cache_ptr on entry");

    
    if (H5AC_validate_config(config_ptr) != SUCCEED)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Bad cache configuration");

    
    
    if (config_ptr->close_trace_file)
        if (H5C_log_tear_down((H5C_t *)cache_ptr) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "mdc logging tear-down failed");

    
    if (config_ptr->open_trace_file) {
        
        if (H5C_log_set_up((H5C_t *)cache_ptr, config_ptr->trace_file_name, H5C_LOG_STYLE_TRACE, true) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "mdc logging setup failed");
    }

    
    if (H5AC__ext_config_2_int_config(config_ptr, &internal_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5AC__ext_config_2_int_config() failed");

    
    if (H5C_set_cache_auto_resize_config(cache_ptr, &internal_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_set_cache_auto_resize_config() failed");
    if (H5C_set_evictions_enabled(cache_ptr, config_ptr->evictions_enabled) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_set_evictions_enabled() failed");

#ifdef H5_HAVE_PARALLEL
    {
        H5AC_aux_t *aux_ptr;

        
        
        if (NULL != (aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(cache_ptr))) {
            aux_ptr->dirty_bytes_threshold   = config_ptr->dirty_bytes_threshold;
            aux_ptr->metadata_write_strategy = config_ptr->metadata_write_strategy;
        } 
    }
#endif 

done:
    
    if (cache_ptr->log_info->logging)
        if (H5C_log_write_set_cache_config_msg(cache_ptr, config_ptr, ret_value) < 0)
            HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_validate_config(const H5AC_cache_config_t *config_ptr)
{
    H5C_auto_size_ctl_t internal_config;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (config_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "NULL config_ptr on entry");
    if (config_ptr->version != H5AC__CURR_CACHE_CONFIG_VERSION)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Unknown config version");

    
    if (config_ptr->open_trace_file) {
        size_t name_len;

        
        name_len = strlen(config_ptr->trace_file_name);
        if (name_len == 0)
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "config_ptr->trace_file_name is empty");
        else if (name_len > H5AC__MAX_TRACE_FILE_NAME_LEN)
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "config_ptr->trace_file_name too long");
    } 

    if ((config_ptr->evictions_enabled == false) &&
        ((config_ptr->incr_mode != H5C_incr__off) || (config_ptr->flash_incr_mode != H5C_flash_incr__off) ||
         (config_ptr->decr_mode != H5C_decr__off)))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Can't disable evictions while auto-resize is enabled");

    if (config_ptr->dirty_bytes_threshold < H5AC__MIN_DIRTY_BYTES_THRESHOLD)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "dirty_bytes_threshold too small");
    else if (config_ptr->dirty_bytes_threshold > H5AC__MAX_DIRTY_BYTES_THRESHOLD)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "dirty_bytes_threshold too big");

    if ((config_ptr->metadata_write_strategy != H5AC_METADATA_WRITE_STRATEGY__PROCESS_0_ONLY) &&
        (config_ptr->metadata_write_strategy != H5AC_METADATA_WRITE_STRATEGY__DISTRIBUTED))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "config_ptr->metadata_write_strategy out of range");

    if (H5AC__ext_config_2_int_config(config_ptr, &internal_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5AC__ext_config_2_int_config() failed");

    if (H5C_validate_resize_config(&internal_config, H5C_RESIZE_CFG__VALIDATE_ALL) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "error(s) in new config");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_validate_cache_image_config(H5AC_cache_image_config_t *config_ptr)
{
    H5C_cache_image_ctl_t internal_config = H5C__DEFAULT_CACHE_IMAGE_CTL;
    herr_t                ret_value       = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (config_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "NULL config_ptr on entry");

    if (config_ptr->version != H5AC__CURR_CACHE_IMAGE_CONFIG_VERSION)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "Unknown image config version");

    
    internal_config.version            = config_ptr->version;
    internal_config.generate_image     = config_ptr->generate_image;
    internal_config.save_resize_status = config_ptr->save_resize_status;
    internal_config.entry_ageout       = config_ptr->entry_ageout;

    if (H5C_validate_cache_image_config(&internal_config) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "error(s) in new cache image config");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5AC__check_if_write_permitted(const H5F_t
#ifndef H5_HAVE_PARALLEL
                                   H5_ATTR_UNUSED
#endif 
                                       *f,
                               bool    *write_permitted_ptr)
{
#ifdef H5_HAVE_PARALLEL
    H5AC_aux_t *aux_ptr = NULL;
#endif 
    bool write_permitted = true;

    FUNC_ENTER_PACKAGE_NOERR

#ifdef H5_HAVE_PARALLEL
    
    assert(f != NULL);
    assert(f->shared != NULL);
    assert(f->shared->cache != NULL);
    aux_ptr = (H5AC_aux_t *)H5C_get_aux_ptr(f->shared->cache);
    if (aux_ptr != NULL) {
        if ((aux_ptr->mpi_rank == 0) ||
            (aux_ptr->metadata_write_strategy == H5AC_METADATA_WRITE_STRATEGY__DISTRIBUTED))
            write_permitted = aux_ptr->write_permitted;
        else
            write_permitted = false;
    }  
#endif 

    *write_permitted_ptr = write_permitted;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5AC__ext_config_2_int_config(const H5AC_cache_config_t *ext_conf_ptr, H5C_auto_size_ctl_t *int_conf_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if ((ext_conf_ptr == NULL) || (ext_conf_ptr->version != H5AC__CURR_CACHE_CONFIG_VERSION) ||
        (int_conf_ptr == NULL))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad ext_conf_ptr or inf_conf_ptr on entry");

    int_conf_ptr->version = H5C__CURR_AUTO_SIZE_CTL_VER;
    if (ext_conf_ptr->rpt_fcn_enabled)
        int_conf_ptr->rpt_fcn = H5C_def_auto_resize_rpt_fcn;
    else
        int_conf_ptr->rpt_fcn = NULL;

    int_conf_ptr->set_initial_size   = ext_conf_ptr->set_initial_size;
    int_conf_ptr->initial_size       = ext_conf_ptr->initial_size;
    int_conf_ptr->min_clean_fraction = ext_conf_ptr->min_clean_fraction;
    int_conf_ptr->max_size           = ext_conf_ptr->max_size;
    int_conf_ptr->min_size           = ext_conf_ptr->min_size;
    int_conf_ptr->epoch_length       = (int64_t)(ext_conf_ptr->epoch_length);

    int_conf_ptr->incr_mode           = ext_conf_ptr->incr_mode;
    int_conf_ptr->lower_hr_threshold  = ext_conf_ptr->lower_hr_threshold;
    int_conf_ptr->increment           = ext_conf_ptr->increment;
    int_conf_ptr->apply_max_increment = ext_conf_ptr->apply_max_increment;
    int_conf_ptr->max_increment       = ext_conf_ptr->max_increment;
    int_conf_ptr->flash_incr_mode     = ext_conf_ptr->flash_incr_mode;
    int_conf_ptr->flash_multiple      = ext_conf_ptr->flash_multiple;
    int_conf_ptr->flash_threshold     = ext_conf_ptr->flash_threshold;

    int_conf_ptr->decr_mode              = ext_conf_ptr->decr_mode;
    int_conf_ptr->upper_hr_threshold     = ext_conf_ptr->upper_hr_threshold;
    int_conf_ptr->decrement              = ext_conf_ptr->decrement;
    int_conf_ptr->apply_max_decrement    = ext_conf_ptr->apply_max_decrement;
    int_conf_ptr->max_decrement          = ext_conf_ptr->max_decrement;
    int_conf_ptr->epochs_before_eviction = (int32_t)(ext_conf_ptr->epochs_before_eviction);
    int_conf_ptr->apply_empty_reserve    = ext_conf_ptr->apply_empty_reserve;
    int_conf_ptr->empty_reserve          = ext_conf_ptr->empty_reserve;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_ignore_tags(const H5F_t *f)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);

    
    if (H5C_ignore_tags(f->shared->cache) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL, "H5C_ignore_tags() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5AC_tag(haddr_t metadata_tag, haddr_t *prev_tag)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    if (prev_tag)
        *prev_tag = H5CX_get_tag();

    
    H5CX_set_tag(metadata_tag);

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5AC_retag_copied_metadata(const H5F_t *f, haddr_t metadata_tag)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    if (H5C_retag_entries(f->shared->cache, H5AC__COPIED_TAG, metadata_tag) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTSET, FAIL, "Can't retag metadata");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_flush_tagged_metadata(H5F_t *f, haddr_t metadata_tag)
{
    
    herr_t ret_value = SUCCEED;

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    if (H5C_flush_tagged_entries(f, metadata_tag) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Cannot flush metadata");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_evict_tagged_metadata(H5F_t *f, haddr_t metadata_tag, bool match_global)
{
    
    herr_t ret_value = SUCCEED;

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    if (H5C_evict_tagged_entries(f, metadata_tag, match_global) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Cannot evict metadata");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_expunge_tag_type_metadata(H5F_t *f, haddr_t tag, int type_id, unsigned flags)
{
    
    herr_t ret_value = SUCCEED;

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);

    
    if (H5C_expunge_tag_type_metadata(f, tag, type_id, flags) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "Cannot expunge tagged type entries");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_tag(const void *thing, haddr_t *tag)
{
    
    herr_t ret_value = SUCCEED;

    
    FUNC_ENTER_NOAPI(FAIL)

    
    assert(thing);
    assert(tag);

    
    if (H5C_get_tag(thing, tag) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTTAG, FAIL, "Cannot get tag for metadata cache entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_cork(H5F_t *f, haddr_t obj_addr, unsigned action, bool *corked)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    assert(f->shared->cache);
    assert(H5_addr_defined(obj_addr));
    assert(action == H5AC__SET_CORK || action == H5AC__UNCORK || action == H5AC__GET_CORKED);

    
    if (action == H5AC__GET_CORKED) {
        assert(corked);
        if (H5C_get_num_objs_corked(f->shared->cache) == 0) {
            *corked = false;
            HGOTO_DONE(SUCCEED);
        }
    }

    if (H5C_cork(f->shared->cache, obj_addr, action, corked) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Cannot perform the cork action");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#if H5AC_DO_TAGGING_SANITY_CHECKS

static herr_t
H5AC__verify_tag(const H5AC_class_t *type)
{
    haddr_t tag;                 
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    tag = H5CX_get_tag();

    
    if (H5C_verify_tag(type->id, tag) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "tag verification failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5AC_get_entry_ring(const H5F_t *f, haddr_t addr, H5AC_ring_t *ring)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(ring);

    
    if (H5C_get_entry_ring(f, addr, ring) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "Can't retrieve ring for entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5AC_set_ring(H5AC_ring_t ring, H5AC_ring_t *orig_ring)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    

    
    if (orig_ring)
        *orig_ring = H5CX_get_ring();

    
    H5CX_set_ring(ring);

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5AC_unsettle_entry_ring(void *_entry)
{
    H5AC_info_t *entry     = (H5AC_info_t *)_entry; 
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);

    
    if (H5C_unsettle_entry_ring(entry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_unsettle_ring(H5F_t *f, H5C_ring_t ring)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (FAIL == (ret_value = H5C_unsettle_ring(f, ring)))
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "H5C_unsettle_ring() failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_remove_entry(void *_entry)
{
    H5AC_info_t *entry     = (H5AC_info_t *)_entry; 
    H5C_t       *cache     = NULL;                  
    herr_t       ret_value = SUCCEED;               

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(entry);
    cache = entry->cache_ptr;
    assert(cache);

    
    if (H5C_remove_entry(entry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "can't remove entry");

done:
    
    if (cache != NULL && cache->log_info != NULL)
        if (cache->log_info->logging)
            if (H5C_log_write_remove_entry_msg(cache, entry, ret_value) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_LOGGING, FAIL, "unable to emit log message");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_get_mdc_image_info(const H5AC_t *cache_ptr, haddr_t *image_addr, hsize_t *image_len)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (H5C_get_mdc_image_info((const H5C_t *)cache_ptr, image_addr, image_len) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "can't retrieve cache image info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
