% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/dose_fit.R, R/dose_predict.R
\docType{methods}
\name{doserate}
\alias{doserate}
\alias{dose_fit}
\alias{dose_fit-method}
\alias{dose_predict}
\alias{dose_predict-method}
\alias{dose_fit,GammaSpectra,GammaSpectrumOrNumeric,matrix-method}
\alias{dose_fit,GammaSpectra,GammaSpectrum,matrix-method}
\alias{dose_fit,GammaSpectra,GammaSpectrumOrNumeric,data.frame-method}
\alias{dose_fit,GammaSpectra,data.frame-method}
\alias{dose_predict,CalibrationCurve,missing-method}
\alias{dose_predict,CalibrationCurve,GammaSpectrum-method}
\alias{dose_predict,CalibrationCurve,GammaSpectra-method}
\title{Dose Rate Estimation}
\usage{
dose_fit(object, background, doses, ...)

dose_predict(object, spectrum, ...)

\S4method{dose_fit}{GammaSpectra,GammaSpectrumOrNumeric,matrix}(
  object,
  background,
  doses,
  range_Ni,
  range_NiEi,
  details = list(authors = "", date = Sys.time())
)

\S4method{dose_fit}{GammaSpectra,GammaSpectrumOrNumeric,data.frame}(
  object,
  background,
  doses,
  range_Ni,
  range_NiEi,
  details = list(authors = "", date = Sys.time())
)

\S4method{dose_predict}{CalibrationCurve,missing}(
  object,
  sigma = 1,
  epsilon = 0.015,
  water_content = NULL,
  use_MC = FALSE
)

\S4method{dose_predict}{CalibrationCurve,GammaSpectrum}(
  object,
  spectrum,
  sigma = 1,
  epsilon = 0.015,
  water_content = NULL,
  use_MC = FALSE
)

\S4method{dose_predict}{CalibrationCurve,GammaSpectra}(
  object,
  spectrum,
  sigma = 1,
  epsilon = 0.015,
  water_content = NULL,
  use_MC = FALSE
)
}
\arguments{
\item{object}{A \linkS4class{GammaSpectra} or \linkS4class{CalibrationCurve} object.}

\item{background}{A \linkS4class{GammaSpectrum} object or a length-two \code{\link{numeric}}
vector giving the background noise integration value and error,
respectively. If no background subtraction is wanted, you can set \code{background = c(0,0,)}}

\item{doses}{A \code{\link{matrix}} or \code{\link{data.frame}} object with gamma dose values and uncertainties.
The row names must match the names of the spectrum.}

\item{...}{Currently not used.}

\item{spectrum}{An optional \linkS4class{GammaSpectrum} or \linkS4class{GammaSpectra}
object in which to look for variables with which to predict. If omitted,
the fitted values are used.}

\item{range_Ni, range_NiEi}{A length-two \code{\link{numeric}} vector giving the energy
range to integrate within (in keV).}

\item{details}{A \code{\link{list}} of length-one vector specifying additional
informations about the instrument for which the curve is built.}

\item{sigma}{A \code{\link{numeric}} value giving the confidence level of which the error from the
slope is considered in the final uncertainty calculation}

\item{epsilon}{A \code{\link{numeric}} value giving an extra relative error term,
introduced by the calibration of the energy scale of the spectrum,
e.g., \code{0.015} for an additional 1.5\% error}

\item{water_content}{\code{\link{numeric}} or \code{\link{matrix}} gravimetric field water content to correct
the gamma-dose rate to using the correction factor by Aitken (1985) to obtain the dry gamma-dose
rate. Example: \code{c(0.05,0.0001)} for water content of 5\% +/- 0.01 \%. The default is \code{NULL} (nothing is corrected).
The correction only works on the final dose rate. For more information see details.}

\item{use_MC}{A \code{\link{logical}} parameter, enabling/disabling Monte Carlo simulations for estimating
the dose rate uncertainty}
}
\value{
\itemize{
\item \code{dose_fit()} returns a \linkS4class{CalibrationCurve} object.
\item \code{dose_predict()} returns a \code{\link{data.frame}} with the following columns:
\describe{
\item{\code{name}}{(\code{\link{character}}) the name of the spectra.}
\item{\code{signal_Ni}}{(\code{\link{numeric}}) the integrated signal value (according to
the value of \code{threshold}; see \code{\link[=signal_integrate]{signal_integrate()}}) for \code{energy = FALSE}}
\item{\code{signal_err_Ni}}{(\code{\link{numeric}}) the integrated signal error value
(according to the value of \code{threshold}; see \code{\link[=signal_integrate]{signal_integrate()}}) for \code{energy = FALSE}.}
\item{\code{dose_Ni}}{(\code{\link{numeric}}) the predicted gamma dose rate for \code{energy = FALSE}.}
\item{\code{dose_err_Ni}}{(\code{\link{numeric}}) the predicted gamma dose rate error for \code{energy = FALSE}.}
\item{\code{signal_Ni}}{(\code{\link{numeric}}) the integrated signal value (according to
the value of \code{threshold}; see \code{\link[=signal_integrate]{signal_integrate()}}).}
\item{\code{signal_err_NiEi}}{(\code{\link{numeric}}) the integrated signal error value
(according to the value of \code{threshold}; see \code{\link[=signal_integrate]{signal_integrate()}}) for \code{energy = TRUE}.}
\item{\code{dose_NiEi}}{(\code{\link{numeric}}) the predicted gamma dose rate for \code{energy = TRUE}.}
\item{\code{dose_err_NiEi}}{(\code{\link{numeric}}) the predicted gamma dose rate error for \code{energy = TRUE}.}
\item{\code{dose_final}}{(\code{\link{numeric}}) the predicted final gamma dose rate as the mean of \code{dose_Ni} and \code{dose_NiEi}}
\item{\code{dose_err_final}}{(\code{\link{numeric}}) the predicted final gamma dose rate error as
\eqn{SE(\dot{D}_{\gamma}) = \sqrt{(\frac{SE(\dot{D}_{\gamma\mathrm{Ni}})}{\dot{D}_{\gamma\mathrm{Ni}}})^2 +
   (\frac{SE(\dot{D}_{\gamma\mathrm{NiEi}})}{\dot{D}_{\gamma\mathrm{NiEi}}})^2}}}
}
}
}
\description{
\itemize{
\item \code{dose_fit()} builds a calibration curve for gamma dose rate estimation.
\item \code{dose_predict()} predicts in situ gamma dose rate.
}
}
\details{
To estimate the gamma dose rate, one of the calibration curves distributed
with this package can be used. These built-in curves are in use in several
luminescence dating laboratories and can be used to replicate published
results. As these curves are instrument specific, the user may have to build
its own curve.

The construction of a calibration curve requires a set of reference spectra
for which the gamma dose rate is known and a background noise measurement.
First, each reference spectrum is integrated over a given interval, then
normalized to active time and corrected for background noise. The dose rate
is finally modelled by the integrated signal value used as a linear
predictor (York \emph{et al.}, 2004).
}
\note{
See \code{vignette(doserate)} for a reproducible example.
}
\section{Uncertainty calculation of the gamma-dose rate}{

The analytical uncertainties of the final gamma-dose rate (\eqn{SE(\dot{D}_{\gamma})}) are calculated as
follows:

\deqn{
 \sigma_{\dot{D_\gamma}} = \sqrt((\frac{m_{\delta}s}{m})^2 + (\frac{s_{\delta}}{s})^2 + \epsilon^2)
 }

with \eqn{m} and \eqn{m_{\delta}} being the slope of the fit an its uncertainty,
\eqn{\sigma} the error scaler for the slope uncertainty, \eqn{s} and \eqn{s_{\delta}}
the integrated signal and its uncertainty, and \eqn{\epsilon} an additional relative uncertainty
term that can be set by the user using the argument \code{epsilon}.

If the parameter \code{use_MC} is set to \code{TRUE}, the a Monte Carlo sampling approach
is chosen to approximate the uncertainties on the dose rate:

\deqn{
   \sigma_{\dot{D_\gamma}} :=
   \sqrt((\frac{SD(\mathcal{N}(\mu_{slope}, \sigma_{slope}) \times \mathcal{N}(\mu_{signal}, \sigma_{signal}) +
   \mathcal{N}(\mu_{intercept}, \sigma_{intercept})) * \rho}{\dot{D_\gamma}})^2 +
   \epsilon^2) * \dot{D_\gamma}
 }

\eqn{\ rho} is the parameter \code{sigma} provided with the function call, \eqn{SD} equals the
the call to \code{sd()}, i.e. the calculation of the standard deviation. To achieve a good
Gaussian normal approximation with sample 1+e06 times (the values is fixed).
}

\section{Water content correction}{

If gamma-dose rates are measured in the field, they are measured at  "as-is"
conditions. In dating studies, however, using the dry dose rate is often
more desirable to model the long-term effect of different assumptions for
the water content. If the parameter \code{water_content}, either as \link{numeric}
vector or as \link{matrix} with the number of rows equal to the number of
processed spectra,  if different values are desired, the \strong{final}
gamma-dose rate is corrected for the water content provided. Final
uncertainties are obtained using the square root of the summed squared
relative uncertainties of the dose rate and the water content.

A word of caution: When estimating the water content in the laboratory,
the water analytical uncertainty is usually minimal, and it does not make
sense to correct with a relative water content of, e.g., c(0.02,0.02)
(2\% +/- 2\%) because this massively inflates the final dose rate error.
}

\examples{
## Import CNF files
## Spectra
spc_dir <- system.file("extdata/BDX_LaBr_1/calibration", package = "gamma")
spc <- read(spc_dir)

## Background
bkg_dir <- system.file("extdata/BDX_LaBr_1/background", package = "gamma")
bkg <- read(bkg_dir)

## Get dose rate values
data("clermont")
(doses <- clermont[, c("gamma_dose", "gamma_error")])

## Build the calibration curve
calib_curve <- dose_fit(spc, bkg, doses,
                        range_Ni = c(300, 2800),
                        range_NiEi = c(165, 2800))

## Plot the curve
plot(calib_curve, threshold = "Ni")

## Estimate gamma dose rates
dose_predict(calib_curve, spc)
}
\references{
Aitken, M.J. (1985). \emph{Thermoluminescence dating}. London: Academic Press.

Mercier, N. & Falguères, C. (2007). Field Gamma Dose-Rate Measurement with
a NaI(Tl) Detector: Re-Evaluation of the "Threshold" Technique.
\emph{Ancient TL}, 25(1), p. 1-4.

York, D., Evensen, N. M., Martínez, M. L. & De Basabe Delgado, J. (2004).
Unified Equations for the Slope, Intercept, and Standard Errors of the Best
Straight Line. \emph{American Journal of Physics}, 72(3), p. 367-75.
\doi{10.1119/1.1632486}.
}
\seealso{
\code{\link[=signal_integrate]{signal_integrate()}}
}
\author{
N. Frerebeau
}
\concept{dose rate}
