#' Fractionally Integrated ARMA Model
#'
#' This function applies fractional differencing and fits an ARMA model to time series data.
#'
#' @param ts A time series object (class `ts`).
#' @param p Integer; the AR order.
#' @param d Numeric; the degree of fractional differencing.
#' @param q Integer; the MA order.
#' @param s Numeric; the proportion of the data to be used for training.
#' @return A list containing the model summary, fitted values, and forecasted results.
#' @import forecast
#' @import fracdiff
#' @export
fracARMA <- function(ts, p, d, q, s) {
  # Check if conditions are met
  if (d < 1 & s > 0.1 & s < 1 & p < 5 & q < 5) {
    # Apply fractional differencing
    diff_data <- fracdiff::diffseries(ts, d)

    # Calculate number of training points
    n_train <- floor(length(diff_data) * s)

    # Split data into training and testing sets
    train_data1 <- diff_data[1:n_train]
    test_data1 <- diff_data[(n_train + 1):length(diff_data)]

    # Fit ARIMA model on training data
    model <- forecast::Arima(train_data1, order = c(p, 0, q))

    # Forecast the test data
    forecast_result <- forecast::forecast(model, h = length(test_data1))

    # Extract fitted values and model summary
    fitted_values <- model$fitted
    model_summary <- summary(model)

    # Return results as a list
    return(list(
      model_summary = model_summary,
      fitted = fitted_values,
      forecasted = forecast_result
    ))
  } else {
    # Return message if conditions are not met
    return("Invalid variables")
  }
}