#' CSEM of IRT Model via Information
#'
#' @description
#' Compute the CSEM for a unidimensional IRT model using either MLE- or EAP-based test information.
#'
#' @param theta A numeric vector (or object coercible to a numeric vector)
#'   containing the ability values at which to compute CSEM.
#'
#' @param ip A data frame or matrix of item parameters. Columns are interpreted
#'   in the same way as in \code{info()}:
#'   \itemize{
#'     \item 3 columns: \code{b}, \code{a}, \code{c} (3PL; \code{a} on the \code{D = 1.702} metric),
#'     \item 2 columns: \code{b}, \code{a} (2PL; \code{c} internally set to 0),
#'     \item 1 column: \code{b} (1PL/Rasch; \code{a = 1}, \code{c = 0}).
#'   }
#'
#' @param est A character string specifying the estimation method:
#'   \code{"MLE"} for maximum likelihood or \code{"EAP"} for empirical Bayes.
#'
#' @return A list containing:
#'   \itemize{
#'     \item \code{theta} — vector of ability values.
#'     \item \code{csemMLE} — CSEM values for MLE (if \code{est = "MLE"}).
#'     \item \code{csemEAP} — CSEM values for EAP (if \code{est = "EAP"}).
#'   }
#'
#' @export
csem_info <- function(theta, ip, est = c("MLE", "EAP")) {

  # match argument for estimation type ----------------------------------------
  est <- match.arg(est)

  # --- theta checks -----------------------------------------------------------
  if (missing(theta)) {
    stop("`theta` must be supplied as a numeric vector.")
  }

  theta_num <- as.numeric(theta)
  if (!is.numeric(theta_num) || length(theta_num) < 1L) {
    stop("`theta` must be a non-empty numeric vector (or coercible to one).")
  }
  if (any(is.na(theta_num))) {
    stop("`theta` contains NA values after coercion to numeric.")
  }

  # --- item parameter checks --------------------------------------------------
  if (missing(ip)) {
    stop("`ip` must be supplied as a data frame or matrix of item parameters.")
  }
  if (!is.data.frame(ip) && !is.matrix(ip)) {
    stop("`ip` must be a data frame or matrix.")
  }

  ip <- as.data.frame(ip)
  if (!all(vapply(ip, is.numeric, logical(1L)))) {
    stop("All columns in `ip` must be numeric.")
  }
  if (nrow(ip) < 1L) {
    stop("`ip` must contain at least one item (one row).")
  }

  # --- compute information using your `info()` function -----------------------
  info_res <- info(theta = theta_num, ip = ip, est = est)
  info_df  <- as.data.frame(info_res)

  # --- compute CSEM -----------------------------------------------------------
  if (est == "MLE") {

    if (!"infoMLE" %in% names(info_df)) {
      stop("`info()` must return `infoMLE` when est = 'MLE'.")
    }

    csemMLE <- sqrt(1 / info_df$infoMLE)

    return(list(theta = info_df$theta, csemMLE = csemMLE))

  } else {  # est == "EAP"

    if (!"infoEAP" %in% names(info_df)) {
      stop("`info()` must return `infoEAP` when est = 'EAP'.")
    }

    csemEAP <- sqrt(1 / info_df$infoEAP)

    return(list(theta = info_df$theta, csemEAP = csemEAP))
  }
}
