#' Calculate Multifunctionality Regularity (MFreg)
#'
#' @description
#' This function calculates the Multifunctionality Regularity (MFreg) for each row in a dataset.
#' MFreg quantifies the evenness of ecosystem function distribution using a normalized Shannon
#' entropy index. Higher values indicate more even distribution of functions, while lower values
#' suggest dominance by fewer functions.
#'
#' @param data A numeric data frame or matrix where rows represent observations (e.g., sites, plots)
#'   and columns represent different ecosystem functions. Function values should be non-negative.
#' @param weights A numeric vector of weights for each function (column) in the data.
#'   If NULL (default), equal weights of 1 are assigned to all functions.
#'   The length of the weights vector must match the number of columns in data.
#'
#' @return A data frame with a single column named "MFreg" containing the calculated
#'   Multifunctionality Regularity values for each row in the input data.
#'   Row names are preserved from the input data if available.
#'
#' @details
#' Multifunctionality Regularity (MFreg) quantifies the evenness of function distribution
#' in an ecosystem. It is calculated as:
#'
#' \deqn{MFreg = \frac{-\sum_{i=1}^{n}\frac{w_i f_i}{\sum_{i=1}^{n}w_i f_i}\ln{\frac{w_i f_i}{\sum_{i=1}^{n}w_i f_i}}}{\ln(n)}}
#' where fi represents the normalized performance level of function i, wi is the weight of function i, and n is the total number of functions examined.
#'
#' @note
#' \itemize{
#'   \item Functions with zero or negative weighted contributions (\eqn{w_i f_i \leq 0}) are
#'         automatically excluded from calculations
#'   \item If all weighted function values in a row are \eqn{\leq 0}, the result will be NaN
#'   \item Natural logarithm (ln) is used in the calculations
#'   \item The function assumes input data represents normalized or standardized function values
#' }
#'
#' @examples
#' # Example with sample data
#' data <- data.frame(
#'   func1 = c(0.5, 0.3, 0.2),
#'   func2 = c(0.2, 0.4, 0.4),
#'   func3 = c(0.3, 0.3, 0.4)
#' )
#' rownames(data) <- c("Site1", "Site2", "Site3")
#'
#' # Calculate MFreg with equal weights
#' MFreg(data)
#'
#' # Calculate MFreg with custom weights (emphasizing func2)
#' MFreg(data, weights = c(1, 2, 1))
#'
#'
#' @export
MFreg <- function(data, weights = NULL) {
  # If no weights are provided, create a weight vector with all 1's
  if (is.null(weights)) {
    weights <- rep(1, ncol(data))
  }
  if (length(weights) != ncol(data)) {
    stop("The length of the weight vector must be equal to the number of columns in the data frame")
  }
  mfreg_values <- apply(data, 1, function(fi) {
    wi <- weights
    pi <- wi * fi / sum(wi * fi)
    pi <- pi[pi > 0]
    H <- -sum(pi * log(pi))
    S <- length(pi)
    H / log(S)
  })
  result <- data.frame(MFreg = mfreg_values)
  rownames(result) <- rownames(data)
  return(result)
}
