% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powersim.R, R/s3_methods.R
\name{powersim.cord}
\alias{powersim.cord}
\alias{powersim}
\title{Provide power estimates for multivariate abundance models}
\usage{
\method{powersim}{cord}(
  object,
  coeffs,
  term,
  N = nrow(object$obj$data),
  coeffs0 = effect_null(object$obj, term),
  nsim = 1000,
  ncrit = 999,
  test = "score",
  alpha = 0.05,
  newdata = NULL,
  ncores = detectCores() - 1,
  show.time = TRUE,
  long_power = FALSE,
  n.samp = 10,
  nlv = 2
)

powersim(
  object,
  coeffs,
  term,
  N = nrow(object$obj$data),
  coeffs0 = effect_null(object$obj, term),
  nsim = 999,
  ncrit = nsim,
  test = "score",
  alpha = 0.05,
  newdata = NULL,
  ncores = detectCores() - 1,
  show.time = TRUE,
  long_power = FALSE,
  n.samp = 10,
  nlv = 2
)
}
\arguments{
\item{object}{objects of class \code{cord}, typically the result of a call to \code{\link[ecoCopula]{cord}}.}

\item{coeffs}{Coefficient matrix for a \code{\link[mvabund]{manyglm}} object that characterises the size of effects to be simulated.
See \code{\link{effect_alt}} for help in producing this matrix.}

\item{term}{Name of predictor of interest in quotes.}

\item{N}{Number of samples for power estimate. Defaults to the number of observations in the original sample.}

\item{coeffs0}{Coefficient matrix under the null hypothesis. Defaults to being specified by \code{\link{effect_null}}.}

\item{nsim}{Number of simulated test statistics under the specified effect size (\code{coeffs}) to estimate power. Defaults to \code{999}.}

\item{ncrit}{Number of simulated test statistics under the null effect to estimate the critical value. Defaults to \code{999}.}

\item{test}{Test statistic for power estimate to based upon. Defaults to \code{"score"}, however \code{"wald"} is also allowed.}

\item{alpha}{Type I error rate for power estimate, defaults to \code{0.05}.}

\item{newdata}{Data frame of the same size as the original data frame from the \code{\link[ecoCopula]{cord}} object
(\code{object$obj$data}), that specifies a different design of interest.}

\item{ncores}{Number of cores for parallel computing. Defaults to the total number of cores available on the
machine minus 1.}

\item{show.time}{Logical. Displays time elapsed. Defaults to \code{TRUE}.}

\item{long_power}{Logical. Whether to estimate power using separate critical test statistics for each \code{nsim} test statistics
simulated under the alternative hypothesis. Note that although this will give a more accurate estimate of power, it will
take a considerably large amount of time. First try increasing \code{ncrit}. Defaults to \code{FALSE}.}

\item{n.samp}{integer, number of sets of residuals for importance sampling for the copula model with cord. Defaults to \code{10}, recommend setting this
higher for smaller sample sizes \code{N}.}

\item{nlv}{number of latent variables (default = 2) for the copula model with cord, recommend setting this lower for smaller sample sizes \code{N}.}
}
\value{
Power estimate result, and;
\item{\code{power}}{power.}
}
\description{
\code{powersim} returns a power estimate for a \code{\link[ecoCopula]{cord}} object for a given sample size \code{N}
and effect size of interest.
}
\details{
\code{powersim} takes a \code{\link[ecoCopula]{cord}} object, sample size \code{N} and coefficient matrix \code{coeffs} which
specifies an effect size of interest and returns a power estimate.

The power estimate is obtained by first parsing the \code{\link[ecoCopula]{cord}} object into \code{\link{extend}},
\code{nsim} times with an effect size specified by \code{coeffs}. Next, the \code{\link[ecoCopula]{cord}} object is parsed into
\code{\link{extend}} an additional \code{ncrit} times with a null effect, which is defined by default by
\code{\link{effect_null}}. This effectively simulates \code{nsim} + \code{ncrit} \code{manyglm} models under both the null
and alternative hypothesis.

For each simulated \code{\link[mvabund]{manyglm}} object, a test statistic \code{test} is obtained. A critical test statistic
is then obtained as the upper 1 - \code{alpha} quantile of simulated test statistics under the null
hypothesis. Power is then estimated as the proportion of times the test statistics simulated under
the alternative hypothesis exceed the critical test statistic under the null.

To improve computation time, simulations are computed in parallel using the "socket" approach, which
by default uses all available cores minus 1 for clustering. Using 1 less than the number of available cores for your
machine (\code{detectCores()-1}) is recommended to better avoid errors relating to clustering or nodes.
}
\section{Functions}{
\itemize{
\item \code{powersim()}: Provide power estimates for multivariate abundance models

}}
\section{Author(s)}{

Ben Maslen <b.maslen@unsw.edu.au>.
}

\section{References}{

Maslen, B., Popovic, G., Lim, M., Marzinelli, E., & Warton, D. (2023). How many sites? Methods to assist design decisions when collecting multivariate data in ecology. Methods in Ecology and Evolution, 14(6), 1564-1573.
Popovic, G. C., Hui, F. K., & Warton, D. I. (2018). A general algorithm for covariance modeling of discrete data. Journal of Multivariate Analysis, 165, 86-100.
}

\examples{
\donttest{
library(ecoCopula)
library(mvabund)
data(spider)
spiddat = mvabund(spider$abund)
X = data.frame(spider$x)

# Specify increasers and decreasers
increasers = c("Alopacce", "Arctlute", "Arctperi", "Pardnigr", "Pardpull")
decreasers = c("Alopcune", "Alopfabr", "Zoraspin")

# Find power for continuous predictor at effect_size=1.5
fit.glm = manyglm(spiddat~bare.sand, family="negative.binomial", data=X)
effect_mat = effect_alt(fit.glm, effect_size=1.5,
       increasers, decreasers, term="bare.sand")
fit.cord = cord(fit.glm)
powersim(fit.cord, coeffs=effect_mat, term="bare.sand", nsim=99, ncrit=99, ncores=2)

# Find power for categorical predictor with 4 levels at effect_size=1.5
X$Treatment = rep(c("A","B","C","D"),each=7)
fit_factors.glm = manyglm(spiddat~Treatment, family="negative.binomial", data=X)
effect_mat = effect_alt(fit_factors.glm, effect_size=1.5,
       increasers, decreasers, term="Treatment")
fit_factors.cord = cord(fit_factors.glm)
powersim(fit_factors.cord, coeffs=effect_mat, term="Treatment", nsim=99, ncrit=99, ncores=2)

# Change effect size parameterisation
effect_mat = effect_alt(fit_factors.glm, effect_size=1.5,
                         increasers, decreasers, term="Treatment",
                         K=c(3,1,2))
powersim(fit_factors.cord, coeffs=effect_mat, term="Treatment", nsim=99, ncrit=99, ncores=2)
}
}
\seealso{
\code{\link{effect_alt}}, \code{\link{effect_null}}, \code{\link{extend}}
}
