% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.R
\name{predict.drf}
\alias{predict.drf}
\title{Predict from Distributional Random Forests object}
\usage{
\method{predict}{drf}(
  object,
  newdata = NULL,
  functional = NULL,
  transformation = NULL,
  custom.functional = NULL,
  num.threads = NULL,
  ...
)
}
\arguments{
\item{object}{Trained DRF forest object.}

\item{newdata}{Points at which predictions should be made. If NULL, returns out-of-bag
predictions on the training set (i.e., for every training point \eqn{X_i}, provides predictions using only trees which did not use
this point for tree construction). Can be either a data frame, matrix or a vector. Each row represents a data point of interest and 
the number and ordering of columns is assumed the be the same as in the training set.}

\item{functional}{Optional. String indicating the statistical functional that we want to compute from the weights. One option between:
\describe{
 \item{"mean"}{ - Conditional mean, the returned value is a matrix \code{mean} of dimension \code{n} x \code{f},
 where \code{n} denotes the number of observations in \code{newdata} and \code{f} the dimension of the \code{transformation}.}
 \item{"sd"}{ - Conditional standard deviation for each component of the (transformed) response, the returned value is
 a matrix of dimension \code{n} x \code{f}, where \code{n} denotes the number of observations in \code{newdata} and \code{f} the dimension of the \code{transformation}.}
 \item{"quantile"}{ - Conditional quantiles. It requires additional parameter \code{quantiles} containing the list of quantile levels we want to compute. 
 The returned value is an array of dimension \code{n} x \code{f}  x \code{q}, where \code{n} denotes the number of observations in \code{newdata}, 
 \code{f} the dimension of the \code{transformation} and \code{q} the number of desired quantiles.}
 \item{"cor"}{ - Conditional correlation matrix, the returned value is an array of dimension \code{n} x \code{f} x \code{f},
 where \code{n} denotes the number of observations in \code{newdata} and \code{f} the dimension of the \code{transformation}.}
 \item{"cov"}{ - Conditional covariance matrix, the returned value is an array of dimension \code{n} x \code{f}  x \code{f},
 where \code{n} denotes the number of observations in \code{newdata}, \code{f} the dimension of the \code{transformation}.}
 \item{"custom"}{ - A custom function provided by the user, the returned value is a matrix of dimension \code{n} x \code{f},
 where \code{n} denotes the number of observations in \code{newdata} and \code{f} the dimension of the output of the function \code{custom.functional} provided by the user.}
}}

\item{transformation}{An optional transformation function that is applied to the responses before computing the target functional. It helps to extend the functionality to a much wider range of targets. 
The responses are not transformed by default, i.e. the identity function \eqn{f(y) = y} is used.}

\item{custom.functional}{A user-defined function when \code{functional} is set to "custom". This should be a function \code{f(y,w)} which for a single test point
takes the \code{n} x \code{f} matrix \code{y} and the corresponding \code{n}-dimensional vector of weights \code{w} and returns the quantity of interest given as a list of values.
\code{n} denotes the number of training observations and \code{f} the dimension of the \code{transformation}.}

\item{num.threads}{Number of threads used for computing. If set to NULL, the software automatically selects an appropriate amount.}

\item{...}{additional parameters.}
}
\value{
If functional equals NULL, returns a list containing the matrix of training responses as well as the matrix of weights, whose number of rows corresponds the number of rows of "newdata" and the number of columns corresponds to the number of training data points.
If functional is specified, the desired quantity is returned, in the format described above.
}
\description{
Obtain predictions from a DRF forest object. For any point \eqn{x} in the predictor space, it returns the estimate of the conditional distribution 
\eqn{P(Y | X=x)} represented as a weighted distribution \eqn{\sum_i w_i y_i} of the training observations \eqn{{y_i}}. 
Additionally, this function also provides support to directly obtain estimates of certain target quantities \eqn{\tau(P(Y | X))}, 
such as e.g. conditional quantiles, variances or correlations.
}
\examples{
\donttest{
library(drf)

n = 10000
p = 20
d = 3

# Generate training data
X = matrix(rnorm(n * p), nrow=n)
Y = matrix(rnorm(n * d), nrow=n)
Y[, 1] = Y[, 1] + X[, 1]
Y[, 2] = Y[, 2] * X[, 2]
Y[, 3] = Y[, 3] * X[, 1] + X[, 2]

# Fit DRF object
drf.forest = drf(X, Y)

# Generate test data
X_test = matrix(rnorm(10 * p), nrow=10)

out = predict(drf.forest, newdata=X_test)
# Compute E[Y_1 | X] for all data in X_test directly from
# the weights representing the estimated distribution
out$weights \%*\% out$y[,1]

out = predict(drf.forest, newdata=X_test,
              functional='mean')
# Compute E[Y_1 | X] for all data in X_test using built-in functionality
out[,1]

out = predict(drf.forest, newdata=X_test,
              functional='quantile',
              quantiles=c(0.25, 0.75),
              transformation=function(y){y[1] * y[2] * y[3]})
# Compute 25\% and 75\% quantiles of Y_1*Y_2*Y_3, conditionally on X = X_test[1, ]
out[1,,]

out = predict(drf.forest, newdata=X_test,
              functional='cov',
              transformation=function(y){matrix(1:6, nrow=2) \%*\% y})
# Compute 2x2 covariance matrix for (1*Y_1 + 3*Y_2 + 5*Y_3, 2*Y_1 + 4*Y_2 + 6*Y_3),
# conditionally on X = X_test[1, ]
out[1,,]

out = predict(drf.forest, newdata=X_test,
              functional='custom',
              custom.functional=function(y, w){c(sum(y[, 1] * w), sum(y[, 2] * w))})
# Compute E[Y_1, Y_2 | X] for all data in X_test by providing custom functional that
# computes it from the weights
out
}

}
