% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Normal.R
\name{Normal}
\alias{Normal}
\title{Create a Normal distribution}
\usage{
Normal(mu = 0, sigma = 1)
}
\arguments{
\item{mu}{The location parameter, written \eqn{\mu} in textbooks,
which is also the mean of the distribution. Can be any real number.
Defaults to \code{0}.}

\item{sigma}{The scale parameter, written \eqn{\sigma} in textbooks,
which is also the \strong{standard deviation} of the distribution. Can be any
positive number. Defaults to \code{1}. If you would like a Normal
distribution with \strong{variance} \eqn{\sigma^2}, be sure to take the
square root, as this is a common source of errors.}
}
\value{
A \code{Normal} object.
}
\description{
The Normal distribution is ubiquitous in statistics, partially because
of the central limit theorem, which states that sums of i.i.d. random
variables eventually become Normal. Linear transformations of Normal
random variables result in new random variables that are also Normal. If
you are taking an intro stats course, you'll likely use the Normal
distribution for Z-tests and in simple linear regression. Under
regularity conditions, maximum likelihood estimators are
asymptotically Normal. The Normal distribution is also called the
gaussian distribution.
}
\details{
We recommend reading this documentation on
\url{https://alexpghayes.github.io/distributions3/}, where the math
will render with additional detail and much greater clarity.

In the following, let \eqn{X} be a Normal random variable with mean
\code{mu} = \eqn{\mu} and standard deviation \code{sigma} = \eqn{\sigma}.

\strong{Support}: \eqn{R}, the set of all real numbers

\strong{Mean}: \eqn{\mu}

\strong{Variance}: \eqn{\sigma^2}

\strong{Probability density function (p.d.f)}:

\deqn{
    f(x) = \frac{1}{\sqrt{2 \pi \sigma^2}} e^{-(x - \mu)^2 / 2 \sigma^2}
  }{
    f(x) = 1 / (2 \pi \sigma^2) exp(-(x - \mu)^2 / (2 \sigma^2))
  }

\strong{Cumulative distribution function (c.d.f)}:

The cumulative distribution function has the form

\deqn{
    F(t) = \int_{-\infty}^t \frac{1}{\sqrt{2 \pi \sigma^2}} e^{-(x - \mu)^2 / 2 \sigma^2} dx
  }{
    F(t) = integral_{-\infty}^t 1 / (2 \pi \sigma^2) exp(-(x - \mu)^2 / (2 \sigma^2)) dx
  }

but this integral does not have a closed form solution and must be
approximated numerically. The c.d.f. of a standard Normal is sometimes
called the "error function". The notation \eqn{\Phi(t)} also stands
for the c.d.f. of a standard Normal evaluated at \eqn{t}. Z-tables
list the value of \eqn{\Phi(t)} for various \eqn{t}.

\strong{Moment generating function (m.g.f)}:

\deqn{
    E(e^{tX}) = e^{\mu t + \sigma^2 t^2 / 2}
  }{
    E(e^(tX)) = e^(\mu t + \sigma^2 t^2 / 2)
  }
}
\examples{

set.seed(27)

X <- Normal(5, 2)
X

mean(X)
variance(X)
skewness(X)
kurtosis(X)

random(X, 10)

pdf(X, 2)
log_pdf(X, 2)

cdf(X, 4)
quantile(X, 0.7)

### example: calculating p-values for two-sided Z-test

# here the null hypothesis is H_0: mu = 3
# and we assume sigma = 2

# exactly the same as: Z <- Normal(0, 1)
Z <- Normal()

# data to test
x <- c(3, 7, 11, 0, 7, 0, 4, 5, 6, 2)
nx <- length(x)

# calculate the z-statistic
z_stat <- (mean(x) - 3) / (2 / sqrt(nx))
z_stat

# calculate the two-sided p-value
1 - cdf(Z, abs(z_stat)) + cdf(Z, -abs(z_stat))

# exactly equivalent to the above
2 * cdf(Z, -abs(z_stat))

# p-value for one-sided test
# H_0: mu <= 3   vs   H_A: mu > 3
1 - cdf(Z, z_stat)

# p-value for one-sided test
# H_0: mu >= 3   vs   H_A: mu < 3
cdf(Z, z_stat)

### example: calculating a 88 percent Z CI for a mean

# same `x` as before, still assume `sigma = 2`

# lower-bound
mean(x) - quantile(Z, 1 - 0.12 / 2) * 2 / sqrt(nx)

# upper-bound
mean(x) + quantile(Z, 1 - 0.12 / 2) * 2 / sqrt(nx)

# equivalent to
mean(x) + c(-1, 1) * quantile(Z, 1 - 0.12 / 2) * 2 / sqrt(nx)

# also equivalent to
mean(x) + quantile(Z, 0.12 / 2) * 2 / sqrt(nx)
mean(x) + quantile(Z, 1 - 0.12 / 2) * 2 / sqrt(nx)

### generating random samples and plugging in ks.test()

set.seed(27)

# generate a random sample
ns <- random(Normal(3, 7), 26)

# test if sample is Normal(3, 7)
ks.test(ns, pnorm, mean = 3, sd = 7)

# test if sample is gamma(8, 3) using base R pgamma()
ks.test(ns, pgamma, shape = 8, rate = 3)

### MISC

# note that the cdf() and quantile() functions are inverses
cdf(X, quantile(X, 0.7))
quantile(X, cdf(X, 7))
}
\seealso{
Other continuous distributions: 
\code{\link{Beta}()},
\code{\link{Cauchy}()},
\code{\link{ChiSquare}()},
\code{\link{Erlang}()},
\code{\link{Exponential}()},
\code{\link{FisherF}()},
\code{\link{Frechet}()},
\code{\link{GEV}()},
\code{\link{GP}()},
\code{\link{Gamma}()},
\code{\link{Gumbel}()},
\code{\link{LogNormal}()},
\code{\link{Logistic}()},
\code{\link{RevWeibull}()},
\code{\link{StudentsT}()},
\code{\link{Tukey}()},
\code{\link{Uniform}()},
\code{\link{Weibull}()}
}
\concept{continuous distributions}
