% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DiscSurvDataTransform.R
\name{dataLongTimeDep}
\alias{dataLongTimeDep}
\title{Data Long Time Dependent Covariates}
\usage{
dataLongTimeDep(
  dataSemiLong,
  timeColumn,
  eventColumn,
  idColumn,
  timeAsFactor = FALSE
)
}
\arguments{
\item{dataSemiLong}{Original data in semi-long format ("class data.frame").}

\item{timeColumn}{Character giving the column name of the observed times ("character vector"). 
It is required that the observed times are discrete ("integer vector").}

\item{eventColumn}{Column name of the event indicator ("character vector").
It is required that this is a binary variable with 1=="event" and
0=="censored".}

\item{idColumn}{Name of column of identification number of persons ("character vector").}

\item{timeAsFactor}{Should the time intervals be coded as factor ("logical vector")? Default is
FALSE. In case of default settings the discrete time intervals are treated as quantitative ("numeric vector").}
}
\value{
Original data in long format with three additional columns: \itemize{ \item
{obj} Index of persons as integer vector \item {timeInt} Index of time
intervals (factor) \item {y} Response in long format as binary vector.
1=="event happens in period timeInt" and zero otherwise }
}
\description{
Transforms short data format to long format for discrete survival modelling
of single event analysis with right censoring. Covariates may vary over
time.
}
\details{
There may be some intervals, where no additional information on the
covariates is observed (e. g. observed values in interval one and three but
two is missing). In this case it is assumed, that the values from the last
observation stay constant over time until a new measurement was done.

In contrast to continuous survival (see e. g. \code{\link[survival]{Surv}}) 
the start and stop time notation is not used here. In discrete time survival analysis the only relevant
information is to use the stop time. Start time does not matter, because all discrete intervals need to be  
included in the long data set format to ensure consistent estimation. It is assumed that the supplied 
data set "dataSemiLong" contains all repeated measurements of each cluster in semi-long format (e. g. persons). 
For further information see example \emph{Start-stop notation}.
}
\examples{

# Example Primary Biliary Cirrhosis data
library(survival)
dataSet1 <- pbcseq

# Only event death is of interest
dataSet1$status [dataSet1$status == 1] <- 0
dataSet1$status [dataSet1$status == 2] <- 1
table(dataSet1$status)

# Convert to months
dataSet1$day <- ceiling(dataSet1$day/30) + 1
names(dataSet1) [7] <- "month"

# Convert to long format for time varying effects
pbcseqLong <- dataLongTimeDep (dataSemiLong = dataSet1, timeColumn = "month", 
eventColumn = "status", idColumn = "id")
pbcseqLong [pbcseqLong$obj == 1, ]

#####################
# Start-stop notation

library(survival)
?survival::heart

# Assume that time was measured on a discrete scale.
# Discrete interval lengths are assumed to vary.
intervalLimits <- quantile(heart$stop, probs = seq(0.1, 1, by=0.1))
intervalLimits[length(intervalLimits)] <- intervalLimits[length(intervalLimits)] + 1
heart_disc <- contToDisc(dataShort = heart, timeColumn = "stop", 
intervalLimits = intervalLimits, equi = FALSE)
table(heart_disc$timeDisc)

# Conversion to long format
heart_disc_long <- dataLongTimeDep(dataSemiLong = heart_disc, timeColumn = "timeDisc", 
eventColumn = "event", idColumn = "id")
head(heart_disc_long)

}
\references{
\insertRef{tutzModelDisc}{discSurv} \cr\cr
\insertRef{fahrmeirDiscSurv}{discSurv} \cr\cr
\insertRef{thompsonTreatment}{discSurv}
}
\seealso{
\code{\link{contToDisc}}, \code{\link{dataLong}},
\code{\link{dataLongCompRisks}}
}
\author{
Thomas Welchowski \email{welchow@imbie.meb.uni-bonn.de}
}
\keyword{datagen}
