#' Find the chromosome arm for each marker
#'
#' @param marker.data A two-column numeric matrix of marker position data for markers in the autosomes.  
#'   Column 1 contains the chromosome number for each marker, and column 2 contains the position (in base 
#'   pairs) for each marker.  This is a submatrix of the marker position matrix used by \code{\link{quickLook}} 
#'   and \code{\link{detailedLook}}.
#'
#' @param annot.file A dataframe containing cytoband annotation for the autosomes.  Each row corresponds to a 
#'   distinct cytoband, and column 1 contains the chromosome number, column 2 contains the start position (in 
#'   base pairs), column 3 contains the end position (in base pairs), and column 4 contains the cytoband name 
#'   (e.g. p21.3).  Additional columns may be present, but they are not used.
#'
#' @param reformat.annot A logical value that indicates whether \code{annot.file} needs to be reformatted.
#'
#' @return A character vector of length m, where m is the number of markers.
#'
#' @details DiNAMIC's peeling procedure is detailed in Bioinformatics (2011) 27(5) 678 - 685, and it is performed 
#'   by the \code{\link{peeling}} function.  By construction, the peeling procedure only affects markers in a given 
#'   chromosome arm.  This function is used internally by the \code{\link{peeling}} function to restrict the peeling 
#'   procedure to the chromosome arm containing the marker that corresponds to \code{max(colSums(x))}.
#'
#'
#' @examples
#' wilms.pq = makeCytoband(wilms.markers, annot.file)
#' #A character vector of length 3288, and each entry is either
#' #"p" or "q", depending on the chromosome arm of the given marker.
#' table(wilms.pq)
#' #Produces the following output:
#' #wilms.pq
#' #   p    q 
#' #1147 2141 
#'
#' @export

makeCytoband = function(marker.data, annot.file, reformat.annot = FALSE) 
{
    if (reformat.annot == TRUE) {
        autosome.stop = min(which(as.matrix(annot.file[, 1]) == 
            "chrX")) - 1
        annot.file = annot.file[1:autosome.stop, ]
        new.chr.col = unlist(strsplit(as.matrix(annot.file[, 
            1]), "r", fixed = TRUE))[2 * c(1:dim(annot.file)[1])]
        annot.file[, 1] = as.data.frame(new.chr.col)
    }
    chr.vec = intersect(unique(as.matrix(marker.data[, 1])), 
        c(1:22))
    output = c()
    for (i in chr.vec) {
        temp.annot.rows = which(as.matrix(annot.file[, 1]) == 
            i)
        temp.annot = annot.file[temp.annot.rows, ]
        temp.pq.vec = substring(as.matrix(temp.annot[, 4]), 1, 
            1)
        temp.q.start = temp.annot[min(which(temp.pq.vec == "q")), 
            2]
        temp.marker.rows = which(marker.data[, 1] == i)
        temp.marker = marker.data[temp.marker.rows, ]
        temp.num.markers = dim(temp.marker)[1]
        temp.num.p = sum(temp.marker[, 2] < temp.q.start)
        temp.num.q = temp.num.markers - temp.num.p
        temp.output = c(rep("p", temp.num.p), rep("q", temp.num.q))
        output = c(output, temp.output)
    }
    return(output)
}
