#' Plot DRI Intersubjective Consistency (IC)
#'
#' `plot_dri_ic` creates a dot plot of deliberation IC where each dot represents
#' a pair of participants
#'
#' @param ic dataframe generated by `get_dri_ic(data)`
#' @param title title of the plot
#' @param suffix string to be added after the title separated by `:`
#' @param dri numeric value generated by `get_dri(ic)`; if omitted, `get_dri` is
#' called by default
#' @param caption a string to be displayed under the plot
#' @import grid
#' @rawNamespace import(ggplot2, except = alpha)
#' @importFrom rlang .data
#'
#' @returns an IC plot
#' @export
#'
#' @seealso [get_dri_ic()] for how to generate the `ic` parameter
#' @seealso [get_dri()] for how to generate the `dri` parameter
#'
#' @examples
#'
#' # get post-deliberation (stage_id == 2) data from Zukunft case
#' data <- human_data[human_data$stage_id == 2 & human_data$case == "Zukunft", ]
#'
#' # set plot optional parameters
#' title <- "Case Zukunft"
#' suffix <- "Post-Deliberation IC Plot"
#' caption <- "this is an example plot"
#'
#' # calculate ic
#' ic <- get_dri_ic(data)
#'
#' plot_dri_ic(ic, title, suffix, caption = caption)
#'
plot_dri_ic <- function(ic,
                     title = NA_character_,
                     suffix = NA_character_,
                     dri = NA_real_,
                     caption = NULL) {

  # assume llm pnum is negative and set palette
  ic$llm <- ic$pnum1 < 0 | ic$pnum2 < 0
  pal <- c("black", "blue")

  # create title
  plot_title <- if (!is.na(title)) title else "Intersubjective Consistency (IC) Plot"
  plot_title <- if (!is.na(suffix)) paste0(plot_title, ": ", suffix) else plot_title

  # create a DRI annotation
  grob <- grobTree(textGrob(
    paste0("DRI = ", round(dri, 3)),
    x = 0.1,
    y = 0.9,
    hjust = 0,
    gp = gpar(
      col = "red",
      fontsize = 12
      # fontface = "italic"
    )
  ))

  dri <- if (is.na(dri)) get_dri(ic) else dri

  plot <-
    ggplot(ic, aes(x = .data$ccor, y = .data$pcor)) +
    geom_abline(
      intercept = 0,
      slope = 1,
      colour = "black"
    ) +
    geom_hline(yintercept = 0,
               color = "black",
               linetype="dashed") +
    geom_vline(xintercept = 0,
               color = "black",
               linetype="dashed") +
    geom_jitter(
      width = 0.02,
      height = 0.02,
      show.legend = TRUE,
      aes(colour = .data$llm)
    ) +
    scale_colour_manual(values = pal) +
    xlim(-1.1, 1.1) + ylim(-1.1, 1.1) +
    # theme(panel.grid.major = element_blank(),
    #       panel.grid.minor = element_blank()) +
    labs(x = "Intersubjective Agreement - Considerations",
         y = "Intersubjective Agreement - Preferences",
         subtitle = paste0("DRI = ", round(dri, 3)),
         caption = caption) +
    ggtitle(plot_title) +
    # annotation_custom(grob) +
    # geom_density_2d_filled(inherit.aes = T, alpha = 0.4) +
    geom_density_2d(colour = "#FF0000AA") +
    theme_classic(base_size = 16) +
    theme(legend.position = "none")

  plot
}
