% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core_stats.R, R/formatting_stats_dm.R
\name{calc_stats}
\alias{calc_stats}
\alias{calc_stats.data.frame}
\alias{calc_stats.drift_dm}
\alias{calc_stats.fits_ids_dm}
\alias{calc_stats.fits_agg_dm}
\alias{print.stats_dm}
\alias{print.stats_dm_list}
\title{Calculate Statistics}
\usage{
calc_stats(object, type, ...)

\method{calc_stats}{data.frame}(
  object,
  type,
  ...,
  conds = NULL,
  resample = FALSE,
  progress = 1,
  level = "individual",
  b_coding = NULL
)

\method{calc_stats}{drift_dm}(object, type, ..., conds = NULL, resample = FALSE)

\method{calc_stats}{fits_ids_dm}(
  object,
  type,
  ...,
  conds = NULL,
  resample = FALSE,
  progress = 1,
  level = "individual"
)

\method{calc_stats}{fits_agg_dm}(
  object,
  type,
  ...,
  conds = NULL,
  resample = FALSE,
  progress = 1,
  level = "group",
  messaging = TRUE
)

\method{print}{stats_dm}(
  x,
  ...,
  round_digits = NULL,
  print_rows = NULL,
  some = NULL,
  show_header = NULL,
  show_note = NULL
)

\method{print}{stats_dm_list}(x, ...)
}
\arguments{
\item{object}{an object for which statistics are calculated. This can be a
\link{data.frame} of observed data, a \link{drift_dm} object, a
\code{fits_ids_dm} object, or a \code{fits_agg_dm} object (see
\code{\link[=estimate_dm]{estimate_dm()}}).}

\item{type}{a character vector, specifying the statistics to calculate.
Supported values include \code{"basic_stats"}, \code{"cafs"}, \code{"quantiles"},
\code{"delta_funs"}, \code{"densities"}, and \code{"fit_stats"}.}

\item{...}{additional arguments passed to the respective method and the
underlying calculation functions (see Details for mandatory arguments).}

\item{conds}{optional character vector specifying conditions to include.
Conditions must match those found in the \code{object}.}

\item{resample}{logical. If \code{TRUE}, then data is (re-)sampled to create
an uncertainty estimate for the requested summary statistic. See Details for
more information. Default is \code{FALSE}. Note that resampling does not work with
\code{type = "fit_stats"}.}

\item{progress}{integer, indicating if information about the progress
should be displayed. 0 -> no information, 1 -> a progress bar. Default is 1.}

\item{level}{a single character string, indicating at which "level" the
statistic should be calculated. Options are \code{"group"} or \code{"individual"}. If
\code{"individual"}, the returned \code{stats_dm} object contains an \code{"ID"} column.}

\item{b_coding}{a list for boundary coding (see \link{b_coding}). Only
relevant when \code{object} is a \link{data.frame}. For other \code{object} types, the
\code{b_coding} of the \code{object} is used.}

\item{messaging}{logical, if \code{FALSE}, no message is provided.}

\item{x}{an object of type \code{stats_dm} or \code{stats_dm_list}, as returned by
the function \code{calc_stats()}.}

\item{round_digits}{integer, controls the number of digits shown.
Default is 3.}

\item{print_rows}{integer, controls the number of rows shown.}

\item{some}{logical. If \code{TRUE}, a subset of randomly sampled rows is shown.}

\item{show_header}{logical. If \code{TRUE}, a header specifying the type of
statistic will be displayed.}

\item{show_note}{logical. If \code{TRUE}, a footnote  is displayed indicating
that the underlying \link{data.frame} can be accessed as usual.}
}
\value{
If \code{type} is a single character string, then a subclass of \link{data.frame} is
returned, containing the respective statistic. Objects of type \code{sum_dist}
will have an additional attribute storing the boundary encoding (see also
\link{b_coding}). The reason for returning subclasses of \link{data.frame} is
to provide custom \code{plot()} methods (e.g., \link{plot.cafs}). To get rid
of the subclass label and additional attributes (i.e., to get just the plain
underlying \link{data.frame}, users can use \code{\link[=unpack_obj]{unpack_obj()}}).

If \code{type} contains multiple character strings (i.e., is a character vector) a
subclass of \link{list} with the calculated statistics is returned. The list will
be of type \code{stats_dm_list} (to easily create multiple panels using the
respective \code{\link[=plot.stats_dm_list]{plot.stats_dm_list()}} method).

The print methods \code{print.stats_dm()} and \code{print.stats_dm_list()} each
invisibly return the supplied object \code{x}.
}
\description{
\code{calc_stats} provides an interface for calculating statistics/metrics on
model predictions and/or observed data. Supported statistics include
basic statistics on mean and standard deviation, Conditional Accuracy
Functions (CAFs), Quantiles, Delta Functions, and fit statistics. Results can
be aggregated across individuals.
}
\details{
\code{calc_stats} is a generic function to handle the calculation of different
statistics/metrics for the supported object types. Per default, it returns
the requested statistics/metrics.
\subsection{List of Supported Statistics}{

\strong{Basic Statistics}

With "basic statistics", we refer to a summary of the mean and standard
deviation of response times, including a proportion of response choices.

\strong{Conditional Accuracy Function (CAFs)}

CAFs are a way to quantify response accuracy against speed. To calculate
CAFs, RTs (whether correct or incorrect) are first binned and then the
percent correct responses per bin is calculated.

When calculating model-based CAFs, a joint CDF combining both the pdf
of correct and incorrect responses is calculated. Afterwards, this CDF
is separated into even-spaced segments and the contribution of
the pdf associated with a correct response relative to the joint CDF is
calculated.

The number of bins can be controlled by passing the argument \code{n_bins}.
The default is 5.

\strong{Quantiles}

For observed response times, the function \link[stats:quantile]{stats::quantile} is used with
default settings.

Which quantiles are calcuated can be controlled by providing the
probabilites, \code{probs}, with values in \eqn{[0, 1]}. Default is
\code{seq(0.1, 0.9, 0.1)}.

\strong{Delta Functions}

Delta functions calculate the difference between quantiles
of two conditions against their mean:
\itemize{
\item \eqn{Delta_i = Q_{i,j} - Q_{i,k}}
\item \eqn{Avg_i = 0.5 \cdot Q_{i,j} + 0.5 \cdot Q_{i,k}}
}

With i indicating a quantile, and j and k two conditions.

To calculate delta functions, users have to specify:
\itemize{
\item \code{minuends}: character vector, specifying condition(s) j. Must be in
\code{conds(drift_dm_obj)}.
\item \code{subtrahends}: character vector, specifying condition(s) k. Must be in
\code{conds(drift_dm_obj)}
\item \code{dvs}: character, indicating which quantile columns to use.
Default is "Quant_<u_label>". If multiple dvs are provided,
then minuends and subtrahends must have the same length,
and matching occurs pairwise. In this case, if only one
minuend/subtrahend is specified, minuend and subtrahend are recycled to
the necessary length.
\item specifying \code{probs} is possible (see Quantiles)
}

\strong{Densities}

With "densities", we refer to a summary of the distribution of observed
or predicted data. For observed data, histogram values and kernel density
estimates are provided. For predicted data, the model's predicted PDFs are
provided.

Optional arguments are:
\itemize{
\item \code{discr}: numeric, the band-width when calculating the histogram or the
kernel density estimates. Defaults to \code{0.015} seconds
\item \code{t_max}: numeric, the maximum time window when calculating the distribution
summaries of observe data. Defaults to the longest RT (for observed data)
or the maximum of the time domain of a model (which is the preferred choice,
if possible). If necessary, \code{t_max} is slightly adjusted to match with
\code{discr}.
\item \code{scale_mass}: logical, only relevant if observed data is available. If
\code{TRUE}, density masses are scaled proportional to the number of trials per
condition.
}

\strong{Fit Statistics}

Calculates the Log-Likelihood, Akaike and Bayesian Information Criteria,
and root-mean squared-error statistic.

Optional arguments are:
\itemize{
\item \code{k}: numeric, for penalizing the AIC statistic (see also \link[stats:AIC]{stats::AIC}
and \link{AIC.fits_ids_dm}).
\item \code{n_bins}, \code{probs}: numeric vectors, see the section on CAFs and Quantiles
above
\item \code{weight_err}: numeric scalar, determines how CAFs and quantiles are
weighted. Default is \code{1.5}.
}
}

\subsection{Resampling}{

When \code{resampling = TRUE}, an uncertainty interval is provided via simulation.
The default number of iterations is \code{R = 100}, which can be changed by
passing the optional argument \code{R}.

If resampling is requested, the returned \code{stats_dm} object contains the
column \code{"Estimate"}, coding the interval. The interval width is controlled
via the optional argument \code{interval_level}, a single numeric value between
0 and 1 (default: \code{0.95}). The interpretation of this interval depends on
the specific situation (see below).

\strong{Resampling at the Individual Level}

If \code{object} is a \code{drift_dm} object (i.e., a single model created by
\code{\link[=drift_dm]{drift_dm()}}), synthetic data are simulated under the model, and
for each synthetic data set the requested statistic is calculated. The
interval then reflects the range of these simulated statistics. To determine
the number of trials for each synthetic data set, dRiftDM either uses the
observed data attached to the model (if available) or the optional argument
\code{n_sim} (passed to \code{\link[=simulate_data]{simulate_data()}}). Note that \code{n_sim} must be
provided if no observed data are available, and that \code{n_sim} always has
priority.

If \code{object} is a \code{drift_dm} object with attached observed data, resampling
is also performed for the observed data. In this case, trials are
bootstrapped, and for each bootstrap sample the requested statistic is
calculated.

If \code{object} is a \code{data.frame}, \code{fits_agg_dm}, or \code{fits_ids_dm} object,
resampling is performed for each individual if \code{level = "individual"}. For
both models and observed data, synthetic or bootstrapped data sets are
generated as described above.

\strong{Resampling at the Group Level}

Group-level resampling is possible only if \code{object} is a \code{data.frame}
(with an \code{"ID"} column), \code{fits_agg_dm}, or \code{fits_ids_dm} object. To request
this, set \code{level = "group"}. Participants are then bootstrapped, and
for each bootstrapped sample the aggregated statistic is calculated.

\strong{Interpretation of Intervals}

For \code{level = "group"}, intervals represent bootstrapped confidence intervals
For \code{level = "individual"}, intervals represent the variability in the
statistic when data for a single participant are resampled or simulated
under the model.

\strong{Note}

For objects of type \code{fits_agg_dm}, which contain a mixture of group- and
individual-level information, the \code{level} argument only affects resampling
for the observed data. For the model itself, resampling is always performed
under the fitted model, in the same way as for a \code{drift_dm} object.
}
}
\note{
When a model's predicted density function integrates to a value of less than
\code{\link[=drift_dm_skip_if_contr_low]{drift_dm_skip_if_contr_low()}}, means and quantiles return the
values \code{NA}. Users can alter this by explicitly passing the argument
\code{skip_if_contr_low} when calling \code{calc_stats()} (e.g.,
\code{calc_stats(..., skip_if_contr_low = -Inf)})
}
\examples{
# Example 1: Calculate CAFs and Quantiles from a model ---------------------
# get a model for demonstration purpose
a_model <- ssp_dm()
# and then calculate cafs and quantiles
some_stats <- calc_stats(a_model, type = c("cafs", "quantiles"))
print(some_stats)

# Example 2: Calculate a Delta Function from a data.frame ------------------
# get a data set for demonstration purpose
some_data <- ulrich_simon_data
conds(some_data) # relevant for minuends and subtrahends
some_stats <- calc_stats(
  a_model,
  type = "delta_funs",
  minuends = "incomp",
  subtrahends = "comp"
)
print(some_stats, print_rows = 5)


# Example 3: Calculate Quantiles from a fits_ids_dm object -----------------
# get an auxiliary fits_ids_dm object
all_fits <- get_example_fits("fits_ids_dm")
some_stats <- calc_stats(all_fits, type = "quantiles")
print(some_stats, print_rows = 5) # note the ID column

# one can also request that the statistics are averaged across individuals
print(
  calc_stats(all_fits, type = "quantiles", average = TRUE)
)

}
