kidney_risk_num <- function(egfr, sex, acr, age, alb, phos, bicarb, calc, units = ""){
  x <- 0
  if(units == "SI"){
    alb = as.numeric(alb)
    alb <- alb / 10
    phos = as.numeric(phos)
    phos <- 3.0974 * phos
    calc = as.numeric(calc)
    calc <- 4.0078 * calc
  }
  egfr = as.numeric(egfr)
  if(!is.na(egfr) & !is.na(sex)& !is.na(acr)& !is.na(age)& !is.na(alb)& !is.na(phos)& !is.na(bicarb)& !is.na(calc)){
    if(egfr >=10 & egfr <=14){
      x <- x -35
    }
    else if(egfr >=15 & egfr <=19){
      x <- x -30
    }
    else if(egfr >=20 & egfr <=24){
      x <- x -25
    }
    else if(egfr >=25 & egfr <=29){
      x <- x -20
    }
    else if(egfr >=30 & egfr <=34){
      x <- x -15
    }
    else if(egfr >=35 & egfr <=39){
      x <- x -10
    }
    else if(egfr >=40 & egfr <=44){
      x <- x -5
    }
    else if(egfr >=50 & egfr <=54){
      x <- x +5
    }
    else if(egfr >=55 & egfr <=59){
      x <- x +10
    }
    sex = as.character(sex)
    if(sex == 'M'){
      x <- x -2
    }
    acr = as.numeric(acr)
    if(acr >=30 & acr <=300){
      x <- x -14
    }
    else if(acr >300){
      x <- x -22
    }
    age = as.numeric(age)
    if(age <30){
      x <- x -4
    }
    else if(age >=30 & age <= 39){
      x <- x -2
    }
    else if(age >=50 & age <= 59){
      x <- x +2
    }
    else if(age >=60 & age <= 69){
      x <- x +4
    }
    else if(age >=70 & age <= 79){
      x <- x +6
    }
    else if(age >=80 & age <= 89){
      x <- x +8
    }
    else if(age >=90){
      x <- x +10
    }
    alb = as.numeric(alb)
    if(alb <=2.5){
      x <- x -5
    }
    else if(alb >=3.1 & alb <= 3.5){
      x <- x +2
    }
    else if(alb >=3.6){
      x <- x +4
    }
    phos = as.numeric(phos)
    if(phos <3.5){
      x <- x +3
    }
    else if(phos >=4.6 & phos <=5.5 ){
      x <- x -3
    }
    else if(phos >=5.5){
      x <- x -5
    }
    bicarb = as.numeric(bicarb)
    if(bicarb < 18){
      x <- x -7
    }
    else if(bicarb >=18 & bicarb <= 22){
      x <- x -4
    }
    else if(bicarb >=23 & bicarb <= 25){
      x <- x -1
    }
    calc = as.numeric(calc)
    if(alb <=8.5){
      x <- x -3
    }
    else if(alb >=9.6){
      x <- x +2
    }
    return(x)
  }
  else
    return(NA)
}

kid_risk_perc<-function(egfr, sex, acr, age, alb, phos, bicarb, calc, units = ""){
  x<- kidney_risk_num(egfr, sex, acr, age, alb, phos, bicarb, calc, units)
  if(!is.na(x)){
    x <- as.numeric(x)
    y <- 0
    if(x < -41)
      y = 90
    else if(x == -41)
      y = 89
    else if(x == -40)
      y = 86.9
    else if(x == -39)
      y = 84.1
    else if(x == -38)
      y = 81
    else if(x == -37)
      y = 77.8
    else if(x == -36)
      y = 74.4
    else if(x== -35)
      y = 70.9
    else if(x == -34)
      y = 67.3
    else if(x == -33)
      y = 63.6
    else if(x == -32)
      y = 59.9
    else if(x== -31)
      y = 56.3
    else if(x == -30)
      y = 52.8
    else if(x == -29)
      y = 49.3
    else if(x == -28)
      y = 45.9
    else if(x == -27)
      y = 42.7
    else if(x == -26)
      y = 39.6
    else if(x == -25)
      y = 36.6
    else if(x == -24)
      y = 33.8
    else if(x == -23)
      y = 31.2
    else if(x == -22)
      y = 28.7
    else if(x == -21)
      y = 26.4
    else if(x == -20)
      y = 24.2
    else if(x == -19)
      y = 22.2
    else if(x == -18)
      y = 20.3
    else if(x == -17)
      y = 18.6
    else if(x == -16)
      y = 17.0
    else if(x == -15)
      y = 15.5
    else if(x == -14)
      y = 14.1
    else if(x == -13)
      y = 12.9
    else if(x == -12)
      y = 11.7
    else if(x == -11)
      y = 10.7
    else if(x == -10)
      y = 9.7
    else if(x == -9)
      y = 8.8
    else if(x == -8)
      y = 8.0
    else if(x == -7)
      y = 7.3
    else if(x == -6)
      y = 6.6
    else if(x == -5)
      y = 6.0
    else if(x == -4)
      y = 5.5
    else if(x >= -3)
      y = 5.0
    return(y)
  }
  else
    return(NA)
}

meld_xi<-function(sbill, screat){
  if(!is.na(sbill)&!is.na(screat)){
    sbill <- as.numeric(sbill)
    screat <- as.numeric(screat)
    x <- 5.112 + log(sbill) + 11.76 * log(screat) + 9.44
    return(x)
  }
  else
    return(NA)
}

fib4<-function(age, ast, alt, plt){
  if(!is.na(age) & !is.na(ast) & !is.na(alt)&!is.na(plt)){
    age <- as.numeric(age)
    ast <- as.numeric(ast)
    alt <- as.numeric(alt)
    plt <- as.numeric(plt)
    x <- (age * ast) / (plt * sqrt(alt))
    return(x)
  }
  else
    return(NA)
}

apri<-function(ast, plt){
  if(!is.na(ast)&!is.na(plt)){
    ast <- as.numeric(ast)
    plt <- as.numeric(plt)
    x <- (ast / 40) / (plt * 100)
    return(x)
  }
  else
    return(NA)
}

albi<-function(sbill, salb, units = ""){
  if(units == "SI"){
    sbill <- as.numeric(sbill)
    sbill <- sbill /17.1
    salb <- as.numeric(salb)
    salb <- salb/ 10
  }
  if(!is.na(sbill)&!is.na(salb)){
    sbill <- as.numeric(sbill)
    salb <- as.numeric(salb)
    x <- (log10(sbill * 17.1) *0.66)  + (-0.085 * (salb * 10))
    return(x)
  }
  else
    return(NA)
}

fli<-function(trigly, bmi, waist, ggt, units = ""){
  if(units == "SI"){
    trigly <- as.numeric(trigly)
  trigly <- trigly * 88.4
  }
  if(!is.na(trigly) & !is.na(bmi) & !is.na(waist)&!is.na(ggt)){
    trigly <- as.numeric(trigly)
    bmi <- as.numeric(bmi)
    ggt <- as.numeric(ggt)
    waist <- as.numeric(waist)
    y <-  0.953 * log(trigly) + 0.139 * bmi + 0.718 * log(ggt) + 0.053 * waist - 15.745
    x <- (exp(1)^y)/(1+exp(1)^y) * 100
    return(x)
  }
  else
    return(NA)
}

eAG <- function(a1c){
  a1c = as.numeric(a1c)
  if(!is.na(a1c))
    return((28.7 * a1c) - 46.7)
  else
    return(NA)
}

homair <- function(insulin,glucose, units = ""){
  if(units == "SI"){
    insulin = as.numeric(insulin)
    insulin <- insulin / 6
    glucose = as.numeric(glucose)
    glucose <- glucose * 18
  }
  insulin = as.numeric(insulin)
  glucose = as.numeric(glucose)
  if(!is.na(insulin) & !is.na(glucose))
    return((insulin * glucose)/405)
  else
    return(NA)
}

bard_score <- function(bmi, ast, alt, diabetes){
  bmi <- as.numeric(bmi)
  ast <- as.numeric(ast)
  alt <- as.numeric(alt)
  diabetes <- as.numeric(diabetes)
  if(is.na(diabetes) | is.na(bmi) | is.na(ast) | is.na(alt)){
    return(NA)
  }
  y <- ast/alt
  if(is.na(y)){
    return(NA)
  }
  x<- 0
  if(bmi >= 28){
    x <- x + 1
  }
  if((y) >= 0.8){
    x <- x + 2
  }
  if(diabetes == 1){
    x <- x + 1
  }
  return(x)
}

bard_score_interpret <- function(bmi, ast, alt, diabetes){
  x<- bard_score(bmi, ast, alt, diabetes)
  if(is.na(x))
    return(NA)
  if(x >1){
    y<- "high"
  }
  else{
    y<- "low"
  }
  return(y)
}

ast_alt_ratio_interpret <- function(ast, alt){
  ast <- as.numeric(ast)
  alt <- as.numeric(alt)
  if(is.na(ast) | is.na(alt)){
    return(NA)
  }
  y <- ast/alt
  if(is.na(y)){
    return(NA)
  }
  x<- 0
  if((y) >= 0.8){
    x <- 1
  }
  return(x)
}

nafld_score <- function(age, bmi, diabetes, ast, alt, platelet, albumin, units = ""){
  age = as.numeric(age)
  bmi = as.numeric(bmi)
  diabetes = as.numeric(diabetes)
  ast = as.numeric(ast)
  alt = as.numeric(alt)
  platelet = as.numeric(platelet)
  albumin = as.numeric(albumin)
  if(units == "SI"){
  albumin = albumin / 10
  }
  if(is.na(age) | is.na(bmi)| is.na(diabetes)| is.na(ast)| is.na(alt)| is.na(platelet)| is.na(albumin)){
    return(NA)
  }
  x<- -1.675 + (0.037 * age) + (0.094 * bmi) + (1.13 * diabetes) + (0.99 * (ast/alt)) - (0.013 * platelet) - (0.66 * albumin)
  return(x)
}

rfactor <- function(alt, alp){
  alt = as.numeric(alt)
  alp = as.numeric(alp)
  if(is.na(alp) | is.na(alt)){
    return(NA)
  }
  x<- (alt/40)/(alp/120)
  return(x)
}

rfactor_interpret <- function(alt, alp){
  x<- rfactor(alt, alp)
  x = as.numeric(x)
  if(is.na(x)){
    return(NA)
  }
  if(x <2){
    return("cholestatic")
  }
  else if(x>5){
    return("hepatocellular")
  }
  else{
    return("indeterminate")
  }
}

aha_stroke <- function(bp, afib, sugar, bmi, diet, chol, dm, physical, hx, smoke){
  bp <- as.numeric(bp)
  afib <- as.numeric(afib)
  sugar <- as.numeric(sugar)
  bmi <- as.numeric(bmi)
  diet <- as.numeric(diet)
  chol <- as.numeric(chol)
  dm <- as.numeric(dm)
  physical <- as.numeric(physical)
  hx <- as.numeric(hx)
  smoke <- as.numeric(smoke)
  if(is.na(bp)){
    bp <- 1
  }
  if(is.na(afib)){
    afib <- 1
  }
  if(is.na(sugar)){
    sugar <- 1
  }
  if(is.na(bmi)){
    bmi <- 1
  }
  if(is.na(diet)){
    diet <- 1
  }
  if(is.na(chol)){
    chol <- 1
  }
  if(is.na(dm)){
    dm <- 1
  }
  if(is.na(physical)){
    physical <- 1
  }
  if(is.na(hx)){
    hx <- 1
  }
  if(is.na(smoke)){
    smoke <- 1
  }
  x <- 0
  if(bp == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(afib == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(sugar == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(bmi == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(diet == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(chol == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(dm == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(physical == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(hx == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  if(smoke == 1){
    x = x + 1
  }
  else{
    x = x - 1
  }
  return(x)
}

#the following was written 7/3/2025 -

eular_acr_pmr_calc <- function(age, stiffness_gt_45, hip_pain_limited_rom, neg_serology, no_peripheral_pain, use_ultrasound = FALSE, us_abn_shoulder_and_hip = 0, us_abn_both_shoulders = 0) {

  age <- as.numeric(age)
  stiffness_gt_45 <- as.numeric(stiffness_gt_45)
  hip_pain_limited_rom <- as.numeric(hip_pain_limited_rom)
  neg_serology <- as.numeric(neg_serology)
  no_peripheral_pain <- as.numeric(no_peripheral_pain)
  us_abn_shoulder_and_hip <- as.numeric(us_abn_shoulder_and_hip)
  us_abn_both_shoulders <- as.numeric(us_abn_both_shoulders)

  if (any(is.na(c(age, stiffness_gt_45, hip_pain_limited_rom, neg_serology, no_peripheral_pain)))) {
    stop("Error: Inputs must be numeric or convertible strings (e.g., '1' for Yes, '0' for No).")
  }

  if (age < 50) {
    return(list(
      Score = NA,
      Result = "Patient does not meet entry criteria (Age must be >= 50)",
      Algorithm = "Excluded"
    ))
  }

  score <- 0

  if (stiffness_gt_45 == 1) score <- score + 2
  if (hip_pain_limited_rom == 1) score <- score + 1
  if (neg_serology == 1) score <- score + 2
  if (no_peripheral_pain == 1) score <- score + 1

  cutoff <- 4
  algo_name <- "Clinical Criteria"

  if (use_ultrasound) {
    if (us_abn_shoulder_and_hip == 1) score <- score + 1
    if (us_abn_both_shoulders == 1) score <- score + 1
    cutoff <- 5
    algo_name <- "Clinical + Ultrasound Criteria"
  }

  classification <- "Not Classified as PMR"
  if (score >= cutoff) {
    classification <- "Classified as PMR"
  }

  return(list(
    Total_Score = score,
    Classification_Threshold = cutoff,
    Result = classification,
    Method_Used = algo_name,
    Input_Verification = list(
      Age = age,
      Morning_Stiffness_Points = if (stiffness_gt_45 == 1) 2 else 0,
      Hip_Pain_Points = if (hip_pain_limited_rom == 1) 1 else 0,
      Serology_Points = if (neg_serology == 1) 2 else 0,
      Peripheral_Pain_Points = if (no_peripheral_pain == 1) 1 else 0,
      Ultrasound_Included = use_ultrasound
    )
  ))
}

leibovich_2018_rcc <- function(t_stage, n_stage, tumor_size_cm, nuclear_grade, necrosis) {

  tumor_size_cm <- as.numeric(tumor_size_cm)
  nuclear_grade <- as.numeric(nuclear_grade)
  necrosis <- as.numeric(necrosis)

  t_stage <- tolower(as.character(t_stage))
  t_stage <- gsub("^[pt]", "", t_stage)

  n_stage <- tolower(as.character(n_stage))
  n_stage <- gsub("^[pn]", "", n_stage)

  if (any(is.na(c(tumor_size_cm, nuclear_grade, necrosis)))) {
    stop("Error: Tumor size, grade, and necrosis must be numeric or convertible to numeric.")
  }

  score <- 0

  if (t_stage %in% c("1a", "1b", "2a")) {
    score <- score + 0
  } else if (t_stage == "2b") {
    score <- score + 2
  } else if (t_stage %in% c("3a", "3b", "3c")) {
    score <- score + 3
  } else if (t_stage == "4") {
    score <- score + 4
  } else {
    stop("Error: Invalid T Stage. Use '1a', '1b', '2a', '2b', '3a', '3b', '3c', or '4'.")
  }

  if (n_stage %in% c("x", "0")) {
    score <- score + 0
  } else if (n_stage == "1") {
    score <- score + 2
  } else {
    stop("Error: Invalid N Stage. Use 'Nx', 'N0', or 'N1'.")
  }

  if (tumor_size_cm >= 10) {
    score <- score + 1
  }

  if (nuclear_grade == 3) {
    score <- score + 1
  } else if (nuclear_grade == 4) {
    score <- score + 3
  }

  if (necrosis == 1) {
    score <- score + 2
  }

  risk_group <- ""
  metastasis_free_survival <- list()
  cancer_specific_survival <- list()

  if (score <= 2) {
    risk_group <- "Low Risk"
    metastasis_free_survival <- list("1yr" = "98.8%", "3yr" = "98.1%", "5yr" = "97.1%", "10yr" = "94.6%")
    cancer_specific_survival <- list("1yr" = "99.5%", "3yr" = "99.4%", "5yr" = "99.0%", "10yr" = "97.0%")
  } else if (score >= 3 && score <= 5) {
    risk_group <- "Intermediate Risk"
    metastasis_free_survival <- list("1yr" = "95.5%", "3yr" = "88.6%", "5yr" = "81.4%", "10yr" = "69.6%")
    cancer_specific_survival <- list("1yr" = "98.5%", "3yr" = "95.2%", "5yr" = "91.1%", "10yr" = "81.0%")
  } else {
    risk_group <- "High Risk"
    metastasis_free_survival <- list("1yr" = "77.5%", "3yr" = "53.2%", "5yr" = "37.1%", "10yr" = "19.8%")
    cancer_specific_survival <- list("1yr" = "89.0%", "3yr" = "66.7%", "5yr" = "53.1%", "10yr" = "31.0%")
  }

  return(list(
    Score = score,
    Risk_Group = risk_group,
    Progression_Metastasis_Free_Survival = metastasis_free_survival,
    Cancer_Specific_Survival = cancer_specific_survival,
    Inputs = list(
      T_Stage = t_stage,
      N_Stage = n_stage,
      Size_cm = tumor_size_cm,
      Grade = nuclear_grade,
      Necrosis = necrosis
    )
  ))
}

duke_iscvid_2023 <- function(pathologic_criteria, major_microbiology, major_imaging, predisposition, fever, vascular, immunologic, minor_microbiology) {

  pathologic_criteria <- as.numeric(pathologic_criteria)
  major_microbiology <- as.numeric(major_microbiology)
  major_imaging <- as.numeric(major_imaging)
  predisposition <- as.numeric(predisposition)
  fever <- as.numeric(fever)
  vascular <- as.numeric(vascular)
  immunologic <- as.numeric(immunologic)
  minor_microbiology <- as.numeric(minor_microbiology)

  if (any(is.na(c(pathologic_criteria, major_microbiology, major_imaging, predisposition, fever, vascular, immunologic, minor_microbiology)))) {
    stop("Error: All inputs must be numeric (0 or 1) or convertible strings.")
  }

  num_major <- major_microbiology + major_imaging
  num_minor <- predisposition + fever + vascular + immunologic + minor_microbiology

  classification <- "Rejected"
  reason <- "Does not meet criteria for Possible or Definite IE"

  if (pathologic_criteria == 1) {
    classification <- "Definite IE"
    reason <- "Pathologic criteria met"
  } else if (num_major == 2) {
    classification <- "Definite IE"
    reason <- "2 Major Clinical Criteria met"
  } else if (num_major == 1 && num_minor >= 3) {
    classification <- "Definite IE"
    reason <- "1 Major and >= 3 Minor Clinical Criteria met"
  } else if (num_minor >= 5) {
    classification <- "Definite IE"
    reason <- "5 Minor Clinical Criteria met"
  } else if (num_major == 1 && num_minor >= 1) {
    classification <- "Possible IE"
    reason <- "1 Major and >= 1 Minor Clinical Criteria met"
  } else if (num_minor >= 3) {
    classification <- "Possible IE"
    reason <- ">= 3 Minor Clinical Criteria met"
  }

  return(list(
    Classification = classification,
    Reasoning = reason,
    Criteria_Counts = list(
      Major = num_major,
      Minor = num_minor
    ),
    Inputs = list(
      Pathologic = pathologic_criteria,
      Major_Micro = major_microbiology,
      Major_Imaging = major_imaging,
      Minor_Predisposition = predisposition,
      Minor_Fever = fever,
      Minor_Vascular = vascular,
      Minor_Immunologic = immunologic,
      Minor_Micro = minor_microbiology
    )
  ))
}

em_coding_2023 <- function(problem_complexity, risk_level, tests_ordered = 0, tests_reviewed = 0,
                           prior_notes_reviewed = 0, independent_historian = 0,
                           independent_interpretation = 0, discussion_management = 0) {

  tests_ordered <- as.numeric(tests_ordered)
  tests_reviewed <- as.numeric(tests_reviewed)
  prior_notes_reviewed <- as.numeric(prior_notes_reviewed)
  independent_historian <- as.numeric(independent_historian)
  independent_interpretation <- as.numeric(independent_interpretation)
  discussion_management <- as.numeric(discussion_management)

  if (any(is.na(c(tests_ordered, tests_reviewed, prior_notes_reviewed,
                  independent_historian, independent_interpretation, discussion_management)))) {
    stop("Error: Data inputs must be numeric.")
  }

  problem_score <- 0
  problem_complexity <- tolower(problem_complexity)

  if (grepl("severe exacerbation|threat to life", problem_complexity)) {
    problem_score <- 4
  } else if (grepl("exacerbation|2.*stable|undiagnosed|systemic|complicated injury", problem_complexity)) {
    problem_score <- 3
  } else if (grepl("2.*minor|1.*stable|uncomplicated|hospitalization", problem_complexity)) {
    problem_score <- 2
  } else {
    problem_score <- 1
  }

  risk_score <- 0
  risk_level <- tolower(risk_level)

  if (risk_level == "high" || grepl("parenteral|major surgery|hospitalized|dnr", risk_level)) {
    risk_score <- 4
  } else if (risk_level == "moderate" || grepl("prescription|minor surgery|social doh", risk_level)) {
    risk_score <- 3
  } else if (risk_level == "low") {
    risk_score <- 2
  } else {
    risk_score <- 1
  }

  data_score <- 1

  l4_cat1_count <- tests_ordered + tests_reviewed + prior_notes_reviewed + independent_historian
  l4_cat1_met <- ifelse(l4_cat1_count >= 3, 1, 0)
  l4_cat2_met <- independent_interpretation
  l4_cat3_met <- discussion_management

  categories_met_count <- l4_cat1_met + l4_cat2_met + l4_cat3_met

  if (categories_met_count >= 2) {
    data_score <- 4
  } else if (categories_met_count >= 1) {
    data_score <- 3
  } else {
    l3_cat1_count <- tests_ordered + tests_reviewed + prior_notes_reviewed
    l3_cat1_met <- ifelse(l3_cat1_count >= 2, 1, 0)
    l3_cat2_met <- independent_historian

    if (l3_cat1_met == 1 || l3_cat2_met == 1) {
      data_score <- 2
    } else {
      data_score <- 1
    }
  }

  scores <- sort(c(problem_score, risk_score, data_score), decreasing = TRUE)

  final_level <- scores[2]

  cpt_code <- ""
  mdm_level <- ""

  if (final_level >= 4) {
    cpt_code <- "99285"
    mdm_level <- "High"
  } else if (final_level == 3) {
    cpt_code <- "99284"
    mdm_level <- "Moderate"
  } else if (final_level == 2) {
    cpt_code <- "99283"
    mdm_level <- "Low"
  } else {
    cpt_code <- "99282"
    mdm_level <- "Straightforward"
  }

  return(list(
    CPT_Code = cpt_code,
    MDM_Level = mdm_level,
    Component_Scores = list(
      COPA_Level = switch(problem_score, "Minimal", "Low", "Moderate", "High"),
      Risk_Level = switch(risk_score, "Minimal", "Low", "Moderate", "High"),
      Data_Level = switch(data_score, "Minimal/None", "Limited", "Moderate", "Extensive")
    ),
    Data_Breakdown = list(
      Tests_Documents_Count = l4_cat1_count,
      Independent_Interpretation = as.logical(independent_interpretation),
      Discussion_Management = as.logical(discussion_management)
    )
  ))
}

score_2helps2b <- function(frequency_gt_2hz, sporadic_discharges, lpd_lrda_bipd,
                           plus_features, prior_seizure, bird) {

  frequency_gt_2hz <- as.numeric(frequency_gt_2hz)
  sporadic_discharges <- as.numeric(sporadic_discharges)
  lpd_lrda_bipd <- as.numeric(lpd_lrda_bipd)
  plus_features <- as.numeric(plus_features)
  prior_seizure <- as.numeric(prior_seizure)
  bird <- as.numeric(bird)

  if (any(is.na(c(frequency_gt_2hz, sporadic_discharges, lpd_lrda_bipd,
                  plus_features, prior_seizure, bird)))) {
    stop("Error: All inputs must be numeric (0 or 1) or convertible strings.")
  }

  score <- 0

  if (frequency_gt_2hz == 1) score <- score + 1
  if (sporadic_discharges == 1) score <- score + 1
  if (lpd_lrda_bipd == 1) score <- score + 1
  if (plus_features == 1) score <- score + 1
  if (prior_seizure == 1) score <- score + 1
  if (bird == 1) score <- score + 2

  seizure_risk <- ""
  monitoring_rec <- ""
  risk_category <- ""

  if (score == 0) {
    seizure_risk <- "5%"
    risk_category <- "Low Risk"
    monitoring_rec <- "1 hour (Screening EEG typically sufficient)"
  } else if (score == 1) {
    seizure_risk <- "12%"
    risk_category <- "Medium Risk"
    monitoring_rec <- "12 hours"
  } else if (score == 2) {
    seizure_risk <- "27%"
    risk_category <- "High Risk"
    monitoring_rec <- "24+ hours"
  } else if (score == 3) {
    seizure_risk <- "50%"
    risk_category <- "High Risk"
    monitoring_rec <- "24+ hours"
  } else if (score == 4) {
    seizure_risk <- "73%"
    risk_category <- "High Risk"
    monitoring_rec <- "24+ hours"
  } else if (score == 5) {
    seizure_risk <- "88%"
    risk_category <- "High Risk"
    monitoring_rec <- "24+ hours"
  } else {
    seizure_risk <- ">95%"
    risk_category <- "High Risk"
    monitoring_rec <- "24+ hours"
  }

  return(list(
    Score = score,
    Risk_Category = risk_category,
    Seizure_Risk_Percentage = seizure_risk,
    Recommended_Monitoring = monitoring_rec,
    Inputs = list(
      Freq_gt_2Hz = frequency_gt_2hz,
      Sporadic_Discharges = sporadic_discharges,
      LPD_LRDA_BIPD = lpd_lrda_bipd,
      Plus_Features = plus_features,
      Prior_Seizure = prior_seizure,
      BIRD_Present = bird
    )
  ))
}

score_4at <- function(alertness, amt4_mistakes, attention_months, acute_change) {

  alertness <- as.numeric(alertness)
  amt4_mistakes <- as.numeric(amt4_mistakes)
  attention_months <- as.numeric(attention_months)
  acute_change <- as.numeric(acute_change)

  if (any(is.na(c(alertness, amt4_mistakes, attention_months, acute_change)))) {
    stop("Error: Inputs must be numeric.")
  }

  score <- 0

  if (alertness == 2) {
    score <- score + 4
  } else if (alertness == 1) {
    score <- score + 0
  } else {
    score <- score + 0
  }

  if (amt4_mistakes >= 2) {
    score <- score + 2
  } else if (amt4_mistakes == 1) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (attention_months == 2) {
    score <- score + 2
  } else if (attention_months == 1) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (acute_change == 1) {
    score <- score + 4
  } else {
    score <- score + 0
  }

  interpretation <- ""
  if (score >= 4) {
    interpretation <- "Possible delirium +/- cognitive impairment"
  } else if (score >= 1) {
    interpretation <- "Possible cognitive impairment"
  } else {
    interpretation <- "Delirium or severe cognitive impairment unlikely"
  }

  return(list(
    Total_Score = score,
    Interpretation = interpretation,
    Item_Breakdown = list(
      Alertness_Score = if(alertness == 2) 4 else 0,
      AMT4_Score = if(amt4_mistakes >= 2) 2 else amt4_mistakes,
      Attention_Score = if(attention_months == 2) 2 else attention_months,
      Acute_Change_Score = if(acute_change == 1) 4 else 0
    )
  ))
}

score_4peps <- function(age, sex, chronic_resp_disease, heart_rate_lt_80,
                        chest_pain_and_dyspnea, estrogen_use, history_vte,
                        syncope, immobility_4wks, o2_sat_lt_95,
                        calf_pain_unilateral_edema, pe_most_likely) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  chronic_resp_disease <- as.numeric(chronic_resp_disease)
  heart_rate_lt_80 <- as.numeric(heart_rate_lt_80)
  chest_pain_and_dyspnea <- as.numeric(chest_pain_and_dyspnea)
  estrogen_use <- as.numeric(estrogen_use)
  history_vte <- as.numeric(history_vte)
  syncope <- as.numeric(syncope)
  immobility_4wks <- as.numeric(immobility_4wks)
  o2_sat_lt_95 <- as.numeric(o2_sat_lt_95)
  calf_pain_unilateral_edema <- as.numeric(calf_pain_unilateral_edema)
  pe_most_likely <- as.numeric(pe_most_likely)

  if (any(is.na(c(age, chronic_resp_disease, heart_rate_lt_80,
                  chest_pain_and_dyspnea, estrogen_use, history_vte,
                  syncope, immobility_4wks, o2_sat_lt_95,
                  calf_pain_unilateral_edema, pe_most_likely)))) {
    stop("Error: All binary inputs must be numeric (0 or 1) and age must be numeric.")
  }

  score <- 0

  if (age < 50) {
    score <- score - 2
  } else if (age >= 50 && age <= 64) {
    score <- score - 1
  }

  if (sex == "male") score <- score + 2

  if (chronic_resp_disease == 1) score <- score - 1
  if (heart_rate_lt_80 == 1) score <- score - 1

  if (chest_pain_and_dyspnea == 1) score <- score + 1
  if (estrogen_use == 1) score <- score + 2
  if (history_vte == 1) score <- score + 2
  if (syncope == 1) score <- score + 2
  if (immobility_4wks == 1) score <- score + 2
  if (o2_sat_lt_95 == 1) score <- score + 3
  if (calf_pain_unilateral_edema == 1) score <- score + 3
  if (pe_most_likely == 1) score <- score + 5

  risk_category <- ""
  strategy <- ""

  if (score < 0) {
    risk_category <- "Very Low Risk (<2%)"
    strategy <- "PE ruled out without testing."
  } else if (score >= 0 && score <= 5) {
    risk_category <- "Low Risk"
    strategy <- "PE ruled out if D-dimer < 1.0 ug/mL (1000 ng/mL)."
  } else if (score >= 6 && score <= 12) {
    risk_category <- "Moderate Risk"
    strategy <- paste0("PE ruled out if D-dimer < 0.5 ug/mL (500 ng/mL) OR ",
                       "Age-Adjusted (Age x 0.01 ug/mL).")
  } else {
    risk_category <- "High Risk (>65%)"
    strategy <- "PE cannot be ruled out without imaging (CTPA)."
  }

  return(list(
    Score = score,
    Risk_Category = risk_category,
    Management_Strategy = strategy,
    Inputs = list(
      Age = age,
      Sex = sex,
      Chronic_Resp_Disease = chronic_resp_disease,
      HR_lt_80 = heart_rate_lt_80,
      ChestPain_Dyspnea = chest_pain_and_dyspnea,
      Estrogen = estrogen_use,
      Hx_VTE = history_vte,
      Syncope = syncope,
      Immobility = immobility_4wks,
      O2_Sat_lt_95 = o2_sat_lt_95,
      Calf_Pain_Edema = calf_pain_unilateral_edema,
      PE_Likely = pe_most_likely
    )
  ))
}

isaric_4c_mortality <- function(age, sex, comorbidities, resp_rate, o2_sat, gcs, urea, crp, urea_units = "mmol/L", crp_units = "mg/L") {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  comorbidities <- as.numeric(comorbidities)
  resp_rate <- as.numeric(resp_rate)
  o2_sat <- as.numeric(o2_sat)
  gcs <- as.numeric(gcs)
  urea <- as.numeric(urea)
  crp <- as.numeric(crp)

  if (any(is.na(c(age, comorbidities, resp_rate, o2_sat, gcs, urea, crp)))) {
    stop("Error: All numeric inputs (Age, Comorbidities, RR, O2, GCS, Urea, CRP) must be valid numbers.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age < 50) {
    score <- score + 0
  } else if (age >= 50 && age <= 59) {
    score <- score + 2
  } else if (age >= 60 && age <= 69) {
    score <- score + 4
  } else if (age >= 70 && age <= 79) {
    score <- score + 6
  } else {
    score <- score + 7
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (comorbidities == 1) {
    score <- score + 1
  } else if (comorbidities >= 2) {
    score <- score + 2
  }

  if (resp_rate < 20) {
    score <- score + 0
  } else if (resp_rate >= 20 && resp_rate <= 29) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (o2_sat < 92) {
    score <- score + 2
  }

  if (gcs < 15) {
    score <- score + 2
  }

  urea_mmol <- urea
  if (urea_units == "mg/dL") {
    urea_mmol <- urea / 2.8
  }

  if (urea_mmol < 7) {
    score <- score + 0
  } else if (urea_mmol >= 7 && urea_mmol <= 14) {
    score <- score + 1
  } else {
    score <- score + 3
  }

  crp_mg_L <- crp
  if (crp_units == "mg/dL") {
    crp_mg_L <- crp * 10
  }

  if (crp_mg_L < 50) {
    score <- score + 0
  } else if (crp_mg_L >= 50 && crp_mg_L < 100) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  mortality_rate <- ""
  risk_group <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_group <- "Low Risk"
    mortality_rate <- "1.2 - 1.7%"
    recommendation <- "May be suitable for outpatient management."
  } else if (score >= 4 && score <= 8) {
    risk_group <- "Intermediate Risk"
    mortality_rate <- "9.1 - 9.9%"
    recommendation <- "May be suitable for general inpatient treatment."
  } else if (score >= 9 && score <= 14) {
    risk_group <- "High Risk"
    mortality_rate <- "31.4 - 34.9%"
    recommendation <- "May require aggressive treatment and higher monitoring."
  } else {
    risk_group <- "Very High Risk"
    mortality_rate <- "61.5 - 66.2%"
    recommendation <- "Urgent critical care assessment recommended."
  }

  return(list(
    Total_Score = score,
    Risk_Group = risk_group,
    Est_In_Hospital_Mortality = mortality_rate,
    Recommendation = recommendation,
    Inputs_Processed = list(
      Age_Points = if(age >= 80) 7 else if(age >= 70) 6 else if(age >= 60) 4 else if(age >= 50) 2 else 0,
      Sex_Points = if(sex == "male") 1 else 0,
      Comorb_Points = if(comorbidities >= 2) 2 else if(comorbidities == 1) 1 else 0,
      RR_Points = if(resp_rate >= 30) 2 else if(resp_rate >= 20) 1 else 0,
      O2_Points = if(o2_sat < 92) 2 else 0,
      GCS_Points = if(gcs < 15) 2 else 0,
      Urea_Points = if(urea_mmol > 14) 3 else if(urea_mmol >= 7) 1 else 0,
      CRP_Points = if(crp_mg_L >= 100) 2 else if(crp_mg_L >= 50) 1 else 0
    )
  ))
}

hit_4ts <- function(thrombocytopenia_score, timing_score, thrombosis_score, other_causes_score) {

  thrombocytopenia_score <- as.numeric(thrombocytopenia_score)
  timing_score <- as.numeric(timing_score)
  thrombosis_score <- as.numeric(thrombosis_score)
  other_causes_score <- as.numeric(other_causes_score)

  if (any(is.na(c(thrombocytopenia_score, timing_score, thrombosis_score, other_causes_score)))) {
    stop("Error: All inputs must be numeric integers (0, 1, or 2).")
  }

  if (any(c(thrombocytopenia_score, timing_score, thrombosis_score, other_causes_score) < 0) ||
      any(c(thrombocytopenia_score, timing_score, thrombosis_score, other_causes_score) > 2)) {
    stop("Error: Score components must be 0, 1, or 2.")
  }

  total_score <- thrombocytopenia_score + timing_score + thrombosis_score + other_causes_score

  probability <- ""
  interpretation <- ""
  management <- ""

  if (total_score <= 3) {
    probability <- "Low Probability (<5%)"
    interpretation <- "HIT is unlikely."
    management <- "Continue to monitor platelet counts. Consider other causes."
  } else if (total_score <= 5) {
    probability <- "Intermediate Probability (~14%)"
    interpretation <- "HIT is possible."
    management <- "Consider stopping heparin and starting alternative anticoagulation. Order HIT testing (PF4 ELISA)."
  } else {
    probability <- "High Probability (~64%)"
    interpretation <- "HIT is likely."
    management <- "Stop heparin immediately. Start alternative anticoagulation. Order confirmatory testing (Serotonin Release Assay)."
  }

  return(list(
    Total_Score = total_score,
    Pretest_Probability = probability,
    Interpretation = interpretation,
    Management_Guidance = management,
    Inputs = list(
      Thrombocytopenia_Pts = thrombocytopenia_score,
      Timing_Pts = timing_score,
      Thrombosis_Pts = thrombosis_score,
      Other_Causes_Pts = other_causes_score
    )
  ))
}

six_minute_walk_prediction <- function(sex, age, height_cm, weight_kg, actual_distance_m = NULL) {

  age <- as.numeric(age)
  height_cm <- as.numeric(height_cm)
  weight_kg <- as.numeric(weight_kg)

  if (!is.null(actual_distance_m)) {
    actual_distance_m <- as.numeric(actual_distance_m)
  }

  if (any(is.na(c(age, height_cm, weight_kg)))) {
    stop("Error: Age, Height, and Weight must be numeric.")
  }

  sex <- tolower(as.character(sex))
  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  pred_distance <- 0
  lln <- 0

  if (sex == "male") {
    pred_distance <- (7.57 * height_cm) - (5.02 * age) - (1.76 * weight_kg) - 309
    lln <- pred_distance - 153
  } else {
    pred_distance <- (2.11 * height_cm) - (2.29 * weight_kg) - (5.78 * age) + 667
    lln <- pred_distance - 139
  }

  pct_predicted <- NA
  interpretation <- ""

  if (!is.null(actual_distance_m)) {
    pct_predicted <- (actual_distance_m / pred_distance) * 100

    if (actual_distance_m < lln) {
      interpretation <- "Below Lower Limit of Normal (LLN)"
    } else {
      interpretation <- "Normal (Above LLN)"
    }
  }

  return(list(
    Predicted_Distance_m = round(pred_distance, 1),
    Lower_Limit_Normal_m = round(lln, 1),
    Actual_Distance_m = if (!is.null(actual_distance_m)) actual_distance_m else NA,
    Percent_Predicted = if (!is.null(pct_predicted)) round(pct_predicted, 1) else NA,
    Interpretation = interpretation,
    Inputs = list(
      Sex = sex,
      Age = age,
      Height_cm = height_cm,
      Weight_kg = weight_kg
    )
  ))
}

a_a_gradient <- function(pao2, paco2, fio2, age = NULL, patm = 760, elevation_m = NULL) {

  pao2 <- as.numeric(pao2)
  paco2 <- as.numeric(paco2)
  fio2 <- as.numeric(fio2)
  patm <- as.numeric(patm)

  if (!is.null(age)) {
    age <- as.numeric(age)
  }

  if (!is.null(elevation_m)) {
    elevation_m <- as.numeric(elevation_m)
  }

  if (any(is.na(c(pao2, paco2, fio2)))) {
    stop("Error: PaO2, PaCO2, and FiO2 must be valid numeric inputs.")
  }

  if (!is.null(elevation_m)) {
    patm <- 760 * exp(elevation_m / -7000)
  }

  fio2_fraction <- fio2
  if (fio2 > 1.0) {
    fio2_fraction <- fio2 / 100
  }

  p_alv_o2 <- (fio2_fraction * (patm - 47)) - (paco2 / 0.8)

  gradient <- p_alv_o2 - pao2

  expected_gradient <- NA
  interpretation <- "Correlation with clinical picture required"

  if (!is.null(age)) {
    expected_gradient <- (age / 4) + 4

    if (gradient > expected_gradient) {
      interpretation <- "Abnormally Elevated A-a Gradient (Suggests V/Q mismatch, shunt, or diffusion defect)"
    } else {
      interpretation <- "Normal A-a Gradient (Suggests hypoventilation or low environmental O2)"
    }
  } else {
    if (gradient > 15) {
      interpretation <- "Likely Elevated (Reference range dependent on age)"
    } else {
      interpretation <- "Likely Normal (Reference range dependent on age)"
    }
  }

  return(list(
    A_a_Gradient_mmHg = round(gradient, 1),
    PAO2_Alveolar_Oxygen_mmHg = round(p_alv_o2, 1),
    Expected_Gradient_mmHg = if (!is.na(expected_gradient)) round(expected_gradient, 1) else NA,
    Interpretation = interpretation,
    Inputs = list(
      PaO2 = pao2,
      PaCO2 = paco2,
      FiO2 = fio2_fraction,
      Patm_Used = round(patm, 1),
      Age = if (!is.null(age)) age else NA
    )
  ))
}

aap_pediatric_htn_guidelines <- function(age, systolic, diastolic, sbp_90th = NULL, sbp_95th = NULL, dbp_90th = NULL, dbp_95th = NULL) {

  age <- as.numeric(age)
  systolic <- as.numeric(systolic)
  diastolic <- as.numeric(diastolic)

  if (!is.null(sbp_90th)) sbp_90th <- as.numeric(sbp_90th)
  if (!is.null(sbp_95th)) sbp_95th <- as.numeric(sbp_95th)
  if (!is.null(dbp_90th)) dbp_90th <- as.numeric(dbp_90th)
  if (!is.null(dbp_95th)) dbp_95th <- as.numeric(dbp_95th)

  if (any(is.na(c(age, systolic, diastolic)))) {
    stop("Error: Age, Systolic BP, and Diastolic BP must be valid numbers.")
  }

  category <- ""
  management <- ""

  if (age >= 13) {
    if (systolic >= 140 || diastolic >= 90) {
      category <- "Stage 2 Hypertension"
      management <- "Refer to subspecialty care within 1 week or start evaluation/treatment."
    } else if ((systolic >= 130 && systolic <= 139) || (diastolic >= 80 && diastolic <= 89)) {
      category <- "Stage 1 Hypertension"
      management <- "Recheck in 1-2 weeks. Lifestyle interventions."
    } else if (systolic >= 120 && systolic <= 129 && diastolic < 80) {
      category <- "Elevated Blood Pressure"
      management <- "Recheck in 6 months. Lifestyle interventions."
    } else {
      category <- "Normal Blood Pressure"
      management <- "Recheck at next routine visit."
    }

  } else {
    if (any(is.null(c(sbp_90th, sbp_95th, dbp_90th, dbp_95th)))) {
      stop("Error: For children < 13 years, you must provide the 90th and 95th percentile BP thresholds.")
    }

    sbp_95_plus_12 <- sbp_95th + 12
    dbp_95_plus_12 <- dbp_95th + 12

    if ((systolic >= sbp_95_plus_12 || systolic >= 140) || (diastolic >= dbp_95_plus_12 || diastolic >= 90)) {
      category <- "Stage 2 Hypertension"
      management <- "Refer to subspecialty care within 1 week or start evaluation/treatment."
    } else if ((systolic >= sbp_95th || systolic >= 130) || (diastolic >= dbp_95th || diastolic >= 80)) {
      category <- "Stage 1 Hypertension"
      management <- "Recheck in 1-2 weeks. Lifestyle interventions."
    } else if ((systolic >= sbp_90th || systolic >= 120) || (diastolic >= dbp_90th || diastolic >= 80)) {
      category <- "Elevated Blood Pressure"
      management <- "Recheck in 6 months. Lifestyle interventions."
    } else {
      category <- "Normal Blood Pressure"
      management <- "Recheck at next routine visit."
    }
  }

  return(list(
    Age = age,
    Blood_Pressure = paste0(systolic, "/", diastolic),
    Classification = category,
    Management_Recommendation = management,
    Applied_Criteria = if (age >= 13) "Adult (Static) Guidelines" else "Pediatric (Percentile) Guidelines"
  ))
}

abbey_pain_scale <- function(vocalization, facial_expression, body_language, behavioral_change, physiological_change, physical_changes) {

  vocalization <- as.numeric(vocalization)
  facial_expression <- as.numeric(facial_expression)
  body_language <- as.numeric(body_language)
  behavioral_change <- as.numeric(behavioral_change)
  physiological_change <- as.numeric(physiological_change)
  physical_changes <- as.numeric(physical_changes)

  if (any(is.na(c(vocalization, facial_expression, body_language, behavioral_change, physiological_change, physical_changes)))) {
    stop("Error: All inputs must be numeric integers (0, 1, 2, or 3).")
  }

  if (any(c(vocalization, facial_expression, body_language, behavioral_change, physiological_change, physical_changes) < 0) ||
      any(c(vocalization, facial_expression, body_language, behavioral_change, physiological_change, physical_changes) > 3)) {
    stop("Error: Score components must be between 0 (Absent) and 3 (Severe).")
  }

  total_score <- vocalization + facial_expression + body_language + behavioral_change + physiological_change + physical_changes

  pain_category <- ""
  management <- ""

  if (total_score <= 2) {
    pain_category <- "No pain"
    management <- "Routine care. Re-evaluate as needed."
  } else if (total_score <= 7) {
    pain_category <- "Mild pain"
    management <- "Consider non-pharmacologic interventions or mild analgesia. Re-evaluate."
  } else if (total_score <= 13) {
    pain_category <- "Moderate pain"
    management <- "Pain intervention required. Re-evaluate effectiveness."
  } else {
    pain_category <- "Severe pain"
    management <- "Urgent pain intervention required. Re-evaluate effectiveness closely."
  }

  return(list(
    Total_Score = total_score,
    Pain_Category = pain_category,
    Management_Recommendation = management,
    Inputs = list(
      Vocalization = vocalization,
      Facial_Expression = facial_expression,
      Body_Language = body_language,
      Behavioral_Change = behavioral_change,
      Physiological_Change = physiological_change,
      Physical_Changes = physical_changes
    )
  ))
}

ais_inhalation <- function(grade) {

  grade <- as.numeric(grade)

  if (is.na(grade) || !grade %in% 0:4) {
    stop("Error: Grade must be an integer between 0 and 4.")
  }

  description <- ""
  mortality_risk <- ""

  if (grade == 0) {
    description <- "No Injury: Absence of carbonaceous deposits, erythema, edema, bronchorrhea, or obstruction."
    mortality_risk <- "Low (similar to baseline burn risk)"
  } else if (grade == 1) {
    description <- "Mild Injury: Minor/patchy erythema, carbonaceous deposits in proximal or distal bronchi."
    mortality_risk <- "~20% (Increased risk relative to no inhalation injury)"
  } else if (grade == 2) {
    description <- "Moderate Injury: Moderate erythema, carbonaceous deposits, bronchorrhea, with or without compromise."
    mortality_risk <- "~30% (Significantly increased risk)"
  } else if (grade == 3) {
    description <- "Severe Injury: Severe inflammation with friability, copious deposits, bronchorrhea, obstruction."
    mortality_risk <- "~50% (High risk)"
  } else {
    description <- "Massive Injury: Mucosal sloughing, necrosis, endoluminal obliteration."
    mortality_risk <- "~90% (Very high risk)"
  }

  return(list(
    Grade = grade,
    Classification = description,
    Estimated_Mortality = mortality_risk
  ))
}

abbreviated_mental_test_10 <- function(age_correct, time_correct, address_recall_correct, year_correct, place_correct, recognition_correct, dob_correct, ww1_date_correct, monarch_correct, count_backwards_correct) {

  age_correct <- as.numeric(age_correct)
  time_correct <- as.numeric(time_correct)
  address_recall_correct <- as.numeric(address_recall_correct)
  year_correct <- as.numeric(year_correct)
  place_correct <- as.numeric(place_correct)
  recognition_correct <- as.numeric(recognition_correct)
  dob_correct <- as.numeric(dob_correct)
  ww1_date_correct <- as.numeric(ww1_date_correct)
  monarch_correct <- as.numeric(monarch_correct)
  count_backwards_correct <- as.numeric(count_backwards_correct)

  if (any(is.na(c(age_correct, time_correct, address_recall_correct, year_correct, place_correct, recognition_correct, dob_correct, ww1_date_correct, monarch_correct, count_backwards_correct)))) {
    stop("Error: All inputs must be numeric integers (0 for incorrect, 1 for correct).")
  }

  total_score <- age_correct + time_correct + address_recall_correct + year_correct + place_correct + recognition_correct + dob_correct + ww1_date_correct + monarch_correct + count_backwards_correct

  interpretation <- ""
  if (total_score >= 8) {
    interpretation <- "Normal (Cognitive impairment unlikely)"
  } else {
    interpretation <- "Abnormal (Possible cognitive impairment)"
  }

  return(list(
    Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Age = age_correct,
      Time = time_correct,
      Address_Recall = address_recall_correct,
      Year = year_correct,
      Place = place_correct,
      Recognition = recognition_correct,
      DOB = dob_correct,
      WW1_Date = ww1_date_correct,
      Monarch = monarch_correct,
      Count_Backwards = count_backwards_correct
    )
  ))
}

abbreviated_mental_test_4 <- function(age_correct, dob_correct, place_correct, year_correct) {

  age_correct <- as.numeric(age_correct)
  dob_correct <- as.numeric(dob_correct)
  place_correct <- as.numeric(place_correct)
  year_correct <- as.numeric(year_correct)

  if (any(is.na(c(age_correct, dob_correct, place_correct, year_correct)))) {
    stop("Error: All inputs must be numeric integers (0 for incorrect, 1 for correct).")
  }

  total_score <- age_correct + dob_correct + place_correct + year_correct

  interpretation <- ""
  if (total_score < 4) {
    interpretation <- "Abnormal (Possible cognitive impairment)"
  } else {
    interpretation <- "Normal (Cognitive impairment unlikely)"
  }

  return(list(
    Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Age = age_correct,
      Date_of_Birth = dob_correct,
      Place = place_correct,
      Year = year_correct
    )
  ))
}

abc_massive_transfusion <- function(penetrating_mechanism, positive_fast, sbp_mmHg, hr_bpm) {

  penetrating_mechanism <- as.numeric(penetrating_mechanism)
  positive_fast <- as.numeric(positive_fast)
  sbp_mmHg <- as.numeric(sbp_mmHg)
  hr_bpm <- as.numeric(hr_bpm)

  if (any(is.na(c(penetrating_mechanism, positive_fast, sbp_mmHg, hr_bpm)))) {
    stop("Error: Inputs must be numeric.")
  }

  score <- 0

  if (penetrating_mechanism == 1) {
    score <- score + 1
  }

  if (positive_fast == 1) {
    score <- score + 1
  }

  if (sbp_mmHg <= 90) {
    score <- score + 1
  }

  if (hr_bpm >= 120) {
    score <- score + 1
  }

  interpretation <- ""
  if (score >= 2) {
    interpretation <- "Likely to require massive transfusion (Sensitivity ~75-90%, Specificity ~67-88%)"
  } else {
    interpretation <- "Unlikely to require massive transfusion"
  }

  return(list(
    Total_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Penetrating_Mechanism = penetrating_mechanism,
      Positive_FAST = positive_fast,
      SBP_mmHg = sbp_mmHg,
      HR_bpm = hr_bpm
    )
  ))
}

abc2_ich_volume <- function(hemorrhage_length_cm, hemorrhage_width_cm, num_slices, slice_thickness_mm) {

  hemorrhage_length_cm <- as.numeric(hemorrhage_length_cm)
  hemorrhage_width_cm <- as.numeric(hemorrhage_width_cm)
  num_slices <- as.numeric(num_slices)
  slice_thickness_mm <- as.numeric(slice_thickness_mm)

  if (any(is.na(c(hemorrhage_length_cm, hemorrhage_width_cm, num_slices, slice_thickness_mm)))) {
    stop("Error: All inputs must be numeric.")
  }


  slice_thickness_cm <- slice_thickness_mm / 10
  c_vertical <- num_slices * slice_thickness_cm
  volume_cc <- (hemorrhage_length_cm * hemorrhage_width_cm * c_vertical) / 2

  interpretation <- ""
  if (volume_cc >= 30) {
    interpretation <- "Volume >= 30 mL (Associated with higher mortality and poor functional outcome)"
  } else {
    interpretation <- "Volume < 30 mL"
  }

  return(list(
    ICH_Volume_mL = round(volume_cc, 1),
    Interpretation = interpretation,
    Inputs = list(
      A_Length_cm = hemorrhage_length_cm,
      B_Width_cm = hemorrhage_width_cm,
      C_Depth_cm = round(c_vertical, 1)
    )
  ))
}

abcd2_tia <- function(age, bp_systolic, bp_diastolic, clinical_features, duration_min, diabetes) {

  age <- as.numeric(age)
  bp_systolic <- as.numeric(bp_systolic)
  bp_diastolic <- as.numeric(bp_diastolic)
  clinical_features <- as.numeric(clinical_features)
  duration_min <- as.numeric(duration_min)
  diabetes <- as.numeric(diabetes)

  if (any(is.na(c(age, bp_systolic, bp_diastolic, clinical_features, duration_min, diabetes)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 60) {
    score <- score + 1
  }

  if (bp_systolic >= 140 || bp_diastolic >= 90) {
    score <- score + 1
  }

  if (clinical_features == 2) {
    score <- score + 2
  } else if (clinical_features == 1) {
    score <- score + 1
  }

  if (duration_min >= 60) {
    score <- score + 2
  } else if (duration_min >= 10) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  risk_2_day <- ""
  risk_7_day <- ""
  risk_90_day <- ""
  risk_category <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_category <- "Low Risk"
    risk_2_day <- "1.0%"
    risk_7_day <- "1.2%"
    risk_90_day <- "3.1%"
    recommendation <- "Hospital observation may be unnecessary without another indication (e.g., new afib)."
  } else if (score <= 5) {
    risk_category <- "Moderate Risk"
    risk_2_day <- "4.1%"
    risk_7_day <- "5.9%"
    risk_90_day <- "9.8%"
    recommendation <- "Hospital observation justified in most situations."
  } else {
    risk_category <- "High Risk"
    risk_2_day <- "8.1%"
    risk_7_day <- "11.7%"
    risk_90_day <- "17.8%"
    recommendation <- "Hospital observation justified in most situations."
  }

  return(list(
    Score = score,
    Risk_Category = risk_category,
    Stroke_Risk = list(
      Two_Day = risk_2_day,
      Seven_Day = risk_7_day,
      Ninety_Day = risk_90_day
    ),
    Management = recommendation
  ))
}

abic_alcoholic_hepatitis <- function(age, bilirubin, inr, creatinine, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  bilirubin <- as.numeric(bilirubin)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)

  if (any(is.na(c(age, bilirubin, inr, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_mg_dl <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_mg_dl <- bilirubin / 17.1
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  abic_val <- (age * 0.1) + (bili_mg_dl * 0.08) + (creat_mg_dl * 0.3) + (inr * 0.8)

  risk_group <- ""
  survival_90_day <- ""

  if (abic_val < 6.71) {
    risk_group <- "Low Risk"
    survival_90_day <- "100%"
  } else if (abic_val >= 6.71 && abic_val <= 9.0) {
    risk_group <- "Intermediate Risk"
    survival_90_day <- "70%"
  } else {
    risk_group <- "High Risk"
    survival_90_day <- "25%"
  }

  return(list(
    ABIC_Score = round(abic_val, 2),
    Risk_Group = risk_group,
    Est_90_Day_Survival = survival_90_day,
    Inputs = list(
      Age = age,
      Bilirubin_mg_dL = round(bili_mg_dl, 2),
      INR = inr,
      Creatinine_mg_dL = round(creat_mg_dl, 2)
    )
  ))
}

aims_tardive_dyskinesia <- function(facial, lips, jaw, tongue, upper_ext, lower_ext, trunk) {

  facial <- as.numeric(facial)
  lips <- as.numeric(lips)
  jaw <- as.numeric(jaw)
  tongue <- as.numeric(tongue)
  upper_ext <- as.numeric(upper_ext)
  lower_ext <- as.numeric(lower_ext)
  trunk <- as.numeric(trunk)

  if (any(is.na(c(facial, lips, jaw, tongue, upper_ext, lower_ext, trunk)))) {
    stop("Error: All inputs must be numeric integers between 0 and 4.")
  }

  inputs <- c(facial, lips, jaw, tongue, upper_ext, lower_ext, trunk)

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All scores must be between 0 (None) and 4 (Severe).")
  }

  total_score <- sum(inputs)

  items_ge_2 <- sum(inputs >= 2)
  items_ge_3 <- sum(inputs >= 3)

  interpretation <- ""
  if (items_ge_3 >= 1 || items_ge_2 >= 2) {
    interpretation <- "Positive for Tardive Dyskinesia (Score 3+ in 1 area OR Score 2+ in 2 areas)"
  } else {
    interpretation <- "Negative for Tardive Dyskinesia"
  }

  return(list(
    Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Facial_Expression = facial,
      Lips_Perioral = lips,
      Jaw = jaw,
      Tongue = tongue,
      Upper_Extremities = upper_ext,
      Lower_Extremities = lower_ext,
      Trunk = trunk
    )
  ))
}

absolute_lymphocyte_count <- function(wbc, lymphs_percent) {

  wbc <- as.numeric(wbc)
  lymphs_percent <- as.numeric(lymphs_percent)

  if (any(is.na(c(wbc, lymphs_percent)))) {
    stop("Error: WBC and Lymphocyte percentage must be numeric.")
  }

  alc <- wbc * (lymphs_percent / 100)

  interpretation <- "Clinical correlation required."

  alc_for_interp <- alc
  if (alc < 50) {
    alc_for_interp <- alc * 1000
  }

  if (alc_for_interp < 1000) {
    interpretation <- "Lymphopenia"
  } else if (alc_for_interp > 4000) {
    interpretation <- "Lymphocytosis"
  } else {
    interpretation <- "Normal Range (approximate)"
  }

  return(list(
    ALC = round(alc, 2),
    Interpretation = interpretation,
    Inputs = list(
      WBC = wbc,
      Lymphocyte_Percent = lymphs_percent
    )
  ))
}

absolute_neutrophil_count <- function(wbc, neutrophils_percent, bands_percent = 0) {

  wbc <- as.numeric(wbc)
  neutrophils_percent <- as.numeric(neutrophils_percent)
  bands_percent <- as.numeric(bands_percent)

  if (any(is.na(c(wbc, neutrophils_percent, bands_percent)))) {
    stop("Error: WBC, Neutrophils, and Bands must be numeric.")
  }

  anc <- wbc * ((neutrophils_percent + bands_percent) / 100)

  interpretation <- ""

  anc_check <- anc
  if (anc < 100) {
    anc_check <- anc * 1000
  }

  if (anc_check < 500) {
    interpretation <- "Severe Neutropenia (< 500 cells/uL)"
  } else if (anc_check < 1000) {
    interpretation <- "Moderate Neutropenia (500-1000 cells/uL)"
  } else if (anc_check < 1500) {
    interpretation <- "Mild Neutropenia (1000-1500 cells/uL)"
  } else {
    interpretation <- "Normal Range (> 1500 cells/uL)"
  }

  return(list(
    ANC = round(anc, 2),
    Interpretation = interpretation,
    Inputs = list(
      WBC = wbc,
      Segs_Percent = neutrophils_percent,
      Bands_Percent = bands_percent
    )
  ))
}

abuse_assessment_screen <- function(ever_abused, past_year_physically_hurt, forced_sexual_activities, afraid_of_partner, pregnant_and_hurt = 0) {

  ever_abused <- as.numeric(ever_abused)
  past_year_physically_hurt <- as.numeric(past_year_physically_hurt)
  forced_sexual_activities <- as.numeric(forced_sexual_activities)
  afraid_of_partner <- as.numeric(afraid_of_partner)
  pregnant_and_hurt <- as.numeric(pregnant_and_hurt)

  if (any(is.na(c(ever_abused, past_year_physically_hurt, forced_sexual_activities, afraid_of_partner, pregnant_and_hurt)))) {
    stop("Error: All inputs must be numeric (0 for No, 1 for Yes).")
  }

  positive_screen <- FALSE
  if (ever_abused == 1 || past_year_physically_hurt == 1 ||
      forced_sexual_activities == 1 || afraid_of_partner == 1 ||
      pregnant_and_hurt == 1) {
    positive_screen <- TRUE
  }

  interpretation <- ""
  if (positive_screen) {
    interpretation <- "Positive Screen: Potential Domestic Abuse/IPV Detected."
  } else {
    interpretation <- "Negative Screen: No abuse disclosed at this time."
  }

  recommendation <- ""
  if (positive_screen) {
    recommendation <- "Validate experience, assess immediate safety, document findings, and refer to resources (e.g., social services, shelters)."
  } else {
    recommendation <- "Monitor and consider revisiting in future visits."
  }

  return(list(
    Screen_Result = if(positive_screen) "Positive" else "Negative",
    Interpretation = interpretation,
    Recommendation = recommendation,
    Questions_Affirmed = list(
      Ever_Abused = ever_abused == 1,
      Hurt_Past_Year = past_year_physically_hurt == 1,
      Forced_Sex = forced_sexual_activities == 1,
      Afraid = afraid_of_partner == 1,
      Pregnant_Hurt = pregnant_and_hurt == 1
    )
  ))
}

acc_aha_hf_staging <- function(risk_factors, structural_disease, hf_symptoms, refractory_hf) {

  risk_factors <- as.numeric(risk_factors)
  structural_disease <- as.numeric(structural_disease)
  hf_symptoms <- as.numeric(hf_symptoms)
  refractory_hf <- as.numeric(refractory_hf)

  if (any(is.na(c(risk_factors, structural_disease, hf_symptoms, refractory_hf)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  stage <- "None"
  description <- "No identified risk factors or structural disease."
  management <- "Routine prevention."

  if (refractory_hf == 1) {
    stage <- "Stage D"
    description <- "Refractory HF requiring specialized interventions."
    management <- "Advanced measures (transplant, MCS, palliative care)."
  } else if (hf_symptoms == 1) {
    stage <- "Stage C"
    description <- "Structural heart disease with prior or current symptoms of HF."
    management <- "Diuretics, ACEI/ARB/ARNI, Beta-blockers, Aldosterone antagonists, etc."
  } else if (structural_disease == 1) {
    stage <- "Stage B"
    description <- "Structural heart disease without signs or symptoms of HF."
    management <- "ACEI/ARB, Beta-blockers. Treat etiology."
  } else if (risk_factors == 1) {
    stage <- "Stage A"
    description <- "At risk for HF but without structural heart disease or symptoms."
    management <- "Treat risk factors (HTN, lipids, DM). Lifestyle modification."
  }

  return(list(
    Stage = stage,
    Description = description,
    Management_Guidance = management,
    Inputs = list(
      Risk_Factors = risk_factors == 1,
      Structural_Disease = structural_disease == 1,
      Symptoms = hf_symptoms == 1,
      Refractory = refractory_hf == 1
    )
  ))
}

acef_ii_cardiac_surgery <- function(age, ejection_fraction, creatinine_mg_dl,
                                    hematocrit, emergency_surgery) {

  age <- as.numeric(age)
  ejection_fraction <- as.numeric(ejection_fraction)
  creatinine_mg_dl <- as.numeric(creatinine_mg_dl)
  hematocrit <- as.numeric(hematocrit)
  emergency_surgery <- as.numeric(emergency_surgery)

  if (any(is.na(c(age, ejection_fraction, creatinine_mg_dl, hematocrit, emergency_surgery)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (ejection_fraction <= 0) {
    stop("Error: Ejection Fraction must be greater than 0.")
  }
  score <- age / ejection_fraction

  if (creatinine_mg_dl > 2.0) {
    score <- score + 2.0
  }

  if (emergency_surgery == 1) {
    score <- score + 3.0
  }

  if (hematocrit < 36) {
    anemia_points <- 0.2 * (36 - hematocrit)
    score <- score + anemia_points
  }

  interpretation <- "Predicts 30-day operative mortality. Higher scores indicate higher risk."

  return(list(
    ACEF_II_Score = round(score, 3),
    Interpretation = interpretation,
    Inputs = list(
      Age = age,
      EF = ejection_fraction,
      Creatinine = creatinine_mg_dl,
      Hematocrit = hematocrit,
      Emergency = emergency_surgery
    )
  ))
}

acep_covid19_management <- function(spo2, resp_rate, signs_of_shock_mods = FALSE,
                                    resp_failure = FALSE, lower_resp_disease = FALSE,
                                    infiltrates_gt_50 = FALSE, pao2_fio2 = NULL,
                                    symptoms_present = TRUE) {

  spo2 <- as.numeric(spo2)
  resp_rate <- as.numeric(resp_rate)

  if (any(is.na(c(spo2, resp_rate)))) {
    stop("Error: SpO2 and Respiratory Rate must be numeric.")
  }

  severity <- "Unclassified"
  management <- "Clinical correlation required."
  definition <- ""

  is_critical <- FALSE
  is_severe <- FALSE
  is_moderate <- FALSE
  is_mild <- FALSE

  if (signs_of_shock_mods || resp_failure) {
    is_critical <- TRUE
  }

  if (!is_critical) {
    if (spo2 < 94 || resp_rate > 30 || infiltrates_gt_50) {
      is_severe <- TRUE
    }
    if (!is.null(pao2_fio2) && pao2_fio2 < 300) {
      is_severe <- TRUE
    }
  }

  if (!is_critical && !is_severe) {
    if (lower_resp_disease || (symptoms_present && spo2 >= 94)) {

      if (lower_resp_disease) {
        is_moderate <- TRUE
      } else {
        if (lower_resp_disease) {
          is_moderate <- TRUE
        } else {
          is_mild <- TRUE
        }
      }
    }
  }

  if (is_critical) {
    severity <- "Critical Illness"
    definition <- "Respiratory failure, septic shock, and/or multiple organ dysfunction."
    management <- "Admit to ICU. Hemodynamic/Respiratory support (HFNC, NIPPV, Intubation, Pressors). Consider Steroids/Immunomodulators per guidelines."
  } else if (is_severe) {
    severity <- "Severe Illness"
    definition <- "SpO2 < 94% on room air, RR > 30, PaO2/FiO2 < 300, or lung infiltrates > 50%."
    management <- "Admit to Inpatient/ICU. Supplemental Oxygen. Dexamethasone. Consider Remdesivir/Immunomodulators."
  } else if (is_moderate) {
    severity <- "Moderate Illness"
    definition <- "Evidence of lower respiratory disease (clinical/imaging) with SpO2 >= 94% on room air."
    management <- "Admit or Observation (depending on risk factors). If discharging: Close follow-up. No specific immunomodulators usually indicated unless rapid deterioration."
  } else if (is_mild) {
    severity <- "Mild Illness"
    definition <- "Signs/symptoms (fever, cough, sore throat, malaise) WITHOUT shortness of breath, dyspnea, or abnormal imaging."
    management <- "Discharge home likely appropriate. Symptomatic care. Isolation instructions. Return precautions."
  } else {
    severity <- "Asymptomatic/Uncertain"
    definition <- "No symptoms reported or criteria not met."
    management <- "Screening/Isolation as per public health guidelines."
  }

  return(list(
    Severity_Classification = severity,
    Clinical_Definition = definition,
    Suggested_Management = management,
    Inputs = list(
      SpO2 = spo2,
      RR = resp_rate,
      Critical_Signs = signs_of_shock_mods || resp_failure,
      Lower_Resp_Disease = lower_resp_disease,
      Infiltrates_gt_50 = infiltrates_gt_50
    )
  ))
}

acetaminophen_nomogram <- function(time_ingestion_hrs, acetaminophen_level, units = "mcg/mL") {

  time_ingestion_hrs <- as.numeric(time_ingestion_hrs)
  acetaminophen_level <- as.numeric(acetaminophen_level)

  if (any(is.na(c(time_ingestion_hrs, acetaminophen_level)))) {
    stop("Error: Time and Acetaminophen Level must be numeric.")
  }

  level_mcg_ml <- 0

  if (units == "mcg/mL" || units == "mg/L") {
    level_mcg_ml <- acetaminophen_level
  } else if (units == "mg/dL") {
    level_mcg_ml <- acetaminophen_level * 10
  } else if (units == "micromol/L") {
    level_mcg_ml <- acetaminophen_level * 0.15116
  } else {
    stop("Error: Invalid units. Use 'mcg/mL', 'mg/L', 'mg/dL', or 'micromol/L'.")
  }

  recommendation <- ""
  threshold_level <- NA

  if (time_ingestion_hrs < 4) {
    recommendation <- "Too early to interpret. Repeat level at 4 hours post-ingestion."
  } else if (time_ingestion_hrs > 24) {
    recommendation <- "Nomogram not valid > 24 hours. Consult Poison Control. Treatment generally indicated if acetaminophen is detected."
  } else {
    threshold_level <- 150 * (0.5 ^ ((time_ingestion_hrs - 4) / 4))

    if (level_mcg_ml >= threshold_level) {
      recommendation <- "Treatment Recommended (Above Treatment Line)"
    } else {
      recommendation <- "Treatment unlikely needed (Below Treatment Line)"
    }
  }

  return(list(
    Standardized_Level_mcg_mL = round(level_mcg_ml, 2),
    Treatment_Threshold_mcg_mL = if (!is.na(threshold_level)) round(threshold_level, 2) else "N/A",
    Time_Post_Ingestion_hrs = time_ingestion_hrs,
    Recommendation = recommendation,
    Input_Units = units
  ))
}

acr_eular_ra_2010 <- function(joint_involvement, serology, acute_phase_reactants, duration_symptoms) {

  joint_involvement <- as.numeric(joint_involvement)
  serology <- as.numeric(serology)
  acute_phase_reactants <- as.numeric(acute_phase_reactants)
  duration_symptoms <- as.numeric(duration_symptoms)

  if (any(is.na(c(joint_involvement, serology, acute_phase_reactants, duration_symptoms)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (joint_involvement == 0) {
    score <- score + 0
  } else if (joint_involvement == 1) {
    score <- score + 1
  } else if (joint_involvement == 2) {
    score <- score + 2
  } else if (joint_involvement == 3) {
    score <- score + 3
  } else if (joint_involvement == 4) {
    score <- score + 5
  } else {
    stop("Error: Invalid joint_involvement input. See documentation for codes.")
  }

  if (serology == 0) {
    score <- score + 0
  } else if (serology == 1) {
    score <- score + 2
  } else if (serology == 2) {
    score <- score + 3
  } else {
    stop("Error: Invalid serology input. See documentation for codes.")
  }

  if (acute_phase_reactants == 0) {
    score <- score + 0
  } else if (acute_phase_reactants == 1) {
    score <- score + 1
  } else {
    stop("Error: Invalid acute_phase_reactants input. See documentation for codes.")
  }

  if (duration_symptoms == 0) {
    score <- score + 0
  } else if (duration_symptoms == 1) {
    score <- score + 1
  } else {
    stop("Error: Invalid duration_symptoms input. See documentation for codes.")
  }

  classification <- ""
  if (score >= 6) {
    classification <- "Definite Rheumatoid Arthritis"
  } else {
    classification <- "Not Classified as Rheumatoid Arthritis (Reassess over time)"
  }

  return(list(
    Total_Score = score,
    Classification = classification,
    Inputs = list(
      Joints = joint_involvement,
      Serology = serology,
      Acute_Phase = acute_phase_reactants,
      Duration = duration_symptoms
    )
  ))
}

acr_eular_gout_2015 <- function(entry_criteria, sufficient_criteria, joint_pattern, symptom_characteristics, episode_time_course, clinical_tophus, serum_urate, synovial_fluid_negative, imaging_urate, imaging_erosion) {

  entry_criteria <- as.numeric(entry_criteria)
  sufficient_criteria <- as.numeric(sufficient_criteria)
  joint_pattern <- as.numeric(joint_pattern)
  symptom_characteristics <- as.numeric(symptom_characteristics)
  episode_time_course <- as.numeric(episode_time_course)
  clinical_tophus <- as.numeric(clinical_tophus)
  serum_urate <- as.numeric(serum_urate)
  synovial_fluid_negative <- as.numeric(synovial_fluid_negative)
  imaging_urate <- as.numeric(imaging_urate)
  imaging_erosion <- as.numeric(imaging_erosion)

  if (any(is.na(c(entry_criteria, sufficient_criteria, joint_pattern, symptom_characteristics, episode_time_course, clinical_tophus, serum_urate, synovial_fluid_negative, imaging_urate, imaging_erosion)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (entry_criteria == 0) {
    return(list(
      Classification = "Not Classified",
      Total_Score = NA,
      Interpretation = "Entry criterion not met (At least one episode of swelling, pain, or tenderness in a peripheral joint/bursa)."
    ))
  }

  if (sufficient_criteria == 1) {
    return(list(
      Classification = "Classified as Gout",
      Total_Score = NA,
      Interpretation = "Sufficient criterion met (Presence of MSU crystals in symptomatic joint/bursa or tophus)."
    ))
  }

  score <- 0

  if (joint_pattern == 1) {
    score <- score + 1
  } else if (joint_pattern == 2) {
    score <- score + 2
  }

  if (symptom_characteristics == 1) {
    score <- score + 1
  } else if (symptom_characteristics == 2) {
    score <- score + 2
  } else if (symptom_characteristics == 3) {
    score <- score + 3
  }

  if (episode_time_course == 1) {
    score <- score + 1
  } else if (episode_time_course == 2) {
    score <- score + 2
  }

  if (clinical_tophus == 1) {
    score <- score + 4
  }

  if (serum_urate < 4) {
    score <- score - 4
  } else if (serum_urate >= 6 && serum_urate < 8) {
    score <- score + 2
  } else if (serum_urate >= 8 && serum_urate < 10) {
    score <- score + 3
  } else if (serum_urate >= 10) {
    score <- score + 4
  }

  if (synovial_fluid_negative == 1) {
    score <- score - 2
  }

  if (imaging_urate == 1) {
    score <- score + 4
  }

  if (imaging_erosion == 1) {
    score <- score + 4
  }

  classification <- "Not Classified as Gout"
  if (score >= 8) {
    classification <- "Classified as Gout"
  }

  return(list(
    Classification = classification,
    Total_Score = score,
    Interpretation = if(score >= 8) "Score >= 8 supports diagnosis of Gout." else "Score < 8 does not support diagnosis of Gout."
  ))
}

action_icu_nstemi <- function(age, creatinine, heart_rate, sbp, troponin_x_uln,
                              heart_failure, st_depression, prior_revasc, chronic_lung_disease) {

  age <- as.numeric(age)
  creatinine <- as.numeric(creatinine)
  heart_rate <- as.numeric(heart_rate)
  sbp <- as.numeric(sbp)
  troponin_x_uln <- as.numeric(troponin_x_uln)
  heart_failure <- as.numeric(heart_failure)
  st_depression <- as.numeric(st_depression)
  prior_revasc <- as.numeric(prior_revasc)
  chronic_lung_disease <- as.numeric(chronic_lung_disease)

  if (any(is.na(c(age, creatinine, heart_rate, sbp, troponin_x_uln,
                  heart_failure, st_depression, prior_revasc, chronic_lung_disease)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 70) {
    score <- score + 1
  }

  if (creatinine >= 1.1) {
    score <- score + 1
  }

  if (heart_rate >= 100) {
    score <- score + 3
  } else if (heart_rate >= 85) {
    score <- score + 1
  }

  if (sbp < 125) {
    score <- score + 3
  } else if (sbp <= 145) {
    score <- score + 1
  }

  if (troponin_x_uln >= 12) {
    score <- score + 2
  }

  if (heart_failure == 1) {
    score <- score + 5
  }

  if (st_depression == 1) {
    score <- score + 1
  }

  if (prior_revasc == 0) {
    score <- score + 1
  }

  if (chronic_lung_disease == 1) {
    score <- score + 2
  }

  risk_complications <- ""
  recommendation <- ""

  if (score <= 4) {
    risk_complications <- "Low Risk (<5%)"
    recommendation <- "May be considered for admission to a non-ICU setting (e.g., telemetry floor)."
  } else if (score >= 5 && score < 14) {
    risk_complications <- "Intermediate Risk (approx. 9.3% at score 5)"
    recommendation <- "Consider ICU admission. Score >= 5 is the suggested cutoff for ICU benefit."
  } else {
    risk_complications <- "High Risk (~40%)"
    recommendation <- "Strongly consider early admission to the ICU."
  }

  return(list(
    Total_Score = score,
    Risk_Prediction = risk_complications,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Creatinine = creatinine,
      Heart_Rate = heart_rate,
      SBP = sbp,
      Troponin_Multiplier = troponin_x_uln,
      Heart_Failure = heart_failure == 1,
      ST_Depression = st_depression == 1,
      No_Prior_Revasc = prior_revasc == 0,
      Chronic_Lung_Disease = chronic_lung_disease == 1
    )
  ))
}

adhere_algorithm <- function(bun, sbp, creatinine) {

  bun <- as.numeric(bun)
  sbp <- as.numeric(sbp)
  creatinine <- as.numeric(creatinine)

  if (any(is.na(c(bun, sbp, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  mortality <- ""
  risk_group <- ""

  if (bun < 43) {
    risk_group <- "Low Risk"
    mortality <- "2.1%"
  } else {
    if (sbp >= 115) {
      risk_group <- "Intermediate Risk"
      mortality <- "6.0%"
    } else {
      if (creatinine < 2.75) {
        risk_group <- "High Risk"
        mortality <- "15.3%"
      } else {
        risk_group <- "Very High Risk"
        mortality <- "21.9%"
      }
    }
  }

  return(list(
    Risk_Group = risk_group,
    In_Hospital_Mortality_Risk = mortality,
    Inputs = list(
      BUN = bun,
      SBP = sbp,
      Creatinine = creatinine
    )
  ))
}

acute_gout_diagnosis <- function(sex, prev_attack, onset_1day, joint_redness, mtp1_involvement, htn_cvd, uric_acid_elevated) {

  sex <- tolower(as.character(sex))
  prev_attack <- as.numeric(prev_attack)
  onset_1day <- as.numeric(onset_1day)
  joint_redness <- as.numeric(joint_redness)
  mtp1_involvement <- as.numeric(mtp1_involvement)
  htn_cvd <- as.numeric(htn_cvd)
  uric_acid_elevated <- as.numeric(uric_acid_elevated)

  if (any(is.na(c(prev_attack, onset_1day, joint_redness, mtp1_involvement, htn_cvd, uric_acid_elevated)))) {
    stop("Error: All clinical inputs must be numeric (0 or 1).")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (sex == "male") {
    score <- score + 2
  }

  if (prev_attack == 1) {
    score <- score + 2
  }

  if (onset_1day == 1) {
    score <- score + 0.5
  }

  if (joint_redness == 1) {
    score <- score + 1
  }

  if (mtp1_involvement == 1) {
    score <- score + 2.5
  }

  if (htn_cvd == 1) {
    score <- score + 1.5
  }

  if (uric_acid_elevated == 1) {
    score <- score + 3.5
  }

  probability <- ""
  recommendation <- ""

  if (score <= 4) {
    probability <- "Low Probability (Gout diagnosis unlikely, <2.2%)"
    recommendation <- "Consider other diagnoses."
  } else if (score < 8) {
    probability <- "Intermediate Probability (Gout diagnosis possible, ~31.2%)"
    recommendation <- "Synovial fluid analysis recommended to confirm diagnosis."
  } else {
    probability <- "High Probability (Gout diagnosis likely, >80%)"
    recommendation <- "Treat for gout. Synovial fluid analysis may not be necessary."
  }

  return(list(
    Total_Score = score,
    Probability = probability,
    Recommendation = recommendation,
    Inputs = list(
      Sex = sex,
      Prev_Attack = prev_attack == 1,
      Onset_1_Day = onset_1day == 1,
      Redness = joint_redness == 1,
      MTP1 = mtp1_involvement == 1,
      HTN_CVD = htn_cvd == 1,
      High_Uric_Acid = uric_acid_elevated == 1
    )
  ))
}

ain_risk_calculator <- function(creatinine, bun, urine_specific_gravity, urine_protein_high, local_ain_prevalence = 0.23) {

  creatinine <- as.numeric(creatinine)
  bun <- as.numeric(bun)
  urine_specific_gravity <- as.numeric(urine_specific_gravity)
  urine_protein_high <- as.numeric(urine_protein_high)
  local_ain_prevalence <- as.numeric(local_ain_prevalence)

  if (any(is.na(c(creatinine, bun, urine_specific_gravity, urine_protein_high, local_ain_prevalence)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (creatinine <= 0 || bun <= 0) {
    stop("Error: Creatinine and BUN must be positive.")
  }

  if (local_ain_prevalence <= 0 || local_ain_prevalence >= 1) {
    stop("Error: Prevalence must be between 0 and 1.")
  }

  log_creatinine <- log(creatinine)

  ratio <- bun / creatinine
  log_ratio <- log(ratio)

  sg_transformed <- (urine_specific_gravity - 1) * 1000

  protein_score <- 0
  if (urine_protein_high == 0) {
    protein_score <- 1
  }

  logit <- (0.8367518 * log_creatinine) +
    (-0.9181938 * log_ratio) +
    (-0.0501203 * sg_transformed) +
    (0.9360127 * protein_score) +
    0.770398

  prevalence_adjustment <- log(local_ain_prevalence / 0.23)
  final_logit <- logit + prevalence_adjustment

  probability <- 1 / (1 + exp(-final_logit))
  probability_percent <- probability * 100

  interpretation <- ""
  if (probability_percent < 20) {
    interpretation <- "Low Probability: AIN is unlikely."
  } else if (probability_percent >= 20 && probability_percent < 50) {
    interpretation <- "Intermediate Probability: AIN is possible."
  } else {
    interpretation <- "High Probability: AIN is likely."
  }

  return(list(
    Probability_AIN_Percent = round(probability_percent, 1),
    Interpretation = interpretation,
    Inputs = list(
      Creatinine = creatinine,
      BUN = bun,
      Specific_Gravity = urine_specific_gravity,
      Protein_High = urine_protein_high == 1,
      Local_Prevalence = local_ain_prevalence
    )
  ))
}

acute_pancreatitis_prediction_model <- function(prior_episodes, cholelithiasis, recent_surgery, epigastric_pain, worsening_severity, pain_duration_days, pain_severity_score, lipase_uln_multiplier) {

  prior_episodes <- as.numeric(prior_episodes)
  cholelithiasis <- as.numeric(cholelithiasis)
  recent_surgery <- as.numeric(recent_surgery)
  epigastric_pain <- as.numeric(epigastric_pain)
  worsening_severity <- as.numeric(worsening_severity)
  pain_duration_days <- as.numeric(pain_duration_days)
  pain_severity_score <- as.numeric(pain_severity_score)
  lipase_uln_multiplier <- as.numeric(lipase_uln_multiplier)

  if (any(is.na(c(prior_episodes, cholelithiasis, recent_surgery, epigastric_pain, worsening_severity, pain_duration_days, pain_severity_score, lipase_uln_multiplier)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (prior_episodes == 2) {
    score <- score + 2
  } else if (prior_episodes == 3) {
    score <- score + 3
  } else if (prior_episodes >= 4) {
    score <- score + 4
  }

  if (cholelithiasis == 1) {
    score <- score + 2
  }

  if (recent_surgery == 1) {
    score <- score - 2
  }

  if (epigastric_pain == 1) {
    score <- score + 2
  }

  if (worsening_severity == 1) {
    score <- score + 1
  }

  if (pain_duration_days >= 1 && pain_duration_days <= 5) {
    score <- score + 1
  }

  if (pain_severity_score >= 4 && pain_severity_score <= 6) {
    score <- score + 2
  } else if (pain_severity_score >= 7 && pain_severity_score <= 10) {
    score <- score + 3
  }

  if (lipase_uln_multiplier >= 10 && lipase_uln_multiplier < 20) {
    score <- score + 1
  } else if (lipase_uln_multiplier >= 20) {
    score <- score + 2
  }

  risk_group <- ""

  if (score < 6) {
    risk_group <- "Low Risk (<5%) for Acute Pancreatitis"
  } else {
    risk_group <- "High Probability of Acute Pancreatitis"
  }

  return(list(
    Total_Score = score,
    Risk_Assessment = risk_group,
    Inputs = list(
      Prior_Episodes = prior_episodes,
      Cholelithiasis = cholelithiasis,
      Recent_Surgery = recent_surgery,
      Epigastric_Pain = epigastric_pain,
      Worsening = worsening_severity,
      Duration_Days = pain_duration_days,
      Pain_Severity = pain_severity_score,
      Lipase_Multiplier = lipase_uln_multiplier
    )
  ))
}

thakar_arf_cardiac_surgery <- function(gender, chf, lvef_lt_35, preop_iabp, copd,
                                       insulin_dependent_dm, prev_cardiac_surgery,
                                       emergency_surgery, surgery_type, creatinine_mg_dl) {

  gender <- tolower(as.character(gender))
  chf <- as.numeric(chf)
  lvef_lt_35 <- as.numeric(lvef_lt_35)
  preop_iabp <- as.numeric(preop_iabp)
  copd <- as.numeric(copd)
  insulin_dependent_dm <- as.numeric(insulin_dependent_dm)
  prev_cardiac_surgery <- as.numeric(prev_cardiac_surgery)
  emergency_surgery <- as.numeric(emergency_surgery)
  surgery_type <- tolower(as.character(surgery_type))
  creatinine_mg_dl <- as.numeric(creatinine_mg_dl)

  if (any(is.na(c(chf, lvef_lt_35, preop_iabp, copd, insulin_dependent_dm,
                  prev_cardiac_surgery, emergency_surgery, creatinine_mg_dl)))) {
    stop("Error: All binary/numeric inputs must be valid.")
  }

  if (!gender %in% c("male", "female")) {
    stop("Error: Gender must be 'male' or 'female'.")
  }

  valid_surgeries <- c("cabg", "valve", "cabg+valve", "other")
  if (!surgery_type %in% valid_surgeries) {
    stop("Error: Surgery type must be 'cabg', 'valve', 'cabg+valve', or 'other'.")
  }

  score <- 0

  if (gender == "female") score <- score + 1
  if (chf == 1) score <- score + 1
  if (lvef_lt_35 == 1) score <- score + 1
  if (preop_iabp == 1) score <- score + 2
  if (copd == 1) score <- score + 1
  if (insulin_dependent_dm == 1) score <- score + 1
  if (prev_cardiac_surgery == 1) score <- score + 1
  if (emergency_surgery == 1) score <- score + 2

  if (surgery_type == "valve") {
    score <- score + 1
  } else if (surgery_type == "cabg+valve") {
    score <- score + 2
  } else if (surgery_type == "other") {
    score <- score + 2
  }

  if (creatinine_mg_dl >= 1.2 && creatinine_mg_dl < 2.1) {
    score <- score + 2
  } else if (creatinine_mg_dl >= 2.1) {
    score <- score + 5
  }

  risk_dialysis <- ""
  risk_group <- ""

  if (score <= 2) {
    risk_group <- "Low Risk"
    risk_dialysis <- "0.4%"
  } else if (score >= 3 && score <= 5) {
    risk_group <- "Moderate Risk"
    risk_dialysis <- "2.2%"
  } else if (score >= 6 && score <= 8) {
    risk_group <- "High Risk"
    risk_dialysis <- "7.0%"
  } else {
    risk_group <- "Very High Risk"
    risk_dialysis <- "21.5%"
  }

  return(list(
    Thakar_Score = score,
    Risk_Group = risk_group,
    Est_Risk_Dialysis_Requirement = risk_dialysis,
    Inputs = list(
      Gender = gender,
      CHF = chf == 1,
      LVEF_lt_35 = lvef_lt_35 == 1,
      IABP = preop_iabp == 1,
      COPD = copd == 1,
      Insulin_DM = insulin_dependent_dm == 1,
      Prev_Surgery = prev_cardiac_surgery == 1,
      Emergency = emergency_surgery == 1,
      Surgery_Type = surgery_type,
      Creatinine = creatinine_mg_dl
    )
  ))
}

adapt_cardiac_risk <- function(timi_score, ekg_new_ischemia, troponin_0h_positive, troponin_2h_positive) {

  timi_score <- as.numeric(timi_score)
  ekg_new_ischemia <- as.numeric(ekg_new_ischemia)
  troponin_0h_positive <- as.numeric(troponin_0h_positive)
  troponin_2h_positive <- as.numeric(troponin_2h_positive)

  if (any(is.na(c(timi_score, ekg_new_ischemia, troponin_0h_positive, troponin_2h_positive)))) {
    stop("Error: All inputs must be numeric.")
  }

  is_low_risk <- FALSE

  if (timi_score == 0 && ekg_new_ischemia == 0 && troponin_0h_positive == 0 && troponin_2h_positive == 0) {
    is_low_risk <- TRUE
  }

  risk_group <- ""
  management <- ""
  mace_risk <- ""

  if (is_low_risk) {
    risk_group <- "Low Risk"
    mace_risk <- "<1%"
    management <- "Safe for discharge with outpatient follow-up (Sensitivity 99.7%, NPV 99.7%)."
  } else {
    risk_group <- "Not Low Risk"
    mace_risk <- "Higher Risk"
    management <- "Proceed with standard observation and further testing (e.g., 6h troponin, stress test, or admission)."
  }

  return(list(
    Risk_Category = risk_group,
    MACE_Risk_30_Day = mace_risk,
    Recommendation = management,
    Inputs = list(
      TIMI_Score = timi_score,
      EKG_New_Ischemia = ekg_new_ischemia == 1,
      Trop_0h_Pos = troponin_0h_positive == 1,
      Trop_2h_Pos = troponin_2h_positive == 1
    )
  ))
}

mskcc_additional_nodal_metastasis <- function(frozen_section_pos, method_detection, tumor_size_cm, lvi, multifocality, num_pos_sln, num_neg_sln) {

  frozen_section_pos <- as.numeric(frozen_section_pos)
  tumor_size_cm <- as.numeric(tumor_size_cm)
  lvi <- as.numeric(lvi)
  multifocality <- as.numeric(multifocality)
  num_pos_sln <- as.numeric(num_pos_sln)
  num_neg_sln <- as.numeric(num_neg_sln)
  method_detection <- tolower(as.character(method_detection))

  if (any(is.na(c(frozen_section_pos, tumor_size_cm, lvi, multifocality, num_pos_sln, num_neg_sln)))) {
    stop("Error: All numeric and binary inputs must be valid.")
  }

  logit <- -3.183

  if (frozen_section_pos == 1) {
    logit <- logit + 1.0
  } else {
    if (method_detection == "routine" || method_detection == "he") {
      logit <- logit + 0.5
    }
  }

  logit <- logit + (0.36 * tumor_size_cm)

  if (lvi == 1) {
    logit <- logit + 0.68
  }

  if (multifocality == 1) {
    logit <- logit + 0.45
  }

  logit <- logit + (0.8 * num_pos_sln)

  logit <- logit - (0.32 * num_neg_sln)

  probability <- 1 / (1 + exp(-logit))
  probability_percent <- probability * 100

  return(list(
    Probability_Non_SLN_Metastasis = round(probability_percent, 1),
    Inputs = list(
      Frozen_Section = frozen_section_pos == 1,
      Method = method_detection,
      Size_cm = tumor_size_cm,
      LVI = lvi == 1,
      Multifocality = multifocality == 1,
      Pos_SLN = num_pos_sln,
      Neg_SLN = num_neg_sln
    )
  ))
}

adult_adhd_self_report_scale <- function(q1_trouble_wrapping_up, q2_difficulty_getting_organized, q3_problems_remembering_appointments, q4_procrastinate_difficult_tasks, q5_fidgety, q6_overly_active) {

  q1 <- as.numeric(q1_trouble_wrapping_up)
  q2 <- as.numeric(q2_difficulty_getting_organized)
  q3 <- as.numeric(q3_problems_remembering_appointments)
  q4 <- as.numeric(q4_procrastinate_difficult_tasks)
  q5 <- as.numeric(q5_fidgety)
  q6 <- as.numeric(q6_overly_active)

  if (any(is.na(c(q1, q2, q3, q4, q5, q6)))) {
    stop("Error: All inputs must be numeric integers (0=Never, 1=Rarely, 2=Sometimes, 3=Often, 4=Very Often).")
  }

  if (any(c(q1, q2, q3, q4, q5, q6) < 0) || any(c(q1, q2, q3, q4, q5, q6) > 4)) {
    stop("Error: Scores must be between 0 and 4.")
  }

  score_part_a <- 0

  if (q1 >= 2) score_part_a <- score_part_a + 1
  if (q2 >= 2) score_part_a <- score_part_a + 1
  if (q3 >= 2) score_part_a <- score_part_a + 1

  if (q4 >= 3) score_part_a <- score_part_a + 1
  if (q5 >= 3) score_part_a <- score_part_a + 1
  if (q6 >= 3) score_part_a <- score_part_a + 1

  result <- ""
  if (score_part_a >= 4) {
    result <- "Screening Positive (Highly consistent with ADHD)"
  } else {
    result <- "Screening Negative"
  }

  return(list(
    Part_A_Score = score_part_a,
    Screening_Result = result,
    Inputs = list(
      Q1 = q1, Q2 = q2, Q3 = q3,
      Q4 = q4, Q5 = q5, Q6 = q6
    )
  ))
}

ace_score <- function(emotional_abuse, physical_abuse, sexual_abuse, emotional_neglect, physical_neglect, divorce_separation, domestic_violence, substance_abuse, mental_illness, incarcerated_relative) {

  emotional_abuse <- as.numeric(emotional_abuse)
  physical_abuse <- as.numeric(physical_abuse)
  sexual_abuse <- as.numeric(sexual_abuse)
  emotional_neglect <- as.numeric(emotional_neglect)
  physical_neglect <- as.numeric(physical_neglect)
  divorce_separation <- as.numeric(divorce_separation)
  domestic_violence <- as.numeric(domestic_violence)
  substance_abuse <- as.numeric(substance_abuse)
  mental_illness <- as.numeric(mental_illness)
  incarcerated_relative <- as.numeric(incarcerated_relative)

  if (any(is.na(c(emotional_abuse, physical_abuse, sexual_abuse, emotional_neglect, physical_neglect, divorce_separation, domestic_violence, substance_abuse, mental_illness, incarcerated_relative)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  total_ace <- emotional_abuse + physical_abuse + sexual_abuse + emotional_neglect + physical_neglect + divorce_separation + domestic_violence + substance_abuse + mental_illness + incarcerated_relative

  interpretation <- ""
  if (total_ace == 0) {
    interpretation <- "Low Risk: No reported adverse childhood experiences."
  } else if (total_ace >= 4) {
    interpretation <- "High Risk: Significantly increased risk for alcoholism, drug abuse, depression, and suicide attempts."
  } else {
    interpretation <- "Intermediate Risk: Presence of ACEs is associated with graded increase in health risks."
  }

  return(list(
    Total_Score = total_ace,
    Interpretation = interpretation,
    Inputs = list(
      Emotional_Abuse = emotional_abuse == 1,
      Physical_Abuse = physical_abuse == 1,
      Sexual_Abuse = sexual_abuse == 1,
      Emotional_Neglect = emotional_neglect == 1,
      Physical_Neglect = physical_neglect == 1,
      Parents_Divorced = divorce_separation == 1,
      Domestic_Violence = domestic_violence == 1,
      Substance_Abuse = substance_abuse == 1,
      Mental_Illness = mental_illness == 1,
      Incarcerated_Member = incarcerated_relative == 1
    )
  ))
}

age_adjusted_d_dimer <- function(age, d_dimer, d_dimer_units = "ng/mL") {

  age <- as.numeric(age)
  d_dimer <- as.numeric(d_dimer)

  if (any(is.na(c(age, d_dimer)))) {
    stop("Error: Age and D-dimer level must be numeric.")
  }

  d_dimer_ng_ml <- d_dimer
  if (d_dimer_units == "ug/mL" || d_dimer_units == "mg/L") {
    d_dimer_ng_ml <- d_dimer * 1000
  }

  cutoff <- 500

  if (age > 50) {
    cutoff <- age * 10
  }

  result <- ""
  interpretation <- ""

  if (d_dimer_ng_ml < cutoff) {
    result <- "Negative (VTE Ruled Out)"
    interpretation <- "D-dimer is below the age-adjusted cutoff. PE/DVT ruled out in low/moderate risk patients."
  } else {
    result <- "Positive (VTE Not Ruled Out)"
    interpretation <- "D-dimer is above the age-adjusted cutoff. Further imaging (CTPA or Ultrasound) recommended."
  }

  return(list(
    Age_Adjusted_Cutoff_ng_mL = cutoff,
    Patient_D_dimer_ng_mL = d_dimer_ng_ml,
    Result = result,
    Interpretation = interpretation,
    Inputs = list(
      Age = age,
      D_dimer_Input = d_dimer,
      Units = d_dimer_units
    )
  ))
}

age_adjusted_esr_crp <- function(age, sex, esr = NULL, crp = NULL) {

  age <- as.numeric(age)

  if (!is.null(esr)) esr <- as.numeric(esr)
  if (!is.null(crp)) crp <- as.numeric(crp)

  if (is.na(age)) {
    stop("Error: Age must be a valid number.")
  }

  sex <- tolower(as.character(sex))
  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  esr_upper_limit <- 0
  if (sex == "male") {
    esr_upper_limit <- age / 2
  } else {
    esr_upper_limit <- (age + 10) / 2
  }

  crp_upper_limit <- 0
  if (sex == "male") {
    crp_upper_limit <- age / 50
  } else {
    crp_upper_limit <- (age / 50) + 0.6
  }

  esr_status <- "Not Provided"
  if (!is.null(esr)) {
    if (esr > esr_upper_limit) {
      esr_status <- "Elevated"
    } else {
      esr_status <- "Normal"
    }
  }

  crp_status <- "Not Provided"
  if (!is.null(crp)) {
    if (crp > crp_upper_limit) {
      crp_status <- "Elevated"
    } else {
      crp_status <- "Normal"
    }
  }

  return(list(
    Calculated_ESR_Upper_Limit = round(esr_upper_limit, 1),
    Calculated_CRP_Upper_Limit = round(crp_upper_limit, 2),
    Patient_Values = list(
      ESR = if (!is.null(esr)) esr else NA,
      CRP = if (!is.null(crp)) crp else NA
    ),
    Interpretation = list(
      ESR_Status = esr_status,
      CRP_Status = crp_status
    )
  ))
}

aims65_upper_gi_bleed <- function(albumin_g_dl, inr, altered_mental_status, sbp_mmhg, age_years) {

  albumin_g_dl <- as.numeric(albumin_g_dl)
  inr <- as.numeric(inr)
  altered_mental_status <- as.numeric(altered_mental_status)
  sbp_mmhg <- as.numeric(sbp_mmhg)
  age_years <- as.numeric(age_years)

  if (any(is.na(c(albumin_g_dl, inr, altered_mental_status, sbp_mmhg, age_years)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (albumin_g_dl < 3.0) {
    score <- score + 1
  }

  if (inr > 1.5) {
    score <- score + 1
  }

  if (altered_mental_status == 1) {
    score <- score + 1
  }

  if (sbp_mmhg <= 90) {
    score <- score + 1
  }

  if (age_years >= 65) {
    score <- score + 1
  }

  mortality <- ""
  if (score == 0) {
    mortality <- "0.3%"
  } else if (score == 1) {
    mortality <- "1.2%"
  } else if (score == 2) {
    mortality <- "5.3%"
  } else if (score == 3) {
    mortality <- "10.3%"
  } else if (score == 4) {
    mortality <- "16.5%"
  } else {
    mortality <- "24.5%"
  }

  risk_stratification <- "Low Risk"
  if (score >= 2) {
    risk_stratification <- "High Risk"
  }

  return(list(
    Total_Score = score,
    In_Hospital_Mortality_Risk = mortality,
    Risk_Stratification = risk_stratification,
    Inputs = list(
      Albumin_lt_3 = albumin_g_dl < 3.0,
      INR_gt_1_5 = inr > 1.5,
      Altered_Mental_Status = altered_mental_status == 1,
      SBP_le_90 = sbp_mmhg <= 90,
      Age_ge_65 = age_years >= 65
    )
  ))
}

akin_aki_staging <- function(baseline_creatinine, max_creatinine, urine_output_stage = 0, rrt_initiated = 0) {

  baseline_creatinine <- as.numeric(baseline_creatinine)
  max_creatinine <- as.numeric(max_creatinine)
  urine_output_stage <- as.numeric(urine_output_stage)
  rrt_initiated <- as.numeric(rrt_initiated)

  if (any(is.na(c(baseline_creatinine, max_creatinine, urine_output_stage, rrt_initiated)))) {
    stop("Error: All inputs must be numeric.")
  }

  cr_stage <- 0

  creatinine_ratio <- max_creatinine / baseline_creatinine
  creatinine_increase <- max_creatinine - baseline_creatinine

  if (rrt_initiated == 1) {
    cr_stage <- 3
  } else {
    if (creatinine_ratio > 3.0) {
      cr_stage <- 3
    } else if (max_creatinine >= 4.0 && creatinine_increase >= 0.5) {
      cr_stage <- 3
    } else if (creatinine_ratio > 2.0) {
      cr_stage <- 2
    } else if (creatinine_ratio >= 1.5 || creatinine_increase >= 0.3) {
      cr_stage <- 1
    } else {
      cr_stage <- 0
    }
  }

  final_stage <- max(cr_stage, urine_output_stage)

  classification <- ""
  if (final_stage == 0) {
    classification <- "No AKI"
  } else if (final_stage == 1) {
    classification <- "Stage 1"
  } else if (final_stage == 2) {
    classification <- "Stage 2"
  } else {
    classification <- "Stage 3"
  }

  return(list(
    AKIN_Stage = classification,
    Numeric_Stage = final_stage,
    Criteria_Met = list(
      Creatinine_Stage = cr_stage,
      Urine_Output_Stage = urine_output_stage,
      RRT = rrt_initiated == 1
    )
  ))
}

alberta_stroke_aspects <- function(caudate, lentiform, internal_capsule, insular_cortex, m1, m2, m3, m4, m5, m6) {

  caudate <- as.numeric(caudate)
  lentiform <- as.numeric(lentiform)
  internal_capsule <- as.numeric(internal_capsule)
  insular_cortex <- as.numeric(insular_cortex)
  m1 <- as.numeric(m1)
  m2 <- as.numeric(m2)
  m3 <- as.numeric(m3)
  m4 <- as.numeric(m4)
  m5 <- as.numeric(m5)
  m6 <- as.numeric(m6)

  if (any(is.na(c(caudate, lentiform, internal_capsule, insular_cortex, m1, m2, m3, m4, m5, m6)))) {
    stop("Error: All inputs must be numeric (0 for No, 1 for Yes).")
  }

  points_deducted <- caudate + lentiform + internal_capsule + insular_cortex + m1 + m2 + m3 + m4 + m5 + m6

  final_value <- 10 - points_deducted

  interpretation <- ""
  if (final_value >= 8) {
    interpretation <- "Score 8-10: Independent lifestyle likely (Lower risk of symptomatic hemorrhage)."
  } else {
    interpretation <- "Score < 8: Associated with poor functional outcome and increased risk of symptomatic hemorrhage with thrombolysis."
  }

  return(list(
    ASPECTS_Value = final_value,
    Interpretation = interpretation,
    Regions_Affected = list(
      Subcortical = list(
        Caudate = caudate == 1,
        Lentiform = lentiform == 1,
        Internal_Capsule = internal_capsule == 1
      ),
      Cortical = list(
        Insula = insular_cortex == 1,
        M1_Anterior_MCA = m1 == 1,
        M2_Lateral_MCA = m2 == 1,
        M3_Posterior_MCA = m3 == 1,
        M4_Anterior_Superior_MCA = m4 == 1,
        M5_Lateral_Superior_MCA = m5 == 1,
        M6_Posterior_Superior_MCA = m6 == 1
      )
    )
  ))
}

aldrete_score <- function(activity, respiration, circulation, consciousness, oxygenation) {

  activity <- as.numeric(activity)
  respiration <- as.numeric(respiration)
  circulation <- as.numeric(circulation)
  consciousness <- as.numeric(consciousness)
  oxygenation <- as.numeric(oxygenation)

  if (any(is.na(c(activity, respiration, circulation, consciousness, oxygenation)))) {
    stop("Error: All inputs must be numeric integers (0, 1, or 2).")
  }

  if (any(c(activity, respiration, circulation, consciousness, oxygenation) < 0) ||
      any(c(activity, respiration, circulation, consciousness, oxygenation) > 2)) {
    stop("Error: Individual scores must be between 0 and 2.")
  }

  total_score <- activity + respiration + circulation + consciousness + oxygenation

  interpretation <- ""
  if (total_score >= 9) {
    interpretation <- "Ready for discharge from Phase I recovery (PACU)."
  } else {
    interpretation <- "Not yet ready for discharge from Phase I recovery."
  }

  return(list(
    Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Activity = activity,
      Respiration = respiration,
      Circulation = circulation,
      Consciousness = consciousness,
      Oxygenation = oxygenation
    )
  ))
}

alt_70_cellulitis <- function(asymmetry, leukocytosis, tachycardia, age_ge_70) {

  asymmetry <- as.numeric(asymmetry)
  leukocytosis <- as.numeric(leukocytosis)
  tachycardia <- as.numeric(tachycardia)
  age_ge_70 <- as.numeric(age_ge_70)

  if (any(is.na(c(asymmetry, leukocytosis, tachycardia, age_ge_70)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (asymmetry == 1) {
    score <- score + 3
  }

  if (leukocytosis == 1) {
    score <- score + 1
  }

  if (tachycardia == 1) {
    score <- score + 1
  }

  if (age_ge_70 == 1) {
    score <- score + 2
  }

  probability <- ""
  recommendation <- ""

  if (score <= 2) {
    probability <- "Low Likelihood (<20%)"
    recommendation <- "Consider alternative diagnoses (e.g., stasis dermatitis, acute lipodermatosclerosis). Review for bilateral involvement."
  } else if (score <= 4) {
    probability <- "Indeterminate"
    recommendation <- "Consult Dermatology or Infectious Disease recommended."
  } else {
    probability <- "High Likelihood (>82%)"
    recommendation <- "Treat for cellulitis."
  }

  return(list(
    Total_Score = score,
    Likelihood = probability,
    Recommendation = recommendation,
    Inputs = list(
      Asymmetry_Unilateral = asymmetry == 1,
      Leukocytosis_WBC_ge_10 = leukocytosis == 1,
      Tachycardia_HR_ge_90 = tachycardia == 1,
      Age_ge_70 = age_ge_70 == 1
    )
  ))
}

altitude_adjusted_perc <- function(age, heart_rate, o2_sat, hemoptysis, estrogen_use, surgery_trauma_4wks, prior_vte, unilateral_leg_swelling) {

  age <- as.numeric(age)
  heart_rate <- as.numeric(heart_rate)
  o2_sat <- as.numeric(o2_sat)
  hemoptysis <- as.numeric(hemoptysis)
  estrogen_use <- as.numeric(estrogen_use)
  surgery_trauma_4wks <- as.numeric(surgery_trauma_4wks)
  prior_vte <- as.numeric(prior_vte)
  unilateral_leg_swelling <- as.numeric(unilateral_leg_swelling)

  if (any(is.na(c(age, heart_rate, o2_sat, hemoptysis, estrogen_use, surgery_trauma_4wks, prior_vte, unilateral_leg_swelling)))) {
    stop("Error: All inputs must be numeric.")
  }

  criteria_met <- TRUE
  reasons <- c()

  if (age >= 50) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Age >= 50")
  }

  if (heart_rate >= 100) {
    criteria_met <- FALSE
    reasons <- c(reasons, "HR >= 100")
  }

  if (o2_sat < 90) {
    criteria_met <- FALSE
    reasons <- c(reasons, "SaO2 < 90% (Altitude Adjusted)")
  }

  if (hemoptysis == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Hemoptysis present")
  }

  if (estrogen_use == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Estrogen use")
  }

  if (surgery_trauma_4wks == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Surgery/Trauma in last 4 weeks")
  }

  if (prior_vte == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Prior VTE")
  }

  if (unilateral_leg_swelling == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Unilateral leg swelling")
  }

  result <- ""
  recommendation <- ""

  if (criteria_met) {
    result <- "PERC Negative (Rule Out)"
    recommendation <- "Low risk for PE (<2%). No further testing (D-dimer/Imaging) needed."
  } else {
    result <- "PERC Positive (Cannot Rule Out)"
    recommendation <- "Rule out criteria not met. Consider D-dimer testing or imaging based on pre-test probability (Wells Score)."
  }

  return(list(
    Result = result,
    Recommendation = recommendation,
    Criteria_Violated = if (length(reasons) > 0) reasons else "None",
    Inputs = list(
      Age = age,
      HR = heart_rate,
      O2_Sat = o2_sat,
      Hemoptysis = hemoptysis == 1,
      Estrogen = estrogen_use == 1,
      Surgery_Trauma = surgery_trauma_4wks == 1,
      Prior_VTE = prior_vte == 1,
      Leg_Swelling = unilateral_leg_swelling == 1
    )
  ))
}

alvarado_appendicitis <- function(migratory_pain, anorexia, nausea_vomiting, rlq_tenderness, rebound_tenderness, elevated_temp, leukocytosis, shift_left) {

  migratory_pain <- as.numeric(migratory_pain)
  anorexia <- as.numeric(anorexia)
  nausea_vomiting <- as.numeric(nausea_vomiting)
  rlq_tenderness <- as.numeric(rlq_tenderness)
  rebound_tenderness <- as.numeric(rebound_tenderness)
  elevated_temp <- as.numeric(elevated_temp)
  leukocytosis <- as.numeric(leukocytosis)
  shift_left <- as.numeric(shift_left)

  if (any(is.na(c(migratory_pain, anorexia, nausea_vomiting, rlq_tenderness, rebound_tenderness, elevated_temp, leukocytosis, shift_left)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (migratory_pain == 1) score <- score + 1
  if (anorexia == 1) score <- score + 1
  if (nausea_vomiting == 1) score <- score + 1
  if (rlq_tenderness == 1) score <- score + 2
  if (rebound_tenderness == 1) score <- score + 1
  if (elevated_temp == 1) score <- score + 1
  if (leukocytosis == 1) score <- score + 2
  if (shift_left == 1) score <- score + 1

  risk_stratification <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_stratification <- "Low Risk (0-3)"
    recommendation <- "Appendicitis unlikely. Consider other diagnoses or observation."
  } else if (score <= 6) {
    risk_stratification <- "Moderate Risk (4-6)"
    recommendation <- "Appendicitis possible. Consider CT scan or further observation."
  } else {
    risk_stratification <- "High Risk (7-10)"
    recommendation <- "Appendicitis probable. Surgical consultation indicated."
  }

  return(list(
    Total_Score = score,
    Risk_Group = risk_stratification,
    Recommendation = recommendation,
    Inputs = list(
      Migratory_Pain = migratory_pain == 1,
      Anorexia = anorexia == 1,
      Nausea_Vomiting = nausea_vomiting == 1,
      RLQ_Tenderness = rlq_tenderness == 1,
      Rebound = rebound_tenderness == 1,
      Elevated_Temp = elevated_temp == 1,
      Leukocytosis = leukocytosis == 1,
      Shift_Left = shift_left == 1
    )
  ))
}

ada_diabetes_risk <- function(age, sex, family_history, high_bp, physically_active, height_in, weight_lbs, history_gestational_dm = 0) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  family_history <- as.numeric(family_history)
  high_bp <- as.numeric(high_bp)
  physically_active <- as.numeric(physically_active)
  height_in <- as.numeric(height_in)
  weight_lbs <- as.numeric(weight_lbs)
  history_gestational_dm <- as.numeric(history_gestational_dm)

  if (any(is.na(c(age, family_history, high_bp, physically_active, height_in, weight_lbs)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age >= 40 && age <= 49) {
    score <- score + 1
  } else if (age >= 50 && age <= 59) {
    score <- score + 2
  } else if (age >= 60) {
    score <- score + 3
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (sex == "female" && history_gestational_dm == 1) {
    score <- score + 1
  }

  if (family_history == 1) {
    score <- score + 1
  }

  if (high_bp == 1) {
    score <- score + 1
  }

  if (physically_active == 0) {
    score <- score + 1
  }

  bmi <- (weight_lbs / (height_in^2)) * 703

  if (bmi >= 25 && bmi < 30) {
    score <- score + 1
  } else if (bmi >= 30 && bmi < 40) {
    score <- score + 2
  } else if (bmi >= 40) {
    score <- score + 3
  }

  risk_category <- ""
  recommendation <- ""

  if (score >= 5) {
    risk_category <- "High Risk"
    recommendation <- "You are at increased risk for type 2 diabetes. Discuss testing with your doctor."
  } else {
    risk_category <- "Low Risk"
    recommendation <- "You are currently at low risk for type 2 diabetes. Keep up the good work."
  }

  return(list(
    Total_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Calculated_BMI = round(bmi, 1),
    Inputs = list(
      Age = age,
      Sex = sex,
      Family_Hx = family_history == 1,
      High_BP = high_bp == 1,
      Active = physically_active == 1,
      Gestational_DM = if (sex == "female") (history_gestational_dm == 1) else "N/A"
    )
  ))
}

aub_has2_cardiovascular_risk <- function(hypertension, age, smoking, hyperlipidemia) {

  hypertension <- as.numeric(hypertension)
  age <- as.numeric(age)
  smoking <- as.numeric(smoking)
  hyperlipidemia <- as.numeric(hyperlipidemia)

  if (any(is.na(c(hypertension, age, smoking, hyperlipidemia)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (hypertension == 1) {
    score <- score + 1
  }

  if (age >= 60) {
    score <- score + 1
  }

  if (smoking == 1) {
    score <- score + 1
  }

  if (hyperlipidemia == 1) {
    score <- score + 2
  }

  risk_category <- ""

  if (score <= 1) {
    risk_category <- "Low Risk"
  } else if (score == 2) {
    risk_category <- "Intermediate Risk"
  } else {
    risk_category <- "High Risk"
  }

  return(list(
    Total_Score = score,
    Risk_Category = risk_category,
    Inputs = list(
      Hypertension = hypertension == 1,
      Age_ge_60 = age >= 60,
      Smoking = smoking == 1,
      Hyperlipidemia = hyperlipidemia == 1
    )
  ))
}

aua_symptom_index <- function(incomplete_emptying, frequency, intermittency, urgency, weak_stream, straining, nocturia, quality_of_life = NULL) {

  incomplete_emptying <- as.numeric(incomplete_emptying)
  frequency <- as.numeric(frequency)
  intermittency <- as.numeric(intermittency)
  urgency <- as.numeric(urgency)
  weak_stream <- as.numeric(weak_stream)
  straining <- as.numeric(straining)
  nocturia <- as.numeric(nocturia)

  if (any(is.na(c(incomplete_emptying, frequency, intermittency, urgency, weak_stream, straining, nocturia)))) {
    stop("Error: All 7 symptom inputs must be numeric integers (0-5).")
  }

  symptom_score <- incomplete_emptying + frequency + intermittency + urgency + weak_stream + straining + nocturia

  severity <- ""
  if (symptom_score <= 7) {
    severity <- "Mild"
  } else if (symptom_score <= 19) {
    severity <- "Moderate"
  } else {
    severity <- "Severe"
  }

  qol_score <- NA
  qol_text <- ""
  if (!is.null(quality_of_life)) {
    qol_score <- as.numeric(quality_of_life)
    if (qol_score == 0) qol_text <- "Delighted"
    else if (qol_score == 1) qol_text <- "Pleased"
    else if (qol_score == 2) qol_text <- "Mostly Satisfied"
    else if (qol_score == 3) qol_text <- "Mixed"
    else if (qol_score == 4) qol_text <- "Mostly Dissatisfied"
    else if (qol_score == 5) qol_text <- "Unhappy"
    else if (qol_score == 6) qol_text <- "Terrible"
  }

  return(list(
    AUA_Symptom_Score = symptom_score,
    Severity_Category = severity,
    Quality_of_Life_Score = qol_score,
    Quality_of_Life_Description = qol_text,
    Inputs = list(
      Incomplete_Emptying = incomplete_emptying,
      Frequency = frequency,
      Intermittency = intermittency,
      Urgency = urgency,
      Weak_Stream = weak_stream,
      Straining = straining,
      Nocturia = nocturia
    )
  ))
}

asdas_crp <- function(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling, crp_mg_l) {

  back_pain <- as.numeric(back_pain)
  duration_morning_stiffness <- as.numeric(duration_morning_stiffness)
  patient_global <- as.numeric(patient_global)
  peripheral_pain_swelling <- as.numeric(peripheral_pain_swelling)
  crp_mg_l <- as.numeric(crp_mg_l)

  if (any(is.na(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling, crp_mg_l)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling) < 0) ||
      any(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling) > 10)) {
    stop("Error: Clinical scores must be between 0 and 10.")
  }

  crp_val <- crp_mg_l
  if (crp_val < 2) {
    crp_val <- 2
  }

  asdas_score <- (0.121 * back_pain) +
    (0.110 * patient_global) +
    (0.073 * peripheral_pain_swelling) +
    (0.058 * duration_morning_stiffness) +
    (0.579 * log(crp_val + 1))

  disease_activity <- ""
  if (asdas_score < 1.3) {
    disease_activity <- "Inactive Disease"
  } else if (asdas_score >= 1.3 && asdas_score < 2.1) {
    disease_activity <- "Moderate Disease Activity"
  } else if (asdas_score >= 2.1 && asdas_score <= 3.5) {
    disease_activity <- "High Disease Activity"
  } else {
    disease_activity <- "Very High Disease Activity"
  }

  return(list(
    ASDAS_CRP_Score = round(asdas_score, 2),
    Disease_Activity_State = disease_activity,
    Inputs = list(
      Back_Pain = back_pain,
      Duration_Stiffness = duration_morning_stiffness,
      Patient_Global = patient_global,
      Peripheral_Pain = peripheral_pain_swelling,
      CRP_mg_L = crp_mg_l
    )
  ))
}

asdas_esr <- function(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling, esr_mm_hr) {

  back_pain <- as.numeric(back_pain)
  duration_morning_stiffness <- as.numeric(duration_morning_stiffness)
  patient_global <- as.numeric(patient_global)
  peripheral_pain_swelling <- as.numeric(peripheral_pain_swelling)
  esr_mm_hr <- as.numeric(esr_mm_hr)

  if (any(is.na(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling, esr_mm_hr)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling) < 0) ||
      any(c(back_pain, duration_morning_stiffness, patient_global, peripheral_pain_swelling) > 10)) {
    stop("Error: Clinical scores must be between 0 and 10.")
  }

  asdas_score <- (0.079 * back_pain) +
    (0.113 * patient_global) +
    (0.063 * peripheral_pain_swelling) +
    (0.069 * duration_morning_stiffness) +
    (0.293 * sqrt(esr_mm_hr))

  disease_activity <- ""
  if (asdas_score < 1.3) {
    disease_activity <- "Inactive Disease"
  } else if (asdas_score >= 1.3 && asdas_score < 2.1) {
    disease_activity <- "Moderate Disease Activity"
  } else if (asdas_score >= 2.1 && asdas_score <= 3.5) {
    disease_activity <- "High Disease Activity"
  } else {
    disease_activity <- "Very High Disease Activity"
  }

  return(list(
    ASDAS_ESR_Score = round(asdas_score, 2),
    Disease_Activity_State = disease_activity,
    Inputs = list(
      Back_Pain = back_pain,
      Duration_Stiffness = duration_morning_stiffness,
      Patient_Global = patient_global,
      Peripheral_Pain = peripheral_pain_swelling,
      ESR_mm_hr = esr_mm_hr
    )
  ))
}

antivenom_dosing_algorithm <- function(signs_of_envenomation, initial_control_achieved = FALSE) {

  signs_of_envenomation <- as.numeric(signs_of_envenomation)
  initial_control_achieved <- as.numeric(initial_control_achieved)

  if (any(is.na(c(signs_of_envenomation, initial_control_achieved)))) {
    stop("Error: All inputs must be numeric (0 for No, 1 for Yes).")
  }

  recommendation <- ""
  maintenance <- ""

  if (signs_of_envenomation == 0) {
    recommendation <- "No antivenom indicated at this time. Monitor for progression for at least 8-12 hours."
    maintenance <- "N/A"
  } else {
    if (initial_control_achieved == 0) {
      recommendation <- "Initial Dose: Administer 4-6 vials of CroFab IV. Re-evaluate in 1 hour. If control not achieved, repeat 4-6 vials."
      maintenance <- "Pending initial control."
    } else {
      recommendation <- "Initial control achieved. Proceed to maintenance therapy."
      maintenance <- "Administer 2 vials of CroFab every 6 hours for 3 doses (at 6, 12, and 18 hours after initial control)."
    }
  }

  return(list(
    Recommendation = recommendation,
    Maintenance_Therapy = maintenance,
    Inputs = list(
      Signs_Envenomation = signs_of_envenomation == 1,
      Initial_Control_Achieved = initial_control_achieved == 1
    )
  ))
}

aortic_dissection_detection_risk_score <- function(high_risk_conditions, high_risk_pain, high_risk_exam) {

  high_risk_conditions <- as.numeric(high_risk_conditions)
  high_risk_pain <- as.numeric(high_risk_pain)
  high_risk_exam <- as.numeric(high_risk_exam)

  if (any(is.na(c(high_risk_conditions, high_risk_pain, high_risk_exam)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (high_risk_conditions == 1) score <- score + 1
  if (high_risk_pain == 1) score <- score + 1
  if (high_risk_exam == 1) score <- score + 1

  risk_category <- ""
  recommendation <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    recommendation <- "Consider D-dimer to rule out aortic dissection. If D-dimer is negative (<500 ng/mL), AD is unlikely."
  } else if (score == 1) {
    risk_category <- "Moderate Risk"
    recommendation <- "Consider CTA or TEE. D-dimer may be used to rule out in selected cases, but imaging is often preferred."
  } else {
    risk_category <- "High Risk"
    recommendation <- "Immediate surgical consultation and urgent imaging (CTA or TEE) required. D-dimer should not be used to rule out."
  }

  return(list(
    ADD_RS_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Conditions_Present = high_risk_conditions == 1,
      Pain_Present = high_risk_pain == 1,
      Exam_Present = high_risk_exam == 1
    )
  ))
}

apache_ii_score <- function(age, temperature_c, mean_arterial_pressure, heart_rate, respiratory_rate, fio2, pao2, paco2 = NULL, arterial_ph, sodium, potassium, creatinine, acute_renal_failure, hematocrit, wbc_count, gcs_total, chronic_organ_insufficiency, admission_type) {

  age <- as.numeric(age)
  temperature_c <- as.numeric(temperature_c)
  mean_arterial_pressure <- as.numeric(mean_arterial_pressure)
  heart_rate <- as.numeric(heart_rate)
  respiratory_rate <- as.numeric(respiratory_rate)
  fio2 <- as.numeric(fio2)
  pao2 <- as.numeric(pao2)
  arterial_ph <- as.numeric(arterial_ph)
  sodium <- as.numeric(sodium)
  potassium <- as.numeric(potassium)
  creatinine <- as.numeric(creatinine)
  hematocrit <- as.numeric(hematocrit)
  wbc_count <- as.numeric(wbc_count)
  gcs_total <- as.numeric(gcs_total)

  if (!is.null(paco2)) paco2 <- as.numeric(paco2)

  if (any(is.na(c(age, temperature_c, mean_arterial_pressure, heart_rate, respiratory_rate, fio2, pao2, arterial_ph, sodium, potassium, creatinine, hematocrit, wbc_count, gcs_total)))) {
    stop("Error: All physiological inputs must be numeric.")
  }

  admission_type <- tolower(as.character(admission_type))
  if (!admission_type %in% c("non_operative", "emergency_postop", "elective_postop")) {
    stop("Error: Admission type must be 'non_operative', 'emergency_postop', or 'elective_postop'.")
  }

  score <- 0

  if (age <= 44) score <- score + 0
  else if (age <= 54) score <- score + 2
  else if (age <= 64) score <- score + 3
  else if (age <= 74) score <- score + 5
  else score <- score + 6

  if (temperature_c >= 41 || temperature_c <= 29.9) score <- score + 4
  else if (temperature_c >= 39 || temperature_c <= 31.9) score <- score + 3
  else if (temperature_c <= 33.9) score <- score + 2
  else if (temperature_c >= 38.5 || temperature_c <= 35.9) score <- score + 1
  else score <- score + 0

  if (mean_arterial_pressure >= 160 || mean_arterial_pressure <= 49) score <- score + 4
  else if (mean_arterial_pressure >= 130) score <- score + 3
  else if (mean_arterial_pressure >= 110 || mean_arterial_pressure <= 69) score <- score + 2
  else score <- score + 0

  if (heart_rate >= 180 || heart_rate <= 39) score <- score + 4
  else if (heart_rate >= 140 || heart_rate <= 54) score <- score + 3
  else if (heart_rate >= 110 || heart_rate <= 69) score <- score + 2
  else score <- score + 0

  if (respiratory_rate >= 50 || respiratory_rate <= 5) score <- score + 4
  else if (respiratory_rate >= 35) score <- score + 3
  else if (respiratory_rate <= 9) score <- score + 2
  else if (respiratory_rate >= 25 || respiratory_rate <= 11) score <- score + 1
  else score <- score + 0

  if (fio2 >= 0.5) {
    if (is.null(paco2)) stop("Error: PaCO2 is required to calculate A-a gradient when FiO2 >= 0.5.")
    patm <- 760
    ph2o <- 47
    p_alv_o2 <- (fio2 * (patm - ph2o)) - (paco2 / 0.8)
    aa_gradient <- p_alv_o2 - pao2

    if (aa_gradient >= 500) score <- score + 4
    else if (aa_gradient >= 350) score <- score + 3
    else if (aa_gradient >= 200) score <- score + 2
    else score <- score + 0
  } else {
    if (pao2 < 55) score <- score + 4
    else if (pao2 <= 60) score <- score + 3
    else if (pao2 <= 70) score <- score + 1
    else score <- score + 0
  }

  if (arterial_ph >= 7.7 || arterial_ph < 7.15) score <- score + 4
  else if (arterial_ph >= 7.6 || arterial_ph <= 7.24) score <- score + 3
  else if (arterial_ph <= 7.32) score <- score + 2
  else if (arterial_ph >= 7.5) score <- score + 1
  else score <- score + 0

  if (sodium >= 180 || sodium <= 110) score <- score + 4
  else if (sodium >= 160 || sodium <= 119) score <- score + 3
  else if (sodium >= 155 || sodium <= 129) score <- score + 2
  else if (sodium >= 150) score <- score + 1
  else score <- score + 0

  if (potassium >= 7 || potassium < 2.5) score <- score + 4
  else if (potassium >= 6) score <- score + 3
  else if (potassium <= 2.9) score <- score + 2
  else if (potassium >= 5.5 || potassium <= 3.4) score <- score + 1
  else score <- score + 0

  creat_points <- 0
  if (creatinine >= 3.5) creat_points <- 4
  else if (creatinine >= 2.0) creat_points <- 3
  else if (creatinine >= 1.5 || creatinine < 0.6) creat_points <- 2
  else creat_points <- 0

  if (acute_renal_failure == 1) {
    creat_points <- creat_points * 2
  }
  score <- score + creat_points

  if (hematocrit >= 60 || hematocrit < 20) score <- score + 4
  else if (hematocrit >= 50 || hematocrit <= 29.9) score <- score + 2
  else if (hematocrit >= 46) score <- score + 1
  else score <- score + 0

  if (wbc_count >= 40 || wbc_count < 1) score <- score + 4
  else if (wbc_count >= 20 || wbc_count <= 2.9) score <- score + 2
  else if (wbc_count >= 15) score <- score + 1
  else score <- score + 0

  score <- score + (15 - gcs_total)

  chronic_points <- 0
  if (chronic_organ_insufficiency == 1) {
    if (admission_type == "elective_postop") {
      chronic_points <- 2
    } else {
      chronic_points <- 5
    }
  }
  score <- score + chronic_points

  mortality_est <- ""
  if (score <= 4) mortality_est <- "4%"
  else if (score <= 9) mortality_est <- "8%"
  else if (score <= 14) mortality_est <- "15%"
  else if (score <= 19) mortality_est <- "25%"
  else if (score <= 24) mortality_est <- "40%"
  else if (score <= 29) mortality_est <- "55%"
  else if (score <= 34) mortality_est <- "75%"
  else mortality_est <- "85%"

  return(list(
    APACHE_II_Score = score,
    Est_Hospital_Mortality = mortality_est,
    Inputs = list(
      Age = age,
      Temp = temperature_c,
      MAP = mean_arterial_pressure,
      HR = heart_rate,
      RR = respiratory_rate,
      Oxygenation = if(fio2 >= 0.5) "A-a Gradient" else "PaO2",
      pH = arterial_ph,
      Na = sodium,
      K = potassium,
      Cr = creatinine,
      ARF = acute_renal_failure == 1,
      Hct = hematocrit,
      WBC = wbc_count,
      GCS = gcs_total,
      Chronic_Health_Pts = chronic_points
    )
  ))
}

apfel_ponv <- function(female_sex, history_ponv_motion_sickness, non_smoker, postop_opioids) {

  female_sex <- as.numeric(female_sex)
  history_ponv_motion_sickness <- as.numeric(history_ponv_motion_sickness)
  non_smoker <- as.numeric(non_smoker)
  postop_opioids <- as.numeric(postop_opioids)

  if (any(is.na(c(female_sex, history_ponv_motion_sickness, non_smoker, postop_opioids)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  total_points <- female_sex + history_ponv_motion_sickness + non_smoker + postop_opioids

  risk_percent <- ""

  if (total_points == 0) {
    risk_percent <- "10%"
  } else if (total_points == 1) {
    risk_percent <- "21%"
  } else if (total_points == 2) {
    risk_percent <- "39%"
  } else if (total_points == 3) {
    risk_percent <- "61%"
  } else {
    risk_percent <- "79%"
  }

  return(list(
    Apfel_Score = total_points,
    Est_Risk_PONV = risk_percent,
    Inputs = list(
      Female = female_sex == 1,
      Hx_PONV_MotionSickness = history_ponv_motion_sickness == 1,
      Non_Smoker = non_smoker == 1,
      Postop_Opioids = postop_opioids == 1
    )
  ))
}

apgar_score <- function(appearance, pulse, grimace, activity, respiration) {

  appearance <- as.numeric(appearance)
  pulse <- as.numeric(pulse)
  grimace <- as.numeric(grimace)
  activity <- as.numeric(activity)
  respiration <- as.numeric(respiration)

  if (any(is.na(c(appearance, pulse, grimace, activity, respiration)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(appearance, pulse, grimace, activity, respiration) < 0) ||
      any(c(appearance, pulse, grimace, activity, respiration) > 2)) {
    stop("Error: Individual scores must be 0, 1, or 2.")
  }

  total_score <- appearance + pulse + grimace + activity + respiration

  interpretation <- ""
  if (total_score >= 7) {
    interpretation <- "Reassuring (Normal)"
  } else if (total_score >= 4) {
    interpretation <- "Moderately Abnormal"
  } else {
    interpretation <- "Low (Concern for neurological status/respiratory failure)"
  }

  return(list(
    Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Appearance = appearance,
      Pulse = pulse,
      Grimace = grimace,
      Activity = activity,
      Respiration = respiration
    )
  ))
}

air_appendicitis <- function(vomiting, rlq_pain, rebound_tenderness, temp_c, wbc_count, pmn_percent, crp_mg_l) {

  vomiting <- as.numeric(vomiting)
  rlq_pain <- as.numeric(rlq_pain)
  rebound_tenderness <- as.numeric(rebound_tenderness)
  temp_c <- as.numeric(temp_c)
  wbc_count <- as.numeric(wbc_count)
  pmn_percent <- as.numeric(pmn_percent)
  crp_mg_l <- as.numeric(crp_mg_l)

  if (any(is.na(c(vomiting, rlq_pain, rebound_tenderness, temp_c, wbc_count, pmn_percent, crp_mg_l)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (vomiting == 1) score <- score + 1
  if (rlq_pain == 1) score <- score + 1

  if (rebound_tenderness == 1) {
    score <- score + 1
  } else if (rebound_tenderness == 2) {
    score <- score + 2
  } else if (rebound_tenderness == 3) {
    score <- score + 3
  }

  if (temp_c >= 38.5) score <- score + 1

  if (wbc_count >= 15) {
    score <- score + 2
  } else if (wbc_count >= 10) {
    score <- score + 1
  }

  if (pmn_percent >= 85) {
    score <- score + 2
  } else if (pmn_percent >= 70) {
    score <- score + 1
  }

  if (crp_mg_l >= 50) {
    score <- score + 2
  } else if (crp_mg_l >= 10) {
    score <- score + 1
  }

  risk_group <- ""
  recommendation <- ""

  if (score <= 4) {
    risk_group <- "Low Probability"
    recommendation <- "Outpatient follow-up if unaltered general condition."
  } else if (score <= 8) {
    risk_group <- "Indeterminate Group"
    recommendation <- "In-hospital active observation, diagnostic imaging, or diagnostic laparoscopy."
  } else {
    risk_group <- "High Probability"
    recommendation <- "Surgical exploration."
  }

  return(list(
    AIR_Score = score,
    Risk_Group = risk_group,
    Recommendation = recommendation,
    Inputs = list(
      Vomiting = vomiting == 1,
      RLQ_Pain = rlq_pain == 1,
      Rebound_Grade = rebound_tenderness,
      Temp_ge_38_5 = temp_c >= 38.5,
      WBC = wbc_count,
      PMN_Percent = pmn_percent,
      CRP = crp_mg_l
    )
  ))
}

ariscat_score <- function(age, spo2, resp_infection_last_month, anemia_hb_le_10, incision_site, duration_surgery_hours, emergency_surgery) {

  age <- as.numeric(age)
  spo2 <- as.numeric(spo2)
  resp_infection_last_month <- as.numeric(resp_infection_last_month)
  anemia_hb_le_10 <- as.numeric(anemia_hb_le_10)
  duration_surgery_hours <- as.numeric(duration_surgery_hours)
  emergency_surgery <- as.numeric(emergency_surgery)
  incision_site <- tolower(as.character(incision_site))

  if (any(is.na(c(age, spo2, resp_infection_last_month, anemia_hb_le_10, duration_surgery_hours, emergency_surgery)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!incision_site %in% c("peripheral", "upper_abdominal", "intrathoracic")) {
    stop("Error: Incision site must be 'peripheral', 'upper_abdominal', or 'intrathoracic'.")
  }

  score <- 0

  if (age <= 50) {
    score <- score + 0
  } else if (age <= 80) {
    score <- score + 3
  } else {
    score <- score + 16
  }

  if (spo2 >= 96) {
    score <- score + 0
  } else if (spo2 >= 91) {
    score <- score + 8
  } else {
    score <- score + 24
  }

  if (resp_infection_last_month == 1) {
    score <- score + 17
  }

  if (anemia_hb_le_10 == 1) {
    score <- score + 11
  }

  if (incision_site == "upper_abdominal") {
    score <- score + 15
  } else if (incision_site == "intrathoracic") {
    score <- score + 24
  }

  if (duration_surgery_hours <= 2) {
    score <- score + 0
  } else if (duration_surgery_hours <= 3) {
    score <- score + 16
  } else {
    score <- score + 23
  }

  if (emergency_surgery == 1) {
    score <- score + 8
  }

  risk_group <- ""
  complication_rate <- ""

  if (score < 26) {
    risk_group <- "Low Risk"
    complication_rate <- "1.6%"
  } else if (score < 45) {
    risk_group <- "Intermediate Risk"
    complication_rate <- "13.3%"
  } else {
    risk_group <- "High Risk"
    complication_rate <- "42.1%"
  }

  return(list(
    ARISCAT_Score = score,
    Risk_Group = risk_group,
    Est_Pulmonary_Complication_Rate = complication_rate,
    Inputs = list(
      Age = age,
      SpO2 = spo2,
      Resp_Infection = resp_infection_last_month == 1,
      Anemia = anemia_hb_le_10 == 1,
      Incision = incision_site,
      Duration = duration_surgery_hours,
      Emergency = emergency_surgery == 1
    )
  ))
}

abg_analyzer <- function(ph, paco2, hco3, na = NULL, cl = NULL, albumin = 4.0) {

  ph <- as.numeric(ph)
  paco2 <- as.numeric(paco2)
  hco3 <- as.numeric(hco3)

  if (!is.null(na)) na <- as.numeric(na)
  if (!is.null(cl)) cl <- as.numeric(cl)
  if (!is.null(albumin)) albumin <- as.numeric(albumin)

  if (any(is.na(c(ph, paco2, hco3)))) {
    stop("Error: pH, PaCO2, and HCO3 must be numeric.")
  }

  primary_disorder <- "Normal Acid-Base Status"
  compensation_status <- ""
  anion_gap_analysis <- "Not calculated (Missing Na/Cl)"

  is_acidemia <- ph < 7.35
  is_alkalemia <- ph > 7.45
  is_normal_ph <- ph >= 7.35 && ph <= 7.45

  paco2_high <- paco2 > 45
  paco2_low <- paco2 < 35
  hco3_high <- hco3 > 26
  hco3_low <- hco3 < 22

  if (is_acidemia) {
    if (paco2_high && hco3_low) {
      primary_disorder <- "Mixed Respiratory and Metabolic Acidosis"
      compensation_status <- "None possible (Both driving pH down)"
    } else if (paco2_high) {
      primary_disorder <- "Respiratory Acidosis"
      exp_hco3_acute <- 24 + ((paco2 - 40) / 10) * 1
      exp_hco3_chronic <- 24 + ((paco2 - 40) / 10) * 4

      if (abs(hco3 - exp_hco3_acute) <= 2) {
        compensation_status <- "Uncompensated (Acute)"
      } else if (abs(hco3 - exp_hco3_chronic) <= 2) {
        compensation_status <- "Compensated (Chronic)"
      } else if (hco3 > exp_hco3_chronic) {
        compensation_status <- "Superimposed Metabolic Alkalosis"
      } else if (hco3 < exp_hco3_acute) {
        compensation_status <- "Superimposed Metabolic Acidosis"
      } else {
        compensation_status <- "Partially Compensated"
      }
    } else if (hco3_low) {
      primary_disorder <- "Metabolic Acidosis"
      exp_paco2 <- (1.5 * hco3) + 8
      if (paco2 >= (exp_paco2 - 2) && paco2 <= (exp_paco2 + 2)) {
        compensation_status <- "Adequately Compensated (Winter's Formula)"
      } else if (paco2 < (exp_paco2 - 2)) {
        compensation_status <- "Superimposed Respiratory Alkalosis"
      } else {
        compensation_status <- "Superimposed Respiratory Acidosis (Inadequate Compensation)"
      }
    } else {
      primary_disorder <- "Unclassified Acidemia"
    }
  } else if (is_alkalemia) {
    if (paco2_low && hco3_high) {
      primary_disorder <- "Mixed Respiratory and Metabolic Alkalosis"
      compensation_status <- "None possible (Both driving pH up)"
    } else if (paco2_low) {
      primary_disorder <- "Respiratory Alkalosis"
      exp_hco3_acute <- 24 - ((40 - paco2) / 10) * 2
      exp_hco3_chronic <- 24 - ((40 - paco2) / 10) * 5

      if (abs(hco3 - exp_hco3_acute) <= 2) {
        compensation_status <- "Uncompensated (Acute)"
      } else if (abs(hco3 - exp_hco3_chronic) <= 2) {
        compensation_status <- "Compensated (Chronic)"
      } else if (hco3 < exp_hco3_chronic) {
        compensation_status <- "Superimposed Metabolic Acidosis"
      } else if (hco3 > exp_hco3_acute) {
        compensation_status <- "Superimposed Metabolic Alkalosis"
      } else {
        compensation_status <- "Partially Compensated"
      }
    } else if (hco3_high) {
      primary_disorder <- "Metabolic Alkalosis"
      exp_paco2 <- 40 + (0.7 * (hco3 - 24))
      if (paco2 >= (exp_paco2 - 2) && paco2 <= (exp_paco2 + 2)) {
        compensation_status <- "Adequately Compensated"
      } else if (paco2 < (exp_paco2 - 2)) {
        compensation_status <- "Superimposed Respiratory Alkalosis"
      } else {
        compensation_status <- "Superimposed Respiratory Acidosis"
      }
    } else {
      primary_disorder <- "Unclassified Alkalemia"
    }
  } else {
    if (paco2_high && hco3_high) {
      primary_disorder <- "Compensated Respiratory Acidosis / Metabolic Alkalosis"
      compensation_status <- "Mixed Disorder (pH Normal)"
    } else if (paco2_low && hco3_low) {
      primary_disorder <- "Compensated Respiratory Alkalosis / Metabolic Acidosis"
      compensation_status <- "Mixed Disorder (pH Normal)"
    } else if (paco2 >= 35 && paco2 <= 45 && hco3 >= 22 && hco3 <= 26) {
      primary_disorder <- "Normal Acid-Base Status"
    } else {
      primary_disorder <- "Mixed Disorder (Normal pH with abnormal components)"
    }
  }

  ag_val <- NA
  corrected_ag <- NA
  delta_ratio <- NA

  if (!is.null(na) && !is.null(cl)) {
    ag_val <- na - (cl + hco3)
    corrected_ag <- ag_val + (2.5 * (4.0 - albumin))

    ag_status <- "Normal Anion Gap"
    if (corrected_ag > 12) {
      ag_status <- "High Anion Gap"
      delta_gap <- corrected_ag - 12
      delta_hco3 <- 24 - hco3
      delta_ratio <- delta_gap / delta_hco3

      gap_interp <- ""
      if (delta_ratio < 0.4) {
        gap_interp <- "Hyperchloremic Normal AG Acidosis"
      } else if (delta_ratio < 0.8) {
        gap_interp <- "Combined High AG and Normal AG Acidosis"
      } else if (delta_ratio >= 0.8 && delta_ratio <= 2.0) {
        gap_interp <- "Pure High AG Metabolic Acidosis"
      } else if (delta_ratio > 2.0) {
        gap_interp <- "High AG Acidosis + Metabolic Alkalosis"
      }
      anion_gap_analysis <- paste0(ag_status, " (", round(corrected_ag, 1), "). ", gap_interp)
    } else {
      anion_gap_analysis <- paste0(ag_status, " (", round(corrected_ag, 1), ")")
    }
  }

  return(list(
    Diagnosis = primary_disorder,
    Compensation = compensation_status,
    Anion_Gap_Analysis = anion_gap_analysis,
    Data = list(
      pH = ph,
      PaCO2 = paco2,
      HCO3 = hco3,
      Anion_Gap = if(!is.na(corrected_ag)) round(corrected_ag, 1) else "N/A"
    )
  ))
}

asa_physical_status <- function(asa_class, emergency = 0) {

  asa_class <- as.numeric(asa_class)
  emergency <- as.numeric(emergency)

  if (is.na(asa_class) || !asa_class %in% 1:6) {
    stop("Error: ASA Class must be an integer between 1 and 6.")
  }

  definition <- ""
  adult_examples <- ""

  if (asa_class == 1) {
    definition <- "A normal healthy patient."
    adult_examples <- "Healthy, non-smoking, no or minimal alcohol use."
  } else if (asa_class == 2) {
    definition <- "A patient with mild systemic disease."
    adult_examples <- "Smoker, social alcohol drinker, pregnancy, obesity (BMI 30-40), well-controlled DM/HTN, mild lung disease."
  } else if (asa_class == 3) {
    definition <- "A patient with severe systemic disease."
    adult_examples <- "Poorly controlled DM/HTN, COPD, BMI >= 40, active hepatitis, alcohol dependence/abuse, implanted pacemaker, moderate reduction of ejection fraction, ESRD undergoing regularly scheduled dialysis, history (>3 months) of MI, CVA, TIA, or CAD/stents."
  } else if (asa_class == 4) {
    definition <- "A patient with severe systemic disease that is a constant threat to life."
    adult_examples <- "Recent (<3 months) MI, CVA, TIA, or CAD/stents, ongoing cardiac ischemia or severe valve dysfunction, severe reduction of ejection fraction, sepsis, DIC, ARD, ESRD not undergoing regularly scheduled dialysis."
  } else if (asa_class == 5) {
    definition <- "A moribund patient who is not expected to survive without the operation."
    adult_examples <- "Ruptured abdominal/thoracic aneurysm, massive trauma, intracranial bleed with mass effect, ischemic bowel in the face of significant cardiac pathology or multiple organ/system dysfunction."
  } else {
    definition <- "A declared brain-dead patient whose organs are being removed for donor purposes."
    adult_examples <- "Transplant donor."
  }

  roman_map <- c("I", "II", "III", "IV", "V", "VI")
  status_string <- paste0("ASA ", roman_map[asa_class])
  if (emergency == 1) {
    status_string <- paste0(status_string, "-E")
  }

  return(list(
    Classification = status_string,
    Definition = definition,
    Examples = adult_examples,
    Emergency_Status = emergency == 1
  ))
}

asas_axial_spa <- function(back_pain_gt_3mo, age_onset_lt_45, sacroiliitis_imaging, hla_b27,
                           inflammatory_bp, arthritis, enthesitis_heel, uveitis, dactylitis,
                           psoriasis, crohns_colitis, good_nsaid_response, family_history_spa,
                           elevated_crp_esr) {

  back_pain_gt_3mo <- as.numeric(back_pain_gt_3mo)
  age_onset_lt_45 <- as.numeric(age_onset_lt_45)
  sacroiliitis_imaging <- as.numeric(sacroiliitis_imaging)
  hla_b27 <- as.numeric(hla_b27)
  inflammatory_bp <- as.numeric(inflammatory_bp)
  arthritis <- as.numeric(arthritis)
  enthesitis_heel <- as.numeric(enthesitis_heel)
  uveitis <- as.numeric(uveitis)
  dactylitis <- as.numeric(dactylitis)
  psoriasis <- as.numeric(psoriasis)
  crohns_colitis <- as.numeric(crohns_colitis)
  good_nsaid_response <- as.numeric(good_nsaid_response)
  family_history_spa <- as.numeric(family_history_spa)
  elevated_crp_esr <- as.numeric(elevated_crp_esr)

  if (any(is.na(c(back_pain_gt_3mo, age_onset_lt_45, sacroiliitis_imaging, hla_b27,
                  inflammatory_bp, arthritis, enthesitis_heel, uveitis, dactylitis,
                  psoriasis, crohns_colitis, good_nsaid_response, family_history_spa,
                  elevated_crp_esr)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  if (back_pain_gt_3mo == 0 || age_onset_lt_45 == 0) {
    return(list(
      Classification = "Not Applicable",
      Reason = "Patient does not meet entry criteria (Back pain >= 3 months AND Age at onset < 45 years)."
    ))
  }

  clinical_features_count <- inflammatory_bp + arthritis + enthesitis_heel + uveitis +
    dactylitis + psoriasis + crohns_colitis + good_nsaid_response +
    family_history_spa + elevated_crp_esr

  met_imaging_arm <- FALSE
  if (sacroiliitis_imaging == 1) {
    total_features_imaging <- clinical_features_count + hla_b27
    if (total_features_imaging >= 1) {
      met_imaging_arm <- TRUE
    }
  }

  met_clinical_arm <- FALSE
  if (hla_b27 == 1) {
    if (clinical_features_count >= 2) {
      met_clinical_arm <- TRUE
    }
  }

  classification <- "Not Classified as Axial SpA"
  criteria_met <- "None"

  if (met_imaging_arm || met_clinical_arm) {
    classification <- "Classified as Axial Spondyloarthritis"
    if (met_imaging_arm && met_clinical_arm) {
      criteria_met <- "Both Imaging and Clinical Arms"
    } else if (met_imaging_arm) {
      criteria_met <- "Imaging Arm"
    } else {
      criteria_met <- "Clinical Arm"
    }
  }

  return(list(
    Classification = classification,
    Criteria_Arm_Met = criteria_met,
    Inputs = list(
      Entry_Criteria_Met = TRUE,
      Sacroiliitis = sacroiliitis_imaging == 1,
      HLA_B27 = hla_b27 == 1,
      Clinical_Features_Count = clinical_features_count
    )
  ))
}

asas_peripheral_spa <- function(arthritis, enthesitis, dactylitis, uveitis, psoriasis, crohns_colitis, preceding_infection, hla_b27, sacroiliitis_imaging, inflammatory_bp, family_history_spa) {

  arthritis <- as.numeric(arthritis)
  enthesitis <- as.numeric(enthesitis)
  dactylitis <- as.numeric(dactylitis)
  uveitis <- as.numeric(uveitis)
  psoriasis <- as.numeric(psoriasis)
  crohns_colitis <- as.numeric(crohns_colitis)
  preceding_infection <- as.numeric(preceding_infection)
  hla_b27 <- as.numeric(hla_b27)
  sacroiliitis_imaging <- as.numeric(sacroiliitis_imaging)
  inflammatory_bp <- as.numeric(inflammatory_bp)
  family_history_spa <- as.numeric(family_history_spa)

  if (any(is.na(c(arthritis, enthesitis, dactylitis, uveitis, psoriasis, crohns_colitis, preceding_infection, hla_b27, sacroiliitis_imaging, inflammatory_bp, family_history_spa)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  if (arthritis == 0 && enthesitis == 0 && dactylitis == 0) {
    return(list(
      Classification = "Not Applicable",
      Reason = "Patient does not meet entry criteria (Must have Arthritis, Enthesitis, or Dactylitis)."
    ))
  }

  group1_count <- uveitis + psoriasis + crohns_colitis + preceding_infection + hla_b27 + sacroiliitis_imaging

  group2_count <- arthritis + enthesitis + dactylitis + inflammatory_bp + family_history_spa

  classification <- "Not Classified as Peripheral SpA"
  criteria_met <- "None"

  if (group1_count >= 1) {
    classification <- "Classified as Peripheral Spondyloarthritis"
    criteria_met <- ">=1 SpA Feature (Group 1)"
  } else if (group2_count >= 2) {
    classification <- "Classified as Peripheral Spondyloarthritis"
    criteria_met <- ">=2 Other SpA Features (Group 2)"
  }

  return(list(
    Classification = classification,
    Criteria_Met = criteria_met,
    Inputs = list(
      Entry_Met = TRUE,
      Group1_Count = group1_count,
      Group2_Count = group2_count
    )
  ))
}

ascod_stroke_phenotype <- function(a_grade, s_grade, c_grade, o_grade, d_grade) {

  a_grade <- as.numeric(a_grade)
  s_grade <- as.numeric(s_grade)
  c_grade <- as.numeric(c_grade)
  o_grade <- as.numeric(o_grade)
  d_grade <- as.numeric(d_grade)

  valid_grades <- c(0, 1, 2, 3, 9)

  if (!all(c(a_grade, s_grade, c_grade, o_grade, d_grade) %in% valid_grades)) {
    stop("Error: All grades must be 0 (Absent), 1 (Potential), 2 (Uncertain), 3 (Unlikely), or 9 (Insufficient).")
  }

  get_desc <- function(grade) {
    if (grade == 1) return("Potential Cause")
    if (grade == 2) return("Causal Link Uncertain")
    if (grade == 3) return("Unlikely Cause")
    if (grade == 9) return("Insufficient Workup")
    return("Disease Absent")
  }

  phenotype_string <- paste0("A", a_grade, "-S", s_grade, "-C", c_grade, "-O", o_grade, "-D", d_grade)

  return(list(
    ASCOD_Phenotype = phenotype_string,
    Interpretation = list(
      A_Atherosclerosis = paste0("Grade ", a_grade, ": ", get_desc(a_grade)),
      S_Small_Vessel = paste0("Grade ", s_grade, ": ", get_desc(s_grade)),
      C_Cardiac = paste0("Grade ", c_grade, ": ", get_desc(c_grade)),
      O_Other = paste0("Grade ", o_grade, ": ", get_desc(o_grade)),
      D_Dissection = paste0("Grade ", d_grade, ": ", get_desc(d_grade))
    )
  ))
}

ascvd_2013_risk_calculator <- function(age, sex, race, total_cholesterol, hdl_cholesterol,
                                       systolic_bp, bp_medication, diabetes, smoker) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  race <- tolower(as.character(race))
  total_cholesterol <- as.numeric(total_cholesterol)
  hdl_cholesterol <- as.numeric(hdl_cholesterol)
  systolic_bp <- as.numeric(systolic_bp)
  bp_medication <- as.numeric(bp_medication)
  diabetes <- as.numeric(diabetes)
  smoker <- as.numeric(smoker)

  if (any(is.na(c(age, total_cholesterol, hdl_cholesterol, systolic_bp, bp_medication, diabetes, smoker)))) {
    stop("Error: All inputs must be numeric or valid strings.")
  }

  if (age < 40 || age > 79) {
    return(list(Risk = NA, Message = "Calculator valid for ages 40-79."))
  }

  ln_age <- log(age)
  ln_tot_chol <- log(total_cholesterol)
  ln_hdl <- log(hdl_cholesterol)
  ln_sbp <- log(systolic_bp)

  is_african_american <- race %in% c("aa", "african american", "black")
  is_female <- sex %in% c("female", "f")

  coeff <- list()
  mean_val <- 0
  baseline_surv <- 0

  if (is_female) {
    if (is_african_american) {
      coeff$ln_age <- 17.114
      coeff$ln_age_sq <- 0
      coeff$ln_tot_chol <- 0.940
      coeff$ln_age_ln_tot_chol <- 0
      coeff$ln_hdl <- -18.920
      coeff$ln_age_ln_hdl <- 4.475
      coeff$ln_sbp_treated <- 29.291
      coeff$ln_age_ln_sbp_treated <- -6.432
      coeff$ln_sbp_untreated <- 27.820
      coeff$ln_age_ln_sbp_untreated <- -6.087
      coeff$smoker <- 0.691
      coeff$ln_age_smoker <- 0
      coeff$diabetes <- 0.874

      mean_val <- 86.61
      baseline_surv <- 0.9533

    } else {
      coeff$ln_age <- -29.799
      coeff$ln_age_sq <- 4.884
      coeff$ln_tot_chol <- 13.540
      coeff$ln_age_ln_tot_chol <- -3.114
      coeff$ln_hdl <- -13.578
      coeff$ln_age_ln_hdl <- 3.149
      coeff$ln_sbp_treated <- 2.019
      coeff$ln_age_ln_sbp_treated <- 0
      coeff$ln_sbp_untreated <- 1.957
      coeff$ln_age_ln_sbp_untreated <- 0
      coeff$smoker <- 7.574
      coeff$ln_age_smoker <- -1.665
      coeff$diabetes <- 0.661

      mean_val <- -29.18
      baseline_surv <- 0.9665
    }
  } else {
    if (is_african_american) {
      coeff$ln_age <- 2.469
      coeff$ln_age_sq <- 0
      coeff$ln_tot_chol <- 0.302
      coeff$ln_age_ln_tot_chol <- 0
      coeff$ln_hdl <- -0.307
      coeff$ln_age_ln_hdl <- 0
      coeff$ln_sbp_treated <- 1.916
      coeff$ln_age_ln_sbp_treated <- 0
      coeff$ln_sbp_untreated <- 1.809
      coeff$ln_age_ln_sbp_untreated <- 0
      coeff$smoker <- 0.549
      coeff$ln_age_smoker <- 0
      coeff$diabetes <- 0.645

      mean_val <- 19.54
      baseline_surv <- 0.8954

    } else {
      coeff$ln_age <- 12.344
      coeff$ln_age_sq <- 0
      coeff$ln_tot_chol <- 11.853
      coeff$ln_age_ln_tot_chol <- -2.664
      coeff$ln_hdl <- -7.990
      coeff$ln_age_ln_hdl <- 1.769
      coeff$ln_sbp_treated <- 1.797
      coeff$ln_age_ln_sbp_treated <- 0
      coeff$ln_sbp_untreated <- 1.764
      coeff$ln_age_ln_sbp_untreated <- 0
      coeff$smoker <- 7.837
      coeff$ln_age_smoker <- -1.795
      coeff$diabetes <- 0.658

      mean_val <- 61.18
      baseline_surv <- 0.9144
    }
  }

  sum_calc <- (coeff$ln_age * ln_age) +
    (coeff$ln_age_sq * (ln_age^2)) +
    (coeff$ln_tot_chol * ln_tot_chol) +
    (coeff$ln_age_ln_tot_chol * ln_age * ln_tot_chol) +
    (coeff$ln_hdl * ln_hdl) +
    (coeff$ln_age_ln_hdl * ln_age * ln_hdl) +
    (coeff$smoker * smoker) +
    (coeff$ln_age_smoker * ln_age * smoker) +
    (coeff$diabetes * diabetes)

  if (bp_medication == 1) {
    sum_calc <- sum_calc +
      (coeff$ln_sbp_treated * ln_sbp) +
      (coeff$ln_age_ln_sbp_treated * ln_age * ln_sbp)
  } else {
    sum_calc <- sum_calc +
      (coeff$ln_sbp_untreated * ln_sbp) +
      (coeff$ln_age_ln_sbp_untreated * ln_age * ln_sbp)
  }

  risk_score <- 1 - (baseline_surv ^ exp(sum_calc - mean_val))
  risk_percentage <- risk_score * 100

  risk_category <- ""
  recommendation <- ""

  if (risk_percentage < 5.0) {
    risk_category <- "Low Risk (<5%)"
    recommendation <- "Lifestyle modifications."
  } else if (risk_percentage < 7.5) {
    risk_category <- "Borderline Risk (5% - 7.4%)"
    recommendation <- "Consider moderate-intensity statin if risk enhancers present."
  } else if (risk_percentage < 20.0) {
    risk_category <- "Intermediate Risk (7.5% - 19.9%)"
    recommendation <- "Moderate-intensity statin indicated. Consider high-intensity if risk enhancers."
  } else {
    risk_category <- "High Risk (>=20%)"
    recommendation <- "High-intensity statin indicated."
  }

  return(list(
    Ten_Year_ASCVD_Risk_Percent = round(risk_percentage, 1),
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      Race = race,
      Total_Chol = total_cholesterol,
      HDL = hdl_cholesterol,
      SBP = systolic_bp,
      Treated_BP = bp_medication == 1,
      Diabetes = diabetes == 1,
      Smoker = smoker == 1
    )
  ))
}

assure_rcc_prognosis <- function(histology, fuhrman_grade, tumor_size_cm, necrosis, vascular_invasion, nodal_involvement, sarcomatoid) {

  histology <- tolower(as.character(histology))
  fuhrman_grade <- as.numeric(fuhrman_grade)
  tumor_size_cm <- as.numeric(tumor_size_cm)
  necrosis <- as.numeric(necrosis)
  vascular_invasion <- tolower(as.character(vascular_invasion))
  nodal_involvement <- as.numeric(nodal_involvement)
  sarcomatoid <- as.numeric(sarcomatoid)

  if (any(is.na(c(fuhrman_grade, tumor_size_cm, necrosis, nodal_involvement, sarcomatoid)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  valid_histologies <- c("clear cell", "papillary type 1", "papillary type 2", "chromophobe", "mixed", "other")
  if (!histology %in% valid_histologies) {
    stop("Error: Histology must be 'clear cell', 'papillary type 1', 'papillary type 2', 'chromophobe', 'mixed', or 'other'.")
  }

  valid_invasion <- c("none", "segmental", "renal vein", "caval")
  if (!vascular_invasion %in% valid_invasion) {
    stop("Error: Vascular Invasion must be 'none', 'segmental', 'renal vein', or 'caval'.")
  }

  points <- 0

  if (histology == "chromophobe") {
    points <- points + 0
  } else if (histology == "papillary type 1") {
    points <- points + 17
  } else if (histology == "clear cell") {
    points <- points + 35
  } else if (histology == "papillary type 2" || histology == "mixed") {
    points <- points + 43
  } else {
    points <- points + 100
  }

  if (fuhrman_grade == 1) points <- points + 0
  if (fuhrman_grade == 2) points <- points + 13
  if (fuhrman_grade == 3) points <- points + 44
  if (fuhrman_grade == 4) points <- points + 78

  points <- points + (tumor_size_cm * 4)

  if (necrosis == 1) {
    points <- points + 23
  }

  if (vascular_invasion == "none") points <- points + 0
  if (vascular_invasion == "segmental") points <- points + 15
  if (vascular_invasion == "renal vein") points <- points + 30
  if (vascular_invasion == "caval") points <- points + 45

  if (nodal_involvement == 1) {
    points <- points + 65
  }

  if (sarcomatoid == 1) {
    points <- points + 40
  }

  dfs_risk <- ""
  os_risk <- ""

  if (points < 140) {
    dfs_risk <- "Low Risk"
    os_risk <- "Low Risk"
  } else if (points >= 140 && points < 220) {
    dfs_risk <- "Intermediate Risk"
    os_risk <- "Intermediate Risk"
  } else {
    dfs_risk <- "High Risk"
    os_risk <- "High Risk"
  }

  return(list(
    Total_Points = round(points, 0),
    DFS_Risk_Group = dfs_risk,
    OS_Risk_Group = os_risk,
    Inputs = list(
      Histology = histology,
      Grade = fuhrman_grade,
      Size_cm = tumor_size_cm,
      Necrosis = necrosis == 1,
      Vascular_Invasion = vascular_invasion,
      Nodes_Positive = nodal_involvement == 1,
      Sarcomatoid = sarcomatoid == 1
    )
  ))
}

airq_asthma_score <- function(symptoms_gt_4days, awakening_gt_1time, activity_limited_daily, rescue_inhaler_daily, social_limited, exercise_limited, difficult_control, oral_steroids_12mo, er_visit_12mo, hospital_overnight_12mo) {

  symptoms_gt_4days <- as.numeric(symptoms_gt_4days)
  awakening_gt_1time <- as.numeric(awakening_gt_1time)
  activity_limited_daily <- as.numeric(activity_limited_daily)
  rescue_inhaler_daily <- as.numeric(rescue_inhaler_daily)
  social_limited <- as.numeric(social_limited)
  exercise_limited <- as.numeric(exercise_limited)
  difficult_control <- as.numeric(difficult_control)
  oral_steroids_12mo <- as.numeric(oral_steroids_12mo)
  er_visit_12mo <- as.numeric(er_visit_12mo)
  hospital_overnight_12mo <- as.numeric(hospital_overnight_12mo)

  if (any(is.na(c(symptoms_gt_4days, awakening_gt_1time, activity_limited_daily, rescue_inhaler_daily, social_limited, exercise_limited, difficult_control, oral_steroids_12mo, er_visit_12mo, hospital_overnight_12mo)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- symptoms_gt_4days + awakening_gt_1time + activity_limited_daily + rescue_inhaler_daily + social_limited + exercise_limited + difficult_control + oral_steroids_12mo + er_visit_12mo + hospital_overnight_12mo

  control_level <- ""
  risk_prediction <- ""

  if (score <= 1) {
    control_level <- "Well-controlled"
    risk_prediction <- "Low risk of exacerbation"
  } else if (score <= 4) {
    control_level <- "Not well-controlled"
    risk_prediction <- "Increased risk (approx. 2x) of exacerbation in next 12 months"
  } else {
    control_level <- "Very poorly controlled"
    risk_prediction <- "High risk (approx. 4x) of exacerbation in next 12 months"
  }

  return(list(
    AIRQ_Score = score,
    Control_Level = control_level,
    Exacerbation_Risk = risk_prediction,
    Inputs = list(
      Impairment_Items = list(
        Symptoms_gt_4d = symptoms_gt_4days == 1,
        Awakening_gt_1 = awakening_gt_1time == 1,
        Daily_Activity_Limit = activity_limited_daily == 1,
        Daily_Rescue_Use = rescue_inhaler_daily == 1,
        Social_Limit = social_limited == 1,
        Exercise_Limit = exercise_limited == 1,
        Diff_Control_Feeling = difficult_control == 1
      ),
      Risk_Items = list(
        Steroids_12mo = oral_steroids_12mo == 1,
        ER_Visit_12mo = er_visit_12mo == 1,
        Hospital_12mo = hospital_overnight_12mo == 1
      )
    )
  ))
}

asthma_predictive_index <- function(wheezing_freq_ge_4, parent_asthma, eczema, allergic_rhinitis, wheezing_apart_colds, eosinophilia_ge_4) {

  wheezing_freq_ge_4 <- as.numeric(wheezing_freq_ge_4)
  parent_asthma <- as.numeric(parent_asthma)
  eczema <- as.numeric(eczema)
  allergic_rhinitis <- as.numeric(allergic_rhinitis)
  wheezing_apart_colds <- as.numeric(wheezing_apart_colds)
  eosinophilia_ge_4 <- as.numeric(eosinophilia_ge_4)

  if (any(is.na(c(wheezing_freq_ge_4, parent_asthma, eczema, allergic_rhinitis, wheezing_apart_colds, eosinophilia_ge_4)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  major_criteria <- 0
  if (parent_asthma == 1) major_criteria <- major_criteria + 1
  if (eczema == 1) major_criteria <- major_criteria + 1

  minor_criteria <- 0
  if (allergic_rhinitis == 1) minor_criteria <- minor_criteria + 1
  if (wheezing_apart_colds == 1) minor_criteria <- minor_criteria + 1
  if (eosinophilia_ge_4 == 1) minor_criteria <- minor_criteria + 1

  result <- ""
  interpretation <- ""
  positive_predictive_value <- ""
  negative_predictive_value <- ""

  is_positive <- FALSE

  if (wheezing_freq_ge_4 == 1) {
    if (major_criteria >= 1 || minor_criteria >= 2) {
      is_positive <- TRUE
    }
  }

  if (is_positive) {
    result <- "Positive Stringent API"
    interpretation <- "High risk of developing active asthma at school age (6-13 years)."
    positive_predictive_value <- "77% (Specificity 97%)"
    negative_predictive_value <- "68%"
  } else {
    result <- "Negative Stringent API"
    interpretation <- "Lower risk of developing active asthma at school age."
    positive_predictive_value <- "N/A"
    negative_predictive_value <- "High NPV suggests asthma unlikely if criteria not met."
  }

  if (wheezing_freq_ge_4 == 0) {
    result <- "Criteria Not Met"
    interpretation <- "The Stringent API requires a history of frequent wheezing (>= 4 episodes/year) as an entry criterion."
  }

  return(list(
    Result = result,
    Interpretation = interpretation,
    Stats = list(
      PPV = positive_predictive_value,
      Specificity = if (is_positive) "97%" else "N/A"
    ),
    Inputs = list(
      Frequent_Wheezing = wheezing_freq_ge_4 == 1,
      Major_Count = major_criteria,
      Minor_Count = minor_criteria
    )
  ))
}

astral_stroke_score <- function(age, nihss, time_gt_3h, visual_field_defect, glucose, loc_reduced, glucose_units = "mg/dL") {

  age <- as.numeric(age)
  nihss <- as.numeric(nihss)
  time_gt_3h <- as.numeric(time_gt_3h)
  visual_field_defect <- as.numeric(visual_field_defect)
  glucose <- as.numeric(glucose)
  loc_reduced <- as.numeric(loc_reduced)

  if (any(is.na(c(age, nihss, time_gt_3h, visual_field_defect, glucose, loc_reduced)))) {
    stop("Error: All inputs must be numeric.")
  }

  glucose_mg_dl <- glucose
  if (glucose_units == "mmol/L") {
    glucose_mg_dl <- glucose * 18.018
  }

  score <- 0

  score <- score + floor(age / 5)

  score <- score + nihss

  if (time_gt_3h == 1) {
    score <- score + 2
  }

  if (visual_field_defect == 1) {
    score <- score + 2
  }

  if (glucose_mg_dl > 131) {
    score <- score + 1
  }

  if (loc_reduced == 1) {
    score <- score + 2
  }

  logit <- -3.86 + (0.126 * score)
  probability <- 1 / (1 + exp(-logit))
  probability_percent <- probability * 100

  return(list(
    ASTRAL_Score = score,
    Unfavorable_Outcome_Probability_3mo = paste0(round(probability_percent, 1), "%"),
    Inputs = list(
      Age_Points = floor(age / 5),
      NIHSS_Points = nihss,
      Time_Points = if (time_gt_3h == 1) 2 else 0,
      Visual_Points = if (visual_field_defect == 1) 2 else 0,
      Glucose_Points = if (glucose_mg_dl > 131) 1 else 0,
      LOC_Points = if (loc_reduced == 1) 2 else 0
    )
  ))
}

asymptomatic_myeloma_prognosis <- function(bmpc_percent, m_protein_g_dl, flc_ratio) {

  bmpc_percent <- as.numeric(bmpc_percent)
  m_protein_g_dl <- as.numeric(m_protein_g_dl)
  flc_ratio <- as.numeric(flc_ratio)

  if (any(is.na(c(bmpc_percent, m_protein_g_dl, flc_ratio)))) {
    stop("Error: All inputs must be numeric.")
  }

  risk_factors <- 0

  if (bmpc_percent > 20) {
    risk_factors <- risk_factors + 1
  }

  if (m_protein_g_dl > 2) {
    risk_factors <- risk_factors + 1
  }

  if (flc_ratio > 20) {
    risk_factors <- risk_factors + 1
  }

  risk_group <- ""
  median_ttp <- ""
  progression_2yr <- ""

  if (risk_factors == 0) {
    risk_group <- "Low Risk"
    median_ttp <- "110 months"
    progression_2yr <- "6%"
  } else if (risk_factors == 1) {
    risk_group <- "Intermediate Risk"
    median_ttp <- "68 months"
    progression_2yr <- "18%"
  } else {
    risk_group <- "High Risk"
    median_ttp <- "29 months"
    progression_2yr <- "46%"
  }

  return(list(
    Total_Risk_Factors = risk_factors,
    Risk_Group = risk_group,
    Median_Time_to_Progression = median_ttp,
    Progression_Risk_2yr = progression_2yr,
    Inputs = list(
      BMPC_gt_20 = bmpc_percent > 20,
      M_Protein_gt_2 = m_protein_g_dl > 2,
      FLC_Ratio_gt_20 = flc_ratio > 20
    )
  ))
}

atlas_cdi_score <- function(age, systemic_abx, wbc_count, albumin, creatinine, albumin_units = "g/dL", creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  systemic_abx <- as.numeric(systemic_abx)
  wbc_count <- as.numeric(wbc_count)
  albumin <- as.numeric(albumin)
  creatinine <- as.numeric(creatinine)

  if (any(is.na(c(age, systemic_abx, wbc_count, albumin, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 60 && age <= 79) {
    score <- score + 1
  } else if (age >= 80) {
    score <- score + 2
  }

  if (systemic_abx == 1) {
    score <- score + 2
  }

  wbc_val <- wbc_count
  if (wbc_val < 100) {
    wbc_val <- wbc_val * 1000
  }

  if (wbc_val >= 16000 && wbc_val <= 25000) {
    score <- score + 1
  } else if (wbc_val > 25000) {
    score <- score + 2
  }

  alb_g_dl <- albumin
  if (albumin_units == "g/L") {
    alb_g_dl <- albumin / 10
  }

  if (alb_g_dl > 2.5 && alb_g_dl <= 3.5) {
    score <- score + 1
  } else if (alb_g_dl <= 2.5) {
    score <- score + 2
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  if (creat_mg_dl >= 1.4 && creat_mg_dl <= 2.0) {
    score <- score + 1
  } else if (creat_mg_dl > 2.0) {
    score <- score + 2
  }

  mortality_risk <- ""
  cure_rate <- ""

  if (score <= 2) {
    mortality_risk <- "0%"
    cure_rate <- "93%"
  } else if (score <= 5) {
    mortality_risk <- "12%"
    cure_rate <- "86%"
  } else if (score <= 7) {
    mortality_risk <- "31%"
    cure_rate <- "74%"
  } else {
    mortality_risk <- "49%"
    cure_rate <- "60%"
  }

  return(list(
    ATLAS_Score = score,
    Predicted_Cure_Rate = cure_rate,
    Predicted_Mortality_30_Day = mortality_risk,
    Inputs = list(
      Age = age,
      Systemic_Antibiotics = systemic_abx == 1,
      WBC_Count = wbc_val,
      Albumin_g_dL = round(alb_g_dl, 2),
      Creatinine_mg_dL = round(creat_mg_dl, 2)
    )
  ))
}

atria_bleeding_risk <- function(age, sex, hemoglobin_g_dl, egfr_ml_min, prior_bleeding, hypertension) {

  age <- as.numeric(age)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  egfr_ml_min <- as.numeric(egfr_ml_min)
  prior_bleeding <- as.numeric(prior_bleeding)
  hypertension <- as.numeric(hypertension)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, hemoglobin_g_dl, egfr_ml_min, prior_bleeding, hypertension)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  is_anemic <- FALSE
  if (sex == "male" && hemoglobin_g_dl < 13) {
    is_anemic <- TRUE
  } else if (sex == "female" && hemoglobin_g_dl < 12) {
    is_anemic <- TRUE
  }

  if (is_anemic) {
    score <- score + 3
  }

  if (egfr_ml_min < 30) {
    score <- score + 3
  }

  if (age >= 75) {
    score <- score + 2
  }

  if (prior_bleeding == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  risk_category <- ""
  risk_percentage <- ""

  if (score <= 3) {
    risk_category <- "Low Risk"
    risk_percentage <- "<1%"
  } else if (score == 4) {
    risk_category <- "Intermediate Risk"
    risk_percentage <- "2.6%"
  } else {
    risk_category <- "High Risk"
    risk_percentage <- "5.8%"
  }

  return(list(
    ATRIA_Score = score,
    Risk_Category = risk_category,
    Annual_Major_Bleeding_Risk = risk_percentage,
    Inputs = list(
      Age = age,
      Anemia = is_anemic,
      Severe_Renal_Disease = egfr_ml_min < 30,
      Prior_Bleeding = prior_bleeding == 1,
      Hypertension = hypertension == 1
    )
  ))
}

atria_stroke_risk <- function(age, prior_stroke, female_sex, diabetes, chf, hypertension, proteinuria, egfr_lt_45_or_esrd) {

  age <- as.numeric(age)
  prior_stroke <- as.numeric(prior_stroke)
  female_sex <- as.numeric(female_sex)
  diabetes <- as.numeric(diabetes)
  chf <- as.numeric(chf)
  hypertension <- as.numeric(hypertension)
  proteinuria <- as.numeric(proteinuria)
  egfr_lt_45_or_esrd <- as.numeric(egfr_lt_45_or_esrd)

  if (any(is.na(c(age, prior_stroke, female_sex, diabetes, chf, hypertension, proteinuria, egfr_lt_45_or_esrd)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 85) {
    score <- score + 6
  } else if (age >= 75) {
    score <- score + 5
  } else if (age >= 65) {
    score <- score + 3
  }

  if (prior_stroke == 1) {
    score <- score + 4
  }

  if (female_sex == 1) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  if (chf == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (proteinuria == 1) {
    score <- score + 1
  }

  if (egfr_lt_45_or_esrd == 1) {
    score <- score + 1
  }

  risk_category <- ""
  stroke_rate <- ""

  if (score <= 5) {
    risk_category <- "Low Risk"
    stroke_rate <- "<1%"
  } else if (score == 6) {
    risk_category <- "Moderate Risk"
    stroke_rate <- "1.5%"
  } else {
    risk_category <- "High Risk"
    stroke_rate <- ">2%"
  }

  return(list(
    ATRIA_Score = score,
    Risk_Category = risk_category,
    Annual_Stroke_Risk = stroke_rate,
    Inputs = list(
      Age = age,
      Prior_Stroke = prior_stroke == 1,
      Female = female_sex == 1,
      Diabetes = diabetes == 1,
      CHF = chf == 1,
      Hypertension = hypertension == 1,
      Proteinuria = proteinuria == 1,
      Severe_Renal_Disease = egfr_lt_45_or_esrd == 1
    )
  ))
}

atropine_toxicity_dosing <- function(last_dose_mg, secretions_controlled, total_cumulative_dose_mg = NULL) {

  last_dose_mg <- as.numeric(last_dose_mg)
  secretions_controlled <- as.numeric(secretions_controlled)
  if (!is.null(total_cumulative_dose_mg)) total_cumulative_dose_mg <- as.numeric(total_cumulative_dose_mg)

  if (any(is.na(c(last_dose_mg, secretions_controlled)))) {
    stop("Error: Last dose and secretions status must be numeric.")
  }

  next_action <- ""
  next_dose <- NA
  infusion_rate <- NA

  if (secretions_controlled == 0) {
    next_dose <- last_dose_mg * 2
    next_action <- "Secretions NOT controlled. Administer double the previous dose IV immediately. Re-assess in 5-10 minutes."
  } else {
    next_action <- "Secretions controlled (Dry Axillae/Clear Lungs). Stop bolus dosing. Initiate maintenance infusion."
    if (!is.null(total_cumulative_dose_mg)) {
      min_rate <- total_cumulative_dose_mg * 0.10
      max_rate <- total_cumulative_dose_mg * 0.20
      infusion_rate <- paste0(round(min_rate, 1), " - ", round(max_rate, 1), " mg/hour")
    } else {
      infusion_rate <- "Calculate 10-20% of total cumulative loading dose per hour."
    }
  }

  return(list(
    Status = if(secretions_controlled == 1) "Controlled" else "Uncontrolled",
    Recommendation = next_action,
    Next_Bolus_Dose_mg = if(secretions_controlled == 0) next_dose else "N/A",
    Maintenance_Infusion_Rate = if(secretions_controlled == 1) infusion_rate else "N/A"
  ))
}

audit_c_alcohol_screen <- function(sex, frequency_score, amount_score, binge_score) {

  sex <- tolower(as.character(sex))
  frequency_score <- as.numeric(frequency_score)
  amount_score <- as.numeric(amount_score)
  binge_score <- as.numeric(binge_score)

  if (any(is.na(c(frequency_score, amount_score, binge_score)))) {
    stop("Error: All scores must be numeric integers (0-4).")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (any(c(frequency_score, amount_score, binge_score) < 0) ||
      any(c(frequency_score, amount_score, binge_score) > 4)) {
    stop("Error: Individual question scores must be between 0 and 4.")
  }

  total_score <- frequency_score + amount_score + binge_score

  result <- ""
  interpretation <- ""

  threshold <- if (sex == "male") 4 else 3

  if (total_score >= threshold) {
    result <- "Positive Screen"
    interpretation <- "Suggests misuse or alcohol use disorder. Further comprehensive assessment (full AUDIT) recommended."
  } else {
    result <- "Negative Screen"
    interpretation <- "Low risk for alcohol use disorder."
  }

  return(list(
    AUDIT_C_Score = total_score,
    Screen_Result = result,
    Interpretation = interpretation,
    Inputs = list(
      Sex = sex,
      Frequency_Pts = frequency_score,
      Amount_Pts = amount_score,
      Binge_Pts = binge_score
    )
  ))
}

ausdrisk_score <- function(age, sex, ethnicity_high_risk, family_history, high_glucose_history, bp_meds, smoker, fruit_veg_daily, physical_activity_sufficient, waist_cm) {

  age <- as.numeric(age)
  ethnicity_high_risk <- as.numeric(ethnicity_high_risk)
  family_history <- as.numeric(family_history)
  high_glucose_history <- as.numeric(high_glucose_history)
  bp_meds <- as.numeric(bp_meds)
  smoker <- as.numeric(smoker)
  fruit_veg_daily <- as.numeric(fruit_veg_daily)
  physical_activity_sufficient <- as.numeric(physical_activity_sufficient)
  waist_cm <- as.numeric(waist_cm)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, ethnicity_high_risk, family_history, high_glucose_history, bp_meds, smoker, fruit_veg_daily, physical_activity_sufficient, waist_cm)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age < 35) {
    score <- score + 0
  } else if (age >= 35 && age <= 44) {
    score <- score + 2
  } else if (age >= 45 && age <= 54) {
    score <- score + 4
  } else if (age >= 55 && age <= 64) {
    score <- score + 6
  } else {
    score <- score + 8
  }

  if (sex == "male") {
    score <- score + 3
  }

  if (ethnicity_high_risk == 1) {
    score <- score + 2
  }

  if (family_history == 1) {
    score <- score + 3
  }

  if (high_glucose_history == 1) {
    score <- score + 6
  }

  if (bp_meds == 1) {
    score <- score + 2
  }

  if (smoker == 1) {
    score <- score + 2
  }

  if (fruit_veg_daily == 0) {
    score <- score + 1
  }

  if (physical_activity_sufficient == 0) {
    score <- score + 2
  }

  if (sex == "male") {
    if (waist_cm >= 90 && waist_cm <= 100) {
      score <- score + 4
    } else if (waist_cm > 100) {
      score <- score + 7
    }
  } else {
    if (waist_cm >= 80 && waist_cm <= 90) {
      score <- score + 4
    } else if (waist_cm > 90) {
      score <- score + 7
    }
  }

  risk_level <- ""
  probability_5yr <- ""

  if (score <= 5) {
    risk_level <- "Low Risk"
    probability_5yr <- "Approximately 1 in 100 people will develop diabetes within 5 years."
  } else if (score <= 11) {
    risk_level <- "Intermediate Risk"
    probability_5yr <- "Approximately 1 in 50 people will develop diabetes within 5 years."
  } else {
    risk_level <- "High Risk"
    probability_5yr <- "Approximately 1 in 14 (scores 12-15) to 1 in 3 (scores 20+) people will develop diabetes within 5 years."
  }

  return(list(
    AUSDRISK_Score = score,
    Risk_Level = risk_level,
    Probability_5_Year = probability_5yr,
    Inputs = list(
      Age = age,
      Sex = sex,
      Ethnicity_Risk = ethnicity_high_risk == 1,
      Family_Hx = family_history == 1,
      Glucose_Hx = high_glucose_history == 1,
      BP_Meds = bp_meds == 1,
      Smoker = smoker == 1,
      Diet_Adequate = fruit_veg_daily == 1,
      Activity_Adequate = physical_activity_sufficient == 1,
      Waist_cm = waist_cm
    )
  ))
}

awol_delirium_risk <- function(age, spell_world_failed, orientation_failed, illness_severity_moderate_severe) {

  age <- as.numeric(age)
  spell_world_failed <- as.numeric(spell_world_failed)
  orientation_failed <- as.numeric(orientation_failed)
  illness_severity_moderate_severe <- as.numeric(illness_severity_moderate_severe)

  if (any(is.na(c(age, spell_world_failed, orientation_failed, illness_severity_moderate_severe)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 80) {
    score <- score + 1
  }

  if (spell_world_failed == 1) {
    score <- score + 1
  }

  if (orientation_failed == 1) {
    score <- score + 1
  }

  if (illness_severity_moderate_severe == 1) {
    score <- score + 1
  }

  risk_probability <- ""
  if (score == 0) {
    risk_probability <- "~2%"
  } else if (score == 1) {
    risk_probability <- "~5%"
  } else if (score == 2) {
    risk_probability <- "~17%"
  } else if (score == 3) {
    risk_probability <- "~37%"
  } else {
    risk_probability <- "~60%"
  }

  risk_category <- ""
  if (score >= 2) {
    risk_category <- "High Risk"
  } else {
    risk_category <- "Low Risk"
  }

  return(list(
    AWOL_Score = score,
    Risk_Category = risk_category,
    Est_Delirium_Risk = risk_probability,
    Inputs = list(
      Age_ge_80 = age >= 80,
      Spell_World_Failed = spell_world_failed == 1,
      Orientation_Failed = orientation_failed == 1,
      Illness_Severity_High = illness_severity_moderate_severe == 1
    )
  ))
}

bacterial_meningitis_score <- function(gram_stain_positive, csf_protein_mg_dl, peripheral_anc, seizure_history, csf_neutrophils) {

  gram_stain_positive <- as.numeric(gram_stain_positive)
  csf_protein_mg_dl <- as.numeric(csf_protein_mg_dl)
  peripheral_anc <- as.numeric(peripheral_anc)
  seizure_history <- as.numeric(seizure_history)
  csf_neutrophils <- as.numeric(csf_neutrophils)

  if (any(is.na(c(gram_stain_positive, csf_protein_mg_dl, peripheral_anc, seizure_history, csf_neutrophils)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (gram_stain_positive == 1) {
    score <- score + 2
  }

  if (csf_protein_mg_dl >= 80) {
    score <- score + 1
  }

  if (peripheral_anc >= 10000) {
    score <- score + 1
  }

  if (seizure_history == 1) {
    score <- score + 1
  }

  if (csf_neutrophils >= 1000) {
    score <- score + 1
  }

  risk_category <- ""
  interpretation <- ""

  if (score == 0) {
    risk_category <- "Very Low Risk"
    interpretation <- "Risk of bacterial meningitis is <0.1% (NPV 99.9%)."
  } else {
    risk_category <- "Not Low Risk"
    interpretation <- "Patient requires admission and empiric antibiotics."
  }

  return(list(
    BMS_Score = score,
    Risk_Category = risk_category,
    Interpretation = interpretation,
    Inputs = list(
      Pos_Gram_Stain = gram_stain_positive == 1,
      CSF_Protein_ge_80 = csf_protein_mg_dl >= 80,
      ANC_ge_10000 = peripheral_anc >= 10000,
      Seizure = seizure_history == 1,
      CSF_Neutrophils_ge_1000 = csf_neutrophils >= 1000
    )
  ))
}

ball_score_cll <- function(beta2_microglobulin_mg_l, hemoglobin_g_dl, sex, ldh_elevated, last_therapy_pfs_lt_24mo) {

  beta2_microglobulin_mg_l <- as.numeric(beta2_microglobulin_mg_l)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  ldh_elevated <- as.numeric(ldh_elevated)
  last_therapy_pfs_lt_24mo <- as.numeric(last_therapy_pfs_lt_24mo)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(beta2_microglobulin_mg_l, hemoglobin_g_dl, ldh_elevated, last_therapy_pfs_lt_24mo)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (beta2_microglobulin_mg_l > 3.5) {
    score <- score + 1
  }

  if (sex == "male") {
    if (hemoglobin_g_dl < 12.0) {
      score <- score + 1
    }
  } else {
    if (hemoglobin_g_dl < 11.0) {
      score <- score + 1
    }
  }

  if (ldh_elevated == 1) {
    score <- score + 1
  }

  if (last_therapy_pfs_lt_24mo == 1) {
    score <- score + 1
  }

  risk_group <- ""
  survival_2yr <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    survival_2yr <- "85-90%"
  } else if (score <= 3) {
    risk_group <- "Intermediate Risk"
    survival_2yr <- "60-65%"
  } else {
    risk_group <- "High Risk"
    survival_2yr <- "<40%"
  }

  return(list(
    BALL_Score = score,
    Risk_Group = risk_group,
    Est_2Yr_Overall_Survival = survival_2yr,
    Inputs = list(
      B2M_gt_3_5 = beta2_microglobulin_mg_l > 3.5,
      Anemia = if(sex == "male") hemoglobin_g_dl < 12 else hemoglobin_g_dl < 11,
      LDH_Elevated = ldh_elevated == 1,
      Last_Therapy_lt_24mo = last_therapy_pfs_lt_24mo == 1
    )
  ))
}

bap65_copd <- function(bun, altered_mental_status, pulse, age) {

  bun <- as.numeric(bun)
  altered_mental_status <- as.numeric(altered_mental_status)
  pulse <- as.numeric(pulse)
  age <- as.numeric(age)

  if (any(is.na(c(bun, altered_mental_status, pulse, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (bun >= 25) {
    score <- score + 1
  }

  if (altered_mental_status == 1) {
    score <- score + 1
  }

  if (pulse >= 109) {
    score <- score + 1
  }

  if (age > 65) {
    score <- score + 1
  }

  risk_class <- ""
  mortality <- ""
  recommendation <- ""

  if (score == 0) {
    risk_class <- "Class I (Low Risk)"
    mortality <- "<1%"
    recommendation <- "Consider early discharge with follow-up."
  } else if (score == 1) {
    risk_class <- "Class II (Moderate Risk)"
    mortality <- "~2%"
    recommendation <- "Admit to general ward."
  } else if (score == 2) {
    risk_class <- "Class III (High Risk)"
    mortality <- "~6%"
    recommendation <- "Admit to general ward or respiratory unit."
  } else {
    risk_class <- "Class IV (Very High Risk)"
    mortality <- "12-30%"
    recommendation <- "Consider ICU admission and/or Non-Invasive Ventilation (NIV)."
  }

  return(list(
    BAP65_Score = score,
    Risk_Class = risk_class,
    In_Hospital_Mortality = mortality,
    Recommendation = recommendation,
    Inputs = list(
      BUN_ge_25 = bun >= 25,
      AMS = altered_mental_status == 1,
      Pulse_ge_109 = pulse >= 109,
      Age_gt_65 = age > 65
    )
  ))
}

bclc_staging <- function(ecog_ps, tumor_characteristics, child_pugh_class) {

  ecog_ps <- as.numeric(ecog_ps)
  tumor_characteristics <- tolower(as.character(tumor_characteristics))
  child_pugh_class <- toupper(as.character(child_pugh_class))

  if (is.na(ecog_ps) || !ecog_ps %in% 0:4) {
    stop("Error: ECOG PS must be an integer between 0 and 4.")
  }

  valid_tumors <- c("single_lt_2cm", "single_gt_2cm_or_3_lt_3cm", "multinodular", "invasion_or_metastasis")
  if (!tumor_characteristics %in% valid_tumors) {
    stop("Error: Tumor characteristics must be 'single_lt_2cm', 'single_gt_2cm_or_3_lt_3cm', 'multinodular', or 'invasion_or_metastasis'.")
  }

  if (!child_pugh_class %in% c("A", "B", "C")) {
    stop("Error: Child-Pugh Class must be 'A', 'B', or 'C'.")
  }

  stage <- ""
  prognosis <- ""
  treatment <- ""

  if (child_pugh_class == "C" || ecog_ps > 2) {
    stage <- "Stage D (Terminal)"
    prognosis <- "Median Survival: ~3 months"
    treatment <- "Best Supportive Care (Consider transplant if within Milan criteria and poor liver function is the only driver)."
  } else if (tumor_characteristics == "invasion_or_metastasis" || (ecog_ps >= 1 && ecog_ps <= 2)) {
    stage <- "Stage C (Advanced)"
    prognosis <- "Median Survival: ~11 months (varies with therapy)"
    treatment <- "Systemic Therapy (e.g., Atezolizumab+Bevacizumab, Sorafenib, Lenvatinib)."
  } else if (tumor_characteristics == "multinodular") {
    stage <- "Stage B (Intermediate)"
    prognosis <- "Median Survival: ~20 months"
    treatment <- "Chemoembolization (TACE)."
  } else if (tumor_characteristics == "single_gt_2cm_or_3_lt_3cm") {
    stage <- "Stage A (Early)"
    prognosis <- "Median Survival: >5 years (50-70% at 5 years)"
    treatment <- "Curative intent: Resection, Liver Transplant, or Ablation."
  } else {
    if (child_pugh_class == "A") {
      stage <- "Stage 0 (Very Early)"
      prognosis <- "Median Survival: >5 years"
      treatment <- "Resection or Ablation."
    } else {
      stage <- "Stage A (Early)"
      prognosis <- "Median Survival: >5 years"
      treatment <- "Consider Liver Transplant or Ablation."
    }
  }

  return(list(
    BCLC_Stage = stage,
    Prognosis = prognosis,
    Treatment_Recommendation = treatment,
    Inputs = list(
      ECOG = ecog_ps,
      Tumor = tumor_characteristics,
      Child_Pugh = child_pugh_class
    )
  ))
}

barnes_jewish_dysphagia_screen <- function(gcs, facial_asymmetry, tongue_asymmetry, palatal_asymmetry, water_swallow_signs = NULL) {

  gcs <- as.numeric(gcs)
  facial_asymmetry <- as.numeric(facial_asymmetry)
  tongue_asymmetry <- as.numeric(tongue_asymmetry)
  palatal_asymmetry <- as.numeric(palatal_asymmetry)

  if (!is.null(water_swallow_signs)) {
    water_swallow_signs <- as.numeric(water_swallow_signs)
  }

  if (any(is.na(c(gcs, facial_asymmetry, tongue_asymmetry, palatal_asymmetry)))) {
    stop("Error: GCS and anatomical assessment inputs must be numeric.")
  }

  step1_fail <- FALSE
  reasons <- c()

  if (gcs < 13) {
    step1_fail <- TRUE
    reasons <- c(reasons, "GCS < 13")
  }

  if (facial_asymmetry == 1) {
    step1_fail <- TRUE
    reasons <- c(reasons, "Facial Asymmetry")
  }

  if (tongue_asymmetry == 1) {
    step1_fail <- TRUE
    reasons <- c(reasons, "Tongue Asymmetry")
  }

  if (palatal_asymmetry == 1) {
    step1_fail <- TRUE
    reasons <- c(reasons, "Palatal Asymmetry")
  }

  result <- ""
  action <- ""

  if (step1_fail) {
    result <- "Failed Bedside Assessment"
    action <- "Keep patient NPO. Consult Speech Therapy for formal evaluation."
  } else {
    if (is.null(water_swallow_signs)) {
      result <- "Passed Bedside Assessment"
      action <- "Proceed to 3-oz Water Swallow Test."
    } else {
      if (water_swallow_signs == 1) {
        result <- "Failed Water Swallow Test"
        action <- "Keep patient NPO. Consult Speech Therapy."
        reasons <- c(reasons, "Signs of aspiration on water swallow")
      } else {
        result <- "Passed Screen"
        action <- "Patient may start oral intake (e.g., standardized stroke diet). Observe closely."
      }
    }
  }

  return(list(
    Screen_Result = result,
    Action_Plan = action,
    Fail_Criteria = if (length(reasons) > 0) reasons else "None",
    Inputs = list(
      GCS = gcs,
      Facial_Asym = facial_asymmetry == 1,
      Tongue_Asym = tongue_asymmetry == 1,
      Palatal_Asym = palatal_asymmetry == 1,
      Water_Test_Signs = if (!is.null(water_swallow_signs)) (water_swallow_signs == 1) else "Not Performed"
    )
  ))
}

barthel_index_adl <- function(feeding, bathing, grooming, dressing, bowels, bladder, toilet_use, transfers, mobility, stairs) {

  feeding <- as.numeric(feeding)
  bathing <- as.numeric(bathing)
  grooming <- as.numeric(grooming)
  dressing <- as.numeric(dressing)
  bowels <- as.numeric(bowels)
  bladder <- as.numeric(bladder)
  toilet_use <- as.numeric(toilet_use)
  transfers <- as.numeric(transfers)
  mobility <- as.numeric(mobility)
  stairs <- as.numeric(stairs)

  if (any(is.na(c(feeding, bathing, grooming, dressing, bowels, bladder, toilet_use, transfers, mobility, stairs)))) {
    stop("Error: All inputs must be numeric.")
  }

  total_score <- feeding + bathing + grooming + dressing + bowels + bladder + toilet_use + transfers + mobility + stairs

  dependency_level <- ""

  if (total_score <= 20) {
    dependency_level <- "Total Dependence"
  } else if (total_score <= 60) {
    dependency_level <- "Severe Dependence"
  } else if (total_score <= 90) {
    dependency_level <- "Moderate Dependence"
  } else if (total_score <= 99) {
    dependency_level <- "Slight Dependence"
  } else {
    dependency_level <- "Independence"
  }

  return(list(
    Barthel_Index_Score = total_score,
    Dependency_Level = dependency_level,
    Inputs = list(
      Feeding = feeding,
      Bathing = bathing,
      Grooming = grooming,
      Dressing = dressing,
      Bowels = bowels,
      Bladder = bladder,
      Toilet_Use = toilet_use,
      Transfers = transfers,
      Mobility = mobility,
      Stairs = stairs
    )
  ))
}

basal_energy_expenditure <- function(sex, height_cm, weight_kg, age) {

  sex <- tolower(as.character(sex))
  height_cm <- as.numeric(height_cm)
  weight_kg <- as.numeric(weight_kg)
  age <- as.numeric(age)

  if (any(is.na(c(height_cm, weight_kg, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  bee_val <- 0

  if (sex == "male") {
    bee_val <- 66.4730 + (13.7516 * weight_kg) + (5.0033 * height_cm) - (6.7550 * age)
  } else {
    bee_val <- 655.0955 + (9.5634 * weight_kg) + (1.8496 * height_cm) - (4.6756 * age)
  }

  return(list(
    BEE_kcal_day = round(bee_val, 0),
    Formula = "Harris-Benedict Equation (Original 1919)",
    Inputs = list(
      Sex = sex,
      Height_cm = height_cm,
      Weight_kg = weight_kg,
      Age = age
    )
  ))
}

basic_stats_calculator <- function(tp, fp, fn, tn) {

  tp <- as.numeric(tp)
  fp <- as.numeric(fp)
  fn <- as.numeric(fn)
  tn <- as.numeric(tn)

  if (any(is.na(c(tp, fp, fn, tn)))) {
    stop("Error: All inputs must be numeric.")
  }

  total <- tp + fp + fn + tn

  if (total == 0) {
    stop("Error: Total N must be greater than 0.")
  }

  sens_denom <- tp + fn
  spec_denom <- tn + fp
  ppv_denom <- tp + fp
  npv_denom <- tn + fn

  sensitivity <- if (sens_denom > 0) (tp / sens_denom) * 100 else NA
  specificity <- if (spec_denom > 0) (tn / spec_denom) * 100 else NA
  ppv <- if (ppv_denom > 0) (tp / ppv_denom) * 100 else NA
  npv <- if (npv_denom > 0) (tn / npv_denom) * 100 else NA
  accuracy <- ((tp + tn) / total) * 100
  prevalence <- (sens_denom / total) * 100

  lr_pos <- NA
  if (!is.na(sensitivity) && !is.na(specificity)) {
    if (specificity != 100) {
      lr_pos <- sensitivity / (100 - specificity)
    } else {
      lr_pos <- Inf
    }
  }

  lr_neg <- NA
  if (!is.na(sensitivity) && !is.na(specificity)) {
    if (specificity != 0) {
      lr_neg <- (100 - sensitivity) / specificity
    } else {
      lr_neg <- Inf
    }
  }

  return(list(
    Diagnostic_Accuracy = list(
      Sensitivity_Percent = if(!is.na(sensitivity)) round(sensitivity, 1) else NA,
      Specificity_Percent = if(!is.na(specificity)) round(specificity, 1) else NA,
      Accuracy_Percent = round(accuracy, 1)
    ),
    Predictive_Values = list(
      PPV_Percent = if(!is.na(ppv)) round(ppv, 1) else NA,
      NPV_Percent = if(!is.na(npv)) round(npv, 1) else NA
    ),
    Likelihood_Ratios = list(
      LR_Positive = if(!is.na(lr_pos)) round(lr_pos, 2) else NA,
      LR_Negative = if(!is.na(lr_neg)) round(lr_neg, 2) else NA
    ),
    Population_Stats = list(
      Prevalence_Percent = round(prevalence, 1),
      Total_N = total
    ),
    Inputs = list(
      TP = tp, FP = fp, FN = fn, TN = tn
    )
  ))
}

bastion_classification_lower_limb <- function(injury_level, segmental_injury = 0, abdominal_injury = 0, genital_perineal_injury = 0, pelvic_ring_injury = 0, upper_limb_injury = 0) {

  injury_level <- as.numeric(injury_level)
  segmental_injury <- as.numeric(segmental_injury)
  abdominal_injury <- as.numeric(abdominal_injury)
  genital_perineal_injury <- as.numeric(genital_perineal_injury)
  pelvic_ring_injury <- as.numeric(pelvic_ring_injury)
  upper_limb_injury <- as.numeric(upper_limb_injury)

  if (is.na(injury_level) || !injury_level %in% 1:5) {
    stop("Error: Injury Level must be an integer from 1 to 5.")
  }

  if (any(is.na(c(segmental_injury, abdominal_injury, genital_perineal_injury, pelvic_ring_injury, upper_limb_injury)))) {
    stop("Error: Associated injury inputs must be numeric (0 or 1).")
  }

  class_description <- ""
  tourniquet_status <- ""

  if (injury_level == 1) {
    class_description <- "Class 1: Injury confined to the foot."
    tourniquet_status <- "Effective below-knee tourniquet possible."
  } else if (injury_level == 2) {
    class_description <- "Class 2: Injury involving lower leg."
    tourniquet_status <- "Effective below-knee tourniquet possible."
  } else if (injury_level == 3) {
    class_description <- "Class 3: Injury involving proximal lower leg or thigh."
    tourniquet_status <- "Effective above-knee tourniquet possible."
  } else if (injury_level == 4) {
    class_description <- "Class 4: Proximal thigh injury."
    tourniquet_status <- "Prevents effective tourniquet application."
  } else {
    class_description <- "Class 5: Any injury with buttock involvement."
    tourniquet_status <- "Prevents effective tourniquet application."
  }

  associated_injuries <- list()

  if (segmental_injury == 1) associated_injuries$Segmental <- "Yes (Viable tissue distal to injury)"
  if (abdominal_injury == 1) associated_injuries$Abdominal <- "Yes (Intraperitoneal)"
  if (genital_perineal_injury == 1) associated_injuries$Genital_Perineal <- "Yes"
  if (pelvic_ring_injury == 1) associated_injuries$Pelvic_Ring <- "Yes"
  if (upper_limb_injury == 1) associated_injuries$Upper_Limb <- "Yes"

  if (length(associated_injuries) == 0) {
    associated_injuries <- "None"
  }

  return(list(
    Bastion_Class = paste("Class", injury_level),
    Description = class_description,
    Tourniquet_Guide = tourniquet_status,
    Associated_Injuries = associated_injuries
  ))
}

beam_value_diabetes <- function(bedtime_glucose, am_glucose, units = "mg/dL") {

  bedtime_glucose <- as.numeric(bedtime_glucose)
  am_glucose <- as.numeric(am_glucose)

  if (any(is.na(c(bedtime_glucose, am_glucose)))) {
    stop("Error: Glucose inputs must be numeric.")
  }

  beam_val <- bedtime_glucose - am_glucose

  threshold <- 50
  if (units == "mmol/L") {
    threshold <- 50 / 18.018
  }

  interpretation <- ""
  if (beam_val > threshold) {
    interpretation <- "High BeAM Value. Indicates significant rise in glucose from morning to bedtime, suggesting uncontrolled postprandial glucose. Consider adding prandial insulin."
  } else {
    interpretation <- "Low/Normal BeAM Value. Basal insulin titration may be the primary focus if HbA1c is elevated, or control is adequate."
  }

  return(list(
    BeAM_Value = round(beam_val, 1),
    Interpretation = interpretation,
    Units = units
  ))
}

bars_agitation_score <- function(patient_state_score) {

  patient_state_score <- as.numeric(patient_state_score)

  if (is.na(patient_state_score) || !patient_state_score %in% 1:7) {
    stop("Error: Input must be an integer from 1 to 7 corresponding to the BARS level.")
  }

  description <- ""
  clinical_status <- ""

  if (patient_state_score == 1) {
    description <- "Difficult or unable to rouse."
    clinical_status <- "Over-sedated"
  } else if (patient_state_score == 2) {
    description <- "Asleep but responds normally to verbal or physical contact."
    clinical_status <- "Sedated"
  } else if (patient_state_score == 3) {
    description <- "Drowsy, appears sedated."
    clinical_status <- "Sedated"
  } else if (patient_state_score == 4) {
    description <- "Quiet and awake (normal level of activity)."
    clinical_status <- "Normal / Euthymic"
  } else if (patient_state_score == 5) {
    description <- "Signs of overt (physical or verbal) activity, calms down with instructions."
    clinical_status <- "Agitated"
  } else if (patient_state_score == 6) {
    description <- "Extremely or continuously active, not requiring restraint."
    clinical_status <- "Highly Agitated"
  } else {
    description <- "Violent, requires restraint."
    clinical_status <- "Severely Agitated / Dangerous"
  }

  return(list(
    BARS_Score = patient_state_score,
    Description = description,
    Clinical_Status = clinical_status
  ))
}

bops_pediatric_pain <- function(facial_expression, verbalization, body_position) {

  facial_expression <- as.numeric(facial_expression)
  verbalization <- as.numeric(verbalization)
  body_position <- as.numeric(body_position)

  if (any(is.na(c(facial_expression, verbalization, body_position)))) {
    stop("Error: All inputs must be numeric integers (0, 1, or 2).")
  }

  if (any(c(facial_expression, verbalization, body_position) < 0) ||
      any(c(facial_expression, verbalization, body_position) > 2)) {
    stop("Error: Individual scores must be between 0 and 2.")
  }

  total_score <- facial_expression + verbalization + body_position

  interpretation <- ""
  if (total_score < 3) {
    interpretation <- "Minimal or no pain. No intervention likely required."
  } else {
    interpretation <- "Significant pain (Moderate to Severe). Analgesia recommended."
  }

  return(list(
    BOPS_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Facial = facial_expression,
      Verbal = verbalization,
      Body = body_position
    )
  ))
}

bps_intubated_pain <- function(facial_expression, upper_limb_movements, compliance_ventilation) {

  facial_expression <- as.numeric(facial_expression)
  upper_limb_movements <- as.numeric(upper_limb_movements)
  compliance_ventilation <- as.numeric(compliance_ventilation)

  if (any(is.na(c(facial_expression, upper_limb_movements, compliance_ventilation)))) {
    stop("Error: All inputs must be numeric integers (1-4).")
  }

  if (any(c(facial_expression, upper_limb_movements, compliance_ventilation) < 1) ||
      any(c(facial_expression, upper_limb_movements, compliance_ventilation) > 4)) {
    stop("Error: Individual scores must be between 1 and 4.")
  }

  total_score <- facial_expression + upper_limb_movements + compliance_ventilation

  interpretation <- ""
  if (total_score <= 3) {
    interpretation <- "No pain (Score 3)"
  } else if (total_score <= 5) {
    interpretation <- "Mild pain (Score 4-5)"
  } else {
    interpretation <- "Significant pain (Score >= 6). Analgesia recommended."
  }

  return(list(
    BPS_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Facial = facial_expression,
      Upper_Limbs = upper_limb_movements,
      Ventilation = compliance_ventilation
    )
  ))
}

benzodiazepine_conversion <- function(source_drug, source_dose, target_drug) {

  source_drug <- tolower(as.character(source_drug))
  target_drug <- tolower(as.character(target_drug))
  source_dose <- as.numeric(source_dose)

  if (is.na(source_dose) || source_dose < 0) {
    stop("Error: Source dose must be a positive number.")
  }

  equiv_to_10mg_diazepam <- list(
    "alprazolam" = 0.5,
    "chlordiazepoxide" = 25,
    "clobazam" = 20,
    "clonazepam" = 0.5,
    "clorazepate" = 15,
    "diazepam" = 10,
    "estazolam" = 2,
    "flurazepam" = 30,
    "lorazepam" = 1,
    "midazolam_oral" = 7.5,
    "midazolam_iv" = 4,
    "nitrazepam" = 10,
    "oxazepam" = 20,
    "quazepam" = 20,
    "temazepam" = 20,
    "triazolam" = 0.25
  )

  if (!source_drug %in% names(equiv_to_10mg_diazepam)) {
    stop(paste("Error: Source drug not recognized. Options:", paste(names(equiv_to_10mg_diazepam), collapse = ", ")))
  }

  if (!target_drug %in% names(equiv_to_10mg_diazepam)) {
    stop(paste("Error: Target drug not recognized. Options:", paste(names(equiv_to_10mg_diazepam), collapse = ", ")))
  }

  source_factor <- equiv_to_10mg_diazepam[[source_drug]]
  target_factor <- equiv_to_10mg_diazepam[[target_drug]]

  diazepam_equivalent <- (source_dose / source_factor) * 10

  target_dose <- (diazepam_equivalent / 10) * target_factor

  return(list(
    Input = list(Drug = source_drug, Dose_mg = source_dose),
    Output = list(Drug = target_drug, Estimated_Dose_mg = round(target_dose, 3)),
    Diazepam_Equivalent_mg = round(diazepam_equivalent, 2),
    Note = "Doses are approximate equivalents. Incomplete cross-tolerance may require dose reduction (e.g., 25-50%) when switching agents."
  ))
}

berg_balance_scale <- function(sit_to_stand, standing_unsupported, sitting_unsupported, stand_to_sit, transfers, standing_eyes_closed, standing_feet_together, reaching_forward, retrieving_object, turning_to_look_behind, turning_360, stool_stepping, tandem_standing, one_leg_stand) {

  inputs <- c(sit_to_stand, standing_unsupported, sitting_unsupported, stand_to_sit, transfers, standing_eyes_closed, standing_feet_together, reaching_forward, retrieving_object, turning_to_look_behind, turning_360, stool_stepping, tandem_standing, one_leg_stand)

  if (any(is.na(inputs))) {
    stop("Error: All 14 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be between 0 and 4.")
  }

  total_score <- sum(inputs)

  risk_category <- ""

  if (total_score >= 41) {
    risk_category <- "Low Fall Risk"
  } else if (total_score >= 21) {
    risk_category <- "Medium Fall Risk"
  } else {
    risk_category <- "High Fall Risk"
  }

  return(list(
    BBS_Total_Score = total_score,
    Fall_Risk_Classification = risk_category,
    Inputs = list(
      Sit_to_Stand = sit_to_stand,
      Standing_Unsupported = standing_unsupported,
      Sitting_Unsupported = sitting_unsupported,
      Stand_to_Sit = stand_to_sit,
      Transfers = transfers,
      Standing_Eyes_Closed = standing_eyes_closed,
      Standing_Feet_Together = standing_feet_together,
      Reaching_Forward = reaching_forward,
      Retrieving_Object = retrieving_object,
      Turning_Behind = turning_to_look_behind,
      Turning_360 = turning_360,
      Stool_Stepping = stool_stepping,
      Tandem_Standing = tandem_standing,
      One_Leg_Stand = one_leg_stand
    )
  ))
}

berlin_ards_criteria <- function(timing_within_1_week, bilateral_opacities, edema_not_cardiac, pao2_fio2_ratio, peep_or_cpap) {

  timing_within_1_week <- as.numeric(timing_within_1_week)
  bilateral_opacities <- as.numeric(bilateral_opacities)
  edema_not_cardiac <- as.numeric(edema_not_cardiac)
  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  peep_or_cpap <- as.numeric(peep_or_cpap)

  if (any(is.na(c(timing_within_1_week, bilateral_opacities, edema_not_cardiac, pao2_fio2_ratio, peep_or_cpap)))) {
    stop("Error: All inputs must be numeric.")
  }

  criteria_met <- TRUE
  reasons <- c()

  if (timing_within_1_week == 0) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Respiratory symptoms must begin within 1 week of a known clinical insult.")
  }

  if (bilateral_opacities == 0) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Bilateral opacities on chest imaging not fully explained by effusions, lobar/lung collapse, or nodules.")
  }

  if (edema_not_cardiac == 0) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Respiratory failure not fully explained by cardiac failure or fluid overload.")
  }

  if (peep_or_cpap < 5) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Minimum PEEP or CPAP of 5 cmH2O is required.")
  }

  severity <- "Not ARDS"
  mortality <- "N/A"

  if (criteria_met) {
    if (pao2_fio2_ratio <= 100) {
      severity <- "Severe ARDS"
      mortality <- "45% (95% CI, 42-48%)"
    } else if (pao2_fio2_ratio <= 200) {
      severity <- "Moderate ARDS"
      mortality <- "32% (95% CI, 29-34%)"
    } else if (pao2_fio2_ratio <= 300) {
      severity <- "Mild ARDS"
      mortality <- "27% (95% CI, 24-30%)"
    } else {
      severity <- "Not ARDS"
      reasons <- c(reasons, "PaO2/FiO2 ratio > 300 mmHg.")
    }
  }

  return(list(
    Classification = severity,
    Predicted_Mortality = mortality,
    Criteria_Met = criteria_met,
    Failure_Reasons = if (length(reasons) > 0) reasons else "None",
    Inputs = list(
      Timing = timing_within_1_week == 1,
      Imaging = bilateral_opacities == 1,
      Edema_Origin = edema_not_cardiac == 1,
      PF_Ratio = pao2_fio2_ratio,
      PEEP = peep_or_cpap
    )
  ))
}

bicarbonate_deficit <- function(weight_kg, current_hco3, desired_hco3 = 24, volume_distribution_fraction = 0.4) {

  weight_kg <- as.numeric(weight_kg)
  current_hco3 <- as.numeric(current_hco3)
  desired_hco3 <- as.numeric(desired_hco3)
  volume_distribution_fraction <- as.numeric(volume_distribution_fraction)

  if (any(is.na(c(weight_kg, current_hco3, desired_hco3, volume_distribution_fraction)))) {
    stop("Error: All inputs must be numeric.")
  }

  deficit_meq <- volume_distribution_fraction * weight_kg * (desired_hco3 - current_hco3)

  if (deficit_meq < 0) {
    deficit_meq <- 0
  }

  admin_50_percent <- deficit_meq * 0.5

  return(list(
    Bicarbonate_Deficit_mEq = round(deficit_meq, 1),
    Recommended_Initial_Dose_mEq = round(admin_50_percent, 1),
    Note = "Clinical practice typically suggests replacing only 50% of the calculated deficit initially to avoid overcorrection and hypernatremia.",
    Inputs = list(
      Weight_kg = weight_kg,
      Current_HCO3 = current_hco3,
      Desired_HCO3 = desired_hco3,
      Vol_Dist_Factor = volume_distribution_fraction
    )
  ))
}

binet_staging_cll <- function(hemoglobin_g_dl, platelets_per_ul, cervical_nodes, axillary_nodes, inguinal_nodes, liver_spleen_palpable) {

  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  platelets_per_ul <- as.numeric(platelets_per_ul)
  cervical_nodes <- as.numeric(cervical_nodes)
  axillary_nodes <- as.numeric(axillary_nodes)
  inguinal_nodes <- as.numeric(inguinal_nodes)
  liver_spleen_palpable <- as.numeric(liver_spleen_palpable)

  if (any(is.na(c(hemoglobin_g_dl, platelets_per_ul, cervical_nodes, axillary_nodes, inguinal_nodes, liver_spleen_palpable)))) {
    stop("Error: All inputs must be numeric.")
  }

  platelet_count <- platelets_per_ul
  if (platelet_count < 1000) {
    platelet_count <- platelet_count * 1000
  }

  organomegaly_count <- 0
  if (cervical_nodes == 1) organomegaly_count <- organomegaly_count + 1
  if (axillary_nodes == 1) organomegaly_count <- organomegaly_count + 1
  if (inguinal_nodes == 1) organomegaly_count <- organomegaly_count + 1
  if (liver_spleen_palpable == 1) organomegaly_count <- organomegaly_count + 1

  stage <- ""
  description <- ""
  median_survival <- ""

  if (hemoglobin_g_dl < 10 || platelet_count < 100000) {
    stage <- "Stage C"
    description <- "Anemia (Hb < 10 g/dL) and/or Thrombocytopenia (Plt < 100k). Number of enlarged areas does not matter."
    median_survival <- "2 years"
  } else if (organomegaly_count >= 3) {
    stage <- "Stage B"
    description <- "Hb >= 10 g/dL, Plt >= 100k, and >= 3 involved lymphoid areas."
    median_survival <- "7 years"
  } else {
    stage <- "Stage A"
    description <- "Hb >= 10 g/dL, Plt >= 100k, and < 3 involved lymphoid areas."
    median_survival <- "12 years"
  }

  return(list(
    Binet_Stage = stage,
    Criteria_Description = description,
    Median_Survival = median_survival,
    Inputs = list(
      Hemoglobin = hemoglobin_g_dl,
      Platelets = platelet_count,
      Enlarged_Areas_Count = organomegaly_count
    )
  ))
}

bisap_score <- function(bun, impaired_mental_status, sirs_criteria_met, age, pleural_effusion, bun_units = "mg/dL") {

  bun <- as.numeric(bun)
  impaired_mental_status <- as.numeric(impaired_mental_status)
  sirs_criteria_met <- as.numeric(sirs_criteria_met)
  age <- as.numeric(age)
  pleural_effusion <- as.numeric(pleural_effusion)

  if (any(is.na(c(bun, impaired_mental_status, sirs_criteria_met, age, pleural_effusion)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  bun_val <- bun
  if (bun_units == "mmol/L") {
    bun_val <- bun * 2.8
  }

  if (bun_val > 25) {
    score <- score + 1
  }

  if (impaired_mental_status == 1) {
    score <- score + 1
  }

  if (sirs_criteria_met == 1) {
    score <- score + 1
  }

  if (age > 60) {
    score <- score + 1
  }

  if (pleural_effusion == 1) {
    score <- score + 1
  }

  mortality_risk <- ""
  if (score <= 2) {
    mortality_risk <- "Low Mortality (<2%)"
  } else {
    mortality_risk <- "High Mortality (>15%)"
  }

  return(list(
    BISAP_Score = score,
    Mortality_Risk = mortality_risk,
    Inputs = list(
      BUN_gt_25 = bun_val > 25,
      Impaired_Mental_Status = impaired_mental_status == 1,
      SIRS_Met = sirs_criteria_met == 1,
      Age_gt_60 = age > 60,
      Pleural_Effusion = pleural_effusion == 1
    )
  ))
}

bishop_score_induction <- function(dilation_cm, effacement_percent, station, consistency, position) {

  dilation_cm <- as.numeric(dilation_cm)
  effacement_percent <- as.numeric(effacement_percent)
  station <- as.numeric(station)
  consistency <- tolower(as.character(consistency))
  position <- tolower(as.character(position))

  if (any(is.na(c(dilation_cm, effacement_percent, station)))) {
    stop("Error: Dilation, Effacement, and Station must be numeric.")
  }

  if (!consistency %in% c("firm", "medium", "soft")) {
    stop("Error: Consistency must be 'firm', 'medium', or 'soft'.")
  }

  if (!position %in% c("posterior", "midposition", "anterior")) {
    stop("Error: Position must be 'posterior', 'midposition', or 'anterior'.")
  }

  score <- 0

  if (dilation_cm == 0) {
    score <- score + 0
  } else if (dilation_cm >= 1 && dilation_cm <= 2) {
    score <- score + 1
  } else if (dilation_cm >= 3 && dilation_cm <= 4) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (effacement_percent <= 30) {
    score <- score + 0
  } else if (effacement_percent >= 40 && effacement_percent <= 50) {
    score <- score + 1
  } else if (effacement_percent >= 60 && effacement_percent <= 70) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (station <= -3) {
    score <- score + 0
  } else if (station == -2) {
    score <- score + 1
  } else if (station == -1 || station == 0) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (consistency == "firm") {
    score <- score + 0
  } else if (consistency == "medium") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (position == "posterior") {
    score <- score + 0
  } else if (position == "midposition") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  interpretation <- ""
  if (score > 8) {
    interpretation <- "Likelihood of vaginal delivery is similar to that of spontaneous labor (Favorable)."
  } else if (score < 6) {
    interpretation <- "Unfavorable cervix. Cervical ripening may be indicated."
  } else {
    interpretation <- "Indeterminate/Intermediate."
  }

  return(list(
    Bishop_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Dilation = dilation_cm,
      Effacement = effacement_percent,
      Station = station,
      Consistency = consistency,
      Position = position
    )
  ))
}

blast_lung_injury_severity <- function(pao2_fio2_ratio, chest_xray_findings, bronchopleural_fistula) {

  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  chest_xray_findings <- as.numeric(chest_xray_findings)
  bronchopleural_fistula <- as.numeric(bronchopleural_fistula)

  if (any(is.na(c(pao2_fio2_ratio, chest_xray_findings, bronchopleural_fistula)))) {
    stop("Error: All inputs must be numeric.")
  }

  pf_score <- 0
  if (pao2_fio2_ratio > 200) {
    pf_score <- 0
  } else if (pao2_fio2_ratio >= 60) {
    pf_score <- 2
  } else {
    pf_score <- 4
  }

  cxr_score <- 0
  if (chest_xray_findings == 0) {
    cxr_score <- 1
  } else if (chest_xray_findings == 1) {
    cxr_score <- 2
  } else {
    cxr_score <- 3
  }

  bpf_score <- 0
  if (bronchopleural_fistula == 1) {
    bpf_score <- 1
  }

  total_score <- pf_score + cxr_score + bpf_score

  severity_category <- ""
  vent_management <- ""

  if (total_score < 3) {
    severity_category <- "Mild"
    vent_management <- "Volume-controlled or pressure support. PEEP <= 5 cmH2O."
  } else if (total_score <= 6) {
    severity_category <- "Moderate"
    vent_management <- "Conventional modes, inverse-ratio if needed. PEEP 5-10 cmH2O."
  } else {
    severity_category <- "Severe"
    vent_management <- "Conventional modes + advanced therapies (NO, HFJV, ILV, ECMO). PEEP > 10 cmH2O."
  }

  return(list(
    BLISS_Score = total_score,
    Severity_Category = severity_category,
    Ventilator_Management = vent_management,
    Inputs = list(
      PF_Ratio = pao2_fio2_ratio,
      CXR_Score = cxr_score,
      BPF_Present = bronchopleural_fistula == 1
    )
  ))
}

blood_volume_calculation <- function(sex, height_cm, weight_kg) {

  sex <- tolower(as.character(sex))
  height_cm <- as.numeric(height_cm)
  weight_kg <- as.numeric(weight_kg)

  if (any(is.na(c(height_cm, weight_kg)))) {
    stop("Error: Height and Weight must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  height_m <- height_cm / 100
  blood_volume_l <- 0

  if (sex == "male") {
    blood_volume_l <- (0.3669 * (height_m ^ 3)) + (0.03219 * weight_kg) + 0.6041
  } else {
    blood_volume_l <- (0.3561 * (height_m ^ 3)) + (0.03308 * weight_kg) + 0.1833
  }

  return(list(
    Estimated_Blood_Volume_L = round(blood_volume_l, 2),
    Formula = "Nadler Equation",
    Inputs = list(
      Sex = sex,
      Height_cm = height_cm,
      Weight_kg = weight_kg
    )
  ))
}

body_mass_index_bsa <- function(weight_kg, height_cm) {

  weight_kg <- as.numeric(weight_kg)
  height_cm <- as.numeric(height_cm)

  if (any(is.na(c(weight_kg, height_cm)))) {
    stop("Error: Weight and Height must be numeric.")
  }

  if (weight_kg <= 0 || height_cm <= 0) {
    stop("Error: Weight and Height must be positive numbers.")
  }

  height_m <- height_cm / 100

  bmi <- weight_kg / (height_m ^ 2)

  bmi_category <- ""
  if (bmi < 18.5) {
    bmi_category <- "Underweight"
  } else if (bmi >= 18.5 && bmi < 25) {
    bmi_category <- "Normal weight"
  } else if (bmi >= 25 && bmi < 30) {
    bmi_category <- "Overweight"
  } else if (bmi >= 30 && bmi < 35) {
    bmi_category <- "Obesity Class I"
  } else if (bmi >= 35 && bmi < 40) {
    bmi_category <- "Obesity Class II"
  } else {
    bmi_category <- "Obesity Class III"
  }

  bsa_mosteller <- sqrt((height_cm * weight_kg) / 3600)

  bsa_dubois <- 0.007184 * (weight_kg ^ 0.425) * (height_cm ^ 0.725)

  bsa_haycock <- 0.024265 * (weight_kg ^ 0.5378) * (height_cm ^ 0.3964)

  bsa_gehan <- 0.0235 * (weight_kg ^ 0.51456) * (height_cm ^ 0.42246)

  weight_grams <- weight_kg * 1000
  exponent_boyd <- 0.7285 - (0.0188 * log10(weight_grams))
  bsa_boyd <- 0.0003207 * (weight_grams ^ exponent_boyd) * (height_cm ^ 0.3)

  return(list(
    BMI_kg_m2 = round(bmi, 2),
    BMI_Category = bmi_category,
    BSA_Mosteller_m2 = round(bsa_mosteller, 2),
    BSA_DuBois_m2 = round(bsa_dubois, 2),
    BSA_Haycock_m2 = round(bsa_haycock, 2),
    BSA_Gehan_George_m2 = round(bsa_gehan, 2),
    BSA_Boyd_m2 = round(bsa_boyd, 2),
    Inputs = list(
      Weight_kg = weight_kg,
      Height_cm = height_cm
    )
  ))
}

bode_index_copd <- function(bmi, fev1_percent_predicted, mmrc_scale, six_minute_walk_distance_m) {

  bmi <- as.numeric(bmi)
  fev1_percent_predicted <- as.numeric(fev1_percent_predicted)
  mmrc_scale <- as.numeric(mmrc_scale)
  six_minute_walk_distance_m <- as.numeric(six_minute_walk_distance_m)

  if (any(is.na(c(bmi, fev1_percent_predicted, mmrc_scale, six_minute_walk_distance_m)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (mmrc_scale < 0 || mmrc_scale > 4) {
    stop("Error: mMRC scale must be between 0 and 4.")
  }

  score <- 0

  if (bmi > 21) {
    score <- score + 0
  } else {
    score <- score + 1
  }

  if (fev1_percent_predicted >= 65) {
    score <- score + 0
  } else if (fev1_percent_predicted >= 50) {
    score <- score + 1
  } else if (fev1_percent_predicted >= 36) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (mmrc_scale == 0 || mmrc_scale == 1) {
    score <- score + 0
  } else if (mmrc_scale == 2) {
    score <- score + 1
  } else if (mmrc_scale == 3) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (six_minute_walk_distance_m >= 350) {
    score <- score + 0
  } else if (six_minute_walk_distance_m >= 250) {
    score <- score + 1
  } else if (six_minute_walk_distance_m >= 150) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  survival_4yr <- ""

  if (score <= 2) {
    survival_4yr <- "Approx. 80%"
  } else if (score <= 4) {
    survival_4yr <- "Approx. 67%"
  } else if (score <= 6) {
    survival_4yr <- "Approx. 57%"
  } else {
    survival_4yr <- "Approx. 18%"
  }

  return(list(
    BODE_Index = score,
    Est_4_Year_Survival = survival_4yr,
    Inputs = list(
      BMI_le_21 = bmi <= 21,
      FEV1_Percent = fev1_percent_predicted,
      mMRC_Score = mmrc_scale,
      Walk_Distance_m = six_minute_walk_distance_m
    )
  ))
}

body_fluid_balance <- function(oral_input_ml = 0, iv_input_ml = 0, enteral_input_ml = 0, other_input_ml = 0, urine_output_ml = 0, stool_output_ml = 0, emesis_ng_output_ml = 0, drain_output_ml = 0, insensible_loss_ml = 0) {

  oral_input_ml <- as.numeric(oral_input_ml)
  iv_input_ml <- as.numeric(iv_input_ml)
  enteral_input_ml <- as.numeric(enteral_input_ml)
  other_input_ml <- as.numeric(other_input_ml)
  urine_output_ml <- as.numeric(urine_output_ml)
  stool_output_ml <- as.numeric(stool_output_ml)
  emesis_ng_output_ml <- as.numeric(emesis_ng_output_ml)
  drain_output_ml <- as.numeric(drain_output_ml)
  insensible_loss_ml <- as.numeric(insensible_loss_ml)

  if (any(is.na(c(oral_input_ml, iv_input_ml, enteral_input_ml, other_input_ml, urine_output_ml, stool_output_ml, emesis_ng_output_ml, drain_output_ml, insensible_loss_ml)))) {
    stop("Error: All inputs must be numeric.")
  }

  total_input <- oral_input_ml + iv_input_ml + enteral_input_ml + other_input_ml
  total_output <- urine_output_ml + stool_output_ml + emesis_ng_output_ml + drain_output_ml + insensible_loss_ml

  net_balance <- total_input - total_output

  balance_status <- ""
  if (net_balance > 0) {
    balance_status <- "Positive Balance (Net Gain)"
  } else if (net_balance < 0) {
    balance_status <- "Negative Balance (Net Loss)"
  } else {
    balance_status <- "Neutral Balance"
  }

  return(list(
    Net_Fluid_Balance_mL = net_balance,
    Status = balance_status,
    Summary = list(
      Total_Input_mL = total_input,
      Total_Output_mL = total_output
    ),
    Detailed_Inputs = list(
      Oral = oral_input_ml,
      IV = iv_input_ml,
      Enteral = enteral_input_ml,
      Other = other_input_ml
    ),
    Detailed_Outputs = list(
      Urine = urine_output_ml,
      Stool = stool_output_ml,
      Emesis_NG = emesis_ng_output_ml,
      Drains = drain_output_ml,
      Insensible = insensible_loss_ml
    )
  ))
}

body_roundness_index <- function(waist_cm, height_cm) {

  waist_cm <- as.numeric(waist_cm)
  height_cm <- as.numeric(height_cm)

  if (any(is.na(c(waist_cm, height_cm)))) {
    stop("Error: Waist circumference and Height must be numeric.")
  }

  if (waist_cm <= 0 || height_cm <= 0) {
    stop("Error: Measurements must be positive.")
  }

  waist_radius <- waist_cm / (2 * pi)
  height_radius <- 0.5 * height_cm

  if (waist_radius >= height_radius) {
    stop("Error: Waist circumference is geometrically incompatible with height for this formula (approaches spherical limit).")
  }

  eccentricity_squared <- 1 - ((waist_radius / height_radius) ^ 2)

  eccentricity <- sqrt(eccentricity_squared)

  bri <- 364.2 - (365.5 * eccentricity)

  return(list(
    BRI_Score = round(bri, 2),
    Inputs = list(
      Waist_cm = waist_cm,
      Height_cm = height_cm
    )
  ))
}

bova_pe_score <- function(sbp, heart_rate, troponin_elevated, rv_dysfunction) {

  sbp <- as.numeric(sbp)
  heart_rate <- as.numeric(heart_rate)
  troponin_elevated <- as.numeric(troponin_elevated)
  rv_dysfunction <- as.numeric(rv_dysfunction)

  if (any(is.na(c(sbp, heart_rate, troponin_elevated, rv_dysfunction)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (sbp >= 90 && sbp <= 100) {
    score <- score + 2
  }

  if (heart_rate >= 110) {
    score <- score + 1
  }

  if (troponin_elevated == 1) {
    score <- score + 2
  }

  if (rv_dysfunction == 1) {
    score <- score + 2
  }

  risk_stage <- ""
  complication_risk <- ""

  if (score <= 2) {
    risk_stage <- "Stage I (Low Risk)"
    complication_risk <- "4.2%"
  } else if (score <= 4) {
    risk_stage <- "Stage II (Intermediate Risk)"
    complication_risk <- "17.9%"
  } else {
    risk_stage <- "Stage III (High Risk)"
    complication_risk <- "41.9%"
  }

  return(list(
    Bova_Score = score,
    Risk_Stage = risk_stage,
    PE_Related_Complication_Risk_30Day = complication_risk,
    Inputs = list(
      SBP_90_100 = (sbp >= 90 && sbp <= 100),
      HR_ge_110 = heart_rate >= 110,
      Troponin_Pos = troponin_elevated == 1,
      RV_Dysfunction = rv_dysfunction == 1
    )
  ))
}

braden_score <- function(sensory_perception, moisture, activity, mobility, nutrition, friction_shear) {

  sensory_perception <- as.numeric(sensory_perception)
  moisture <- as.numeric(moisture)
  activity <- as.numeric(activity)
  mobility <- as.numeric(mobility)
  nutrition <- as.numeric(nutrition)
  friction_shear <- as.numeric(friction_shear)

  if (any(is.na(c(sensory_perception, moisture, activity, mobility, nutrition, friction_shear)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(sensory_perception, moisture, activity, mobility, nutrition) < 1) ||
      any(c(sensory_perception, moisture, activity, mobility, nutrition) > 4)) {
    stop("Error: Sensory, Moisture, Activity, Mobility, and Nutrition must be between 1 and 4.")
  }

  if (friction_shear < 1 || friction_shear > 3) {
    stop("Error: Friction & Shear must be between 1 and 3.")
  }

  total_score <- sensory_perception + moisture + activity + mobility + nutrition + friction_shear

  risk_category <- ""
  if (total_score <= 9) {
    risk_category <- "Severe Risk"
  } else if (total_score <= 12) {
    risk_category <- "High Risk"
  } else if (total_score <= 14) {
    risk_category <- "Moderate Risk"
  } else if (total_score <= 18) {
    risk_category <- "Mild Risk"
  } else {
    risk_category <- "No Risk"
  }

  return(list(
    Braden_Score = total_score,
    Risk_Category = risk_category,
    Inputs = list(
      Sensory = sensory_perception,
      Moisture = moisture,
      Activity = activity,
      Mobility = mobility,
      Nutrition = nutrition,
      Friction_Shear = friction_shear
    )
  ))
}

brain_metastasis_velocity <- function(new_mets_count, time_interval_years) {

  new_mets_count <- as.numeric(new_mets_count)
  time_interval_years <- as.numeric(time_interval_years)

  if (any(is.na(c(new_mets_count, time_interval_years)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (new_mets_count < 0 || time_interval_years <= 0) {
    stop("Error: Number of metastases must be non-negative and time interval must be positive.")
  }

  bmv_score <- new_mets_count / time_interval_years

  risk_group <- ""
  median_survival <- ""

  if (bmv_score < 4) {
    risk_group <- "Low Risk"
    median_survival <- "12.4 months"
  } else if (bmv_score >= 4 && bmv_score <= 13) {
    risk_group <- "Intermediate Risk"
    median_survival <- "8.2 months"
  } else {
    risk_group <- "High Risk"
    median_survival <- "4.3 months"
  }

  return(list(
    BMV_Score = round(bmv_score, 2),
    Risk_Class = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      New_Metastases = new_mets_count,
      Time_Interval_Years = time_interval_years
    )
  ))
}

brain_metastasis_velocity <- function(new_mets_count, time_interval_years) {

  new_mets_count <- as.numeric(new_mets_count)
  time_interval_years <- as.numeric(time_interval_years)

  if (any(is.na(c(new_mets_count, time_interval_years)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (new_mets_count < 0 || time_interval_years <= 0) {
    stop("Error: Number of metastases must be non-negative and time interval must be positive.")
  }

  bmv_score <- new_mets_count / time_interval_years

  risk_group <- ""
  median_survival <- ""

  if (bmv_score < 4) {
    risk_group <- "Low Risk"
    median_survival <- "12.4 months"
  } else if (bmv_score >= 4 && bmv_score <= 13) {
    risk_group <- "Intermediate Risk"
    median_survival <- "8.2 months"
  } else {
    risk_group <- "High Risk"
    median_survival <- "4.3 months"
  }

  return(list(
    BMV_Score = round(bmv_score, 2),
    Risk_Class = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      New_Metastases = new_mets_count,
      Time_Interval_Years = time_interval_years
    )
  ))
}

brescia_covid_severity <- function(wheezing_present, rr_gt_22_or_desat, worsening_cxr_or_unstable) {

  wheezing_present <- as.numeric(wheezing_present)
  rr_gt_22_or_desat <- as.numeric(rr_gt_22_or_desat)
  worsening_cxr_or_unstable <- as.numeric(worsening_cxr_or_unstable)

  if (any(is.na(c(wheezing_present, rr_gt_22_or_desat, worsening_cxr_or_unstable)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0
  management <- ""

  if (wheezing_present == 1) {
    score <- 0
    management <- "Manage as Asthma/COPD exacerbation (not typical COVID-19 progression). Re-evaluate."
  } else {
    if (rr_gt_22_or_desat == 0) {
      score <- 0
      management <- "Observation. Monitor SpO2/RR. If stable, consider discharge/home monitoring."
    } else {
      if (worsening_cxr_or_unstable == 0) {
        score <- 1
        management <- "Supplemental O2. Monitor closely. If stable, continue O2. If worsening, escalate."
      } else {
        score <- 2
        management <- "Consider NIV/CPAP or Intubation. Consult ICU. Prone positioning."
      }
    }
  }

  return(list(
    BCRSS_Severity_Level = score,
    Management_Recommendation = management,
    Inputs = list(
      Wheezing = wheezing_present == 1,
      Respiratory_Distress = rr_gt_22_or_desat == 1,
      Worsening_Clinical_Pic = worsening_cxr_or_unstable == 1
    )
  ))
}

brief_addiction_monitor <- function(alcohol_days, drug_days, marijuana_days,
                                    risk_health, risk_sleep, risk_anxiety, risk_depression, risk_cravings, risk_situations,
                                    prot_confidence, prot_support, prot_spiritual, prot_work, prot_income, prot_family, prot_self_help_days, prot_commitment) {

  alcohol_days <- as.numeric(alcohol_days)
  drug_days <- as.numeric(drug_days)
  marijuana_days <- as.numeric(marijuana_days)

  risk_health <- as.numeric(risk_health)
  risk_sleep <- as.numeric(risk_sleep)
  risk_anxiety <- as.numeric(risk_anxiety)
  risk_depression <- as.numeric(risk_depression)
  risk_cravings <- as.numeric(risk_cravings)
  risk_situations <- as.numeric(risk_situations)

  prot_confidence <- as.numeric(prot_confidence)
  prot_support <- as.numeric(prot_support)
  prot_spiritual <- as.numeric(prot_spiritual)
  prot_work <- as.numeric(prot_work)
  prot_income <- as.numeric(prot_income)
  prot_family <- as.numeric(prot_family)
  prot_self_help_days <- as.numeric(prot_self_help_days)
  prot_commitment <- as.numeric(prot_commitment)

  inputs <- c(alcohol_days, drug_days, marijuana_days, risk_health, risk_sleep, risk_anxiety, risk_depression, risk_cravings, risk_situations, prot_confidence, prot_support, prot_spiritual, prot_work, prot_income, prot_family, prot_self_help_days, prot_commitment)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  likert_inputs <- c(risk_health, risk_sleep, risk_anxiety, risk_depression, risk_cravings, risk_situations, prot_confidence, prot_support, prot_spiritual, prot_work, prot_income, prot_family, prot_commitment)
  if (any(likert_inputs < 0) || any(likert_inputs > 4)) {
    stop("Error: Likert scale items must be between 0 and 4.")
  }

  day_inputs <- c(alcohol_days, drug_days, marijuana_days, prot_self_help_days)
  if (any(day_inputs < 0) || any(day_inputs > 30)) {
    stop("Error: Day count items must be between 0 and 30.")
  }

  use_score <- alcohol_days + drug_days + marijuana_days

  risk_score <- risk_health + risk_sleep + risk_anxiety + risk_depression + risk_cravings + risk_situations

  prot_likert_sum <- prot_confidence + prot_support + prot_spiritual + prot_work + prot_income + prot_family + prot_commitment

  return(list(
    Use_Total_Days = use_score,
    Risk_Score_Total = risk_score,
    Protective_Likert_Sum = prot_likert_sum,
    Self_Help_Days = prot_self_help_days,
    Interpretation = list(
      Use = "Total days of substance use in past 30 days.",
      Risk = "Sum of 6 risk items (Range 0-24). Higher score indicates higher risk of relapse.",
      Protection = "Sum of 7 protective items (Range 0-28) plus self-help attendance. Higher score indicates stronger recovery factors."
    )
  ))
}

brief_alcohol_withdrawal_scale <- function(tremor, sweating, agitation, orientation, hallucinations) {

  tremor <- as.numeric(tremor)
  sweating <- as.numeric(sweating)
  agitation <- as.numeric(agitation)
  orientation <- as.numeric(orientation)
  hallucinations <- as.numeric(hallucinations)

  if (any(is.na(c(tremor, sweating, agitation, orientation, hallucinations)))) {
    stop("Error: All inputs must be numeric integers (0-3).")
  }

  if (any(c(tremor, sweating, agitation, orientation, hallucinations) < 0) ||
      any(c(tremor, sweating, agitation, orientation, hallucinations) > 3)) {
    stop("Error: All item scores must be between 0 and 3.")
  }

  total_score <- tremor + sweating + agitation + orientation + hallucinations

  severity <- ""
  management <- ""
  reassessment <- ""

  if (total_score < 3) {
    severity <- "Mild Withdrawal"
    management <- "Supportive care. No medication typically required (unless anxiety persists, then consider Diazepam 5mg)."
    reassessment <- "Reassess every 4 hours."
  } else if (total_score <= 5) {
    severity <- "Moderate Withdrawal"
    management <- "Administer Diazepam 10 mg PO (or Lorazepam 2 mg IV/PO) x1."
    reassessment <- "Reassess every 4 hours."
  } else if (total_score <= 8) {
    severity <- "Severe Withdrawal"
    management <- "Administer Diazepam 20 mg PO (or Lorazepam 4 mg IV/PO) x1."
    reassessment <- "Reassess every 1-2 hours. Notify physician if score > 5 after 6 hours."
  } else {
    severity <- "Very Severe Withdrawal"
    management <- "Administer Diazepam 20 mg PO (or Lorazepam 4 mg IV/PO) immediately. Notify physician."
    reassessment <- "Reassess every 1 hour. Consider ICU transfer."
  }

  return(list(
    BAWS_Score = total_score,
    Severity = severity,
    Management_Protocol = management,
    Next_Step = reassessment,
    Inputs = list(
      Tremor = tremor,
      Sweating = sweating,
      Agitation = agitation,
      Orientation = orientation,
      Hallucinations = hallucinations
    )
  ))
}

brue_risk_stratification <- function(age_days, gestational_age_weeks, duration_less_than_1_min, cpr_performed, recurring_event, concerning_history_or_exam) {

  age_days <- as.numeric(age_days)
  gestational_age_weeks <- as.numeric(gestational_age_weeks)
  duration_less_than_1_min <- as.numeric(duration_less_than_1_min)
  cpr_performed <- as.numeric(cpr_performed)
  recurring_event <- as.numeric(recurring_event)
  concerning_history_or_exam <- as.numeric(concerning_history_or_exam)

  if (any(is.na(c(age_days, gestational_age_weeks, duration_less_than_1_min, cpr_performed, recurring_event, concerning_history_or_exam)))) {
    stop("Error: All inputs must be numeric.")
  }

  is_low_risk <- TRUE
  reasons <- c()

  if (age_days <= 60) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Age <= 60 days")
  }

  if (gestational_age_weeks < 32) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Gestational Age < 32 weeks")
  }

  post_conceptional_age_weeks <- gestational_age_weeks + (age_days / 7)
  if (post_conceptional_age_weeks < 45) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Post-conceptional age < 45 weeks")
  }

  if (duration_less_than_1_min == 0) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Duration >= 1 minute")
  }

  if (cpr_performed == 1) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "CPR performed by trained provider")
  }

  if (recurring_event == 1) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Recurrent event (not first time)")
  }

  if (concerning_history_or_exam == 1) {
    is_low_risk <- FALSE
    reasons <- c(reasons, "Concerning history or physical exam findings")
  }

  risk_category <- ""
  recommendation <- ""

  if (is_low_risk) {
    risk_category <- "Low Risk BRUE"
    recommendation <- "May discharge with education and follow-up. Viral testing, EKG, and brief observation (1-4h) may be considered but are not mandatory."
  } else {
    risk_category <- "Not Low Risk (High Risk) BRUE"
    recommendation <- "Further investigation, admission, and monitoring indicated based on specific risk factors."
  }

  return(list(
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Criteria_Not_Met = if (length(reasons) > 0) reasons else "None",
    Inputs = list(
      Age_Days = age_days,
      Gestational_Age = gestational_age_weeks,
      Corrected_Age_Weeks = round(post_conceptional_age_weeks, 1),
      Duration_lt_1min = duration_less_than_1_min == 1,
      CPR = cpr_performed == 1,
      Recurring = recurring_event == 1,
      Concerning_Findings = concerning_history_or_exam == 1
    )
  ))
}

brue_2_0_criteria <- function(age_days, gestational_age_weeks, duration_less_than_1_min, cpr_performed, recurring_event, concerning_history, concerning_exam) {

  age_days <- as.numeric(age_days)
  gestational_age_weeks <- as.numeric(gestational_age_weeks)
  duration_less_than_1_min <- as.numeric(duration_less_than_1_min)
  cpr_performed <- as.numeric(cpr_performed)
  recurring_event <- as.numeric(recurring_event)
  concerning_history <- as.numeric(concerning_history)
  concerning_exam <- as.numeric(concerning_exam)

  if (any(is.na(c(age_days, gestational_age_weeks, duration_less_than_1_min, cpr_performed, recurring_event, concerning_history, concerning_exam)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (age_days >= 365) {
    return(list(
      Risk_Category = "Not BRUE",
      Reason = "Patient age is >= 1 year. BRUE criteria apply only to infants < 1 year."
    ))
  }

  is_low_risk <- TRUE
  risk_factors <- c()

  if (age_days <= 60) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Age <= 60 days")
  }

  if (gestational_age_weeks < 32) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Gestational Age < 32 weeks")
  }

  post_conceptional_age <- gestational_age_weeks + (age_days / 7)
  if (post_conceptional_age < 45) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Post-conceptional Age < 45 weeks")
  }

  if (duration_less_than_1_min == 0) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Duration >= 1 minute")
  }

  if (cpr_performed == 1) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "CPR performed by trained provider")
  }

  if (recurring_event == 1) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Recurring event (not first episode)")
  }

  if (concerning_history == 1) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Concerning features in history")
  }

  if (concerning_exam == 1) {
    is_low_risk <- FALSE
    risk_factors <- c(risk_factors, "Concerning physical examination")
  }

  classification <- ""
  management <- ""

  if (is_low_risk) {
    classification <- "Lower-Risk BRUE"
    management <- "Serious underlying disorder unlikely. Reassurance and education (CPR, safe sleep). Observation (1-4h) may be considered. Routine testing/admission not recommended."
  } else {
    classification <- "Higher-Risk BRUE"
    management <- "Increased risk of serious underlying diagnosis or recurrence. Further investigation (ECG, viral testing, monitoring) and/or admission recommended based on specific risk factors."
  }

  return(list(
    Classification = classification,
    Management_Guidance = management,
    Risk_Factors_Present = if (length(risk_factors) > 0) risk_factors else "None",
    Inputs = list(
      Age_Days = age_days,
      GA_Weeks = gestational_age_weeks,
      PCA_Weeks = round(post_conceptional_age, 1),
      Duration_Short = duration_less_than_1_min == 1,
      CPR = cpr_performed == 1,
      Recurring = recurring_event == 1,
      History_Concern = concerning_history == 1,
      Exam_Concern = concerning_exam == 1
    )
  ))
}

bristol_stool_scale <- function(stool_type) {

  stool_type <- as.numeric(stool_type)

  if (is.na(stool_type) || !stool_type %in% 1:7) {
    stop("Error: Stool type must be an integer between 1 and 7.")
  }

  description <- ""
  interpretation <- ""
  transit_time <- ""

  if (stool_type == 1) {
    description <- "Separate hard lumps, like nuts (hard to pass)."
    interpretation <- "Severe Constipation"
    transit_time <- "Slowest (~100 hours)"
  } else if (stool_type == 2) {
    description <- "Sausage-shaped but lumpy."
    interpretation <- "Mild Constipation"
    transit_time <- "Slow"
  } else if (stool_type == 3) {
    description <- "Like a sausage but with cracks on its surface."
    interpretation <- "Normal"
    transit_time <- "Normal"
  } else if (stool_type == 4) {
    description <- "Like a sausage or snake, smooth and soft."
    interpretation <- "Normal (Ideal)"
    transit_time <- "Normal"
  } else if (stool_type == 5) {
    description <- "Soft blobs with clear-cut edges (passed easily)."
    interpretation <- "Lacking Fiber / Normal"
    transit_time <- "Normal"
  } else if (stool_type == 6) {
    description <- "Fluffy pieces with ragged edges, a mushy stool."
    interpretation <- "Mild Diarrhea"
    transit_time <- "Fast"
  } else {
    description <- "Watery, no solid pieces. Entirely liquid."
    interpretation <- "Severe Diarrhea"
    transit_time <- "Fastest (~10 hours)"
  }

  return(list(
    Type = stool_type,
    Description = description,
    Interpretation = interpretation,
    Est_Transit_Time = transit_time
  ))
}

brugada_criteria_vt <- function(absent_rs_precordial, rs_interval_gt_100, av_dissociation, morphology_criteria_vt) {

  absent_rs_precordial <- as.numeric(absent_rs_precordial)
  rs_interval_gt_100 <- as.numeric(rs_interval_gt_100)
  av_dissociation <- as.numeric(av_dissociation)
  morphology_criteria_vt <- as.numeric(morphology_criteria_vt)

  if (any(is.na(c(absent_rs_precordial, rs_interval_gt_100, av_dissociation, morphology_criteria_vt)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  diagnosis <- ""
  step_positive <- 0

  if (absent_rs_precordial == 1) {
    diagnosis <- "Ventricular Tachycardia (VT)"
    step_positive <- 1
  } else if (rs_interval_gt_100 == 1) {
    diagnosis <- "Ventricular Tachycardia (VT)"
    step_positive <- 2
  } else if (av_dissociation == 1) {
    diagnosis <- "Ventricular Tachycardia (VT)"
    step_positive <- 3
  } else if (morphology_criteria_vt == 1) {
    diagnosis <- "Ventricular Tachycardia (VT)"
    step_positive <- 4
  } else {
    diagnosis <- "SVT with Aberrancy"
    step_positive <- 0
  }

  return(list(
    Diagnosis = diagnosis,
    Step_Positive = step_positive,
    Inputs = list(
      Step1_Absent_RS = absent_rs_precordial == 1,
      Step2_RS_gt_100 = rs_interval_gt_100 == 1,
      Step3_AV_Dissoc = av_dissociation == 1,
      Step4_Morphology = morphology_criteria_vt == 1
    )
  ))
}

bun_creatinine_ratio <- function(bun, creatinine, bun_units = "mg/dL", creatinine_units = "mg/dL") {

  bun <- as.numeric(bun)
  creatinine <- as.numeric(creatinine)

  if (any(is.na(c(bun, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (creatinine == 0) {
    stop("Error: Creatinine cannot be zero (division by zero).")
  }

  bun_mg_dl <- bun
  if (bun_units == "mmol/L") {
    bun_mg_dl <- bun * 2.8
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  } else if (creatinine_units == "mmol/L") {
    creat_mg_dl <- creatinine / 0.0884
  }

  ratio <- bun_mg_dl / creat_mg_dl

  interpretation <- ""
  possible_causes <- ""

  if (ratio > 20) {
    interpretation <- "Ratio > 20:1 (Prerenal Etiology)"
    possible_causes <- "Dehydration, Congestive Heart Failure, GI Bleed, High Protein Diet, Steroids."
  } else if (ratio >= 10 && ratio <= 20) {
    interpretation <- "Ratio 10-20:1 (Normal or Postrenal)"
    possible_causes <- "Normal range, or Postrenal obstruction (ratio may vary)."
  } else {
    interpretation <- "Ratio < 10:1 (Intrarenal Etiology)"
    possible_causes <- "Acute Tubular Necrosis (ATN), Renal damage, Starvation, Liver Disease (reduced urea synthesis)."
  }

  return(list(
    BUN_Creatinine_Ratio = round(ratio, 1),
    Interpretation = interpretation,
    Possible_Causes = possible_causes,
    Standardized_Values_mg_dL = list(
      BUN = round(bun_mg_dl, 2),
      Creatinine = round(creat_mg_dl, 2)
    )
  ))
}

burch_wartofsky_score <- function(temp_c, cns_scale, gi_scale, heart_rate, chf_scale, afib_present, precipitating_event) {

  temp_c <- as.numeric(temp_c)
  cns_scale <- as.numeric(cns_scale)
  gi_scale <- as.numeric(gi_scale)
  heart_rate <- as.numeric(heart_rate)
  chf_scale <- as.numeric(chf_scale)
  afib_present <- as.numeric(afib_present)
  precipitating_event <- as.numeric(precipitating_event)

  if (any(is.na(c(temp_c, cns_scale, gi_scale, heart_rate, chf_scale, afib_present, precipitating_event)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (temp_c >= 37.2 && temp_c < 37.8) {
    score <- score + 5
  } else if (temp_c >= 37.8 && temp_c < 38.3) {
    score <- score + 10
  } else if (temp_c >= 38.3 && temp_c < 38.9) {
    score <- score + 15
  } else if (temp_c >= 38.9 && temp_c < 39.4) {
    score <- score + 20
  } else if (temp_c >= 39.4 && temp_c < 40.0) {
    score <- score + 25
  } else if (temp_c >= 40.0) {
    score <- score + 30
  }

  if (cns_scale == 1) {
    score <- score + 10
  } else if (cns_scale == 2) {
    score <- score + 20
  } else if (cns_scale == 3) {
    score <- score + 30
  }

  if (gi_scale == 1) {
    score <- score + 10
  } else if (gi_scale == 2) {
    score <- score + 20
  }

  if (heart_rate >= 99 && heart_rate < 110) {
    score <- score + 5
  } else if (heart_rate >= 110 && heart_rate < 120) {
    score <- score + 10
  } else if (heart_rate >= 120 && heart_rate < 130) {
    score <- score + 15
  } else if (heart_rate >= 130 && heart_rate < 140) {
    score <- score + 20
  } else if (heart_rate >= 140) {
    score <- score + 25
  }

  if (chf_scale == 1) {
    score <- score + 5
  } else if (chf_scale == 2) {
    score <- score + 10
  } else if (chf_scale == 3) {
    score <- score + 15
  }

  if (afib_present == 1) {
    score <- score + 10
  }

  if (precipitating_event == 1) {
    score <- score + 10
  }

  interpretation <- ""
  if (score < 25) {
    interpretation <- "Thyroid Storm Unlikely"
  } else if (score >= 25 && score <= 44) {
    interpretation <- "Impending Storm (Suggestive)"
  } else {
    interpretation <- "Thyroid Storm Highly Suggestive"
  }

  return(list(
    BWPS_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Temp_C = temp_c,
      CNS_Points = cns_scale * 10,
      GI_Points = gi_scale * 10,
      HR = heart_rate,
      CHF_Points = chf_scale * 5,
      Afib = afib_present == 1,
      Precipitant = precipitating_event == 1
    )
  ))
}

bush_francis_catatonia_scale <- function(excitement, immobility, mutism, staring, posturing, grimacing, echophenomena, stereotypy, mannerisms, verbigeration, rigidity, negativism, waxy_flexibility, withdrawal, impulsivity, automatic_obedience, mitgehen, gegenhalten, ambitendency, grasp_reflex, perseverance, combativeness, autonomic_abnormality) {

  inputs <- c(excitement, immobility, mutism, staring, posturing, grimacing, echophenomena, stereotypy, mannerisms, verbigeration, rigidity, negativism, waxy_flexibility, withdrawal, impulsivity, automatic_obedience, mitgehen, gegenhalten, ambitendency, grasp_reflex, perseverance, combativeness, autonomic_abnormality)

  if (any(is.na(inputs))) {
    stop("Error: All 23 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 3)) {
    stop("Error: All item scores must be between 0 and 3.")
  }

  screening_items <- c(excitement, immobility, mutism, staring, posturing, grimacing, echophenomena, stereotypy, mannerisms, verbigeration, rigidity, negativism, waxy_flexibility, withdrawal)

  screening_positive_count <- sum(screening_items > 0)

  total_score <- sum(inputs)

  screening_result <- ""
  if (screening_positive_count >= 2) {
    screening_result <- "Positive Screen (Suggests Catatonia)"
  } else {
    screening_result <- "Negative Screen"
  }

  return(list(
    BFCRS_Total_Score = total_score,
    Screening_Result = screening_result,
    Screening_Items_Present = screening_positive_count,
    Inputs = list(
      Excitement = excitement,
      Immobility = immobility,
      Mutism = mutism,
      Staring = staring,
      Posturing = posturing,
      Grimacing = grimacing,
      Echophenomena = echophenomena,
      Stereotypy = stereotypy,
      Mannerisms = mannerisms,
      Verbigeration = verbigeration,
      Rigidity = rigidity,
      Negativism = negativism,
      Waxy_Flexibility = waxy_flexibility,
      Withdrawal = withdrawal,
      Impulsivity = impulsivity,
      Automatic_Obedience = automatic_obedience,
      Mitgehen = mitgehen,
      Gegenhalten = gegenhalten,
      Ambitendency = ambitendency,
      Grasp_Reflex = grasp_reflex,
      Perseverance = perseverance,
      Combativeness = combativeness,
      Autonomic_Abnormality = autonomic_abnormality
    )
  ))
}

bwh_egg_freezing_tool <- function(age, num_mature_eggs) {

  age <- as.numeric(age)
  num_mature_eggs <- as.numeric(num_mature_eggs)

  if (any(is.na(c(age, num_mature_eggs)))) {
    stop("Error: Age and Number of Mature Eggs must be numeric.")
  }

  if (age < 24 || age > 44) {
    return(list(
      Error = "The BWH Egg Freezing Counseling Tool is validated for women aged 24-44 years."
    ))
  }

  p_efficiency <- 0

  if (age <= 35) {
    p_efficiency <- 0.111
  } else if (age == 36) {
    p_efficiency <- 0.086
  } else if (age == 37) {
    p_efficiency <- 0.067
  } else if (age == 38) {
    p_efficiency <- 0.056
  } else if (age == 39) {
    p_efficiency <- 0.045
  } else if (age == 40) {
    p_efficiency <- 0.035
  } else if (age == 41) {
    p_efficiency <- 0.028
  } else if (age == 42) {
    p_efficiency <- 0.020
  } else if (age == 43) {
    p_efficiency <- 0.012
  } else {
    p_efficiency <- 0.006
  }

  n <- num_mature_eggs
  p <- p_efficiency


p_x_0 <- choose(n, 0) * (p^0) * ((1 - p)^(n - 0))


p_x_1 <- choose(n, 1) * (p^1) * ((1 - p)^(n - 1))


p_x_2 <- choose(n, 2) * (p^2) * ((1 - p)^(n - 2))

  prob_1_plus <- (1 - p_x_0) * 100

  prob_2_plus <- (1 - p_x_0 - p_x_1) * 100

  prob_3_plus <- (1 - p_x_0 - p_x_1 - p_x_2) * 100

  return(list(
    Age = age,
    Mature_Eggs = num_mature_eggs,
    Likelihood_Live_Birth = list(
      At_Least_One_Child = paste0(round(prob_1_plus, 1), "%"),
      At_Least_Two_Children = paste0(round(prob_2_plus, 1), "%"),
      At_Least_Three_Children = paste0(round(prob_3_plus, 1), "%")
    ),
    Model_Efficiency_Per_Egg = paste0(round(p_efficiency * 100, 1), "%")
  ))
}

c_peptide_glucose_ratio <- function(c_peptide, glucose, c_peptide_units = "ng/mL", glucose_units = "mg/dL") {

  c_peptide <- as.numeric(c_peptide)
  glucose <- as.numeric(glucose)

  if (any(is.na(c(c_peptide, glucose)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (glucose <= 0) {
    stop("Error: Glucose must be greater than 0.")
  }

  c_pep_ng_ml <- c_peptide
  if (c_peptide_units == "nmol/L") {
    c_pep_ng_ml <- c_peptide * 3.020
  } else if (c_peptide_units == "pmol/L") {
    c_pep_ng_ml <- c_peptide * 0.003020
  }

  gluc_mg_dl <- glucose
  if (glucose_units == "mmol/L") {
    gluc_mg_dl <- glucose * 18.018
  }

  cpgr <- (c_pep_ng_ml / gluc_mg_dl) * 100

  interpretation <- ""
  if (cpgr >= 1.7) {
    interpretation <- "Ratio >= 1.7 suggests significant endogenous insulin production (e.g., T2DM or MODY rather than T1DM)."
  } else {
    interpretation <- "Ratio < 1.7 suggests insulin deficiency consistent with Type 1 Diabetes."
  }

  return(list(
    CPGR_Score = round(cpgr, 2),
    Interpretation = interpretation,
    Inputs = list(
      C_Peptide_ng_mL = round(c_pep_ng_ml, 3),
      Glucose_mg_dL = round(gluc_mg_dl, 1)
    )
  ))
}

cage_alcohol_screen <- function(cut_down, annoyed, guilty, eye_opener) {

  cut_down <- as.numeric(cut_down)
  annoyed <- as.numeric(annoyed)
  guilty <- as.numeric(guilty)
  eye_opener <- as.numeric(eye_opener)

  if (any(is.na(c(cut_down, annoyed, guilty, eye_opener)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  total_score <- cut_down + annoyed + guilty + eye_opener

  result <- ""
  interpretation <- ""

  if (total_score >= 2) {
    result <- "Positive Screen"
    interpretation <- "Clinically significant. High suspicion for alcohol dependence."
  } else {
    result <- "Negative Screen"
    interpretation <- "Low risk for alcohol dependence."
  }

  return(list(
    CAGE_Score = total_score,
    Screen_Result = result,
    Interpretation = interpretation,
    Inputs = list(
      Cut_Down = cut_down == 1,
      Annoyed = annoyed == 1,
      Guilty = guilty == 1,
      Eye_Opener = eye_opener == 1
    )
  ))
}

cahp_cardiac_arrest_score <- function(age, location_home, initial_rhythm_shockable, no_flow_minutes, low_flow_minutes, epinephrine_dose_mg, arterial_ph) {

  age <- as.numeric(age)
  location_home <- as.numeric(location_home)
  initial_rhythm_shockable <- as.numeric(initial_rhythm_shockable)
  no_flow_minutes <- as.numeric(no_flow_minutes)
  low_flow_minutes <- as.numeric(low_flow_minutes)
  epinephrine_dose_mg <- as.numeric(epinephrine_dose_mg)
  arterial_ph <- as.numeric(arterial_ph)

  if (any(is.na(c(age, location_home, initial_rhythm_shockable, no_flow_minutes, low_flow_minutes, epinephrine_dose_mg, arterial_ph)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age < 45) score <- score + 0
  else if (age >= 45 && age <= 54) score <- score + 13
  else if (age >= 55 && age <= 64) score <- score + 27
  else if (age >= 65 && age <= 74) score <- score + 39
  else if (age >= 75 && age <= 84) score <- score + 53
  else score <- score + 66

  if (location_home == 1) score <- score + 16

  if (initial_rhythm_shockable == 0) score <- score + 27

  if (no_flow_minutes == 0) score <- score + 0
  else if (no_flow_minutes >= 1 && no_flow_minutes <= 3) score <- score + 15
  else score <- score + 29

  if (low_flow_minutes < 15) score <- score + 0
  else if (low_flow_minutes >= 15 && low_flow_minutes <= 29) score <- score + 16
  else if (low_flow_minutes >= 30 && low_flow_minutes <= 59) score <- score + 32
  else score <- score + 48

  if (epinephrine_dose_mg == 0) score <- score + 0
  else if (epinephrine_dose_mg >= 1 && epinephrine_dose_mg <= 2) score <- score + 16
  else if (epinephrine_dose_mg >= 3 && epinephrine_dose_mg <= 4) score <- score + 32
  else score <- score + 48

  if (arterial_ph > 7.30) score <- score + 0
  else if (arterial_ph >= 7.15 && arterial_ph <= 7.30) score <- score + 12
  else score <- score + 25

  risk_group <- ""
  poor_outcome_prob <- ""

  if (score < 150) {
    risk_group <- "Low Risk"
    poor_outcome_prob <- "~39% (High chance of good neurological outcome)"
  } else if (score >= 150 && score <= 200) {
    risk_group <- "Intermediate Risk"
    poor_outcome_prob <- "~81%"
  } else {
    risk_group <- "High Risk"
    poor_outcome_prob <- "100% (Poor neurological outcome CPC 3-5 highly likely)"
  }

  return(list(
    CAHP_Score = score,
    Risk_Group = risk_group,
    Prob_Poor_Neuro_Outcome = poor_outcome_prob,
    Inputs = list(
      Age = age,
      Location_Home = location_home == 1,
      Rhythm_Shockable = initial_rhythm_shockable == 1,
      No_Flow_Min = no_flow_minutes,
      Low_Flow_Min = low_flow_minutes,
      Epi_Dose_mg = epinephrine_dose_mg,
      pH = arterial_ph
    )
  ))
}

calcium_correction_hypoalbuminemia <- function(calcium, albumin, calcium_units = "mg/dL", albumin_units = "g/dL") {

  calcium <- as.numeric(calcium)
  albumin <- as.numeric(albumin)

  if (any(is.na(c(calcium, albumin)))) {
    stop("Error: Calcium and Albumin inputs must be numeric.")
  }

  calcium_mg_dl <- calcium
  if (calcium_units == "mmol/L") {
    calcium_mg_dl <- calcium / 0.25
  }

  albumin_g_dl <- albumin
  if (albumin_units == "g/L") {
    albumin_g_dl <- albumin / 10
  }

  corrected_ca_mg_dl <- calcium_mg_dl + 0.8 * (4.0 - albumin_g_dl)

  corrected_ca_mmol_l <- corrected_ca_mg_dl * 0.25

  return(list(
    Corrected_Calcium_mg_dL = round(corrected_ca_mg_dl, 2),
    Corrected_Calcium_mmol_L = round(corrected_ca_mmol_l, 2),
    Inputs = list(
      Measured_Calcium = calcium,
      Measured_Albumin = albumin,
      Calcium_Units = calcium_units,
      Albumin_Units = albumin_units
    )
  ))
}

cambridge_diabetes_risk <- function(age, sex, bmi, family_history, smoking_status, rx_steroids_htn) {

  age <- as.numeric(age)
  bmi <- as.numeric(bmi)
  family_history <- as.numeric(family_history)
  rx_steroids_htn <- as.numeric(rx_steroids_htn)
  sex <- tolower(as.character(sex))
  smoking_status <- tolower(as.character(smoking_status))

  if (any(is.na(c(age, bmi, family_history, rx_steroids_htn)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (!smoking_status %in% c("non", "past", "current")) {
    stop("Error: Smoking status must be 'non', 'past', or 'current'.")
  }

  score <- 0

  if (age < 45) {
    score <- score + 0
  } else if (age >= 45 && age <= 54) {
    score <- score + 6
  } else if (age >= 55 && age <= 64) {
    score <- score + 11
  } else {
    score <- score + 13
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (bmi < 25) {
    score <- score + 0
  } else if (bmi >= 25 && bmi < 27.5) {
    score <- score + 4
  } else if (bmi >= 27.5 && bmi < 30) {
    score <- score + 9
  } else {
    score <- score + 15
  }

  if (family_history == 1) {
    score <- score + 5
  }

  if (smoking_status == "past") {
    score <- score + 2
  } else if (smoking_status == "current") {
    score <- score + 4
  }

  if (rx_steroids_htn == 1) {
    score <- score + 2
  }

  risk_probability <- ""

  if (score <= 0.16) {
    risk_probability <- "Low Risk"
  }

  exponent <- -6.322 + (score * 0.176)
  if (score >= 14) {
    interpretation <- "High Risk. >1 in 30 chance of undiagnosed diabetes."
  } else {
    interpretation <- "Low/Moderate Risk."
  }

  return(list(
    Cambridge_Risk_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Age = age,
      Sex = sex,
      BMI = bmi,
      Family_History = family_history == 1,
      Smoking = smoking_status,
      Medication_Hx = rx_steroids_htn == 1
    )
  ))
}

caroc_fracture_risk <- function(sex, age, femoral_neck_t_score, prior_fragility_fracture, glucocorticoid_use) {

  sex <- tolower(as.character(sex))
  age <- as.numeric(age)
  femoral_neck_t_score <- as.numeric(femoral_neck_t_score)
  prior_fragility_fracture <- as.numeric(prior_fragility_fracture)
  glucocorticoid_use <- as.numeric(glucocorticoid_use)

  if (any(is.na(c(age, femoral_neck_t_score, prior_fragility_fracture, glucocorticoid_use)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (age < 50) {
    return(list(
      Risk_Category = "Assessment Not Valid",
      Message = "CAROC system is intended for women and men over age 50."
    ))
  }

  base_risk_level <- 0

  if (sex == "female") {
    mod_thresh <- -2.5 + ((age - 50) * (2.0 / 35))
    high_thresh <- -3.9 + ((age - 50) * (1.9 / 35))
  } else {
    mod_thresh <- -3.5 + ((age - 50) * (2.0 / 35))
    high_thresh <- -4.5 + ((age - 50) * (1.7 / 35))
  }

  if (femoral_neck_t_score > mod_thresh) {
    base_risk_level <- 0
  } else if (femoral_neck_t_score > high_thresh) {
    base_risk_level <- 1
  } else {
    base_risk_level <- 2
  }

  final_risk_level <- base_risk_level

  if (prior_fragility_fracture == 1 && glucocorticoid_use == 1) {
    final_risk_level <- 2
  } else if (prior_fragility_fracture == 1 || glucocorticoid_use == 1) {
    final_risk_level <- final_risk_level + 1
    if (final_risk_level > 2) final_risk_level <- 2
  }

  risk_category <- ""
  risk_percentage <- ""

  if (final_risk_level == 0) {
    risk_category <- "Low Risk"
    risk_percentage <- "<10%"
  } else if (final_risk_level == 1) {
    risk_category <- "Moderate Risk"
    risk_percentage <- "10% - 20%"
  } else {
    risk_category <- "High Risk"
    risk_percentage <- ">20%"
  }

  return(list(
    Risk_Category = risk_category,
    Ten_Year_Fracture_Risk = risk_percentage,
    Inputs = list(
      Sex = sex,
      Age = age,
      T_Score = femoral_neck_t_score,
      Prior_Fracture = prior_fragility_fracture == 1,
      Steroids = glucocorticoid_use == 1
    )
  ))
}

canadian_c_spine_rule <- function(age, dangerous_mechanism, paresthesias_extremities, simple_rear_end_mva, sitting_in_ed, ambulatory_at_any_time, delayed_onset_neck_pain, midline_c_spine_tenderness, able_to_rotate_45_degrees) {

  age <- as.numeric(age)
  dangerous_mechanism <- as.numeric(dangerous_mechanism)
  paresthesias_extremities <- as.numeric(paresthesias_extremities)
  simple_rear_end_mva <- as.numeric(simple_rear_end_mva)
  sitting_in_ed <- as.numeric(sitting_in_ed)
  ambulatory_at_any_time <- as.numeric(ambulatory_at_any_time)
  delayed_onset_neck_pain <- as.numeric(delayed_onset_neck_pain)
  midline_c_spine_tenderness <- as.numeric(midline_c_spine_tenderness)
  able_to_rotate_45_degrees <- as.numeric(able_to_rotate_45_degrees)

  if (any(is.na(c(age, dangerous_mechanism, paresthesias_extremities, simple_rear_end_mva, sitting_in_ed, ambulatory_at_any_time, delayed_onset_neck_pain, midline_c_spine_tenderness, able_to_rotate_45_degrees)))) {
    stop("Error: All inputs must be numeric (0 or 1, and age).")
  }

  recommendation <- ""
  risk_level <- ""

  is_high_risk <- FALSE
  if (age >= 65 || dangerous_mechanism == 1 || paresthesias_extremities == 1) {
    is_high_risk <- TRUE
  }

  if (is_high_risk) {
    risk_level <- "High Risk"
    recommendation <- "Radiography Indicated."
  } else {

    has_low_risk_factor <- FALSE
    if (simple_rear_end_mva == 1 || sitting_in_ed == 1 || ambulatory_at_any_time == 1 || delayed_onset_neck_pain == 1 || midline_c_spine_tenderness == 0) {
      has_low_risk_factor <- TRUE
    }

    if (!has_low_risk_factor) {
      risk_level <- "No Low Risk Factors Present"
      recommendation <- "Radiography Indicated."
    } else {

      if (able_to_rotate_45_degrees == 0) {
        risk_level <- "Inability to Rotate Neck"
        recommendation <- "Radiography Indicated."
      } else {
        risk_level <- "Low Risk"
        recommendation <- "No Radiography Needed."
      }
    }
  }

  return(list(
    Recommendation = recommendation,
    Risk_Reasoning = risk_level,
    Inputs = list(
      High_Risk_Criteria = list(
        Age_ge_65 = age >= 65,
        Dangerous_Mechanism = dangerous_mechanism == 1,
        Paresthesias = paresthesias_extremities == 1
      ),
      Low_Risk_Criteria = list(
        Simple_Rear_End = simple_rear_end_mva == 1,
        Sitting_ED = sitting_in_ed == 1,
        Ambulatory = ambulatory_at_any_time == 1,
        Delayed_Onset = delayed_onset_neck_pain == 1,
        No_Midline_Tenderness = midline_c_spine_tenderness == 0
      ),
      Rotation_Ability = able_to_rotate_45_degrees == 1
    )
  ))
}

ccs_angina_grade <- function(inability_or_rest, marked_limitation, slight_limitation, strenuous_only) {

  inability_or_rest <- as.numeric(inability_or_rest)
  marked_limitation <- as.numeric(marked_limitation)
  slight_limitation <- as.numeric(slight_limitation)
  strenuous_only <- as.numeric(strenuous_only)

  if (any(is.na(c(inability_or_rest, marked_limitation, slight_limitation, strenuous_only)))) {
    stop("Error: All inputs must be numeric.")
  }

  grade <- "None"
  description <- "No angina symptoms reported based on inputs."

  if (inability_or_rest == 1) {
    grade <- "Grade IV"
    description <- "Inability to carry on any physical activity without discomfort; angina syndrome may be present at rest."
  } else if (marked_limitation == 1) {
    grade <- "Grade III"
    description <- "Marked limitation of ordinary physical activity (e.g., walking 1-2 blocks on the level and climbing 1 flight of stairs in normal conditions)."
  } else if (slight_limitation == 1) {
    grade <- "Grade II"
    description <- "Slight limitation of ordinary activity (e.g., walking >2 blocks on the level and climbing >1 flight of stairs in normal conditions)."
  } else if (strenuous_only == 1) {
    grade <- "Grade I"
    description <- "Ordinary physical activity does not cause angina (e.g., walking, climbing stairs). Angina with strenuous, rapid, or prolonged exertion."
  }

  return(list(
    CCS_Grade = grade,
    Description = description,
    Inputs = list(
      Grade_IV_Criteria = inability_or_rest == 1,
      Grade_III_Criteria = marked_limitation == 1,
      Grade_II_Criteria = slight_limitation == 1,
      Grade_I_Criteria = strenuous_only == 1
    )
  ))
}

canadian_ct_head_rule <- function(gcs_less_than_15_at_2h, suspected_open_skull_fracture, basal_skull_fracture_signs, vomiting_ge_2, age, amnesia_pre_impact_gt_30m, dangerous_mechanism) {

  gcs_less_than_15_at_2h <- as.numeric(gcs_less_than_15_at_2h)
  suspected_open_skull_fracture <- as.numeric(suspected_open_skull_fracture)
  basal_skull_fracture_signs <- as.numeric(basal_skull_fracture_signs)
  vomiting_ge_2 <- as.numeric(vomiting_ge_2)
  age <- as.numeric(age)
  amnesia_pre_impact_gt_30m <- as.numeric(amnesia_pre_impact_gt_30m)
  dangerous_mechanism <- as.numeric(dangerous_mechanism)

  if (any(is.na(c(gcs_less_than_15_at_2h, suspected_open_skull_fracture, basal_skull_fracture_signs, vomiting_ge_2, age, amnesia_pre_impact_gt_30m, dangerous_mechanism)))) {
    stop("Error: All inputs must be numeric.")
  }

  high_risk <- FALSE
  medium_risk <- FALSE
  risk_level <- "Low Risk"
  recommendation <- "CT Head not indicated."
  reason <- "No high or medium risk factors present."

  if (gcs_less_than_15_at_2h == 1) high_risk <- TRUE
  if (suspected_open_skull_fracture == 1) high_risk <- TRUE
  if (basal_skull_fracture_signs == 1) high_risk <- TRUE
  if (vomiting_ge_2 == 1) high_risk <- TRUE
  if (age >= 65) high_risk <- TRUE

  if (!high_risk) {
    if (amnesia_pre_impact_gt_30m == 1) medium_risk <- TRUE
    if (dangerous_mechanism == 1) medium_risk <- TRUE
  }

  if (high_risk) {
    risk_level <- "High Risk"
    recommendation <- "CT Head is indicated."
    reason <- "High risk for neurological intervention."
  } else if (medium_risk) {
    risk_level <- "Medium Risk"
    recommendation <- "CT Head is indicated."
    reason <- "High risk for brain injury on CT."
  }

  return(list(
    Risk_Level = risk_level,
    Recommendation = recommendation,
    Reasoning = reason,
    Inputs = list(
      GCS_lt_15_2h = gcs_less_than_15_at_2h == 1,
      Open_Fracture = suspected_open_skull_fracture == 1,
      Basal_Fracture = basal_skull_fracture_signs == 1,
      Vomiting = vomiting_ge_2 == 1,
      Age_ge_65 = age >= 65,
      Amnesia = amnesia_pre_impact_gt_30m == 1,
      Mechanism = dangerous_mechanism == 1
    )
  ))
}

canrisk_diabetes_score <- function(age, sex, bmi, waist_cm, physical_activity_daily, fruit_veg_daily, hypertension_meds, high_glucose_history, gave_birth_large_baby, family_history, ethnicity_risk, education_level) {

  age <- as.numeric(age)
  bmi <- as.numeric(bmi)
  waist_cm <- as.numeric(waist_cm)
  sex <- tolower(as.character(sex))
  physical_activity_daily <- as.numeric(physical_activity_daily)
  fruit_veg_daily <- as.numeric(fruit_veg_daily)
  hypertension_meds <- as.numeric(hypertension_meds)
  high_glucose_history <- as.numeric(high_glucose_history)
  gave_birth_large_baby <- as.numeric(gave_birth_large_baby)
  family_history <- tolower(as.character(family_history))
  ethnicity_risk <- tolower(as.character(ethnicity_risk))
  education_level <- tolower(as.character(education_level))

  if (any(is.na(c(age, bmi, waist_cm, physical_activity_daily, fruit_veg_daily, hypertension_meds, high_glucose_history)))) {
    stop("Error: Numeric inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age >= 45 && age <= 54) {
    score <- score + 7
  } else if (age >= 55 && age <= 64) {
    score <- score + 13
  } else if (age >= 65 && age <= 74) {
    score <- score + 15
  } else if (age >= 75) {
    score <- score + 15
  }

  if (sex == "male") {
    score <- score + 6
  }

  if (bmi >= 25 && bmi <= 29.9) {
    score <- score + 4
  } else if (bmi >= 30 && bmi <= 34.9) {
    score <- score + 9
  } else if (bmi >= 35) {
    score <- score + 14
  }

  if (sex == "male") {
    if (waist_cm >= 94 && waist_cm <= 102) {
      score <- score + 4
    } else if (waist_cm > 102) {
      score <- score + 6
    }
  } else {
    if (waist_cm >= 80 && waist_cm <= 88) {
      score <- score + 4
    } else if (waist_cm > 88) {
      score <- score + 6
    }
  }

  if (physical_activity_daily == 0) {
    score <- score + 2
  }

  if (fruit_veg_daily == 0) {
    score <- score + 2
  }

  if (hypertension_meds == 1) {
    score <- score + 2
  }

  if (high_glucose_history == 1) {
    score <- score + 14
  }

  if (sex == "female" && gave_birth_large_baby == 1) {
    score <- score + 1
  }

  if (family_history == "parent_sibling") {
    score <- score + 4
  } else if (family_history == "other_relative") {
    score <- score + 2
  }

  if (ethnicity_risk == "high") {
    score <- score + 11
  } else if (ethnicity_risk == "moderate" || ethnicity_risk == "east_asian") {
    score <- score + 10
  } else if (ethnicity_risk == "mixed" || ethnicity_risk == "other_non_white") {
    score <- score + 3
  }

  if (education_level == "secondary") {
    score <- score + 1
  } else if (education_level == "primary" || education_level == "less_than_high_school") {
    score <- score + 5
  }

  risk_level <- ""
  if (score < 21) {
    risk_level <- "Low Risk (1 in 100 chance)"
  } else if (score <= 32) {
    risk_level <- "Moderate Risk (1 in 10 chance)"
  } else {
    risk_level <- "High Risk (1 in 3 to 1 in 2 chance)"
  }

  return(list(
    CANRISK_Score = score,
    Risk_Category = risk_level,
    Inputs = list(
      Age = age,
      Sex = sex,
      BMI = bmi,
      Waist = waist_cm,
      Activity = physical_activity_daily,
      Diet = fruit_veg_daily,
      HTN = hypertension_meds,
      High_Sugar = high_glucose_history,
      Large_Baby = if(sex=="female") gave_birth_large_baby else "N/A",
      Fam_Hx = family_history,
      Ethnicity = ethnicity_risk,
      Education = education_level
    )
  ))
}

canadian_syncope_risk_score <- function(vasovagal_predisposition, heart_disease_history, sbp_any, sbp_ed, troponin_elevated, axis_qrs_qt_abnormal, ed_diagnosis_cardiac, ed_diagnosis_vasovagal) {

  vasovagal_predisposition <- as.numeric(vasovagal_predisposition)
  heart_disease_history <- as.numeric(heart_disease_history)
  sbp_any <- as.numeric(sbp_any)
  sbp_ed <- as.numeric(sbp_ed)
  troponin_elevated <- as.numeric(troponin_elevated)
  axis_qrs_qt_abnormal <- as.numeric(axis_qrs_qt_abnormal)
  ed_diagnosis_cardiac <- as.numeric(ed_diagnosis_cardiac)
  ed_diagnosis_vasovagal <- as.numeric(ed_diagnosis_vasovagal)

  if (any(is.na(c(vasovagal_predisposition, heart_disease_history, sbp_any, sbp_ed, troponin_elevated, axis_qrs_qt_abnormal, ed_diagnosis_cardiac, ed_diagnosis_vasovagal)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (vasovagal_predisposition == 1) {
    score <- score - 1
  }

  if (heart_disease_history == 1) {
    score <- score + 1
  }

  if (sbp_any == 1) {
    score <- score + 2
  }

  if (troponin_elevated == 1) {
    score <- score + 2
  }

  if (axis_qrs_qt_abnormal == 1) {
    score <- score + 1
  }

  if (ed_diagnosis_vasovagal == 1) {
    score <- score - 2
  }

  if (ed_diagnosis_cardiac == 1) {
    score <- score + 2
  }

  risk_category <- ""
  risk_30_day <- ""

  if (score <= -2) {
    risk_category <- "Very Low Risk"
    risk_30_day <- "< 1%"
  } else if (score >= -1 && score <= 0) {
    risk_category <- "Low Risk"
    risk_30_day <- "1-4%"
  } else if (score >= 1 && score <= 3) {
    risk_category <- "Medium Risk"
    risk_30_day <- "8-19%"
  } else if (score >= 4 && score <= 5) {
    risk_category <- "High Risk"
    risk_30_day <- "30-50%"
  } else {
    risk_category <- "Very High Risk"
    risk_30_day <- "> 80%"
  }

  return(list(
    CSRS_Score = score,
    Risk_Category = risk_category,
    Est_30_Day_Serious_Outcome_Risk = risk_30_day,
    Inputs = list(
      Vasovagal_Predisposition = vasovagal_predisposition == 1,
      Heart_Disease_Hx = heart_disease_history == 1,
      Abnormal_SBP = sbp_any == 1,
      Elevated_Troponin = troponin_elevated == 1,
      Abnormal_ECG = axis_qrs_qt_abnormal == 1,
      Dx_Vasovagal = ed_diagnosis_vasovagal == 1,
      Dx_Cardiac = ed_diagnosis_cardiac == 1
    )
  ))
}

canadian_tia_score <- function(history_tia_stroke, history_afib, unilateral_weakness, speech_disturbance, duration_ge_10min, dbp_ge_110, platelets_ge_400, glucose_ge_15, ct_findings, carotid_stenosis_ge_50) {

  history_tia_stroke <- as.numeric(history_tia_stroke)
  history_afib <- as.numeric(history_afib)
  unilateral_weakness <- as.numeric(unilateral_weakness)
  speech_disturbance <- as.numeric(speech_disturbance)
  duration_ge_10min <- as.numeric(duration_ge_10min)
  dbp_ge_110 <- as.numeric(dbp_ge_110)
  platelets_ge_400 <- as.numeric(platelets_ge_400)
  glucose_ge_15 <- as.numeric(glucose_ge_15)
  ct_findings <- as.numeric(ct_findings)
  carotid_stenosis_ge_50 <- as.numeric(carotid_stenosis_ge_50)

  if (any(is.na(c(history_tia_stroke, history_afib, unilateral_weakness, speech_disturbance, duration_ge_10min, dbp_ge_110, platelets_ge_400, glucose_ge_15, ct_findings, carotid_stenosis_ge_50)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (history_tia_stroke == 1) score <- score + 1
  if (history_afib == 1) score <- score + 2
  if (unilateral_weakness == 1) score <- score + 2
  if (speech_disturbance == 1) score <- score + 2
  if (duration_ge_10min == 1) score <- score + 2
  if (dbp_ge_110 == 1) score <- score + 2
  if (platelets_ge_400 == 1) score <- score + 2
  if (glucose_ge_15 == 1) score <- score + 2

  if (ct_findings == 1) {
    score <- score + 1
  } else if (ct_findings == 2) {
    score <- score + 2
  }

  if (carotid_stenosis_ge_50 == 1) score <- score + 2

  risk_level <- ""
  stroke_risk_7_days <- ""

  if (score <= 3) {
    risk_level <- "Low Risk"
    stroke_risk_7_days <- "<0.5%"
  } else if (score <= 8) {
    risk_level <- "Medium Risk"
    stroke_risk_7_days <- "~2.3%"
  } else {
    risk_level <- "High Risk"
    stroke_risk_7_days <- ">5.9% (Up to 26.6% for scores >= 13)"
  }

  return(list(
    Canadian_TIA_Score = score,
    Risk_Category = risk_level,
    Subsequent_Stroke_Risk_7_Day = stroke_risk_7_days,
    Inputs = list(
      Hx_TIA_Stroke = history_tia_stroke == 1,
      Hx_New_AFib = history_afib == 1,
      Weakness = unilateral_weakness == 1,
      Speech = speech_disturbance == 1,
      Duration_10min = duration_ge_10min == 1,
      DBP_110 = dbp_ge_110 == 1,
      Plt_400 = platelets_ge_400 == 1,
      Glucose_15 = glucose_ge_15 == 1,
      CT_Result = switch(as.character(ct_findings), "0"="Normal", "1"="Old Infarct", "2"="Acute Infarct"),
      Carotid_Stenosis = carotid_stenosis_ge_50 == 1
    )
  ))
}

carg_chemo_toxicity_score <- function(age, sex, weight_kg, height_cm, creatinine_mg_dl, hemoglobin_g_dl,
                                      cancer_type_gi_gu, standard_dose, polychemotherapy,
                                      falls_last_6mo, hearing_impairment, limit_walking_1block,
                                      assistance_meds, limit_social_activity) {

  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)
  height_cm <- as.numeric(height_cm)
  creatinine_mg_dl <- as.numeric(creatinine_mg_dl)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)

  cancer_type_gi_gu <- as.numeric(cancer_type_gi_gu)
  standard_dose <- as.numeric(standard_dose)
  polychemotherapy <- as.numeric(polychemotherapy)
  falls_last_6mo <- as.numeric(falls_last_6mo)
  hearing_impairment <- as.numeric(hearing_impairment)
  limit_walking_1block <- as.numeric(limit_walking_1block)
  assistance_meds <- as.numeric(assistance_meds)
  limit_social_activity <- as.numeric(limit_social_activity)

  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, weight_kg, height_cm, creatinine_mg_dl, hemoglobin_g_dl, cancer_type_gi_gu,
                  standard_dose, polychemotherapy, falls_last_6mo, hearing_impairment,
                  limit_walking_1block, assistance_meds, limit_social_activity)))) {
    stop("Error: All inputs must be numeric/valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  jelliffe_crcl <- ((98 - (0.8 * (age - 20))) / creatinine_mg_dl)
  if (sex == "female") {
    jelliffe_crcl <- jelliffe_crcl * 0.90
  }

  score <- 0

  if (age >= 72) {
    score <- score + 2
  }

  if (cancer_type_gi_gu == 1) {
    score <- score + 2
  }

  if (standard_dose == 1) {
    score <- score + 2
  }

  if (polychemotherapy == 1) {
    score <- score + 2
  }

  is_anemic <- FALSE
  if (sex == "male" && hemoglobin_g_dl < 11) is_anemic <- TRUE
  if (sex == "female" && hemoglobin_g_dl < 10) is_anemic <- TRUE

  if (is_anemic) {
    score <- score + 3
  }

  if (jelliffe_crcl < 34) {
    score <- score + 3
  }

  if (falls_last_6mo == 1) {
    score <- score + 3
  }

  if (hearing_impairment == 1) {
    score <- score + 2
  }

  if (limit_walking_1block == 1) {
    score <- score + 2
  }

  if (assistance_meds == 1) {
    score <- score + 1
  }

  if (limit_social_activity == 1) {
    score <- score + 1
  }

  risk_level <- ""
  toxicity_risk <- ""

  if (score <= 5) {
    risk_level <- "Low Risk"
    toxicity_risk <- "30%"
  } else if (score <= 9) {
    risk_level <- "Intermediate Risk"
    toxicity_risk <- "52%"
  } else {
    risk_level <- "High Risk"
    toxicity_risk <- "83%"
  }

  return(list(
    CARG_Score = score,
    Risk_Level = risk_level,
    Grade_3_5_Toxicity_Risk = toxicity_risk,
    Calculated_Jelliffe_CrCl = round(jelliffe_crcl, 1),
    Inputs = list(
      Age_ge_72 = age >= 72,
      Cancer_GI_GU = cancer_type_gi_gu == 1,
      Standard_Dose = standard_dose == 1,
      Polychemo = polychemotherapy == 1,
      Anemia = is_anemic,
      CrCl_lt_34 = jelliffe_crcl < 34,
      Falls = falls_last_6mo == 1,
      Hearing_Fair_Poor = hearing_impairment == 1,
      Walk_Limited = limit_walking_1block == 1,
      Meds_Help = assistance_meds == 1,
      Social_Limited = limit_social_activity == 1
    )
  ))
}

caprini_vte_score <- function(age, risk_factors_1pt_count, risk_factors_2pt_count, risk_factors_3pt_count, risk_factors_5pt_count) {

  age <- as.numeric(age)
  risk_factors_1pt_count <- as.numeric(risk_factors_1pt_count)
  risk_factors_2pt_count <- as.numeric(risk_factors_2pt_count)
  risk_factors_3pt_count <- as.numeric(risk_factors_3pt_count)
  risk_factors_5pt_count <- as.numeric(risk_factors_5pt_count)

  if (any(is.na(c(age, risk_factors_1pt_count, risk_factors_2pt_count, risk_factors_3pt_count, risk_factors_5pt_count)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 41 && age <= 60) {
    score <- score + 1
  } else if (age >= 61 && age <= 74) {
    score <- score + 2
  } else if (age >= 75) {
    score <- score + 3
  }

  score <- score + (risk_factors_1pt_count * 1)
  score <- score + (risk_factors_2pt_count * 2)
  score <- score + (risk_factors_3pt_count * 3)
  score <- score + (risk_factors_5pt_count * 5)

  risk_level <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_level <- "Very Low Risk (0-1)"
    recommendation <- "Early ambulation."
  } else if (score == 2) {
    risk_level <- "Low Risk (2)"
    recommendation <- "Mechanical prophylaxis (IPC) or chemical prophylaxis (LMWH/UH)."
  } else if (score >= 3 && score <= 4) {
    risk_level <- "Moderate Risk (3-4)"
    recommendation <- "Chemical prophylaxis (LMWH/UH) +/- Mechanical prophylaxis."
  } else {
    risk_level <- "High Risk (>=5)"
    recommendation <- "Chemical prophylaxis (LMWH/UH) + Mechanical prophylaxis."
  }

  return(list(
    Caprini_Score = score,
    Risk_Category = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Count_1pt_Items = risk_factors_1pt_count,
      Count_2pt_Items = risk_factors_2pt_count,
      Count_3pt_Items = risk_factors_3pt_count,
      Count_5pt_Items = risk_factors_5pt_count
    )
  ))
}

cardiac_anesthesia_risk_evaluation <- function(history_cardiac_surgery, complex_surgery, uncontrolled_medical_condition, emergency_surgery) {

  history_cardiac_surgery <- as.numeric(history_cardiac_surgery)
  complex_surgery <- as.numeric(complex_surgery)
  uncontrolled_medical_condition <- as.numeric(uncontrolled_medical_condition)
  emergency_surgery <- as.numeric(emergency_surgery)

  if (any(is.na(c(history_cardiac_surgery, complex_surgery, uncontrolled_medical_condition, emergency_surgery)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  care_score <- 1

  if (emergency_surgery == 1) {
    care_score <- 5
  } else if (uncontrolled_medical_condition == 1) {
    care_score <- 4
  } else if (complex_surgery == 1) {
    care_score <- 3
  } else if (history_cardiac_surgery == 1) {
    care_score <- 2
  } else {
    care_score <- 1
  }

  mortality_risk <- ""
  morbidity_risk <- ""

  if (care_score == 1) {
    mortality_risk <- "0.5%"
    morbidity_risk <- "3.1%"
  } else if (care_score == 2) {
    mortality_risk <- "2.2%"
    morbidity_risk <- "8.8%"
  } else if (care_score == 3) {
    mortality_risk <- "4.7%"
    morbidity_risk <- "14.2%"
  } else if (care_score == 4) {
    mortality_risk <- "16.2%"
    morbidity_risk <- "30.8%"
  } else {
    mortality_risk <- "23.2%"
    morbidity_risk <- "55.6%"
  }

  return(list(
    CARE_Score = care_score,
    In_Hospital_Mortality_Risk = mortality_risk,
    Morbidity_Risk = morbidity_risk,
    Inputs = list(
      History_Cardiac_Surgery = history_cardiac_surgery == 1,
      Complex_Surgery = complex_surgery == 1,
      Uncontrolled_Medical_Condition = uncontrolled_medical_condition == 1,
      Emergency = emergency_surgery == 1
    )
  ))
}

fick_cardiac_output <- function(hemoglobin_g_dl, sa_o2_percent, sv_o2_percent, vo2_ml_min, pa_o2_mmhg = NULL, pv_o2_mmhg = NULL) {

  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  sa_o2_percent <- as.numeric(sa_o2_percent)
  sv_o2_percent <- as.numeric(sv_o2_percent)
  vo2_ml_min <- as.numeric(vo2_ml_min)

  if (any(is.na(c(hemoglobin_g_dl, sa_o2_percent, sv_o2_percent, vo2_ml_min)))) {
    stop("Error: Hemoglobin, O2 Saturations, and VO2 must be numeric.")
  }

  sa_o2_frac <- sa_o2_percent
  if (sa_o2_percent > 1) {
    sa_o2_frac <- sa_o2_percent / 100
  }

  sv_o2_frac <- sv_o2_percent
  if (sv_o2_percent > 1) {
    sv_o2_frac <- sv_o2_percent / 100
  }

  dissolved_o2_a <- 0
  if (!is.null(pa_o2_mmhg)) {
    dissolved_o2_a <- as.numeric(pa_o2_mmhg) * 0.0031
  }

  dissolved_o2_v <- 0
  if (!is.null(pv_o2_mmhg)) {
    dissolved_o2_v <- as.numeric(pv_o2_mmhg) * 0.0031
  }

  ca_o2 <- (1.34 * hemoglobin_g_dl * sa_o2_frac) + dissolved_o2_a
  cv_o2 <- (1.34 * hemoglobin_g_dl * sv_o2_frac) + dissolved_o2_v

  delta_av_content <- ca_o2 - cv_o2

  if (delta_av_content <= 0) {
    stop("Error: Arterial O2 content must be greater than Venous O2 content.")
  }

  delta_av_content_ml_l <- delta_av_content * 10

  cardiac_output <- vo2_ml_min / delta_av_content_ml_l

  return(list(
    Cardiac_Output_L_min = round(cardiac_output, 2),
    Arterial_O2_Content_mL_dL = round(ca_o2, 1),
    Venous_O2_Content_mL_dL = round(cv_o2, 1),
    AV_O2_Difference_mL_dL = round(delta_av_content, 1),
    Inputs = list(
      Hgb = hemoglobin_g_dl,
      SaO2 = sa_o2_percent,
      SvO2 = sv_o2_percent,
      VO2 = vo2_ml_min
    )
  ))
}

cardiac_power_output <- function(mean_arterial_pressure, cardiac_output) {

  mean_arterial_pressure <- as.numeric(mean_arterial_pressure)
  cardiac_output <- as.numeric(cardiac_output)

  if (any(is.na(c(mean_arterial_pressure, cardiac_output)))) {
    stop("Error: All inputs must be numeric.")
  }

  cpo_watts <- (mean_arterial_pressure * cardiac_output) / 451

  interpretation <- ""
  if (cpo_watts < 0.6) {
    interpretation <- "Low CPO (< 0.6 W). Associated with significantly increased in-hospital mortality in cardiogenic shock."
  } else {
    interpretation <- "CPO >= 0.6 W. Associated with better survival outcomes in cardiogenic shock."
  }

  return(list(
    CPO_Watts = round(cpo_watts, 2),
    Interpretation = interpretation,
    Inputs = list(
      MAP_mmHg = mean_arterial_pressure,
      CO_L_min = cardiac_output
    )
  ))
}

car_olt_score <- function(age, sex, race, employment_status, education_level, history_hcc, diabetes, heart_failure, atrial_fibrillation, pulmonary_hypertension, systemic_hypertension, respiratory_failure) {

  age <- as.numeric(age)
  history_hcc <- as.numeric(history_hcc)
  diabetes <- as.numeric(diabetes)
  heart_failure <- as.numeric(heart_failure)
  atrial_fibrillation <- as.numeric(atrial_fibrillation)
  pulmonary_hypertension <- as.numeric(pulmonary_hypertension)
  systemic_hypertension <- as.numeric(systemic_hypertension)
  respiratory_failure <- as.numeric(respiratory_failure)

  sex <- tolower(as.character(sex))
  race <- tolower(as.character(race))
  employment_status <- tolower(as.character(employment_status))
  education_level <- tolower(as.character(education_level))

  if (any(is.na(c(age, history_hcc, diabetes, heart_failure, atrial_fibrillation, pulmonary_hypertension, systemic_hypertension, respiratory_failure)))) {
    stop("Error: All clinical history inputs must be numeric (0 or 1) and Age must be a number.")
  }

  score <- 0

  if (age >= 60) score <- score + 3

  if (history_hcc == 1) score <- score + 2
  if (diabetes == 1) score <- score + 3
  if (heart_failure == 1) score <- score + 4
  if (atrial_fibrillation == 1) score <- score + 5
  if (pulmonary_hypertension == 1) score <- score + 4
  if (systemic_hypertension == 1) score <- score + 2
  if (respiratory_failure == 1) score <- score + 6

  risk_category <- ""

  if (score <= 23) {
    risk_category <- "Low Risk (Score <= 23)"
  } else {
    risk_category <- "Elevated Risk (Score > 23)"
  }

  return(list(
    CAR_OLT_Score = score,
    Risk_Threshold = risk_category,
    Inputs = list(
      Age = age,
      Sex = sex,
      Race = race,
      Employment = employment_status,
      Education = education_level,
      History_HCC = history_hcc == 1,
      Diabetes = diabetes == 1,
      Heart_Failure = heart_failure == 1,
      AFib = atrial_fibrillation == 1,
      Pulm_HTN = pulmonary_hypertension == 1,
      Syst_HTN = systemic_hypertension == 1,
      Resp_Failure = respiratory_failure == 1
    )
  ))
}

cts6_score <- function(numbness_median_territory, nocturnal_numbness, thenar_atrophy_weakness, phalens_test, tinels_sign, two_point_discrim_gt_5mm) {

  numbness_median_territory <- as.numeric(numbness_median_territory)
  nocturnal_numbness <- as.numeric(nocturnal_numbness)
  thenar_atrophy_weakness <- as.numeric(thenar_atrophy_weakness)
  phalens_test <- as.numeric(phalens_test)
  tinels_sign <- as.numeric(tinels_sign)
  two_point_discrim_gt_5mm <- as.numeric(two_point_discrim_gt_5mm)

  if (any(is.na(c(numbness_median_territory, nocturnal_numbness, thenar_atrophy_weakness, phalens_test, tinels_sign, two_point_discrim_gt_5mm)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (numbness_median_territory == 1) {
    score <- score + 3.5
  }

  if (nocturnal_numbness == 1) {
    score <- score + 4.0
  }

  if (thenar_atrophy_weakness == 1) {
    score <- score + 5.0
  }

  if (phalens_test == 1) {
    score <- score + 5.0
  }

  if (tinels_sign == 1) {
    score <- score + 4.0
  }

  if (two_point_discrim_gt_5mm == 1) {
    score <- score + 4.5
  }

  probability_category <- ""

  if (score < 5) {
    probability_category <- "Low Probability (<25%)"
  } else if (score <= 12) {
    probability_category <- "Intermediate Probability"
  } else {
    probability_category <- "High Probability (>80%)"
  }

  return(list(
    CTS6_Score = score,
    Probability_Category = probability_category,
    Inputs = list(
      Numbness_Median = numbness_median_territory == 1,
      Nocturnal_Numbness = nocturnal_numbness == 1,
      Thenar_Atrophy = thenar_atrophy_weakness == 1,
      Phalens = phalens_test == 1,
      Tinels = tinels_sign == 1,
      Two_Point_Discrim = two_point_discrim_gt_5mm == 1
    )
  ))
}

cart_cardiac_arrest_score <- function(respiratory_rate, heart_rate, diastolic_bp, age) {

  respiratory_rate <- as.numeric(respiratory_rate)
  heart_rate <- as.numeric(heart_rate)
  diastolic_bp <- as.numeric(diastolic_bp)
  age <- as.numeric(age)

  if (any(is.na(c(respiratory_rate, heart_rate, diastolic_bp, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (respiratory_rate < 21) {
    score <- score + 0
  } else if (respiratory_rate >= 21 && respiratory_rate <= 23) {
    score <- score + 8
  } else if (respiratory_rate >= 24 && respiratory_rate <= 25) {
    score <- score + 12
  } else if (respiratory_rate >= 26 && respiratory_rate <= 29) {
    score <- score + 15
  } else {
    score <- score + 22
  }

  if (heart_rate < 110) {
    score <- score + 0
  } else if (heart_rate >= 110 && heart_rate <= 139) {
    score <- score + 4
  } else {
    score <- score + 13
  }

  if (diastolic_bp > 49) {
    score <- score + 0
  } else if (diastolic_bp >= 40 && diastolic_bp <= 49) {
    score <- score + 4
  } else if (diastolic_bp >= 35 && diastolic_bp <= 39) {
    score <- score + 6
  } else {
    score <- score + 13
  }

  if (age < 55) {
    score <- score + 0
  } else if (age >= 55 && age <= 69) {
    score <- score + 4
  } else {
    score <- score + 9
  }

  return(list(
    CART_Score = score,
    Inputs = list(
      RR = respiratory_rate,
      HR = heart_rate,
      DBP = diastolic_bp,
      Age = age
    )
  ))
}

caspar_psoriatic_arthritis <- function(inflammatory_disease_present, current_psoriasis, personal_history_psoriasis, family_history_psoriasis, nail_dystrophy, negative_rf, dactylitis, juxta_articular_bone_formation) {

  inflammatory_disease_present <- as.numeric(inflammatory_disease_present)
  current_psoriasis <- as.numeric(current_psoriasis)
  personal_history_psoriasis <- as.numeric(personal_history_psoriasis)
  family_history_psoriasis <- as.numeric(family_history_psoriasis)
  nail_dystrophy <- as.numeric(nail_dystrophy)
  negative_rf <- as.numeric(negative_rf)
  dactylitis <- as.numeric(dactylitis)
  juxta_articular_bone_formation <- as.numeric(juxta_articular_bone_formation)

  if (any(is.na(c(inflammatory_disease_present, current_psoriasis, personal_history_psoriasis, family_history_psoriasis, nail_dystrophy, negative_rf, dactylitis, juxta_articular_bone_formation)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  if (inflammatory_disease_present == 0) {
    return(list(
      Classification = "Not Applicable",
      Reason = "CASPAR criteria require the presence of inflammatory articular disease (joint, spine, or entheseal) for application."
    ))
  }

  score <- 0

  if (current_psoriasis == 1) {
    score <- score + 2
  } else if (personal_history_psoriasis == 1) {
    score <- score + 1
  } else if (family_history_psoriasis == 1) {
    score <- score + 1
  }

  if (nail_dystrophy == 1) {
    score <- score + 1
  }

  if (negative_rf == 1) {
    score <- score + 1
  }

  if (dactylitis == 1) {
    score <- score + 1
  }

  if (juxta_articular_bone_formation == 1) {
    score <- score + 1
  }

  classification <- "Not Classified as Psoriatic Arthritis"
  if (score >= 3) {
    classification <- "Classified as Psoriatic Arthritis"
  }

  return(list(
    Total_Score = score,
    Classification = classification,
    Specifics = list(
      Psoriasis_Score = if (current_psoriasis == 1) 2 else if (personal_history_psoriasis == 1 || family_history_psoriasis == 1) 1 else 0,
      Nail_Score = nail_dystrophy,
      RF_Score = negative_rf,
      Dactylitis_Score = dactylitis,
      Bone_Score = juxta_articular_bone_formation
    )
  ))
}

catch_head_injury_rule <- function(gcs_lt_15_2hr, suspected_open_skull_fx, worsening_headache, irritability, basal_skull_fx_signs, large_boggy_hematoma, dangerous_mechanism) {

  gcs_lt_15_2hr <- as.numeric(gcs_lt_15_2hr)
  suspected_open_skull_fx <- as.numeric(suspected_open_skull_fx)
  worsening_headache <- as.numeric(worsening_headache)
  irritability <- as.numeric(irritability)
  basal_skull_fx_signs <- as.numeric(basal_skull_fx_signs)
  large_boggy_hematoma <- as.numeric(large_boggy_hematoma)
  dangerous_mechanism <- as.numeric(dangerous_mechanism)

  if (any(is.na(c(gcs_lt_15_2hr, suspected_open_skull_fx, worsening_headache, irritability, basal_skull_fx_signs, large_boggy_hematoma, dangerous_mechanism)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  is_high_risk <- FALSE
  if (gcs_lt_15_2hr == 1) is_high_risk <- TRUE
  if (suspected_open_skull_fx == 1) is_high_risk <- TRUE
  if (worsening_headache == 1) is_high_risk <- TRUE
  if (irritability == 1) is_high_risk <- TRUE

  is_medium_risk <- FALSE
  if (!is_high_risk) {
    if (basal_skull_fx_signs == 1) is_medium_risk <- TRUE
    if (large_boggy_hematoma == 1) is_medium_risk <- TRUE
    if (dangerous_mechanism == 1) is_medium_risk <- TRUE
  }

  risk_level <- ""
  recommendation <- ""

  if (is_high_risk) {
    risk_level <- "High Risk"
    recommendation <- "CT Head is indicated (High risk for neurological intervention)."
  } else if (is_medium_risk) {
    risk_level <- "Medium Risk"
    recommendation <- "CT Head may be considered (Risk for brain injury on CT)."
  } else {
    risk_level <- "Low Risk"
    recommendation <- "CT Head usually not indicated."
  }

  return(list(
    Risk_Level = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      High_Risk_Factors = list(
        GCS_lt_15_at_2h = gcs_lt_15_2hr == 1,
        Suspected_Open_Fx = suspected_open_skull_fx == 1,
        Worsening_Headache = worsening_headache == 1,
        Irritability = irritability == 1
      ),
      Medium_Risk_Factors = list(
        Basal_Fx_Signs = basal_skull_fx_signs == 1,
        Boggy_Hematoma = large_boggy_hematoma == 1,
        Dangerous_Mechanism = dangerous_mechanism == 1
      )
    )
  ))
}

cedocs_score <- function(critical_care_patients, longest_admit_wait_hours, waiting_room_patients, total_ed_patients, ed_beds, annual_visits) {

  critical_care_patients <- as.numeric(critical_care_patients)
  longest_admit_wait_hours <- as.numeric(longest_admit_wait_hours)
  waiting_room_patients <- as.numeric(waiting_room_patients)
  total_ed_patients <- as.numeric(total_ed_patients)
  ed_beds <- as.numeric(ed_beds)
  annual_visits <- as.numeric(annual_visits)

  if (any(is.na(c(critical_care_patients, longest_admit_wait_hours, waiting_room_patients, total_ed_patients, ed_beds, annual_visits)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (ed_beds <= 0) {
    stop("Error: Number of ED beds must be greater than 0.")
  }

  ratio_patients_beds <- total_ed_patients / ed_beds

  v_factor <- (1.11e-12 * (max(annual_visits - 18811, 0)^3)) +
    (8.23e-12 * (max(annual_visits - 43012, 0)^3)) -
    (8.20e-12 * (max(annual_visits - 49466, 0)^3)) +
    (1.08e-12 * (max(annual_visits - 67273, 0)^3))

  score <- (3 * critical_care_patients) +
    (0.53 * longest_admit_wait_hours) +
    (1.16 * waiting_room_patients) +
    (20.66 * ratio_patients_beds) +
    (0.00126 * annual_visits) -
    30.39 -
    v_factor

  score <- round(score, 0)
  if (score < 0) score <- 0
  if (score > 200) score <- 200

  interpretation <- ""
  if (score <= 20) {
    interpretation <- "Not Busy"
  } else if (score <= 60) {
    interpretation <- "Busy"
  } else if (score <= 100) {
    interpretation <- "Extremely Busy but Not Overcrowded"
  } else if (score <= 140) {
    interpretation <- "Overcrowded"
  } else if (score <= 180) {
    interpretation <- "Severely Overcrowded"
  } else {
    interpretation <- "Dangerously Overcrowded"
  }

  return(list(
    CEDOCS_Score = score,
    Level_of_Crowding = interpretation,
    Inputs = list(
      Critical_Care_Patients = critical_care_patients,
      Longest_Admit_Wait_Hours = longest_admit_wait_hours,
      Waiting_Room_Patients = waiting_room_patients,
      Total_Patients = total_ed_patients,
      ED_Beds = ed_beds,
      Annual_Visits = annual_visits
    )
  ))
}

centor_score_modified <- function(age, tonsillar_exudate, tender_anterior_cervical_adenopathy, fever_history, absence_of_cough) {

  age <- as.numeric(age)
  tonsillar_exudate <- as.numeric(tonsillar_exudate)
  tender_anterior_cervical_adenopathy <- as.numeric(tender_anterior_cervical_adenopathy)
  fever_history <- as.numeric(fever_history)
  absence_of_cough <- as.numeric(absence_of_cough)

  if (any(is.na(c(age, tonsillar_exudate, tender_anterior_cervical_adenopathy, fever_history, absence_of_cough)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 3 && age <= 14) {
    score <- score + 1
  } else if (age >= 15 && age <= 44) {
    score <- score + 0
  } else if (age >= 45) {
    score <- score - 1
  }

  if (tonsillar_exudate == 1) {
    score <- score + 1
  }

  if (tender_anterior_cervical_adenopathy == 1) {
    score <- score + 1
  }

  if (fever_history == 1) {
    score <- score + 1
  }

  if (absence_of_cough == 1) {
    score <- score + 1
  }

  risk <- ""
  management <- ""

  if (score <= 0) {
    risk <- "1-2.5%"
    management <- "No further testing or antibiotics."
  } else if (score == 1) {
    risk <- "5-10%"
    management <- "No further testing or antibiotics."
  } else if (score == 2) {
    risk <- "11-17%"
    management <- "Optional rapid strep test and/or culture. Treat if positive."
  } else if (score == 3) {
    risk <- "28-35%"
    management <- "Rapid strep test and/or culture. Treat if positive."
  } else {
    risk <- "51-53%"
    management <- "Rapid strep test and/or culture. Treat if positive. Empiric antibiotics may be considered depending on local guidelines."
  }

  return(list(
    Centor_Score = score,
    Strep_Probability = risk,
    Management_Guideline = management,
    Inputs = list(
      Age = age,
      Exudate = tonsillar_exudate == 1,
      Adenopathy = tender_anterior_cervical_adenopathy == 1,
      Fever = fever_history == 1,
      No_Cough = absence_of_cough == 1
    )
  ))
}

cns_ipi_score <- function(age, ldh_elevated, ecog_gt_1, stage_iii_iv, extranodal_gt_1, kidney_adrenal_involvement) {

  age <- as.numeric(age)
  ldh_elevated <- as.numeric(ldh_elevated)
  ecog_gt_1 <- as.numeric(ecog_gt_1)
  stage_iii_iv <- as.numeric(stage_iii_iv)
  extranodal_gt_1 <- as.numeric(extranodal_gt_1)
  kidney_adrenal_involvement <- as.numeric(kidney_adrenal_involvement)

  if (any(is.na(c(age, ldh_elevated, ecog_gt_1, stage_iii_iv, extranodal_gt_1, kidney_adrenal_involvement)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age > 60) {
    score <- score + 1
  }

  if (ldh_elevated == 1) {
    score <- score + 1
  }

  if (ecog_gt_1 == 1) {
    score <- score + 1
  }

  if (stage_iii_iv == 1) {
    score <- score + 1
  }

  if (extranodal_gt_1 == 1) {
    score <- score + 1
  }

  if (kidney_adrenal_involvement == 1) {
    score <- score + 1
  }

  risk_group <- ""
  cns_relapse_risk_2yr <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    cns_relapse_risk_2yr <- "0.6%"
  } else if (score <= 3) {
    risk_group <- "Intermediate Risk"
    cns_relapse_risk_2yr <- "3.4%"
  } else {
    risk_group <- "High Risk"
    cns_relapse_risk_2yr <- "10.2%"
  }

  return(list(
    CNS_IPI_Score = score,
    Risk_Group = risk_group,
    Est_2yr_CNS_Relapse_Risk = cns_relapse_risk_2yr,
    Inputs = list(
      Age_gt_60 = age > 60,
      LDH_Elevated = ldh_elevated == 1,
      ECOG_Performance_gt_1 = ecog_gt_1 == 1,
      Stage_III_IV = stage_iii_iv == 1,
      Extranodal_Sites_gt_1 = extranodal_gt_1 == 1,
      Kidney_Adrenal_Inv = kidney_adrenal_involvement == 1
    )
  ))
}

cerebral_perfusion_pressure <- function(mean_arterial_pressure, intracranial_pressure) {

  map <- as.numeric(mean_arterial_pressure)
  icp <- as.numeric(intracranial_pressure)

  if (any(is.na(c(map, icp)))) {
    stop("Error: MAP and ICP must be numeric.")
  }

  cpp <- map - icp

  interpretation <- ""
  if (cpp < 50) {
    interpretation <- "Low CPP (<50 mmHg). Risk of hypoperfusion and cerebral ischemia."
  } else if (cpp >= 60 && cpp <= 70) {
    interpretation <- "Target Range (60-70 mmHg) for TBI management per Brain Trauma Foundation."
  } else if (cpp > 70) {
    interpretation <- "High CPP (>70 mmHg). Risk of hyperemia and ARDS in TBI patients."
  } else {
    interpretation <- "Borderline (50-60 mmHg). Monitor closely."
  }

  return(list(
    CPP_mmHg = round(cpp, 1),
    Interpretation = interpretation,
    Inputs = list(
      MAP = map,
      ICP = icp
    )
  ))
}

chads_65_score <- function(age, chf, hypertension, diabetes, stroke_tia) {

  age <- as.numeric(age)
  chf <- as.numeric(chf)
  hypertension <- as.numeric(hypertension)
  diabetes <- as.numeric(diabetes)
  stroke_tia <- as.numeric(stroke_tia)

  if (any(is.na(c(age, chf, hypertension, diabetes, stroke_tia)))) {
    stop("Error: All inputs must be numeric (0 or 1, and Age).")
  }

  score <- 0

  if (chf == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (age >= 65) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  if (stroke_tia == 1) {
    score <- score + 2
  }

  recommendation <- ""

  if (score == 0) {
    recommendation <- "No antithrombotic therapy is recommended."
  } else {
    recommendation <- "Oral Anticoagulant (OAC) is recommended (DOAC preferred over Warfarin)."
  }

  return(list(
    CHADS_65_Score = score,
    Recommendation = recommendation,
    Inputs = list(
      Age_ge_65 = age >= 65,
      CHF = chf == 1,
      Hypertension = hypertension == 1,
      Diabetes = diabetes == 1,
      Stroke_TIA = stroke_tia == 1
    )
  ))
}

chads2_score <- function(chf, hypertension, age, diabetes, stroke_tia) {

  chf <- as.numeric(chf)
  hypertension <- as.numeric(hypertension)
  age <- as.numeric(age)
  diabetes <- as.numeric(diabetes)
  stroke_tia <- as.numeric(stroke_tia)

  if (any(is.na(c(chf, hypertension, age, diabetes, stroke_tia)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (chf == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (age >= 75) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  if (stroke_tia == 1) {
    score <- score + 2
  }

  risk_level <- ""
  stroke_risk <- ""

  if (score == 0) {
    risk_level <- "Low Risk"
    stroke_risk <- "1.9%"
  } else if (score == 1) {
    risk_level <- "Moderate Risk"
    stroke_risk <- "2.8%"
  } else if (score == 2) {
    risk_level <- "Moderate Risk"
    stroke_risk <- "4.0%"
  } else if (score == 3) {
    risk_level <- "High Risk"
    stroke_risk <- "5.9%"
  } else if (score == 4) {
    risk_level <- "High Risk"
    stroke_risk <- "8.5%"
  } else if (score == 5) {
    risk_level <- "High Risk"
    stroke_risk <- "12.5%"
  } else {
    risk_level <- "High Risk"
    stroke_risk <- "18.2%"
  }

  return(list(
    CHADS2_Score = score,
    Risk_Category = risk_level,
    Annual_Stroke_Risk = stroke_risk,
    Inputs = list(
      CHF = chf == 1,
      Hypertension = hypertension == 1,
      Age_ge_75 = age >= 75,
      Diabetes = diabetes == 1,
      Stroke_TIA = stroke_tia == 1
    )
  ))
}

chalice_head_injury_rule <- function(age_years, gcs_score, witnessed_loc_gt_5min, amnesia_gt_5min, abnormal_drowsiness, vomiting_ge_3, suspicion_nai, seizure, high_speed_mva, fall_gt_3m, high_speed_object, open_depressed_fx_tense_fontanelle, basal_skull_fx_signs, focal_neuro_deficit, scalp_swelling_laceration_gt_5cm) {

  age_years <- as.numeric(age_years)
  gcs_score <- as.numeric(gcs_score)
  witnessed_loc_gt_5min <- as.numeric(witnessed_loc_gt_5min)
  amnesia_gt_5min <- as.numeric(amnesia_gt_5min)
  abnormal_drowsiness <- as.numeric(abnormal_drowsiness)
  vomiting_ge_3 <- as.numeric(vomiting_ge_3)
  suspicion_nai <- as.numeric(suspicion_nai)
  seizure <- as.numeric(seizure)
  high_speed_mva <- as.numeric(high_speed_mva)
  fall_gt_3m <- as.numeric(fall_gt_3m)
  high_speed_object <- as.numeric(high_speed_object)
  open_depressed_fx_tense_fontanelle <- as.numeric(open_depressed_fx_tense_fontanelle)
  basal_skull_fx_signs <- as.numeric(basal_skull_fx_signs)
  focal_neuro_deficit <- as.numeric(focal_neuro_deficit)
  scalp_swelling_laceration_gt_5cm <- as.numeric(scalp_swelling_laceration_gt_5cm)

  if (any(is.na(c(age_years, gcs_score, witnessed_loc_gt_5min, amnesia_gt_5min, abnormal_drowsiness, vomiting_ge_3, suspicion_nai, seizure, high_speed_mva, fall_gt_3m, high_speed_object, open_depressed_fx_tense_fontanelle, basal_skull_fx_signs, focal_neuro_deficit, scalp_swelling_laceration_gt_5cm)))) {
    stop("Error: All inputs must be numeric.")
  }

  ct_indicated <- FALSE
  reasons <- c()

  if (witnessed_loc_gt_5min == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Witnessed LOC > 5 min")
  }

  if (amnesia_gt_5min == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Amnesia > 5 min")
  }

  if (abnormal_drowsiness == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Abnormal drowsiness")
  }

  if (vomiting_ge_3 == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Vomiting >= 3 times")
  }

  if (suspicion_nai == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Suspicion of Non-Accidental Injury")
  }

  if (seizure == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Seizure after injury")
  }

  if (high_speed_mva == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "High-speed MVA (>40mph)")
  }

  if (fall_gt_3m == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Fall > 3m")
  }

  if (high_speed_object == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "High-speed projectile/object")
  }

  if (age_years < 1) {
    if (gcs_score < 15) {
      ct_indicated <- TRUE
      reasons <- c(reasons, "GCS < 15 (Infant < 1 year)")
    }
    if (scalp_swelling_laceration_gt_5cm == 1) {
      ct_indicated <- TRUE
      reasons <- c(reasons, "Scalp bruise/swelling/lac > 5cm (Infant < 1 year)")
    }
  } else {
    if (gcs_score < 14) {
      ct_indicated <- TRUE
      reasons <- c(reasons, "GCS < 14 (Child >= 1 year)")
    }
  }

  if (open_depressed_fx_tense_fontanelle == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Suspected open/depressed fracture or tense fontanelle")
  }

  if (basal_skull_fx_signs == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Signs of basal skull fracture")
  }

  if (focal_neuro_deficit == 1) {
    ct_indicated <- TRUE
    reasons <- c(reasons, "Focal neurological deficit")
  }

  recommendation <- ""
  if (ct_indicated) {
    recommendation <- "CT Head Indicated (High Risk)"
  } else {
    recommendation <- "CT Head Not Indicated (Low Risk)"
  }

  return(list(
    Recommendation = recommendation,
    Risk_Factors_Present = if (length(reasons) > 0) reasons else "None",
    Inputs = list(
      Age = age_years,
      GCS = gcs_score,
      History_Factors = sum(witnessed_loc_gt_5min, amnesia_gt_5min, abnormal_drowsiness, vomiting_ge_3, suspicion_nai, seizure),
      Mechanism_Factors = sum(high_speed_mva, fall_gt_3m, high_speed_object),
      Exam_Factors = sum(open_depressed_fx_tense_fontanelle, basal_skull_fx_signs, focal_neuro_deficit, scalp_swelling_laceration_gt_5cm)
    )
  ))
}

charlson_comorbidity_index <- function(age, myocardial_infarction, chf, pvd, cva_tia, dementia, copd, connective_tissue_disease, peptic_ulcer_disease, liver_disease, diabetes, hemiplegia, renal_disease, solid_tumor, leukemia, lymphoma, aids) {

  age <- as.numeric(age)
  myocardial_infarction <- as.numeric(myocardial_infarction)
  chf <- as.numeric(chf)
  pvd <- as.numeric(pvd)
  cva_tia <- as.numeric(cva_tia)
  dementia <- as.numeric(dementia)
  copd <- as.numeric(copd)
  connective_tissue_disease <- as.numeric(connective_tissue_disease)
  peptic_ulcer_disease <- as.numeric(peptic_ulcer_disease)
  liver_disease <- tolower(as.character(liver_disease))
  diabetes <- tolower(as.character(diabetes))
  hemiplegia <- as.numeric(hemiplegia)
  renal_disease <- as.numeric(renal_disease)
  solid_tumor <- tolower(as.character(solid_tumor))
  leukemia <- as.numeric(leukemia)
  lymphoma <- as.numeric(lymphoma)
  aids <- as.numeric(aids)

  if (any(is.na(c(age, myocardial_infarction, chf, pvd, cva_tia, dementia, copd, connective_tissue_disease, peptic_ulcer_disease, hemiplegia, renal_disease, leukemia, lymphoma, aids)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  score <- 0

  if (age >= 50 && age <= 59) {
    score <- score + 1
  } else if (age >= 60 && age <= 69) {
    score <- score + 2
  } else if (age >= 70 && age <= 79) {
    score <- score + 3
  } else if (age >= 80) {
    score <- score + 4
  }

  if (myocardial_infarction == 1) score <- score + 1
  if (chf == 1) score <- score + 1
  if (pvd == 1) score <- score + 1
  if (cva_tia == 1) score <- score + 1
  if (dementia == 1) score <- score + 1
  if (copd == 1) score <- score + 1
  if (connective_tissue_disease == 1) score <- score + 1
  if (peptic_ulcer_disease == 1) score <- score + 1

  if (liver_disease == "mild") {
    score <- score + 1
  } else if (liver_disease == "moderate_severe") {
    score <- score + 3
  }

  if (diabetes == "uncomplicated") {
    score <- score + 1
  } else if (diabetes == "end_organ_damage") {
    score <- score + 2
  }

  if (hemiplegia == 1) score <- score + 2
  if (renal_disease == 1) score <- score + 2

  if (solid_tumor == "localized") {
    score <- score + 2
  } else if (solid_tumor == "metastatic") {
    score <- score + 6
  }

  if (leukemia == 1) score <- score + 2
  if (lymphoma == 1) score <- score + 2
  if (aids == 1) score <- score + 6

  est_10yr_survival <- ""
  if (score == 0) {
    est_10yr_survival <- "98%"
  } else if (score == 1) {
    est_10yr_survival <- "96%"
  } else if (score == 2) {
    est_10yr_survival <- "90%"
  } else if (score == 3) {
    est_10yr_survival <- "77%"
  } else if (score == 4) {
    est_10yr_survival <- "53%"
  } else {
    est_10yr_survival <- "21%"
  }

  return(list(
    CCI_Score = score,
    Estimated_10_Year_Survival = est_10yr_survival,
    Inputs = list(
      Age = age,
      MI = myocardial_infarction == 1,
      CHF = chf == 1,
      PVD = pvd == 1,
      CVA_TIA = cva_tia == 1,
      Dementia = dementia == 1,
      COPD = copd == 1,
      CTD = connective_tissue_disease == 1,
      PUD = peptic_ulcer_disease == 1,
      Liver = liver_disease,
      Diabetes = diabetes,
      Hemiplegia = hemiplegia == 1,
      Renal = renal_disease == 1,
      Tumor = solid_tumor,
      Leukemia = leukemia == 1,
      Lymphoma = lymphoma == 1,
      AIDS = aids == 1
    )
  ))
}

cha2ds2_va_score <- function(age, chf_history, hypertension, diabetes, stroke_tia_thromboembolism, vascular_disease) {

  age <- as.numeric(age)
  chf_history <- as.numeric(chf_history)
  hypertension <- as.numeric(hypertension)
  diabetes <- as.numeric(diabetes)
  stroke_tia_thromboembolism <- as.numeric(stroke_tia_thromboembolism)
  vascular_disease <- as.numeric(vascular_disease)

  if (any(is.na(c(age, chf_history, hypertension, diabetes, stroke_tia_thromboembolism, vascular_disease)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 75) {
    score <- score + 2
  } else if (age >= 65) {
    score <- score + 1
  }

  if (chf_history == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  if (stroke_tia_thromboembolism == 1) {
    score <- score + 2
  }

  if (vascular_disease == 1) {
    score <- score + 1
  }

  annual_stroke_risk <- ""
  risk_category <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    annual_stroke_risk <- "0.5%"
  } else if (score == 1) {
    risk_category <- "Intermediate Risk"
    annual_stroke_risk <- "1.5%"
  } else if (score == 2) {
    risk_category <- "Moderate/High Risk"
    annual_stroke_risk <- "2.9%"
  } else if (score == 3) {
    risk_category <- "Moderate/High Risk"
    annual_stroke_risk <- "5.1%"
  } else if (score == 4) {
    risk_category <- "High Risk"
    annual_stroke_risk <- "7.3%"
  } else if (score == 5) {
    risk_category <- "High Risk"
    annual_stroke_risk <- "11.2%"
  } else if (score == 6) {
    risk_category <- "High Risk"
    annual_stroke_risk <- "15.5%"
  } else {
    risk_category <- "High Risk"
    annual_stroke_risk <- "> 15%"
  }

  recommendation <- ""
  if (score == 0) {
    recommendation <- "No anticoagulation recommended."
  } else if (score == 1) {
    recommendation <- "Consider anticoagulation based on clinical judgment and patient preference."
  } else {
    recommendation <- "Oral anticoagulation is recommended."
  }

  return(list(
    CHA2DS2_VA_Score = score,
    Risk_Category = risk_category,
    Annual_Stroke_Risk_No_Anticoagulation = annual_stroke_risk,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      CHF = chf_history == 1,
      Hypertension = hypertension == 1,
      Diabetes = diabetes == 1,
      Stroke_TIA = stroke_tia_thromboembolism == 1,
      Vascular_Disease = vascular_disease == 1
    )
  ))
}

cha2ds2_vasc_score <- function(age, sex, chf_history, hypertension_history, stroke_tia_history, vascular_disease_history, diabetes_history) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  chf_history <- as.numeric(chf_history)
  hypertension_history <- as.numeric(hypertension_history)
  stroke_tia_history <- as.numeric(stroke_tia_history)
  vascular_disease_history <- as.numeric(vascular_disease_history)
  diabetes_history <- as.numeric(diabetes_history)

  if (any(is.na(c(age, chf_history, hypertension_history, stroke_tia_history, vascular_disease_history, diabetes_history)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (chf_history == 1) {
    score <- score + 1
  }

  if (hypertension_history == 1) {
    score <- score + 1
  }

  if (age >= 75) {
    score <- score + 2
  } else if (age >= 65) {
    score <- score + 1
  }

  if (diabetes_history == 1) {
    score <- score + 1
  }

  if (stroke_tia_history == 1) {
    score <- score + 2
  }

  if (vascular_disease_history == 1) {
    score <- score + 1
  }

  if (sex == "female") {
    score <- score + 1
  }

  stroke_risk <- ""
  recommendation <- ""

  if (score == 0) stroke_risk <- "0%"
  else if (score == 1) stroke_risk <- "1.3%"
  else if (score == 2) stroke_risk <- "2.2%"
  else if (score == 3) stroke_risk <- "3.2%"
  else if (score == 4) stroke_risk <- "4.0%"
  else if (score == 5) stroke_risk <- "6.7%"
  else if (score == 6) stroke_risk <- "9.8%"
  else if (score == 7) stroke_risk <- "9.6%"
  else if (score == 8) stroke_risk <- "6.7%"
  else stroke_risk <- "15.2%"

  if (sex == "male") {
    if (score == 0) {
      recommendation <- "Low risk. No anticoagulation recommended."
    } else if (score == 1) {
      recommendation <- "Moderate risk. Consider oral anticoagulation."
    } else {
      recommendation <- "High risk. Oral anticoagulation recommended."
    }
  } else {
    if (score == 1) {
      recommendation <- "Low risk. No anticoagulation recommended."
    } else if (score == 2) {
      recommendation <- "Moderate risk. Consider oral anticoagulation."
    } else {
      recommendation <- "High risk. Oral anticoagulation recommended."
    }
  }

  return(list(
    CHA2DS2_VASc_Score = score,
    Annual_Stroke_Risk_No_Anticoagulation = stroke_risk,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      CHF = chf_history == 1,
      Hypertension = hypertension_history == 1,
      Stroke_TIA = stroke_tia_history == 1,
      Vascular_Disease = vascular_disease_history == 1,
      Diabetes = diabetes_history == 1
    )
  ))
}

crash_chemo_toxicity_score <- function(diastolic_bp, iadl_score, ldh_elevated, chemo_risk_max2, ecog_ps, mmse_score, mna_score) {

  diastolic_bp <- as.numeric(diastolic_bp)
  iadl_score <- as.numeric(iadl_score)
  ldh_elevated <- as.numeric(ldh_elevated)
  ecog_ps <- as.numeric(ecog_ps)
  mmse_score <- as.numeric(mmse_score)
  mna_score <- as.numeric(mna_score)
  chemo_risk_max2 <- tolower(as.character(chemo_risk_max2))

  if (any(is.na(c(diastolic_bp, iadl_score, ldh_elevated, ecog_ps, mmse_score, mna_score)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  if (!chemo_risk_max2 %in% c("low", "intermediate", "high")) {
    stop("Error: Chemo Risk must be 'low', 'intermediate', or 'high'.")
  }

  hema_score <- 0

  if (diastolic_bp <= 72) {
    hema_score <- hema_score + 2
  }

  if (iadl_score < 28) {
    hema_score <- hema_score + 2
  }

  if (ldh_elevated == 1) {
    hema_score <- hema_score + 2
  }

  if (chemo_risk_max2 == "intermediate" || chemo_risk_max2 == "high") {
    hema_score <- hema_score + 1
  }

  non_hema_score <- 0

  if (ecog_ps == 1 || ecog_ps == 2) {
    non_hema_score <- non_hema_score + 1
  } else if (ecog_ps >= 3) {
    non_hema_score <- non_hema_score + 2
  }

  if (mmse_score < 30) {
    non_hema_score <- non_hema_score + 1
  }

  if (mna_score < 28) {
    non_hema_score <- non_hema_score + 2
  }

  if (chemo_risk_max2 == "intermediate") {
    non_hema_score <- non_hema_score + 2
  } else if (chemo_risk_max2 == "high") {
    non_hema_score <- non_hema_score + 4
  }

  hema_risk_category <- ""
  hema_toxicity_rate <- ""

  if (hema_score <= 2) {
    hema_risk_category <- "Low Risk"
    hema_toxicity_rate <- "7%"
  } else if (hema_score == 3) {
    hema_risk_category <- "Intermediate Risk"
    hema_toxicity_rate <- "23%"
  } else {
    hema_risk_category <- "High Risk"
    hema_toxicity_rate <- "50%"
  }

  non_hema_risk_category <- ""
  non_hema_toxicity_rate <- ""

  if (non_hema_score <= 3) {
    non_hema_risk_category <- "Low Risk"
    non_hema_toxicity_rate <- "23%"
  } else if (non_hema_score <= 6) {
    non_hema_risk_category <- "Intermediate Risk"
    non_hema_toxicity_rate <- "43%"
  } else {
    non_hema_risk_category <- "High Risk"
    non_hema_toxicity_rate <- "73%"
  }

  combined_score <- hema_score + non_hema_score

  return(list(
    Scores = list(
      Hematologic_Score = hema_score,
      Non_Hematologic_Score = non_hema_score,
      Combined_Score = combined_score
    ),
    Hematologic_Toxicity_Risk = list(
      Risk_Category = hema_risk_category,
      Est_Grade4_Toxicity = hema_toxicity_rate
    ),
    Non_Hematologic_Toxicity_Risk = list(
      Risk_Category = non_hema_risk_category,
      Est_Grade3_4_Toxicity = non_hema_toxicity_rate
    ),
    Inputs = list(
      DBP = diastolic_bp,
      IADL = iadl_score,
      LDH_High = ldh_elevated == 1,
      ECOG = ecog_ps,
      MMSE = mmse_score,
      MNA = mna_score,
      Chemo_Risk = chemo_risk_max2
    )
  ))
}

child_pugh_score <- function(bilirubin, albumin, inr, ascites, encephalopathy, pbc_or_psc = 0, bilirubin_units = "mg/dL", albumin_units = "g/dL") {

  bilirubin <- as.numeric(bilirubin)
  albumin <- as.numeric(albumin)
  inr <- as.numeric(inr)
  ascites <- tolower(as.character(ascites))
  encephalopathy <- tolower(as.character(encephalopathy))
  pbc_or_psc <- as.numeric(pbc_or_psc)

  if (any(is.na(c(bilirubin, albumin, inr)))) {
    stop("Error: Bilirubin, Albumin, and INR must be numeric.")
  }

  if (!ascites %in% c("none", "slight", "moderate")) {
    stop("Error: Ascites must be 'none', 'slight', or 'moderate'.")
  }

  if (!encephalopathy %in% c("none", "grade_1_2", "grade_3_4")) {
    stop("Error: Encephalopathy must be 'none', 'grade_1_2', or 'grade_3_4'.")
  }

  score <- 0

  bili_mg_dl <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_mg_dl <- bilirubin / 17.1
  }

  if (pbc_or_psc == 1) {
    if (bili_mg_dl < 4) {
      score <- score + 1
    } else if (bili_mg_dl >= 4 && bili_mg_dl <= 10) {
      score <- score + 2
    } else {
      score <- score + 3
    }
  } else {
    if (bili_mg_dl < 2) {
      score <- score + 1
    } else if (bili_mg_dl >= 2 && bili_mg_dl <= 3) {
      score <- score + 2
    } else {
      score <- score + 3
    }
  }

  alb_g_dl <- albumin
  if (albumin_units == "g/L") {
    alb_g_dl <- albumin / 10
  }

  if (alb_g_dl > 3.5) {
    score <- score + 1
  } else if (alb_g_dl >= 2.8 && alb_g_dl <= 3.5) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (inr < 1.7) {
    score <- score + 1
  } else if (inr >= 1.7 && inr <= 2.3) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (ascites == "none") {
    score <- score + 1
  } else if (ascites == "slight") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (encephalopathy == "none") {
    score <- score + 1
  } else if (encephalopathy == "grade_1_2") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  class_grade <- ""
  survival_1yr <- ""
  survival_2yr <- ""

  if (score <= 6) {
    class_grade <- "Class A"
    survival_1yr <- "100%"
    survival_2yr <- "85%"
  } else if (score <= 9) {
    class_grade <- "Class B"
    survival_1yr <- "80%"
    survival_2yr <- "60%"
  } else {
    class_grade <- "Class C"
    survival_1yr <- "45%"
    survival_2yr <- "35%"
  }

  return(list(
    Child_Pugh_Score = score,
    Class = class_grade,
    Life_Expectancy_1_Year = survival_1yr,
    Life_Expectancy_2_Year = survival_2yr,
    Inputs = list(
      Bilirubin = bili_mg_dl,
      Albumin = alb_g_dl,
      INR = inr,
      Ascites = ascites,
      Encephalopathy = encephalopathy,
      PBC_PSC_Modified = pbc_or_psc == 1
    )
  ))
}

cheops_pain_scale <- function(cry, facial, verbal, torso, touch, legs) {

  cry <- as.numeric(cry)
  facial <- as.numeric(facial)
  verbal <- as.numeric(verbal)
  torso <- as.numeric(torso)
  touch <- as.numeric(touch)
  legs <- as.numeric(legs)

  if (any(is.na(c(cry, facial, verbal, torso, touch, legs)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (cry < 1 || cry > 3) stop("Error: Cry score must be 1, 2, or 3.")
  if (facial < 0 || facial > 2) stop("Error: Facial score must be 0, 1, or 2.")
  if (verbal < 0 || verbal > 2) stop("Error: Verbal score must be 0, 1, or 2.")
  if (torso < 1 || torso > 2) stop("Error: Torso score must be 1 or 2.")
  if (touch < 1 || touch > 2) stop("Error: Touch score must be 1 or 2.")
  if (legs < 1 || legs > 2) stop("Error: Legs score must be 1 or 2.")

  score <- cry + facial + verbal + torso + touch + legs

  interpretation <- ""
  if (score >= 6) {
    interpretation <- "Significant pain likely. Consider analgesia."
  } else {
    interpretation <- "Pain likely controlled or minimal."
  }

  return(list(
    CHEOPS_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Cry = cry,
      Facial = facial,
      Verbal = verbal,
      Torso = torso,
      Touch = touch,
      Legs = legs
    )
  ))
}

chip_head_injury_rule <- function(age, gcs, vomiting, amnesia, skull_fracture_signs, neurological_deficit, seizure, coagulopathy, dangerous_mechanism) {

  age <- as.numeric(age)
  gcs <- as.numeric(gcs)
  vomiting <- as.numeric(vomiting)
  amnesia <- as.numeric(amnesia)
  skull_fracture_signs <- as.numeric(skull_fracture_signs)
  neurological_deficit <- as.numeric(neurological_deficit)
  seizure <- as.numeric(seizure)
  coagulopathy <- as.numeric(coagulopathy)
  dangerous_mechanism <- as.numeric(dangerous_mechanism)

  if (any(is.na(c(age, gcs, vomiting, amnesia, skull_fracture_signs, neurological_deficit, seizure, coagulopathy, dangerous_mechanism)))) {
    stop("Error: All inputs must be numeric.")
  }

  ct_indicated <- FALSE
  risk_factors <- c()

  if (age >= 60) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Age >= 60")
  }

  if (gcs < 15) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "GCS < 15")
  }

  if (vomiting == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Vomiting")
  }

  if (amnesia == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Amnesia (Retrograde > 30 min or Anterograde > 4 hrs)")
  }

  if (skull_fracture_signs == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Signs of Skull Fracture")
  }

  if (neurological_deficit == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Neurological Deficit")
  }

  if (seizure == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Post-traumatic Seizure")
  }

  if (coagulopathy == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Coagulopathy/Anticoagulant Use")
  }

  if (dangerous_mechanism == 1) {
    ct_indicated <- TRUE
    risk_factors <- c(risk_factors, "Dangerous Mechanism")
  }

  recommendation <- ""
  if (ct_indicated) {
    recommendation <- "CT Head Indicated"
  } else {
    recommendation <- "CT Head Not Indicated (Low Risk)"
  }

  return(list(
    Recommendation = recommendation,
    Risk_Factors_Present = if (length(risk_factors) > 0) risk_factors else "None",
    Inputs = list(
      Age = age,
      GCS = gcs,
      Vomiting = vomiting == 1,
      Amnesia = amnesia == 1,
      Skull_Fx = skull_fracture_signs == 1,
      Neuro_Deficit = neurological_deficit == 1,
      Seizure = seizure == 1,
      Coagulopathy = coagulopathy == 1,
      Mechanism = dangerous_mechanism == 1
    )
  ))
}

choles_score_duration <- function(sex, age, history_acute_cholecystitis, bmi, asa_class_ge_3) {

  sex <- tolower(as.character(sex))
  age <- as.numeric(age)
  history_acute_cholecystitis <- as.numeric(history_acute_cholecystitis)
  bmi <- as.numeric(bmi)
  asa_class_ge_3 <- as.numeric(asa_class_ge_3)

  if (any(is.na(c(age, history_acute_cholecystitis, bmi, asa_class_ge_3)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (sex == "male") {
    score <- score + 1
  }

  if (age >= 60 && age <= 70) {
    score <- score + 1
  } else if (age > 70) {
    score <- score + 2
  }

  if (history_acute_cholecystitis == 1) {
    score <- score + 1
  }

  if (bmi >= 30 && bmi <= 35) {
    score <- score + 1
  } else if (bmi > 35) {
    score <- score + 2
  }

  if (asa_class_ge_3 == 1) {
    score <- score + 1
  }

  risk_category <- ""
  mean_duration <- ""
  difficulty <- ""

  if (score <= 2) {
    risk_category <- "Low Risk"
    mean_duration <- "~60 minutes"
    difficulty <- "Low probability of difficult cholecystectomy."
  } else if (score <= 4) {
    risk_category <- "Medium Risk"
    mean_duration <- "~80 minutes"
    difficulty <- "Intermediate probability of difficult cholecystectomy."
  } else {
    risk_category <- "High Risk"
    mean_duration <- "~100 minutes"
    difficulty <- "High probability of difficult cholecystectomy. Consider senior surgeon presence."
  }

  return(list(
    CholeS_Score = score,
    Risk_Category = risk_category,
    Est_Duration = mean_duration,
    Difficulty_Prediction = difficulty,
    Inputs = list(
      Male_Sex = sex == "male",
      Age = age,
      Hx_Acute_Cholecystitis = history_acute_cholecystitis == 1,
      BMI = bmi,
      ASA_III_IV_V = asa_class_ge_3 == 1
    )
  ))
}

ckid_u25_egfr <- function(height_cm, creatinine, creatinine_units = "mg/dL") {

  height_cm <- as.numeric(height_cm)
  creatinine <- as.numeric(creatinine)

  if (any(is.na(c(height_cm, creatinine)))) {
    stop("Error: Height and Creatinine must be numeric.")
  }

  if (height_cm <= 0 || creatinine <= 0) {
    stop("Error: Inputs must be positive numbers.")
  }

  scr_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    scr_mg_dl <- creatinine / 88.4
  }

  k_value <- 0.413

  egfr <- k_value * (height_cm / scr_mg_dl)

  return(list(
    eGFR_mL_min_1_73m2 = round(egfr, 1),
    Method = "CKiD U25 (Bedside Schwartz 2009, validated for 1-25 years)",
    Inputs = list(
      Height_cm = height_cm,
      Creatinine_mg_dL = round(scr_mg_dl, 2)
    )
  ))
}

cp_sss_stroke_score <- function(conjugate_gaze_deviation, severe_arm_weakness, loc_questions_incorrect) {

  conjugate_gaze_deviation <- as.numeric(conjugate_gaze_deviation)
  severe_arm_weakness <- as.numeric(severe_arm_weakness)
  loc_questions_incorrect <- as.numeric(loc_questions_incorrect)

  if (any(is.na(c(conjugate_gaze_deviation, severe_arm_weakness, loc_questions_incorrect)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (conjugate_gaze_deviation == 1) {
    score <- score + 2
  }

  if (severe_arm_weakness == 1) {
    score <- score + 1
  }

  if (loc_questions_incorrect == 1) {
    score <- score + 1
  }

  likelihood_lvo <- ""
  recommendation <- ""

  if (score >= 2) {
    likelihood_lvo <- "High Likelihood of Large Vessel Occlusion (LVO)"
    recommendation <- "Transport to Comprehensive Stroke Center (CSC) if feasible."
  } else {
    likelihood_lvo <- "Low Likelihood of Large Vessel Occlusion (LVO)"
    recommendation <- "Transport to nearest Stroke Center (Primary or Comprehensive)."
  }

  return(list(
    CP_SSS_Score = score,
    LVO_Likelihood = likelihood_lvo,
    Recommendation = recommendation,
    Inputs = list(
      Gaze_Deviation = conjugate_gaze_deviation == 1,
      Severe_Arm_Weakness = severe_arm_weakness == 1,
      LOC_Incorrect = loc_questions_incorrect == 1
    )
  ))
}

cp_aki_risk_score <- function(age_ge_61, systolic_bp_gt_140, low_hemoglobin, low_albumin, low_potassium, low_calcium, low_magnesium, high_creatinine_dose, high_cisplatin_dose) {

  age_ge_61 <- as.numeric(age_ge_61)
  systolic_bp_gt_140 <- as.numeric(systolic_bp_gt_140)
  low_hemoglobin <- as.numeric(low_hemoglobin)
  low_albumin <- as.numeric(low_albumin)
  low_potassium <- as.numeric(low_potassium)
  low_calcium <- as.numeric(low_calcium)
  low_magnesium <- as.numeric(low_magnesium)
  high_creatinine_dose <- as.numeric(high_creatinine_dose)
  high_cisplatin_dose <- as.numeric(high_cisplatin_dose)

  if (any(is.na(c(age_ge_61, systolic_bp_gt_140, low_hemoglobin, low_albumin, low_potassium, low_calcium, low_magnesium, high_creatinine_dose, high_cisplatin_dose)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (age_ge_61 == 1) score <- score + 1.5
  if (systolic_bp_gt_140 == 1) score <- score + 1.5
  if (low_hemoglobin == 1) score <- score + 1.0
  if (low_albumin == 1) score <- score + 2.5
  if (low_potassium == 1) score <- score + 2.0
  if (low_calcium == 1) score <- score + 1.5
  if (low_magnesium == 1) score <- score + 1.5
  if (high_creatinine_dose == 1) score <- score + 1.5
  if (high_cisplatin_dose == 1) score <- score + 2.5

  risk_group <- ""
  probability_aki <- ""

  if (score <= 3.5) {
    risk_group <- "Low Risk"
    probability_aki <- "< 0.1%"
  } else if (score > 3.5 && score <= 8.0) {
    risk_group <- "Moderate Risk"
    probability_aki <- "3.0%"
  } else {
    risk_group <- "High Risk"
    probability_aki <- "6.0%"
  }

  return(list(
    CP_AKI_Score = score,
    Risk_Group = risk_group,
    Est_Risk_AKI = probability_aki,
    Inputs = list(
      Age_ge_61 = age_ge_61 == 1,
      SBP_gt_140 = systolic_bp_gt_140 == 1,
      Low_Hgb = low_hemoglobin == 1,
      Low_Alb = low_albumin == 1,
      Low_K = low_potassium == 1,
      Low_Ca = low_calcium == 1,
      Low_Mg = low_magnesium == 1,
      High_Cr_Dose = high_creatinine_dose == 1,
      High_Cisplatin = high_cisplatin_dose == 1
    )
  ))
}

ciwa_ar_alcohol_withdrawal <- function(nausea_vomiting, tremor, paroxysmal_sweats, anxiety, agitation, tactile_disturbances, auditory_disturbances, visual_disturbances, headache, orientation) {

  nausea_vomiting <- as.numeric(nausea_vomiting)
  tremor <- as.numeric(tremor)
  paroxysmal_sweats <- as.numeric(paroxysmal_sweats)
  anxiety <- as.numeric(anxiety)
  agitation <- as.numeric(agitation)
  tactile_disturbances <- as.numeric(tactile_disturbances)
  auditory_disturbances <- as.numeric(auditory_disturbances)
  visual_disturbances <- as.numeric(visual_disturbances)
  headache <- as.numeric(headache)
  orientation <- as.numeric(orientation)

  if (any(is.na(c(nausea_vomiting, tremor, paroxysmal_sweats, anxiety, agitation, tactile_disturbances, auditory_disturbances, visual_disturbances, headache, orientation)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(nausea_vomiting, tremor, paroxysmal_sweats, anxiety, agitation, tactile_disturbances, auditory_disturbances, visual_disturbances, headache) < 0) ||
      any(c(nausea_vomiting, tremor, paroxysmal_sweats, anxiety, agitation, tactile_disturbances, auditory_disturbances, visual_disturbances, headache) > 7)) {
    stop("Error: Items 1-9 must be scored between 0 and 7.")
  }

  if (orientation < 0 || orientation > 4) {
    stop("Error: Orientation score must be between 0 and 4.")
  }

  total_score <- nausea_vomiting + tremor + paroxysmal_sweats + anxiety + agitation + tactile_disturbances + auditory_disturbances + visual_disturbances + headache + orientation

  severity <- ""
  management <- ""

  if (total_score < 8) {
    severity <- "Mild Withdrawal"
    management <- "Supportive care. Pharmacotherapy typically not required."
  } else if (total_score <= 15) {
    severity <- "Moderate Withdrawal"
    management <- "Pharmacotherapy indicated (e.g., Benzodiazepines)."
  } else {
    severity <- "Severe Withdrawal"
    management <- "Aggressive pharmacotherapy indicated. High risk for complications (seizures, DTs)."
  }

  return(list(
    CIWA_Ar_Score = total_score,
    Severity_Category = severity,
    Management_Guideline = management,
    Inputs = list(
      Nausea = nausea_vomiting,
      Tremor = tremor,
      Sweats = paroxysmal_sweats,
      Anxiety = anxiety,
      Agitation = agitation,
      Tactile = tactile_disturbances,
      Auditory = auditory_disturbances,
      Visual = visual_disturbances,
      Headache = headache,
      Orientation = orientation
    )
  ))
}

ckd_prediction_hiv <- function(age, sex, iv_drug_user, baseline_egfr, current_cd4, nadir_cd4, viral_load, hepatitis_c, history_cvd, hypertension, diabetes) {

  age <- as.numeric(age)
  baseline_egfr <- as.numeric(baseline_egfr)
  current_cd4 <- as.numeric(current_cd4)
  nadir_cd4 <- as.numeric(nadir_cd4)
  viral_load <- as.numeric(viral_load)

  iv_drug_user <- as.numeric(iv_drug_user)
  hepatitis_c <- as.numeric(hepatitis_c)
  history_cvd <- as.numeric(history_cvd)
  hypertension <- as.numeric(hypertension)
  diabetes <- as.numeric(diabetes)

  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, baseline_egfr, current_cd4, nadir_cd4, viral_load, iv_drug_user, hepatitis_c, history_cvd, hypertension, diabetes)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (baseline_egfr < 60) {
    return(list(
      Status = "Pre-existing CKD",
      Message = "This calculator is designed to predict the development of CKD in patients with baseline eGFR > 60 mL/min/1.73 m^2."
    ))
  }

  score <- 0

  if (sex == "male") {
    score <- score + 1
  }

  if (age >= 40 && age <= 44) {
    score <- score + 2
  } else if (age >= 45 && age <= 49) {
    score <- score + 4
  } else if (age >= 50 && age <= 54) {
    score <- score + 6
  } else if (age >= 55 && age <= 59) {
    score <- score + 9
  } else if (age >= 60) {
    score <- score + 11
  }

  if (iv_drug_user == 1) {
    score <- score + 3
  }

  if (baseline_egfr >= 80 && baseline_egfr < 90) {
    score <- score + 2
  } else if (baseline_egfr >= 70 && baseline_egfr < 80) {
    score <- score + 7
  } else if (baseline_egfr >= 60 && baseline_egfr < 70) {
    score <- score + 15
  }

  if (current_cd4 < 50) {
    score <- score + 3
  } else if (current_cd4 < 200) {
    score <- score + 2
  }

  if (nadir_cd4 < 50) {
    score <- score + 2
  } else if (nadir_cd4 < 200) {
    score <- score + 1
  }

  if (viral_load > 100000) {
    score <- score + 2
  }

  if (hepatitis_c == 1) {
    score <- score + 1
  }

  if (history_cvd == 1) {
    score <- score + 2
  }

  if (hypertension == 1) {
    score <- score + 2
  }

  if (diabetes == 1) {
    score <- score + 4
  }

  risk_category <- ""
  risk_fraction <- ""

  if (score < 0) {
    risk_category <- "Low Risk"
    risk_fraction <- "1:439"
  } else if (score <= 9) {
    risk_category <- "Low Risk"
    risk_fraction <- "1:103"
  } else if (score <= 19) {
    risk_category <- "Medium Risk"
    risk_fraction <- "1:32"
  } else {
    risk_category <- "High Risk"
    risk_fraction <- "1:7"
  }

  return(list(
    DAD_CKD_Score = score,
    Risk_Category = risk_category,
    Five_Year_Risk_Ratio = risk_fraction,
    Inputs = list(
      Age = age,
      Sex = sex,
      IVDU = iv_drug_user == 1,
      eGFR = baseline_egfr,
      CD4 = current_cd4,
      Nadir_CD4 = nadir_cd4,
      Viral_Load_gt_100k = viral_load > 100000,
      HCV = hepatitis_c == 1,
      CVD = history_cvd == 1,
      HTN = hypertension == 1,
      DM = diabetes == 1
    )
  ))
}

ckd_epi_gfr <- function(age, sex, creatinine, cystatin_c = NULL, race_black = FALSE, creatinine_units = "mg/dL", cystatin_units = "mg/L") {

  age <- as.numeric(age)
  creatinine <- as.numeric(creatinine)
  sex <- tolower(as.character(sex))
  race_black <- as.numeric(race_black)

  if (any(is.na(c(age, creatinine)))) {
    stop("Error: Age and Creatinine must be numeric.")
  }

  if (creatinine <= 0) {
    stop("Error: Creatinine must be positive.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  scr <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    scr <- creatinine / 88.4
  }

  kappa <- if (sex == "female") 0.7 else 0.9
  alpha <- if (sex == "female") -0.241 else -0.302
  female_factor <- if (sex == "female") 1.012 else 1.0

  egfr_2021_cr <- 142 * (min(scr / kappa, 1) ^ alpha) * (max(scr / kappa, 1) ^ -1.200) * (0.9938 ^ age) * female_factor

  egfr_2021_cr_cys <- NA
  egfr_2012_cys <- NA

  if (!is.null(cystatin_c) && !is.na(cystatin_c)) {
    scys <- as.numeric(cystatin_c)
    if (scys <= 0) {
      stop("Error: Cystatin C must be positive.")
    }

    k_comb <- if (sex == "female") 0.7 else 0.9
    a_comb <- if (sex == "female") -0.219 else -0.144
    f_comb <- if (sex == "female") 0.963 else 1.0

    egfr_2021_cr_cys <- 135 * (min(scr / k_comb, 1) ^ a_comb) * (max(scr / k_comb, 1) ^ -0.544) * (min(scys / 0.8, 1) ^ -0.323) * (max(scys / 0.8, 1) ^ -0.778) * (0.9961 ^ age) * f_comb

    f_cys <- if (sex == "female") 0.932 else 1.0

    egfr_2012_cys <- 133 * (min(scys / 0.8, 1) ^ -0.499) * (max(scys / 0.8, 1) ^ -1.328) * (0.996 ^ age) * f_cys
  }

  k_09 <- if (sex == "female") 0.7 else 0.9
  a_09 <- if (sex == "female") -0.329 else -0.411
  f_09 <- if (sex == "female") 1.018 else 1.0
  race_09 <- if (race_black == 1) 1.159 else 1.0

  egfr_2009_cr <- 141 * (min(scr / k_09, 1) ^ a_09) * (max(scr / k_09, 1) ^ -1.209) * (0.993 ^ age) * f_09 * race_09

  return(list(
    eGFR_2021_Creatinine = round(egfr_2021_cr, 1),
    eGFR_2021_Creatinine_Cystatin_C = if (!is.na(egfr_2021_cr_cys)) round(egfr_2021_cr_cys, 1) else "Cystatin C Required",
    eGFR_2012_Cystatin_C = if (!is.na(egfr_2012_cys)) round(egfr_2012_cys, 1) else "Cystatin C Required",
    eGFR_2009_Creatinine = round(egfr_2009_cr, 1),
    Inputs = list(
      Age = age,
      Sex = sex,
      Creatinine_mg_dL = round(scr, 2),
      Cystatin_C_mg_L = if (!is.null(cystatin_c)) cystatin_c else "Not Provided",
      Race_Black = race_black == 1
    )
  ))
}

clif_c_aclf_score <- function(age, wbc_count, bilirubin, creatinine, inr, map, use_vasopressors, pao2_fio2_ratio, he_grade, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  wbc_count <- as.numeric(wbc_count)
  bilirubin <- as.numeric(bilirubin)
  creatinine <- as.numeric(creatinine)
  inr <- as.numeric(inr)
  map <- as.numeric(map)
  use_vasopressors <- as.numeric(use_vasopressors)
  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  he_grade <- as.numeric(he_grade)

  if (any(is.na(c(age, wbc_count, bilirubin, creatinine, inr, map, use_vasopressors, pao2_fio2_ratio, he_grade)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_mg_dl <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_mg_dl <- bilirubin / 17.1
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  liver_score <- 1
  if (bili_mg_dl > 12) {
    liver_score <- 3
  } else if (bili_mg_dl >= 6) {
    liver_score <- 2
  }

  kidney_score <- 1
  if (creat_mg_dl > 3.5) {
    kidney_score <- 3
  } else if (creat_mg_dl >= 2) {
    kidney_score <- 2
  }

  brain_score <- 1
  if (he_grade >= 3) {
    brain_score <- 3
  } else if (he_grade >= 1) {
    brain_score <- 2
  }

  coag_score <- 1
  if (inr > 2.5) {
    coag_score <- 3
  } else if (inr >= 2) {
    coag_score <- 2
  }

  circ_score <- 1
  if (use_vasopressors == 1) {
    circ_score <- 3
  } else if (map < 70) {
    circ_score <- 2
  }

  resp_score <- 1
  if (pao2_fio2_ratio <= 200) {
    resp_score <- 3
  } else if (pao2_fio2_ratio <= 300) {
    resp_score <- 2
  }

  clif_of_score <- liver_score + kidney_score + brain_score + coag_score + circ_score + resp_score

  clif_c_aclf <- 10 * ((0.33 * clif_of_score) + (0.04 * age) + (0.63 * log(wbc_count)) - 2)

  mortality_28_day <- ""
  mortality_90_day <- ""

  if (clif_c_aclf <= 38) {
    mortality_28_day <- "~4%"
    mortality_90_day <- "~13%"
  } else if (clif_c_aclf <= 44) {
    mortality_28_day <- "~17%"
    mortality_90_day <- "~35%"
  } else if (clif_c_aclf <= 50) {
    mortality_28_day <- "~36%"
    mortality_90_day <- "~58%"
  } else if (clif_c_aclf <= 57) {
    mortality_28_day <- "~60%"
    mortality_90_day <- "~80%"
  } else if (clif_c_aclf <= 64) {
    mortality_28_day <- "~80%"
    mortality_90_day <- "~93%"
  } else {
    mortality_28_day <- ">90%"
    mortality_90_day <- ">98%"
  }

  return(list(
    CLIF_C_ACLF_Score = round(clif_c_aclf, 1),
    CLIF_OF_Score = clif_of_score,
    Est_28_Day_Mortality = mortality_28_day,
    Est_90_Day_Mortality = mortality_90_day,
    Inputs = list(
      Age = age,
      WBC = wbc_count,
      Bilirubin_mg_dL = round(bili_mg_dl, 2),
      Creatinine_mg_dL = round(creat_mg_dl, 2),
      INR = inr,
      MAP = map,
      Vasopressors = use_vasopressors == 1,
      PF_Ratio = pao2_fio2_ratio,
      HE_Grade = he_grade
    )
  ))
}

clinical_dementia_rating <- function(memory, orientation, judgment, community, home_hobbies, personal_care) {

  inputs <- c(memory, orientation, judgment, community, home_hobbies, personal_care)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  valid_scores <- c(0, 0.5, 1, 2, 3)
  if (!all(inputs %in% valid_scores)) {
    stop("Error: Scores must be one of 0, 0.5, 1, 2, or 3.")
  }

  cdr_sb <- sum(inputs)

  m <- memory
  secondary <- c(orientation, judgment, community, home_hobbies, personal_care)

  global_cdr <- m

  if (m == 0) {
    count_ge_0_5 <- sum(secondary >= 0.5)
    if (count_ge_0_5 >= 2) {
      global_cdr <- 0.5
    } else {
      global_cdr <- 0
    }
  } else if (m == 0.5) {
    count_ge_1 <- sum(secondary >= 1)
    if (count_ge_1 >= 3) {
      global_cdr <- 1
    } else {
      global_cdr <- 0.5
    }
  } else {
    count_greater <- sum(secondary > m)
    count_less <- sum(secondary < m)

    if (count_greater >= 3) {
      global_cdr <- if (m == 3) 3 else m + 1
    } else if (count_less >= 3) {
      global_cdr <- if (m == 1) 0.5 else m - 1
    } else {
      global_cdr <- m
    }
  }

  stage_description <- ""
  if (global_cdr == 0) stage_description <- "Normal"
  else if (global_cdr == 0.5) stage_description <- "Very Mild Dementia"
  else if (global_cdr == 1) stage_description <- "Mild Dementia"
  else if (global_cdr == 2) stage_description <- "Moderate Dementia"
  else if (global_cdr == 3) stage_description <- "Severe Dementia"

  return(list(
    Global_CDR_Score = global_cdr,
    CDR_Sum_of_Boxes = cdr_sb,
    Stage = stage_description,
    Inputs = list(
      Memory = memory,
      Orientation = orientation,
      Judgment_ProblemSolving = judgment,
      Community_Affairs = community,
      Home_Hobbies = home_hobbies,
      Personal_Care = personal_care
    )
  ))
}

cdai_rheumatoid_arthritis <- function(tender_joint_count, swollen_joint_count, patient_global_assessment, provider_global_assessment) {

  tender_joint_count <- as.numeric(tender_joint_count)
  swollen_joint_count <- as.numeric(swollen_joint_count)
  patient_global_assessment <- as.numeric(patient_global_assessment)
  provider_global_assessment <- as.numeric(provider_global_assessment)

  if (any(is.na(c(tender_joint_count, swollen_joint_count, patient_global_assessment, provider_global_assessment)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (tender_joint_count < 0 || tender_joint_count > 28) {
    stop("Error: Tender joint count must be between 0 and 28.")
  }

  if (swollen_joint_count < 0 || swollen_joint_count > 28) {
    stop("Error: Swollen joint count must be between 0 and 28.")
  }

  if (patient_global_assessment < 0 || patient_global_assessment > 10) {
    stop("Error: Patient global assessment must be between 0 and 10.")
  }

  if (provider_global_assessment < 0 || provider_global_assessment > 10) {
    stop("Error: Provider global assessment must be between 0 and 10.")
  }

  cdai_score <- tender_joint_count + swollen_joint_count + patient_global_assessment + provider_global_assessment

  disease_activity <- ""

  if (cdai_score <= 2.8) {
    disease_activity <- "Remission"
  } else if (cdai_score <= 10.0) {
    disease_activity <- "Low Disease Activity"
  } else if (cdai_score <= 22.0) {
    disease_activity <- "Moderate Disease Activity"
  } else {
    disease_activity <- "High Disease Activity"
  }

  return(list(
    CDAI_Score = cdai_score,
    Disease_Activity = disease_activity,
    Inputs = list(
      Tender_Joints = tender_joint_count,
      Swollen_Joints = swollen_joint_count,
      Patient_Global = patient_global_assessment,
      Provider_Global = provider_global_assessment
    )
  ))
}

cisne_score <- function(ecog_ge_2, copd, cvd, mucositis_ge_2, monocytes_lt_200, stress_hyperglycemia) {

  ecog_ge_2 <- as.numeric(ecog_ge_2)
  copd <- as.numeric(copd)
  cvd <- as.numeric(cvd)
  mucositis_ge_2 <- as.numeric(mucositis_ge_2)
  monocytes_lt_200 <- as.numeric(monocytes_lt_200)
  stress_hyperglycemia <- as.numeric(stress_hyperglycemia)

  if (any(is.na(c(ecog_ge_2, copd, cvd, mucositis_ge_2, monocytes_lt_200, stress_hyperglycemia)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (ecog_ge_2 == 1) {
    score <- score + 2
  }

  if (copd == 1) {
    score <- score + 1
  }

  if (cvd == 1) {
    score <- score + 1
  }

  if (mucositis_ge_2 == 1) {
    score <- score + 1
  }

  if (monocytes_lt_200 == 1) {
    score <- score + 1
  }

  if (stress_hyperglycemia == 1) {
    score <- score + 2
  }

  risk_class <- ""
  complication_rate <- ""
  recommendation <- ""

  if (score == 0) {
    risk_class <- "Low Risk"
    complication_rate <- "1.1%"
    recommendation <- "May be eligible for outpatient care."
  } else if (score <= 2) {
    risk_class <- "Intermediate Risk"
    complication_rate <- "6.2%"
    recommendation <- "Observation or inpatient admission usually recommended."
  } else {
    risk_class <- "High Risk"
    complication_rate <- "36%"
    recommendation <- "Inpatient admission required. High risk of serious complications."
  }

  return(list(
    CISNE_Score = score,
    Risk_Class = risk_class,
    Est_Serious_Complication_Rate = complication_rate,
    Recommendation = recommendation,
    Inputs = list(
      ECOG_PS_ge_2 = ecog_ge_2 == 1,
      COPD = copd == 1,
      Cardiovascular_Disease = cvd == 1,
      Mucositis_Grade_ge_2 = mucositis_ge_2 == 1,
      Monocytes_lt_200 = monocytes_lt_200 == 1,
      Stress_Hyperglycemia = stress_hyperglycemia == 1
    )
  ))
}

cpis_vap_score <- function(temp_c, wbc_count, band_forms_present, tracheal_secretions, pao2_fio2_ratio, ards_present, chest_xray, culture_results = 0) {

  temp_c <- as.numeric(temp_c)
  wbc_count <- as.numeric(wbc_count)
  band_forms_present <- as.numeric(band_forms_present)
  tracheal_secretions <- as.numeric(tracheal_secretions)
  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  ards_present <- as.numeric(ards_present)
  chest_xray <- as.numeric(chest_xray)
  culture_results <- as.numeric(culture_results)

  if (any(is.na(c(temp_c, wbc_count, band_forms_present, tracheal_secretions, pao2_fio2_ratio, ards_present, chest_xray, culture_results)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (temp_c >= 36.5 && temp_c <= 38.4) {
    score <- score + 0
  } else if (temp_c >= 38.5 && temp_c <= 38.9) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (wbc_count >= 4000 && wbc_count <= 11000) {
    score <- score + 0
  } else {
    if (band_forms_present == 1) {
      score <- score + 2
    } else {
      score <- score + 1
    }
  }

  if (tracheal_secretions == 0) {
    score <- score + 0
  } else if (tracheal_secretions == 1) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (ards_present == 1) {
    score <- score + 0
  } else {
    if (pao2_fio2_ratio > 240) {
      score <- score + 0
    } else {
      score <- score + 2
    }
  }

  if (chest_xray == 0) {
    score <- score + 0
  } else if (chest_xray == 1) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (culture_results == 0) {
    score <- score + 0
  } else if (culture_results == 1) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  interpretation <- ""
  if (score > 6) {
    interpretation <- "Suggestive of Ventilator-Associated Pneumonia (VAP)"
  } else {
    interpretation <- "Low likelihood of VAP"
  }

  return(list(
    CPIS_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Temp_C = temp_c,
      WBC = wbc_count,
      Bands = band_forms_present == 1,
      Secretions_Grade = tracheal_secretions,
      PF_Ratio = pao2_fio2_ratio,
      ARDS = ards_present == 1,
      CXR_Grade = chest_xray,
      Culture_Grade = culture_results
    )
  ))
}

ishihara_screening_result <- function(total_plates_administered, error_count) {

  total_plates_administered <- as.numeric(total_plates_administered)
  error_count <- as.numeric(error_count)

  if (any(is.na(c(total_plates_administered, error_count)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (error_count > total_plates_administered) {
    stop("Error: Error count cannot exceed total plates.")
  }

  result <- ""
  interpretation <- ""
  next_step <- ""

  is_normal <- FALSE

  if (total_plates_administered >= 38) {
    if (error_count <= 4) {
      is_normal <- TRUE
    } else if (error_count >= 8) {
      result <- "Deficient"
    } else {
      result <- "Indeterminate"
    }
  } else if (total_plates_administered >= 24) {
    if (error_count <= 3) {
      is_normal <- TRUE
    } else {
      result <- "Deficient"
    }
  } else {
    if (error_count <= 1) {
      is_normal <- TRUE
    } else {
      result <- "Deficient"
    }
  }

  if (is_normal) {
    result <- "Normal Color Vision"
    interpretation <- "The patient identified the majority of plates correctly."
    next_step <- "No further testing required for color deficiency."
  } else {
    if (result == "Indeterminate") {
      interpretation <- "Results are borderline."
      next_step <- "Retest or refer to an eye care specialist for comprehensive testing."
    } else {
      result <- "Color Vision Deficiency Suspected"
      interpretation <- "High likelihood of Red-Green color deficiency (Protanopia or Deuteranopia)."
      next_step <- "Refer to ophthalmologist for Anomaloscope testing to distinguish type and severity."
    }
  }

  return(list(
    Test_Format = paste(total_plates_administered, "Plate Edition"),
    Errors = error_count,
    Screening_Result = result,
    Interpretation = interpretation,
    Next_Steps = next_step
  ))
}

cssrs_screener <- function(wish_to_be_dead, suicidal_thoughts, thoughts_with_method, intent_without_plan, intent_with_plan, suicidal_behavior_3mo) {

  wish_to_be_dead <- as.numeric(wish_to_be_dead)
  suicidal_thoughts <- as.numeric(suicidal_thoughts)
  thoughts_with_method <- as.numeric(thoughts_with_method)
  intent_without_plan <- as.numeric(intent_without_plan)
  intent_with_plan <- as.numeric(intent_with_plan)
  suicidal_behavior_3mo <- as.numeric(suicidal_behavior_3mo)

  if (any(is.na(c(wish_to_be_dead, suicidal_thoughts, thoughts_with_method, intent_without_plan, intent_with_plan, suicidal_behavior_3mo)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  risk_level <- "None"
  action <- "No specific intervention required based on this screener."
  color_code <- "Green"

  if (wish_to_be_dead == 1 || suicidal_thoughts == 1) {
    risk_level <- "Low Risk"
    color_code <- "Yellow"
    action <- "Behavioral health referral at discharge. Provide crisis resources."
  }

  if (thoughts_with_method == 1) {
    risk_level <- "Moderate Risk"
    color_code <- "Orange"
    action <- "Behavioral health consult required. Precautions may be needed pending evaluation."
  }

  if (intent_without_plan == 1 || intent_with_plan == 1 || suicidal_behavior_3mo == 1) {
    risk_level <- "High Risk"
    color_code <- "Red"
    action <- "Immediate evaluation required. Constant observation/safety precautions. Do not leave patient unattended."
  }

  max_ideation_score <- 0
  if (intent_with_plan == 1) max_ideation_score <- 5
  else if (intent_without_plan == 1) max_ideation_score <- 4
  else if (thoughts_with_method == 1) max_ideation_score <- 3
  else if (suicidal_thoughts == 1) max_ideation_score <- 2
  else if (wish_to_be_dead == 1) max_ideation_score <- 1

  return(list(
    Risk_Level = risk_level,
    Triage_Color = color_code,
    Recommended_Action = action,
    Max_Ideation_Severity = max_ideation_score,
    Recent_Behavior = suicidal_behavior_3mo == 1,
    Inputs = list(
      Q1_Wish_Dead = wish_to_be_dead == 1,
      Q2_Thoughts = suicidal_thoughts == 1,
      Q3_Method = thoughts_with_method == 1,
      Q4_Intent = intent_without_plan == 1,
      Q5_Plan_Intent = intent_with_plan == 1,
      Q6_Behavior_3mo = suicidal_behavior_3mo == 1
    )
  ))
}

ctcae_v5_grading <- function(grade_index = NULL, anc_count = NULL, platelet_count = NULL) {

  if (!is.null(grade_index)) {
    grade_index <- as.numeric(grade_index)
    if (!grade_index %in% 1:5) {
      stop("Error: Grade index must be an integer between 1 and 5.")
    }

    definition <- ""
    if (grade_index == 1) {
      definition <- "Mild; asymptomatic or mild symptoms; clinical or diagnostic observations only; intervention not indicated."
    } else if (grade_index == 2) {
      definition <- "Moderate; minimal, local or noninvasive intervention indicated; limiting age-appropriate instrumental ADL."
    } else if (grade_index == 3) {
      definition <- "Severe or medically significant but not immediately life-threatening; hospitalization or prolongation of hospitalization indicated; disabling; limiting self care ADL."
    } else if (grade_index == 4) {
      definition <- "Life-threatening consequences; urgent intervention indicated."
    } else {
      definition <- "Death related to adverse event."
    }

    return(list(
      Grade = grade_index,
      General_Definition = definition
    ))
  }

  if (!is.null(anc_count)) {
    anc <- as.numeric(anc_count)
    calc_grade <- 0
    if (anc >= 1500) {
      calc_grade <- 0
    } else if (anc >= 1000) {
      calc_grade <- 2
    } else if (anc >= 500) {
      calc_grade <- 3
    } else {
      calc_grade <- 4
    }

    return(list(
      Toxicity = "Neutropenia (ANC)",
      Input_Value = anc,
      Calculated_Grade = if(calc_grade == 0) "Normal / Grade 1 (depends on LLN)" else paste("Grade", calc_grade),
      Note = "Grade 1 is defined as < LLN - 1500/mm3. Grade 4 is < 500/mm3."
    ))
  }

  if (!is.null(platelet_count)) {
    plt <- as.numeric(platelet_count)
    calc_grade <- 0

    if (plt >= 100000) {
      calc_grade <- 0
    } else if (plt >= 75000) {
      calc_grade <- 1
    } else if (plt >= 50000) {
      calc_grade <- 2
    } else if (plt >= 25000) {
      calc_grade <- 3
    } else {
      calc_grade <- 4
    }

    return(list(
      Toxicity = "Thrombocytopenia (Platelets)",
      Input_Value = plt,
      Calculated_Grade = if(calc_grade == 0) "Normal" else paste("Grade", calc_grade)
    ))
  }

  stop("Error: Please provide either a 'grade_index' to define, or a specific lab value ('anc_count', 'platelet_count') to grade.")
}

cam_icu_delirium_assessment <- function(acute_fluctuating, inattention, altered_loc, disorganized_thinking) {

  acute_fluctuating <- as.numeric(acute_fluctuating)
  inattention <- as.numeric(inattention)
  altered_loc <- as.numeric(altered_loc)
  disorganized_thinking <- as.numeric(disorganized_thinking)

  if (any(is.na(c(acute_fluctuating, inattention, altered_loc, disorganized_thinking)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  is_positive <- FALSE

  if (acute_fluctuating == 1 && inattention == 1) {
    if (altered_loc == 1 || disorganized_thinking == 1) {
      is_positive <- TRUE
    }
  }

  diagnosis <- ""
  if (is_positive) {
    diagnosis <- "Delirium Positive"
  } else {
    diagnosis <- "Delirium Negative"
  }

  return(list(
    Diagnosis = diagnosis,
    Required_Features_Met = list(
      Feature_1_Acute_Fluctuating = acute_fluctuating == 1,
      Feature_2_Inattention = inattention == 1,
      Feature_3_OR_4_Met = altered_loc == 1 || disorganized_thinking == 1
    )
  ))
}

copd_assessment_test <- function(cough, phlegm, chest_tightness, breathlessness_stairs, activity_limitation, confidence_leaving_home, sleep, energy) {

  cough <- as.numeric(cough)
  phlegm <- as.numeric(phlegm)
  chest_tightness <- as.numeric(chest_tightness)
  breathlessness_stairs <- as.numeric(breathlessness_stairs)
  activity_limitation <- as.numeric(activity_limitation)
  confidence_leaving_home <- as.numeric(confidence_leaving_home)
  sleep <- as.numeric(sleep)
  energy <- as.numeric(energy)

  inputs <- c(cough, phlegm, chest_tightness, breathlessness_stairs, activity_limitation, confidence_leaving_home, sleep, energy)

  if (any(is.na(inputs))) {
    stop("Error: All 8 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 5)) {
    stop("Error: All item scores must be between 0 and 5.")
  }

  total_score <- sum(inputs)

  impact_level <- ""
  if (total_score <= 10) {
    impact_level <- "Low Impact (Well-controlled)"
  } else if (total_score <= 20) {
    impact_level <- "Medium Impact"
  } else if (total_score <= 30) {
    impact_level <- "High Impact"
  } else {
    impact_level <- "Very High Impact"
  }

  return(list(
    CAT_Score = total_score,
    Impact_Level = impact_level,
    Inputs = list(
      Cough = cough,
      Phlegm = phlegm,
      Tightness = chest_tightness,
      Breathlessness = breathlessness_stairs,
      Activity_Limit = activity_limitation,
      Confidence = confidence_leaving_home,
      Sleep = sleep,
      Energy = energy
    )
  ))
}

cornell_assessment_pediatric_delirium <- function(q1_eye_contact, q2_purposeful_actions, q3_aware_surroundings, q4_communicate_needs, q5_restless, q6_inconsolable, q7_underactive, q8_response_time) {

  inputs <- c(q1_eye_contact, q2_purposeful_actions, q3_aware_surroundings, q4_communicate_needs, q5_restless, q6_inconsolable, q7_underactive, q8_response_time)

  if (any(is.na(inputs))) {
    stop("Error: All 8 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be between 0 and 4.")
  }

  total_score <- sum(inputs)

  interpretation <- ""
  if (total_score >= 9) {
    interpretation <- "Positive for Delirium (Score >= 9)"
  } else {
    interpretation <- "Negative for Delirium (Score < 9)"
  }

  return(list(
    CAPD_Total_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Q1_Eye_Contact = q1_eye_contact,
      Q2_Purposeful = q2_purposeful_actions,
      Q3_Awareness = q3_aware_surroundings,
      Q4_Communication = q4_communicate_needs,
      Q5_Restless = q5_restless,
      Q6_Inconsolable = q6_inconsolable,
      Q7_Underactive = q7_underactive,
      Q8_Response_Time = q8_response_time
    )
  ))
}

coronavirus_anxiety_scale <- function(dizziness, sleep_disturbance, paralyzed_frozen, appetite_loss, nausea_stomach_pain) {

  dizziness <- as.numeric(dizziness)
  sleep_disturbance <- as.numeric(sleep_disturbance)
  paralyzed_frozen <- as.numeric(paralyzed_frozen)
  appetite_loss <- as.numeric(appetite_loss)
  nausea_stomach_pain <- as.numeric(nausea_stomach_pain)

  inputs <- c(dizziness, sleep_disturbance, paralyzed_frozen, appetite_loss, nausea_stomach_pain)

  if (any(is.na(inputs))) {
    stop("Error: All 5 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be between 0 and 4.")
  }

  total_score <- sum(inputs)

  result <- ""
  interpretation <- ""

  if (total_score >= 9) {
    result <- "Positive Screen"
    interpretation <- "Probable dysfunctional anxiety associated with the COVID-19 crisis (Sensitivity 90%, Specificity 85%)."
  } else {
    result <- "Negative Screen"
    interpretation <- "No significant dysfunctional anxiety detected."
  }

  return(list(
    CAS_Score = total_score,
    Result = result,
    Interpretation = interpretation,
    Inputs = list(
      Dizziness = dizziness,
      Sleep = sleep_disturbance,
      Tonic_Immobility = paralyzed_frozen,
      Appetite = appetite_loss,
      Nausea = nausea_stomach_pain
    )
  ))
}

cci_platelet_transfusion <- function(post_platelet_count, pre_platelet_count, bsa, platelets_transfused_10_11) {

  post_platelet_count <- as.numeric(post_platelet_count)
  pre_platelet_count <- as.numeric(pre_platelet_count)
  bsa <- as.numeric(bsa)
  platelets_transfused_10_11 <- as.numeric(platelets_transfused_10_11)

  if (any(is.na(c(post_platelet_count, pre_platelet_count, bsa, platelets_transfused_10_11)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (platelets_transfused_10_11 <= 0) {
    stop("Error: Platelets transfused must be greater than 0.")
  }

  cci <- ((post_platelet_count - pre_platelet_count) * bsa) / platelets_transfused_10_11

  interpretation <- ""
  if (cci < 5000) {
    interpretation <- "Refractoriness likely (CCI < 5000). Suggests immune or non-immune destruction."
  } else if (cci < 7500) {
    interpretation <- "Possible Refractoriness (CCI 5000-7500). Borderline result."
  } else {
    interpretation <- "Successful Transfusion (CCI >= 7500). Adequate increment."
  }

  return(list(
    CCI_Value = round(cci, 0),
    Interpretation = interpretation,
    Inputs = list(
      Post_Plt = post_platelet_count,
      Pre_Plt = pre_platelet_count,
      BSA_m2 = bsa,
      Transfused_Units_10_11 = platelets_transfused_10_11
    )
  ))
}

calculate_qtc <- function(qt_interval_msec, heart_rate_bpm) {

  qt_interval_msec <- as.numeric(qt_interval_msec)
  heart_rate_bpm <- as.numeric(heart_rate_bpm)

  if (any(is.na(c(qt_interval_msec, heart_rate_bpm)))) {
    stop("Error: QT interval and Heart Rate must be numeric.")
  }

  if (qt_interval_msec <= 0 || heart_rate_bpm <= 0) {
    stop("Error: Inputs must be positive numbers.")
  }

  rr_interval_sec <- 60 / heart_rate_bpm

  qtc_bazett <- qt_interval_msec / sqrt(rr_interval_sec)

  qtc_fridericia <- qt_interval_msec / (rr_interval_sec ^ (1/3))

  qtc_framingham <- qt_interval_msec + (154 * (1 - rr_interval_sec))

  qtc_hodges <- qt_interval_msec + (1.75 * (heart_rate_bpm - 60))

  return(list(
    QTc_Bazett = round(qtc_bazett, 0),
    QTc_Fridericia = round(qtc_fridericia, 0),
    QTc_Framingham = round(qtc_framingham, 0),
    QTc_Hodges = round(qtc_hodges, 0),
    Inputs = list(
      QT_msec = qt_interval_msec,
      HR_bpm = heart_rate_bpm,
      RR_sec = round(rr_interval_sec, 3)
    )
  ))
}

reticulocyte_indices <- function(retic_percent, rbc_count_million, hematocrit_percent, normal_hct = 45) {

  retic_percent <- as.numeric(retic_percent)
  rbc_count_million <- as.numeric(rbc_count_million)
  hematocrit_percent <- as.numeric(hematocrit_percent)
  normal_hct <- as.numeric(normal_hct)

  if (any(is.na(c(retic_percent, rbc_count_million, hematocrit_percent, normal_hct)))) {
    stop("Error: All inputs must be numeric.")
  }

  arc <- retic_percent * rbc_count_million * 10000

  retic_index <- retic_percent * (hematocrit_percent / normal_hct)

  maturation_factor <- 1.0
  if (hematocrit_percent >= 45) {
    maturation_factor <- 1.0
  } else if (hematocrit_percent >= 35) {
    maturation_factor <- 1.5
  } else if (hematocrit_percent >= 25) {
    maturation_factor <- 2.0
  } else if (hematocrit_percent >= 15) {
    maturation_factor <- 2.5
  } else {
    maturation_factor <- 3.0
  }

  rpi <- retic_index / maturation_factor

  response_status <- ""
  if (rpi > 3) {
    response_status <- "Adequate marrow response (e.g., for hemolysis/blood loss)"
  } else if (rpi < 2) {
    response_status <- "Inadequate marrow response (e.g., hypoproliferative anemia)"
  } else {
    response_status <- "Indeterminate"
  }

  return(list(
    Absolute_Reticulocyte_Count = paste(round(arc, 0), "cells/uL"),
    Reticulocyte_Index = round(retic_index, 2),
    Reticulocyte_Production_Index = round(rpi, 2),
    Marrow_Response = response_status,
    Inputs = list(
      Retic_Percent = retic_percent,
      RBC_Count = rbc_count_million,
      Hct = hematocrit_percent,
      Maturation_Factor_Used = maturation_factor
    )
  ))
}

chosen_covid_risk_score <- function(age, sex, respiratory_rate, spo2, albumin, albumin_units = "g/dL") {

  age <- as.numeric(age)
  respiratory_rate <- as.numeric(respiratory_rate)
  spo2 <- as.numeric(spo2)
  albumin <- as.numeric(albumin)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, respiratory_rate, spo2, albumin)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  alb_val <- albumin
  if (albumin_units == "g/L") {
    alb_val <- albumin / 10
  }

  sex_coeff <- if (sex == "male") 0.48 else 0


  logit <- -12.2 +
    (0.04 * age) +
    sex_coeff +
    (0.12 * respiratory_rate) -
    (0.13 * spo2) -
    (0.43 * alb_val)

  prob_adverse <- 1 / (1 + exp(-logit))


  prob_safe <- (1 - prob_adverse) * 100

  risk_level <- ""
  recommendation <- ""

  if (prob_safe >= 90) {
    risk_level <- "Low Risk (Safe for Discharge)"
    recommendation <- "High likelihood of safe recovery at home. Consider discharge with follow-up."
  } else if (prob_safe >= 70) {
    risk_level <- "Intermediate Risk"
    recommendation <- "Observation or short admission recommended."
  } else {
    risk_level <- "High Risk (Unsafe for Discharge)"
    recommendation <- "Inpatient admission warranted."
  }

  return(list(
    Probability_Safe_Discharge = paste0(round(prob_safe, 1), "%"),
    Risk_Category = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      RR = respiratory_rate,
      SpO2 = spo2,
      Albumin_g_dL = round(alb_val, 2)
    )
  ))
}

circ_covid_risk <- function(age, race_black, nursing_home_resident, charlson_score, bmi, temp_c, respiratory_rate, heart_rate, spo2, sbp) {

  age <- as.numeric(age)
  race_black <- as.numeric(race_black)
  nursing_home_resident <- as.numeric(nursing_home_resident)
  charlson_score <- as.numeric(charlson_score)
  bmi <- as.numeric(bmi)
  temp_c <- as.numeric(temp_c)
  respiratory_rate <- as.numeric(respiratory_rate)
  heart_rate <- as.numeric(heart_rate)
  spo2 <- as.numeric(spo2)
  sbp <- as.numeric(sbp)

  if (any(is.na(c(age, race_black, nursing_home_resident, charlson_score, bmi, temp_c, respiratory_rate, heart_rate, spo2, sbp)))) {
    stop("Error: All inputs must be numeric.")
  }

  logit <- -10.5

  logit <- logit + (0.065 * age)

  if (race_black == 1) {
    logit <- logit - 0.3
  }

  if (nursing_home_resident == 1) {
    logit <- logit + 0.6
  }

  logit <- logit + (0.15 * charlson_score)

  if (bmi >= 30) {
    logit <- logit + 0.2
  } else if (bmi >= 40) {
    logit <- logit + 0.4
  }

  if (respiratory_rate > 24) {
    logit <- logit + 0.5
  }

  if (spo2 < 92) {
    logit <- logit + 1.2
  } else if (spo2 < 96) {
    logit <- logit + 0.6
  }

  if (heart_rate > 100) {
    logit <- logit + 0.3
  }

  if (sbp < 100) {
    logit <- logit + 0.4
  }

  if (temp_c > 38) {
    logit <- logit + 0.2
  }

  probability <- 1 / (1 + exp(-logit))
  prob_percent <- probability * 100

  risk_level <- ""
  if (prob_percent < 5) {
    risk_level <- "Low Risk"
  } else if (prob_percent < 20) {
    risk_level <- "Moderate Risk"
  } else {
    risk_level <- "High Risk"
  }

  return(list(
    Est_Mortality_Risk_7_Day = paste0(round(prob_percent, 1), "%"),
    Risk_Category = risk_level,
    Inputs = list(
      Age = age,
      Race_Black = race_black == 1,
      Nursing_Home = nursing_home_resident == 1,
      CCI = charlson_score,
      BMI = bmi,
      Vitals = list(Temp=temp_c, RR=respiratory_rate, HR=heart_rate, SpO2=spo2, SBP=sbp)
    )
  ))
}

covid_gram_score <- function(age, x_ray_abnormality, hemoptysis, dyspnea, unconsciousness, number_of_comorbidities, cancer_history, nlr, ldh, direct_bilirubin, bilirubin_units = "umol/L") {

  age <- as.numeric(age)
  x_ray_abnormality <- as.numeric(x_ray_abnormality)
  hemoptysis <- as.numeric(hemoptysis)
  dyspnea <- as.numeric(dyspnea)
  unconsciousness <- as.numeric(unconsciousness)
  number_of_comorbidities <- as.numeric(number_of_comorbidities)
  cancer_history <- as.numeric(cancer_history)
  nlr <- as.numeric(nlr)
  ldh <- as.numeric(ldh)
  direct_bilirubin <- as.numeric(direct_bilirubin)

  if (any(is.na(c(age, x_ray_abnormality, hemoptysis, dyspnea, unconsciousness, number_of_comorbidities, cancer_history, nlr, ldh, direct_bilirubin)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_umol <- direct_bilirubin
  if (bilirubin_units == "mg/dL") {
    bili_umol <- direct_bilirubin * 17.1
  }

  logit <- -4.436

  logit <- logit + (0.022 * age)

  if (x_ray_abnormality == 1) {
    logit <- logit + 1.295
  }

  if (hemoptysis == 1) {
    logit <- logit + 2.303
  }

  if (dyspnea == 1) {
    logit <- logit + 0.770
  }

  if (unconsciousness == 1) {
    logit <- logit + 1.824
  }

  if (number_of_comorbidities == 1) {
    logit <- logit + 0.616
  } else if (number_of_comorbidities >= 2) {
    logit <- logit + 1.164
  }

  if (cancer_history == 1) {
    logit <- logit + 1.079
  }

  logit <- logit + (0.075 * nlr)

  logit <- logit + (0.003 * ldh)

  logit <- logit + (0.037 * bili_umol)

  probability <- 1 / (1 + exp(-logit))
  prob_percent <- probability * 100

  risk_category <- ""
  if (prob_percent < 1.7) {
    risk_category <- "Low Risk"
  } else if (prob_percent >= 1.7 && prob_percent < 40.4) {
    risk_category <- "Medium Risk"
  } else {
    risk_category <- "High Risk"
  }

  return(list(
    Critical_Illness_Probability = paste0(round(prob_percent, 1), "%"),
    Risk_Category = risk_category,
    Inputs = list(
      Age = age,
      CXR_Abnormal = x_ray_abnormality == 1,
      Hemoptysis = hemoptysis == 1,
      Dyspnea = dyspnea == 1,
      Unconsciousness = unconsciousness == 1,
      Comorbidities = number_of_comorbidities,
      Cancer_Hx = cancer_history == 1,
      NLR = nlr,
      LDH = ldh,
      Direct_Bilirubin_umol = round(bili_umol, 2)
    )
  ))
}

cows_opiate_withdrawal <- function(pulse_score, sweating_score, restlessness_score, pupil_size_score, bone_joint_ache_score, runny_nose_tearing_score, gi_upset_score, tremor_score, yawning_score, anxiety_irritability_score, gooseflesh_score) {

  pulse_score <- as.numeric(pulse_score)
  sweating_score <- as.numeric(sweating_score)
  restlessness_score <- as.numeric(restlessness_score)
  pupil_size_score <- as.numeric(pupil_size_score)
  bone_joint_ache_score <- as.numeric(bone_joint_ache_score)
  runny_nose_tearing_score <- as.numeric(runny_nose_tearing_score)
  gi_upset_score <- as.numeric(gi_upset_score)
  tremor_score <- as.numeric(tremor_score)
  yawning_score <- as.numeric(yawning_score)
  anxiety_irritability_score <- as.numeric(anxiety_irritability_score)
  gooseflesh_score <- as.numeric(gooseflesh_score)

  if (any(is.na(c(pulse_score, sweating_score, restlessness_score, pupil_size_score, bone_joint_ache_score, runny_nose_tearing_score, gi_upset_score, tremor_score, yawning_score, anxiety_irritability_score, gooseflesh_score)))) {
    stop("Error: All inputs must be numeric scores.")
  }

  total_score <- pulse_score + sweating_score + restlessness_score + pupil_size_score + bone_joint_ache_score + runny_nose_tearing_score + gi_upset_score + tremor_score + yawning_score + anxiety_irritability_score + gooseflesh_score

  severity <- ""
  if (total_score <= 4) {
    severity <- "No Withdrawal / Minimal"
  } else if (total_score <= 12) {
    severity <- "Mild Withdrawal"
  } else if (total_score <= 24) {
    severity <- "Moderate Withdrawal"
  } else if (total_score <= 36) {
    severity <- "Moderately Severe Withdrawal"
  } else {
    severity <- "Severe Withdrawal"
  }

  return(list(
    COWS_Score = total_score,
    Severity_Category = severity,
    Inputs = list(
      Pulse = pulse_score,
      Sweating = sweating_score,
      Restlessness = restlessness_score,
      Pupils = pupil_size_score,
      Aches = bone_joint_ache_score,
      RunnyNose = runny_nose_tearing_score,
      GI = gi_upset_score,
      Tremor = tremor_score,
      Yawning = yawning_score,
      Anxiety = anxiety_irritability_score,
      Gooseflesh = gooseflesh_score
    )
  ))
}

crb65_pneumonia_severity <- function(confusion, respiratory_rate, systolic_bp, diastolic_bp, age) {

  confusion <- as.numeric(confusion)
  respiratory_rate <- as.numeric(respiratory_rate)
  systolic_bp <- as.numeric(systolic_bp)
  diastolic_bp <- as.numeric(diastolic_bp)
  age <- as.numeric(age)

  if (any(is.na(c(confusion, respiratory_rate, systolic_bp, diastolic_bp, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (confusion == 1) {
    score <- score + 1
  }

  if (respiratory_rate >= 30) {
    score <- score + 1
  }

  if (systolic_bp < 90 || diastolic_bp <= 60) {
    score <- score + 1
  }

  if (age >= 65) {
    score <- score + 1
  }

  risk_level <- ""
  mortality <- ""
  recommendation <- ""

  if (score == 0) {
    risk_level <- "Very Low Risk"
    mortality <- "1.2%"
    recommendation <- "Likely suitable for home treatment."
  } else if (score >= 1 && score <= 2) {
    risk_level <- "Increased Risk"
    mortality <- "8.15%"
    recommendation <- "Consider hospital referral for assessment."
  } else {
    risk_level <- "High Risk"
    mortality <- "31%"
    recommendation <- "Urgent hospital admission."
  }

  return(list(
    CRB65_Score = score,
    Risk_Level = risk_level,
    Est_Mortality_30_Day = mortality,
    Recommendation = recommendation,
    Inputs = list(
      Confusion = confusion == 1,
      RR_ge_30 = respiratory_rate >= 30,
      Hypotension = (systolic_bp < 90 || diastolic_bp <= 60),
      Age_ge_65 = age >= 65
    )
  ))
}

cockcroft_gault_crcl <- function(sex, age, creatinine, weight_kg, height_cm = NULL, creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  creatinine <- as.numeric(creatinine)
  weight_kg <- as.numeric(weight_kg)

  if (!is.null(height_cm)) {
    height_cm <- as.numeric(height_cm)
  }

  if (any(is.na(c(age, creatinine, weight_kg)))) {
    stop("Error: Age, Creatinine, and Weight must be numeric.")
  }

  sex <- tolower(as.character(sex))
  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  scr_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    scr_mg_dl <- creatinine / 88.4
  }

  if (scr_mg_dl <= 0) stop("Error: Creatinine must be positive.")

  weight_to_use <- weight_kg
  weight_type <- "Actual Body Weight"

  if (!is.null(height_cm) && !is.na(height_cm)) {
    height_m <- height_cm / 100
    height_inches <- height_cm / 2.54
    bmi <- weight_kg / (height_m ^ 2)

    ibw <- 0
    if (sex == "male") {
      ibw <- 50 + 2.3 * (height_inches - 60)
    } else {
      ibw <- 45.5 + 2.3 * (height_inches - 60)
    }

    if (bmi < 18.5) {
      weight_to_use <- weight_kg
      weight_type <- "Actual Body Weight (Underweight)"
    } else if (bmi >= 18.5 && bmi < 25) {
      weight_to_use <- ibw
      weight_type <- "Ideal Body Weight (Normal BMI)"
    } else {
      abw <- ibw + 0.4 * (weight_kg - ibw)
      weight_to_use <- abw
      weight_type <- "Adjusted Body Weight (Overweight/Obese)"
    }
  }

  constant <- if (sex == "female") 0.85 else 1.0

  crcl <- ((140 - age) * weight_to_use) / (72 * scr_mg_dl) * constant

  return(list(
    CrCl_mL_min = round(crcl, 1),
    Weight_Used_kg = round(weight_to_use, 1),
    Method = weight_type,
    Inputs = list(
      Sex = sex,
      Age = age,
      Creatinine_mg_dL = round(scr_mg_dl, 2),
      Weight_Actual = weight_kg,
      Height = if (!is.null(height_cm)) height_cm else "Not Provided"
    )
  ))
}

cpot_pain_score <- function(facial_expression, body_movements, muscle_tension, compliance_or_vocalization) {

  facial_expression <- as.numeric(facial_expression)
  body_movements <- as.numeric(body_movements)
  muscle_tension <- as.numeric(muscle_tension)
  compliance_or_vocalization <- as.numeric(compliance_or_vocalization)

  if (any(is.na(c(facial_expression, body_movements, muscle_tension, compliance_or_vocalization)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(facial_expression, body_movements, muscle_tension, compliance_or_vocalization) < 0) ||
      any(c(facial_expression, body_movements, muscle_tension, compliance_or_vocalization) > 2)) {
    stop("Error: All item scores must be between 0 and 2.")
  }

  total_score <- facial_expression + body_movements + muscle_tension + compliance_or_vocalization

  interpretation <- ""
  if (total_score > 2) {
    interpretation <- "Positive for Pain (Score > 2). Consider analgesic intervention."
  } else {
    interpretation <- "Negative for Pain (Score 0-2). Pain likely controlled or minimal."
  }

  return(list(
    CPOT_Score = total_score,
    Interpretation = interpretation,
    Inputs = list(
      Facial = facial_expression,
      Body = body_movements,
      Muscle = muscle_tension,
      Compliance_Vocal = compliance_or_vocalization
    )
  ))
}

cdai_score <- function(liquid_stools_sum_7d, abdominal_pain_sum_7d, well_being_sum_7d, complications_count, antidiarrheal_use, abdominal_mass, hematocrit, weight_kg, standard_weight_kg, sex) {

  liquid_stools_sum_7d <- as.numeric(liquid_stools_sum_7d)
  abdominal_pain_sum_7d <- as.numeric(abdominal_pain_sum_7d)
  well_being_sum_7d <- as.numeric(well_being_sum_7d)
  complications_count <- as.numeric(complications_count)
  antidiarrheal_use <- as.numeric(antidiarrheal_use)
  abdominal_mass <- as.numeric(abdominal_mass)
  hematocrit <- as.numeric(hematocrit)
  weight_kg <- as.numeric(weight_kg)
  standard_weight_kg <- as.numeric(standard_weight_kg)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(liquid_stools_sum_7d, abdominal_pain_sum_7d, well_being_sum_7d, complications_count, antidiarrheal_use, abdominal_mass, hematocrit, weight_kg, standard_weight_kg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  score <- score + (liquid_stools_sum_7d * 2)

  score <- score + (abdominal_pain_sum_7d * 5)

  score <- score + (well_being_sum_7d * 7)

  score <- score + (complications_count * 20)

  if (antidiarrheal_use == 1) {
    score <- score + 30
  }

  if (abdominal_mass == 2) {
    score <- score + 20
  } else if (abdominal_mass == 5) {
    score <- score + 50
  }

  hct_diff <- 0
  if (sex == "male") {
    hct_diff <- 47 - hematocrit
  } else {
    hct_diff <- 42 - hematocrit
  }
  score <- score + (hct_diff * 6)

  weight_deviation <- ((standard_weight_kg - weight_kg) / standard_weight_kg) * 100
  if (weight_deviation < 0) {
    weight_deviation <- 0
  }
  score <- score + (weight_deviation * 1)

  interpretation <- ""
  if (score < 150) {
    interpretation <- "Remission"
  } else if (score <= 219) {
    interpretation <- "Mildly Active Disease"
  } else if (score <= 450) {
    interpretation <- "Moderately Active Disease"
  } else {
    interpretation <- "Severely Active Disease"
  }

  return(list(
    CDAI_Score = round(score, 1),
    Disease_Activity = interpretation,
    Inputs = list(
      Stools_Sum = liquid_stools_sum_7d,
      Pain_Sum = abdominal_pain_sum_7d,
      Well_Being_Sum = well_being_sum_7d,
      Complications = complications_count,
      Antidiarrheal = antidiarrheal_use == 1,
      Mass_Score = abdominal_mass,
      Hct_Diff = hct_diff,
      Weight_Dev_Percent = round(weight_deviation, 1)
    )
  ))
}

crusade_bleeding_score <- function(hematocrit_percent, creatinine_clearance, heart_rate, sex, chf_signs, prior_vascular_disease, diabetes, systolic_bp) {

  hematocrit_percent <- as.numeric(hematocrit_percent)
  creatinine_clearance <- as.numeric(creatinine_clearance)
  heart_rate <- as.numeric(heart_rate)
  chf_signs <- as.numeric(chf_signs)
  prior_vascular_disease <- as.numeric(prior_vascular_disease)
  diabetes <- as.numeric(diabetes)
  systolic_bp <- as.numeric(systolic_bp)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(hematocrit_percent, creatinine_clearance, heart_rate, chf_signs, prior_vascular_disease, diabetes, systolic_bp)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (hematocrit_percent < 31) {
    score <- score + 9
  } else if (hematocrit_percent >= 31 && hematocrit_percent < 34) {
    score <- score + 7
  } else if (hematocrit_percent >= 34 && hematocrit_percent < 37) {
    score <- score + 3
  } else if (hematocrit_percent >= 37 && hematocrit_percent < 40) {
    score <- score + 2
  }

  if (creatinine_clearance <= 15) {
    score <- score + 39
  } else if (creatinine_clearance > 15 && creatinine_clearance <= 30) {
    score <- score + 35
  } else if (creatinine_clearance > 30 && creatinine_clearance <= 60) {
    score <- score + 28
  } else if (creatinine_clearance > 60 && creatinine_clearance <= 90) {
    score <- score + 17
  } else if (creatinine_clearance > 90 && creatinine_clearance <= 120) {
    score <- score + 7
  }

  if (heart_rate <= 70) {
    score <- score + 0
  } else if (heart_rate >= 71 && heart_rate <= 80) {
    score <- score + 1
  } else if (heart_rate >= 81 && heart_rate <= 90) {
    score <- score + 3
  } else if (heart_rate >= 91 && heart_rate <= 100) {
    score <- score + 6
  } else if (heart_rate >= 101 && heart_rate <= 110) {
    score <- score + 8
  } else if (heart_rate >= 111 && heart_rate <= 120) {
    score <- score + 10
  } else {
    score <- score + 11
  }

  if (sex == "female") {
    score <- score + 8
  }

  if (chf_signs == 1) {
    score <- score + 7
  }

  if (prior_vascular_disease == 1) {
    score <- score + 6
  }

  if (diabetes == 1) {
    score <- score + 6
  }

  if (systolic_bp <= 90) {
    score <- score + 10
  } else if (systolic_bp >= 91 && systolic_bp <= 100) {
    score <- score + 8
  } else if (systolic_bp >= 101 && systolic_bp <= 120) {
    score <- score + 5
  } else if (systolic_bp >= 121 && systolic_bp <= 180) {
    score <- score + 1
  } else if (systolic_bp >= 181 && systolic_bp <= 200) {
    score <- score + 3
  } else {
    score <- score + 5
  }

  risk_category <- ""
  bleeding_risk <- ""

  if (score <= 20) {
    risk_category <- "Very Low Risk"
    bleeding_risk <- "3.1%"
  } else if (score <= 30) {
    risk_category <- "Low Risk"
    bleeding_risk <- "5.5%"
  } else if (score <= 40) {
    risk_category <- "Moderate Risk"
    bleeding_risk <- "8.6%"
  } else if (score <= 50) {
    risk_category <- "High Risk"
    bleeding_risk <- "11.9%"
  } else {
    risk_category <- "Very High Risk"
    bleeding_risk <- "19.5%"
  }

  return(list(
    CRUSADE_Score = score,
    Risk_Category = risk_category,
    In_Hospital_Major_Bleeding_Risk = bleeding_risk,
    Inputs = list(
      Hct = hematocrit_percent,
      CrCl = creatinine_clearance,
      HR = heart_rate,
      Sex = sex,
      CHF = chf_signs == 1,
      Vascular_Disease = prior_vascular_disease == 1,
      Diabetes = diabetes == 1,
      SBP = systolic_bp
    )
  ))
}

dire_opioid_score <- function(diagnosis, intractability, risk_psychological, risk_chemical, risk_reliability, risk_social, efficacy) {

  inputs <- c(diagnosis, intractability, risk_psychological, risk_chemical, risk_reliability, risk_social, efficacy)

  if (any(is.na(inputs))) {
    stop("Error: All 7 inputs must be numeric.")
  }

  if (any(inputs < 1) || any(inputs > 3)) {
    stop("Error: All item scores must be integers between 1 and 3.")
  }

  total_score <- sum(inputs)

  suitability <- ""
  recommendation <- ""

  if (total_score <= 13) {
    suitability <- "Not a suitable candidate for long-term opioid prescribing."
    recommendation <- "Consider other pain management modalities. High risk of adverse outcomes."
  } else {
    suitability <- "Suitable candidate for long-term opioid prescribing."
    recommendation <- "Monitor closely, but patient demonstrates factors associated with successful opioid therapy."
  }

  return(list(
    DIRE_Score = total_score,
    Suitability = suitability,
    Recommendation = recommendation,
    Inputs = list(
      Diagnosis = diagnosis,
      Intractability = intractability,
      Risk_Psych = risk_psychological,
      Risk_Chemical = risk_chemical,
      Risk_Reliability = risk_reliability,
      Risk_Social = risk_social,
      Efficacy = efficacy
    )
  ))
}

doac_bleeding_risk <- function(age, criteria_count_gt_1, creatinine_clearance_ml_min, prior_bleeding, hypertension, nsaids_antiplatelets, cirrhosis_history, history_stroke, prior_major_bleeding, extreme_body_weight) {

  age <- as.numeric(age)
  criteria_count_gt_1 <- as.numeric(criteria_count_gt_1)
  creatinine_clearance_ml_min <- as.numeric(creatinine_clearance_ml_min)
  prior_bleeding <- as.numeric(prior_bleeding)
  hypertension <- as.numeric(hypertension)
  nsaids_antiplatelets <- as.numeric(nsaids_antiplatelets)
  cirrhosis_history <- as.numeric(cirrhosis_history)
  history_stroke <- as.numeric(history_stroke)
  prior_major_bleeding <- as.numeric(prior_major_bleeding)
  extreme_body_weight <- as.numeric(extreme_body_weight)

  if (any(is.na(c(age, criteria_count_gt_1, creatinine_clearance_ml_min, prior_bleeding, hypertension, nsaids_antiplatelets, cirrhosis_history, history_stroke, prior_major_bleeding, extreme_body_weight)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 75) {
    score <- score + 2
  } else if (age >= 65 && age <= 74) {
    score <- score + 1
  }

  if (creatinine_clearance_ml_min < 60) {
    score <- score + 1
  }

  if (criteria_count_gt_1 == 1) {
    score <- score + 2
  }

  if (prior_bleeding == 1) {
    score <- score + 2
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (nsaids_antiplatelets == 1) {
    score <- score + 1
  }

  if (cirrhosis_history == 1) {
    score <- score + 1
  }

  if (history_stroke == 1) {
    score <- score + 1
  }

  if (prior_major_bleeding == 1) {
    score <- score + 1
  }

  if (extreme_body_weight == 1) {
    score <- score + 1
  }

  risk_level <- ""

  if (score <= 3) {
    risk_level <- "Low Bleeding Risk"
  } else if (score <= 7) {
    risk_level <- "Moderate Bleeding Risk"
  } else {
    risk_level <- "High Bleeding Risk"
  }

  return(list(
    DOAC_Score = score,
    Risk_Category = risk_level,
    Inputs = list(
      Age = age,
      CrCl_lt_60 = creatinine_clearance_ml_min < 60,
      Comorb_Cirrhosis = cirrhosis_history == 1,
      Prior_Bleed = prior_bleeding == 1,
      Hypertension = hypertension == 1,
      Antiplatelet_NSAID = nsaids_antiplatelets == 1,
      Stroke_History = history_stroke == 1,
      Major_Bleed_History = prior_major_bleeding == 1,
      Extreme_Weight = extreme_body_weight == 1
    )
  ))
}

das28_crp_rheumatoid_arthritis <- function(tender_joint_count, swollen_joint_count, crp, patient_global_health, crp_units = "mg/L") {

  tender_joint_count <- as.numeric(tender_joint_count)
  swollen_joint_count <- as.numeric(swollen_joint_count)
  crp <- as.numeric(crp)
  patient_global_health <- as.numeric(patient_global_health)

  if (any(is.na(c(tender_joint_count, swollen_joint_count, crp, patient_global_health)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (tender_joint_count < 0 || tender_joint_count > 28) stop("Error: Tender joint count must be between 0 and 28.")
  if (swollen_joint_count < 0 || swollen_joint_count > 28) stop("Error: Swollen joint count must be between 0 and 28.")
  if (patient_global_health < 0 || patient_global_health > 100) stop("Error: Patient Global Health must be between 0 and 100.")

  crp_mg_l <- crp
  if (crp_units == "mg/dL") {
    crp_mg_l <- crp * 10
  }

  das28 <- (0.56 * sqrt(tender_joint_count)) +
    (0.28 * sqrt(swollen_joint_count)) +
    (0.36 * log(crp_mg_l + 1)) +
    (0.014 * patient_global_health) +
    0.96

  disease_activity <- ""
  if (das28 < 2.6) {
    disease_activity <- "Remission"
  } else if (das28 >= 2.6 && das28 <= 3.2) {
    disease_activity <- "Low Disease Activity"
  } else if (das28 > 3.2 && das28 <= 5.1) {
    disease_activity <- "Moderate Disease Activity"
  } else {
    disease_activity <- "High Disease Activity"
  }

  return(list(
    DAS28_CRP_Score = round(das28, 2),
    Disease_Activity = disease_activity,
    Inputs = list(
      Tender_Joints = tender_joint_count,
      Swollen_Joints = swollen_joint_count,
      CRP_mg_L = crp_mg_l,
      Global_Health = patient_global_health
    )
  ))
}

das28_esr_rheumatoid_arthritis <- function(tender_joint_count, swollen_joint_count, esr_mm_hr, patient_global_health) {

  tender_joint_count <- as.numeric(tender_joint_count)
  swollen_joint_count <- as.numeric(swollen_joint_count)
  esr_mm_hr <- as.numeric(esr_mm_hr)
  patient_global_health <- as.numeric(patient_global_health)

  if (any(is.na(c(tender_joint_count, swollen_joint_count, esr_mm_hr, patient_global_health)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (tender_joint_count < 0 || tender_joint_count > 28) stop("Error: Tender joint count must be between 0 and 28.")
  if (swollen_joint_count < 0 || swollen_joint_count > 28) stop("Error: Swollen joint count must be between 0 and 28.")
  if (patient_global_health < 0 || patient_global_health > 100) stop("Error: Patient Global Health must be between 0 and 100.")

  das28 <- (0.56 * sqrt(tender_joint_count)) +
    (0.28 * sqrt(swollen_joint_count)) +
    (0.70 * log(esr_mm_hr)) +
    (0.014 * patient_global_health)

  disease_activity <- ""
  if (das28 < 2.6) {
    disease_activity <- "Remission"
  } else if (das28 >= 2.6 && das28 <= 3.2) {
    disease_activity <- "Low Disease Activity"
  } else if (das28 > 3.2 && das28 <= 5.1) {
    disease_activity <- "Moderate Disease Activity"
  } else {
    disease_activity <- "High Disease Activity"
  }

  return(list(
    DAS28_ESR_Score = round(das28, 2),
    Disease_Activity = disease_activity,
    Inputs = list(
      Tender_Joints = tender_joint_count,
      Swollen_Joints = swollen_joint_count,
      ESR_mm_hr = esr_mm_hr,
      Global_Health = patient_global_health
    )
  ))
}

disease_steps_ms <- function(classification) {

  classification <- as.numeric(classification)

  if (is.na(classification) || !classification %in% 0:6) {
    stop("Error: Classification must be an integer between 0 and 6.")
  }

  description <- ""

  if (classification == 0) {
    description <- "Normal: Normal examination. Functional status normal."
  } else if (classification == 1) {
    description <- "Mild Disability: Mild signs or symptoms. No visible effect on gait."
  } else if (classification == 2) {
    description <- "Moderate Disability: Visibly abnormal gait. Can walk 25 feet in <= 20 seconds."
  } else if (classification == 3) {
    description <- "Early Cane: Intermittent use of cane (or other unilateral support). Can walk 25 feet in <= 20 seconds."
  } else if (classification == 4) {
    description <- "Late Cane: Cane dependent. Cannot walk 25 feet without a cane (or other unilateral support)."
  } else if (classification == 5) {
    description <- "Bilateral Support: Requires bilateral support (canes, crutches, walker) to walk 25 feet."
  } else {
    description <- "Confined to Wheelchair: Cannot walk 25 feet even with bilateral support. Uses wheelchair."
  }

  return(list(
    Disease_Step = classification,
    Clinical_Description = description
  ))
}

dli_volume_calculator <- function(recipient_weight_kg, target_cd3_dose_per_kg, donor_wbc_count_cells_ul, donor_lymphocyte_percent, donor_cd3_percent, collection_efficiency_percent, num_infusions) {

  recipient_weight_kg <- as.numeric(recipient_weight_kg)
  target_cd3_dose_per_kg <- as.numeric(target_cd3_dose_per_kg)
  donor_wbc_count_cells_ul <- as.numeric(donor_wbc_count_cells_ul)
  donor_lymphocyte_percent <- as.numeric(donor_lymphocyte_percent)
  donor_cd3_percent <- as.numeric(donor_cd3_percent)
  collection_efficiency_percent <- as.numeric(collection_efficiency_percent)
  num_infusions <- as.numeric(num_infusions)

  if (any(is.na(c(recipient_weight_kg, target_cd3_dose_per_kg, donor_wbc_count_cells_ul, donor_lymphocyte_percent, donor_cd3_percent, collection_efficiency_percent, num_infusions)))) {
    stop("Error: All inputs must be numeric.")
  }

  total_cd3_cells_needed <- recipient_weight_kg * target_cd3_dose_per_kg * num_infusions

  donor_wbc_cells_ml <- donor_wbc_count_cells_ul * 1000

  donor_cd3_conc_cells_ml <- donor_wbc_cells_ml * (donor_lymphocyte_percent / 100) * (donor_cd3_percent / 100)


  efficiency_decimal <- collection_efficiency_percent / 100

  if (donor_cd3_conc_cells_ml <= 0 || efficiency_decimal <= 0) {
    stop("Error: Donor cell concentration or collection efficiency is zero or invalid.")
  }

  volume_to_process_ml <- total_cd3_cells_needed / (donor_cd3_conc_cells_ml * efficiency_decimal)

  volume_to_process_l <- volume_to_process_ml / 1000

  return(list(
    Volume_to_Process_L = round(volume_to_process_l, 2),
    Total_CD3_Cells_Required = total_cd3_cells_needed,
    Donor_CD3_Concentration_cells_mL = round(donor_cd3_conc_cells_ml, 0),
    Inputs = list(
      Recipient_Weight_kg = recipient_weight_kg,
      Target_Dose = target_cd3_dose_per_kg,
      Infusions_Count = num_infusions,
      Efficiency_Percent = collection_efficiency_percent,
      Donor_WBC = donor_wbc_count_cells_ul,
      Donor_Lymph_Pct = donor_lymphocyte_percent,
      Donor_CD3_Pct = donor_cd3_percent
    )
  ))
}

dragon_score_stroke <- function(hyperdense_cerebral_artery, early_infarct_signs, mrs_prestroke_gt_1, age, glucose, onset_to_treatment_gt_90, baseline_nihss, glucose_units = "mg/dL") {

  hyperdense_cerebral_artery <- as.numeric(hyperdense_cerebral_artery)
  early_infarct_signs <- as.numeric(early_infarct_signs)
  mrs_prestroke_gt_1 <- as.numeric(mrs_prestroke_gt_1)
  age <- as.numeric(age)
  glucose <- as.numeric(glucose)
  onset_to_treatment_gt_90 <- as.numeric(onset_to_treatment_gt_90)
  baseline_nihss <- as.numeric(baseline_nihss)

  if (any(is.na(c(hyperdense_cerebral_artery, early_infarct_signs, mrs_prestroke_gt_1, age, glucose, onset_to_treatment_gt_90, baseline_nihss)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (hyperdense_cerebral_artery == 1 && early_infarct_signs == 1) {
    score <- score + 2
  } else if (hyperdense_cerebral_artery == 1 || early_infarct_signs == 1) {
    score <- score + 1
  }

  if (mrs_prestroke_gt_1 == 1) {
    score <- score + 1
  }

  if (age >= 80) {
    score <- score + 2
  } else if (age >= 65) {
    score <- score + 1
  }

  glucose_mg_dl <- glucose
  if (glucose_units == "mmol/L") {
    glucose_mg_dl <- glucose * 18.018
  }

  if (glucose_mg_dl > 144) {
    score <- score + 1
  }

  if (onset_to_treatment_gt_90 == 1) {
    score <- score + 1
  }

  if (baseline_nihss > 15) {
    score <- score + 3
  } else if (baseline_nihss >= 10) {
    score <- score + 2
  } else if (baseline_nihss >= 5) {
    score <- score + 1
  }

  outcome_probability <- ""
  risk_category <- ""

  if (score <= 2) {
    risk_category <- "Good Prognosis"
    outcome_probability <- "High likelihood of good functional outcome (mRS 0-2) at 3 months (>90%)."
  } else if (score <= 7) {
    risk_category <- "Intermediate Prognosis"
    outcome_probability <- "Variable outcome. Increasing score correlates with worse prognosis."
  } else {
    risk_category <- "Poor Prognosis"
    outcome_probability <- "High likelihood of poor functional outcome (mRS 5-6) at 3 months. Scores 8-10 associated with 0% good outcome in validation cohorts."
  }

  return(list(
    DRAGON_Score = score,
    Risk_Category = risk_category,
    Outcome_Probability_3_Months = outcome_probability,
    Inputs = list(
      CT_Findings = if (hyperdense_cerebral_artery == 1 && early_infarct_signs == 1) "Both" else if (hyperdense_cerebral_artery == 1 || early_infarct_signs == 1) "One" else "None",
      PreStroke_mRS_gt_1 = mrs_prestroke_gt_1 == 1,
      Age = age,
      Glucose_gt_144 = glucose_mg_dl > 144,
      Onset_to_Treatment_gt_90min = onset_to_treatment_gt_90 == 1,
      Baseline_NIHSS = baseline_nihss
    )
  ))
}

dast_10_screening <- function(used_drugs_non_medical, abuse_multiple_drugs, able_to_stop_when_wanted, blackouts_flashbacks, feel_guilty, spouse_parents_complain, neglected_family, illegal_activities, withdrawal_symptoms, medical_problems) {

  used_drugs_non_medical <- as.numeric(used_drugs_non_medical)
  abuse_multiple_drugs <- as.numeric(abuse_multiple_drugs)
  able_to_stop_when_wanted <- as.numeric(able_to_stop_when_wanted)
  blackouts_flashbacks <- as.numeric(blackouts_flashbacks)
  feel_guilty <- as.numeric(feel_guilty)
  spouse_parents_complain <- as.numeric(spouse_parents_complain)
  neglected_family <- as.numeric(neglected_family)
  illegal_activities <- as.numeric(illegal_activities)
  withdrawal_symptoms <- as.numeric(withdrawal_symptoms)
  medical_problems <- as.numeric(medical_problems)

  if (any(is.na(c(used_drugs_non_medical, abuse_multiple_drugs, able_to_stop_when_wanted, blackouts_flashbacks, feel_guilty, spouse_parents_complain, neglected_family, illegal_activities, withdrawal_symptoms, medical_problems)))) {
    stop("Error: All inputs must be numeric (0 for No, 1 for Yes).")
  }

  score <- 0

  if (used_drugs_non_medical == 1) score <- score + 1
  if (abuse_multiple_drugs == 1) score <- score + 1

  if (able_to_stop_when_wanted == 0) score <- score + 1

  if (blackouts_flashbacks == 1) score <- score + 1
  if (feel_guilty == 1) score <- score + 1
  if (spouse_parents_complain == 1) score <- score + 1
  if (neglected_family == 1) score <- score + 1
  if (illegal_activities == 1) score <- score + 1
  if (withdrawal_symptoms == 1) score <- score + 1
  if (medical_problems == 1) score <- score + 1

  severity <- ""
  action <- ""

  if (score == 0) {
    severity <- "No problems reported"
    action <- "None suggested; monitor at re-assessment."
  } else if (score <= 2) {
    severity <- "Low Level"
    action <- "Monitor, reassess at a later date."
  } else if (score <= 5) {
    severity <- "Moderate Level"
    action <- "Further investigation. Consider Brief Intervention."
  } else if (score <= 8) {
    severity <- "Substantial Level"
    action <- "Intensive assessment required."
  } else {
    severity <- "Severe Level"
    action <- "Intensive assessment required."
  }

  return(list(
    DAST_10_Score = score,
    Severity_Level = severity,
    Suggested_Action = action,
    Inputs = list(
      Used_Non_Medical = used_drugs_non_medical == 1,
      Multiple_Drugs = abuse_multiple_drugs == 1,
      Unable_To_Stop = able_to_stop_when_wanted == 0,
      Blackouts = blackouts_flashbacks == 1,
      Guilt = feel_guilty == 1,
      Complaints = spouse_parents_complain == 1,
      Neglect = neglected_family == 1,
      Illegal_Acts = illegal_activities == 1,
      Withdrawal = withdrawal_symptoms == 1,
      Medical_Issues = medical_problems == 1
    )
  ))
}

drip_pneumonia_score <- function(antibiotic_use_60d, long_term_care, tube_feeding, prior_drug_resistant_organism_1yr, hospitalization_60d, chronic_pulmonary_disease, poor_functional_status, acid_suppression_14d, wound_care, mrsa_colonization_1yr) {

  antibiotic_use_60d <- as.numeric(antibiotic_use_60d)
  long_term_care <- as.numeric(long_term_care)
  tube_feeding <- as.numeric(tube_feeding)
  prior_drug_resistant_organism_1yr <- as.numeric(prior_drug_resistant_organism_1yr)
  hospitalization_60d <- as.numeric(hospitalization_60d)
  chronic_pulmonary_disease <- as.numeric(chronic_pulmonary_disease)
  poor_functional_status <- as.numeric(poor_functional_status)
  acid_suppression_14d <- as.numeric(acid_suppression_14d)
  wound_care <- as.numeric(wound_care)
  mrsa_colonization_1yr <- as.numeric(mrsa_colonization_1yr)

  if (any(is.na(c(antibiotic_use_60d, long_term_care, tube_feeding, prior_drug_resistant_organism_1yr, hospitalization_60d, chronic_pulmonary_disease, poor_functional_status, acid_suppression_14d, wound_care, mrsa_colonization_1yr)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (antibiotic_use_60d == 1) score <- score + 2
  if (long_term_care == 1) score <- score + 2
  if (tube_feeding == 1) score <- score + 2
  if (prior_drug_resistant_organism_1yr == 1) score <- score + 2

  if (hospitalization_60d == 1) score <- score + 1
  if (chronic_pulmonary_disease == 1) score <- score + 1
  if (poor_functional_status == 1) score <- score + 1
  if (acid_suppression_14d == 1) score <- score + 1
  if (wound_care == 1) score <- score + 1
  if (mrsa_colonization_1yr == 1) score <- score + 1

  risk_level <- ""
  recommendation <- ""

  if (score < 4) {
    risk_level <- "Low Risk of Drug-Resistant Pneumonia"
    recommendation <- "Consider treating without extended-spectrum antibiotics (e.g., standard CAP therapy)."
  } else {
    risk_level <- "High Risk of Drug-Resistant Pneumonia"
    recommendation <- "Extended-spectrum antibiotics likely necessary (e.g., coverage for MRSA/Pseudomonas)."
  }

  return(list(
    DRIP_Score = score,
    Risk_Level = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      Antibiotics_60d = antibiotic_use_60d == 1,
      LTC_Resident = long_term_care == 1,
      Tube_Feeding = tube_feeding == 1,
      Prior_DRP_1yr = prior_drug_resistant_organism_1yr == 1,
      Hospitalization_60d = hospitalization_60d == 1,
      Chronic_Lung_Disease = chronic_pulmonary_disease == 1,
      Poor_Functional_Status = poor_functional_status == 1,
      Acid_Suppression_14d = acid_suppression_14d == 1,
      Wound_Care = wound_care == 1,
      MRSA_Colonization = mrsa_colonization_1yr == 1
    )
  ))
}

dsm5_binge_eating_disorder <- function(recurrent_episodes, symptom_count, marked_distress, time_criteria_met, no_compensatory_behavior, frequency_per_week) {

  recurrent_episodes <- as.numeric(recurrent_episodes)
  symptom_count <- as.numeric(symptom_count)
  marked_distress <- as.numeric(marked_distress)
  time_criteria_met <- as.numeric(time_criteria_met)
  no_compensatory_behavior <- as.numeric(no_compensatory_behavior)
  frequency_per_week <- as.numeric(frequency_per_week)

  if (any(is.na(c(recurrent_episodes, symptom_count, marked_distress, time_criteria_met, no_compensatory_behavior, frequency_per_week)))) {
    stop("Error: All inputs must be numeric.")
  }

  criteria_met <- TRUE
  missing_criteria <- c()

  if (recurrent_episodes == 0) {
    criteria_met <- FALSE
    missing_criteria <- c(missing_criteria, "Criterion A: Recurrent episodes of binge eating")
  }

  if (symptom_count < 3) {
    criteria_met <- FALSE
    missing_criteria <- c(missing_criteria, "Criterion B: Less than 3 associated symptoms (e.g., eating rapidy, until uncomfortable)")
  }

  if (marked_distress == 0) {
    criteria_met <- FALSE
    missing_criteria <- c(missing_criteria, "Criterion C: Marked distress regarding binge eating")
  }

  if (time_criteria_met == 0) {
    criteria_met <- FALSE
    missing_criteria <- c(missing_criteria, "Criterion D: Frequency < 1/week for 3 months")
  }

  if (no_compensatory_behavior == 0) {
    criteria_met <- FALSE
    missing_criteria <- c(missing_criteria, "Criterion E: Presence of compensatory behavior (Bulimia) or occurs strictly during Anorexia")
  }

  diagnosis <- ""
  severity <- "N/A"

  if (criteria_met) {
    diagnosis <- "Criteria Met for Binge Eating Disorder"

    if (frequency_per_week >= 1 && frequency_per_week <= 3) {
      severity <- "Mild (1-3 episodes/week)"
    } else if (frequency_per_week >= 4 && frequency_per_week <= 7) {
      severity <- "Moderate (4-7 episodes/week)"
    } else if (frequency_per_week >= 8 && frequency_per_week <= 13) {
      severity <- "Severe (8-13 episodes/week)"
    } else if (frequency_per_week >= 14) {
      severity <- "Extreme (>=14 episodes/week)"
    } else {
      severity <- "Frequency Unspecified"
    }
  } else {
    diagnosis <- "Criteria Not Met"
  }

  return(list(
    Diagnosis = diagnosis,
    Severity_Specification = severity,
    Missing_Criteria = if (length(missing_criteria) > 0) missing_criteria else "None",
    Inputs = list(
      Recurrent_Episodes = recurrent_episodes == 1,
      Symptom_Count = symptom_count,
      Distress = marked_distress == 1,
      Duration_Met = time_criteria_met == 1,
      Exclusion_Criteria_Met = no_compensatory_behavior == 1,
      Episodes_Per_Week = frequency_per_week
    )
  ))
}

dsm5_major_depressive_disorder <- function(depressed_mood, diminished_interest, weight_change, sleep_disturbance, psychomotor_change, fatigue, worthlessness_guilt, diminished_concentration, suicidal_ideation) {

  depressed_mood <- as.numeric(depressed_mood)
  diminished_interest <- as.numeric(diminished_interest)
  weight_change <- as.numeric(weight_change)
  sleep_disturbance <- as.numeric(sleep_disturbance)
  psychomotor_change <- as.numeric(psychomotor_change)
  fatigue <- as.numeric(fatigue)
  worthlessness_guilt <- as.numeric(worthlessness_guilt)
  diminished_concentration <- as.numeric(diminished_concentration)
  suicidal_ideation <- as.numeric(suicidal_ideation)

  if (any(is.na(c(depressed_mood, diminished_interest, weight_change, sleep_disturbance, psychomotor_change, fatigue, worthlessness_guilt, diminished_concentration, suicidal_ideation)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  symptoms <- c(depressed_mood, diminished_interest, weight_change, sleep_disturbance, psychomotor_change, fatigue, worthlessness_guilt, diminished_concentration, suicidal_ideation)

  symptom_count <- sum(symptoms)

  criteria_met <- FALSE

  if (symptom_count >= 5) {
    if (depressed_mood == 1 || diminished_interest == 1) {
      criteria_met <- TRUE
    }
  }

  diagnosis <- ""
  if (criteria_met) {
    diagnosis <- "Criteria Met for Major Depressive Disorder"
  } else {
    diagnosis <- "Criteria Not Met"
  }

  return(list(
    Diagnosis = diagnosis,
    Symptom_Count = symptom_count,
    Criteria_Check = list(
      Core_Symptoms_Present = (depressed_mood == 1 || diminished_interest == 1),
      Total_Symptoms_ge_5 = symptom_count >= 5
    )
  ))
}

dsm5_bipolar_criteria <- function(symptom_count, irritable_mood_only, duration_days, hospitalized, psychotic_features, marked_impairment, history_mde, substance_medical_excluded) {

  symptom_count <- as.numeric(symptom_count)
  irritable_mood_only <- as.numeric(irritable_mood_only)
  duration_days <- as.numeric(duration_days)
  hospitalized <- as.numeric(hospitalized)
  psychotic_features <- as.numeric(psychotic_features)
  marked_impairment <- as.numeric(marked_impairment)
  history_mde <- as.numeric(history_mde)
  substance_medical_excluded <- as.numeric(substance_medical_excluded)

  if (any(is.na(c(symptom_count, irritable_mood_only, duration_days, hospitalized, psychotic_features, marked_impairment, history_mde, substance_medical_excluded)))) {
    stop("Error: All inputs must be numeric.")
  }

  required_symptoms <- if (irritable_mood_only == 1) 4 else 3

  symptoms_met <- symptom_count >= required_symptoms

  is_manic <- FALSE
  is_hypomanic <- FALSE

  if (substance_medical_excluded == 1 && symptoms_met) {
    if ((duration_days >= 7 || hospitalized == 1) && (marked_impairment == 1 || psychotic_features == 1 || hospitalized == 1)) {
      is_manic <- TRUE
    } else if (duration_days >= 4 && marked_impairment == 0 && psychotic_features == 0 && hospitalized == 0) {
      is_hypomanic <- TRUE
    }
  }

  diagnosis <- ""
  episode_type <- ""

  if (is_manic) {
    episode_type <- "Manic Episode"
    diagnosis <- "Bipolar I Disorder"
  } else if (is_hypomanic) {
    episode_type <- "Hypomanic Episode"
    if (history_mde == 1) {
      diagnosis <- "Bipolar II Disorder"
    } else {
      diagnosis <- "Criteria for Bipolar I or II not met (Hypomania present without MDE)."
    }
  } else {
    episode_type <- "No Manic/Hypomanic Episode defined"
    diagnosis <- "Criteria for Bipolar I or II not met."
  }

  return(list(
    Diagnosis = diagnosis,
    Current_Episode_Type = episode_type,
    Inputs = list(
      Symptoms_Count = symptom_count,
      Threshold_Needed = required_symptoms,
      Duration = duration_days,
      Severe_Features = (hospitalized == 1 || psychotic_features == 1 || marked_impairment == 1),
      History_MDE = history_mde == 1
    )
  ))
}

dsm5_ptsd_criteria <- function(exposure_criterion_a, intrusion_count_b, avoidance_count_c, cognition_mood_count_d, arousal_reactivity_count_e, duration_gt_1month, distress_impairment, exclusion_criteria_ruled_out, dissociative_symptoms = 0) {

  exposure_criterion_a <- as.numeric(exposure_criterion_a)
  intrusion_count_b <- as.numeric(intrusion_count_b)
  avoidance_count_c <- as.numeric(avoidance_count_c)
  cognition_mood_count_d <- as.numeric(cognition_mood_count_d)
  arousal_reactivity_count_e <- as.numeric(arousal_reactivity_count_e)
  duration_gt_1month <- as.numeric(duration_gt_1month)
  distress_impairment <- as.numeric(distress_impairment)
  exclusion_criteria_ruled_out <- as.numeric(exclusion_criteria_ruled_out)
  dissociative_symptoms <- as.numeric(dissociative_symptoms)

  if (any(is.na(c(exposure_criterion_a, intrusion_count_b, avoidance_count_c, cognition_mood_count_d, arousal_reactivity_count_e, duration_gt_1month, distress_impairment, exclusion_criteria_ruled_out)))) {
    stop("Error: All inputs must be numeric.")
  }

  criteria_met <- TRUE
  missing_list <- c()

  if (exposure_criterion_a == 0) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion A (Stressor)")
  }

  if (intrusion_count_b < 1) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion B (Intrusion symptoms: Need 1)")
  }

  if (avoidance_count_c < 1) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion C (Avoidance: Need 1)")
  }

  if (cognition_mood_count_d < 2) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion D (Negative alterations in cognitions/mood: Need 2)")
  }

  if (arousal_reactivity_count_e < 2) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion E (Alterations in arousal/reactivity: Need 2)")
  }

  if (duration_gt_1month == 0) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion F (Duration > 1 month)")
  }

  if (distress_impairment == 0) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion G (Functional impairment)")
  }

  if (exclusion_criteria_ruled_out == 0) {
    criteria_met <- FALSE
    missing_list <- c(missing_list, "Criterion H (Not due to medication/substance/illness)")
  }

  diagnosis <- ""
  subtype <- ""

  if (criteria_met) {
    diagnosis <- "Criteria Met for PTSD"
    if (dissociative_symptoms == 1) {
      subtype <- "With Dissociative Symptoms (Depersonalization/Derealization)"
    }
  } else {
    diagnosis <- "Criteria Not Met"
  }

  return(list(
    Diagnosis = diagnosis,
    Subtype = subtype,
    Missing_Criteria = if (length(missing_list) > 0) missing_list else "None",
    Inputs = list(
      Criterion_A = exposure_criterion_a == 1,
      Criterion_B_Count = intrusion_count_b,
      Criterion_C_Count = avoidance_count_c,
      Criterion_D_Count = cognition_mood_count_d,
      Criterion_E_Count = arousal_reactivity_count_e,
      Duration_Met = duration_gt_1month == 1,
      Functional_Impairment = distress_impairment == 1,
      Exclusions_Clear = exclusion_criteria_ruled_out == 1
    )
  ))
}

du_bois_ipf_score <- function(age, history_resp_hospitalization, baseline_fvc_percent, fvc_change_24wk_percent) {

  age <- as.numeric(age)
  history_resp_hospitalization <- as.numeric(history_resp_hospitalization)
  baseline_fvc_percent <- as.numeric(baseline_fvc_percent)
  fvc_change_24wk_percent <- as.numeric(fvc_change_24wk_percent)

  if (any(is.na(c(age, history_resp_hospitalization, baseline_fvc_percent, fvc_change_24wk_percent)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 60 && age <= 69) {
    score <- score + 4
  } else if (age >= 70) {
    score <- score + 8
  }

  if (history_resp_hospitalization == 1) {
    score <- score + 14
  }

  if (baseline_fvc_percent <= 50) {
    score <- score + 18
  } else if (baseline_fvc_percent >= 51 && baseline_fvc_percent <= 65) {
    score <- score + 13
  } else if (baseline_fvc_percent >= 66 && baseline_fvc_percent <= 79) {
    score <- score + 8
  }

  if (fvc_change_24wk_percent <= -10) {
    score <- score + 21
  } else if (fvc_change_24wk_percent >= -9.9 && fvc_change_24wk_percent <= -5) {
    score <- score + 10
  }

  est_mortality <- ""

  if (score <= 4) {
    est_mortality <- "< 2%"
  } else if (score >= 8 && score <= 14) {
    est_mortality <- "2 - 5%"
  } else if (score >= 16 && score <= 21) {
    est_mortality <- "5 - 10%"
  } else if (score >= 22 && score <= 29) {
    est_mortality <- "10 - 20%"
  } else if (score >= 30 && score <= 33) {
    est_mortality <- "20 - 30%"
  } else if (score >= 34 && score <= 37) {
    est_mortality <- "30 - 40%"
  } else if (score >= 38 && score <= 40) {
    est_mortality <- "40 - 50%"
  } else if (score >= 41 && score <= 43) {
    est_mortality <- "50 - 60%"
  } else if (score >= 44 && score <= 45) {
    est_mortality <- "60 - 70%"
  } else if (score >= 47 && score <= 49) {
    est_mortality <- "70 - 80%"
  } else {
    est_mortality <- "> 80%"
  }

  return(list(
    du_Bois_Score = score,
    Est_1_Year_Mortality = est_mortality,
    Inputs = list(
      Age = age,
      Hx_Resp_Hospitalization = history_resp_hospitalization == 1,
      Baseline_FVC_Pct = baseline_fvc_percent,
      FVC_Change_24wk_Pct = fvc_change_24wk_percent
    )
  ))
}

dapt_score <- function(age, smoker, diabetes, mi_presentation, prior_pci_mi, paclitaxel_stent, stent_diameter_lt_3mm, chf_lvef_lt_30, vein_graft_stent) {

  age <- as.numeric(age)
  smoker <- as.numeric(smoker)
  diabetes <- as.numeric(diabetes)
  mi_presentation <- as.numeric(mi_presentation)
  prior_pci_mi <- as.numeric(prior_pci_mi)
  paclitaxel_stent <- as.numeric(paclitaxel_stent)
  stent_diameter_lt_3mm <- as.numeric(stent_diameter_lt_3mm)
  chf_lvef_lt_30 <- as.numeric(chf_lvef_lt_30)
  vein_graft_stent <- as.numeric(vein_graft_stent)

  if (any(is.na(c(age, smoker, diabetes, mi_presentation, prior_pci_mi, paclitaxel_stent, stent_diameter_lt_3mm, chf_lvef_lt_30, vein_graft_stent)))) {
    stop("Error: All inputs must be numeric (0 or 1 for boolean, age as number).")
  }

  score <- 0

  if (age >= 75) {
    score <- score - 2
  } else if (age >= 65 && age < 75) {
    score <- score - 1
  } else {
    score <- score + 0
  }

  if (smoker == 1) {
    score <- score + 1
  }

  if (diabetes == 1) {
    score <- score + 1
  }

  if (mi_presentation == 1) {
    score <- score + 1
  }

  if (prior_pci_mi == 1) {
    score <- score + 1
  }

  if (paclitaxel_stent == 1) {
    score <- score + 1
  }

  if (stent_diameter_lt_3mm == 1) {
    score <- score + 1
  }

  if (chf_lvef_lt_30 == 1) {
    score <- score + 2
  }

  if (vein_graft_stent == 1) {
    score <- score + 2
  }

  recommendation <- ""
  risk_profile <- ""

  if (score < 2) {
    risk_profile <- "Low Score (< 2)"
    recommendation <- "Standard DAPT duration (12 months) is favored. The risk of bleeding exceeds the benefit of reduced ischemia with prolonged therapy."
  } else {
    risk_profile <- "High Score (>= 2)"
    recommendation <- "Prolonged DAPT duration (> 12 months, e.g., 30 months) is favored. The reduction in ischemic events outweighs the increased risk of bleeding."
  }

  return(list(
    DAPT_Score = score,
    Risk_Profile = risk_profile,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Smoker = smoker == 1,
      Diabetes = diabetes == 1,
      MI_Presentation = mi_presentation == 1,
      Prior_PCI_MI = prior_pci_mi == 1,
      Paclitaxel_Stent = paclitaxel_stent == 1,
      Small_Stent = stent_diameter_lt_3mm == 1,
      CHF_LVEF = chf_lvef_lt_30 == 1,
      Vein_Graft = vein_graft_stent == 1
    )
  ))
}

duke_activity_status_index <- function(self_care, walk_indoors, walk_blocks, climb_stairs, run_short, light_housework, moderate_housework, heavy_housework, yard_work, sexual_relations, moderate_recreation, strenuous_sports) {

  self_care <- as.numeric(self_care)
  walk_indoors <- as.numeric(walk_indoors)
  walk_blocks <- as.numeric(walk_blocks)
  climb_stairs <- as.numeric(climb_stairs)
  run_short <- as.numeric(run_short)
  light_housework <- as.numeric(light_housework)
  moderate_housework <- as.numeric(moderate_housework)
  heavy_housework <- as.numeric(heavy_housework)
  yard_work <- as.numeric(yard_work)
  sexual_relations <- as.numeric(sexual_relations)
  moderate_recreation <- as.numeric(moderate_recreation)
  strenuous_sports <- as.numeric(strenuous_sports)

  if (any(is.na(c(self_care, walk_indoors, walk_blocks, climb_stairs, run_short, light_housework, moderate_housework, heavy_housework, yard_work, sexual_relations, moderate_recreation, strenuous_sports)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (self_care == 1) score <- score + 2.75
  if (walk_indoors == 1) score <- score + 1.75
  if (walk_blocks == 1) score <- score + 2.75
  if (climb_stairs == 1) score <- score + 5.50
  if (run_short == 1) score <- score + 8.00
  if (light_housework == 1) score <- score + 2.70
  if (moderate_housework == 1) score <- score + 3.50
  if (heavy_housework == 1) score <- score + 8.00
  if (yard_work == 1) score <- score + 4.50
  if (sexual_relations == 1) score <- score + 5.25
  if (moderate_recreation == 1) score <- score + 6.00
  if (strenuous_sports == 1) score <- score + 7.50

  vo2_peak <- (0.43 * score) + 9.6

  mets <- vo2_peak / 3.5

  return(list(
    DASI_Score = round(score, 2),
    Estimated_VO2_Peak_mL_kg_min = round(vo2_peak, 1),
    Estimated_METs = round(mets, 1),
    Inputs = list(
      Self_Care = self_care == 1,
      Walk_Indoors = walk_indoors == 1,
      Walk_Blocks = walk_blocks == 1,
      Climb_Stairs = climb_stairs == 1,
      Run = run_short == 1,
      Light_Work = light_housework == 1,
      Mod_Work = moderate_housework == 1,
      Heavy_Work = heavy_housework == 1,
      Yard_Work = yard_work == 1,
      Sex = sexual_relations == 1,
      Mod_Rec = moderate_recreation == 1,
      Strenuous = strenuous_sports == 1
    )
  ))
}

duke_endocarditis_criteria <- function(pathologic_lesions, major_blood_cultures, major_echo, predisposition_heart_condition, fever_gt_38, vascular_phenomena, immunologic_phenomena, minor_blood_culture_findings) {

  pathologic_lesions <- as.numeric(pathologic_lesions)
  major_blood_cultures <- as.numeric(major_blood_cultures)
  major_echo <- as.numeric(major_echo)
  predisposition_heart_condition <- as.numeric(predisposition_heart_condition)
  fever_gt_38 <- as.numeric(fever_gt_38)
  vascular_phenomena <- as.numeric(vascular_phenomena)
  immunologic_phenomena <- as.numeric(immunologic_phenomena)
  minor_blood_culture_findings <- as.numeric(minor_blood_culture_findings)

  if (any(is.na(c(pathologic_lesions, major_blood_cultures, major_echo, predisposition_heart_condition, fever_gt_38, vascular_phenomena, immunologic_phenomena, minor_blood_culture_findings)))) {
    stop("Error: All inputs must be numeric.")
  }

  major_count <- major_blood_cultures + major_echo
  minor_count <- predisposition_heart_condition + fever_gt_38 + vascular_phenomena + immunologic_phenomena + minor_blood_culture_findings

  classification <- "Rejected"

  if (pathologic_lesions == 1) {
    classification <- "Definite Infective Endocarditis"
  } else if (major_count == 2) {
    classification <- "Definite Infective Endocarditis"
  } else if (major_count == 1 && minor_count >= 3) {
    classification <- "Definite Infective Endocarditis"
  } else if (minor_count >= 5) {
    classification <- "Definite Infective Endocarditis"
  } else if (major_count == 1 && minor_count >= 1) {
    classification <- "Possible Infective Endocarditis"
  } else if (minor_count >= 3) {
    classification <- "Possible Infective Endocarditis"
  } else {
    classification <- "Rejected"
  }

  return(list(
    Classification = classification,
    Criteria_Met = list(
      Major_Criteria_Count = major_count,
      Minor_Criteria_Count = minor_count,
      Pathologic_Evidence = pathologic_lesions == 1
    )
  ))
}

duke_treadmill_score <- function(exercise_time_min, max_st_depression_mm, angina_index) {

  exercise_time_min <- as.numeric(exercise_time_min)
  max_st_depression_mm <- as.numeric(max_st_depression_mm)
  angina_index <- as.numeric(angina_index)

  if (any(is.na(c(exercise_time_min, max_st_depression_mm, angina_index)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!angina_index %in% c(0, 1, 2)) {
    stop("Error: Angina Index must be 0 (None), 1 (Non-limiting), or 2 (Limiting).")
}

  score <- exercise_time_min - (5 * max_st_depression_mm) - (4 * angina_index)

  risk_group <- ""
  annual_mortality <- ""
  survival_5yr <- ""

  if (score >= 5) {
    risk_group <- "Low Risk"
    annual_mortality <- "0.25%"
    survival_5yr <- "99%"
  } else if (score >= -10) {
    risk_group <- "Moderate Risk"
    annual_mortality <- "1.25%"
    survival_5yr <- "95%"
  } else {
    risk_group <- "High Risk"
    annual_mortality <- "5.25%"
    survival_5yr <- "79%"
  }

  return(list(
    Duke_Score = round(score, 1),
    Risk_Group = risk_group,
    Annual_Mortality_Risk = annual_mortality,
    Five_Year_Survival = survival_5yr,
    Inputs = list(
      Time_min = exercise_time_min,
      Max_ST_Depression_mm = max_st_depression_mm,
      Angina_Index = switch(as.character(angina_index), "0"="None", "1"="Non-limiting", "2"="Limiting")
    )
  ))
}

dutch_lipid_clinic_score <- function(ldl_cholesterol, family_history_score, personal_premature_cad, personal_premature_cerebral_peripheral, exam_tendon_xanthoma, exam_arcus_cornealis_lt45, genetic_mutation, ldl_units = "mg/dL") {

  ldl_cholesterol <- as.numeric(ldl_cholesterol)
  family_history_score <- as.numeric(family_history_score)
  personal_premature_cad <- as.numeric(personal_premature_cad)
  personal_premature_cerebral_peripheral <- as.numeric(personal_premature_cerebral_peripheral)
  exam_tendon_xanthoma <- as.numeric(exam_tendon_xanthoma)
  exam_arcus_cornealis_lt45 <- as.numeric(exam_arcus_cornealis_lt45)
  genetic_mutation <- as.numeric(genetic_mutation)

  if (any(is.na(c(ldl_cholesterol, family_history_score, personal_premature_cad, personal_premature_cerebral_peripheral, exam_tendon_xanthoma, exam_arcus_cornealis_lt45, genetic_mutation)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!family_history_score %in% c(0, 1, 2)) {
    stop("Error: Family History Score must be 0, 1, or 2 based on specific criteria met.")
  }

  score <- 0

  score <- score + family_history_score

  if (personal_premature_cad == 1) {
    score <- score + 2
  }

  if (personal_premature_cerebral_peripheral == 1) {
    score <- score + 1
  }

  exam_points <- 0
  if (exam_tendon_xanthoma == 1) {
    exam_points <- 6
  } else if (exam_arcus_cornealis_lt45 == 1) {
    exam_points <- 4
  }
  score <- score + exam_points

  ldl_mg_dl <- ldl_cholesterol
  if (ldl_units == "mmol/L") {
    ldl_mg_dl <- ldl_cholesterol * 38.67
  }

  if (ldl_mg_dl >= 330) {
    score <- score + 8
  } else if (ldl_mg_dl >= 250) {
    score <- score + 5
  } else if (ldl_mg_dl >= 190) {
    score <- score + 3
  } else if (ldl_mg_dl >= 155) {
    score <- score + 1
  }

  if (genetic_mutation == 1) {
    score <- score + 8
  }

  diagnosis <- ""
  if (score > 8) {
    diagnosis <- "Definite Familial Hypercholesterolemia"
  } else if (score >= 6) {
    diagnosis <- "Probable Familial Hypercholesterolemia"
  } else if (score >= 3) {
    diagnosis <- "Possible Familial Hypercholesterolemia"
  } else {
    diagnosis <- "Unlikely Familial Hypercholesterolemia"
  }

  return(list(
    DLCN_Score = score,
    Diagnosis = diagnosis,
    Inputs = list(
      LDL_mg_dL = round(ldl_mg_dl, 1),
      Family_Hx_Points = family_history_score,
      Personal_Hx_Points = (personal_premature_cad * 2) + (personal_premature_cerebral_peripheral * 1),
      Exam_Points = exam_points,
      Genetic_Points = genetic_mutation * 8
    )
  ))
}

delta_p_lems_score <- function(bulbar_weakness, erectile_dysfunction_male, weight_loss_ge_5pct, tobacco_at_onset, age_onset_ge_50, kps_lt_70) {

  bulbar_weakness <- as.numeric(bulbar_weakness)
  erectile_dysfunction_male <- as.numeric(erectile_dysfunction_male)
  weight_loss_ge_5pct <- as.numeric(weight_loss_ge_5pct)
  tobacco_at_onset <- as.numeric(tobacco_at_onset)
  age_onset_ge_50 <- as.numeric(age_onset_ge_50)
  kps_lt_70 <- as.numeric(kps_lt_70)

  if (any(is.na(c(bulbar_weakness, erectile_dysfunction_male, weight_loss_ge_5pct, tobacco_at_onset, age_onset_ge_50, kps_lt_70)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (bulbar_weakness == 1) {
    score <- score + 1
  }

  if (erectile_dysfunction_male == 1) {
    score <- score + 1
  }

  if (weight_loss_ge_5pct == 1) {
    score <- score + 1
  }

  if (tobacco_at_onset == 1) {
    score <- score + 1
  }

  if (age_onset_ge_50 == 1) {
    score <- score + 1
  }

  if (kps_lt_70 == 1) {
    score <- score + 1
  }

  risk_probability <- ""
  if (score <= 1) {
    risk_probability <- "0% - 2.6% (Low Risk)"
  } else if (score == 2) {
    risk_probability <- "29% (Intermediate Risk)"
  } else if (score == 3) {
    risk_probability <- "84% (High Risk)"
  } else if (score == 4) {
    risk_probability <- "94% (High Risk)"
  } else if (score == 5) {
    risk_probability <- "97% (High Risk)"
  } else {
    risk_probability <- "100% (Very High Risk)"
  }

  return(list(
    DELTA_P_Score = score,
    SCLC_Probability = risk_probability,
    Inputs = list(
      Bulbar_Weakness = bulbar_weakness == 1,
      Male_Impotence = erectile_dysfunction_male == 1,
      Weight_Loss = weight_loss_ge_5pct == 1,
      Tobacco_Use = tobacco_at_onset == 1,
      Age_ge_50 = age_onset_ge_50 == 1,
      KPS_lt_70 = kps_lt_70 == 1
    )
  ))
}

duval_cibmtr_aml_survival <- function(disease_group, cytogenetics, hla_match, circulating_blasts, karnofsky_score) {

  disease_group <- as.numeric(disease_group)
  cytogenetics <- as.numeric(cytogenetics)
  hla_match <- as.numeric(hla_match)
  circulating_blasts <- as.numeric(circulating_blasts)
  karnofsky_score <- as.numeric(karnofsky_score)

  if (any(is.na(c(disease_group, cytogenetics, hla_match, circulating_blasts, karnofsky_score)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (disease_group == 1) {
    score <- score + 1
  }

  if (cytogenetics == 1) {
    score <- score + 1
  }

  if (hla_match == 1) {
    score <- score + 2
  }

  if (circulating_blasts == 1) {
    score <- score + 1
  }

  if (karnofsky_score == 1) {
    score <- score + 1
  }

  estimated_survival <- ""

  if (score == 0) {
    estimated_survival <- "42%"
  } else if (score == 1) {
    estimated_survival <- "28%"
  } else if (score == 2) {
    estimated_survival <- "15%"
  } else {
    estimated_survival <- "6%"
  }

  return(list(
    Duval_CIBMTR_Score = score,
    Estimated_3_Year_Survival = estimated_survival,
    Inputs = list(
      Disease_Status_Duration_CR1_lt_6mo = disease_group == 1,
      Poor_Cytogenetics = cytogenetics == 1,
      Donor_Mismatch = hla_match == 1,
      Circulating_Blasts = circulating_blasts == 1,
      KPS_lt_90 = karnofsky_score == 1
    )
  ))
}

csha_clinical_frailty_scale <- function(cfs_level) {

  cfs_level <- as.numeric(cfs_level)

  if (is.na(cfs_level) || !cfs_level %in% 1:9) {
    stop("Error: CFS Level must be an integer between 1 and 9.")
  }

  category <- ""
  description <- ""

  if (cfs_level == 1) {
    category <- "Very Fit"
    description <- "Robust, active, energetic, and motivated. Exercises regularly. Among the fittest for their age."
  } else if (cfs_level == 2) {
    category <- "Well"
    description <- "No active disease symptoms but less fit than Level 1. Exercises or is active occasionally."
  } else if (cfs_level == 3) {
    category <- "Managing Well"
    description <- "Medical problems are well controlled, but not regularly active beyond routine walking."
  } else if (cfs_level == 4) {
    category <- "Vulnerable"
    description <- "Not dependent on others for daily help, but symptoms limit activities. Common complaints are being 'slowed up' or tired."
  } else if (cfs_level == 5) {
    category <- "Mildly Frail"
    description <- "Evident slowing, needs help with high-order IADLs (finances, transportation, heavy housework, meds). Impaired shopping/walking outside alone."
  } else if (cfs_level == 6) {
    category <- "Moderately Frail"
    description <- "Needs help with all outside activities and keeping house. Inside, problems with stairs, bathing, and minimal assistance with dressing."
  } else if (cfs_level == 7) {
    category <- "Severely Frail"
    description <- "Completely dependent for personal care. Seem stable and not at high risk of dying (within ~6 months)."
  } else if (cfs_level == 8) {
    category <- "Very Severely Frail"
    description <- "Completely dependent, approaching end of life. Could not recover even from a minor illness."
  } else {
    category <- "Terminally Ill"
    description <- "Approaching the end of life (life expectancy <6 months) but not otherwise evidently frail."
  }

  return(list(
    CFS_Level = cfs_level,
    Category = category,
    Description = description
  ))
}

cryoprecipitate_dosing <- function(current_fibrinogen, target_fibrinogen, weight_kg, hematocrit, fibrinogen_per_unit_mg = 250) {

  current_fibrinogen <- as.numeric(current_fibrinogen)
  target_fibrinogen <- as.numeric(target_fibrinogen)
  weight_kg <- as.numeric(weight_kg)
  hematocrit <- as.numeric(hematocrit)
  fibrinogen_per_unit_mg <- as.numeric(fibrinogen_per_unit_mg)

  if (any(is.na(c(current_fibrinogen, target_fibrinogen, weight_kg, hematocrit, fibrinogen_per_unit_mg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  if (target_fibrinogen <= current_fibrinogen) {
    return(list(
      Units_Needed = 0,
      Message = "Current fibrinogen is already at or above target."
    ))
  }

  total_blood_volume_ml <- weight_kg * 70

  plasma_volume_ml <- total_blood_volume_ml * (1 - (hematocrit / 100))

  plasma_volume_dl <- plasma_volume_ml / 100

  fibrinogen_deficit_mg_dl <- target_fibrinogen - current_fibrinogen

  total_fibrinogen_needed_mg <- fibrinogen_deficit_mg_dl * plasma_volume_dl

  units_needed <- total_fibrinogen_needed_mg / fibrinogen_per_unit_mg

  return(list(
    Estimated_Units = ceiling(units_needed),
    Exact_Units_Calculation = round(units_needed, 2),
    Total_Fibrinogen_Required_mg = round(total_fibrinogen_needed_mg, 0),
    Plasma_Volume_dL = round(plasma_volume_dl, 1),
    Inputs = list(
      Current_Fib = current_fibrinogen,
      Target_Fib = target_fibrinogen,
      Weight = weight_kg,
      Hct = hematocrit,
      Fib_Per_Unit = fibrinogen_per_unit_mg
    )
  ))
}

cirs_geriatric_score <- function(heart, vascular, hematopoietic, respiratory, eent, upper_gi, lower_gi, liver, renal, gu, msk, neuro, endocrine, psych) {

  inputs <- c(heart, vascular, hematopoietic, respiratory, eent, upper_gi, lower_gi, liver, renal, gu, msk, neuro, endocrine, psych)

  if (any(is.na(inputs))) {
    stop("Error: All 14 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be integers between 0 and 4.")
  }

  total_score <- sum(inputs)
  severity_index <- total_score / 14

  comorbidities_count <- sum(inputs > 0)

  return(list(
    CIRS_G_Total_Score = total_score,
    Severity_Index = round(severity_index, 2),
    Number_of_Categories_Endorsed = comorbidities_count,
    Inputs = list(
      Heart = heart, Vascular = vascular, Hematopoietic = hematopoietic,
      Respiratory = respiratory, EENT = eent, Upper_GI = upper_gi, Lower_GI = lower_gi,
      Liver = liver, Renal = renal, GU = gu, MSK = msk, Neuro = neuro,
      Endocrine = endocrine, Psych = psych
    )
  ))
}

ecog_performance_status <- function(grade) {

  grade <- as.numeric(grade)

  if (is.na(grade) || !grade %in% 0:5) {
    stop("Error: ECOG Grade must be an integer between 0 and 5.")
  }

  definition <- ""

  if (grade == 0) {
    definition <- "Fully active, able to carry on all pre-disease performance without restriction."
  } else if (grade == 1) {
    definition <- "Restricted in physically strenuous activity but ambulatory and able to carry out work of a light or sedentary nature (e.g., light house work, office work)."
  } else if (grade == 2) {
    definition <- "Ambulatory and capable of all selfcare but unable to carry out any work activities; up and about more than 50% of waking hours."
  } else if (grade == 3) {
    definition <- "Capable of only limited selfcare; confined to bed or chair more than 50% of waking hours."
  } else if (grade == 4) {
    definition <- "Completely disabled; cannot carry on any selfcare; totally confined to bed or chair."
  } else {
    definition <- "Dead."
  }

  return(list(
    ECOG_Grade = grade,
    Definition = definition
  ))
}

eat_sleep_console_score <- function(eating_well, sleeping_well, easy_to_console) {

  eating_well <- as.numeric(eating_well)
  sleeping_well <- as.numeric(sleeping_well)
  easy_to_console <- as.numeric(easy_to_console)

  if (any(is.na(c(eating_well, sleeping_well, easy_to_console)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (eating_well == 1) {
    score <- score + 1
  }

  if (sleeping_well == 1) {
    score <- score + 1
  }

  if (easy_to_console == 1) {
    score <- score + 1
  }

  management_recommendation <- ""

  if (score == 3) {
    management_recommendation <- "Well managed. No further interventions necessary."
  } else {
    management_recommendation <- "Increase non-pharmacologic interventions (feeding on demand, swaddling, low stimulation, parental presence). If no improvement, consider NICU consult and pharmacotherapy."
  }

  return(list(
    ESC_Total_Score = score,
    Recommendation = management_recommendation,
    Inputs = list(
      Eat = eating_well == 1,
      Sleep = sleeping_well == 1,
      Console = easy_to_console == 1
    )
  ))
}

easi_score <- function(age_years, area_scores, erythema_scores, induration_scores, excoriation_scores, lichenification_scores) {

  if (length(area_scores) != 4 || length(erythema_scores) != 4 ||
      length(induration_scores) != 4 || length(excoriation_scores) != 4 ||
      length(lichenification_scores) != 4) {
    stop("Error: All symptom input vectors must have exactly 4 elements (Head, Upper, Trunk, Lower).")
  }

  if (any(is.na(c(age_years, area_scores, erythema_scores, induration_scores, excoriation_scores, lichenification_scores)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(area_scores < 0) || any(area_scores > 6)) stop("Error: Area scores must be between 0 and 6.")
  if (any(erythema_scores < 0) || any(erythema_scores > 3)) stop("Error: Erythema scores must be between 0 and 3.")
  if (any(induration_scores < 0) || any(induration_scores > 3)) stop("Error: Induration/Papulation scores must be between 0 and 3.")
  if (any(excoriation_scores < 0) || any(excoriation_scores > 3)) stop("Error: Excoriation scores must be between 0 and 3.")
  if (any(lichenification_scores < 0) || any(lichenification_scores > 3)) stop("Error: Lichenification scores must be between 0 and 3.")

  multipliers <- numeric(4)

  if (age_years < 8) {
    multipliers <- c(0.2, 0.2, 0.3, 0.3)
  } else {
    multipliers <- c(0.1, 0.2, 0.3, 0.4)
  }

  region_scores <- numeric(4)

  for (i in 1:4) {
    sum_severity <- erythema_scores[i] + induration_scores[i] + excoriation_scores[i] + lichenification_scores[i]
    region_scores[i] <- sum_severity * area_scores[i] * multipliers[i]
  }

  total_score <- sum(region_scores)

  severity_category <- ""

  if (total_score == 0) {
    severity_category <- "Clear"
  } else if (total_score <= 1.0) {
    severity_category <- "Almost Clear"
  } else if (total_score <= 7.0) {
    severity_category <- "Mild"
  } else if (total_score <= 21.0) {
    severity_category <- "Moderate"
  } else if (total_score <= 50.0) {
    severity_category <- "Severe"
  } else {
    severity_category <- "Very Severe"
  }

  return(list(
    EASI_Score = round(total_score, 1),
    Severity_Category = severity_category,
    Inputs = list(
      Age = age_years,
      Multipliers_Used = multipliers,
      Region_Scores = region_scores
    )
  ))
}

pss_3_screener <- function(depressed_2weeks, suicidal_thoughts_2weeks, suicide_attempt_lifetime) {

  depressed_2weeks <- as.numeric(depressed_2weeks)
  suicidal_thoughts_2weeks <- as.numeric(suicidal_thoughts_2weeks)
  suicide_attempt_lifetime <- as.numeric(suicide_attempt_lifetime)

  if (any(is.na(c(depressed_2weeks, suicidal_thoughts_2weeks, suicide_attempt_lifetime)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  screen_positive <- FALSE
  risk_status <- "Negative"

  if (suicidal_thoughts_2weeks == 1 || suicide_attempt_lifetime == 1) {
    screen_positive <- TRUE
    risk_status <- "Positive for Suicide Risk"
  } else if (depressed_2weeks == 1) {
    screen_positive <- FALSE
    risk_status <- "Positive for Depression (Negative for acute suicide risk based on PSS-3 alone)"
  } else {
    screen_positive <- FALSE
    risk_status <- "Negative"
  }

  action <- ""
  if (screen_positive) {
    action <- "Further assessment required (e.g., C-SSRS, Safety Plan, Mental Health Consult)."
  } else if (risk_status == "Positive for Depression (Negative for acute suicide risk based on PSS-3 alone)") {
    action <- "Evaluate for depression; no immediate suicide precautions indicated by this screen."
  } else {
    action <- "No further suicide risk intervention required."
  }

  return(list(
    Screen_Result = if(screen_positive) "Positive" else "Negative",
    Risk_Interpretation = risk_status,
    Recommended_Action = action,
    Inputs = list(
      Depression = depressed_2weeks == 1,
      Suicidal_Ideation = suicidal_thoughts_2weeks == 1,
      Lifetime_Attempt = suicide_attempt_lifetime == 1
    )
  ))
}

edinburgh_gastric_ulcer_score <- function(age, ulcer_size_cm, ulcer_location) {

  age <- as.numeric(age)
  ulcer_size_cm <- as.numeric(ulcer_size_cm)
  ulcer_location <- tolower(as.character(ulcer_location))

  if (any(is.na(c(age, ulcer_size_cm)))) {
    stop("Error: Age and Ulcer Size must be numeric.")
  }

  if (ulcer_size_cm <= 0) {
    stop("Error: Ulcer size must be greater than 0.")
  }

  if (!ulcer_location %in% c("antral", "non-antral")) {
    stop("Error: Ulcer Location must be 'antral' or 'non-antral'.")
  }

  score <- 0

  if (age < 68) {
    score <- score + 0
  } else if (age >= 68 && age <= 79) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (ulcer_size_cm < 1.25) {
    score <- score + 0
  } else if (ulcer_size_cm >= 1.25 && ulcer_size_cm < 3.00) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (ulcer_location == "non-antral") {
    score <- score + 1
  }

  risk_category <- ""
  recommendation <- ""

  if (score <= 2) {
    risk_category <- "Low Risk (Score 0-2)"
    recommendation <- "Confirm >=6 benign biopsies and symptom resolution. Repeat upper endoscopy may be avoided; reassure patient but advise return if alarm symptoms arise."
  } else {
    risk_category <- "High Risk (Score >= 3)"
    recommendation <- "Arrange repeat upper endoscopy in 6-8 weeks (or sooner if symptoms persist) to confirm healing and obtain additional biopsies."
  }

  return(list(
    EGUS_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Size_cm = ulcer_size_cm,
      Location = ulcer_location
    )
  ))
}

epds_score <- function(q1_laugh, q2_enjoy, q3_blame, q4_anxious, q5_scared, q6_coping, q7_sleep, q8_sad, q9_crying, q10_self_harm) {

  inputs <- c(q1_laugh, q2_enjoy, q3_blame, q4_anxious, q5_scared, q6_coping, q7_sleep, q8_sad, q9_crying, q10_self_harm)

  if (any(is.na(inputs))) {
    stop("Error: All 10 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 3)) {
    stop("Error: All item scores must be integers between 0 and 3.")
  }

  total_score <- sum(inputs)

  interpretation <- ""
  action <- ""

  if (total_score < 10) {
    interpretation <- "Depression not likely"
    action <- "Continue routine postnatal care."
  } else if (total_score >= 10 && total_score <= 12) {
    interpretation <- "Possible depression"
    action <- "Repeat EPDS in 2 weeks and monitor regularly."
  } else {
    interpretation <- "Probable depression (High likelihood)"
    action <- "Further assessment and treatment warranted (referral)."
  }

  suicide_risk <- FALSE
  if (q10_self_harm > 0) {
    suicide_risk <- TRUE
    action <- paste("IMMEDIATE ATTENTION REQUIRED: Patient reported thoughts of self-harm.", action)
  }

  return(list(
    EPDS_Total_Score = total_score,
    Interpretation = interpretation,
    Action_Plan = action,
    Suicide_Risk_Flag = suicide_risk,
    Inputs = list(
      Q1_Laugh = q1_laugh,
      Q2_Enjoy = q2_enjoy,
      Q3_Blame = q3_blame,
      Q4_Anxious = q4_anxious,
      Q5_Scared = q5_scared,
      Q6_Coping = q6_coping,
      Q7_Sleep = q7_sleep,
      Q8_Sad = q8_sad,
      Q9_Crying = q9_crying,
      Q10_Self_Harm = q10_self_harm
    )
  ))
}

edmonton_obesity_staging_system <- function(severe_functional_limitations, end_organ_damage, established_comorbidity, subclinical_risk_factors) {

  severe_functional_limitations <- as.numeric(severe_functional_limitations)
  end_organ_damage <- as.numeric(end_organ_damage)
  established_comorbidity <- as.numeric(established_comorbidity)
  subclinical_risk_factors <- as.numeric(subclinical_risk_factors)

  if (any(is.na(c(severe_functional_limitations, end_organ_damage, established_comorbidity, subclinical_risk_factors)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  stage <- 0
  description <- "No risk factors. No physical or psychological symptoms."
  management <- "Counseling on healthy diet and exercise. Prevention."

  if (severe_functional_limitations == 1) {
    stage <- 4
    description <- "Severe (End-stage) disabilities from obesity-related chronic diseases. Severe psychopathology."
    management <- "Palliative care, pain management, aggressive medical management. Bariatric surgery may be high risk."
  } else if (end_organ_damage == 1) {
    stage <- 3
    description <- "Established end-organ damage (e.g., MI, Heart Failure, Stroke) or significant functional impairment."
    management <- "Intensive medical management. Bariatric surgery is a strong option."
  } else if (established_comorbidity == 1) {
    stage <- 2
    description <- "Established obesity-related chronic disease (e.g., HTN, T2DM, Sleep Apnea) or moderate limitations."
    management <- "Medical management of comorbidities. Lifestyle interventions. Consider bariatric surgery."
  } else if (subclinical_risk_factors == 1) {
    stage <- 1
    description <- "Subclinical obesity-related risk factors (e.g., Borderline HTN, Impaired Fasting Glucose) or mild symptoms."
    management <- "Lifestyle interventions (Diet/Exercise). Monitoring of risk factors."
  }

  return(list(
    EOSS_Stage = stage,
    Description = description,
    Management_Recommendation = management,
    Inputs = list(
      Stage4_Criteria = severe_functional_limitations == 1,
      Stage3_Criteria = end_organ_damage == 1,
      Stage2_Criteria = established_comorbidity == 1,
      Stage1_Criteria = subclinical_risk_factors == 1
    )
  ))
}

esas_r_assessment <- function(pain, tiredness, drowsiness, nausea, lack_of_appetite, shortness_of_breath, depression, anxiety, wellbeing, other_symptom_score = NULL) {

  inputs <- c(pain, tiredness, drowsiness, nausea, lack_of_appetite, shortness_of_breath, depression, anxiety, wellbeing)

  if (!is.null(other_symptom_score)) {
    other_symptom_score <- as.numeric(other_symptom_score)
    if (is.na(other_symptom_score) || other_symptom_score < 0 || other_symptom_score > 10) {
      stop("Error: Other symptom score must be a number between 0 and 10.")
    }
  }

  if (any(is.na(inputs))) {
    stop("Error: All 9 core symptom inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 10)) {
    stop("Error: All symptom scores must be between 0 and 10.")
  }

  physical_subscore <- pain + tiredness + drowsiness + nausea + lack_of_appetite + shortness_of_breath
  emotional_subscore <- depression + anxiety
  total_distress_score <- physical_subscore + emotional_subscore + wellbeing

  if (!is.null(other_symptom_score)) {
    total_distress_score <- total_distress_score + other_symptom_score
  }

  interpret_severity <- function(x) {
    if (x == 0) return("None")
    if (x <= 3) return("Mild")
    if (x <= 6) return("Moderate")
    return("Severe")
  }

  severity_profile <- sapply(inputs, interpret_severity)
  names(severity_profile) <- c("Pain", "Tiredness", "Drowsiness", "Nausea", "Appetite", "SOB", "Depression", "Anxiety", "Wellbeing")

  return(list(
    Total_Symptom_Distress_Score = total_distress_score,
    Subscores = list(
      Physical = physical_subscore,
      Emotional = emotional_subscore
    ),
    Severity_Profile = severity_profile,
    Inputs = list(
      Pain = pain,
      Tiredness = tiredness,
      Drowsiness = drowsiness,
      Nausea = nausea,
      Appetite = lack_of_appetite,
      SOB = shortness_of_breath,
      Depression = depression,
      Anxiety = anxiety,
      Wellbeing = wellbeing,
      Other = if (!is.null(other_symptom_score)) other_symptom_score else "N/A"
    )
  ))
}

egsys_syncope_score <- function(palpitations, abnormal_ecg_or_heart_disease, effort_syncope, supine_syncope, autonomic_prodrome, predisposing_factors) {

  palpitations <- as.numeric(palpitations)
  abnormal_ecg_or_heart_disease <- as.numeric(abnormal_ecg_or_heart_disease)
  effort_syncope <- as.numeric(effort_syncope)
  supine_syncope <- as.numeric(supine_syncope)
  autonomic_prodrome <- as.numeric(autonomic_prodrome)
  predisposing_factors <- as.numeric(predisposing_factors)

  if (any(is.na(c(palpitations, abnormal_ecg_or_heart_disease, effort_syncope, supine_syncope, autonomic_prodrome, predisposing_factors)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (palpitations == 1) {
    score <- score + 4
  }

  if (abnormal_ecg_or_heart_disease == 1) {
    score <- score + 3
  }

  if (effort_syncope == 1) {
    score <- score + 3
  }

  if (supine_syncope == 1) {
    score <- score + 2
  }

  if (autonomic_prodrome == 1) {
    score <- score - 1
  }

  if (predisposing_factors == 1) {
    score <- score - 1
  }

  interpretation <- ""
  etiology <- ""

  if (score < 3) {
    interpretation <- "Low risk of cardiac etiology."
    etiology <- "Likely Neurally Mediated Syncope"
  } else {
    interpretation <- "High risk of cardiac etiology."
    etiology <- "Likely Cardiac Syncope"
  }

  return(list(
    EGSYS_Score = score,
    Interpretation = interpretation,
    Likely_Etiology = etiology,
    Inputs = list(
      Palpitations = palpitations == 1,
      Abnormal_ECG_Heart_Disease = abnormal_ecg_or_heart_disease == 1,
      Effort_Related = effort_syncope == 1,
      Supine = supine_syncope == 1,
      Prodrome_Nausea_Vomiting = autonomic_prodrome == 1,
      Predisposing_Factors = predisposing_factors == 1
    )
  ))
}

el_ganzouri_risk_index <- function(mouth_opening_cm, thyromental_distance_cm, mallampati_score, neck_movement_degrees, can_prognath, weight_kg, history_difficult_intubation) {

  mouth_opening_cm <- as.numeric(mouth_opening_cm)
  thyromental_distance_cm <- as.numeric(thyromental_distance_cm)
  mallampati_score <- as.numeric(mallampati_score)
  neck_movement_degrees <- as.numeric(neck_movement_degrees)
  can_prognath <- as.numeric(can_prognath)
  weight_kg <- as.numeric(weight_kg)
  history_difficult_intubation <- tolower(as.character(history_difficult_intubation))

  if (any(is.na(c(mouth_opening_cm, thyromental_distance_cm, mallampati_score, neck_movement_degrees, can_prognath, weight_kg)))) {
    stop("Error: All physiological inputs must be numeric.")
  }

  if (!history_difficult_intubation %in% c("none", "questionable", "definite")) {
    stop("Error: History of difficult intubation must be 'none', 'questionable', or 'definite'.")
  }

  if (!mallampati_score %in% 1:4) {
    stop("Error: Mallampati Score must be 1, 2, 3, or 4.")
  }

  score <- 0

  if (mouth_opening_cm >= 4) {
    score <- score + 0
  } else {
    score <- score + 1
  }

  if (thyromental_distance_cm > 6.5) {
    score <- score + 0
  } else if (thyromental_distance_cm >= 6.0) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (mallampati_score == 1 || mallampati_score == 2) {
    score <- score + 0
  } else if (mallampati_score == 3) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (neck_movement_degrees > 90) {
    score <- score + 0
  } else if (neck_movement_degrees >= 80) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (can_prognath == 1) {
    score <- score + 0
  } else {
    score <- score + 1
  }

  if (weight_kg < 90) {
    score <- score + 0
  } else if (weight_kg >= 90 && weight_kg < 110) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (history_difficult_intubation == "none") {
    score <- score + 0
  } else if (history_difficult_intubation == "questionable") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  interpretation <- ""
  if (score >= 4) {
    interpretation <- "High Risk for Difficult Intubation"
  } else {
    interpretation <- "Low Risk for Difficult Intubation"
  }

  return(list(
    EGRI_Score = score,
    Risk_Category = interpretation,
    Inputs = list(
      Mouth_Opening_cm = mouth_opening_cm,
      TMD_cm = thyromental_distance_cm,
      Mallampati_Class = mallampati_score,
      Neck_Movement_Deg = neck_movement_degrees,
      Prognath_Ability = if (can_prognath == 1) "Yes" else "No",
      Weight_kg = weight_kg,
      Hx_Difficult_Intubation = history_difficult_intubation
    )
  ))
}

esus_criteria <- function(ischemic_stroke_ct_mri, extracranial_stenosis_lt_50, afib_flutter_absent, cardiac_embolism_source_absent, other_causes_absent) {

  ischemic_stroke_ct_mri <- as.numeric(ischemic_stroke_ct_mri)
  extracranial_stenosis_lt_50 <- as.numeric(extracranial_stenosis_lt_50)
  afib_flutter_absent <- as.numeric(afib_flutter_absent)
  cardiac_embolism_source_absent <- as.numeric(cardiac_embolism_source_absent)
  other_causes_absent <- as.numeric(other_causes_absent)

  if (any(is.na(c(ischemic_stroke_ct_mri, extracranial_stenosis_lt_50, afib_flutter_absent, cardiac_embolism_source_absent, other_causes_absent)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  criteria_met <- TRUE
  reasons_not_met <- c()

  if (ischemic_stroke_ct_mri == 0) {
    criteria_met <- FALSE
    reasons_not_met <- c(reasons_not_met, "No non-lacunar ischemic stroke visualized on CT/MRI.")
  }

  if (extracranial_stenosis_lt_50 == 0) {
    criteria_met <- FALSE
    reasons_not_met <- c(reasons_not_met, "Extracranial or intracranial atherosclerosis >= 50% causing stenosis present.")
  }

  if (afib_flutter_absent == 0) {
    criteria_met <- FALSE
    reasons_not_met <- c(reasons_not_met, "Atrial Fibrillation or Flutter detected (requires >24h monitoring).")
  }

  if (cardiac_embolism_source_absent == 0) {
    criteria_met <- FALSE
    reasons_not_met <- c(reasons_not_met, "Major risk cardioembolic source present (e.g., intracardiac thrombus, prosthetic valve, etc.).")
  }

  if (other_causes_absent == 0) {
    criteria_met <- FALSE
    reasons_not_met <- c(reasons_not_met, "Other specific cause of stroke identified (e.g., arteritis, dissection, migraine).")
  }

  diagnosis <- ""
  if (criteria_met) {
    diagnosis <- "Criteria Met for Embolic Stroke of Undetermined Source (ESUS)"
  } else {
    diagnosis <- "Criteria Not Met for ESUS"
  }

  return(list(
    Diagnosis = diagnosis,
    Criteria_Met = criteria_met,
    Missing_Criteria = if (length(reasons_not_met) > 0) reasons_not_met else "None"
  ))
}

edacs_chest_pain_score <- function(age, sex, known_cad_or_3_risk_factors, diaphoresis, radiation, pain_inspiration, pain_palpation, ekg_ischemia_present = FALSE, troponin_normal = TRUE) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  known_cad_or_3_risk_factors <- as.numeric(known_cad_or_3_risk_factors)
  diaphoresis <- as.numeric(diaphoresis)
  radiation <- as.numeric(radiation)
  pain_inspiration <- as.numeric(pain_inspiration)
  pain_palpation <- as.numeric(pain_palpation)
  ekg_ischemia_present <- as.numeric(ekg_ischemia_present)
  troponin_normal <- as.numeric(troponin_normal)

  if (any(is.na(c(age, known_cad_or_3_risk_factors, diaphoresis, radiation, pain_inspiration, pain_palpation)))) {
    stop("Error: All clinical inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (age < 18) {
    return(list(Error = "EDACS is validated for patients >= 18 years old."))
  }

  score <- 0

  if (age >= 18 && age <= 45) {
    score <- score + 2
  } else if (age >= 46 && age <= 50) {
    score <- score + 4
  } else if (age >= 51 && age <= 55) {
    score <- score + 6
  } else if (age >= 56 && age <= 60) {
    score <- score + 8
  } else if (age >= 61 && age <= 65) {
    score <- score + 10
  } else if (age >= 66 && age <= 70) {
    score <- score + 12
  } else if (age >= 71 && age <= 75) {
    score <- score + 14
  } else if (age >= 76 && age <= 80) {
    score <- score + 16
  } else if (age >= 81 && age <= 85) {
    score <- score + 18
  } else {
    score <- score + 20
  }

  if (sex == "male") {
    score <- score + 6
  }

  if (known_cad_or_3_risk_factors == 1) {
    score <- score + 4
  }

  if (diaphoresis == 1) {
    score <- score + 3
  }

  if (radiation == 1) {
    score <- score + 5
  }

  if (pain_inspiration == 1) {
    score <- score - 4
  }

  if (pain_palpation == 1) {
    score <- score - 6
  }

  protocol_result <- ""
  action <- ""

  if (score < 16 && ekg_ischemia_present == 0 && troponin_normal == 1) {
    protocol_result <- "Low Risk"
    action <- "Safe for early discharge to outpatient follow-up (Risk of Major Adverse Cardiac Event < 1%)."
  } else {
    protocol_result <- "Not Low Risk"
    action <- "Proceed with usual care (observation, serial troponins, further testing)."
  }

  return(list(
    EDACS_Score = score,
    Risk_Category = protocol_result,
    Recommendation = action,
    Inputs = list(
      Age = age,
      Sex = sex,
      History_Risk = known_cad_or_3_risk_factors == 1,
      Diaphoresis = diaphoresis == 1,
      Radiation = radiation == 1,
      Inspiration_Pain = pain_inspiration == 1,
      Palpation_Pain = pain_palpation == 1,
      EKG_Ischemia = ekg_ischemia_present == 1,
      Troponin_Normal = troponin_normal == 1
    )
  ))
}

embed_buprenorphine_initiation <- function(oud_present, treatment_readiness, withdrawal_severity_cows, pregnant, waiver_status) {

  oud_present <- as.numeric(oud_present)
  treatment_readiness <- as.numeric(treatment_readiness)
  withdrawal_severity_cows <- as.numeric(withdrawal_severity_cows)
  pregnant <- as.numeric(pregnant)
  waiver_status <- as.numeric(waiver_status)

  if (any(is.na(c(oud_present, treatment_readiness, withdrawal_severity_cows, pregnant, waiver_status)))) {
    stop("Error: All inputs must be numeric.")
  }

  action <- ""

  if (oud_present == 0) {
    action <- "No OUD: Do not initiate buprenorphine."
  } else {
    if (treatment_readiness == 0) {
      action <- "Not Ready: Offer harm reduction (Naloxone), referral info, and discharge."
    } else {
      if (pregnant == 1) {
        action <- "Pregnant: Consult OB/GYN or Addiction Specialist. Buprenorphine mono-product often preferred."
      } else {
        if (withdrawal_severity_cows < 8) {
          action <- "Mild Withdrawal (COWS < 8): Do not initiate yet (Risk of precipitated withdrawal). Provide home induction instructions or observe."
        } else if (withdrawal_severity_cows >= 8) {
          if (waiver_status == 1) {
            action <- "Moderate/Severe Withdrawal: Initiate Buprenorphine (e.g., 4-8mg SL). Observe. Prescribe maintenance."
          } else {
            action <- "Moderate/Severe Withdrawal: Administer '3-Day Rule' dose in ED. Refer to waivered provider."
          }
        }
      }
    }
  }

  return(list(
    Recommendation = action,
    Inputs = list(
      OUD_Diagnosis = oud_present == 1,
      Ready_for_Tx = treatment_readiness == 1,
      COWS_Score = withdrawal_severity_cows,
      Pregnant = pregnant == 1,
      Provider_Waivered = waiver_status == 1
    )
  ))
}

ehmrg_mortality_score <- function(age, sbp, heart_rate, o2_sat, creatinine, potassium, troponin_positive, active_cancer, creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  sbp <- as.numeric(sbp)
  heart_rate <- as.numeric(heart_rate)
  o2_sat <- as.numeric(o2_sat)
  creatinine <- as.numeric(creatinine)
  potassium <- as.numeric(potassium)
  troponin_positive <- as.numeric(troponin_positive)
  active_cancer <- as.numeric(active_cancer)

  if (any(is.na(c(age, sbp, heart_rate, o2_sat, creatinine, potassium, troponin_positive, active_cancer)))) {
    stop("Error: All inputs must be numeric.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  score <- 0

  score <- score + (2 * age)

  if (sbp < 160) {
    if (sbp < 90) score <- score + 60
    else if (sbp < 110) score <- score + 45
    else if (sbp < 130) score <- score + 30
    else if (sbp < 160) score <- score + 15
  }

  if (heart_rate >= 80 && heart_rate < 100) score <- score + 5
  else if (heart_rate >= 100 && heart_rate < 120) score <- score + 10
  else if (heart_rate >= 120) score <- score + 20

  if (o2_sat < 90) score <- score + 40
  else if (o2_sat <= 95) score <- score + 20


  if (creat_mg_dl >= 2.5) score <- score + 40
  else if (creat_mg_dl >= 1.5) score <- score + 20


  if (potassium >= 4.0 && potassium <= 4.5) score <- score + 5
  else if (potassium > 4.5) score <- score + 10

  if (troponin_positive == 1) {
    score <- score + 60
  }

  if (active_cancer == 1) {
    score <- score + 45
  }

  risk_tier <- ""
  mortality_7d <- ""


  if (score < 60) {
    risk_tier <- "Low Risk"
    mortality_7d <- "< 1%"
  } else if (score < 90) {
    risk_tier <- "Intermediate Risk"
    mortality_7d <- "2-3%"
  } else if (score < 120) {
    risk_tier <- "High Risk"
    mortality_7d <- "5-9%"
  } else {
    risk_tier <- "Very High Risk"
    mortality_7d <- "> 10%"
  }

  return(list(
    EHMRG_Score = score,
    Risk_Classification = risk_tier,
    Est_7_Day_Mortality = mortality_7d,
    Inputs = list(
      Age = age,
      SBP = sbp,
      HR = heart_rate,
      O2_Sat = o2_sat,
      Cr = round(creat_mg_dl, 2),
      K = potassium,
      Trop_Pos = troponin_positive == 1,
      Active_Cancer = active_cancer == 1
    )
  ))
}

ett_depth_tidal_volume <- function(height_cm, sex, tube_size_mm) {

  height_cm <- as.numeric(height_cm)
  sex <- tolower(as.character(sex))
  tube_size_mm <- as.numeric(tube_size_mm)

  if (any(is.na(c(height_cm, tube_size_mm)))) {
    stop("Error: Height and Tube Size must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (height_cm <= 0) {
    stop("Error: Height must be positive.")
  }

  if (tube_size_mm <= 0) {
    stop("Error: Tube size must be positive.")
  }

  pbw <- 0
  if (sex == "male") {
    pbw <- 50 + 0.91 * (height_cm - 152.4)
  } else {
    pbw <- 45.5 + 0.91 * (height_cm - 152.4)
  }

  tv_6ml <- 6 * pbw
  tv_8ml <- 8 * pbw

  depth_chula <- 0.1 * height_cm + 4

  depth_3x <- 3 * tube_size_mm

  return(list(
    Predicted_Body_Weight_kg = round(pbw, 1),
    Tidal_Volume_Target_6_8_ml_kg = paste(round(tv_6ml, 0), "-", round(tv_8ml, 0), "mL"),
    ETT_Depth_Chula_cm = round(depth_chula, 1),
    ETT_Depth_3x_Size_cm = round(depth_3x, 1),
    Inputs = list(
      Height_cm = height_cm,
      Sex = sex,
      Tube_Size_mm = tube_size_mm
    )
  ))
}

erefs_score <- function(edema, rings, exudates, furrows, stricture) {

  edema <- as.numeric(edema)
  rings <- as.numeric(rings)
  exudates <- as.numeric(exudates)
  furrows <- as.numeric(furrows)
  stricture <- as.numeric(stricture)

  if (any(is.na(c(edema, rings, exudates, furrows, stricture)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (edema < 0 || edema > 1) stop("Error: Edema score must be 0 or 1.")
  if (rings < 0 || rings > 3) stop("Error: Rings score must be between 0 and 3.")
  if (exudates < 0 || exudates > 2) stop("Error: Exudates score must be between 0 and 2.")
  if (furrows < 0 || furrows > 2) stop("Error: Furrows score must be between 0 and 2.")
  if (stricture < 0 || stricture > 1) stop("Error: Stricture score must be 0 or 1.")

  inflammatory_score <- edema + exudates + furrows
  fibrostenotic_score <- rings + stricture
  total_score <- inflammatory_score + fibrostenotic_score

  return(list(
    EREFS_Total_Score = total_score,
    Subscores = list(
      Inflammatory_Score = inflammatory_score,
      Fibrostenotic_Score = fibrostenotic_score
    ),
    Inputs = list(
      Edema = edema,
      Rings = rings,
      Exudates = exudates,
      Furrows = furrows,
      Stricture = stricture
    )
  ))
}

estimate_toxic_alcohol_level <- function(substance, volume_ingested, volume_units, concentration_percent, weight_kg, sex = "male") {

  volume_ingested <- as.numeric(volume_ingested)
  concentration_percent <- as.numeric(concentration_percent)
  weight_kg <- as.numeric(weight_kg)
  substance <- tolower(as.character(substance))
  volume_units <- tolower(as.character(volume_units))
  sex <- tolower(as.character(sex))

  if (any(is.na(c(volume_ingested, concentration_percent, weight_kg)))) {
    stop("Error: Volume, Concentration, and Weight must be numeric.")
  }

  if (!substance %in% c("ethanol", "methanol", "ethylene_glycol", "isopropanol")) {
    stop("Error: Substance must be 'ethanol', 'methanol', 'ethylene_glycol', or 'isopropanol'.")
  }

  vol_ml <- volume_ingested
  if (volume_units == "oz") {
    vol_ml <- volume_ingested * 29.5735
  } else if (volume_units == "l") {
    vol_ml <- volume_ingested * 1000
  }

  sg <- 0
  vd_l_kg <- 0

  if (substance == "ethanol") {
    sg <- 0.79
    vd_l_kg <- if (sex == "male") 0.7 else 0.6
  } else if (substance == "methanol") {
    sg <- 0.79
    vd_l_kg <- 0.6
  } else if (substance == "ethylene_glycol") {
    sg <- 1.11
    vd_l_kg <- 0.7
  } else {
    sg <- 0.79
    vd_l_kg <- 0.6
  }

  mass_g <- vol_ml * (concentration_percent / 100) * sg

  total_vd_liters <- vd_l_kg * weight_kg

  peak_conc_g_l <- mass_g / total_vd_liters

  peak_conc_mg_dl <- peak_conc_g_l * 100

  toxic_threshold <- ""
  if (substance == "methanol" || substance == "ethylene_glycol") {
    if (peak_conc_mg_dl >= 20) {
      toxic_threshold <- "Above Toxic Threshold (>= 20 mg/dL). Antidote indicated."
    } else {
      toxic_threshold <- "Below standard treatment threshold (< 20 mg/dL), but monitor for acidosis."
    }
  } else if (substance == "isopropanol") {
    if (peak_conc_mg_dl >= 400) {
      toxic_threshold <- "Severe Toxicity likely (Coma/Hypotension). Hemodialysis may be considered."
    } else {
      toxic_threshold <- "Supportive care usually sufficient."
    }
  } else {
    if (peak_conc_mg_dl >= 300) {
      toxic_threshold <- "Risk of Coma/Respiratory Depression."
    } else if (peak_conc_mg_dl >= 80) {
      toxic_threshold <- "Legal Intoxication (Driving limit typically 80 mg/dL)."
    }
  }

  return(list(
    Estimated_Peak_Concentration_mg_dL = round(peak_conc_mg_dl, 1),
    Substance = substance,
    Interpretation = toxic_threshold,
    Inputs = list(
      Volume_mL = round(vol_ml, 1),
      Concentration_Pct = concentration_percent,
      Weight_kg = weight_kg,
      Specific_Gravity_Used = sg,
      Vd_Used_L_kg = vd_l_kg
    )
  ))
}

estimated_peak_flow <- function(age, height_in, sex, race, actual_pef = NULL) {

  age <- as.numeric(age)
  height_in <- as.numeric(height_in)

  if (any(is.na(c(age, height_in)))) {
    stop("Error: Age and Height must be numeric.")
  }

  height_cm <- height_in * 2.54

  race <- tolower(as.character(race))
  if (!race %in% c("white", "black", "hispanic", "other")) {
    stop("Error: Race must be 'white', 'black', 'hispanic', or 'other'.")
  }

  sex <- tolower(as.character(sex))
  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (age < 8 || age > 80) {
    return(list(
      Error = "The Hankinson (NHANES III) equations are validated for ages 8-80 years."
    ))
  }

  b0 <- 0
  b1 <- 0
  b2 <- 0
  b3 <- 0
  b4 <- 0

  if (sex == "male") {
    if (race == "white" || race == "other") {
      if (age < 20) {
        b0 <- -2.0688
        b1 <- 0.1070
        b2 <- -0.0016
        b3 <- 0.00038
        b4 <- 0
      } else {
        b0 <- 0.5536
        b1 <- -0.0003
        b2 <- -0.0002
        b3 <- 0.0004
        b4 <- 0
      }
    } else if (race == "black") {
      if (age < 20) {
        b0 <- -1.9420
        b1 <- 0.0858
        b2 <- -0.0014
        b3 <- 0.00038
        b4 <- 0
      } else {
        b0 <- 0.3240
        b1 <- 0.0028
        b2 <- -0.00023
        b3 <- 0.00037
        b4 <- 0
      }
    } else {
      if (age < 20) {
        b0 <- -1.6531
        b1 <- 0.0772
        b2 <- -0.0012
        b3 <- 0.00034
        b4 <- 0
      } else {
        b0 <- 0.6812
        b1 <- -0.0023
        b2 <- -0.00017
        b3 <- 0.00035
        b4 <- 0
      }
    }
  } else {
    if (race == "white" || race == "other") {
      if (age < 18) {
        b0 <- -2.0662
        b1 <- 0.0556
        b2 <- 0.0011
        b3 <- 0.00036
        b4 <- 0
      } else {
        b0 <- 0.7107
        b1 <- -0.0077
        b2 <- 0.00004
        b3 <- 0.0003
        b4 <- 0
      }
    } else if (race == "black") {
      if (age < 18) {
        b0 <- -1.5526
        b1 <- 0.0148
        b2 <- 0.0020
        b3 <- 0.00034
        b4 <- 0
      } else {
        b0 <- 0.7526
        b1 <- -0.0031
        b2 <- -0.00007
        b3 <- 0.00025
        b4 <- 0
      }
    } else {
      if (age < 18) {
        b0 <- -1.6575
        b1 <- 0.0371
        b2 <- 0.0013
        b3 <- 0.00035
        b4 <- 0
      } else {
        b0 <- 0.9889
        b1 <- -0.0109
        b2 <- 0.00002
        b3 <- 0.00025
        b4 <- 0
      }
    }
  }

  pef_predicted_ls <- b0 + (b1 * age) + (b2 * (age^2)) + (b3 * (height_cm^2))

  pef_predicted_l_min <- (exp(pef_predicted_ls)) * 60

  percent_predicted <- NA
  if (!is.null(actual_pef)) {
    actual_pef <- as.numeric(actual_pef)
    percent_predicted <- (actual_pef / pef_predicted_l_min) * 100
  }

  return(list(
    Estimated_PEF_L_min = round(pef_predicted_l_min, 0),
    Percent_Predicted = if (!is.na(percent_predicted)) paste0(round(percent_predicted, 1), "%") else NULL,
    Method = "Hankinson (NHANES III) Equations",
    Inputs = list(
      Age = age,
      Height_cm = round(height_cm, 1),
      Sex = sex,
      Race = race
    )
  ))
}

euromacs_rhf_score <- function(cvp_pcwp_ratio_gt_054, multiple_inotropes, severe_rv_dysfunction, intermacs_1_2, hemoglobin_le_10) {

  cvp_pcwp_ratio_gt_054 <- as.numeric(cvp_pcwp_ratio_gt_054)
  multiple_inotropes <- as.numeric(multiple_inotropes)
  severe_rv_dysfunction <- as.numeric(severe_rv_dysfunction)
  intermacs_1_2 <- as.numeric(intermacs_1_2)
  hemoglobin_le_10 <- as.numeric(hemoglobin_le_10)

  if (any(is.na(c(cvp_pcwp_ratio_gt_054, multiple_inotropes, severe_rv_dysfunction, intermacs_1_2, hemoglobin_le_10)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (cvp_pcwp_ratio_gt_054 == 1) {
    score <- score + 2
  }

  if (multiple_inotropes == 1) {
    score <- score + 2.5
  }

  if (severe_rv_dysfunction == 1) {
    score <- score + 2
  }

  if (intermacs_1_2 == 1) {
    score <- score + 2
  }

  if (hemoglobin_le_10 == 1) {
    score <- score + 1
  }

  risk_level <- ""
  risk_percentage <- ""

  if (score <= 2) {
    risk_level <- "Low Risk"
    risk_percentage <- "11%"
  } else if (score <= 4) {
    risk_level <- "Intermediate Risk"
    risk_percentage <- "28%"
  } else {
    risk_level <- "High Risk"
    risk_percentage <- "63%"
  }

  return(list(
    EUROMACS_Score = score,
    Risk_Category = risk_level,
    Early_RHF_Risk = risk_percentage,
    Inputs = list(
      CVP_PCWP_Ratio_gt_0_54 = cvp_pcwp_ratio_gt_054 == 1,
      Multiple_Inotropes = multiple_inotropes == 1,
      Severe_RV_Dysfunction = severe_rv_dysfunction == 1,
      INTERMACS_1_2 = intermacs_1_2 == 1,
      Hemoglobin_le_10 = hemoglobin_le_10 == 1
    )
  ))
}

euroscore_ii <- function(age, gender, renal_impairment, extracardiac_arteriopathy, poor_mobility,
                         previous_cardiac_surgery, chronic_lung_disease, active_endocarditis,
                         critical_preoperative_state, diabetes_on_insulin, nyha_class,
                         ccs_class_4_angina, lv_function, recent_mi, pulmonary_hypertension,
                         urgency, weight_of_intervention, surgery_on_thoracic_aorta) {

  age <- as.numeric(age)

  gender <- tolower(as.character(gender))
  renal_impairment <- tolower(as.character(renal_impairment))
  extracardiac_arteriopathy <- as.numeric(extracardiac_arteriopathy)
  poor_mobility <- as.numeric(poor_mobility)
  previous_cardiac_surgery <- as.numeric(previous_cardiac_surgery)
  chronic_lung_disease <- as.numeric(chronic_lung_disease)
  active_endocarditis <- as.numeric(active_endocarditis)
  critical_preoperative_state <- as.numeric(critical_preoperative_state)
  diabetes_on_insulin <- as.numeric(diabetes_on_insulin)
  nyha_class <- as.numeric(nyha_class)
  ccs_class_4_angina <- as.numeric(ccs_class_4_angina)
  lv_function <- tolower(as.character(lv_function))
  recent_mi <- as.numeric(recent_mi)
  pulmonary_hypertension <- tolower(as.character(pulmonary_hypertension))
  urgency <- tolower(as.character(urgency))
  weight_of_intervention <- tolower(as.character(weight_of_intervention))
  surgery_on_thoracic_aorta <- as.numeric(surgery_on_thoracic_aorta)

  beta_0 <- -5.324537

  age_factor <- 0
  if (age > 60) {
    age_factor <- (age - 60) * 0.0285181
  }

  gender_factor <- ifelse(gender == "female", 0.2196434, 0)

  renal_factor <- 0
  if (renal_impairment == "creatinine_clearance_50_85") {
    renal_factor <- 0.303553
  } else if (renal_impairment == "creatinine_clearance_lt_50") {
    renal_factor <- 0.8592256
  } else if (renal_impairment == "dialysis") {
    renal_factor <- 0.6421508
  }

  arteriopathy_factor <- ifelse(extracardiac_arteriopathy == 1, 0.5360256, 0)
  mobility_factor <- ifelse(poor_mobility == 1, 0.2407181, 0)
  prev_surgery_factor <- ifelse(previous_cardiac_surgery == 1, 1.118599, 0)
  lung_disease_factor <- ifelse(chronic_lung_disease == 1, 0.1886564, 0)
  endocarditis_factor <- ifelse(active_endocarditis == 1, 0.6194522, 0)
  critical_state_factor <- ifelse(critical_preoperative_state == 1, 1.086517, 0)
  diabetes_factor <- ifelse(diabetes_on_insulin == 1, 0.3542749, 0)

  nyha_factor <- 0
  if (nyha_class == 2) {
    nyha_factor <- 0.1070545
  } else if (nyha_class == 3) {
    nyha_factor <- 0.2958358
  } else if (nyha_class == 4) {
    nyha_factor <- 0.5597929
  }

  ccs4_factor <- ifelse(ccs_class_4_angina == 1, 0.2226147, 0)

  lv_factor <- 0
  if (lv_function == "moderate") {
    lv_factor <- 0.3150652
  } else if (lv_function == "poor") {
    lv_factor <- 0.8084096
  } else if (lv_function == "very_poor") {
    lv_factor <- 0.9346919
  }

  recent_mi_factor <- ifelse(recent_mi == 1, 0.1528943, 0)

  ph_factor <- 0
  if (pulmonary_hypertension == "moderate") {
    ph_factor <- 0.1788899
  } else if (pulmonary_hypertension == "severe") {
    ph_factor <- 0.3491475
  }

  urgency_factor <- 0
  if (urgency == "urgent") {
    urgency_factor <- 0.3174673
  } else if (urgency == "emergency") {
    urgency_factor <- 0.7039121
  } else if (urgency == "salvage") {
    urgency_factor <- 1.362947
  }

  weight_factor <- 0
  if (weight_of_intervention == "single_non_cabg") {
    weight_factor <- 0.0062118
  } else if (weight_of_intervention == "two_procedures") {
    weight_factor <- 0.5521478
  } else if (weight_of_intervention == "three_procedures") {
    weight_factor <- 0.9724533
  }

  aorta_surgery_factor <- ifelse(surgery_on_thoracic_aorta == 1, 0.6527205, 0)

  log_odds <- beta_0 + age_factor + gender_factor + renal_factor + arteriopathy_factor +
    mobility_factor + prev_surgery_factor + lung_disease_factor + endocarditis_factor +
    critical_state_factor + diabetes_factor + nyha_factor + ccs4_factor + lv_factor +
    recent_mi_factor + ph_factor + urgency_factor + weight_factor + aorta_surgery_factor

  predicted_mortality <- exp(log_odds) / (1 + exp(log_odds))

  return(list(
    Predicted_In_Hospital_Mortality = predicted_mortality * 100
  ))
}

eutos_cml_score <- function(basophils_percent, spleen_size_cm) {

  basophils_percent <- as.numeric(basophils_percent)
  spleen_size_cm <- as.numeric(spleen_size_cm)

  if (any(is.na(c(basophils_percent, spleen_size_cm)))) {
    stop("Error: Basophils percentage and Spleen size must be numeric.")
  }

  if (basophils_percent < 0 || basophils_percent > 100) {
    stop("Error: Basophils percentage must be between 0 and 100.")
  }

  if (spleen_size_cm < 0) {
    stop("Error: Spleen size cannot be negative.")
  }

  eutos_score <- (7 * basophils_percent) + (4 * spleen_size_cm)

  risk_category <- ""

  if (eutos_score > 87) {
    risk_category <- "High Risk"
  } else {
    risk_category <- "Low Risk"
  }

  return(list(
    EUTOS_Score = eutos_score,
    Risk_Category = risk_category,
    Inputs = list(
      Basophils_Percent = basophils_percent,
      Spleen_Size_cm = spleen_size_cm
    )
  ))
}

evendo_esophageal_varices_score <- function(platelet_count, inr, urea, urea_units = "mg/dL") {

  platelet_count <- as.numeric(platelet_count)
  inr <- as.numeric(inr)
  urea <- as.numeric(urea)

  if (any(is.na(c(platelet_count, inr, urea)))) {
    stop("Error: All inputs must be numeric.")
  }

  urea_mg_dl <- urea
  if (urea_units == "mmol/L") {
    urea_mg_dl <- urea * 6.006
  } else if (urea_units == "BUN_mg_dL") {
    urea_mg_dl <- urea * 2.14
  }

  risk_factor_count <- 0

  if (platelet_count < 150) {
    risk_factor_count <- risk_factor_count + 1
  }

  if (inr > 1.5) {
    risk_factor_count <- risk_factor_count + 1
  }

  if (urea_mg_dl > 40) {
    risk_factor_count <- risk_factor_count + 1
  }

  interpretation <- ""
  if (risk_factor_count == 0) {
    interpretation <- "Low likelihood of Esophageal Varices needing treatment."
  } else {
    interpretation <- "Increased likelihood of Esophageal Varices. Endoscopy recommended."
  }

  return(list(
    Risk_Factors_Present = risk_factor_count,
    Interpretation = interpretation,
    Inputs = list(
      Platelets_x10_9_L = platelet_count,
      INR = inr,
      Urea_mg_dL = round(urea_mg_dl, 1)
    )
  ))
}

calculate_edss <- function(pyramidal, cerebellar, brainstem, sensory, bowel_bladder, visual, cerebral, ambulation_code) {

  inputs <- c(pyramidal, cerebellar, brainstem, sensory, bowel_bladder, visual, cerebral, ambulation_code)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  fss_scores <- c(pyramidal, cerebellar, brainstem, sensory, bowel_bladder, visual, cerebral)
  if (any(fss_scores < 0) || any(fss_scores > 6)) {
    stop("Error: Functional System scores must be between 0 and 6.")
  }

  if (ambulation_code < 0 || ambulation_code > 11) {
    stop("Error: Ambulation code must be between 0 and 11.")
  }

  sorted_fss <- sort(fss_scores, decreasing = TRUE)
  max_score <- sorted_fss[1]
  count_max <- sum(sorted_fss == max_score)
  second_max <- sorted_fss[2]

  edss_fss <- 0

  if (max_score == 0) {
    edss_fss <- 0
  } else if (max_score == 1) {
    if (count_max == 1) edss_fss <- 1.0
    else edss_fss <- 1.5
  } else if (max_score == 2) {
    if (count_max == 1) {
      edss_fss <- 2.0
    } else {
      edss_fss <- 2.5
    }
  } else if (max_score == 3) {
    if (count_max == 1) {
      if (second_max <= 1) edss_fss <- 3.0
      else if (second_max == 2) {
        edss_fss <- 3.5
      } else {
        edss_fss <- 3.5
      }
    } else {
      edss_fss <- 3.5
    }
  } else if (max_score == 4) {
    edss_fss <- 4.0
    if (count_max > 1) edss_fss <- 4.5
  } else if (max_score >= 5) {
    edss_fss <- 5.0
  }

  edss_ambulation <- 0

  if (ambulation_code == 0) {
    edss_ambulation <- 0
  } else if (ambulation_code == 1) {
    edss_ambulation <- 4.5
  } else if (ambulation_code == 2) {
    edss_ambulation <- 5.0
  } else if (ambulation_code == 3) {
    edss_ambulation <- 5.5
  } else if (ambulation_code == 4) {
    edss_ambulation <- 6.0
  } else if (ambulation_code == 5) {
    edss_ambulation <- 6.5
  } else if (ambulation_code == 6) {
    edss_ambulation <- 7.0
  } else if (ambulation_code == 7) {
    edss_ambulation <- 7.5
  } else if (ambulation_code == 8) {
    edss_ambulation <- 8.0
  } else if (ambulation_code == 9) {
    edss_ambulation <- 8.5
  } else if (ambulation_code == 10) {
    edss_ambulation <- 9.0
  } else if (ambulation_code == 11) {
    edss_ambulation <- 9.5
  }

  final_score <- max(edss_fss, edss_ambulation)

  if (ambulation_code == 0 && final_score > 5.0) {
    final_score <- 5.0
  }

  return(list(
    EDSS_Score = final_score,
    Inputs = list(
      FSS = fss_scores,
      Ambulation_Code = ambulation_code
    )
  ))
}

curb65_pneumonia_severity <- function(confusion, bun, respiratory_rate, systolic_bp, diastolic_bp, age, bun_units = "mg/dL") {

  confusion <- as.numeric(confusion)
  bun <- as.numeric(bun)
  respiratory_rate <- as.numeric(respiratory_rate)
  systolic_bp <- as.numeric(systolic_bp)
  diastolic_bp <- as.numeric(diastolic_bp)
  age <- as.numeric(age)

  if (any(is.na(c(confusion, bun, respiratory_rate, systolic_bp, diastolic_bp, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  bun_val <- bun
  if (bun_units == "mmol/L") {
    bun_val <- bun * 2.8
  }

  score <- 0

  if (confusion == 1) {
    score <- score + 1
  }

  if (bun_val > 19) {
    score <- score + 1
  }

  if (respiratory_rate >= 30) {
    score <- score + 1
  }

  if (systolic_bp < 90 || diastolic_bp <= 60) {
    score <- score + 1
  }

  if (age >= 65) {
    score <- score + 1
  }

  risk_group <- ""
  mortality_30d <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    mortality_30d <- "0.6% - 2.7%"
    recommendation <- "Consider home treatment."
  } else if (score == 2) {
    risk_group <- "Moderate Risk"
    mortality_30d <- "6.8%"
    recommendation <- "Consider hospital admission."
  } else {
    risk_group <- "High Risk"
    mortality_30d <- "14% - 27.8%"
    recommendation <- "Consider ICU admission."
  }

  return(list(
    CURB65_Score = score,
    Risk_Group = risk_group,
    Est_30_Day_Mortality = mortality_30d,
    Recommendation = recommendation,
    Inputs = list(
      Confusion = confusion == 1,
      BUN_gt_19 = bun_val > 19,
      RR_ge_30 = respiratory_rate >= 30,
      Hypotension = (systolic_bp < 90 || diastolic_bp <= 60),
      Age_ge_65 = age >= 65
    )
  ))
}

comm_opioid_misuse_measure <- function(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17) {

  inputs <- c(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17)

  if (any(is.na(inputs))) {
    stop("Error: All 17 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be integers between 0 and 4.")
  }

  total_score <- sum(inputs)

  result <- ""
  interpretation <- ""

  if (total_score >= 9) {
    result <- "Positive Screen"
    interpretation <- "Suggests current aberrant medication-related behavior. Further assessment recommended."
  } else {
    result <- "Negative Screen"
    interpretation <- "Low likelihood of current opioid misuse."
  }

  return(list(
    COMM_Score = total_score,
    Result = result,
    Interpretation = interpretation,
    Inputs = list(
      Q1_Thinking = q1,
      Q2_Slurred = q2,
      Q3_WearOff = q3,
      Q4_Focus = q4,
      Q5_MoreFreq = q5,
      Q6_HigherDose = q6,
      Q7_Borrowing = q7,
      Q8_OtherSx = q8,
      Q9_WorkTrouble = q9,
      Q10_FamilyConcern = q10,
      Q11_Anger = q11,
      Q12_MoodSwings = q12,
      Q13_DocWorry = q13,
      Q14_MultiDoc = q14,
      Q15_Lying = q15,
      Q16_LostStolen = q16,
      Q17_LawTrouble = q17
    )
  ))
}

cytokine_release_syndrome_grading <- function(temperature_c, hypotension_present, hypotension_management, hypoxia_present, hypoxia_management) {

  temperature_c <- as.numeric(temperature_c)
  hypotension_present <- as.numeric(hypotension_present)
  hypotension_management <- tolower(as.character(hypotension_management))
  hypoxia_present <- as.numeric(hypoxia_present)
  hypoxia_management <- tolower(as.character(hypoxia_management))

  if (any(is.na(c(temperature_c, hypotension_present, hypoxia_present)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (temperature_c < 38.0) {
    grade <- 0
  } else {
    grade <- 1

    if (hypotension_present == 1) {
      if (hypotension_management == "none" || hypotension_management == "fluids") {
        grade <- 2
      } else if (hypotension_management == "low_dose_vasopressor") {
        grade <- 3
      } else if (hypotension_management == "high_dose_vasopressor") {
        grade <- 4
      }
    }

    if (hypoxia_present == 1) {
      grade_hypoxia <- 0
      if (hypoxia_management == "low_flow_o2") {
        grade_hypoxia <- 2
      } else if (hypoxia_management == "high_flow_o2") {
        grade_hypoxia <- 3
      } else if (hypoxia_management == "positive_pressure") {
        grade_hypoxia <- 4
      }

      if (grade_hypoxia > grade) {
        grade <- grade_hypoxia
      }
    }
  }

  grade_description <- ""

  if (grade == 0) {
    grade_description <- "No CRS"
  } else if (grade == 1) {
    grade_description <- "Grade 1: Fever >= 38.0 C only. No hypotension or hypoxia."
  } else if (grade == 2) {
    grade_description <- "Grade 2: Fever + Hypotension (responds to fluids) OR Hypoxia (Low-flow O2 <6L/min)."
  } else if (grade == 3) {
    grade_description <- "Grade 3: Fever + Hypotension (needs vasopressor +/- vasopressin) OR Hypoxia (High-flow O2 >=6L/min)."
  } else {
    grade_description <- "Grade 4: Fever + Hypotension (needs multiple pressors) OR Hypoxia (Positive pressure ventilation)."
  }

  return(list(
    CRS_Grade = grade,
    Description = grade_description,
    Inputs = list(
      Temp_C = temperature_c,
      Hypotension = if(hypotension_present==1) hypotension_management else "None",
      Hypoxia = if(hypoxia_present==1) hypoxia_management else "None"
    )
  ))
}

damico_prostate_risk <- function(psa, gleason_score, clinical_t_stage) {

  psa <- as.numeric(psa)
  gleason_score <- as.numeric(gleason_score)
  clinical_t_stage <- tolower(as.character(clinical_t_stage))

  if (any(is.na(c(psa, gleason_score)))) {
    stop("Error: PSA and Gleason Score must be numeric.")
  }

  stage_risk <- 0

  if (clinical_t_stage %in% c("t1", "t1a", "t1b", "t1c", "t2a")) {
    stage_risk <- 1
  } else if (clinical_t_stage == "t2b") {
    stage_risk <- 2
  } else if (clinical_t_stage %in% c("t2c", "t3", "t3a", "t3b", "t4")) {
    stage_risk <- 3
  } else {
    stop("Error: Clinical T Stage must be one of: t1, t1a, t1b, t1c, t2a, t2b, t2c, t3, t3a, t3b, t4.")
  }

  psa_risk <- 0
  if (psa < 10) {
    psa_risk <- 1
  } else if (psa >= 10 && psa <= 20) {
    psa_risk <- 2
  } else {
    psa_risk <- 3
  }

  gleason_risk <- 0
  if (gleason_score <= 6) {
    gleason_risk <- 1
  } else if (gleason_score == 7) {
    gleason_risk <- 2
  } else {
    gleason_risk <- 3
  }

  final_risk_level <- max(stage_risk, psa_risk, gleason_risk)

  risk_group <- ""
  if (final_risk_level == 1) {
    risk_group <- "Low Risk"
  } else if (final_risk_level == 2) {
    risk_group <- "Intermediate Risk"
  } else {
    risk_group <- "High Risk"
  }

  return(list(
    Risk_Classification = risk_group,
    Inputs = list(
      PSA = psa,
      Gleason = gleason_score,
      T_Stage = clinical_t_stage
    )
  ))
}

danger_assessment_tool <- function(increase_severity, gun_in_house, partner_unemployed, weapon_used, threat_to_kill, spy_on_partner, forced_sex, choking, alcohol_abuse, drug_use, violent_jealousy, controlling_behavior, beaten_while_pregnant, stepchild_in_home, threaten_children, partner_suicide_threat, arrest_record, left_or_separated, capable_of_killing) {

  increase_severity <- as.numeric(increase_severity)
  gun_in_house <- as.numeric(gun_in_house)
  partner_unemployed <- as.numeric(partner_unemployed)
  weapon_used <- as.numeric(weapon_used)
  threat_to_kill <- as.numeric(threat_to_kill)
  spy_on_partner <- as.numeric(spy_on_partner)
  forced_sex <- as.numeric(forced_sex)
  choking <- as.numeric(choking)
  alcohol_abuse <- as.numeric(alcohol_abuse)
  drug_use <- as.numeric(drug_use)
  violent_jealousy <- as.numeric(violent_jealousy)
  controlling_behavior <- as.numeric(controlling_behavior)
  beaten_while_pregnant <- as.numeric(beaten_while_pregnant)
  stepchild_in_home <- as.numeric(stepchild_in_home)
  threaten_children <- as.numeric(threaten_children)
  partner_suicide_threat <- as.numeric(partner_suicide_threat)
  arrest_record <- as.numeric(arrest_record)
  left_or_separated <- as.numeric(left_or_separated)
  capable_of_killing <- as.numeric(capable_of_killing)

  if (any(is.na(c(increase_severity, gun_in_house, partner_unemployed, weapon_used, threat_to_kill, spy_on_partner, forced_sex, choking, alcohol_abuse, drug_use, violent_jealousy, controlling_behavior, beaten_while_pregnant, stepchild_in_home, threaten_children, partner_suicide_threat, arrest_record, left_or_separated, capable_of_killing)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (increase_severity == 1) score <- score + 3
  if (gun_in_house == 1) score <- score + 3
  if (partner_unemployed == 1) score <- score + 4
  if (weapon_used == 1) score <- score + 3
  if (threat_to_kill == 1) score <- score + 3
  if (spy_on_partner == 1) score <- score + 2
  if (forced_sex == 1) score <- score + 3
  if (choking == 1) score <- score + 3
  if (alcohol_abuse == 1) score <- score + 1
  if (drug_use == 1) score <- score + 1
  if (violent_jealousy == 1) score <- score + 1
  if (controlling_behavior == 1) score <- score + 1
  if (beaten_while_pregnant == 1) score <- score + 1
  if (stepchild_in_home == 1) score <- score + 2
  if (threaten_children == 1) score <- score + 1
  if (partner_suicide_threat == 1) score <- score + 1
  if (arrest_record == 1) score <- score + 1
  if (left_or_separated == 1) score <- score + 4
  if (capable_of_killing == 1) score <- score + 3

  danger_level <- ""

  if (score < 8) {
    danger_level <- "Variable Danger"
  } else if (score <= 13) {
    danger_level <- "Increased Danger"
  } else if (score <= 17) {
    danger_level <- "Severe Danger"
  } else {
    danger_level <- "Extreme Danger"
  }

  return(list(
    DA_Score = score,
    Risk_Level = danger_level
  ))
}

dash_vte_recurrence_score <- function(d_dimer_abnormal, age, sex, hormone_use_at_onset = 0) {

  d_dimer_abnormal <- as.numeric(d_dimer_abnormal)
  age <- as.numeric(age)
  hormone_use_at_onset <- as.numeric(hormone_use_at_onset)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(d_dimer_abnormal, age)))) {
    stop("Error: Inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (d_dimer_abnormal == 1) {
    score <- score + 2
  }

  if (age <= 50) {
    score <- score + 1
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (sex == "female" && hormone_use_at_onset == 1) {
    score <- score - 2
  }

  risk_category <- ""
  recurrence_rate <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_category <- "Low Risk"
    recurrence_rate <- "3.1% Annual Risk"
    recommendation <- "Consider stopping anticoagulation after initial treatment period (3-6 months)."
  } else {
    risk_category <- "High Risk"
    recurrence_rate <- "9.3% Annual Risk"
    recommendation <- "Consider indefinite anticoagulation."
  }

  return(list(
    DASH_Score = score,
    Risk_Category = risk_category,
    Annual_Recurrence_Risk = recurrence_rate,
    Recommendation = recommendation,
    Inputs = list(
      Abnormal_D_Dimer = d_dimer_abnormal == 1,
      Age_le_50 = age <= 50,
      Male_Sex = sex == "male",
      Hormone_Use = if (sex == "female") (hormone_use_at_onset == 1) else "N/A"
    )
  ))
}

decaf_score_copd <- function(emrcd_5a_5b, eosinophils, consolidation, ph_value, atrial_fibrillation) {

  emrcd_5a_5b <- as.numeric(emrcd_5a_5b)
  eosinophils <- as.numeric(eosinophils)
  consolidation <- as.numeric(consolidation)
  ph_value <- as.numeric(ph_value)
  atrial_fibrillation <- as.numeric(atrial_fibrillation)

  if (any(is.na(c(emrcd_5a_5b, eosinophils, consolidation, ph_value, atrial_fibrillation)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (emrcd_5a_5b == 1) {
    score <- score + 1
  }

  if (eosinophils < 0.05) {
    score <- score + 1
  }

  if (consolidation == 1) {
    score <- score + 1
  }

  if (ph_value < 7.30) {
    score <- score + 1
  }

  if (atrial_fibrillation == 1) {
    score <- score + 1
  }

  risk_group <- ""
  mortality <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    mortality <- "1-1.5%"
    recommendation <- "Consider early supported discharge or home treatment."
  } else if (score == 2) {
    risk_group <- "Intermediate Risk"
    mortality <- "5.4%"
    recommendation <- "Inpatient management usually required."
  } else {
    risk_group <- "High Risk"
    mortality <- "21% (Score 3) to >30% (Score 4-5)"
    recommendation <- "Consider escalation of care / ICU assessment."
  }

  return(list(
    DECAF_Score = score,
    Risk_Group = risk_group,
    In_Hospital_Mortality = mortality,
    Recommendation = recommendation,
    Inputs = list(
      eMRCD_5a_5b = emrcd_5a_5b == 1,
      Eosinopenia = eosinophils < 0.05,
      Consolidation = consolidation == 1,
      Acidemia = ph_value < 7.30,
      AFib = atrial_fibrillation == 1
    )
  ))
}

dhaka_dehydration_score <- function(general_appearance, sunken_eyes, dry_mucous_membranes, tears_absent, skin_turgor, deep_breathing) {

  general_appearance <- as.numeric(general_appearance)
  sunken_eyes <- as.numeric(sunken_eyes)
  dry_mucous_membranes <- as.numeric(dry_mucous_membranes)
  tears_absent <- as.numeric(tears_absent)
  skin_turgor <- as.numeric(skin_turgor)
  deep_breathing <- as.numeric(deep_breathing)

  if (any(is.na(c(general_appearance, sunken_eyes, dry_mucous_membranes, tears_absent, skin_turgor, deep_breathing)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (general_appearance < 0 || general_appearance > 2) stop("Error: General Appearance must be 0, 1, or 2.")
  if (skin_turgor < 0 || skin_turgor > 2) stop("Error: Skin Turgor must be 0, 1, or 2.")

  score <- general_appearance + sunken_eyes + dry_mucous_membranes + tears_absent + skin_turgor + deep_breathing

  dehydration_status <- ""
  management <- ""
  sensitivity <- ""
  specificity <- ""

  if (score < 2) {
    dehydration_status <- "No Dehydration (< 5% deficit)"
    management <- "Oral rehydration solution (ORS) as needed."
    sensitivity <- "N/A"
    specificity <- "N/A"
  } else if (score <= 3) {
    dehydration_status <- "Some Dehydration (5-9% deficit)"
    management <- "Oral rehydration solution (ORS) under supervision."
    sensitivity <- "84% (for detecting Some/Severe)"
    specificity <- "84%"
  } else {
    dehydration_status <- "Severe Dehydration (> 10% deficit)"
    management <- "Intravenous (IV) fluids immediately."
    sensitivity <- "86% (for detecting Severe)"
    specificity <- "81%"
  }

  return(list(
    DHAKA_Score = score,
    Dehydration_Status = dehydration_status,
    Management_Plan = management,
    Accuracy_Stats = list(Sensitivity = sensitivity, Specificity = specificity),
    Inputs = list(
      Appearance = general_appearance,
      Sunken_Eyes = sunken_eyes == 1,
      Dry_Mucosa = dry_mucous_membranes == 1,
      No_Tears = tears_absent == 1,
      Turgor = skin_turgor,
      Deep_Breathing = deep_breathing == 1
    )
  ))
}

denver_hiv_risk_score <- function(age, sex, msm, injection_drug_use, vaginal_anal_sex_no_condom, std_history) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  msm <- as.numeric(msm)
  injection_drug_use <- as.numeric(injection_drug_use)
  vaginal_anal_sex_no_condom <- as.numeric(vaginal_anal_sex_no_condom)
  std_history <- as.numeric(std_history)

  if (any(is.na(c(age, msm, injection_drug_use, vaginal_anal_sex_no_condom, std_history)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (sex == "male") {
    if (age < 30) {
      score <- score + 13
    } else if (age >= 30 && age <= 39) {
      score <- score + 14
    } else if (age >= 40 && age <= 49) {
      score <- score + 17
    } else {
      score <- score + 7
    }

    if (msm == 1) {
      score <- score + 18
    }

    if (injection_drug_use == 1) {
      score <- score + 13
    }

    if (vaginal_anal_sex_no_condom == 1) {
      score <- score + 12
    }

  } else {
    if (age < 30) {
      score <- score + 3
    } else if (age >= 30 && age <= 39) {
      score <- score + 4
    } else if (age >= 40 && age <= 49) {
      score <- score + 11
    } else {
      score <- score + 4
    }

    if (injection_drug_use == 1) {
      score <- score + 21
    }

    if (vaginal_anal_sex_no_condom == 1) {
      score <- score + 8
    }

    if (std_history == 1) {
      score <- score + 10
    }
  }

  risk_category <- ""
  prevalence <- ""
  recommendation <- ""

  if (score < 20) {
    risk_category <- "Low Risk"
    prevalence <- "< 0.1%"
    recommendation <- "Routine screening may not be indicated unless other factors present."
  } else if (score >= 20 && score < 30) {
    risk_category <- "Moderate Risk"
    prevalence <- "0.4% - 0.9%"
    recommendation <- "HIV Testing Recommended."
  } else if (score >= 30 && score < 40) {
    risk_category <- "High Risk"
    prevalence <- "1.7% - 2.8%"
    recommendation <- "HIV Testing Strongly Recommended."
  } else if (score >= 40 && score < 50) {
    risk_category <- "Very High Risk"
    prevalence <- "3.7% - 6.6%"
    recommendation <- "HIV Testing Strongly Recommended."
  } else {
    risk_category <- "Extremely High Risk"
    prevalence <- "> 13%"
    recommendation <- "HIV Testing Strongly Recommended."
  }

  return(list(
    Denver_Score = score,
    Risk_Category = risk_category,
    Estimated_Prevalence = prevalence,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      MSM = if (sex == "male") (msm == 1) else "N/A",
      IVDU = injection_drug_use == 1,
      Unprotected_Sex = vaginal_anal_sex_no_condom == 1,
      STD_History = if (sex == "female") (std_history == 1) else "N/A"
    )
  ))
}

diabetes_distress_scale <- function(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17) {

  inputs <- c(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17)

  if (any(is.na(inputs))) {
    stop("Error: All 17 inputs must be numeric.")
  }

  if (any(inputs < 1) || any(inputs > 6)) {
    stop("Error: All item scores must be integers between 1 and 6.")
  }

  emotional_burden_items <- c(q1, q3, q8, q11, q14)
  physician_distress_items <- c(q2, q4, q9, q15)
  regimen_distress_items <- c(q5, q6, q10, q12, q16)
  interpersonal_distress_items <- c(q7, q13, q17)

  emotional_burden_score <- mean(emotional_burden_items)
  physician_distress_score <- mean(physician_distress_items)
  regimen_distress_score <- mean(regimen_distress_items)
  interpersonal_distress_score <- mean(interpersonal_distress_items)

  total_mean_score <- mean(inputs)

  get_interpretation <- function(score) {
    if (score < 2.0) {
      return("Little or no distress")
    } else if (score < 3.0) {
      return("Moderate distress")
    } else {
      return("High distress")
    }
  }

  return(list(
    Total_Mean_Score = round(total_mean_score, 2),
    Total_Interpretation = get_interpretation(total_mean_score),
    Subscales = list(
      Emotional_Burden = list(
        Score = round(emotional_burden_score, 2),
        Interpretation = get_interpretation(emotional_burden_score)
      ),
      Physician_Related_Distress = list(
        Score = round(physician_distress_score, 2),
        Interpretation = get_interpretation(physician_distress_score)
      ),
      Regimen_Related_Distress = list(
        Score = round(regimen_distress_score, 2),
        Interpretation = get_interpretation(regimen_distress_score)
      ),
      Interpersonal_Distress = list(
        Score = round(interpersonal_distress_score, 2),
        Interpretation = get_interpretation(interpersonal_distress_score)
      )
    ),
    Inputs = inputs
  ))
}

dka_mpm_score <- function(severe_comorbidities, ph_lt_7, insulin_req_gt_50, glucose_gt_300_12hr, depressed_mental_status, fever_24hr) {

  severe_comorbidities <- as.numeric(severe_comorbidities)
  ph_lt_7 <- as.numeric(ph_lt_7)
  insulin_req_gt_50 <- as.numeric(insulin_req_gt_50)
  glucose_gt_300_12hr <- as.numeric(glucose_gt_300_12hr)
  depressed_mental_status <- as.numeric(depressed_mental_status)
  fever_24hr <- as.numeric(fever_24hr)

  if (any(is.na(c(severe_comorbidities, ph_lt_7, insulin_req_gt_50, glucose_gt_300_12hr, depressed_mental_status, fever_24hr)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (severe_comorbidities == 1) {
    score <- score + 6
  }

  if (ph_lt_7 == 1) {
    score <- score + 4
  }

  if (insulin_req_gt_50 == 1) {
    score <- score + 4
  }

  if (glucose_gt_300_12hr == 1) {
    score <- score + 4
  }

  if (depressed_mental_status == 1) {
    score <- score + 4
  }

  if (fever_24hr == 1) {
    score <- score + 3
  }

  mortality_risk <- ""
  risk_group <- ""

  if (score <= 14) {
    risk_group <- "Low Risk"
    mortality_risk <- "0.86%"
  } else {
    risk_group <- "High Risk"
    mortality_risk <- "93.3%"
  }

  return(list(
    DKA_MPM_Score = score,
    Risk_Group = risk_group,
    Estimated_Mortality = mortality_risk,
    Inputs = list(
      Severe_Comorbidities = severe_comorbidities == 1,
      pH_lt_7 = ph_lt_7 == 1,
      Insulin_Req_gt_50U = insulin_req_gt_50 == 1,
      Glucose_gt_300_12h = glucose_gt_300_12hr == 1,
      Depressed_Mental_Status = depressed_mental_status == 1,
      Fever_24h = fever_24hr == 1
    )
  ))
}

digifab_dosing <- function(poisoning_type, weight_kg, digoxin_level_ng_ml = NULL, amount_ingested_mg = NULL) {

  poisoning_type <- tolower(as.character(poisoning_type))
  weight_kg <- as.numeric(weight_kg)

  if (!poisoning_type %in% c("acute", "chronic")) {
    stop("Error: Poisoning type must be 'acute' or 'chronic'.")
  }

  if (is.na(weight_kg) || weight_kg <= 0) {
    stop("Error: Weight must be a positive number.")
  }

  vials <- 0
  method <- ""

  if (!is.null(digoxin_level_ng_ml) && !is.na(as.numeric(digoxin_level_ng_ml))) {
    level <- as.numeric(digoxin_level_ng_ml)
    vials <- (level * weight_kg) / 100
    method <- "Serum Concentration Formula"
  } else if (!is.null(amount_ingested_mg) && !is.na(as.numeric(amount_ingested_mg)) && poisoning_type == "acute") {
    amount <- as.numeric(amount_ingested_mg)
    vials <- amount * 1.6
    method <- "Amount Ingested Formula (0.80 bioavailability)"
  } else {
    if (poisoning_type == "acute") {
      vials <- 10
      method <- "Empiric Dose (Acute)"
    } else {
      vials <- 6
      method <- "Empiric Dose (Chronic)"
    }
  }

  vials_rounded <- ceiling(vials)
  dose_mg <- vials_rounded * 40

  return(list(
    Recommended_Vials = vials_rounded,
    Total_Dose_mg = dose_mg,
    Calculation_Method = method,
    Inputs = list(
      Type = poisoning_type,
      Weight = weight_kg,
      Level = if (!is.null(digoxin_level_ng_ml)) digoxin_level_ng_ml else "N/A",
      Amount = if (!is.null(amount_ingested_mg)) amount_ingested_mg else "N/A"
    )
  ))
}

dipss_myelofibrosis <- function(age, hemoglobin_g_dl, wbc_count_10_9_l, circulating_blasts_percent, constitutional_symptoms) {

  age <- as.numeric(age)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  wbc_count_10_9_l <- as.numeric(wbc_count_10_9_l)
  circulating_blasts_percent <- as.numeric(circulating_blasts_percent)
  constitutional_symptoms <- as.numeric(constitutional_symptoms)

  if (any(is.na(c(age, hemoglobin_g_dl, wbc_count_10_9_l, circulating_blasts_percent, constitutional_symptoms)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age > 65) {
    score <- score + 1
  }

  if (hemoglobin_g_dl < 10) {
    score <- score + 2
  }

  if (wbc_count_10_9_l > 25) {
    score <- score + 1
  }

  if (circulating_blasts_percent >= 1) {
    score <- score + 1
  }

  if (constitutional_symptoms == 1) {
    score <- score + 1
  }

  risk_category <- ""
  median_survival <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    median_survival <- "Not Reached"
  } else if (score >= 1 && score <= 2) {
    risk_category <- "Intermediate-1 Risk"
    median_survival <- "14.2 years"
  } else if (score >= 3 && score <= 4) {
    risk_category <- "Intermediate-2 Risk"
    median_survival <- "4.0 years"
  } else {
    risk_category <- "High Risk"
    median_survival <- "1.5 years"
  }

  return(list(
    DIPSS_Score = score,
    Risk_Category = risk_category,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Age_gt_65 = age > 65,
      Hb_lt_10 = hemoglobin_g_dl < 10,
      WBC_gt_25 = wbc_count_10_9_l > 25,
      Blasts_ge_1 = circulating_blasts_percent >= 1,
      Symptoms = constitutional_symptoms == 1
    )
  ))
}

face_drops_stroke_score <- function(facial_droop, arm_drift, communication_speech, eye_deviation, duration_sudden, recurrence_or_prior, onset_seizure, seizure_history, age) {

  facial_droop <- as.numeric(facial_droop)
  arm_drift <- as.numeric(arm_drift)
  communication_speech <- as.numeric(communication_speech)
  eye_deviation <- as.numeric(eye_deviation)
  duration_sudden <- as.numeric(duration_sudden)
  recurrence_or_prior <- as.numeric(recurrence_or_prior)
  onset_seizure <- as.numeric(onset_seizure)
  seizure_history <- as.numeric(seizure_history)
  age <- as.numeric(age)

  if (any(is.na(c(facial_droop, arm_drift, communication_speech, eye_deviation, duration_sudden, recurrence_or_prior, onset_seizure, seizure_history, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (facial_droop == 1) {
    score <- score + 1
  }

  if (arm_drift == 1) {
    score <- score + 1
  }

  if (communication_speech == 1) {
    score <- score + 1
  }

  if (eye_deviation == 1) {
    score <- score + 1
  }

  if (duration_sudden == 1) {
    score <- score + 1
  }

  if (recurrence_or_prior == 1) {
    score <- score + 1
  }

  if (onset_seizure == 1) {
    score <- score - 1
  }

  if (seizure_history == 1) {
    score <- score - 1
  }

  if (age >= 55) {
    score <- score + 1
  }

  likelihood <- ""
  if (score >= 1) {
    likelihood <- "Stroke Likely"
  } else {
    likelihood <- "Stroke Unlikely (Consider Mimics)"
  }

  return(list(
    FACE_DROPS_Score = score,
    Interpretation = likelihood,
    Inputs = list(
      Face = facial_droop == 1,
      Arm = arm_drift == 1,
      Communication = communication_speech == 1,
      Eyes = eye_deviation == 1,
      Duration = duration_sudden == 1,
      Recurrence = recurrence_or_prior == 1,
      Onset_Seizure = onset_seizure == 1,
      Prior_Seizure = seizure_history == 1,
      Age = age
    )
  ))
}

fat_free_mass <- function(weight_kg, height_cm, sex) {

  weight_kg <- as.numeric(weight_kg)
  height_cm <- as.numeric(height_cm)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(weight_kg, height_cm)))) {
    stop("Error: Weight and Height must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  height_m <- height_cm / 100
  bmi <- weight_kg / (height_m ^ 2)

  ffm <- 0

  if (sex == "male") {
    ffm <- (9270 * weight_kg) / (6680 + (216 * bmi))
  } else {
    ffm <- (9270 * weight_kg) / (8780 + (244 * bmi))
  }

  return(list(
    Fat_Free_Mass_kg = round(ffm, 2),
    BMI = round(bmi, 1),
    Inputs = list(
      Weight_kg = weight_kg,
      Height_cm = height_cm,
      Sex = sex
    )
  ))
}

fetal_biophysical_profile <- function(nst_reactive, amniotic_fluid_volume, fetal_breathing_movements, fetal_tone, fetal_body_movements) {

  nst_reactive <- as.numeric(nst_reactive)
  amniotic_fluid_volume <- as.numeric(amniotic_fluid_volume)
  fetal_breathing_movements <- as.numeric(fetal_breathing_movements)
  fetal_tone <- as.numeric(fetal_tone)
  fetal_body_movements <- as.numeric(fetal_body_movements)

  if (any(is.na(c(nst_reactive, amniotic_fluid_volume, fetal_breathing_movements, fetal_tone, fetal_body_movements)))) {
    stop("Error: All inputs must be numeric (0 or 2).")
  }

  if (any(!c(nst_reactive, amniotic_fluid_volume, fetal_breathing_movements, fetal_tone, fetal_body_movements) %in% c(0, 2))) {
    stop("Error: BPP components must be scored as 0 (Abnormal) or 2 (Normal).")
  }

  total_score <- nst_reactive + amniotic_fluid_volume + fetal_breathing_movements + fetal_tone + fetal_body_movements

  management <- ""
  risk_level <- ""

  if (total_score == 8 || total_score == 10) {
    if (amniotic_fluid_volume == 0) {
      risk_level <- "Low Risk (but Oligohydramnios present)"
      management <- "Chronic asphyxia suspected due to oligohydramnios. Deliver if >36 weeks or if fetal lung maturity is proven. Otherwise, intensive surveillance."
    } else {
      risk_level <- "Low Risk (Normal)"
      management <- "Risk of fetal asphyxia is extremely rare. No intervention required for fetal indications."
    }
  } else if (total_score == 6) {
    if (amniotic_fluid_volume == 0) {
      risk_level <- "Suspected Chronic Asphyxia"
      management <- "Deliver if >36 weeks. If <36 weeks with equivocal test, repeat BPP in 24 hours. If oligohydramnios, induce delivery."
    } else {
      risk_level <- "Equivocal"
      management <- "Repeat test in 24 hours. If score persists at 6 or drops, deliver if term. If preterm, monitor closely."
    }
  } else if (total_score == 4) {
    risk_level <- "Suspected Acute Asphyxia"
    management <- "High probability of fetal distress. Deliver if >32 weeks. If <32 weeks, extended monitoring required."
  } else if (total_score == 2) {
    risk_level <- "High Probability of Acute Asphyxia"
    management <- "Immediate delivery usually indicated regardless of gestational age (extended testing warranted only to confirm)."
  } else {
    risk_level <- "Certain Acute Asphyxia"
    management <- "Immediate delivery indicated."
  }

  return(list(
    BPP_Total_Score = total_score,
    Risk_Assessment = risk_level,
    Management_Guidance = management,
    Inputs = list(
      NST = if(nst_reactive == 2) "Reactive" else "Non-reactive",
      AFV = if(amniotic_fluid_volume == 2) "Normal (>2cm pocket)" else "Abnormal (Oligohydramnios)",
      Breathing = if(fetal_breathing_movements == 2) "Present (>30s)" else "Absent",
      Tone = if(fetal_tone == 2) "Normal (Flexion/Extension)" else "Abnormal (Absent/Slow)",
      Movement = if(fetal_body_movements == 2) "Normal (>3 movements)" else "Abnormal"
    )
  ))
}

feverpain_score <- function(fever_24hr, purulence, attend_rapidly_3days, inflamed_tonsils, no_cough) {

  fever_24hr <- as.numeric(fever_24hr)
  purulence <- as.numeric(purulence)
  attend_rapidly_3days <- as.numeric(attend_rapidly_3days)
  inflamed_tonsils <- as.numeric(inflamed_tonsils)
  no_cough <- as.numeric(no_cough)

  if (any(is.na(c(fever_24hr, purulence, attend_rapidly_3days, inflamed_tonsils, no_cough)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (fever_24hr == 1) {
    score <- score + 1
  }

  if (purulence == 1) {
    score <- score + 1
  }

  if (attend_rapidly_3days == 1) {
    score <- score + 1
  }

  if (inflamed_tonsils == 1) {
    score <- score + 1
  }

  if (no_cough == 1) {
    score <- score + 1
  }

  strep_risk <- ""
  recommendation <- ""

  if (score <= 1) {
    strep_risk <- "13-18%"
    recommendation <- "Antibiotics not recommended."
  } else if (score <= 3) {
    strep_risk <- "30-39%"
    recommendation <- "Delayed antibiotic prescription strategy."
  } else {
    strep_risk <- "62-65%"
    recommendation <- "Immediate or delayed antibiotic prescription."
  }

  return(list(
    FeverPAIN_Score = score,
    Strep_Probability = strep_risk,
    Recommendation = recommendation,
    Inputs = list(
      Fever = fever_24hr == 1,
      Purulence = purulence == 1,
      Rapid_Attendance = attend_rapidly_3days == 1,
      Inflamed_Tonsils = inflamed_tonsils == 1,
      No_Cough = no_cough == 1
    )
  ))
}

fibrotic_nash_index <- function(ast, hdl_cholesterol, hba1c) {

  ast <- as.numeric(ast)
  hdl_cholesterol <- as.numeric(hdl_cholesterol)
  hba1c <- as.numeric(hba1c)

  if (any(is.na(c(ast, hdl_cholesterol, hba1c)))) {
    stop("Error: All inputs must be numeric.")
  }

  fni_score <- exp(-10.33 + (2.37 * log(ast)) - (2.75 * log(hdl_cholesterol)) + (2.61 * log(hba1c))) /
    (1 + exp(-10.33 + (2.37 * log(ast)) - (2.75 * log(hdl_cholesterol)) + (2.61 * log(hba1c))))

  risk_group <- ""
  interpretation <- ""

  if (fni_score < 0.10) {
    risk_group <- "Low Risk"
    interpretation <- "Low probability of advanced fibrosis."
  } else if (fni_score >= 0.10 && fni_score <= 0.33) {
    risk_group <- "Intermediate Risk"
    interpretation <- "Indeterminate probability of advanced fibrosis."
  } else {
    risk_group <- "High Risk"
    interpretation <- "High probability of advanced fibrosis."
  }

  return(list(
    FNI_Score = round(fni_score, 2),
    Risk_Category = risk_group,
    Interpretation = interpretation,
    Inputs = list(
      AST = ast,
      HDL = hdl_cholesterol,
      HbA1c = hba1c
    )
  ))
}

fast_ed_score <- function(facial_palsy, arm_weakness, speech_changes, eye_deviation, denial_neglect) {

  facial_palsy <- as.numeric(facial_palsy)
  arm_weakness <- as.numeric(arm_weakness)
  speech_changes <- as.numeric(speech_changes)
  eye_deviation <- as.numeric(eye_deviation)
  denial_neglect <- as.numeric(denial_neglect)

  if (any(is.na(c(facial_palsy, arm_weakness, speech_changes, eye_deviation, denial_neglect)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (facial_palsy == 1) {
    score <- score + 1
  }

  if (arm_weakness == 1) {
    score <- score + 1
  } else if (arm_weakness == 2) {
    score <- score + 2
  }

  if (speech_changes == 1) {
    score <- score + 1
  } else if (speech_changes == 2) {
    score <- score + 2
  }

  if (eye_deviation == 1) {
    score <- score + 1
  } else if (eye_deviation == 2) {
    score <- score + 2
  }

  if (denial_neglect == 1) {
    score <- score + 1
  } else if (denial_neglect == 2) {
    score <- score + 2
  }

  lvo_likelihood <- ""
  recommendation <- ""

  if (score >= 4) {
    lvo_likelihood <- "High Likelihood of Large Vessel Occlusion (LVO)"
    recommendation <- "Consider direct transport to a Comprehensive Stroke Center (Thrombectomy Capable)."
  } else {
    lvo_likelihood <- "Low Likelihood of LVO"
    recommendation <- "Transport to the nearest Stroke Center (Primary or Comprehensive)."
  }

  return(list(
    FAST_ED_Score = score,
    Likelihood = lvo_likelihood,
    Recommendation = recommendation,
    Inputs = list(
      Face = facial_palsy,
      Arm = arm_weakness,
      Speech = speech_changes,
      Eyes = eye_deviation,
      Neglect = denial_neglect
    )
  ))
}

figo_ovarian_cancer_staging <- function(tumor_location, capsule_rupture, surface_involvement, ascites_cytology, pelvic_extension, abdominal_extension, lymph_node_involvement, distant_metastasis) {

  tumor_location <- tolower(as.character(tumor_location))
  capsule_rupture <- as.numeric(capsule_rupture)
  surface_involvement <- as.numeric(surface_involvement)
  ascites_cytology <- as.numeric(ascites_cytology)
  pelvic_extension <- tolower(as.character(pelvic_extension))
  abdominal_extension <- tolower(as.character(abdominal_extension))
  lymph_node_involvement <- tolower(as.character(lymph_node_involvement))
  distant_metastasis <- tolower(as.character(distant_metastasis))

  if (!tumor_location %in% c("one_ovary", "both_ovaries", "none")) {
    stop("Error: Tumor location must be 'one_ovary', 'both_ovaries', or 'none'.")
  }

  if (any(is.na(c(capsule_rupture, surface_involvement, ascites_cytology)))) {
    stop("Error: Capsule, Surface, and Cytology inputs must be numeric (0 or 1).")
  }

  stage <- "Unknown"
  description <- ""

  if (distant_metastasis != "none") {
    stage <- "Stage IV"
    if (distant_metastasis == "pleural_effusion") {
      stage <- "Stage IVA"
      description <- "Pleural effusion with positive cytology."
    } else if (distant_metastasis == "parenchymal" || distant_metastasis == "extra_abdominal") {
      stage <- "Stage IVB"
      description <- "Hepatic/splenic parenchymal metastasis, or metastasis to extra-abdominal organs (including inguinal lymph nodes/nodes outside abdominal cavity)."
    }
  } else if (abdominal_extension != "none" || lymph_node_involvement != "none") {
    stage <- "Stage III"

    if (lymph_node_involvement != "none" && abdominal_extension == "none") {
      if (lymph_node_involvement == "retroperitoneal_lt_10mm") {
        stage <- "Stage IIIA1(i)"
        description <- "Retroperitoneal lymph nodes only (cytologically or histologically proven), metastasis <= 10 mm."
      } else if (lymph_node_involvement == "retroperitoneal_gt_10mm") {
        stage <- "Stage IIIA1(ii)"
        description <- "Retroperitoneal lymph nodes only, metastasis > 10 mm."
      }
    } else {
      if (abdominal_extension == "microscopic") {
        stage <- "Stage IIIA2"
        description <- "Microscopic extrapelvic (above the brim) peritoneal involvement with or without positive retroperitoneal lymph nodes."
      } else if (abdominal_extension == "macroscopic_lt_2cm") {
        stage <- "Stage IIIB"
        description <- "Macroscopic extrapelvic peritoneal metastasis <= 2 cm, with or without positive retroperitoneal lymph nodes."
      } else if (abdominal_extension == "macroscopic_gt_2cm") {
        stage <- "Stage IIIC"
        description <- "Macroscopic extrapelvic peritoneal metastasis > 2 cm, with or without positive retroperitoneal lymph nodes. Includes extension to liver/spleen capsule."
      }
    }
  } else if (pelvic_extension != "none") {
    stage <- "Stage II"
    if (pelvic_extension == "uterus_tubes") {
      stage <- "Stage IIA"
      description <- "Extension and/or implants on uterus and/or fallopian tubes."
    } else if (pelvic_extension == "other_pelvic_tissues") {
      stage <- "Stage IIB"
      description <- "Extension to other pelvic intraperitoneal tissues."
    }
  } else if (tumor_location != "none") {
    stage <- "Stage I"
    if (tumor_location == "one_ovary") {
      if (capsule_rupture == 0 && surface_involvement == 0 && ascites_cytology == 0) {
        stage <- "Stage IA"
        description <- "Tumor limited to one ovary (capsule intact) or fallopian tube; no tumor on ovarian or fallopian tube surface; no malignant cells in ascites or peritoneal washings."
      } else {
        stage <- "Stage IC"
      }
    } else if (tumor_location == "both_ovaries") {
      if (capsule_rupture == 0 && surface_involvement == 0 && ascites_cytology == 0) {
        stage <- "Stage IB"
        description <- "Tumor limited to both ovaries (capsules intact) or fallopian tubes; no tumor on ovarian or fallopian tube surface; no malignant cells in ascites or peritoneal washings."
      } else {
        stage <- "Stage IC"
      }
    }

    if (stage == "Stage IC") {
      if (capsule_rupture == 1 && surface_involvement == 0 && ascites_cytology == 0) {
        stage <- "Stage IC"
        description <- "Tumor limited to one or both ovaries/fallopian tubes, with any of the following: surgical spill (IC1), capsule ruptured before surgery or tumor on surface (IC2), or malignant cells in ascites/washings (IC3)."
      } else {
        description <- "Tumor limited to one or both ovaries/fallopian tubes, with any of the following: surgical spill (IC1), capsule ruptured before surgery or tumor on surface (IC2), or malignant cells in ascites/washings (IC3)."
      }
    }
  }

  return(list(
    FIGO_Stage = stage,
    Description = description,
    Inputs = list(
      Tumor_Location = tumor_location,
      Capsule_Surface_Cytology = paste("Rupture:", capsule_rupture, "| Surface:", surface_involvement, "| Cytology:", ascites_cytology),
      Pelvic_Extension = pelvic_extension,
      Abdominal_Extension = abdominal_extension,
      Nodes = lymph_node_involvement,
      Metastasis = distant_metastasis
    )
  ))
}

findrisc_score <- function(age, bmi, waist_cm, sex, physical_activity, vegetables_fruit_daily, bp_meds, high_glucose_history, family_history) {

  age <- as.numeric(age)
  bmi <- as.numeric(bmi)
  waist_cm <- as.numeric(waist_cm)
  sex <- tolower(as.character(sex))
  physical_activity <- as.numeric(physical_activity)
  vegetables_fruit_daily <- as.numeric(vegetables_fruit_daily)
  bp_meds <- as.numeric(bp_meds)
  high_glucose_history <- as.numeric(high_glucose_history)
  family_history <- tolower(as.character(family_history))

  if (any(is.na(c(age, bmi, waist_cm, physical_activity, vegetables_fruit_daily, bp_meds, high_glucose_history)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age < 45) {
    score <- score + 0
  } else if (age >= 45 && age <= 54) {
    score <- score + 2
  } else if (age >= 55 && age <= 64) {
    score <- score + 3
  } else {
    score <- score + 4
  }

  if (bmi < 25) {
    score <- score + 0
  } else if (bmi >= 25 && bmi <= 30) {
    score <- score + 1
  } else {
    score <- score + 3
  }

  if (sex == "male") {
    if (waist_cm < 94) {
      score <- score + 0
    } else if (waist_cm >= 94 && waist_cm <= 102) {
      score <- score + 3
    } else {
      score <- score + 4
    }
  } else {
    if (waist_cm < 80) {
      score <- score + 0
    } else if (waist_cm >= 80 && waist_cm <= 88) {
      score <- score + 3
    } else {
      score <- score + 4
    }
  }

  if (physical_activity == 1) {
    score <- score + 0
  } else {
    score <- score + 2
  }

  if (vegetables_fruit_daily == 1) {
    score <- score + 0
  } else {
    score <- score + 1
  }

  if (bp_meds == 1) {
    score <- score + 2
  }

  if (high_glucose_history == 1) {
    score <- score + 5
  }

  if (family_history == "none") {
    score <- score + 0
  } else if (family_history == "2nd_degree") {
    score <- score + 3
  } else if (family_history == "1st_degree") {
    score <- score + 5
  }

  risk_level <- ""
  prob_10yr <- ""

  if (score < 7) {
    risk_level <- "Low Risk"
    prob_10yr <- "1%"
  } else if (score >= 7 && score <= 11) {
    risk_level <- "Slightly Elevated Risk"
    prob_10yr <- "4%"
  } else if (score >= 12 && score <= 14) {
    risk_level <- "Moderate Risk"
    prob_10yr <- "17%"
  } else if (score >= 15 && score <= 20) {
    risk_level <- "High Risk"
    prob_10yr <- "33%"
  } else {
    risk_level <- "Very High Risk"
    prob_10yr <- "50%"
  }

  return(list(
    FINDRISC_Score = score,
    Risk_Level = risk_level,
    Ten_Year_Diabetes_Risk = prob_10yr,
    Inputs = list(
      Age = age,
      BMI = bmi,
      Waist = waist_cm,
      Activity = physical_activity == 1,
      Diet = vegetables_fruit_daily == 1,
      BP_Meds = bp_meds == 1,
      High_Glucose = high_glucose_history == 1,
      Family_Hx = family_history
    )
  ))
}

fisher_sah_grade <- function(sah_present, ich_ivh_present, clot_thickness_ge_1mm) {

  sah_present <- as.numeric(sah_present)
  ich_ivh_present <- as.numeric(ich_ivh_present)
  clot_thickness_ge_1mm <- as.numeric(clot_thickness_ge_1mm)

  if (any(is.na(c(sah_present, ich_ivh_present, clot_thickness_ge_1mm)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  grade <- 0
  description <- ""
  vasospasm_risk <- ""

  if (sah_present == 0) {
    grade <- 1
    description <- "No blood detected."
    vasospasm_risk <- "21% (Low)"
  } else {
    if (ich_ivh_present == 1) {
      grade <- 4
      description <- "Intracerebral or intraventricular clot with diffuse or no subarachnoid hemorrhage."
      vasospasm_risk <- "35% (Moderate)"
    } else if (clot_thickness_ge_1mm == 1) {
      grade <- 3
      description <- "Localized clots and/or vertical layers >= 1mm thick."
      vasospasm_risk <- "96% (High)"
    } else {
      grade <- 2
      description <- "Diffuse deposition or vertical layers < 1mm thick."
      vasospasm_risk <- "25% (Low-Moderate)"
    }
  }

  return(list(
    Fisher_Grade = grade,
    CT_Findings = description,
    Vasospasm_Risk = vasospasm_risk,
    Inputs = list(
      SAH_Present = sah_present == 1,
      ICH_IVH_Present = ich_ivh_present == 1,
      Clot_ge_1mm = clot_thickness_ge_1mm == 1
    )
  ))
}

fleischner_nodule_guidelines <- function(nodule_type, multiplicity, size_mm, risk_level = "low") {

  nodule_type <- tolower(as.character(nodule_type))
  multiplicity <- tolower(as.character(multiplicity))
  size_mm <- as.numeric(size_mm)
  risk_level <- tolower(as.character(risk_level))

  if (is.na(size_mm)) {
    stop("Error: Nodule size must be numeric.")
  }

  if (!nodule_type %in% c("solid", "ground_glass", "part_solid")) {
    stop("Error: Nodule type must be 'solid', 'ground_glass', or 'part_solid'.")
  }

  if (!multiplicity %in% c("single", "multiple")) {
    stop("Error: Multiplicity must be 'single' or 'multiple'.")
  }

  recommendation <- ""

  if (nodule_type == "solid") {
    if (multiplicity == "single") {
      if (size_mm < 6) {
        if (risk_level == "low") {
          recommendation <- "No routine follow-up needed."
        } else {
          recommendation <- "Optional CT at 12 months."
        }
      } else if (size_mm >= 6 && size_mm <= 8) {
        recommendation <- "CT at 6-12 months, then consider CT at 18-24 months."
      } else {
        recommendation <- "Consider CT at 3 months, PET/CT, or Tissue Sampling."
      }
    } else {
      if (size_mm < 6) {
        if (risk_level == "low") {
          recommendation <- "No routine follow-up needed."
        } else {
          recommendation <- "Optional CT at 12 months."
        }
      } else if (size_mm >= 6 && size_mm <= 8) {
        recommendation <- "CT at 3-6 months, then consider CT at 18-24 months."
      } else {
        recommendation <- "CT at 3-6 months, then consider CT at 18-24 months. (Prioritize most suspicious nodule)."
      }
    }
  } else {
    if (size_mm < 6) {
      recommendation <- "No routine follow-up needed."
    } else {
      if (nodule_type == "ground_glass") {
        recommendation <- "CT at 6-12 months to confirm persistence, then CT every 2 years until 5 years."
      } else {
        recommendation <- "CT at 3-6 months to confirm persistence. If unchanged and solid component <6mm, annual CT for 5 years."
      }
    }
  }

  return(list(
    Guideline_Recommendation = recommendation,
    Inputs = list(
      Type = nodule_type,
      Count = multiplicity,
      Size_mm = size_mm,
      Risk_Factors = risk_level
    )
  ))
}

fast_trauma_algorithm <- function(pericardial_fluid, ruq_fluid, luq_fluid, pelvic_fluid, patient_unstable) {

  pericardial_fluid <- as.numeric(pericardial_fluid)
  ruq_fluid <- as.numeric(ruq_fluid)
  luq_fluid <- as.numeric(luq_fluid)
  pelvic_fluid <- as.numeric(pelvic_fluid)
  patient_unstable <- as.numeric(patient_unstable)

  if (any(is.na(c(pericardial_fluid, ruq_fluid, luq_fluid, pelvic_fluid, patient_unstable)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  positive_abdominal <- (ruq_fluid == 1 || luq_fluid == 1 || pelvic_fluid == 1)
  positive_pericardial <- (pericardial_fluid == 1)
  positive_fast <- (positive_abdominal || positive_pericardial)

  result <- ""
  management <- ""

  if (positive_pericardial) {
    result <- "Positive FAST (Pericardial Fluid Present)"
    if (patient_unstable == 1) {
      management <- "Suspect Cardiac Tamponade. Immediate surgical consult. Consider pericardiocentesis or thoracotomy."
    } else {
      management <- "Hemodynamically Stable. Obtain formal Echocardiogram and/or CT. Consult Cardiology/Surgery."
    }
  } else if (positive_abdominal) {
    result <- "Positive FAST (Intraperitoneal Fluid Present)"
    if (patient_unstable == 1) {
      management <- "Hemodynamically Unstable. Immediate Laparotomy indicated."
    } else {
      management <- "Hemodynamically Stable. Obtain CT Abdomen/Pelvis to identify injury source."
    }
  } else {
    result <- "Negative FAST"
    if (patient_unstable == 1) {
      management <- "Hemodynamically Unstable with Negative FAST. Search for extra-abdominal sources (e.g., pelvic fracture, hemothorax). Consider DPL or CT if patient can be stabilized."
    } else {
      management <- "Hemodynamically Stable. Clinical observation and serial FAST exams."
    }
  }

  return(list(
    Result = result,
    Management_Recommendation = management,
    Inputs = list(
      Pericardial_View = pericardial_fluid == 1,
      RUQ_View = ruq_fluid == 1,
      LUQ_View = luq_fluid == 1,
      Pelvic_View = pelvic_fluid == 1,
      Unstable = patient_unstable == 1
    )
  ))
}

flipi_score <- function(age, ann_arbor_stage, hemoglobin_g_dl, ldh_elevated, num_nodal_sites) {

  age <- as.numeric(age)
  ann_arbor_stage <- as.numeric(ann_arbor_stage)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  ldh_elevated <- as.numeric(ldh_elevated)
  num_nodal_sites <- as.numeric(num_nodal_sites)

  if (any(is.na(c(age, ann_arbor_stage, hemoglobin_g_dl, ldh_elevated, num_nodal_sites)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 60) {
    score <- score + 1
  }

  if (ann_arbor_stage == 3 || ann_arbor_stage == 4) {
    score <- score + 1
  }

  if (hemoglobin_g_dl < 12.0) {
    score <- score + 1
  }

  if (ldh_elevated == 1) {
    score <- score + 1
  }

  if (num_nodal_sites > 4) {
    score <- score + 1
  }

  risk_group <- ""
  survival_10y <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    survival_10y <- "71%"
  } else if (score == 2) {
    risk_group <- "Intermediate Risk"
    survival_10y <- "51%"
  } else {
    risk_group <- "High Risk"
    survival_10y <- "36%"
  }

  return(list(
    FLIPI_Score = score,
    Risk_Group = risk_group,
    Est_10_Year_Survival = survival_10y,
    Inputs = list(
      Age_ge_60 = age >= 60,
      Stage_III_IV = ann_arbor_stage >= 3,
      Hb_lt_12 = hemoglobin_g_dl < 12,
      LDH_High = ldh_elevated == 1,
      Nodal_Sites_gt_4 = num_nodal_sites > 4
    )
  ))
}

fomepizole_dosing <- function(weight_kg, on_hemodialysis = 0) {

  weight_kg <- as.numeric(weight_kg)
  on_hemodialysis <- as.numeric(on_hemodialysis)

  if (is.na(weight_kg) || weight_kg <= 0) {
    stop("Error: Weight must be a positive number.")
  }

  loading_dose <- 15 * weight_kg

  maintenance_dose_initial <- 10 * weight_kg

  maintenance_dose_prolonged <- 15 * weight_kg

  regimen <- list()

  regimen$Loading_Dose = paste0(round(loading_dose, 0), " mg (15 mg/kg) IV once.")

  if (on_hemodialysis == 0) {
    regimen$Maintenance_Doses_1_to_4 = paste0(round(maintenance_dose_initial, 0), " mg (10 mg/kg) IV every 12 hours (Doses 2-5).")
    regimen$Maintenance_Doses_5_Plus = paste0(round(maintenance_dose_prolonged, 0), " mg (15 mg/kg) IV every 12 hours (Dose 6 onwards).")
    regimen$Dialysis_Note = "If dialysis becomes necessary: Change frequency to every 4 hours during dialysis."
  } else {
    regimen$Dialysis_Schedule = "Dosing frequency increases during hemodialysis."
    regimen$During_Dialysis = paste0(round(maintenance_dose_initial, 0), " mg (10 mg/kg) IV every 4 hours.")
    regimen$Post_Dialysis = "Resume standard q12h schedule upon completion of dialysis (if levels remain toxic)."
    regimen$Note = "Administer dose at the beginning of dialysis if >6 hours since last dose. Administer dose at the end of dialysis if <6 hours since last dose (or follow specific replacement protocols)."
  }

  return(list(
    Weight_Used_kg = weight_kg,
    Calculated_Dosing_Regimen = regimen
  ))
}

fong_crs_score <- function(nodal_status_primary, disease_free_interval_lt_12mo, num_tumors_gt_1, preop_cea_gt_200, largest_tumor_gt_5cm) {

  nodal_status_primary <- as.numeric(nodal_status_primary)
  disease_free_interval_lt_12mo <- as.numeric(disease_free_interval_lt_12mo)
  num_tumors_gt_1 <- as.numeric(num_tumors_gt_1)
  preop_cea_gt_200 <- as.numeric(preop_cea_gt_200)
  largest_tumor_gt_5cm <- as.numeric(largest_tumor_gt_5cm)

  if (any(is.na(c(nodal_status_primary, disease_free_interval_lt_12mo, num_tumors_gt_1, preop_cea_gt_200, largest_tumor_gt_5cm)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (nodal_status_primary == 1) {
    score <- score + 1
  }

  if (disease_free_interval_lt_12mo == 1) {
    score <- score + 1
  }

  if (num_tumors_gt_1 == 1) {
    score <- score + 1
  }

  if (preop_cea_gt_200 == 1) {
    score <- score + 1
  }

  if (largest_tumor_gt_5cm == 1) {
    score <- score + 1
  }

  risk_group <- ""
  survival_5yr <- ""
  median_survival <- ""

  if (score == 0) {
    risk_group <- "Low Risk"
    survival_5yr <- "60%"
    median_survival <- "74 months"
  } else if (score == 1) {
    risk_group <- "Intermediate Risk"
    survival_5yr <- "44%"
    median_survival <- "51 months"
  } else if (score == 2) {
    risk_group <- "Intermediate Risk"
    survival_5yr <- "40%"
    median_survival <- "47 months"
  } else if (score == 3) {
    risk_group <- "High Risk"
    survival_5yr <- "20%"
    median_survival <- "33 months"
  } else if (score == 4) {
    risk_group <- "High Risk"
    survival_5yr <- "25%"
    median_survival <- "20 months"
  } else {
    risk_group <- "High Risk"
    survival_5yr <- "14%"
    median_survival <- "22 months"
  }

  return(list(
    Fong_Score = score,
    Risk_Group = risk_group,
    Est_5_Year_Survival = survival_5yr,
    Median_Survival = median_survival,
    Inputs = list(
      Node_Positive_Primary = nodal_status_primary == 1,
      DFI_lt_12mo = disease_free_interval_lt_12mo == 1,
      Multiple_Tumors = num_tumors_gt_1 == 1,
      CEA_gt_200 = preop_cea_gt_200 == 1,
      Tumor_Size_gt_5cm = largest_tumor_gt_5cm == 1
    )
  ))
}

covid_risk_score <- function(age_years, sex, chronic_lung_disease, kidney_disease, chf, cad, hypertension, diabetes, cancer, stroke, nursing_home_resident) {

  age_years <- as.numeric(age_years)
  sex <- tolower(as.character(sex))
  chronic_lung_disease <- as.numeric(chronic_lung_disease)
  kidney_disease <- as.numeric(kidney_disease)
  chf <- as.numeric(chf)
  cad <- as.numeric(cad)
  hypertension <- as.numeric(hypertension)
  diabetes <- as.numeric(diabetes)
  cancer <- as.numeric(cancer)
  stroke <- as.numeric(stroke)
  nursing_home_resident <- as.numeric(nursing_home_resident)

  if (any(is.na(c(age_years, chronic_lung_disease, kidney_disease, chf, cad, hypertension, diabetes, cancer, stroke, nursing_home_resident)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (age_years >= 18 && age_years <= 49) {
    score <- score + 0
  } else if (age_years >= 50 && age_years <= 64) {
    score <- score + 2
  } else if (age_years >= 65 && age_years <= 79) {
    score <- score + 3
  } else if (age_years >= 80) {
    score <- score + 5
  }

  if (sex == "male") {
    score <- score + 2
  }

  if (chronic_lung_disease == 1) score <- score + 1
  if (kidney_disease == 1) score <- score + 1
  if (chf == 1) score <- score + 1
  if (cad == 1) score <- score + 1
  if (hypertension == 1) score <- score + 1
  if (diabetes == 1) score <- score + 1
  if (cancer == 1) score <- score + 1
  if (stroke == 1) score <- score + 1
  if (nursing_home_resident == 1) score <- score + 2

  risk_category <- ""
  if (score <= 5) {
    risk_category <- "Low Risk"
  } else if (score <= 9) {
    risk_category <- "Intermediate Risk"
  } else {
    risk_category <- "High Risk"
  }

  return(list(
    COVID_Risk_Score = score,
    Risk_Category = risk_category,
    Inputs = list(
      Age = age_years,
      Sex = sex,
      Comorbidities_Sum = chronic_lung_disease + kidney_disease + chf + cad + hypertension + diabetes + cancer + stroke + (nursing_home_resident * 2)
    )
  ))
}

forrest_classification <- function(endoscopic_finding) {

  endoscopic_finding <- tolower(as.character(endoscopic_finding))

  valid_findings <- c("spurting_hemorrhage", "oozing_hemorrhage", "visible_vessel", "adherent_clot", "flat_pigmented_spot", "clean_base")

  if (!endoscopic_finding %in% valid_findings) {
    stop("Error: Finding must be 'spurting_hemorrhage', 'oozing_hemorrhage', 'visible_vessel', 'adherent_clot', 'flat_pigmented_spot', or 'clean_base'.")
  }

  class_name <- ""
  rebleed_risk <- ""
  mortality_risk <- ""
  recommendation <- ""

  if (endoscopic_finding == "spurting_hemorrhage") {
    class_name <- "Class Ia"
    rebleed_risk <- "90% (High)"
    mortality_risk <- "11%"
    recommendation <- "Endoscopic hemostasis required."
  } else if (endoscopic_finding == "oozing_hemorrhage") {
    class_name <- "Class Ib"
    rebleed_risk <- "50% (High)"
    mortality_risk <- "11%"
    recommendation <- "Endoscopic hemostasis required."
  } else if (endoscopic_finding == "visible_vessel") {
    class_name <- "Class IIa"
    rebleed_risk <- "25-30% (High)"
    mortality_risk <- "11%"
    recommendation <- "Endoscopic hemostasis required."
  } else if (endoscopic_finding == "adherent_clot") {
    class_name <- "Class IIb"
    rebleed_risk <- "10-20% (Intermediate)"
    mortality_risk <- "7%"
    recommendation <- "Consider clot removal and treatment of underlying lesion."
  } else if (endoscopic_finding == "flat_pigmented_spot") {
    class_name <- "Class IIc"
    rebleed_risk <- "7-10% (Low)"
    mortality_risk <- "3%"
    recommendation <- "Standard medical therapy."
  } else {
    class_name <- "Class III"
    rebleed_risk <- "< 5% (Low)"
    mortality_risk <- "2%"
    recommendation <- "Standard medical therapy; consider early discharge."
  }

  return(list(
    Forrest_Class = class_name,
    Risk_of_Rebleeding = rebleed_risk,
    Mortality_Risk = mortality_risk,
    Management_Recommendation = recommendation,
    Input_Finding = endoscopic_finding
  ))
}

four_score <- function(eye_response, motor_response, brainstem_reflexes, respiration_pattern) {

  eye_response <- as.numeric(eye_response)
  motor_response <- as.numeric(motor_response)
  brainstem_reflexes <- as.numeric(brainstem_reflexes)
  respiration_pattern <- as.numeric(respiration_pattern)

  if (any(is.na(c(eye_response, motor_response, brainstem_reflexes, respiration_pattern)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(eye_response, motor_response, brainstem_reflexes, respiration_pattern) < 0) ||
      any(c(eye_response, motor_response, brainstem_reflexes, respiration_pattern) > 4)) {
    stop("Error: All item scores must be integers between 0 and 4.")
  }

  total_score <- eye_response + motor_response + brainstem_reflexes + respiration_pattern

  prognosis <- ""
  if (total_score == 0) {
    prognosis <- "High mortality probability (89%). Likely brain death or severe injury."
  } else if (total_score <= 4) {
    prognosis <- "Very Poor Prognosis. High mortality."
  } else if (total_score <= 8) {
    prognosis <- "Poor Prognosis. Significant neurological impairment likely."
  } else if (total_score <= 12) {
    prognosis <- "Moderate Impairment."
  } else {
    prognosis <- "Good Prognosis (approaching normal consciousness)."
  }

  return(list(
    FOUR_Score = total_score,
    Interpretation = prognosis,
    Inputs = list(
      Eye = eye_response,
      Motor = motor_response,
      Brainstem = brainstem_reflexes,
      Respiration = respiration_pattern
    )
  ))
}

calculate_fena <- function(serum_sodium, serum_creatinine, urine_sodium, urine_creatinine) {

  serum_sodium <- as.numeric(serum_sodium)
  serum_creatinine <- as.numeric(serum_creatinine)
  urine_sodium <- as.numeric(urine_sodium)
  urine_creatinine <- as.numeric(urine_creatinine)

  if (any(is.na(c(serum_sodium, serum_creatinine, urine_sodium, urine_creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (serum_sodium == 0 || urine_creatinine == 0) {
    stop("Error: Serum Sodium and Urine Creatinine cannot be zero (division by zero).")
  }

  fena <- (urine_sodium * serum_creatinine) / (serum_sodium * urine_creatinine) * 100

  interpretation <- ""
  if (fena < 1) {
    interpretation <- "Prerenal Etiology (Hypoperfusion)"
  } else if (fena > 2) {
    interpretation <- "Intrinsic Renal Etiology (e.g., Acute Tubular Necrosis)"
  } else {
    interpretation <- "Indeterminate (1-2%). Could be prerenal, ATN, or other causes."
  }

  return(list(
    FENa_Percent = round(fena, 2),
    Etiology = interpretation,
    Inputs = list(
      Serum_Na = serum_sodium,
      Serum_Cr = serum_creatinine,
      Urine_Na = urine_sodium,
      Urine_Cr = urine_creatinine
    )
  ))
}

calculate_feurea <- function(serum_urea, serum_creatinine, urine_urea, urine_creatinine, urea_units = "mg/dL", creatinine_units = "mg/dL") {

  serum_urea <- as.numeric(serum_urea)
  serum_creatinine <- as.numeric(serum_creatinine)
  urine_urea <- as.numeric(urine_urea)
  urine_creatinine <- as.numeric(urine_creatinine)

  if (any(is.na(c(serum_urea, serum_creatinine, urine_urea, urine_creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (serum_urea == 0 || urine_creatinine == 0) {
    stop("Error: Serum Urea and Urine Creatinine cannot be zero (division by zero).")
  }

  s_urea_mg_dl <- serum_urea
  if (urea_units == "mmol/L") {
    s_urea_mg_dl <- serum_urea * 6.006
  } else if (urea_units == "BUN_mg_dL") {
    s_urea_mg_dl <- serum_urea * 2.14
  }

  u_urea_mg_dl <- urine_urea
  if (urea_units == "mmol/L") {
    u_urea_mg_dl <- urine_urea * 6.006
  } else if (urea_units == "BUN_mg_dL") {
    u_urea_mg_dl <- urine_urea * 2.14
  }

  s_creat_mg_dl <- serum_creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    s_creat_mg_dl <- serum_creatinine / 88.4
  }

  u_creat_mg_dl <- urine_creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    u_creat_mg_dl <- urine_creatinine / 88.4
  }

  fe_urea <- (u_urea_mg_dl * s_creat_mg_dl) / (s_urea_mg_dl * u_creat_mg_dl) * 100

  interpretation <- ""
  if (fe_urea < 35) {
    interpretation <- "Prerenal Etiology (FEUrea < 35%)"
  } else if (fe_urea > 50) {
    interpretation <- "Intrinsic Renal Etiology (FEUrea > 50%) e.g., Acute Tubular Necrosis"
  } else {
    interpretation <- "Indeterminate (35-50%)"
  }

  return(list(
    FEUrea_Percent = round(fe_urea, 1),
    Etiology = interpretation,
    Inputs = list(
      Serum_Urea = s_urea_mg_dl,
      Serum_Cr = s_creat_mg_dl,
      Urine_Urea = u_urea_mg_dl,
      Urine_Cr = u_creat_mg_dl
    )
  ))
}

fracture_index_score <- function(age, history_fracture_after_50, maternal_hip_fracture, weight_lt_57kg, smoker, uses_arms_to_stand) {

  age <- as.numeric(age)
  history_fracture_after_50 <- as.numeric(history_fracture_after_50)
  maternal_hip_fracture <- as.numeric(maternal_hip_fracture)
  weight_lt_57kg <- as.numeric(weight_lt_57kg)
  smoker <- as.numeric(smoker)
  uses_arms_to_stand <- as.numeric(uses_arms_to_stand)

  if (any(is.na(c(age, history_fracture_after_50, maternal_hip_fracture, weight_lt_57kg, smoker, uses_arms_to_stand)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 65 && age <= 69) {
    score <- score + 1
  } else if (age >= 70 && age <= 74) {
    score <- score + 2
  } else if (age >= 75 && age <= 79) {
    score <- score + 3
  } else if (age >= 80) {
    score <- score + 4
  }

  if (history_fracture_after_50 == 1) {
    score <- score + 1
  }

  if (maternal_hip_fracture == 1) {
    score <- score + 1
  }

  if (weight_lt_57kg == 1) {
    score <- score + 1
  }

  if (smoker == 1) {
    score <- score + 1
  }

  if (uses_arms_to_stand == 1) {
    score <- score + 1
  }

  hip_fracture_risk <- ""
  any_fracture_risk <- ""
  bmd_t_score_est <- ""

  if (score <= 1) {
    hip_fracture_risk <- "0.8%"
    any_fracture_risk <- "5.6%"
    bmd_t_score_est <- "Normal / Osteopenia"
  } else if (score >= 2 && score <= 3) {
    hip_fracture_risk <- "2.5% (Score 2) - 5.4% (Score 3)"
    any_fracture_risk <- "9.6% (Score 2) - 13% (Score 3)"
    bmd_t_score_est <- "Low Bone Mass"
  } else {
    hip_fracture_risk <- "9.4% (Score 4) - >19% (Score >=5)"
    any_fracture_risk <- "17% (Score 4) - >25% (Score >=5)"
    bmd_t_score_est <- "Osteoporosis Likely"
  }

  return(list(
    Fracture_Index_Score = score,
    Est_5Yr_Hip_Fracture_Risk = hip_fracture_risk,
    Est_5Yr_Any_Fracture_Risk = any_fracture_risk,
    Bone_Density_Correlation = bmd_t_score_est,
    Inputs = list(
      Age = age,
      Prior_Fx = history_fracture_after_50 == 1,
      Maternal_Hip_Fx = maternal_hip_fracture == 1,
      Low_Weight = weight_lt_57kg == 1,
      Smoker = smoker == 1,
      Arm_Assist_Stand = uses_arms_to_stand == 1
    )
  ))
}

framingham_hf_criteria <- function(pnd, orthopnea, elevated_jvp, rales, cardiomegaly, pulmonary_edema, s3_gallop, hepatojugular_reflux, weight_loss_response, ankle_edema, night_cough, dyspnea_on_exertion, hepatomegaly, pleural_effusion, tachycardia_gt_120, vital_capacity_decrease) {

  inputs <- c(pnd, orthopnea, elevated_jvp, rales, cardiomegaly, pulmonary_edema, s3_gallop, hepatojugular_reflux, weight_loss_response, ankle_edema, night_cough, dyspnea_on_exertion, hepatomegaly, pleural_effusion, tachycardia_gt_120, vital_capacity_decrease)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  major_count <- sum(pnd, orthopnea, elevated_jvp, rales, cardiomegaly, pulmonary_edema, s3_gallop, hepatojugular_reflux, weight_loss_response)

  minor_count <- sum(ankle_edema, night_cough, dyspnea_on_exertion, hepatomegaly, pleural_effusion, tachycardia_gt_120, vital_capacity_decrease)

  diagnosis_met <- FALSE

  if (major_count >= 2) {
    diagnosis_met <- TRUE
  } else if (major_count == 1 && minor_count >= 2) {
    diagnosis_met <- TRUE
  }

  result <- ""
  if (diagnosis_met) {
    result <- "Diagnostic criteria for Heart Failure met."
  } else {
    result <- "Diagnostic criteria for Heart Failure NOT met."
  }

  return(list(
    Result = result,
    Criteria_Counts = list(
      Major_Criteria = major_count,
      Minor_Criteria = minor_count
    ),
    Diagnosis_Met = diagnosis_met
  ))
}

framingham_hard_chd <- function(age, sex, smoker, total_cholesterol, hdl_cholesterol, systolic_bp, bp_medication) {

  age <- as.numeric(age)
  total_cholesterol <- as.numeric(total_cholesterol)
  hdl_cholesterol <- as.numeric(hdl_cholesterol)
  systolic_bp <- as.numeric(systolic_bp)
  smoker <- as.numeric(smoker)
  bp_medication <- as.numeric(bp_medication)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, total_cholesterol, hdl_cholesterol, systolic_bp, smoker, bp_medication)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (age < 20 || age > 79) {
    return(list(Error = "Framingham Hard CHD Score is validated for ages 20-79."))
  }

  points <- 0

  if (sex == "male") {
    if (age >= 20 && age <= 34) points <- -9
    else if (age >= 35 && age <= 39) points <- -4
    else if (age >= 40 && age <= 44) points <- 0
    else if (age >= 45 && age <= 49) points <- 3
    else if (age >= 50 && age <= 54) points <- 6
    else if (age >= 55 && age <= 59) points <- 8
    else if (age >= 60 && age <= 64) points <- 10
    else if (age >= 65 && age <= 69) points <- 11
    else if (age >= 70 && age <= 74) points <- 12
    else points <- 13

    if (total_cholesterol < 160) {
      points <- points + 0
    } else if (total_cholesterol >= 160 && total_cholesterol <= 199) {
      if (age <= 39) points <- points + 4
      else if (age <= 49) points <- points + 3
      else if (age <= 59) points <- points + 2
      else if (age <= 69) points <- points + 1
      else points <- points + 0
    } else if (total_cholesterol >= 200 && total_cholesterol <= 239) {
      if (age <= 39) points <- points + 7
      else if (age <= 49) points <- points + 5
      else if (age <= 59) points <- points + 3
      else if (age <= 69) points <- points + 1
      else points <- points + 0
    } else if (total_cholesterol >= 240 && total_cholesterol <= 279) {
      if (age <= 39) points <- points + 9
      else if (age <= 49) points <- points + 6
      else if (age <= 59) points <- points + 4
      else if (age <= 69) points <- points + 2
      else points <- points + 1
    } else {
      if (age <= 39) points <- points + 11
      else if (age <= 49) points <- points + 8
      else if (age <= 59) points <- points + 5
      else if (age <= 69) points <- points + 3
      else points <- points + 1
    }

    if (smoker == 1) {
      if (age <= 39) points <- points + 8
      else if (age <= 49) points <- points + 5
      else if (age <= 59) points <- points + 3
      else points <- points + 1
    }

    if (hdl_cholesterol >= 60) points <- points - 1
    else if (hdl_cholesterol >= 50) points <- points + 0
    else if (hdl_cholesterol >= 40) points <- points + 1
    else points <- points + 2

    if (bp_medication == 1) {
      if (systolic_bp < 120) points <- points + 0
      else if (systolic_bp <= 129) points <- points + 1
      else if (systolic_bp <= 139) points <- points + 2
      else if (systolic_bp <= 159) points <- points + 2
      else points <- points + 3
    } else {
      if (systolic_bp < 120) points <- points + 0
      else if (systolic_bp <= 129) points <- points + 0
      else if (systolic_bp <= 139) points <- points + 1
      else if (systolic_bp <= 159) points <- points + 1
      else points <- points + 2
    }

    risk_percent <- "< 1%"
    if (points <= 4) risk_percent <- "< 1%"
    else if (points == 5) risk_percent <- "2%"
    else if (points == 6) risk_percent <- "2%"
    else if (points == 7) risk_percent <- "3%"
    else if (points == 8) risk_percent <- "4%"
    else if (points == 9) risk_percent <- "5%"
    else if (points == 10) risk_percent <- "6%"
    else if (points == 11) risk_percent <- "8%"
    else if (points == 12) risk_percent <- "10%"
    else if (points == 13) risk_percent <- "12%"
    else if (points == 14) risk_percent <- "16%"
    else if (points == 15) risk_percent <- "20%"
    else if (points == 16) risk_percent <- "25%"
    else risk_percent <- ">= 30%"

  } else {
    if (age >= 20 && age <= 34) points <- -7
    else if (age >= 35 && age <= 39) points <- -3
    else if (age >= 40 && age <= 44) points <- 0
    else if (age >= 45 && age <= 49) points <- 3
    else if (age >= 50 && age <= 54) points <- 6
    else if (age >= 55 && age <= 59) points <- 8
    else if (age >= 60 && age <= 64) points <- 10
    else if (age >= 65 && age <= 69) points <- 12
    else if (age >= 70 && age <= 74) points <- 14
    else points <- 16

    if (total_cholesterol < 160) {
      points <- points + 0
    } else if (total_cholesterol >= 160 && total_cholesterol <= 199) {
      if (age <= 39) points <- points + 4
      else if (age <= 49) points <- points + 3
      else if (age <= 59) points <- points + 2
      else if (age <= 69) points <- points + 1
      else points <- points + 1
    } else if (total_cholesterol >= 200 && total_cholesterol <= 239) {
      if (age <= 39) points <- points + 8
      else if (age <= 49) points <- points + 6
      else if (age <= 59) points <- points + 4
      else if (age <= 69) points <- points + 2
      else points <- points + 1
    } else if (total_cholesterol >= 240 && total_cholesterol <= 279) {
      if (age <= 39) points <- points + 11
      else if (age <= 49) points <- points + 8
      else if (age <= 59) points <- points + 5
      else if (age <= 69) points <- points + 3
      else points <- points + 2
    } else {
      if (age <= 39) points <- points + 13
      else if (age <= 49) points <- points + 10
      else if (age <= 59) points <- points + 7
      else if (age <= 69) points <- points + 4
      else points <- points + 2
    }

    if (smoker == 1) {
      if (age <= 39) points <- points + 9
      else if (age <= 49) points <- points + 7
      else if (age <= 59) points <- points + 4
      else if (age <= 69) points <- points + 2
      else points <- points + 1
    }

    if (hdl_cholesterol >= 60) points <- points - 1
    else if (hdl_cholesterol >= 50) points <- points + 0
    else if (hdl_cholesterol >= 40) points <- points + 1
    else points <- points + 2

    if (bp_medication == 1) {
      if (systolic_bp < 120) points <- points + 0
      else if (systolic_bp <= 129) points <- points + 3
      else if (systolic_bp <= 139) points <- points + 4
      else if (systolic_bp <= 159) points <- points + 5
      else points <- points + 6
    } else {
      if (systolic_bp < 120) points <- points + 0
      else if (systolic_bp <= 129) points <- points + 1
      else if (systolic_bp <= 139) points <- points + 2
      else if (systolic_bp <= 159) points <- points + 3
      else points <- points + 4
    }

    if (points <= 8) risk_percent <- "< 1%"
    else if (points == 9) risk_percent <- "1%"
    else if (points == 10) risk_percent <- "1%"
    else if (points == 11) risk_percent <- "1%"
    else if (points == 12) risk_percent <- "1%"
    else if (points == 13) risk_percent <- "2%"
    else if (points == 14) risk_percent <- "2%"
    else if (points == 15) risk_percent <- "3%"
    else if (points == 16) risk_percent <- "4%"
    else if (points == 17) risk_percent <- "5%"
    else if (points == 18) risk_percent <- "6%"
    else if (points == 19) risk_percent <- "8%"
    else if (points == 20) risk_percent <- "11%"
    else if (points == 21) risk_percent <- "14%"
    else if (points == 22) risk_percent <- "17%"
    else if (points == 23) risk_percent <- "22%"
    else if (points == 24) risk_percent <- "27%"
    else risk_percent <- ">= 30%"
  }

  return(list(
    Total_Points = points,
    Ten_Year_Risk_Hard_CHD = risk_percent,
    Inputs = list(
      Age = age,
      Sex = sex,
      Smoker = smoker == 1,
      Tot_Chol = total_cholesterol,
      HDL = hdl_cholesterol,
      SBP = systolic_bp,
      Treated_BP = bp_medication == 1
    )
  ))
}

free_water_deficit <- function(current_na, weight_kg, age, sex, target_na = 140) {

  current_na <- as.numeric(current_na)
  weight_kg <- as.numeric(weight_kg)
  age <- as.numeric(age)
  target_na <- as.numeric(target_na)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(current_na, weight_kg, age, target_na)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (current_na <= 140) {
    return(list(
      Deficit_L = 0,
      Message = "Current Sodium is not hypernatremic relative to the standard target (140). No free water deficit."
    ))
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  tbw_fraction <- 0

  if (age < 18) {
    tbw_fraction <- 0.6
  } else if (age < 65) {
    if (sex == "male") {
      tbw_fraction <- 0.6
    } else {
      tbw_fraction <- 0.5
    }
  } else {
    if (sex == "male") {
      tbw_fraction <- 0.5
    } else {
      tbw_fraction <- 0.45
    }
  }

  total_body_water <- weight_kg * tbw_fraction

  deficit <- total_body_water * ((current_na / target_na) - 1)

  return(list(
    Free_Water_Deficit_L = round(deficit, 1),
    Total_Body_Water_L = round(total_body_water, 1),
    TBW_Fraction_Used = tbw_fraction,
    Inputs = list(
      Current_Na = current_na,
      Target_Na = target_na,
      Weight = weight_kg,
      Age = age,
      Sex = sex
    )
  ))
}

fuhrman_nuclear_grade <- function(nuclear_diameter_um, nuclear_shape_irregular, nucleoli_visible, magnification_needed) {

  nuclear_diameter_um <- as.numeric(nuclear_diameter_um)
  nuclear_shape_irregular <- as.numeric(nuclear_shape_irregular)
  nucleoli_visible <- as.numeric(nucleoli_visible)
  magnification_needed <- tolower(as.character(magnification_needed))

  if (any(is.na(c(nuclear_diameter_um, nuclear_shape_irregular, nucleoli_visible)))) {
    stop("Error: All inputs must be numeric/valid.")
  }

  if (!magnification_needed %in% c("400x", "100x", "10x")) {
    stop("Error: Magnification must be '400x', '100x', or '10x'.")
  }

  grade <- 0
  description <- ""
  prognosis <- ""
  survival_5yr <- ""

  if (nuclear_diameter_um > 20 && nuclear_shape_irregular == 1 && nucleoli_visible == 1) {
    grade <- 4
    description <- "Nuclei > 20 microm, bizarre and multilobated; heavy chromatin clumps."
    prognosis <- "Poor prognosis. Often associated with sarcomatoid differentiation."
    survival_5yr <- "Approximately 30-40%"
  } else if (nuclear_diameter_um >= 15 && nucleoli_visible == 1 && magnification_needed == "10x") {
    grade <- 3
    description <- "Nuclei ~15 microm, irregular contour; nucleoli visible at low power (10x)."
    prognosis <- "Intermediate prognosis."
    survival_5yr <- "Approximately 50-60%"
  } else if (nuclear_diameter_um >= 15 && nucleoli_visible == 1 && magnification_needed == "400x") {
    grade <- 2
    description <- "Nuclei ~15 microm, irregular contour; nucleoli visible at high power (400x) but not low power."
    prognosis <- "Good prognosis."
    survival_5yr <- "Approximately 70-80%"
  } else if (nuclear_diameter_um <= 10 && nuclear_shape_irregular == 0 && nucleoli_visible == 0) {
    grade <- 1
    description <- "Nuclei ~10 microm (similar to lymphocytes), round/uniform; nucleoli absent or inconspicuous."
    prognosis <- "Excellent prognosis."
    survival_5yr <- "Greater than 90%"
  } else {
    if (nuclear_diameter_um > 20 || (nucleoli_visible == 1 && magnification_needed == "10x" && nuclear_shape_irregular == 1)) {
      grade <- 3
      if (nuclear_diameter_um > 20) grade <- 4
    } else if (nucleoli_visible == 1) {
      grade <- 2
    } else {
      grade <- 1
    }
    description <- "Mixed features. Grade assigned based on worst feature (nucleolar prominence/size)."
  }

  return(list(
    Fuhrman_Grade = grade,
    Histologic_Description = description,
    Prognosis = prognosis,
    Est_5_Year_Survival = survival_5yr,
    Inputs = list(
      Nuclei_Size_um = nuclear_diameter_um,
      Irregular_Shape = nuclear_shape_irregular == 1,
      Nucleoli_Visible = nucleoli_visible == 1,
      Magnification = magnification_needed
    )
  ))
}

func_ich_score <- function(ich_volume_cm3, age, location, gcs, cognitive_impairment_pre_ich) {

  ich_volume_cm3 <- as.numeric(ich_volume_cm3)
  age <- as.numeric(age)
  location <- tolower(as.character(location))
  gcs <- as.numeric(gcs)
  cognitive_impairment_pre_ich <- as.numeric(cognitive_impairment_pre_ich)

  if (any(is.na(c(ich_volume_cm3, age, gcs, cognitive_impairment_pre_ich)))) {
    stop("Error: All inputs must be numeric (0 or 1 for binary).")
  }

  if (!location %in% c("lobar", "deep", "infratentorial")) {
    stop("Error: Location must be 'lobar', 'deep', or 'infratentorial'.")
  }

  score <- 0

  if (ich_volume_cm3 < 30) {
    score <- score + 4
  } else if (ich_volume_cm3 >= 30 && ich_volume_cm3 <= 60) {
    score <- score + 2
  } else {
    score <- score + 0
  }

  if (age < 70) {
    score <- score + 2
  } else if (age >= 70 && age <= 79) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (location == "lobar") {
    score <- score + 2
  } else if (location == "deep") {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (gcs >= 9) {
    score <- score + 2
  } else {
    score <- score + 0
  }

  if (cognitive_impairment_pre_ich == 0) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  functional_independence_prob <- ""
  if (score <= 4) {
    functional_independence_prob <- "0%"
  } else if (score >= 5 && score <= 7) {
    functional_independence_prob <- "1-20%"
  } else if (score == 8) {
    functional_independence_prob <- "21-60%"
  } else if (score == 9) {
    functional_independence_prob <- "61-80%"
  } else if (score == 10) {
    functional_independence_prob <- "81-100%"
  } else {
    functional_independence_prob <- "81-100%"
  }

  return(list(
    FUNC_Score = score,
    Prob_Functional_Independence_90_Day = functional_independence_prob,
    Inputs = list(
      Volume_cm3 = ich_volume_cm3,
      Age = age,
      Location = location,
      GCS = gcs,
      Pre_ICH_Cognitive_Impairment = cognitive_impairment_pre_ich == 1
    )
  ))
}

g8_geriatric_screen <- function(food_intake, weight_loss, mobility, neuropsychological, bmi_score, more_than_3_meds, self_rated_health, age) {

  food_intake <- as.numeric(food_intake)
  weight_loss <- as.numeric(weight_loss)
  mobility <- as.numeric(mobility)
  neuropsychological <- as.numeric(neuropsychological)
  bmi_score <- as.numeric(bmi_score)
  more_than_3_meds <- as.numeric(more_than_3_meds)
  self_rated_health <- as.numeric(self_rated_health)
  age <- as.numeric(age)

  if (any(is.na(c(food_intake, weight_loss, mobility, neuropsychological, bmi_score, more_than_3_meds, self_rated_health, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  score <- score + food_intake
  score <- score + weight_loss
  score <- score + mobility
  score <- score + neuropsychological
  score <- score + bmi_score

  if (more_than_3_meds == 0) {
    score <- score + 1
  }

  score <- score + self_rated_health

  if (age < 80) {
    score <- score + 2
  } else if (age <= 85) {
    score <- score + 1
  }

  interpretation <- ""
  if (score <= 14) {
    interpretation <- "Vulnerable / Frail (Score <= 14). Comprehensive Geriatric Assessment (CGA) recommended."
  } else {
    interpretation <- "Fit / Normal (Score > 14). Standard oncologic treatment generally appropriate."
  }

  return(list(
    G8_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Food_Intake_Score = food_intake,
      Weight_Loss_Score = weight_loss,
      Mobility_Score = mobility,
      Neuro_Score = neuropsychological,
      BMI_Score = bmi_score,
      Polypharmacy = more_than_3_meds == 1,
      Self_Rated_Health_Score = self_rated_health,
      Age = age
    )
  ))
}

gad7_anxiety_screen <- function(q1, q2, q3, q4, q5, q6, q7) {

  inputs <- c(q1, q2, q3, q4, q5, q6, q7)

  if (any(is.na(inputs))) {
    stop("Error: All 7 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 3)) {
    stop("Error: All item scores must be integers between 0 and 3.")
  }

  total_score <- sum(inputs)

  severity <- ""
  if (total_score <= 4) {
    severity <- "Minimal anxiety"
  } else if (total_score <= 9) {
    severity <- "Mild anxiety"
  } else if (total_score <= 14) {
    severity <- "Moderate anxiety"
  } else {
    severity <- "Severe anxiety"
  }

  recommendation <- ""
  if (total_score >= 10) {
    recommendation <- "Positive Screen. Further evaluation is recommended."
  } else {
    recommendation <- "Negative Screen. Monitor; repeat if symptoms persist or worsen."
  }

  return(list(
    GAD7_Score = total_score,
    Severity_Category = severity,
    Screening_Result = recommendation,
    Inputs = list(
      Q1_Nervous = q1,
      Q2_Worry_Control = q2,
      Q3_Worry_Too_Much = q3,
      Q4_Trouble_Relaxing = q4,
      Q5_Restless = q5,
      Q6_Irritable = q6,
      Q7_Afraid = q7
    )
  ))
}

gail_model_risk <- function(age, age_menarche, age_first_live_birth, num_first_degree_relatives, num_biopsies, atypical_hyperplasia, race = "white") {

  age <- as.numeric(age)
  age_menarche <- as.numeric(age_menarche)
  age_first_live_birth <- as.numeric(age_first_live_birth)
  num_first_degree_relatives <- as.numeric(num_first_degree_relatives)
  num_biopsies <- as.numeric(num_biopsies)
  atypical_hyperplasia <- as.numeric(atypical_hyperplasia)
  race <- tolower(as.character(race))

  if (any(is.na(c(age, age_menarche, num_first_degree_relatives, num_biopsies)))) {
    stop("Error: All inputs must be numeric.")
  }

  rr_menarche <- 1.0
  if (age_menarche >= 14) {
    rr_menarche <- 1.0
  } else if (age_menarche >= 12) {
    rr_menarche <- 1.10
  } else {
    rr_menarche <- 1.21
  }

  rr_biopsy <- 1.0
  if (num_biopsies == 0) {
    rr_biopsy <- 1.0
  } else if (num_biopsies == 1) {
    if (age < 50) {
      rr_biopsy <- 1.70
    } else {
      rr_biopsy <- 1.27
    }
  } else {
    if (age < 50) {
      rr_biopsy <- 2.88
    } else {
      rr_biopsy <- 1.62
    }
  }

  if (num_biopsies > 0) {
    if (atypical_hyperplasia == 1) {
      rr_biopsy <- rr_biopsy * 1.82
    } else if (atypical_hyperplasia == 0) {
      rr_biopsy <- rr_biopsy * 0.93
    }
  }

  flb_category <- 0
  if (age_first_live_birth == 0) {
    flb_category <- 5
  } else if (age_first_live_birth < 20) {
    flb_category <- 1
  } else if (age_first_live_birth < 25) {
    flb_category <- 2
  } else if (age_first_live_birth < 30) {
    flb_category <- 3
  } else {
    flb_category <- 4
  }

  relatives_category <- 0
  if (num_first_degree_relatives == 0) {
    relatives_category <- 1
  } else if (num_first_degree_relatives == 1) {
    relatives_category <- 2
  } else {
    relatives_category <- 3
  }

  rr_flb_relatives <- 1.0

  if (age < 50) {
    if (relatives_category == 1) {
      if (flb_category == 1) rr_flb_relatives <- 1.00
      else if (flb_category == 2) rr_flb_relatives <- 1.24
      else if (flb_category == 3) rr_flb_relatives <- 1.55
      else rr_flb_relatives <- 1.93
    } else if (relatives_category == 2) {
      if (flb_category == 1) rr_flb_relatives <- 2.61
      else if (flb_category == 2) rr_flb_relatives <- 2.68
      else if (flb_category == 3) rr_flb_relatives <- 2.76
      else rr_flb_relatives <- 2.83
    } else {
      if (flb_category == 1) rr_flb_relatives <- 6.80
      else if (flb_category == 2) rr_flb_relatives <- 5.78
      else if (flb_category == 3) rr_flb_relatives <- 4.91
      else rr_flb_relatives <- 4.17
    }
  } else {
    if (relatives_category == 1) {
      if (flb_category == 1) rr_flb_relatives <- 1.00
      else if (flb_category == 2) rr_flb_relatives <- 1.24
      else if (flb_category == 3) rr_flb_relatives <- 1.55
      else rr_flb_relatives <- 1.93
    } else if (relatives_category == 2) {
      if (flb_category == 1) rr_flb_relatives <- 1.61
      else if (flb_category == 2) rr_flb_relatives <- 1.97
      else if (flb_category == 3) rr_flb_relatives <- 2.42
      else rr_flb_relatives <- 2.94
    } else {
      if (flb_category == 1) rr_flb_relatives <- 2.59
      else if (flb_category == 2) rr_flb_relatives <- 3.13
      else if (flb_category == 3) rr_flb_relatives <- 3.77
      else rr_flb_relatives <- 4.49
    }
  }

  total_relative_risk <- rr_menarche * rr_biopsy * rr_flb_relatives

  return(list(
    Relative_Risk = round(total_relative_risk, 2),
    Interpretation = "This value represents the relative risk compared to a woman of the same age with no risk factors. To calculate absolute 5-year and lifetime risk, this RR must be applied to age-specific baseline hazard rates (NCI Actuarial Data).",
    Inputs = list(
      Age = age,
      Age_Menarche = age_menarche,
      Age_FLB = age_first_live_birth,
      Relatives = num_first_degree_relatives,
      Biopsies = num_biopsies,
      Hyperplasia = atypical_hyperplasia == 1,
      Race = race
    )
  ))
}

galad_score_hcc <- function(sex, age, afp, afp_l3, dcp) {

  age <- as.numeric(age)
  afp <- as.numeric(afp)
  afp_l3 <- as.numeric(afp_l3)
  dcp <- as.numeric(dcp)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(age, afp, afp_l3, dcp)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (afp <= 0 || dcp <= 0) {
    stop("Error: AFP and DCP must be greater than 0 (logarithmic calculation).")
  }

  sex_factor <- if (sex == "male") 1 else 0

  z_score <- -10.08 + (0.09 * age) + (1.67 * sex_factor) + (2.34 * log10(afp)) + (0.04 * afp_l3) + (1.33 * log10(dcp))

  probability <- exp(z_score) / (1 + exp(z_score))
  probability_percent <- probability * 100

  risk_level <- ""
  if (z_score < -0.63) {
    risk_level <- "Low Risk"
  } else {
    risk_level <- "High Risk"
  }

  return(list(
    GALAD_Score_Z = round(z_score, 2),
    Probability_HCC = paste0(round(probability_percent, 1), "%"),
    Risk_Category = risk_level,
    Inputs = list(
      Age = age,
      Sex = sex,
      AFP = afp,
      AFP_L3 = afp_l3,
      DCP = dcp
    )
  ))
}

ganzoni_iron_deficit <- function(weight_kg, actual_hb, target_hb = 13, iron_stores_mg = 500) {

  weight_kg <- as.numeric(weight_kg)
  actual_hb <- as.numeric(actual_hb)
  target_hb <- as.numeric(target_hb)
  iron_stores_mg <- as.numeric(iron_stores_mg)

  if (any(is.na(c(weight_kg, actual_hb, target_hb, iron_stores_mg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  hb_diff <- target_hb - actual_hb

  if (hb_diff < 0) {
    hb_diff <- 0
  }

  total_iron_deficit <- (weight_kg * hb_diff * 2.4) + iron_stores_mg

  return(list(
    Total_Iron_Deficit_mg = round(total_iron_deficit, 0),
    Formula = paste0("(", weight_kg, " x (", target_hb, " - ", actual_hb, ") x 2.4) + ", iron_stores_mg),
    Inputs = list(
      Weight_kg = weight_kg,
      Actual_Hb = actual_hb,
      Target_Hb = target_hb,
      Iron_Stores = iron_stores_mg
    )
  ))
}

gap_index_ipf <- function(sex, age, fvc_percent, dlco_percent, dlco_unable_to_perform = 0) {

  sex <- tolower(as.character(sex))
  age <- as.numeric(age)
  fvc_percent <- as.numeric(fvc_percent)
  dlco_unable_to_perform <- as.numeric(dlco_unable_to_perform)

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (any(is.na(c(age, fvc_percent)))) {
    stop("Error: Age and FVC must be numeric.")
  }

  score <- 0

  if (sex == "male") {
    score <- score + 1
  }

  if (age <= 60) {
    score <- score + 0
  } else if (age >= 61 && age <= 65) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (fvc_percent > 75) {
    score <- score + 0
  } else if (fvc_percent >= 50) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (dlco_unable_to_perform == 1) {
    score <- score + 3
  } else {
    if (is.na(as.numeric(dlco_percent))) {
      stop("Error: DLCO percent must be numeric if performance is possible.")
    }
    dlco <- as.numeric(dlco_percent)

    if (dlco > 55) {
      score <- score + 0
    } else if (dlco >= 36) {
      score <- score + 1
    } else {
      score <- score + 2
    }
  }

  stage <- ""
  mortality_1yr <- ""
  mortality_2yr <- ""
  mortality_3yr <- ""

  if (score <= 3) {
    stage <- "Stage I"
    mortality_1yr <- "5.6%"
    mortality_2yr <- "10.9%"
    mortality_3yr <- "16.3%"
  } else if (score <= 5) {
    stage <- "Stage II"
    mortality_1yr <- "16.2%"
    mortality_2yr <- "29.9%"
    mortality_3yr <- "42.1%"
  } else {
    stage <- "Stage III"
    mortality_1yr <- "39.2%"
    mortality_2yr <- "62.1%"
    mortality_3yr <- "76.8%"
  }

  return(list(
    GAP_Score = score,
    GAP_Stage = stage,
    Mortality_Risk = list(
      Year_1 = mortality_1yr,
      Year_2 = mortality_2yr,
      Year_3 = mortality_3yr
    ),
    Inputs = list(
      Sex = sex,
      Age = age,
      FVC_Percent = fvc_percent,
      DLCO_Status = if (dlco_unable_to_perform == 1) "Unable to Perform" else paste0(dlco_percent, "%")
    )
  ))
}

garfield_af_risk_score <- function(sex, age, weight_kg, heart_rate, systolic_bp, chf_history, acs_history, history_stroke_tia, history_bleeding, ckd_stage_3_to_5, diabetes, treatment_type) {

  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)
  heart_rate <- as.numeric(heart_rate)
  systolic_bp <- as.numeric(systolic_bp)
  chf_history <- as.numeric(chf_history)
  acs_history <- as.numeric(acs_history)
  history_stroke_tia <- as.numeric(history_stroke_tia)
  history_bleeding <- as.numeric(history_bleeding)
  ckd_stage_3_to_5 <- as.numeric(ckd_stage_3_to_5)
  diabetes <- as.numeric(diabetes)
  sex <- tolower(as.character(sex))
  treatment_type <- tolower(as.character(treatment_type))

  if (any(is.na(c(age, weight_kg, heart_rate, systolic_bp, chf_history, acs_history, history_stroke_tia, history_bleeding, ckd_stage_3_to_5, diabetes)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  valid_treatments <- c("none", "vka", "doac", "antiplatelet", "combination")
  if (!treatment_type %in% valid_treatments) {
    stop("Error: Treatment type must be 'none', 'vka', 'doac', 'antiplatelet', or 'combination'.")
  }

  risk_factors <- list()

  if (chf_history == 1) risk_factors$CHF <- "Present"
  if (acs_history == 1) risk_factors$ACS <- "Present"
  if (history_stroke_tia == 1) risk_factors$Stroke_TIA <- "Present"
  if (history_bleeding == 1) risk_factors$Bleeding_Hx <- "Present"
  if (ckd_stage_3_to_5 == 1) risk_factors$CKD <- "Moderate-Severe"
  if (diabetes == 1) risk_factors$Diabetes <- "Present"

  if (length(risk_factors) == 0) risk_factors$None <- "No major comorbidities reported"

  return(list(
    Patient_Profile = list(
      Demographics = paste(age, "yo", sex),
      Vitals = paste("HR:", heart_rate, "SBP:", systolic_bp, "Wt:", weight_kg),
      Treatment = treatment_type
    ),
    Risk_Factors_Identified = risk_factors,
    Note = "The GARFIELD-AF tool uses a complex regression model to estimate 1-year and 2-year mortality, stroke, and bleeding risks. This function structures the inputs for that calculation."
  ))
}

gcs_pupils_score <- function(gcs_total, pupils_unreactive_count) {

  gcs_total <- as.numeric(gcs_total)
  pupils_unreactive_count <- as.numeric(pupils_unreactive_count)

  if (any(is.na(c(gcs_total, pupils_unreactive_count)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (gcs_total < 3 || gcs_total > 15) {
    stop("Error: GCS Total must be between 3 and 15.")
  }

  if (!pupils_unreactive_count %in% c(0, 1, 2)) {
    stop("Error: Count of unreactive pupils must be 0, 1, or 2.")
  }

  gcs_p_score <- gcs_total - pupils_unreactive_count

  mortality_trend <- ""
  if (gcs_p_score <= 3) {
    mortality_trend <- "Very High Mortality Risk (>50%)"
  } else if (gcs_p_score <= 8) {
    mortality_trend <- "High Mortality Risk"
  } else {
    mortality_trend <- "Lower Mortality Risk (Decreases linearly with higher scores)"
  }

  return(list(
    GCS_P_Score = gcs_p_score,
    Interpretation = mortality_trend,
    Inputs = list(
      GCS_Total = gcs_total,
      Unreactive_Pupils = pupils_unreactive_count
    )
  ))
}

gipss_pmf_score <- function(karyotype_category, platelet_count, asxl1_mutation, srsf2_mutation, u2af1_q157_mutation) {

  karyotype_category <- tolower(as.character(karyotype_category))
  platelet_count <- as.numeric(platelet_count)
  asxl1_mutation <- as.numeric(asxl1_mutation)
  srsf2_mutation <- as.numeric(srsf2_mutation)
  u2af1_q157_mutation <- as.numeric(u2af1_q157_mutation)

  if (any(is.na(c(platelet_count, asxl1_mutation, srsf2_mutation, u2af1_q157_mutation)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!karyotype_category %in% c("favorable", "unfavorable", "very_high_risk")) {
    stop("Error: Karyotype category must be 'favorable', 'unfavorable', or 'very_high_risk'.")
  }

  score <- 0

  if (karyotype_category == "very_high_risk") {
    score <- score + 2
  } else if (karyotype_category == "unfavorable") {
    score <- score + 1
  }

  if (platelet_count < 100) {
    score <- score + 1
  }

  if (asxl1_mutation == 1) {
    score <- score + 1
  }

  if (srsf2_mutation == 1) {
    score <- score + 1
  }

  if (u2af1_q157_mutation == 1) {
    score <- score + 1
  }

  risk_group <- ""
  median_survival <- ""

  if (score == 0) {
    risk_group <- "Low Risk"
    median_survival <- "26.4 years"
  } else if (score == 1) {
    risk_group <- "Intermediate-1 Risk"
    median_survival <- "10.3 years"
  } else if (score == 2) {
    risk_group <- "Intermediate-2 Risk"
    median_survival <- "4.6 years"
  } else {
    risk_group <- "High Risk"
    median_survival <- "2.6 years"
  }

  return(list(
    GIPSS_Score = score,
    Risk_Category = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Karyotype = karyotype_category,
      Platelets_lt_100 = platelet_count < 100,
      ASXL1 = asxl1_mutation == 1,
      SRSF2 = srsf2_mutation == 1,
      U2AF1_Q157 = u2af1_q157_mutation == 1
    )
  ))
}

geneva_vte_prophylaxis_score <- function(heart_failure, respiratory_failure, recent_stroke, recent_mi, acute_infection, rheumatic_disease, active_malignancy, myeloproliferative_syndrome, nephrotic_syndrome, prior_vte, hypercoagulable_state, immobilization, recent_travel, age_gt_60, obesity_bmi_gt_30, chronic_venous_insufficiency, pregnancy, hormonal_therapy, dehydration) {

  inputs <- c(heart_failure, respiratory_failure, recent_stroke, recent_mi, acute_infection, rheumatic_disease, active_malignancy, myeloproliferative_syndrome, nephrotic_syndrome, prior_vte, hypercoagulable_state, immobilization, recent_travel, age_gt_60, obesity_bmi_gt_30, chronic_venous_insufficiency, pregnancy, hormonal_therapy, dehydration)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (heart_failure == 1) score <- score + 2
  if (respiratory_failure == 1) score <- score + 2
  if (recent_stroke == 1) score <- score + 2
  if (recent_mi == 1) score <- score + 2
  if (acute_infection == 1) score <- score + 2
  if (rheumatic_disease == 1) score <- score + 2
  if (active_malignancy == 1) score <- score + 2
  if (myeloproliferative_syndrome == 1) score <- score + 2
  if (nephrotic_syndrome == 1) score <- score + 2
  if (prior_vte == 1) score <- score + 2
  if (hypercoagulable_state == 1) score <- score + 2

  if (immobilization == 1) score <- score + 1
  if (recent_travel == 1) score <- score + 1
  if (age_gt_60 == 1) score <- score + 1
  if (obesity_bmi_gt_30 == 1) score <- score + 1
  if (chronic_venous_insufficiency == 1) score <- score + 1
  if (pregnancy == 1) score <- score + 1
  if (hormonal_therapy == 1) score <- score + 1
  if (dehydration == 1) score <- score + 1

  risk_category <- ""
  recommendation <- ""

  if (score < 3) {
    risk_category <- "Low Risk"
    recommendation <- "VTE prophylaxis generally NOT indicated. Prioritize early mobilization."
  } else {
    risk_category <- "High Risk"
    recommendation <- "VTE prophylaxis indicated (e.g., LMWH/UFH) after bleeding risk assessment."
  }

  return(list(
    Geneva_Prophylaxis_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Major_Factors_2pts = list(
        Heart_Failure = heart_failure == 1,
        Resp_Failure = respiratory_failure == 1,
        Stroke = recent_stroke == 1,
        MI = recent_mi == 1,
        Infection = acute_infection == 1,
        Rheumatic = rheumatic_disease == 1,
        Malignancy = active_malignancy == 1,
        Myeloproliferative = myeloproliferative_syndrome == 1,
        Nephrotic = nephrotic_syndrome == 1,
        Prior_VTE = prior_vte == 1,
        Hypercoagulable = hypercoagulable_state == 1
      ),
      Minor_Factors_1pt = list(
        Immobilization = immobilization == 1,
        Travel = recent_travel == 1,
        Age_gt_60 = age_gt_60 == 1,
        Obesity = obesity_bmi_gt_30 == 1,
        Venous_Insufficiency = chronic_venous_insufficiency == 1,
        Pregnancy = pregnancy == 1,
        Hormonal_Tx = hormonal_therapy == 1,
        Dehydration = dehydration == 1
      )
    )
  ))
}

geneva_score_revised <- function(age_gt_65, previous_dvt_pe, surgery_or_fracture_1mo, active_malignancy, unilateral_lower_limb_pain, hemoptysis, heart_rate, lower_limb_deep_vein_palpation_edema) {

  age_gt_65 <- as.numeric(age_gt_65)
  previous_dvt_pe <- as.numeric(previous_dvt_pe)
  surgery_or_fracture_1mo <- as.numeric(surgery_or_fracture_1mo)
  active_malignancy <- as.numeric(active_malignancy)
  unilateral_lower_limb_pain <- as.numeric(unilateral_lower_limb_pain)
  hemoptysis <- as.numeric(hemoptysis)
  heart_rate <- as.numeric(heart_rate)
  lower_limb_deep_vein_palpation_edema <- as.numeric(lower_limb_deep_vein_palpation_edema)

  if (any(is.na(c(age_gt_65, previous_dvt_pe, surgery_or_fracture_1mo, active_malignancy, unilateral_lower_limb_pain, hemoptysis, heart_rate, lower_limb_deep_vein_palpation_edema)))) {
    stop("Error: All inputs must be numeric (0 or 1 for binary variables, and numeric for heart rate).")
  }

  score <- 0

  if (age_gt_65 == 1) {
    score <- score + 1
  }

  if (previous_dvt_pe == 1) {
    score <- score + 3
  }

  if (surgery_or_fracture_1mo == 1) {
    score <- score + 2
  }

  if (active_malignancy == 1) {
    score <- score + 2
  }

  if (unilateral_lower_limb_pain == 1) {
    score <- score + 3
  }

  if (hemoptysis == 1) {
    score <- score + 2
  }

  if (heart_rate >= 75 && heart_rate <= 94) {
    score <- score + 3
  } else if (heart_rate >= 95) {
    score <- score + 5
  }

  if (lower_limb_deep_vein_palpation_edema == 1) {
    score <- score + 4
  }

  risk_level <- ""
  pe_probability <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_level <- "Low Probability"
    pe_probability <- "8%"
    recommendation <- "D-dimer testing to rule out PE."
  } else if (score >= 4 && score <= 10) {
    risk_level <- "Intermediate Probability"
    pe_probability <- "28%"
    recommendation <- "D-dimer testing or CTPA depending on clinical judgment."
  } else {
    risk_level <- "High Probability"
    pe_probability <- "74%"
    recommendation <- "CTPA or V/Q scan."
  }

  return(list(
    Geneva_Score = score,
    Risk_Probability = risk_level,
    PE_Probability = pe_probability,
    Recommendation = recommendation,
    Inputs = list(
      Age_gt_65 = age_gt_65 == 1,
      Prev_DVT_PE = previous_dvt_pe == 1,
      Surgery_Fracture_1mo = surgery_or_fracture_1mo == 1,
      Malignancy = active_malignancy == 1,
      Unilateral_Pain = unilateral_lower_limb_pain == 1,
      Hemoptysis = hemoptysis == 1,
      Heart_Rate = heart_rate,
      Pain_Edema = lower_limb_deep_vein_palpation_edema == 1
    )
  ))
}

gds15_depression_screen <- function(satisfied_with_life, dropped_activities, life_empty, often_bored, good_spirits, afraid_something_bad, happy_most_time, often_helpless, prefer_stay_home, memory_problems, wonderful_alive, worthless, full_energy, hopeless, others_better_off) {

  inputs <- c(satisfied_with_life, dropped_activities, life_empty, often_bored, good_spirits, afraid_something_bad, happy_most_time, often_helpless, prefer_stay_home, memory_problems, wonderful_alive, worthless, full_energy, hopeless, others_better_off)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric (0 for No, 1 for Yes).")
  }

  score <- 0

  if (satisfied_with_life == 0) score <- score + 1
  if (dropped_activities == 1) score <- score + 1
  if (life_empty == 1) score <- score + 1
  if (often_bored == 1) score <- score + 1
  if (good_spirits == 0) score <- score + 1
  if (afraid_something_bad == 1) score <- score + 1
  if (happy_most_time == 0) score <- score + 1
  if (often_helpless == 1) score <- score + 1
  if (prefer_stay_home == 1) score <- score + 1
  if (memory_problems == 1) score <- score + 1
  if (wonderful_alive == 0) score <- score + 1
  if (worthless == 1) score <- score + 1
  if (full_energy == 0) score <- score + 1
  if (hopeless == 1) score <- score + 1
  if (others_better_off == 1) score <- score + 1

  interpretation <- ""
  if (score <= 4) {
    interpretation <- "Normal"
  } else if (score <= 8) {
    interpretation <- "Mild Depression"
  } else if (score <= 11) {
    interpretation <- "Moderate Depression"
  } else {
    interpretation <- "Severe Depression"
  }

  return(list(
    GDS15_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Q1_Satisfied = satisfied_with_life == 1,
      Q2_Dropped_Activities = dropped_activities == 1,
      Q3_Life_Empty = life_empty == 1,
      Q4_Bored = often_bored == 1,
      Q5_Good_Spirits = good_spirits == 1,
      Q6_Afraid = afraid_something_bad == 1,
      Q7_Happy = happy_most_time == 1,
      Q8_Helpless = often_helpless == 1,
      Q9_Stay_Home = prefer_stay_home == 1,
      Q10_Memory = memory_problems == 1,
      Q11_Wonderful = wonderful_alive == 1,
      Q12_Worthless = worthless == 1,
      Q13_Energy = full_energy == 1,
      Q14_Hopeless = hopeless == 1,
      Q15_Others_Better = others_better_off == 1
    )
  ))
}

gillmore_attr_cm_staging <- function(nt_probnp_pg_ml, egfr_ml_min_1_73m2, genotype = "wild-type") {

  nt_probnp_pg_ml <- as.numeric(nt_probnp_pg_ml)
  egfr_ml_min_1_73m2 <- as.numeric(egfr_ml_min_1_73m2)
  genotype <- tolower(as.character(genotype))

  if (any(is.na(c(nt_probnp_pg_ml, egfr_ml_min_1_73m2)))) {
    stop("Error: NT-proBNP and eGFR must be numeric.")
  }

  if (!genotype %in% c("wild-type", "hereditary", "variant")) {
    stop("Error: Genotype must be 'wild-type' or 'hereditary'/'variant'.")
  }

  threshold <- 3000

  stage <- 0
  median_survival <- ""

  if (nt_probnp_pg_ml <= threshold && egfr_ml_min_1_73m2 >= 45) {
    stage <- 1
    median_survival <- "69 months"
  } else if (nt_probnp_pg_ml > threshold && egfr_ml_min_1_73m2 < 45) {
    stage <- 3
    median_survival <- "24 months"
  } else {
    stage <- 2
    median_survival <- "47 months"
  }

  return(list(
    Gillmore_Stage = stage,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      NT_proBNP = nt_probnp_pg_ml,
      eGFR = egfr_ml_min_1_73m2,
      Genotype = genotype
    )
  ))
}

glasgow_alcoholic_hepatitis_score <- function(age, wbc_count, urea, pt_inr, bilirubin, bilirubin_units = "mg/dL", urea_units = "mg/dL") {

  age <- as.numeric(age)
  wbc_count <- as.numeric(wbc_count)
  urea <- as.numeric(urea)
  pt_inr <- as.numeric(pt_inr)
  bilirubin <- as.numeric(bilirubin)

  if (any(is.na(c(age, wbc_count, urea, pt_inr, bilirubin)))) {
    stop("Error: All inputs must be numeric.")
  }

  urea_mmol_l <- urea
  if (urea_units == "mg/dL") {
    urea_mmol_l <- urea * 0.357
  }

  bili_umol_l <- bilirubin
  if (bilirubin_units == "mg/dL") {
    bili_umol_l <- bilirubin * 17.1
  }

  score <- 0

  if (age < 50) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (wbc_count < 15) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (urea_mmol_l < 5) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (pt_inr < 1.5) {
    score <- score + 1
  } else if (pt_inr < 2.0) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (bili_umol_l < 125) {
    score <- score + 1
  } else if (bili_umol_l < 250) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  survival_28_day <- ""
  survival_84_day <- ""

  if (score < 9) {
    survival_28_day <- "87%"
    survival_84_day <- "79%"
  } else {
    survival_28_day <- "46%"
    survival_84_day <- "40%"
  }

  return(list(
    GAHS_Score = score,
    Survival_28_Day = survival_28_day,
    Survival_84_Day = survival_84_day,
    Inputs = list(
      Age = age,
      WBC = wbc_count,
      Urea_mmol_L = round(urea_mmol_l, 2),
      INR = pt_inr,
      Bilirubin_umol_L = round(bili_umol_l, 2)
    )
  ))
}

glasgow_coma_scale <- function(eye_response, verbal_response, motor_response) {

  eye_response <- as.numeric(eye_response)
  verbal_response <- as.numeric(verbal_response)
  motor_response <- as.numeric(motor_response)

  if (any(is.na(c(eye_response, verbal_response, motor_response)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (eye_response < 1 || eye_response > 4) stop("Error: Eye response must be between 1 and 4.")
  if (verbal_response < 1 || verbal_response > 5) stop("Error: Verbal response must be between 1 and 5.")
  if (motor_response < 1 || motor_response > 6) stop("Error: Motor response must be between 1 and 6.")

  gcs_score <- eye_response + verbal_response + motor_response

  severity <- ""
  intervention <- ""

  if (gcs_score >= 13) {
    severity <- "Mild Brain Injury"
    intervention <- "Observation; CT scan if risk factors present."
  } else if (gcs_score >= 9) {
    severity <- "Moderate Brain Injury"
    intervention <- "CT scan required; Observation; Consider ICU."
  } else {
    severity <- "Severe Brain Injury (Coma)"
    intervention <- "Intubation usually indicated (GCS <= 8); ICU admission; ICP monitoring considerations."
  }

  return(list(
    GCS_Score = gcs_score,
    Injury_Severity = severity,
    Clinical_Action = intervention,
    Inputs = list(
      Eye = eye_response,
      Verbal = verbal_response,
      Motor = motor_response
    )
  ))
}

gmaws_score <- function(tremor, hallucinations, orientation, agitation, sweating) {

  tremor <- as.numeric(tremor)
  hallucinations <- as.numeric(hallucinations)
  orientation <- as.numeric(orientation)
  agitation <- as.numeric(agitation)
  sweating <- as.numeric(sweating)

  if (any(is.na(c(tremor, hallucinations, orientation, agitation, sweating)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(tremor, hallucinations, orientation, agitation, sweating) < 0) ||
      any(c(tremor, hallucinations, orientation, agitation, sweating) > 2)) {
    stop("Error: All item scores must be integers between 0 and 2.")
  }

  total_score <- tremor + hallucinations + orientation + agitation + sweating

  severity <- ""
  management <- ""

  if (total_score < 4) {
    severity <- "Mild Withdrawal"
    management <- "Monitor. Pharmacotherapy usually not indicated unless history of seizures."
  } else if (total_score <= 7) {
    severity <- "Moderate Withdrawal"
    management <- "Pharmacotherapy indicated (e.g., Diazepam/Lorazepam). Reassess frequently."
  } else {
    severity <- "Severe Withdrawal"
    management <- "Aggressive pharmacotherapy required. High risk of complications (DTs). Consider senior review."
  }

  return(list(
    GMAWS_Score = total_score,
    Severity = severity,
    Management_Guidance = management,
    Inputs = list(
      Tremor = tremor,
      Hallucinations = hallucinations,
      Orientation = orientation,
      Agitation = agitation,
      Sweating = sweating
    )
  ))
}

gps_cancer_score <- function(crp_mg_l, albumin_g_l) {

  crp_mg_l <- as.numeric(crp_mg_l)
  albumin_g_l <- as.numeric(albumin_g_l)

  if (any(is.na(c(crp_mg_l, albumin_g_l)))) {
    stop("Error: CRP and Albumin must be numeric.")
  }

  score <- 0

  if (crp_mg_l > 10) {
    if (albumin_g_l < 35) {
      score <- 2
    } else {
      score <- 1
    }
  } else {
    score <- 0
  }

  prognosis <- ""
  if (score == 0) {
    prognosis <- "Good Prognosis"
  } else if (score == 1) {
    prognosis <- "Intermediate Prognosis"
  } else {
    prognosis <- "Poor Prognosis"
  }

  return(list(
    GPS_Score = score,
    Prognosis = prognosis,
    Inputs = list(
      CRP_mg_L = crp_mg_l,
      Albumin_g_L = albumin_g_l
    )
  ))
}

glasgow_blatchford_score <- function(bun, hemoglobin, systolic_bp, heart_rate, melena, syncope, liver_disease, cardiac_failure, bun_units = "mg/dL") {

  bun <- as.numeric(bun)
  hemoglobin <- as.numeric(hemoglobin)
  systolic_bp <- as.numeric(systolic_bp)
  heart_rate <- as.numeric(heart_rate)
  melena <- as.numeric(melena)
  syncope <- as.numeric(syncope)
  liver_disease <- as.numeric(liver_disease)
  cardiac_failure <- as.numeric(cardiac_failure)

  if (any(is.na(c(bun, hemoglobin, systolic_bp, heart_rate, melena, syncope, liver_disease, cardiac_failure)))) {
    stop("Error: All inputs must be numeric.")
  }

  bun_mg_dl <- bun
  if (bun_units == "mmol/L") {
    bun_mg_dl <- bun * 2.8
  }

  score <- 0

  if (bun_mg_dl >= 6.5 && bun_mg_dl < 8.0) {
    score <- score + 2
  } else if (bun_mg_dl >= 8.0 && bun_mg_dl < 10.0) {
    score <- score + 3
  } else if (bun_mg_dl >= 10.0 && bun_mg_dl < 25.0) {
    score <- score + 4
  } else if (bun_mg_dl >= 25.0) {
    score <- score + 6
  }

  if (hemoglobin >= 12.0 && hemoglobin < 13.0) {
    score <- score + 1
  } else if (hemoglobin >= 10.0 && hemoglobin < 12.0) {
    score <- score + 3
  } else if (hemoglobin < 10.0) {
    score <- score + 6
  }

  if (systolic_bp >= 100 && systolic_bp < 109) {
    score <- score + 1
  } else if (systolic_bp >= 90 && systolic_bp < 100) {
    score <- score + 2
  } else if (systolic_bp < 90) {
    score <- score + 3
  }

  if (heart_rate >= 100) {
    score <- score + 1
  }

  if (melena == 1) {
    score <- score + 1
  }

  if (syncope == 1) {
    score <- score + 2
  }

  if (liver_disease == 1) {
    score <- score + 2
  }

  if (cardiac_failure == 1) {
    score <- score + 2
  }

  risk_category <- ""
  if (score == 0) {
    risk_category <- "Low Risk (0.5% risk of needing intervention)"
  } else {
    risk_category <- "High Risk (Consider admission/endoscopy)"
  }

  return(list(
    GBS_Score = score,
    Risk_Category = risk_category,
    Inputs = list(
      BUN_mg_dL = round(bun_mg_dl, 1),
      Hemoglobin = hemoglobin,
      SBP = systolic_bp,
      Heart_Rate = heart_rate,
      Melena = melena == 1,
      Syncope = syncope == 1,
      Liver_Disease = liver_disease == 1,
      Cardiac_Failure = cardiac_failure == 1
    )
  ))
}

glasgow_imrie_pancreatitis_score <- function(age, wbc, glucose, urea, pao2, albumin, calcium, ldh, ast, alt, glucose_units = "mmol/L", urea_units = "mmol/L", calcium_units = "mmol/L", albumin_units = "g/L") {

  age <- as.numeric(age)
  wbc <- as.numeric(wbc)
  glucose <- as.numeric(glucose)
  urea <- as.numeric(urea)
  pao2 <- as.numeric(pao2)
  albumin <- as.numeric(albumin)
  calcium <- as.numeric(calcium)
  ldh <- as.numeric(ldh)
  ast <- as.numeric(ast)
  alt <- as.numeric(alt)

  if (any(is.na(c(age, wbc, glucose, urea, pao2, albumin, calcium, ldh, ast, alt)))) {
    stop("Error: All inputs must be numeric.")
  }

  glucose_val <- glucose
  if (glucose_units == "mg/dL") {
    glucose_val <- glucose / 18.018
  }

  urea_val <- urea
  if (urea_units == "mg/dL") {
    urea_val <- urea * 0.357
  } else if (urea_units == "urea_mg_dL") {
    urea_val <- urea * 0.1665
  }

  calcium_val <- calcium
  if (calcium_units == "mg/dL") {
    calcium_val <- calcium * 0.25
  }

  alb_val <- albumin
  if (albumin_units == "g/dL") {
    alb_val <- albumin * 10
  }

  score <- 0

  if (pao2 < 60) {
    score <- score + 1
  }

  if (age > 55) {
    score <- score + 1
  }

  if (wbc > 15) { # > 15 x 10^9/L
    score <- score + 1
  }

  if (calcium_val < 2.0) { # < 8.0 mg/dL
    score <- score + 1
  }

  if (urea_val > 16.0) { # > ~45 mg/dL BUN
    score <- score + 1
  }

  if (ldh > 600 || ast > 200 || alt > 200) {
    score <- score + 1
  }

  if (alb_val < 32) { # < 3.2 g/dL
    score <- score + 1
  }

  if (glucose_val > 10.0) { # > 180 mg/dL
    score <- score + 1
  }

  severity <- ""
  if (score >= 3) {
    severity <- "Severe Pancreatitis"
  } else {
    severity <- "Mild Pancreatitis"
  }

  return(list(
    Glasgow_Imrie_Score = score,
    Severity_Assessment = severity,
    Inputs = list(
      PaO2_lt_60mmHg = pao2 < 60,
      Age_gt_55 = age > 55,
      WBC_gt_15 = wbc > 15,
      Ca_lt_2mmol = calcium_val < 2.0,
      Urea_gt_16mmol = urea_val > 16.0,
      Enzymes_High = (ldh > 600 || ast > 200 || alt > 200),
      Albumin_lt_32gL = alb_val < 32,
      Glucose_gt_10mmol = glucose_val > 10.0
    )
  ))
}

gleason_score_calculator <- function(primary_grade, secondary_grade) {

  primary_grade <- as.numeric(primary_grade)
  secondary_grade <- as.numeric(secondary_grade)

  if (any(is.na(c(primary_grade, secondary_grade)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!primary_grade %in% 1:5 || !secondary_grade %in% 1:5) {
    stop("Error: Gleason grades must be integers between 1 and 5.")
  }

  total_score <- primary_grade + secondary_grade

  grade_group <- 0
  description <- ""

  if (total_score <= 6) {
    grade_group <- 1
    description <- "Gleason Score <= 6. Prognostic Grade Group 1. Favorable prognosis."
  } else if (total_score == 7) {
    if (primary_grade == 3) {
      grade_group <- 2
      description <- "Gleason Score 3+4=7. Prognostic Grade Group 2."
    } else {
      grade_group <- 3
      description <- "Gleason Score 4+3=7. Prognostic Grade Group 3."
    }
  } else if (total_score == 8) {
    grade_group <- 4
    description <- "Gleason Score 8. Prognostic Grade Group 4."
  } else {
    grade_group <- 5
    description <- "Gleason Score 9-10. Prognostic Grade Group 5. Poor prognosis."
  }

  return(list(
    Total_Gleason_Score = total_score,
    ISUP_Grade_Group = grade_group,
    Prognosis_Description = description,
    Inputs = list(
      Primary_Pattern = primary_grade,
      Secondary_Pattern = secondary_grade
    )
  ))
}

gold_copd_criteria <- function(fev1_percent_predicted, exacerbations_last_year, hospitalized_for_exacerbation, mmrc_score = NULL, cat_score = NULL) {

  fev1_percent_predicted <- as.numeric(fev1_percent_predicted)
  exacerbations_last_year <- as.numeric(exacerbations_last_year)
  hospitalized_for_exacerbation <- as.numeric(hospitalized_for_exacerbation)

  if (any(is.na(c(fev1_percent_predicted, exacerbations_last_year, hospitalized_for_exacerbation)))) {
    stop("Error: FEV1, Exacerbation history, and Hospitalization status are required.")
  }

  if (is.null(mmrc_score) && is.null(cat_score)) {
    stop("Error: At least one symptom score (mMRC or CAT) must be provided.")
  }

  gold_grade <- ""
  if (fev1_percent_predicted >= 80) {
    gold_grade <- "GOLD 1 (Mild)"
  } else if (fev1_percent_predicted >= 50) {
    gold_grade <- "GOLD 2 (Moderate)"
  } else if (fev1_percent_predicted >= 30) {
    gold_grade <- "GOLD 3 (Severe)"
  } else {
    gold_grade <- "GOLD 4 (Very Severe)"
  }

  is_high_risk <- FALSE
  if (exacerbations_last_year >= 2 || hospitalized_for_exacerbation == 1) {
    is_high_risk <- TRUE
  }

  is_high_symptoms <- FALSE

  if (!is.null(mmrc_score)) {
    if (as.numeric(mmrc_score) >= 2) is_high_symptoms <- TRUE
  }

  if (!is.null(cat_score)) {
    if (as.numeric(cat_score) >= 10) is_high_symptoms <- TRUE
  }

  gold_group <- ""
  treatment_rec <- ""

  if (!is_high_risk && !is_high_symptoms) {
    gold_group <- "Group A"
    treatment_rec <- "Bronchodilator (Short or Long acting)."
  } else if (!is_high_risk && is_high_symptoms) {
    gold_group <- "Group B"
    treatment_rec <- "Long-acting bronchodilator (LABA or LAMA)."
  } else if (is_high_risk && !is_high_symptoms) {
    gold_group <- "Group C"
    treatment_rec <- "LAMA."
  } else {
    gold_group <- "Group D"
    treatment_rec <- "LAMA or LAMA+LABA (if highly symptomatic) or ICS+LABA (if eos >= 300)."
  }

  return(list(
    GOLD_Spirometry_Grade = gold_grade,
    GOLD_Group = gold_group,
    Treatment_Recommendation = treatment_rec,
    Inputs = list(
      FEV1_Pct = fev1_percent_predicted,
      Exacerbations = exacerbations_last_year,
      Hospitalized = hospitalized_for_exacerbation == 1,
      mMRC = if(!is.null(mmrc_score)) mmrc_score else "N/A",
      CAT = if(!is.null(cat_score)) cat_score else "N/A"
    )
  ))
}

glucose_infusion_rate <- function(iv_rate_ml_hr, dextrose_conc_percent, weight_kg) {

  iv_rate_ml_hr <- as.numeric(iv_rate_ml_hr)
  dextrose_conc_percent <- as.numeric(dextrose_conc_percent)
  weight_kg <- as.numeric(weight_kg)

  if (any(is.na(c(iv_rate_ml_hr, dextrose_conc_percent, weight_kg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  gir <- (iv_rate_ml_hr * dextrose_conc_percent) / (6 * weight_kg)

  interpretation <- ""
  if (gir < 4) {
    interpretation <- "Low GIR. Risk of hypoglycemia. Typical maintenance is 4-6 mg/kg/min."
  } else if (gir >= 4 && gir <= 8) {
    interpretation <- "Normal Maintenance Range (Neonatal)."
  } else {
    interpretation <- "High GIR. Monitor for hyperglycemia. Typical max limits ~12-14 mg/kg/min."
  }

  return(list(
    GIR_mg_kg_min = round(gir, 2),
    Interpretation = interpretation,
    Inputs = list(
      Rate_mL_hr = iv_rate_ml_hr,
      Dextrose_Pct = dextrose_conc_percent,
      Weight_kg = weight_kg
    )
  ))
}

go_far_cardiac_arrest_score <- function(age, neurologically_intact_admission, major_trauma, acute_stroke, metastatic_hematologic_cancer, septicemia, medical_noncardiac_diagnosis, hepatic_insufficiency, skilled_nursing_facility, hypotension_hypoperfusion, renal_insufficiency_dialysis, respiratory_insufficiency, pneumonia) {

  age <- as.numeric(age)
  neurologically_intact_admission <- as.numeric(neurologically_intact_admission)
  major_trauma <- as.numeric(major_trauma)
  acute_stroke <- as.numeric(acute_stroke)
  metastatic_hematologic_cancer <- as.numeric(metastatic_hematologic_cancer)
  septicemia <- as.numeric(septicemia)
  medical_noncardiac_diagnosis <- as.numeric(medical_noncardiac_diagnosis)
  hepatic_insufficiency <- as.numeric(hepatic_insufficiency)
  skilled_nursing_facility <- as.numeric(skilled_nursing_facility)
  hypotension_hypoperfusion <- as.numeric(hypotension_hypoperfusion)
  renal_insufficiency_dialysis <- as.numeric(renal_insufficiency_dialysis)
  respiratory_insufficiency <- as.numeric(respiratory_insufficiency)
  pneumonia <- as.numeric(pneumonia)

  if (any(is.na(c(age, neurologically_intact_admission, major_trauma, acute_stroke, metastatic_hematologic_cancer, septicemia, medical_noncardiac_diagnosis, hepatic_insufficiency, skilled_nursing_facility, hypotension_hypoperfusion, renal_insufficiency_dialysis, respiratory_insufficiency, pneumonia)))) {
    stop("Error: All inputs must be numeric (0 or 1, and age).")
  }

  score <- 0

  if (neurologically_intact_admission == 1) score <- score - 15
  if (major_trauma == 1) score <- score + 10
  if (acute_stroke == 1) score <- score + 8
  if (metastatic_hematologic_cancer == 1) score <- score + 7
  if (septicemia == 1) score <- score + 7
  if (medical_noncardiac_diagnosis == 1) score <- score + 7
  if (hepatic_insufficiency == 1) score <- score + 6
  if (skilled_nursing_facility == 1) score <- score + 6
  if (hypotension_hypoperfusion == 1) score <- score + 5
  if (renal_insufficiency_dialysis == 1) score <- score + 4
  if (respiratory_insufficiency == 1) score <- score + 4
  if (pneumonia == 1) score <- score + 1

  if (age >= 70 && age <= 74) score <- score + 2
  else if (age >= 75 && age <= 79) score <- score + 5
  else if (age >= 80 && age <= 84) score <- score + 6
  else if (age >= 85) score <- score + 11

  survival_prob <- ""
  risk_group <- ""

  if (score >= 24) {
    risk_group <- "Very Low Probability"
    survival_prob <- "< 1%"
  } else if (score >= 14) {
    risk_group <- "Low Probability"
    survival_prob <- "1% - 3%"
  } else if (score >= -5) {
    risk_group <- "Average Probability"
    survival_prob <- "3% - 15%"
  } else {
    risk_group <- "Above Average Probability"
    survival_prob <- "> 15%"
  }

  return(list(
    GO_FAR_Score = score,
    Probability_Neurologically_Intact_Survival = survival_prob,
    Risk_Group = risk_group,
    Inputs = list(
      Age = age,
      Neuro_Intact = neurologically_intact_admission == 1,
      Major_Trauma = major_trauma == 1,
      Acute_Stroke = acute_stroke == 1,
      Metastatic_Heme_Cancer = metastatic_hematologic_cancer == 1,
      Septicemia = septicemia == 1,
      Medical_NonCardiac = medical_noncardiac_diagnosis == 1,
      Hepatic_Insufficiency = hepatic_insufficiency == 1,
      SNF_Admit = skilled_nursing_facility == 1,
      Hypotension = hypotension_hypoperfusion == 1,
      Renal_Insufficiency = renal_insufficiency_dialysis == 1,
      Resp_Insufficiency = respiratory_insufficiency == 1,
      Pneumonia = pneumonia == 1
    )
  ))
}

grace_acs_risk <- function(age, heart_rate, systolic_bp, creatinine, killip_class, cardiac_arrest_admission, st_segment_deviation, elevated_cardiac_enzymes, creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  heart_rate <- as.numeric(heart_rate)
  systolic_bp <- as.numeric(systolic_bp)
  creatinine <- as.numeric(creatinine)
  killip_class <- as.numeric(killip_class)
  cardiac_arrest_admission <- as.numeric(cardiac_arrest_admission)
  st_segment_deviation <- as.numeric(st_segment_deviation)
  elevated_cardiac_enzymes <- as.numeric(elevated_cardiac_enzymes)

  if (any(is.na(c(age, heart_rate, systolic_bp, creatinine, killip_class, cardiac_arrest_admission, st_segment_deviation, elevated_cardiac_enzymes)))) {
    stop("Error: All inputs must be numeric.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  score <- 0

  if (age < 30) score <- score + 0
  else if (age >= 30 && age <= 39) score <- score + 0
  else if (age >= 40 && age <= 49) score <- score + 18
  else if (age >= 50 && age <= 59) score <- score + 36
  else if (age >= 60 && age <= 69) score <- score + 55
  else if (age >= 70 && age <= 79) score <- score + 73
  else if (age >= 80 && age <= 89) score <- score + 91
  else score <- score + 100

  if (heart_rate < 50) score <- score + 0
  else if (heart_rate >= 50 && heart_rate <= 69) score <- score + 3
  else if (heart_rate >= 70 && heart_rate <= 89) score <- score + 9
  else if (heart_rate >= 90 && heart_rate <= 109) score <- score + 15
  else if (heart_rate >= 110 && heart_rate <= 149) score <- score + 24
  else if (heart_rate >= 150 && heart_rate <= 199) score <- score + 38
  else score <- score + 46

  if (systolic_bp < 80) score <- score + 58
  else if (systolic_bp >= 80 && systolic_bp <= 99) score <- score + 53
  else if (systolic_bp >= 100 && systolic_bp <= 119) score <- score + 43
  else if (systolic_bp >= 120 && systolic_bp <= 139) score <- score + 34
  else if (systolic_bp >= 140 && systolic_bp <= 159) score <- score + 24
  else if (systolic_bp >= 160 && systolic_bp <= 199) score <- score + 10
  else score <- score + 0

  if (creat_mg_dl >= 0 && creat_mg_dl <= 0.39) score <- score + 1
  else if (creat_mg_dl >= 0.4 && creat_mg_dl <= 0.79) score <- score + 4
  else if (creat_mg_dl >= 0.8 && creat_mg_dl <= 1.19) score <- score + 7
  else if (creat_mg_dl >= 1.2 && creat_mg_dl <= 1.59) score <- score + 10
  else if (creat_mg_dl >= 1.6 && creat_mg_dl <= 1.99) score <- score + 13
  else if (creat_mg_dl >= 2.0 && creat_mg_dl <= 3.99) score <- score + 21
  else score <- score + 28

  if (killip_class == 1) score <- score + 0
  else if (killip_class == 2) score <- score + 20
  else if (killip_class == 3) score <- score + 39
  else if (killip_class == 4) score <- score + 59

  if (cardiac_arrest_admission == 1) score <- score + 39
  if (st_segment_deviation == 1) score <- score + 28
  if (elevated_cardiac_enzymes == 1) score <- score + 14

  logit <- -7.7035 + (0.0531 * score)
  prob <- 1 / (1 + exp(-logit))
  prob_percent <- prob * 100

  risk_category <- ""
  if (prob_percent < 1) risk_category <- "Low Risk"
  else if (prob_percent <= 3) risk_category <- "Intermediate Risk"
  else risk_category <- "High Risk"

  return(list(
    GRACE_Score = score,
    In_Hospital_Mortality_Risk = paste0(round(prob_percent, 1), "%"),
    Risk_Category = risk_category,
    Inputs = list(
      Age = age,
      HR = heart_rate,
      SBP = systolic_bp,
      Creatinine_mg_dL = round(creat_mg_dl, 2),
      Killip_Class = killip_class,
      Cardiac_Arrest = cardiac_arrest_admission == 1,
      ST_Deviation = st_segment_deviation == 1,
      Elevated_Enzymes = elevated_cardiac_enzymes == 1
    )
  ))
}

gigpa_score <- function(kps, age, extracranial_metastases, num_brain_metastases) {

  kps <- as.numeric(kps)
  age <- as.numeric(age)
  extracranial_metastases <- as.numeric(extracranial_metastases)
  num_brain_metastases <- as.numeric(num_brain_metastases)

  if (any(is.na(c(kps, age, extracranial_metastases, num_brain_metastases)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (kps >= 90) {
    score <- score + 2.0
  } else if (kps == 80) {
    score <- score + 1.5
  } else if (kps == 70) {
    score <- score + 1.0
  } else {
    score <- score + 0.0
  }

  if (age < 70) {
    score <- score + 0.5
  } else {
    score <- score + 0.0
  }

  if (extracranial_metastases == 0) {
    score <- score + 1.0
  } else {
    score <- score + 0.0
  }

  if (num_brain_metastases == 1) {
    score <- score + 0.5
  } else {
    score <- score + 0.0
  }

  median_survival <- ""
  if (score >= 3.5 && score <= 4.0) {
    median_survival <- "17 months"
  } else if (score >= 2.5 && score <= 3.0) {
    median_survival <- "9 months"
  } else if (score >= 1.5 && score <= 2.0) {
    median_survival <- "6 months"
  } else {
    median_survival <- "3 months"
  }

  return(list(
    GI_GPA_Score = score,
    Median_Survival = median_survival,
    Inputs = list(
      KPS = kps,
      Age = age,
      Extracranial_Mets = if(extracranial_metastases == 1) "Present" else "Absent",
      Brain_Mets_Count = num_brain_metastases
    )
  ))
}

grogan_attr_cm_staging <- function(nt_probnp_pg_ml, troponin_t_ng_ml = NULL, troponin_i_ng_ml = NULL) {

  nt_probnp_pg_ml <- as.numeric(nt_probnp_pg_ml)

  if (is.na(nt_probnp_pg_ml)) {
    stop("Error: NT-proBNP must be numeric.")
  }

  troponin_t <- NA
  troponin_i <- NA

  if (!is.null(troponin_t_ng_ml)) {
    troponin_t <- as.numeric(troponin_t_ng_ml)
  }

  if (!is.null(troponin_i_ng_ml)) {
    troponin_i <- as.numeric(troponin_i_ng_ml)
  }

  if (is.na(troponin_t) && is.na(troponin_i)) {
    stop("Error: Either Troponin T or Troponin I must be provided.")
  }

  score <- 0

  if (nt_probnp_pg_ml >= 3000) {
    score <- score + 1
  }

  trop_elevated <- FALSE
  if (!is.na(troponin_t)) {
    if (troponin_t >= 0.05) {
      trop_elevated <- TRUE
    }
  } else if (!is.na(troponin_i)) {
    if (troponin_i >= 0.1) {
      trop_elevated <- TRUE
    }
  }

  if (trop_elevated) {
    score <- score + 1
  }

  stage <- 0
  median_survival <- ""

  if (score == 0) {
    stage <- 1
    median_survival <- "66 months"
  } else if (score == 1) {
    stage <- 2
    median_survival <- "40 months"
  } else {
    stage <- 3
    median_survival <- "20 months"
  }

  return(list(
    Grogan_Stage = stage,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      NT_proBNP = nt_probnp_pg_ml,
      Troponin_Used = if(!is.na(troponin_t)) paste("TnT:", troponin_t) else paste("TnI:", troponin_i)
    )
  ))
}

gelf_criteria_follicular_lymphoma <- function(nodal_mass_gt_7cm, multiple_nodal_sites_gt_3cm, systemic_symptoms, spleen_enlargement, organ_compression, pleural_ascitic_effusion, ldh_or_beta2_elevated) {

  nodal_mass_gt_7cm <- as.numeric(nodal_mass_gt_7cm)
  multiple_nodal_sites_gt_3cm <- as.numeric(multiple_nodal_sites_gt_3cm)
  systemic_symptoms <- as.numeric(systemic_symptoms)
  spleen_enlargement <- as.numeric(spleen_enlargement)
  organ_compression <- as.numeric(organ_compression)
  pleural_ascitic_effusion <- as.numeric(pleural_ascitic_effusion)
  ldh_or_beta2_elevated <- as.numeric(ldh_or_beta2_elevated)

  if (any(is.na(c(nodal_mass_gt_7cm, multiple_nodal_sites_gt_3cm, systemic_symptoms, spleen_enlargement, organ_compression, pleural_ascitic_effusion, ldh_or_beta2_elevated)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  criteria_met_count <- 0

  if (nodal_mass_gt_7cm == 1) criteria_met_count <- criteria_met_count + 1
  if (multiple_nodal_sites_gt_3cm == 1) criteria_met_count <- criteria_met_count + 1
  if (systemic_symptoms == 1) criteria_met_count <- criteria_met_count + 1
  if (spleen_enlargement == 1) criteria_met_count <- criteria_met_count + 1
  if (organ_compression == 1) criteria_met_count <- criteria_met_count + 1
  if (pleural_ascitic_effusion == 1) criteria_met_count <- criteria_met_count + 1
  if (ldh_or_beta2_elevated == 1) criteria_met_count <- criteria_met_count + 1

  result <- ""
  recommendation <- ""

  if (criteria_met_count >= 1) {
    result <- "High Tumor Burden (GELF Criteria Met)"
    recommendation <- "Consider initiation of therapy (e.g., chemoimmunotherapy)."
  } else {
    result <- "Low Tumor Burden (GELF Criteria Not Met)"
    recommendation <- "Consider 'Watch and Wait' strategy."
  }

  return(list(
    GELF_Result = result,
    Criteria_Count = criteria_met_count,
    Recommendation = recommendation,
    Inputs = list(
      Mass_gt_7cm = nodal_mass_gt_7cm == 1,
      Multiple_Nodes_gt_3cm = multiple_nodal_sites_gt_3cm == 1,
      B_Symptoms = systemic_symptoms == 1,
      Splenomegaly = spleen_enlargement == 1,
      Compression = organ_compression == 1,
      Effusion = pleural_ascitic_effusion == 1,
      Elevated_Markers = ldh_or_beta2_elevated == 1
    )
  ))
}

gupta_mica_risk <- function(age, creatinine_mg_dl, asa_class, functional_status, procedure_type) {

  age <- as.numeric(age)
  creatinine_mg_dl <- as.numeric(creatinine_mg_dl)
  asa_class <- as.numeric(asa_class)
  functional_status <- tolower(as.character(functional_status))
  procedure_type <- tolower(as.character(procedure_type))

  if (any(is.na(c(age, creatinine_mg_dl, asa_class)))) {
    stop("Error: Age, Creatinine, and ASA Class must be numeric.")
  }

  if (!asa_class %in% 1:5) {
    stop("Error: ASA Class must be an integer between 1 and 5.")
  }

  if (!functional_status %in% c("independent", "partially_dependent", "totally_dependent")) {
    stop("Error: Functional status must be 'independent', 'partially_dependent', or 'totally_dependent'.")
  }

  logit <- -3.91

  logit <- logit + (0.028 * age)

  if (creatinine_mg_dl > 1.5) {
    logit <- logit + 0.59
  }

  if (asa_class == 2) {
    logit <- logit + 0.47
  } else if (asa_class == 3) {
    logit <- logit + 1.05
  } else if (asa_class == 4) {
    logit <- logit + 1.75
  } else if (asa_class == 5) {
    logit <- logit + 1.75
  }

  if (functional_status == "partially_dependent") {
    logit <- logit + 0.58
  } else if (functional_status == "totally_dependent") {
    logit <- logit + 0.93
  }

  proc_coeff <- 0
  if (procedure_type == "anorectal") proc_coeff <- -0.54
  else if (procedure_type == "aortic") proc_coeff <- 1.07
  else if (procedure_type == "bariatric") proc_coeff <- -0.16
  else if (procedure_type == "brain") proc_coeff <- 0.34
  else if (procedure_type == "breast") proc_coeff <- -0.65
  else if (procedure_type == "cardiac") proc_coeff <- 1.20
  else if (procedure_type == "ent") proc_coeff <- -0.38
  else if (procedure_type == "foregut_hepatopancreatobiliary") proc_coeff <- 0.45
  else if (procedure_type == "gallbladder_appendix_adrenal_spleen") proc_coeff <- 0.05
  else if (procedure_type == "hernia_ventral_inguinal") proc_coeff <- -0.08
  else if (procedure_type == "intestinal") proc_coeff <- 0.48
  else if (procedure_type == "neck_thyroid_parathyroid") proc_coeff <- -0.20
  else if (procedure_type == "obgyn") proc_coeff <- -0.13
  else if (procedure_type == "ortho_non_spine") proc_coeff <- 0.08
  else if (procedure_type == "spine") proc_coeff <- 0.15
  else if (procedure_type == "thoracic_non_cardiac") proc_coeff <- 0.65
  else if (procedure_type == "urology") proc_coeff <- 0.08
  else if (procedure_type == "vascular_non_aortic") proc_coeff <- 0.68
  else if (procedure_type == "vein") proc_coeff <- -0.90
  else stop("Error: Procedure type not recognized. Please check documentation for valid categories.")

  logit <- logit + proc_coeff

  probability <- exp(logit) / (1 + exp(logit))
  probability_percent <- probability * 100

  return(list(
    MICA_Risk_Percent = round(probability_percent, 2),
    Inputs = list(
      Age = age,
      Creatinine = creatinine_mg_dl,
      ASA_Class = asa_class,
      Function = functional_status,
      Procedure = procedure_type
    )
  ))
}

gupta_pneumonia_risk <- function(age, copd, functional_status, asa_class, sepsis_status, smoker, procedure_type) {

  age <- as.numeric(age)
  copd <- as.numeric(copd)
  functional_status <- tolower(as.character(functional_status))
  asa_class <- as.numeric(asa_class)
  sepsis_status <- tolower(as.character(sepsis_status))
  smoker <- as.numeric(smoker)
  procedure_type <- tolower(as.character(procedure_type))

  if (any(is.na(c(age, copd, asa_class, smoker)))) {
    stop("Error: All inputs must be numeric or binary where required.")
  }

  if (!asa_class %in% 1:5) {
    stop("Error: ASA Class must be an integer between 1 and 5.")
  }


  beta_0 <- -5.089

  beta_age <- 0.02 * age

  beta_copd <- if (copd == 1) 0.37 else 0

  beta_func <- 0
  if (functional_status == "partially_dependent") beta_func <- 0.65
  else if (functional_status == "totally_dependent") beta_func <- 1.12

  beta_asa <- 0
  if (asa_class == 2) beta_asa <- 0.53
  else if (asa_class == 3) beta_asa <- 1.19
  else if (asa_class == 4) beta_asa <- 1.76
  else if (asa_class == 5) beta_asa <- 1.76 # Usually clustered with 4 or higher risk

  beta_sepsis <- 0
  if (sepsis_status == "sirs") beta_sepsis <- 0.49
  else if (sepsis_status == "sepsis") beta_sepsis <- 0.71
  else if (sepsis_status == "septic_shock") beta_sepsis <- 1.22

  beta_smoke <- if (smoker == 1) 0.27 else 0

  beta_proc <- 0
  if (procedure_type == "anorectal") beta_proc <- -0.24
  else if (procedure_type == "aortic") beta_proc <- 0.91
  else if (procedure_type == "bariatric") beta_proc <- -0.29
  else if (procedure_type == "brain") beta_proc <- 0.71
  else if (procedure_type == "breast") beta_proc <- -1.44
  else if (procedure_type == "cardiac") beta_proc <- 0.48
  else if (procedure_type == "ent") beta_proc <- 0.33
  else if (procedure_type == "foregut_hepatopancreatobiliary") beta_proc <- 0.70
  else if (procedure_type == "gallbladder_appendix_adrenal_spleen") beta_proc <- 0.22
  else if (procedure_type == "hernia_ventral_inguinal") beta_proc <- -0.24
  else if (procedure_type == "intestinal") beta_proc <- 0.51
  else if (procedure_type == "neck_thyroid_parathyroid") beta_proc <- 0.08
  else if (procedure_type == "obgyn") beta_proc <- -0.36
  else if (procedure_type == "orthopedic") beta_proc <- -0.32
  else if (procedure_type == "other_abdominal") beta_proc <- 0.39
  else if (procedure_type == "peripheral_vascular") beta_proc <- 0.03
  else if (procedure_type == "skin") beta_proc <- -0.11
  else if (procedure_type == "spine") beta_proc <- -0.29
  else if (procedure_type == "thoracic_non_cardiac") beta_proc <- 0.63
  else if (procedure_type == "urology") beta_proc <- -0.33
  else if (procedure_type == "vein") beta_proc <- -1.17
  else stop("Error: Procedure type not recognized.")

  logit <- beta_0 + beta_age + beta_copd + beta_func + beta_asa + beta_sepsis + beta_smoke + beta_proc

  probability <- exp(logit) / (1 + exp(logit))
  probability_percent <- probability * 100

  return(list(
    Pneumonia_Risk_Percent = round(probability_percent, 2),
    Inputs = list(
      Age = age,
      COPD = copd == 1,
      Functional_Status = functional_status,
      ASA_Class = asa_class,
      Sepsis = sepsis_status,
      Smoker = smoker == 1,
      Procedure = procedure_type
    )
  ))
}

gupta_respiratory_failure_risk <- function(procedure_type, asa_class, functional_status, emergency_case, sepsis_status) {

  procedure_type <- tolower(as.character(procedure_type))
  asa_class <- as.numeric(asa_class)
  functional_status <- tolower(as.character(functional_status))
  emergency_case <- as.numeric(emergency_case)
  sepsis_status <- tolower(as.character(sepsis_status))

  if (any(is.na(c(asa_class, emergency_case)))) {
    stop("Error: ASA Class and Emergency status must be numeric.")
  }

  if (!asa_class %in% 1:5) {
    stop("Error: ASA Class must be an integer between 1 and 5.")
  }

  beta_0 <- -6.31

  beta_proc <- 0
  if (procedure_type == "anorectal") beta_proc <- -0.18
  else if (procedure_type == "aortic") beta_proc <- 1.79
  else if (procedure_type == "bariatric") beta_proc <- -0.39
  else if (procedure_type == "brain") beta_proc <- 2.03
  else if (procedure_type == "breast") beta_proc <- -0.86
  else if (procedure_type == "cardiac") beta_proc <- 1.32
  else if (procedure_type == "ent") beta_proc <- -0.02
  else if (procedure_type == "foregut_hepatopancreatobiliary") beta_proc <- 0.97
  else if (procedure_type == "gallbladder_appendix_adrenal_spleen") beta_proc <- 0.43
  else if (procedure_type == "hernia_ventral_inguinal") beta_proc <- -0.14
  else if (procedure_type == "intestinal") beta_proc <- 0.62
  else if (procedure_type == "neck_thyroid_parathyroid") beta_proc <- -0.67
  else if (procedure_type == "obgyn") beta_proc <- -0.49
  else if (procedure_type == "orthopedic") beta_proc <- -0.27
  else if (procedure_type == "other_abdominal") beta_proc <- 0.47
  else if (procedure_type == "peripheral_vascular") beta_proc <- 0.07
  else if (procedure_type == "skin") beta_proc <- -0.15
  else if (procedure_type == "spine") beta_proc <- -0.13
  else if (procedure_type == "thoracic_non_cardiac") beta_proc <- 1.16
  else if (procedure_type == "urology") beta_proc <- -0.41
  else if (procedure_type == "vein") beta_proc <- -1.59
  else stop("Error: Procedure type not recognized.")

  beta_emergency <- if (emergency_case == 1) 0.61 else 0

  beta_func <- 0
  if (functional_status == "partially_dependent") beta_func <- 0.58
  else if (functional_status == "totally_dependent") beta_func <- 1.06

  beta_asa <- 0
  if (asa_class == 2) beta_asa <- 0.49
  else if (asa_class == 3) beta_asa <- 1.23
  else if (asa_class == 4) beta_asa <- 1.96
  else if (asa_class == 5) beta_asa <- 1.96

  beta_sepsis <- 0
  if (sepsis_status == "sirs") beta_sepsis <- 0.61
  else if (sepsis_status == "sepsis") beta_sepsis <- 0.74
  else if (sepsis_status == "septic_shock") beta_sepsis <- 1.45

  logit <- beta_0 + beta_proc + beta_emergency + beta_func + beta_asa + beta_sepsis

  probability <- exp(logit) / (1 + exp(logit))
  probability_percent <- probability * 100

  return(list(
    Respiratory_Failure_Risk_Percent = round(probability_percent, 2),
    Inputs = list(
      Procedure = procedure_type,
      Emergency = emergency_case == 1,
      Functional_Status = functional_status,
      ASA_Class = asa_class,
      Sepsis = sepsis_status
    )
  ))
}

gwtg_hf_risk_score <- function(age, systolic_bp, bun, heart_rate, sodium, copd, race_black = 0, bun_units = "mg/dL", sodium_units = "mEq/L") {

  age <- as.numeric(age)
  systolic_bp <- as.numeric(systolic_bp)
  bun <- as.numeric(bun)
  heart_rate <- as.numeric(heart_rate)
  sodium <- as.numeric(sodium)
  copd <- as.numeric(copd)
  race_black <- as.numeric(race_black)

  if (any(is.na(c(age, systolic_bp, bun, heart_rate, sodium, copd, race_black)))) {
    stop("Error: All inputs must be numeric.")
  }

  bun_mg_dl <- bun
  if (bun_units == "mmol/L") {
    bun_mg_dl <- bun * 2.8
  }

  score <- 0

  if (age < 20) score <- score + 0
  else if (age < 30) score <- score + 0
  else if (age < 40) score <- score + 0
  else if (age < 50) score <- score + 3
  else if (age < 60) score <- score + 8
  else if (age < 70) score <- score + 14
  else if (age < 80) score <- score + 19
  else if (age < 90) score <- score + 25
  else score <- score + 30

  if (systolic_bp < 50) score <- score + 28
  else if (systolic_bp < 60) score <- score + 26
  else if (systolic_bp < 70) score <- score + 24
  else if (systolic_bp < 80) score <- score + 23
  else if (systolic_bp < 90) score <- score + 21
  else if (systolic_bp < 100) score <- score + 19
  else if (systolic_bp < 110) score <- score + 17
  else if (systolic_bp < 120) score <- score + 15
  else if (systolic_bp < 130) score <- score + 13
  else if (systolic_bp < 140) score <- score + 11
  else if (systolic_bp < 150) score <- score + 9
  else if (systolic_bp < 160) score <- score + 7
  else if (systolic_bp < 170) score <- score + 5
  else if (systolic_bp < 180) score <- score + 3
  else if (systolic_bp < 190) score <- score + 1
  else score <- score + 0

  if (bun_mg_dl < 10) score <- score + 0
  else if (bun_mg_dl < 20) score <- score + 2
  else if (bun_mg_dl < 30) score <- score + 4
  else if (bun_mg_dl < 40) score <- score + 6
  else if (bun_mg_dl < 50) score <- score + 8
  else if (bun_mg_dl < 60) score <- score + 9
  else if (bun_mg_dl < 70) score <- score + 11
  else if (bun_mg_dl < 80) score <- score + 13
  else if (bun_mg_dl < 90) score <- score + 15
  else if (bun_mg_dl < 100) score <- score + 17
  else if (bun_mg_dl < 110) score <- score + 19
  else if (bun_mg_dl < 120) score <- score + 21
  else if (bun_mg_dl < 130) score <- score + 23
  else if (bun_mg_dl < 140) score <- score + 25
  else if (bun_mg_dl < 150) score <- score + 27
  else score <- score + 28

  if (heart_rate < 80) score <- score + 0
  else if (heart_rate < 85) score <- score + 1
  else if (heart_rate < 90) score <- score + 3
  else if (heart_rate < 95) score <- score + 4
  else if (heart_rate < 100) score <- score + 5
  else if (heart_rate < 105) score <- score + 6
  else score <- score + 8

  if (sodium < 131) score <- score + 4
  else if (sodium < 134) score <- score + 3
  else if (sodium < 137) score <- score + 2
  else if (sodium < 140) score <- score + 1
  else score <- score + 0

  if (copd == 1) score <- score + 2

  if (race_black == 0) score <- score + 3

  mortality <- ""
  if (score <= 33) mortality <- "< 1%"
  else if (score <= 43) mortality <- "1% - 3%"
  else if (score <= 58) mortality <- "3% - 10%"
  else if (score <= 78) mortality <- "10% - 30%"
  else mortality <- "> 30%"

  return(list(
    GWTG_HF_Score = score,
    Est_In_Hospital_Mortality = mortality,
    Inputs = list(
      Age = age,
      SBP = systolic_bp,
      BUN_mg_dL = round(bun_mg_dl, 1),
      HR = heart_rate,
      Na = sodium,
      COPD = copd == 1,
      Race_Black = race_black == 1
    )
  ))
}

h2fpef_score <- function(bmi, num_hypertensive_meds, atrial_fibrillation, pulmonary_hypertension_gt_35, age_gt_60, e_e_prime_gt_9) {

  bmi <- as.numeric(bmi)
  num_hypertensive_meds <- as.numeric(num_hypertensive_meds)
  atrial_fibrillation <- as.numeric(atrial_fibrillation)
  pulmonary_hypertension_gt_35 <- as.numeric(pulmonary_hypertension_gt_35)
  age_gt_60 <- as.numeric(age_gt_60)
  e_e_prime_gt_9 <- as.numeric(e_e_prime_gt_9)

  if (any(is.na(c(bmi, num_hypertensive_meds, atrial_fibrillation, pulmonary_hypertension_gt_35, age_gt_60, e_e_prime_gt_9)))) {
    stop("Error: All inputs must be numeric (0 or 1 for booleans, BMI as number).")
  }

  score <- 0

  if (bmi > 30) {
    score <- score + 2
  }

  if (num_hypertensive_meds >= 2) {
    score <- score + 1
  }

  if (atrial_fibrillation == 1) {
    score <- score + 3
  }

  if (pulmonary_hypertension_gt_35 == 1) {
    score <- score + 1
  }

  if (age_gt_60 == 1) {
    score <- score + 1
  }

  if (e_e_prime_gt_9 == 1) {
    score <- score + 1
  }

  probability <- ""
  if (score <= 1) {
    probability <- "< 20% (Low Probability)"
  } else if (score >= 2 && score <= 5) {
    probability <- "Intermediate Probability (Requires further testing)"
  } else {
    probability <- "> 90% (High Probability)"
  }

  return(list(
    H2FPEF_Score = score,
    HFpEF_Probability = probability,
    Inputs = list(
      BMI_gt_30 = bmi > 30,
      HTN_Meds_ge_2 = num_hypertensive_meds >= 2,
      AFib = atrial_fibrillation == 1,
      Pulm_HTN = pulmonary_hypertension_gt_35 == 1,
      Age_gt_60 = age_gt_60 == 1,
      Filling_Pressure_High = e_e_prime_gt_9 == 1
    )
  ))
}

hacks_impairment_index <- function(gait_speed_lt_0_8, physical_performance_lt_10, mmse_lt_26, incontinence, adl_impairment, iadl_impairment) {

  gait_speed_lt_0_8 <- as.numeric(gait_speed_lt_0_8)
  physical_performance_lt_10 <- as.numeric(physical_performance_lt_10)
  mmse_lt_26 <- as.numeric(mmse_lt_26)
  incontinence <- as.numeric(incontinence)
  adl_impairment <- as.numeric(adl_impairment)
  iadl_impairment <- as.numeric(iadl_impairment)

  if (any(is.na(c(gait_speed_lt_0_8, physical_performance_lt_10, mmse_lt_26, incontinence, adl_impairment, iadl_impairment)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (gait_speed_lt_0_8 == 1) {
    score <- score + 1
  }

  if (physical_performance_lt_10 == 1) {
    score <- score + 1
  }

  if (mmse_lt_26 == 1) {
    score <- score + 1
  }

  if (incontinence == 1) {
    score <- score + 1
  }

  if (adl_impairment == 1) {
    score <- score + 1
  }

  if (iadl_impairment == 1) {
    score <- score + 1
  }

  risk_level <- ""
  institutionalization_risk <- ""

  if (score == 0) {
    risk_level <- "No Impairment"
    institutionalization_risk <- "Low Risk"
  } else if (score <= 2) {
    risk_level <- "Mild Impairment"
    institutionalization_risk <- "Intermediate Risk"
  } else {
    risk_level <- "Severe Impairment"
    institutionalization_risk <- "High Risk"
  }

  return(list(
    HII_Score = score,
    Impairment_Level = risk_level,
    Institutionalization_Risk = institutionalization_risk,
    Inputs = list(
      Slow_Gait = gait_speed_lt_0_8 == 1,
      Poor_Physical_Perf = physical_performance_lt_10 == 1,
      Cognitive_Impairment = mmse_lt_26 == 1,
      Incontinence = incontinence == 1,
      ADL_Impairment = adl_impairment == 1,
      IADL_Impairment = iadl_impairment == 1
    )
  ))
}

hacor_score <- function(heart_rate, ph, gcs, pao2_fio2_ratio, respiratory_rate) {

  heart_rate <- as.numeric(heart_rate)
  ph <- as.numeric(ph)
  gcs <- as.numeric(gcs)
  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  respiratory_rate <- as.numeric(respiratory_rate)

  if (any(is.na(c(heart_rate, ph, gcs, pao2_fio2_ratio, respiratory_rate)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (heart_rate >= 120) {
    score <- score + 1
  }

  if (ph < 7.25) {
    score <- score + 4
  } else if (ph >= 7.25 && ph < 7.30) {
    score <- score + 3
  } else if (ph >= 7.30 && ph < 7.35) {
    score <- score + 2
  }

  if (gcs < 15) {
    score <- score + 10
  }

  if (pao2_fio2_ratio <= 100) {
    score <- score + 6
  } else if (pao2_fio2_ratio <= 125) {
    score <- score + 5
  } else if (pao2_fio2_ratio <= 150) {
    score <- score + 4
  } else if (pao2_fio2_ratio <= 175) {
    score <- score + 3
  } else if (pao2_fio2_ratio <= 200) {
    score <- score + 2
  }

  if (respiratory_rate > 45) {
    score <- score + 4
  } else if (respiratory_rate >= 41) {
    score <- score + 3
  } else if (respiratory_rate >= 36) {
    score <- score + 2
  } else if (respiratory_rate >= 31) {
    score <- score + 1
  }

  prob_failure <- ""
  recommendation <- ""

  if (score <= 5) {
    prob_failure <- "18% (Low Risk)"
    recommendation <- "Continue NIV."
  } else {
    prob_failure <- "> 80% (High Risk)"
    recommendation <- "Consider early intubation."
  }

  return(list(
    HACOR_Score = score,
    NIV_Failure_Probability = prob_failure,
    Recommendation = recommendation,
    Inputs = list(
      HR = heart_rate,
      pH = ph,
      GCS = gcs,
      PF_Ratio = pao2_fio2_ratio,
      RR = respiratory_rate
    )
  ))
}

hamilton_anxiety_scale <- function(anxious_mood, tension, fears, insomnia, intellectual, depressed_mood, somatic_muscular, somatic_sensory, cardiovascular, respiratory, gastrointestinal, genitourinary, autonomic, behavior_at_interview) {

  inputs <- c(anxious_mood, tension, fears, insomnia, intellectual, depressed_mood, somatic_muscular, somatic_sensory, cardiovascular, respiratory, gastrointestinal, genitourinary, autonomic, behavior_at_interview)

  if (any(is.na(inputs))) {
    stop("Error: All 14 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be integers between 0 and 4.")
  }

  total_score <- sum(inputs)

  severity <- ""
  if (total_score <= 17) {
    severity <- "Mild Severity"
  } else if (total_score <= 24) {
    severity <- "Mild to Moderate Severity"
  } else if (total_score <= 30) {
    severity <- "Moderate to Severe Severity"
  } else {
    severity <- "Very Severe Severity" # Usually score > 30 is severe/very severe
  }

  return(list(
    HAM_A_Score = total_score,
    Severity_Category = severity,
    Inputs = list(
      Q1_Anxious_Mood = anxious_mood,
      Q2_Tension = tension,
      Q3_Fears = fears,
      Q4_Insomnia = insomnia,
      Q5_Intellectual = intellectual,
      Q6_Depressed_Mood = depressed_mood,
      Q7_Somatic_Muscular = somatic_muscular,
      Q8_Somatic_Sensory = somatic_sensory,
      Q9_Cardiovascular = cardiovascular,
      Q10_Respiratory = respiratory,
      Q11_Gastrointestinal = gastrointestinal,
      Q12_Genitourinary = genitourinary,
      Q13_Autonomic = autonomic,
      Q14_Behavior_at_Interview = behavior_at_interview
    )
  ))
}

hamilton_depression_scale <- function(depressed_mood, guilt, suicide, insomnia_early, insomnia_middle, insomnia_late, work_activities, retardation, agitation, anxiety_psychic, anxiety_somatic, somatic_gi, somatic_general, genital_symptoms, hypochondriasis, weight_loss, insight) {

  inputs_0_to_4 <- c(depressed_mood, guilt, suicide, work_activities, retardation, agitation, anxiety_psychic, anxiety_somatic, hypochondriasis)
  inputs_0_to_2 <- c(insomnia_early, insomnia_middle, insomnia_late, somatic_gi, somatic_general, genital_symptoms, weight_loss, insight)

  if (any(is.na(c(inputs_0_to_4, inputs_0_to_2)))) {
    stop("Error: All 17 inputs must be numeric.")
  }

  if (any(inputs_0_to_4 < 0) || any(inputs_0_to_4 > 4)) {
    stop("Error: Items 1, 2, 3, 7, 8, 9, 10, 11, 15 must be scored between 0 and 4.")
  }

  if (any(inputs_0_to_2 < 0) || any(inputs_0_to_2 > 2)) {
    stop("Error: Items 4, 5, 6, 12, 13, 14, 16, 17 must be scored between 0 and 2.")
  }

  total_score <- sum(inputs_0_to_4) + sum(inputs_0_to_2)

  severity <- ""
  if (total_score <= 7) {
    severity <- "Normal"
  } else if (total_score <= 13) {
    severity <- "Mild Depression"
  } else if (total_score <= 18) {
    severity <- "Moderate Depression"
  } else if (total_score <= 22) {
    severity <- "Severe Depression"
  } else {
    severity <- "Very Severe Depression"
  }

  return(list(
    HAM_D_Score = total_score,
    Severity_Category = severity,
    Inputs = list(
      Depressed_Mood = depressed_mood,
      Guilt = guilt,
      Suicide = suicide,
      Insomnia_Early = insomnia_early,
      Insomnia_Middle = insomnia_middle,
      Insomnia_Late = insomnia_late,
      Work_Activities = work_activities,
      Retardation = retardation,
      Agitation = agitation,
      Anxiety_Psychic = anxiety_psychic,
      Anxiety_Somatic = anxiety_somatic,
      Somatic_GI = somatic_gi,
      Somatic_General = somatic_general,
      Genital_Symptoms = genital_symptoms,
      Hypochondriasis = hypochondriasis,
      Weight_Loss = weight_loss,
      Insight = insight
    )
  ))
}

haps_score <- function(rebound_tenderness, hematocrit, creatinine, sex, creatinine_units = "mg/dL") {

  rebound_tenderness <- as.numeric(rebound_tenderness)
  hematocrit <- as.numeric(hematocrit)
  creatinine <- as.numeric(creatinine)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(rebound_tenderness, hematocrit, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  criteria_met <- TRUE
  reasons <- c()

  if (rebound_tenderness == 1) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Rebound tenderness or guarding present")
  }

  if (creat_mg_dl >= 2.0) {
    criteria_met <- FALSE
    reasons <- c(reasons, "Creatinine >= 2.0 mg/dL")
  }

  if (sex == "male") {
    if (hematocrit > 43.0) {
      criteria_met <- FALSE
      reasons <- c(reasons, "Hematocrit > 43% (Male)")
    }
  } else {
    if (hematocrit > 39.6) {
      criteria_met <- FALSE
      reasons <- c(reasons, "Hematocrit > 39.6% (Female)")
    }
  }

  prediction <- ""
  if (criteria_met) {
    prediction <- "Harmless Acute Pancreatitis Likely"
  } else {
    prediction <- "Not Classified as Harmless (Risk of Severe Course)"
  }

  return(list(
    HAPS_Result = prediction,
    Probability_Harmless_Course = if(criteria_met) "98%" else "Variable Risk",
    Criteria_Failed = if(length(reasons) > 0) reasons else "None",
    Inputs = list(
      Rebound = rebound_tenderness == 1,
      Hct = hematocrit,
      Creatinine_mg_dL = round(creat_mg_dl, 2),
      Sex = sex
    )
  ))
}

harvey_bradshaw_index <- function(general_wellbeing, abdominal_pain, liquid_stools_daily, abdominal_mass, complications_count) {

  general_wellbeing <- as.numeric(general_wellbeing)
  abdominal_pain <- as.numeric(abdominal_pain)
  liquid_stools_daily <- as.numeric(liquid_stools_daily)
  abdominal_mass <- as.numeric(abdominal_mass)
  complications_count <- as.numeric(complications_count)

  if (any(is.na(c(general_wellbeing, abdominal_pain, liquid_stools_daily, abdominal_mass, complications_count)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (general_wellbeing < 0 || general_wellbeing > 4) stop("Error: General wellbeing must be 0-4.")
  if (abdominal_pain < 0 || abdominal_pain > 3) stop("Error: Abdominal pain must be 0-3.")
  if (abdominal_mass < 0 || abdominal_mass > 3) stop("Error: Abdominal mass must be 0-3.")

  score <- general_wellbeing + abdominal_pain + liquid_stools_daily + abdominal_mass + complications_count

  disease_activity <- ""
  if (score < 5) {
    disease_activity <- "Remission"
  } else if (score <= 7) {
    disease_activity <- "Mild Activity"
  } else if (score <= 16) {
    disease_activity <- "Moderate Activity"
  } else {
    disease_activity <- "Severe Activity"
  }

  return(list(
    HBI_Score = score,
    Disease_Activity = disease_activity,
    Inputs = list(
      Wellbeing = general_wellbeing,
      Pain = abdominal_pain,
      Stools = liquid_stools_daily,
      Mass = abdominal_mass,
      Complications = complications_count
    )
  ))
}

has_bled_score <- function(hypertension, abnormal_renal_function, abnormal_liver_function, stroke_history, bleeding_history, labile_inr, elderly_gt_65, drug_usage, alcohol_usage) {

  hypertension <- as.numeric(hypertension)
  abnormal_renal_function <- as.numeric(abnormal_renal_function)
  abnormal_liver_function <- as.numeric(abnormal_liver_function)
  stroke_history <- as.numeric(stroke_history)
  bleeding_history <- as.numeric(bleeding_history)
  labile_inr <- as.numeric(labile_inr)
  elderly_gt_65 <- as.numeric(elderly_gt_65)
  drug_usage <- as.numeric(drug_usage)
  alcohol_usage <- as.numeric(alcohol_usage)

  if (any(is.na(c(hypertension, abnormal_renal_function, abnormal_liver_function, stroke_history, bleeding_history, labile_inr, elderly_gt_65, drug_usage, alcohol_usage)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (hypertension == 1) {
    score <- score + 1
  }

  if (abnormal_renal_function == 1) {
    score <- score + 1
  }

  if (abnormal_liver_function == 1) {
    score <- score + 1
  }

  if (stroke_history == 1) {
    score <- score + 1
  }

  if (bleeding_history == 1) {
    score <- score + 1
  }

  if (labile_inr == 1) {
    score <- score + 1
  }

  if (elderly_gt_65 == 1) {
    score <- score + 1
  }

  if (drug_usage == 1) {
    score <- score + 1
  }

  if (alcohol_usage == 1) {
    score <- score + 1
  }

  bleed_risk <- ""
  recommendation <- ""

  if (score == 0) {
    bleed_risk <- "1.13 bleeds per 100 patient-years (Low Risk)"
    recommendation <- "Anticoagulation generally safe."
  } else if (score == 1) {
    bleed_risk <- "1.02 bleeds per 100 patient-years (Low Risk)"
    recommendation <- "Anticoagulation generally safe."
  } else if (score == 2) {
    bleed_risk <- "1.88 bleeds per 100 patient-years (Moderate Risk)"
    recommendation <- "Anticoagulation can be considered; address reversible risk factors."
  } else if (score == 3) {
    bleed_risk <- "3.74 bleeds per 100 patient-years (High Risk)"
    recommendation <- "Caution advised; frequent monitoring and correction of risk factors required."
  } else if (score == 4) {
    bleed_risk <- "8.70 bleeds per 100 patient-years (High Risk)"
    recommendation <- "High risk of bleeding; carefully weigh benefits vs. risks."
  } else {
    bleed_risk <- ">12.5 bleeds per 100 patient-years (Very High Risk)"
    recommendation <- "Very high risk; consider alternatives or extremely close monitoring."
  }

  return(list(
    HAS_BLED_Score = score,
    Annual_Bleeding_Risk = bleed_risk,
    Recommendation = recommendation,
    Inputs = list(
      Hypertension = hypertension == 1,
      Renal_Disease = abnormal_renal_function == 1,
      Liver_Disease = abnormal_liver_function == 1,
      Stroke_History = stroke_history == 1,
      Prior_Bleeding = bleeding_history == 1,
      Labile_INR = labile_inr == 1,
      Elderly_gt_65 = elderly_gt_65 == 1,
      Drugs_NSAIDs_Antiplatelets = drug_usage == 1,
      Alcohol_Use = alcohol_usage == 1
    )
  ))
}

hat_score_tpa_hemorrhage <- function(nihss_score, history_of_diabetes, glucose_mg_dl, history_of_signs_of_early_infarct) {

  nihss_score <- as.numeric(nihss_score)
  history_of_diabetes <- as.numeric(history_of_diabetes)
  glucose_mg_dl <- as.numeric(glucose_mg_dl)
  history_of_signs_of_early_infarct <- as.numeric(history_of_signs_of_early_infarct)

  if (any(is.na(c(nihss_score, history_of_diabetes, glucose_mg_dl, history_of_signs_of_early_infarct)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (nihss_score == 0) {
    score <- score + 0
  } else if (nihss_score >= 15 && nihss_score <= 20) {
    score <- score + 1
  } else if (nihss_score > 20) {
    score <- score + 2
  } else {
    score <- score + 0
  }

  if (history_of_diabetes == 1) {
    score <- score + 1
  }

  if (glucose_mg_dl >= 200) {
    score <- score + 1
  }

  if (history_of_signs_of_early_infarct == 1) {
    score <- score + 1
  }

  risk_category <- ""
  risk_of_sich <- ""
  risk_of_any_ich <- ""
  risk_of_fatal_ich <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    risk_of_sich <- "2%"
    risk_of_any_ich <- "6%"
    risk_of_fatal_ich <- "1%"
  } else if (score == 1) {
    risk_category <- "Moderate Risk"
    risk_of_sich <- "5%"
    risk_of_any_ich <- "10%"
    risk_of_fatal_ich <- "3%"
  } else if (score == 2) {
    risk_category <- "High Risk"
    risk_of_sich <- "10%"
    risk_of_any_ich <- "24%"
    risk_of_fatal_ich <- "6%"
  } else if (score == 3) {
    risk_category <- "High Risk"
    risk_of_sich <- "15%"
    risk_of_any_ich <- "28%"
    risk_of_fatal_ich <- "6%"
  } else if (score == 4) {
    risk_category <- "Very High Risk"
    risk_of_sich <- "44%"
    risk_of_any_ich <- "49%"
    risk_of_fatal_ich <- "17%"
  } else {
    risk_category <- "Very High Risk"
    risk_of_sich <- ">44%"
    risk_of_any_ich <- ">49%"
    risk_of_fatal_ich <- ">17%"
  }

  return(list(
    HAT_Score = score,
    Risk_Category = risk_category,
    Risk_of_Symptomatic_Intracranial_Hemorrhage = risk_of_sich,
    Risk_of_Any_Intracranial_Hemorrhage = risk_of_any_ich,
    Risk_of_Fatal_Hemorrhage = risk_of_fatal_ich,
    Inputs = list(
      NIHSS = nihss_score,
      Diabetes = history_of_diabetes == 1,
      Glucose_gt_200 = glucose_mg_dl >= 200,
      Early_Infarct_Signs = history_of_signs_of_early_infarct == 1
    )
  ))
}

hcm_risk_scd <- function(age, max_wall_thickness_mm, la_diameter_mm, max_lvot_gradient_mmhg, family_history_scd, nsvt, unexplained_syncope) {

  age <- as.numeric(age)
  max_wall_thickness_mm <- as.numeric(max_wall_thickness_mm)
  la_diameter_mm <- as.numeric(la_diameter_mm)
  max_lvot_gradient_mmhg <- as.numeric(max_lvot_gradient_mmhg)
  family_history_scd <- as.numeric(family_history_scd)
  nsvt <- as.numeric(nsvt)
  unexplained_syncope <- as.numeric(unexplained_syncope)

  if (any(is.na(c(age, max_wall_thickness_mm, la_diameter_mm, max_lvot_gradient_mmhg, family_history_scd, nsvt, unexplained_syncope)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (age < 16 || age > 115) {
    stop("Error: Model validated for ages 16-115.")
  }

  if (max_wall_thickness_mm > 35) max_wall_thickness_mm <- 35

  prognostic_index <- (0.15939858 * max_wall_thickness_mm) - (0.00294271 * (max_wall_thickness_mm^2)) + (0.0259082 * la_diameter_mm) + (0.00446131 * max_lvot_gradient_mmhg) + (0.4583082 * family_history_scd) + (0.82639195 * nsvt) + (0.71650361 * unexplained_syncope) - (0.01799934 * age)

  risk_scd_5yr <- 1 - (0.998 ^ exp(prognostic_index))
  risk_percentage <- risk_scd_5yr * 100

  risk_category <- ""
  recommendation <- ""

  if (risk_percentage < 4) {
    risk_category <- "Low Risk (<4%)"
    recommendation <- "ICD generally not indicated."
  } else if (risk_percentage >= 4 && risk_percentage < 6) {
    risk_category <- "Intermediate Risk (4-6%)"
    recommendation <- "ICD may be considered."
  } else {
    risk_category <- "High Risk (>=6%)"
    recommendation <- "ICD should be considered."
  }

  return(list(
    Risk_SCD_5_Year = paste0(round(risk_percentage, 2), "%"),
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Wall_Thickness = max_wall_thickness_mm,
      LA_Diameter = la_diameter_mm,
      Max_Gradient = max_lvot_gradient_mmhg,
      FamHx_SCD = family_history_scd == 1,
      NSVT = nsvt == 1,
      Syncope = unexplained_syncope == 1
    )
  ))
}

heads_ed_score <- function(home, education, activities_peers, drugs_alcohol, suicidality, emotions_behavior, discharge_resources) {

  inputs <- c(home, education, activities_peers, drugs_alcohol, suicidality, emotions_behavior, discharge_resources)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 2)) {
    stop("Error: All item scores must be 0, 1, or 2.")
  }

  total_score <- sum(inputs)

  action <- ""
  if (total_score >= 8 || suicidality == 2) {
    action <- "Consult Crisis/Psychiatry Team. High likelihood of requiring admission or intensive rapid response."
  } else {
    action <- "Discharge likely appropriate with community referrals/follow-up."
  }

  return(list(
    HEADS_ED_Score = total_score,
    Clinical_Action = action,
    Inputs = list(
      Home = home,
      Education = education,
      Activities = activities_peers,
      Drugs = drugs_alcohol,
      Suicidality = suicidality,
      Emotions = emotions_behavior,
      Discharge_Supports = discharge_resources
    )
  ))
}

heart_pathway_early_discharge <- function(history_score, ecg_score, age, risk_factors_score, initial_troponin_score, troponin_3h_negative) {

  history_score <- as.numeric(history_score)
  ecg_score <- as.numeric(ecg_score)
  age <- as.numeric(age)
  risk_factors_score <- as.numeric(risk_factors_score)
  initial_troponin_score <- as.numeric(initial_troponin_score)
  troponin_3h_negative <- as.numeric(troponin_3h_negative)

  if (any(is.na(c(history_score, ecg_score, age, risk_factors_score, initial_troponin_score, troponin_3h_negative)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!initial_troponin_score %in% c(0, 1, 2)) stop("Error: Initial Troponin Score must be 0, 1, or 2.")

  age_score <- 0
  if (age < 45) {
    age_score <- 0
  } else if (age >= 45 && age <= 64) {
    age_score <- 1
  } else {
    age_score <- 2
  }

  heart_score <- history_score + ecg_score + age_score + risk_factors_score + initial_troponin_score

  pathway_result <- ""
  recommendation <- ""
  mace_risk <- ""

  if (heart_score <= 3 && initial_troponin_score == 0 && troponin_3h_negative == 1) {
    pathway_result <- "Low Risk (Candidate for Early Discharge)"
    recommendation <- "Discharge home with outpatient follow-up (e.g., stress test or cardiology visit) is reasonable."
    mace_risk <- "< 1%"
  } else {
    pathway_result <- "Not Low Risk"
    recommendation <- "Admission for observation and further cardiac testing is recommended."
    mace_risk <- ">= 1% (Risk increases with score)"
  }

  return(list(
    HEART_Score = heart_score,
    Pathway_Result = pathway_result,
    Recommendation = recommendation,
    MACE_Risk_30_Day = mace_risk,
    Inputs = list(
      History = history_score,
      ECG = ecg_score,
      Age = age,
      Risk_Factors_Pts = risk_factors_score,
      Initial_Troponin_Pts = initial_troponin_score,
      Troponin_3h_Neg = troponin_3h_negative == 1
    )
  ))
}

heart_score_mace <- function(history_suspicion, ecg_findings, age, risk_factors_count, history_atherosclerosis, troponin_limit_multiplier) {

  history_suspicion <- as.numeric(history_suspicion)
  ecg_findings <- as.numeric(ecg_findings)
  age <- as.numeric(age)
  risk_factors_count <- as.numeric(risk_factors_count)
  history_atherosclerosis <- as.numeric(history_atherosclerosis)
  troponin_limit_multiplier <- as.numeric(troponin_limit_multiplier)

  if (any(is.na(c(history_suspicion, ecg_findings, age, risk_factors_count, history_atherosclerosis, troponin_limit_multiplier)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!history_suspicion %in% c(0, 1, 2)) stop("Error: History suspicion must be 0 (Slight), 1 (Moderate), or 2 (High).")
  if (!ecg_findings %in% c(0, 1, 2)) stop("Error: ECG findings must be 0 (Normal), 1 (Non-specific), or 2 (ST deviation).")

  score <- 0

  score <- score + history_suspicion

  score <- score + ecg_findings

  if (age < 45) {
    score <- score + 0
  } else if (age >= 45 && age <= 64) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (risk_factors_count >= 3 || history_atherosclerosis == 1) {
    score <- score + 2
  } else if (risk_factors_count >= 1) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (troponin_limit_multiplier > 2) { # >2x normal
    score <- score + 2
  } else if (troponin_limit_multiplier > 1) { # 1-2x normal
    score <- score + 1
  } else {
    score <- score + 0
  }

  risk_group <- ""
  mace_risk_6_weeks <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_group <- "Low Risk"
    mace_risk_6_weeks <- "0.9 - 1.7%"
    recommendation <- "Discharge is often appropriate."
  } else if (score <= 6) {
    risk_group <- "Moderate Risk"
    mace_risk_6_weeks <- "12 - 16.6%"
    recommendation <- "Observation and clinical evaluation required."
  } else {
    risk_group <- "High Risk"
    mace_risk_6_weeks <- "50 - 65%"
    recommendation <- "Early invasive strategies recommended."
  }

  return(list(
    HEART_Score = score,
    Risk_Category = risk_group,
    MACE_Risk_6_Weeks = mace_risk_6_weeks,
    Recommendation = recommendation,
    Inputs = list(
      History_Pts = history_suspicion,
      ECG_Pts = ecg_findings,
      Age = age,
      Risk_Factors_Count = risk_factors_count,
      Hx_Atherosclerosis = history_atherosclerosis == 1,
      Troponin_Multiplier = troponin_limit_multiplier
    )
  ))
}

hct_ci_score <- function(arrhythmia, cardiac_comorbidity, ibd, diabetes, cerebrovascular, psychiatric, hepatic_mild, hepatic_severe, obesity, infection, rheumatologic, peptic_ulcer, renal_moderate_severe, pulmonary_moderate, pulmonary_severe, prior_solid_tumor, heart_valve_disease) {

  arrhythmia <- as.numeric(arrhythmia)
  cardiac_comorbidity <- as.numeric(cardiac_comorbidity)
  ibd <- as.numeric(ibd)
  diabetes <- as.numeric(diabetes)
  cerebrovascular <- as.numeric(cerebrovascular)
  psychiatric <- as.numeric(psychiatric)
  hepatic_mild <- as.numeric(hepatic_mild)
  hepatic_severe <- as.numeric(hepatic_severe)
  obesity <- as.numeric(obesity)
  infection <- as.numeric(infection)
  rheumatologic <- as.numeric(rheumatologic)
  peptic_ulcer <- as.numeric(peptic_ulcer)
  renal_moderate_severe <- as.numeric(renal_moderate_severe)
  pulmonary_moderate <- as.numeric(pulmonary_moderate)
  pulmonary_severe <- as.numeric(pulmonary_severe)
  prior_solid_tumor <- as.numeric(prior_solid_tumor)
  heart_valve_disease <- as.numeric(heart_valve_disease)

  if (any(is.na(c(arrhythmia, cardiac_comorbidity, ibd, diabetes, cerebrovascular, psychiatric, hepatic_mild, hepatic_severe, obesity, infection, rheumatologic, peptic_ulcer, renal_moderate_severe, pulmonary_moderate, pulmonary_severe, prior_solid_tumor, heart_valve_disease)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (arrhythmia == 1) score <- score + 1
  if (cardiac_comorbidity == 1) score <- score + 1
  if (ibd == 1) score <- score + 1
  if (diabetes == 1) score <- score + 1
  if (cerebrovascular == 1) score <- score + 1
  if (psychiatric == 1) score <- score + 1
  if (obesity == 1) score <- score + 1
  if (infection == 1) score <- score + 1

  if (hepatic_severe == 1) {
    score <- score + 3
  } else if (hepatic_mild == 1) {
    score <- score + 1
  }

  if (rheumatologic == 1) score <- score + 2
  if (peptic_ulcer == 1) score <- score + 2
  if (renal_moderate_severe == 1) score <- score + 2

  if (pulmonary_severe == 1) {
    score <- score + 3
  } else if (pulmonary_moderate == 1) {
    score <- score + 2
  }

  if (prior_solid_tumor == 1) score <- score + 3
  if (heart_valve_disease == 1) score <- score + 3

  risk_group <- ""
  nrm_2yr <- ""
  os_2yr <- ""

  if (score == 0) {
    risk_group <- "Low Risk"
    nrm_2yr <- "14%"
    os_2yr <- "71%"
  } else if (score <= 2) {
    risk_group <- "Intermediate Risk"
    nrm_2yr <- "21%"
    os_2yr <- "60%"
  } else {
    risk_group <- "High Risk"
    nrm_2yr <- "41%"
    os_2yr <- "34%"
  }

  return(list(
    HCT_CI_Score = score,
    Risk_Group = risk_group,
    Est_2Yr_Non_Relapse_Mortality = nrm_2yr,
    Est_2Yr_Overall_Survival = os_2yr,
    Inputs = list(
      Arrhythmia = arrhythmia == 1,
      Cardiac = cardiac_comorbidity == 1,
      IBD = ibd == 1,
      Diabetes = diabetes == 1,
      Cerebrovascular = cerebrovascular == 1,
      Psychiatric = psychiatric == 1,
      Hepatic = if(hepatic_severe==1) "Severe (3)" else if(hepatic_mild==1) "Mild (1)" else "None",
      Obesity = obesity == 1,
      Infection = infection == 1,
      Rheumatologic = rheumatologic == 1,
      Peptic_Ulcer = peptic_ulcer == 1,
      Renal_Mod_Severe = renal_moderate_severe == 1,
      Pulmonary = if(pulmonary_severe==1) "Severe (3)" else if(pulmonary_moderate==1) "Moderate (2)" else "None",
      Solid_Tumor = prior_solid_tumor == 1,
      Heart_Valve = heart_valve_disease == 1
    )
  ))
}

hemorr2hages_score <- function(hepatic_renal_disease, ethanol_abuse, malignancy, age_gt_75, reduced_platelets_function, prior_bleed, hypertension, anemia, genetic_factors, fall_risk, stroke_history) {

  hepatic_renal_disease <- as.numeric(hepatic_renal_disease)
  ethanol_abuse <- as.numeric(ethanol_abuse)
  malignancy <- as.numeric(malignancy)
  age_gt_75 <- as.numeric(age_gt_75)
  reduced_platelets_function <- as.numeric(reduced_platelets_function)
  prior_bleed <- as.numeric(prior_bleed)
  hypertension <- as.numeric(hypertension)
  anemia <- as.numeric(anemia)
  genetic_factors <- as.numeric(genetic_factors)
  fall_risk <- as.numeric(fall_risk)
  stroke_history <- as.numeric(stroke_history)

  if (any(is.na(c(hepatic_renal_disease, ethanol_abuse, malignancy, age_gt_75, reduced_platelets_function, prior_bleed, hypertension, anemia, genetic_factors, fall_risk, stroke_history)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (hepatic_renal_disease == 1) {
    score <- score + 1
  }

  if (ethanol_abuse == 1) {
    score <- score + 1
  }

  if (malignancy == 1) {
    score <- score + 1
  }

  if (age_gt_75 == 1) {
    score <- score + 1
  }

  if (reduced_platelets_function == 1) {
    score <- score + 1
  }

  if (prior_bleed == 1) {
    score <- score + 2
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (anemia == 1) {
    score <- score + 1
  }

  if (genetic_factors == 1) {
    score <- score + 1
  }

  if (fall_risk == 1) {
    score <- score + 1
  }

  if (stroke_history == 1) {
    score <- score + 1
  }

  bleed_risk_per_100_years <- ""
  risk_category <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    bleed_risk_per_100_years <- "1.9%"
  } else if (score == 1) {
    risk_category <- "Intermediate Risk"
    bleed_risk_per_100_years <- "2.5%"
  } else if (score == 2) {
    risk_category <- "High Risk"
    bleed_risk_per_100_years <- "5.3%"
  } else if (score == 3) {
    risk_category <- "High Risk"
    bleed_risk_per_100_years <- "8.4%"
  } else if (score == 4) {
    risk_category <- "High Risk"
    bleed_risk_per_100_years <- "10.4%"
  } else {
    risk_category <- "High Risk"
    bleed_risk_per_100_years <- ">= 12.3%"
  }

  return(list(
    HEMORR2HAGES_Score = score,
    Risk_Category = risk_category,
    Bleeding_Risk_per_100_Pt_Years = bleed_risk_per_100_years,
    Inputs = list(
      Hepatic_Renal = hepatic_renal_disease == 1,
      Ethanol = ethanol_abuse == 1,
      Malignancy = malignancy == 1,
      Age_gt_75 = age_gt_75 == 1,
      Platelets_Function = reduced_platelets_function == 1,
      Rebleeding_Risk = prior_bleed == 1,
      Hypertension = hypertension == 1,
      Anemia = anemia == 1,
      Genetic_Factors = genetic_factors == 1,
      Fall_Risk = fall_risk == 1,
      Stroke = stroke_history == 1
    )
  ))
}

herdoo2_rule <- function(post_thrombotic_signs, d_dimer_ge_250, bmi_ge_30, age_ge_65) {

  post_thrombotic_signs <- as.numeric(post_thrombotic_signs)
  d_dimer_ge_250 <- as.numeric(d_dimer_ge_250)
  bmi_ge_30 <- as.numeric(bmi_ge_30)
  age_ge_65 <- as.numeric(age_ge_65)

  if (any(is.na(c(post_thrombotic_signs, d_dimer_ge_250, bmi_ge_30, age_ge_65)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (post_thrombotic_signs == 1) {
    score <- score + 1
  }

  if (d_dimer_ge_250 == 1) {
    score <- score + 1
  }

  if (bmi_ge_30 == 1) {
    score <- score + 1
  }

  if (age_ge_65 == 1) {
    score <- score + 1
  }

  risk_category <- ""
  recurrence_risk <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_category <- "Low Risk"
    recurrence_risk <- "3.0% per patient year"
    recommendation <- "Consider discontinuing oral anticoagulation (if patient is female)."
  } else {
    risk_category <- "High Risk"
    recurrence_risk <- "7.4% per patient year"
    recommendation <- "Continue oral anticoagulation."
  }

  return(list(
    HERDOO2_Score = score,
    Risk_Category = risk_category,
    VTE_Recurrence_Risk = recurrence_risk,
    Recommendation = recommendation,
    Note = "Valid only for women with unprovoked VTE. Men are high risk regardless of score.",
    Inputs = list(
      Post_Thrombotic_Signs = post_thrombotic_signs == 1,
      D_Dimer_ge_250 = d_dimer_ge_250 == 1,
      BMI_ge_30 = bmi_ge_30 == 1,
      Age_ge_65 = age_ge_65 == 1
    )
  ))
}

hestia_criteria <- function(hemodynamically_unstable, thrombolysis_needed, high_risk_bleeding, oxygen_needed_gt_24h, pe_on_anticoagulation, iv_pain_meds_needed, medical_social_reason, renal_failure, liver_failure, pregnancy, heparin_hit_history) {

  hemodynamically_unstable <- as.numeric(hemodynamically_unstable)
  thrombolysis_needed <- as.numeric(thrombolysis_needed)
  high_risk_bleeding <- as.numeric(high_risk_bleeding)
  oxygen_needed_gt_24h <- as.numeric(oxygen_needed_gt_24h)
  pe_on_anticoagulation <- as.numeric(pe_on_anticoagulation)
  iv_pain_meds_needed <- as.numeric(iv_pain_meds_needed)
  medical_social_reason <- as.numeric(medical_social_reason)
  renal_failure <- as.numeric(renal_failure)
  liver_failure <- as.numeric(liver_failure)
  pregnancy <- as.numeric(pregnancy)
  heparin_hit_history <- as.numeric(heparin_hit_history)

  if (any(is.na(c(hemodynamically_unstable, thrombolysis_needed, high_risk_bleeding, oxygen_needed_gt_24h, pe_on_anticoagulation, iv_pain_meds_needed, medical_social_reason, renal_failure, liver_failure, pregnancy, heparin_hit_history)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  criteria_count <- 0
  criteria_failed <- c()

  if (hemodynamically_unstable == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Hemodynamically Unstable")
  }

  if (thrombolysis_needed == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Thrombolysis/Embolectomy Needed")
  }

  if (high_risk_bleeding == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "High Bleeding Risk")
  }

  if (oxygen_needed_gt_24h == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Oxygen requirement >24h")
  }

  if (pe_on_anticoagulation == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "PE diagnosis while on anticoagulation")
  }

  if (iv_pain_meds_needed == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Severe pain requiring IV meds >24h")
  }

  if (medical_social_reason == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Medical/Social reason for admission")
  }

  if (renal_failure == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Creatinine clearance <30 mL/min")
  }

  if (liver_failure == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Severe Liver Impairment")
  }

  if (pregnancy == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "Pregnancy")
  }

  if (heparin_hit_history == 1) {
    criteria_count <- criteria_count + 1
    criteria_failed <- c(criteria_failed, "History of HIT")
  }

  result <- ""
  if (criteria_count == 0) {
    result <- "Low Risk (Eligible for Outpatient Treatment)"
  } else {
    result <- "Not Eligible for Outpatient Treatment (High Risk)"
  }

  return(list(
    Hestia_Result = result,
    Criteria_Met_Count = criteria_count,
    Criteria_List = if(criteria_count > 0) criteria_failed else "None",
    Inputs = list(
      Unstable = hemodynamically_unstable == 1,
      Thrombolysis = thrombolysis_needed == 1,
      Bleeding_Risk = high_risk_bleeding == 1,
      Oxygen = oxygen_needed_gt_24h == 1,
      On_Anticoagulation = pe_on_anticoagulation == 1,
      Pain_IV = iv_pain_meds_needed == 1,
      Social_Medical = medical_social_reason == 1,
      Renal = renal_failure == 1,
      Liver = liver_failure == 1,
      Pregnant = pregnancy == 1,
      HIT_Hx = heparin_hit_history == 1
    )
  ))
}

hiet_dosing <- function(weight_kg, serum_glucose_initial, insulin_bolus_dose_u_kg = 1, insulin_drip_rate_u_kg_hr = 0.5, dextrose_bolus_g_kg = 0.5, dextrose_drip_g_kg_hr = 0.5) {

  weight_kg <- as.numeric(weight_kg)
  serum_glucose_initial <- as.numeric(serum_glucose_initial)
  insulin_bolus_dose_u_kg <- as.numeric(insulin_bolus_dose_u_kg)
  insulin_drip_rate_u_kg_hr <- as.numeric(insulin_drip_rate_u_kg_hr)
  dextrose_bolus_g_kg <- as.numeric(dextrose_bolus_g_kg)
  dextrose_drip_g_kg_hr <- as.numeric(dextrose_drip_g_kg_hr)

  if (any(is.na(c(weight_kg, serum_glucose_initial, insulin_bolus_dose_u_kg, insulin_drip_rate_u_kg_hr, dextrose_bolus_g_kg, dextrose_drip_g_kg_hr)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  insulin_bolus <- weight_kg * insulin_bolus_dose_u_kg
  insulin_drip <- weight_kg * insulin_drip_rate_u_kg_hr

  dextrose_bolus_g <- 0
  if (serum_glucose_initial < 200) {
    dextrose_bolus_g <- weight_kg * dextrose_bolus_g_kg
  }

  dextrose_drip_g_hr <- weight_kg * dextrose_drip_g_kg_hr

  return(list(
    Insulin_Bolus_Units = round(insulin_bolus, 1),
    Insulin_Drip_Units_hr = round(insulin_drip, 1),
    Dextrose_Bolus_Grams = round(dextrose_bolus_g, 1),
    Dextrose_Maintenance_Grams_hr = round(dextrose_drip_g_hr, 1),
    Monitoring_Notes = "Monitor Glucose q15-30min initially. Monitor K+ q30min (target 2.8-3.2 mEq/L). Monitor Mg and Phos.",
    Inputs = list(
      Weight_kg = weight_kg,
      Initial_Glucose = serum_glucose_initial
    )
  ))
}

hints_exam_score <- function(head_impulse_normal, nystagmus_direction_changing, skew_deviation_present) {

  head_impulse_normal <- as.numeric(head_impulse_normal)
  nystagmus_direction_changing <- as.numeric(nystagmus_direction_changing)
  skew_deviation_present <- as.numeric(skew_deviation_present)

  if (any(is.na(c(head_impulse_normal, nystagmus_direction_changing, skew_deviation_present)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  central_signs_count <- 0
  central_features <- c()

  if (head_impulse_normal == 1) {
    central_signs_count <- central_signs_count + 1
    central_features <- c(central_features, "Normal Head Impulse (No saccade)")
  }

  if (nystagmus_direction_changing == 1) {
    central_signs_count <- central_signs_count + 1
    central_features <- c(central_features, "Direction-changing Nystagmus")
  }

  if (skew_deviation_present == 1) {
    central_signs_count <- central_signs_count + 1
    central_features <- c(central_features, "Skew Deviation (Vertical ocular misalignment)")
  }

  interpretation <- ""
  recommendation <- ""

  if (central_signs_count > 0) {
    interpretation <- "Central Etiology Likely (e.g., Stroke)"
    recommendation <- "Urgent neuroimaging (MRI) and stroke workup indicated. HINTS is more sensitive than early MRI for posterior circulation stroke."
  } else {
    interpretation <- "Peripheral Etiology Likely (e.g., Vestibular Neuritis)"
    recommendation <- "Consider symptomatic management for peripheral vertigo. Ensure no hearing loss (HINTS+)."
  }

  return(list(
    HINTS_Result = interpretation,
    Recommendation = recommendation,
    Central_Features_Present = if(length(central_features) > 0) central_features else "None",
    Inputs = list(
      Head_Impulse = if(head_impulse_normal==1) "Normal (Worrisome)" else "Abnormal (Reassuring)",
      Nystagmus = if(nystagmus_direction_changing==1) "Direction-Changing (Worrisome)" else "Direction-Fixed (Reassuring)",
      Skew = if(skew_deviation_present==1) "Present (Worrisome)" else "Absent (Reassuring)"
    )
  ))
}

he_macs_score <- function(age, sex, smoker, systolic_bp_lt_100, ecg_ischemia, sweating_observed, vomiting, pain_radiation_right_arm) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  smoker <- as.numeric(smoker)
  systolic_bp_lt_100 <- as.numeric(systolic_bp_lt_100)
  ecg_ischemia <- as.numeric(ecg_ischemia)
  sweating_observed <- as.numeric(sweating_observed)
  vomiting <- as.numeric(vomiting)
  pain_radiation_right_arm <- as.numeric(pain_radiation_right_arm)

  if (any(is.na(c(age, smoker, systolic_bp_lt_100, ecg_ischemia, sweating_observed, vomiting, pain_radiation_right_arm)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  high_risk_features <- 0
  if (ecg_ischemia == 1) high_risk_features <- high_risk_features + 1
  if (sweating_observed == 1) high_risk_features <- high_risk_features + 1
  if (vomiting == 1) high_risk_features <- high_risk_features + 1
  if (pain_radiation_right_arm == 1) high_risk_features <- high_risk_features + 1
  if (systolic_bp_lt_100 == 1) high_risk_features <- high_risk_features + 1
  if (smoker == 1) high_risk_features <- high_risk_features + 1
  if (sex == "male") high_risk_features <- high_risk_features + 1

  interpretation <- "The HE-MACS model calculates a specific probability of Acute Coronary Syndrome (ACS) to stratify patients."
  recommendation <- "If probability is 'Very Low' (<4-5%), guidelines suggest consideration for discharge. Otherwise, further observation/troponin testing is indicated."

  return(list(
    Risk_Model = "HE-MACS (History and ECG-only Manchester Acute Coronary Syndromes)",
    Interpretation = interpretation,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      Smoker = smoker == 1,
      SBP_lt_100 = systolic_bp_lt_100 == 1,
      ECG_Ischemia = ecg_ischemia == 1,
      Sweating = sweating_observed == 1,
      Vomiting = vomiting == 1,
      Right_Arm_Radiation = pain_radiation_right_arm == 1
    )
  ))
}

hep_hit_score <- function(platelet_fall_percent, timing_code, nadir_platelet_count, thrombosis_code, skin_necrosis, acute_systemic_reaction, bleeding, other_causes_present) {

  platelet_fall_percent <- as.numeric(platelet_fall_percent)
  timing_code <- as.numeric(timing_code)
  nadir_platelet_count <- as.numeric(nadir_platelet_count)
  thrombosis_code <- as.numeric(thrombosis_code)
  skin_necrosis <- as.numeric(skin_necrosis)
  acute_systemic_reaction <- as.numeric(acute_systemic_reaction)
  bleeding <- as.numeric(bleeding)
  other_causes_present <- as.numeric(other_causes_present)

  if (any(is.na(c(platelet_fall_percent, timing_code, nadir_platelet_count, thrombosis_code, skin_necrosis, acute_systemic_reaction, bleeding, other_causes_present)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (platelet_fall_percent < 30) {
    score <- score - 1
  } else if (platelet_fall_percent <= 50) {
    score <- score + 1
  } else {
    score <- score + 3
  }

  if (timing_code == 0) score <- score - 2
  else if (timing_code == 1) score <- score + 2
  else if (timing_code == 2) score <- score + 3
  else if (timing_code == 3) score <- score + 2
  else if (timing_code == 4) score <- score - 1
  else if (timing_code == 5) score <- score + 2
  else if (timing_code == 6) score <- score - 1

  if (nadir_platelet_count <= 20) {
    score <- score - 2
  } else {
    score <- score + 2
  }

  if (thrombosis_code == 1) {
    score <- score + 3
  } else if (thrombosis_code == 2) {
    score <- score + 2
  }

  if (skin_necrosis == 1) {
    score <- score + 3
  }

  if (acute_systemic_reaction == 1) {
    score <- score + 2
  }

  if (bleeding == 1) {
    score <- score - 1
  }

  if (other_causes_present == 0) {
    score <- score + 3
  } else {
    score <- score - 2
  }

  probability <- ""
  if (score >= 2) {
    probability <- "High Probability (100% Sensitivity)"
  } else {
    probability <- "Low Probability"
  }

  return(list(
    HEP_Score = score,
    Probability = probability,
    Inputs = list(
      Fall_Percent = platelet_fall_percent,
      Timing_Code = timing_code,
      Nadir_Count = nadir_platelet_count,
      Thrombosis_Code = thrombosis_code,
      Skin_Necrosis = skin_necrosis == 1,
      Systemic_Reaction = acute_systemic_reaction == 1,
      Bleeding = bleeding == 1,
      Other_Causes = other_causes_present == 1
    )
  ))
}

hiri_msm_score <- function(age, partners_6mo, rai_episodes_6mo, iai_episodes_6mo, meth_use, poppers_use) {

  age <- as.numeric(age)
  partners_6mo <- as.numeric(partners_6mo)
  rai_episodes_6mo <- as.numeric(rai_episodes_6mo)
  iai_episodes_6mo <- as.numeric(iai_episodes_6mo)
  meth_use <- as.numeric(meth_use)
  poppers_use <- as.numeric(poppers_use)

  if (any(is.na(c(age, partners_6mo, rai_episodes_6mo, iai_episodes_6mo, meth_use, poppers_use)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 18 && age <= 28) {
    score <- score + 8
  } else if (age >= 29 && age <= 40) {
    score <- score + 5
  } else if (age >= 41 && age <= 48) {
    score <- score + 2
  }

  if (partners_6mo <= 5) {
    score <- score + 0
  } else if (partners_6mo >= 6 && partners_6mo <= 10) {
    score <- score + 4
  } else if (partners_6mo >= 11 && partners_6mo <= 20) {
    score <- score + 7
  } else {
    score <- score + 10
  }

  if (rai_episodes_6mo == 0) {
    score <- score + 0
  } else if (rai_episodes_6mo >= 1 && rai_episodes_6mo <= 5) {
    score <- score + 3
  } else if (rai_episodes_6mo >= 6 && rai_episodes_6mo <= 10) {
    score <- score + 5
  } else if (rai_episodes_6mo >= 11 && rai_episodes_6mo <= 20) {
    score <- score + 8
  } else {
    score <- score + 10
  }

  if (iai_episodes_6mo == 0) {
    score <- score + 0
  } else if (iai_episodes_6mo >= 1 && iai_episodes_6mo <= 5) {
    score <- score + 3
  } else if (iai_episodes_6mo >= 6 && iai_episodes_6mo <= 10) {
    score <- score + 4
  } else if (iai_episodes_6mo >= 11 && iai_episodes_6mo <= 20) {
    score <- score + 5
  } else {
    score <- score + 6
  }

  if (meth_use == 1) {
    score <- score + 5
  }

  if (poppers_use == 1) {
    score <- score + 3
  }

  risk_category <- ""
  recommendation <- ""

  if (score >= 10) {
    risk_category <- "High Risk"
    recommendation <- "Patient may benefit from PrEP (Pre-Exposure Prophylaxis)."
  } else {
    risk_category <- "Low Risk"
    recommendation <- "PrEP may not be indicated based solely on this index."
  }

  return(list(
    HIRI_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Partners_6mo = partners_6mo,
      RAI_6mo = rai_episodes_6mo,
      IAI_6mo = iai_episodes_6mo,
      Meth_Use = meth_use == 1,
      Poppers_Use = poppers_use == 1
    )
  ))
}

rasp_hiv_risk <- function(source_status, exposure_type, deep_injury_or_visible_blood, source_viral_load_high) {

  source_status <- tolower(as.character(source_status))
  exposure_type <- tolower(as.character(exposure_type))
  deep_injury_or_visible_blood <- as.numeric(deep_injury_or_visible_blood)
  source_viral_load_high <- as.numeric(source_viral_load_high)

  if (!source_status %in% c("positive", "negative", "unknown")) {
    stop("Error: Source status must be 'positive', 'negative', or 'unknown'.")
  }

  if (!exposure_type %in% c("percutaneous", "mucous_membrane", "non_intact_skin")) {
    stop("Error: Exposure type must be 'percutaneous', 'mucous_membrane', or 'non_intact_skin'.")
  }

  transmission_risk <- ""
  recommendation <- ""

  if (source_status == "negative") {
    transmission_risk <- "Negligible (0%)"
    recommendation <- "No PEP indicated."
  } else if (source_status == "unknown") {
    transmission_risk <- "Unknown (Depends on local prevalence)"
    recommendation <- "Consider PEP if high-risk setting or source likely HIV positive. Consult expert."
  } else {
    if (exposure_type == "percutaneous") {
      base_risk <- 0.3
      if (deep_injury_or_visible_blood == 1 || source_viral_load_high == 1) {
        transmission_risk <- "> 0.3% (1 in 300)"
        recommendation <- "PEP Recommended (3-drug regimen)."
      } else {
        transmission_risk <- "Approx. 0.3% (1 in 333)"
        recommendation <- "PEP Recommended (3-drug regimen)."
      }
    } else if (exposure_type == "mucous_membrane") {
      transmission_risk <- "Approx. 0.09% (1 in 1000)"
      recommendation <- "PEP Usually Recommended (Consider risks/benefits)."
    } else {
      transmission_risk <- "< 0.09% (Low Risk)"
      recommendation <- "PEP Generally Not Recommended (Unless skin integrity significantly compromised with high viral load blood)."
    }
  }

  return(list(
    Estimated_Transmission_Risk = transmission_risk,
    PEP_Recommendation = recommendation,
    Inputs = list(
      Source = source_status,
      Exposure = exposure_type,
      Severe_Injury = deep_injury_or_visible_blood == 1,
      High_Viral_Load = source_viral_load_high == 1
    )
  ))
}

ho_trauma_index <- function(inr, lactate_mmol_l) {

  inr <- as.numeric(inr)
  lactate_mmol_l <- as.numeric(lactate_mmol_l)

  if (any(is.na(c(inr, lactate_mmol_l)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (inr < 0 || lactate_mmol_l < 0) {
    stop("Error: Inputs must be non-negative.")
  }

  ho_score <- inr + (lactate_mmol_l / 2)

  mortality_risk <- ""

  if (ho_score < 1.5) {
    mortality_risk <- "Low Risk"
  } else if (ho_score >= 1.5 && ho_score < 3.0) {
    mortality_risk <- "Intermediate Risk"
  } else {
    mortality_risk <- "High Risk"
  }

  return(list(
    HO_Score = round(ho_score, 2),
    Risk_Category = mortality_risk,
    Inputs = list(
      INR = inr,
      Lactate = lactate_mmol_l
    )
  ))
}

horowitz_index_pf_ratio <- function(pao2_mmhg, fio2_percent) {

  pao2_mmhg <- as.numeric(pao2_mmhg)
  fio2_percent <- as.numeric(fio2_percent)

  if (any(is.na(c(pao2_mmhg, fio2_percent)))) {
    stop("Error: PaO2 and FiO2 must be numeric.")
  }

  if (fio2_percent > 100 || fio2_percent < 21) {
    warning("FiO2 input typically ranges from 21 to 100. Ensure input is percentage.")
  }

  fio2_decimal <- fio2_percent / 100

  pf_ratio <- pao2_mmhg / fio2_decimal

  classification <- ""
  if (pf_ratio > 300) {
    classification <- "Normal (> 300)"
  } else if (pf_ratio > 200) {
    classification <- "Mild ARDS (200-300)"
  } else if (pf_ratio > 100) {
    classification <- "Moderate ARDS (100-200)"
  } else {
    classification <- "Severe ARDS (<= 100)"
  }

  return(list(
    Horowitz_Index = round(pf_ratio, 1),
    ARDS_Severity = classification,
    Inputs = list(
      PaO2 = pao2_mmhg,
      FiO2 = fio2_decimal
    )
  ))
}

hospital_readmission_score <- function(hemoglobin_level, oncology_service, sodium_level, procedure_during_admission, admission_type_index, length_of_stay, previous_admissions) {

  hemoglobin_level <- as.numeric(hemoglobin_level)
  oncology_service <- as.numeric(oncology_service)
  sodium_level <- as.numeric(sodium_level)
  procedure_during_admission <- as.numeric(procedure_during_admission)
  admission_type_index <- as.numeric(admission_type_index)
  length_of_stay <- as.numeric(length_of_stay)
  previous_admissions <- as.numeric(previous_admissions)

  if (any(is.na(c(hemoglobin_level, oncology_service, sodium_level, procedure_during_admission, admission_type_index, length_of_stay, previous_admissions)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (hemoglobin_level < 12) {
    score <- score + 1
  }

  if (oncology_service == 1) {
    score <- score + 2
  }

  if (sodium_level < 135) {
    score <- score + 1
  }

  if (procedure_during_admission == 0) {
    score <- score + 1
  }

  if (admission_type_index == 1) {
    score <- score + 1
  }

  if (length_of_stay >= 5) {
    score <- score + 2
  }

  if (previous_admissions > 1) {
    score <- score + 5
  } else if (previous_admissions == 1) {
    score <- score + 2
  }

  risk_category <- ""
  prob_readmission <- ""

  if (score <= 4) {
    risk_category <- "Low Risk"
    prob_readmission <- "5%"
  } else if (score <= 6) {
    risk_category <- "Intermediate Risk"
    prob_readmission <- "10%"
  } else {
    risk_category <- "High Risk"
    prob_readmission <- ">15%"
  }

  return(list(
    HOSPITAL_Score = score,
    Risk_Level = risk_category,
    Est_Readmission_Risk_30_Day = prob_readmission,
    Inputs = list(
      Hgb_lt_12 = hemoglobin_level < 12,
      Onc_Service = oncology_service == 1,
      Na_lt_135 = sodium_level < 135,
      No_Procedure = procedure_during_admission == 0,
      Admission_Index_Urgent = admission_type_index == 1,
      LOS_ge_5 = length_of_stay >= 5,
      Prev_Admissions_Count = previous_admissions
    )
  ))
}

bhutani_nomogram_risk <- function(age_hours, total_bilirubin, units = "mg/dL") {

  age_hours <- as.numeric(age_hours)
  total_bilirubin <- as.numeric(total_bilirubin)

  if (any(is.na(c(age_hours, total_bilirubin)))) {
    stop("Error: Inputs must be numeric.")
  }

  if (age_hours < 12) {
    return(list(Error = "Nomogram valid for infants >= 12 hours of age."))
  }

  tsb_mg_dl <- total_bilirubin
  if (units == "umol/L" || units == "micromol/L") {
    tsb_mg_dl <- total_bilirubin / 17.1
  }

  ref_hours <- c(12, 16, 20, 24, 28, 32, 36, 40, 44, 48, 52, 56, 60, 64, 68, 72, 76, 80, 84, 88, 92, 96, 108, 120)

  p40_vals <- c(1.9, 2.6, 3.2, 3.9, 4.5, 5.0, 5.6, 6.1, 6.6, 7.0, 7.4, 7.8, 8.3, 8.6, 9.0, 9.4, 9.7, 9.9, 10.2, 10.5, 10.8, 11.0, 11.7, 12.2)
  p75_vals <- c(3.0, 3.8, 4.6, 5.4, 6.1, 6.8, 7.5, 8.2, 8.8, 9.3, 9.9, 10.5, 11.0, 11.5, 11.9, 12.4, 12.8, 13.2, 13.6, 13.9, 14.2, 14.6, 15.3, 15.9)
  p95_vals <- c(4.0, 4.9, 5.9, 6.8, 7.7, 8.6, 9.4, 10.3, 11.0, 11.7, 12.4, 13.1, 13.7, 14.3, 14.8, 15.3, 15.8, 16.2, 16.6, 16.9, 17.3, 17.5, 18.0, 18.0)

  get_threshold <- function(hr, hours_vec, val_vec) {
    if (hr > max(hours_vec)) return(tail(val_vec, 1))
    approx(hours_vec, val_vec, xout = hr)$y
  }

  limit_40 <- get_threshold(age_hours, ref_hours, p40_vals)
  limit_75 <- get_threshold(age_hours, ref_hours, p75_vals)
  limit_95 <- get_threshold(age_hours, ref_hours, p95_vals)

  risk_zone <- ""
  if (tsb_mg_dl > limit_95) {
    risk_zone <- "High Risk Zone (> 95th percentile)"
  } else if (tsb_mg_dl > limit_75) {
    risk_zone <- "High Intermediate Risk Zone (75th-95th percentile)"
  } else if (tsb_mg_dl > limit_40) {
    risk_zone <- "Low Intermediate Risk Zone (40th-75th percentile)"
  } else {
    risk_zone <- "Low Risk Zone (< 40th percentile)"
  }

  return(list(
    Risk_Zone = risk_zone,
    TSB_mg_dL = round(tsb_mg_dl, 2),
    Thresholds_at_Hour = list(
      P95_High = round(limit_95, 2),
      P75_High_Int = round(limit_75, 2),
      P40_Low_Int = round(limit_40, 2)
    ),
    Inputs = list(
      Age_Hours = age_hours,
      Bilirubin = total_bilirubin
    )
  ))
}

hscore_hemophagocytic_syndrome <- function(immunosuppression, temp_c, hepatomegaly, splenomegaly, cytopenias_count, ferritin_ng_ml, triglyceride_mg_dl, fibrinogen_mg_dl, ast_u_l, hemophagocytosis_marrow) {

  immunosuppression <- as.numeric(immunosuppression)
  temp_c <- as.numeric(temp_c)
  hepatomegaly <- as.numeric(hepatomegaly)
  splenomegaly <- as.numeric(splenomegaly)
  cytopenias_count <- as.numeric(cytopenias_count)
  ferritin_ng_ml <- as.numeric(ferritin_ng_ml)
  triglyceride_mg_dl <- as.numeric(triglyceride_mg_dl)
  fibrinogen_mg_dl <- as.numeric(fibrinogen_mg_dl)
  ast_u_l <- as.numeric(ast_u_l)
  hemophagocytosis_marrow <- as.numeric(hemophagocytosis_marrow)

  if (any(is.na(c(immunosuppression, temp_c, hepatomegaly, splenomegaly, cytopenias_count, ferritin_ng_ml, triglyceride_mg_dl, fibrinogen_mg_dl, ast_u_l, hemophagocytosis_marrow)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (immunosuppression == 1) {
    score <- score + 18
  }

  if (temp_c >= 38.4 && temp_c <= 39.4) {
    score <- score + 33
  } else if (temp_c > 39.4) {
    score <- score + 49
  }

  if (hepatomegaly == 1 && splenomegaly == 1) {
    score <- score + 38
  } else if (hepatomegaly == 1 || splenomegaly == 1) {
    score <- score + 23
  }

  if (cytopenias_count == 2) {
    score <- score + 24
  } else if (cytopenias_count == 3) {
    score <- score + 34
  }

  if (ferritin_ng_ml >= 2000 && ferritin_ng_ml <= 6000) {
    score <- score + 35
  } else if (ferritin_ng_ml > 6000) {
    score <- score + 50
  }

  tri_limit <- 133
  fib_limit <- 250

  if (triglyceride_mg_dl > tri_limit || fibrinogen_mg_dl <= fib_limit) {
    score <- score + 64
  }

  if (ast_u_l >= 30) {
    score <- score + 19
  }

  if (hemophagocytosis_marrow == 1) {
    score <- score + 35
  }

  probability <- ""
  if (score < 90) {
    probability <- "< 1%"
  } else if (score <= 169) {
    probability <- "Variable (< 50%)"
  } else {
    probability <- "> 90% (High Probability of HLH)"
  }

  return(list(
    HScore = score,
    HLH_Probability = probability,
    Inputs = list(
      Immunosuppression = immunosuppression == 1,
      Temp_C = temp_c,
      Organomegaly = if(hepatomegaly == 1 && splenomegaly == 1) "Both" else if(hepatomegaly == 1 || splenomegaly == 1) "One" else "None",
      Cytopenias = cytopenias_count,
      Ferritin = ferritin_ng_ml,
      Triglyceride = triglyceride_mg_dl,
      Fibrinogen = fibrinogen_mg_dl,
      AST = ast_u_l,
      Marrow_Hemophagocytosis = hemophagocytosis_marrow == 1
    )
  ))
}

hark_screen <- function(humiliation, afraid, rape, kick) {

  humiliation <- as.numeric(humiliation)
  afraid <- as.numeric(afraid)
  rape <- as.numeric(rape)
  kick <- as.numeric(kick)

  if (any(is.na(c(humiliation, afraid, rape, kick)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- humiliation + afraid + rape + kick

  result <- ""
  recommendation <- ""

  if (score >= 1) {
    result <- "Positive Screen"
    recommendation <- "The patient has likely experienced intimate partner violence (IPV). Assess immediate safety and refer to support services."
  } else {
    result <- "Negative Screen"
    recommendation <- "No current disclosure of IPV. Monitor and rescreen if clinical suspicion remains."
  }

  return(list(
    HARK_Score = score,
    Screen_Result = result,
    Recommendation = recommendation,
    Inputs = list(
      Humiliation = humiliation == 1,
      Afraid = afraid == 1,
      Rape = rape == 1,
      Kick = kick == 1
    )
  ))
}

hunt_hess_classification <- function(clinical_findings) {

  clinical_findings <- as.numeric(clinical_findings)

  if (is.na(clinical_findings) || !clinical_findings %in% 1:5) {
    stop("Error: Clinical findings must be an integer between 1 and 5 representing the grade criteria.")
  }

  grade <- clinical_findings

  description <- ""
  mortality <- ""

  if (grade == 1) {
    description <- "Asymptomatic, mild headache, or slight nuchal rigidity."
    mortality <- "30%"
  } else if (grade == 2) {
    description <- "Moderate to severe headache, nuchal rigidity, no neurologic deficit other than cranial nerve palsy."
    mortality <- "40%"
  } else if (grade == 3) {
    description <- "Drowsiness, confusion, or mild focal deficit."
    mortality <- "50%"
  } else if (grade == 4) {
    description <- "Stupor, moderate-to-severe hemiparesis, possible early decerebrate rigidity, and vegetative disturbances."
    mortality <- "80%"
  } else {
    description <- "Deep coma, decerebrate rigidity, moribund appearance."
    mortality <- "90%"
  }

  return(list(
    Hunt_Hess_Grade = grade,
    Description = description,
    Survival_Rate = paste0("Survival ~", 100 - as.numeric(gsub("%", "", mortality)), "%"),
    Mortality_Rate = mortality
  ))
}

hits_screening_tool <- function(physically_hurt, insult, threaten_harm, scream_curse) {

  physically_hurt <- as.numeric(physically_hurt)
  insult <- as.numeric(insult)
  threaten_harm <- as.numeric(threaten_harm)
  scream_curse <- as.numeric(scream_curse)

  inputs <- c(physically_hurt, insult, threaten_harm, scream_curse)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(inputs < 1) || any(inputs > 5)) {
    stop("Error: All item scores must be integers between 1 (Never) and 5 (Frequently).")
  }

  total_score <- sum(inputs)

  result <- ""
  recommendation <- ""

  if (total_score > 10) {
    result <- "Positive Screen"
    recommendation <- "Indicates probable domestic abuse. Further assessment and referral to support services are warranted."
  } else {
    result <- "Negative Screen"
    recommendation <- "No current indication of abuse based on this screen."
  }

  return(list(
    HITS_Score = total_score,
    Screening_Result = result,
    Recommendation = recommendation,
    Inputs = list(
      Hurt = physically_hurt,
      Insult = insult,
      Threaten = threaten_harm,
      Scream = scream_curse
    )
  ))
}

hydroxychloroquine_dosing <- function(weight_kg, indication) {

  weight_kg <- as.numeric(weight_kg)
  indication <- tolower(as.character(indication))

  if (is.na(weight_kg) || weight_kg <= 0) {
    stop("Error: Weight must be a positive number.")
  }

  if (!indication %in% c("lupus", "rheumatoid_arthritis", "malaria_prophylaxis", "malaria_treatment")) {
    stop("Error: Indication must be 'lupus', 'rheumatoid_arthritis', 'malaria_prophylaxis', or 'malaria_treatment'.")
  }

  dosing_info <- ""
  max_dose <- ""

  if (indication == "lupus") {
    daily_dose <- min(400, 5 * weight_kg) # Max 400mg/day or 5mg/kg
    dosing_info <- paste0("200-400 mg/day (Max 5 mg/kg actual body weight). Calculated Max: ", round(daily_dose, 0), " mg/day.")
    max_dose <- "Daily dose should not exceed 5 mg/kg (actual body weight) to reduce retinal toxicity risk."
  } else if (indication == "rheumatoid_arthritis") {
    daily_dose <- min(400, 5 * weight_kg) # Often cited similarly to Lupus guidelines now (AAO recommendations)
    dosing_info <- paste0("200-400 mg/day. Calculated Max based on weight: ", round(daily_dose, 0), " mg/day.")
    max_dose <- "Limit to < 5 mg/kg real body weight."
  } else if (indication == "malaria_prophylaxis") {
    weekly_dose <- 400 # Standard adult dose
    # Pediatric: 6.5 mg/kg weekly, max 400mg
    pediatric_dose <- min(400, 6.5 * weight_kg)
    dosing_info <- paste0("Adults: 400 mg weekly. Pediatric (<60kg): ", round(pediatric_dose, 0), " mg weekly.")
    max_dose <- "Start 2 weeks prior to exposure and continue for 4 weeks after leaving."
  } else { # malaria_treatment
    # Standard acute dosing: 800mg stat, then 400mg at 6h, 24h, 48h.
    # Pediatric: 13 mg/kg stat, then 6.5 mg/kg at 6, 24, 48h.
    initial_load <- min(800, 13 * weight_kg)
    subsequent <- min(400, 6.5 * weight_kg)
    dosing_info <- paste0("Initial: ", round(initial_load, 0), " mg. Then ", round(subsequent, 0), " mg at 6, 24, and 48 hours.")
    max_dose <- "Total dose: 2000 mg (Adults) or 32.5 mg/kg (Peds)."
  }

  return(list(
    Indication = indication,
    Weight_kg = weight_kg,
    Recommended_Dosing = dosing_info,
    Safety_Guidance = max_dose
  ))
}

hypoglycemia_risk_score <- function(prior_hypo_episodes, insulin_use, sulfonylurea_use, severe_ckd, age_years, total_ed_visits_12mo) {

  prior_hypo_episodes <- as.numeric(prior_hypo_episodes)
  insulin_use <- as.numeric(insulin_use)
  sulfonylurea_use <- as.numeric(sulfonylurea_use)
  severe_ckd <- as.numeric(severe_ckd)
  age_years <- as.numeric(age_years)
  total_ed_visits_12mo <- as.numeric(total_ed_visits_12mo)

  if (any(is.na(c(prior_hypo_episodes, insulin_use, sulfonylurea_use, severe_ckd, age_years, total_ed_visits_12mo)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (prior_hypo_episodes == 0) {
    score <- score + 0
  } else if (prior_hypo_episodes >= 1 && prior_hypo_episodes <= 2) {
    score <- score + 3
  } else {
    score <- score + 5
  }

  if (insulin_use == 1) {
    score <- score + 3
  }

  if (sulfonylurea_use == 1) {
    score <- score + 3
  }

  if (severe_ckd == 1) {
    score <- score + 2
  }

  if (age_years < 77) {
    score <- score + 2
  }

  if (total_ed_visits_12mo == 0) {
    score <- score + 0
  } else if (total_ed_visits_12mo == 1) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  risk_level <- ""
  risk_percentage <- ""

  if (score <= 1) {
    risk_level <- "Low Risk"
    risk_percentage <- "< 1%"
  } else if (score <= 6) {
    risk_level <- "Intermediate Risk"
    risk_percentage <- "1 - 5%"
  } else {
    risk_level <- "High Risk"
    risk_percentage <- "> 5%"
  }

  return(list(
    Hypoglycemia_Risk_Score = score,
    Risk_Level = risk_level,
    Est_12_Month_Risk = risk_percentage,
    Inputs = list(
      Prior_Hypo_Count = prior_hypo_episodes,
      Insulin = insulin_use == 1,
      Sulfonylurea = sulfonylurea_use == 1,
      Severe_CKD = severe_ckd == 1,
      Age = age_years,
      Total_ED_Visits = total_ed_visits_12mo
    )
  ))
}

hope_score_hypothermia <- function(age, sex, mechanism_hypothermia, cpr_duration_min, potassium_mmol_l, temperature_c) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  mechanism_hypothermia <- tolower(as.character(mechanism_hypothermia))
  cpr_duration_min <- as.numeric(cpr_duration_min)
  potassium_mmol_l <- as.numeric(potassium_mmol_l)
  temperature_c <- as.numeric(temperature_c)

  if (any(is.na(c(age, cpr_duration_min, potassium_mmol_l, temperature_c)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (!mechanism_hypothermia %in% c("asphyxia", "non-asphyxia")) {
    stop("Error: Mechanism must be 'asphyxia' (e.g. submersion/avalanche) or 'non-asphyxia'.")
  }

  score <- 0

  if (age >= 0) {
    score <- score + 0
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (mechanism_hypothermia == "asphyxia") {
    score <- score + 2
  }

  if (cpr_duration_min > 0) {
    score <- score + 0
  }

  if (potassium_mmol_l > 0) {
    score <- score + 0
  }

  logit <- 2.34 - (0.04 * age) - (0.82 * (if (sex == "male") 1 else 0)) + (3.29 * (if (mechanism_hypothermia == "non-asphyxia") 1 else 0)) - (0.01 * cpr_duration_min) - (0.57 * potassium_mmol_l) + (0.08 * temperature_c)

  prob_survival <- exp(logit) / (1 + exp(logit))
  prob_percent <- prob_survival * 100

  recommendation <- ""
  if (prob_percent >= 10) {
    recommendation <- "ECLS Rewarming Recommended. Probability of survival >10%."
  } else {
    recommendation <- "ECLS may be futile (Survival probability <10%). Consider clinical context."
  }

  return(list(
    HOPE_Score_Probability = paste0(round(prob_percent, 1), "%"),
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      Mechanism = mechanism_hypothermia,
      CPR_Duration = cpr_duration_min,
      Potassium = potassium_mmol_l,
      Temperature = temperature_c
    )
  ))
}

body_weight_calculations <- function(sex, height, actual_weight, height_units = "in", weight_units = "kg") {

  sex <- tolower(as.character(sex))
  height <- as.numeric(height)
  actual_weight <- as.numeric(actual_weight)

  if (any(is.na(c(height, actual_weight)))) {
    stop("Error: Height and Weight must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  height_in <- height
  if (height_units == "cm") {
    height_in <- height / 2.54
  }

  weight_kg <- actual_weight
  if (weight_units == "lbs") {
    weight_kg <- actual_weight / 2.20462
  }

  ibw <- 0
  if (sex == "male") {
    ibw <- 50 + (2.3 * (height_in - 60))
  } else {
    ibw <- 45.5 + (2.3 * (height_in - 60))
  }

  if (ibw < 0) {
    ibw <- 1
  }

  abw <- ibw + 0.4 * (weight_kg - ibw)

  weight_used_for_dosing <- 0
  dosing_method <- ""

  if (weight_kg < ibw) {
    weight_used_for_dosing <- weight_kg
    dosing_method <- "Actual Body Weight (Patient is Underweight)"
  } else if (weight_kg > (1.3 * ibw)) {
    weight_used_for_dosing <- abw
    dosing_method <- "Adjusted Body Weight (Patient is Obese >130% IBW)"
  } else {
    weight_used_for_dosing <- weight_kg
    dosing_method <- "Actual Body Weight (Patient is near IBW)"
  }

  return(list(
    Ideal_Body_Weight_kg = round(ibw, 1),
    Adjusted_Body_Weight_kg = round(abw, 1),
    Actual_Body_Weight_kg = round(weight_kg, 1),
    Percent_Over_IBW = paste0(round(((weight_kg - ibw) / ibw) * 100, 1), "%"),
    Dosing_Suggestion = list(
      Weight_to_Use = round(weight_used_for_dosing, 1),
      Reasoning = dosing_method
    ),
    Inputs = list(
      Sex = sex,
      Height_in = round(height_in, 1),
      Actual_Weight_Input = actual_weight
    )
  ))
}

imdc_risk_score <- function(time_to_therapy_lt_1yr, kps_lt_80, hemoglobin_lt_lln, calcium_gt_uln, neutrophils_gt_uln, platelets_gt_uln) {

  time_to_therapy_lt_1yr <- as.numeric(time_to_therapy_lt_1yr)
  kps_lt_80 <- as.numeric(kps_lt_80)
  hemoglobin_lt_lln <- as.numeric(hemoglobin_lt_lln)
  calcium_gt_uln <- as.numeric(calcium_gt_uln)
  neutrophils_gt_uln <- as.numeric(neutrophils_gt_uln)
  platelets_gt_uln <- as.numeric(platelets_gt_uln)

  if (any(is.na(c(time_to_therapy_lt_1yr, kps_lt_80, hemoglobin_lt_lln, calcium_gt_uln, neutrophils_gt_uln, platelets_gt_uln)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (time_to_therapy_lt_1yr == 1) score <- score + 1
  if (kps_lt_80 == 1) score <- score + 1
  if (hemoglobin_lt_lln == 1) score <- score + 1
  if (calcium_gt_uln == 1) score <- score + 1
  if (neutrophils_gt_uln == 1) score <- score + 1
  if (platelets_gt_uln == 1) score <- score + 1

  risk_group <- ""
  median_os <- ""

  if (score == 0) {
    risk_group <- "Favorable Risk"
    median_os <- "43.2 months"
  } else if (score <= 2) {
    risk_group <- "Intermediate Risk"
    median_os <- "22.5 months"
  } else {
    risk_group <- "Poor Risk"
    median_os <- "7.8 months"
  }

  return(list(
    IMDC_Score = score,
    Risk_Group = risk_group,
    Median_Overall_Survival = median_os,
    Inputs = list(
      Time_to_Tx_lt_1yr = time_to_therapy_lt_1yr == 1,
      KPS_lt_80 = kps_lt_80 == 1,
      Anemia = hemoglobin_lt_lln == 1,
      Hypercalcemia = calcium_gt_uln == 1,
      Neutrophilia = neutrophils_gt_uln == 1,
      Thrombocytosis = platelets_gt_uln == 1
    )
  ))
}

irae_endocrine_diabetes_grading <- function(fasting_glucose_mg_dl, metabolic_acidosis_present, insulin_dependence) {

  fasting_glucose_mg_dl <- as.numeric(fasting_glucose_mg_dl)
  metabolic_acidosis_present <- as.numeric(metabolic_acidosis_present)
  insulin_dependence <- as.numeric(insulin_dependence)

  if (any(is.na(c(fasting_glucose_mg_dl, metabolic_acidosis_present, insulin_dependence)))) {
    stop("Error: All inputs must be numeric.")
  }

  grade <- 0

  if (metabolic_acidosis_present == 1) {
    grade <- 4
  } else if (fasting_glucose_mg_dl > 500) {
    grade <- 3
  } else if (fasting_glucose_mg_dl > 250 && fasting_glucose_mg_dl <= 500) {
    grade <- 3
    if (insulin_dependence == 0) {
      grade <- 3
    }
  } else if (fasting_glucose_mg_dl > 160 && fasting_glucose_mg_dl <= 250) {
    grade <- 2
  } else {
    grade <- 1
  }

  definition <- ""
  management <- ""

  if (grade == 1) {
    definition <- "Asymptomatic or mild symptoms; fasting glucose >ULN - 160 mg/dL; no intervention indicated."
    management <- "Continue ICPi. Monitor glucose."
  } else if (grade == 2) {
    definition <- "Moderate symptoms; fasting glucose >160 - 250 mg/dL; medical intervention indicated."
    management <- "Hold ICPi until glucose control achieved. Start insulin if needed."
  } else if (grade == 3) {
    definition <- "Severe symptoms; fasting glucose >250 - 500 mg/dL; hospitalization indicated."
    management <- "Hold ICPi. Hospitalize for stabilization. Insulin therapy required."
  } else {
    definition <- "Life-threatening consequences (DKA); fasting glucose >500 mg/dL."
    management <- "Permanently discontinue ICPi. Urgent intervention indicated (DKA protocol)."
  }

  return(list(
    CTCAE_Grade = grade,
    Definition = definition,
    Management = management,
    Inputs = list(
      Fasting_Glucose = fasting_glucose_mg_dl,
      Metabolic_Acidosis = metabolic_acidosis_present == 1,
      Insulin_Dependent = insulin_dependence == 1
    )
  ))
}

irae_hypothyroidism_grading <- function(symptomatic, adl_limitation, life_threatening) {

  symptomatic <- as.numeric(symptomatic)
  adl_limitation <- as.numeric(adl_limitation)
  life_threatening <- as.numeric(life_threatening)

  if (any(is.na(c(symptomatic, adl_limitation, life_threatening)))) {
    stop("Error: All inputs must be numeric.")
  }

  grade <- 1

  if (life_threatening == 1) {
    grade <- 4
  } else if (adl_limitation == 2) {
    grade <- 3
  } else if (symptomatic == 1 || adl_limitation == 1) {
    grade <- 2
  } else {
    grade <- 1
  }

  definition <- ""
  management <- ""

  if (grade == 1) {
    definition <- "Asymptomatic; clinical or diagnostic observations only; intervention not indicated."
    management <- "Continue ICPi. Monitor TSH/fT4 every 4-6 weeks. Can consider thyroid replacement if TSH > 10."
  } else if (grade == 2) {
    definition <- "Symptomatic; minimal, local or noninvasive intervention indicated; limiting instrumental ADL."
    management <- "Continue ICPi. Initiate thyroid hormone replacement (e.g., levothyroxine). Monitor levels."
  } else if (grade == 3) {
    definition <- "Severe or medically significant but not immediately life-threatening; limiting self-care ADL."
    management <- "Hold ICPi until symptoms resolve to Grade 1. Initiate thyroid replacement. Endocrinology consult."
  } else {
    definition <- "Life-threatening consequences; urgent intervention indicated."
    management <- "Permanently discontinue ICPi. Admit for urgent management (e.g., myxedema coma protocol)."
  }

  return(list(
    CTCAE_Grade = grade,
    Definition = definition,
    Management_Recommendation = management,
    Inputs = list(
      Symptomatic = symptomatic == 1,
      ADL_Limitation = if(adl_limitation == 2) "Self-Care" else if(adl_limitation == 1) "Instrumental" else "None",
      Life_Threatening = life_threatening == 1
    )
  ))
}

irae_colitis_grading <- function(stools_above_baseline, incontinence, life_threatening, pain_mucus_blood) {

  stools_above_baseline <- as.numeric(stools_above_baseline)
  incontinence <- as.numeric(incontinence)
  life_threatening <- as.numeric(life_threatening)
  pain_mucus_blood <- as.numeric(pain_mucus_blood)

  if (any(is.na(c(stools_above_baseline, incontinence, life_threatening, pain_mucus_blood)))) {
    stop("Error: All inputs must be numeric.")
  }

  grade <- 1

  if (life_threatening == 1) {
    grade <- 4
  } else if (stools_above_baseline >= 7 || incontinence == 1 || pain_mucus_blood == 1) {
    grade <- 3
  } else if (stools_above_baseline >= 4 && stools_above_baseline <= 6) {
    grade <- 2
  } else {
    grade <- 1
  }

  definition <- ""
  management <- ""

  if (grade == 1) {
    definition <- "Stools < 4 above baseline/day; asymptomatic; clinical/diagnostic observations only; intervention not indicated."
    management <- "Continue ICPi. Monitor for dehydration and symptom progression."
  } else if (grade == 2) {
    definition <- "Stools 4-6 above baseline/day; IV fluids indicated < 24 hours; not interfering with ADL."
    management <- "Hold ICPi. Administer corticosteroids (0.5-1 mg/kg/day prednisone equivalent). If no improvement in 2-3 days, consider Grade 3 management."
  } else if (grade == 3) {
    definition <- "Stools >= 7 above baseline/day; incontinence; IV fluids > 24 hours; hospitalization indicated; severe pain; interfering with ADL."
    management <- "Permanently discontinue ICPi. Hospitalize. IV Methylprednisolone (1-2 mg/kg/day). If refractory, consider infliximab or vedolizumab."
  } else {
    definition <- "Life-threatening consequences; urgent intervention indicated (e.g., perforation, bleeding, hemodynamic instability)."
    management <- "Permanently discontinue ICPi. Emergent hospitalization. High-dose IV steroids. Surgical consultation. Consider biologics (infliximab/vedolizumab)."
  }

  return(list(
    CTCAE_Grade = grade,
    Definition = definition,
    Management_Recommendation = management,
    Inputs = list(
      Stools_Above_Baseline = stools_above_baseline,
      Incontinence = incontinence == 1,
      Life_Threatening = life_threatening == 1,
      Severe_Symptoms = pain_mucus_blood == 1
    )
  ))
}

irae_hepatitis_grading <- function(ast, alt, total_bilirubin, ast_uln, alt_uln, bilirubin_uln) {

  ast <- as.numeric(ast)
  alt <- as.numeric(alt)
  total_bilirubin <- as.numeric(total_bilirubin)
  ast_uln <- as.numeric(ast_uln)
  alt_uln <- as.numeric(alt_uln)
  bilirubin_uln <- as.numeric(bilirubin_uln)

  if (any(is.na(c(ast, alt, total_bilirubin, ast_uln, alt_uln, bilirubin_uln)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (ast_uln <= 0 || alt_uln <= 0 || bilirubin_uln <= 0) {
    stop("Error: Upper Limits of Normal (ULN) must be positive.")
  }

  ast_ratio <- ast / ast_uln
  alt_ratio <- alt / alt_uln
  bili_ratio <- total_bilirubin / bilirubin_uln

  max_enzyme_ratio <- max(ast_ratio, alt_ratio)

  grade_enzymes <- 0
  if (max_enzyme_ratio > 20.0) {
    grade_enzymes <- 4
  } else if (max_enzyme_ratio > 5.0) {
    grade_enzymes <- 3
  } else if (max_enzyme_ratio > 3.0) {
    grade_enzymes <- 2
  } else if (max_enzyme_ratio > 1.0) {
    grade_enzymes <- 1
  }

  grade_bili <- 0
  if (bili_ratio > 10.0) {
    grade_bili <- 4
  } else if (bili_ratio > 3.0) {
    grade_bili <- 3
  } else if (bili_ratio > 1.5) {
    grade_bili <- 2
  } else if (bili_ratio > 1.0) {
    grade_bili <- 1
  }

  final_grade <- max(grade_enzymes, grade_bili)

  definition <- ""
  management <- ""

  if (final_grade == 1) {
    definition <- "Asymptomatic; AST/ALT >ULN-3.0x; Bilirubin >ULN-1.5x."
    management <- "Continue ICPi with frequent monitoring (1-2x/week)."
  } else if (final_grade == 2) {
    definition <- "AST/ALT >3.0-5.0x ULN; Bilirubin >1.5-3.0x ULN."
    management <- "Hold ICPi. Monitor labs every 2-3 days. Consider Prednisone (0.5-1 mg/kg) if no improvement."
  } else if (final_grade == 3) {
    definition <- "AST/ALT >5.0-20.0x ULN; Bilirubin >3.0-10.0x ULN."
    management <- "Permanently discontinue ICPi. Start Prednisone (1-2 mg/kg/day). If steroid refractory after 2-3 days, add Mycophenolate."
  } else if (final_grade == 4) {
    definition <- "AST/ALT >20.0x ULN; Bilirubin >10.0x ULN."
    management <- "Permanently discontinue ICPi. Admit. Methylprednisolone IV (2 mg/kg/day). If refractory, add Mycophenolate or ATG/Tacrolimus."
  } else {
    definition <- "Normal liver function (<= ULN)."
    management <- "No intervention required."
  }

  return(list(
    CTCAE_Grade = final_grade,
    Definition = definition,
    Management_Recommendation = management,
    Calculated_Ratios = list(
      AST_Multiple = round(ast_ratio, 2),
      ALT_Multiple = round(alt_ratio, 2),
      Bilirubin_Multiple = round(bili_ratio, 2)
    )
  ))
}

irae_pneumonitis_grading <- function(asymptomatic, symptomatic, severe_symptoms, life_threatening, hypoxia_on_room_air, oxygen_needed, oxygen_management) {

  asymptomatic <- as.numeric(asymptomatic)
  symptomatic <- as.numeric(symptomatic)
  severe_symptoms <- as.numeric(severe_symptoms)
  life_threatening <- as.numeric(life_threatening)
  hypoxia_on_room_air <- as.numeric(hypoxia_on_room_air)
  oxygen_needed <- as.numeric(oxygen_needed)
  oxygen_management <- tolower(as.character(oxygen_management))

  if (any(is.na(c(asymptomatic, symptomatic, severe_symptoms, life_threatening, hypoxia_on_room_air, oxygen_needed)))) {
    stop("Error: All inputs must be numeric.")
  }

  grade <- 0

  if (life_threatening == 1) {
    grade <- 4
  } else if (severe_symptoms == 1 || hypoxia_on_room_air == 1 || oxygen_needed == 1) {
    grade <- 3
  } else if (symptomatic == 1) {
    grade <- 2
  } else if (asymptomatic == 1) {
    grade <- 1
  } else {
    grade <- 0
  }

  definition <- ""
  management <- ""

  if (grade == 1) {
    definition <- "Asymptomatic; confined to radiographic findings only."
    management <- "Hold ICPi. Monitor weekly. May resume if radiographic evidence improves/resolves."
  } else if (grade == 2) {
    definition <- "Symptomatic; mild/moderate symptoms; medical intervention indicated; limiting instrumental ADL."
    management <- "Hold ICPi. Prednisone 1-2 mg/kg/day. Pulmonary/ID consult. Bronchoscopy if needed."
  } else if (grade == 3) {
    definition <- "Severe symptoms; limiting self-care ADL; oxygen indicated."
    management <- "Permanently discontinue ICPi. Hospitalize. Prednisone 1-2 mg/kg/day or Methylprednisolone IV. If no improvement in 48h, consider Infliximab/Mycophenolate/IVIG."
  } else if (grade == 4) {
    definition <- "Life-threatening respiratory compromise; urgent intervention indicated (intubation)."
    management <- "Permanently discontinue ICPi. ICU admission. High-dose IV steroids. If refractory, add immunosuppression (e.g., Infliximab, Tocilizumab)."
  } else {
    definition <- "No Pneumonitis."
    management <- "No specific intervention."
  }

  return(list(
    CTCAE_Grade = grade,
    Definition = definition,
    Management_Recommendation = management,
    Inputs = list(
      Asymptomatic = asymptomatic == 1,
      Symptomatic = symptomatic == 1,
      Severe = severe_symptoms == 1,
      Life_Threatening = life_threatening == 1,
      Hypoxia = hypoxia_on_room_air == 1,
      O2_Required = oxygen_needed == 1
    )
  ))
}

irae_nephritis_grading <- function(creatinine_level, creatinine_uln, baseline_creatinine = NULL, life_threatening = 0) {

  creatinine_level <- as.numeric(creatinine_level)
  creatinine_uln <- as.numeric(creatinine_uln)
  life_threatening <- as.numeric(life_threatening)

  if (any(is.na(c(creatinine_level, creatinine_uln)))) {
    stop("Error: Creatinine level and ULN must be numeric.")
  }

  if (creatinine_uln <= 0) {
    stop("Error: Upper Limit of Normal (ULN) must be positive.")
  }

  grade <- 0

  # Grading criteria:
  # Grade 1: Cr > ULN - 1.5x ULN
  # Grade 2: Cr > 1.5 - 3.0x baseline; or > 1.5 - 3.0x ULN
  # Grade 3: Cr > 3.0x baseline; or > 3.0 - 6.0x ULN
  # Grade 4: Cr > 6.0x ULN; or life-threatening (dialysis indicated)

  # Calculate ratios
  ratio_uln <- creatinine_level / creatinine_uln
  ratio_baseline <- 0
  if (!is.null(baseline_creatinine) && !is.na(as.numeric(baseline_creatinine))) {
    bl_cr <- as.numeric(baseline_creatinine)
    if (bl_cr > 0) ratio_baseline <- creatinine_level / bl_cr
  }

  # Determine Grade based on ULN criteria first
  grade_uln <- 0
  if (ratio_uln > 6.0) grade_uln <- 4
  else if (ratio_uln > 3.0) grade_uln <- 3
  else if (ratio_uln > 1.5) grade_uln <- 2
  else if (ratio_uln > 1.0) grade_uln <- 1

  # Determine Grade based on Baseline criteria (if available)
  grade_bl <- 0
  if (ratio_baseline > 3.0) grade_bl <- 3
  else if (ratio_baseline > 1.5) grade_bl <- 2

  # Final grade is the higher of the two assessments
  grade <- max(grade_uln, grade_bl)

  # Override for life-threatening
  if (life_threatening == 1) {
    grade <- 4
  }

  definition <- ""
  management <- ""

  if (grade == 1) {
    definition <- "Creatinine increased > ULN - 1.5 x ULN."
    management <- "Continue ICPi. Monitor creatinine weekly. Ensure hydration and avoid nephrotoxins."
  } else if (grade == 2) {
    definition <- "Creatinine increased > 1.5 - 3.0 x baseline; or > 1.5 - 3.0 x ULN."
    management <- "Hold ICPi. Monitor creatinine every 2-3 days. If no improvement, start Prednisone 0.5-1 mg/kg/day. Nephrology consult."
  } else if (grade == 3) {
    definition <- "Creatinine increased > 3.0 x baseline; or > 3.0 - 6.0 x ULN."
    management <- "Permanently discontinue ICPi. Hospitalize. Prednisone 1-2 mg/kg/day. Nephrology consult."
  } else if (grade == 4) {
    definition <- "Creatinine increased > 6.0 x ULN; or life-threatening consequences (dialysis)."
    management <- "Permanently discontinue ICPi. Hospitalize. Methylprednisolone IV. Consider other immunosuppressants (e.g., Azathioprine/Mycophenolate) if refractory."
  } else {
    definition <- "Normal renal function (<= ULN and <= 1.5x Baseline)."
    management <- "No intervention required."
  }

  return(list(
    CTCAE_Grade = grade,
    Definition = definition,
    Management_Recommendation = management,
    Inputs = list(
      Creatinine = creatinine_level,
      ULN = creatinine_uln,
      Baseline = if(!is.null(baseline_creatinine)) baseline_creatinine else "Not Provided",
      Life_Threatening = life_threatening == 1
    )
  ))
}

immunization_schedule_cdc <- function(current_age_months) {

  if (is.na(current_age_months) || current_age_months < 0) {
    stop("Error: Age in months must be a non-negative number.")
  }

  vaccines_due <- c()

  if (current_age_months < 1) {
    vaccines_due <- c(vaccines_due, "Hepatitis B (1st dose)")
  }

  if (current_age_months >= 1 && current_age_months <= 2) {
    vaccines_due <- c(vaccines_due, "Hepatitis B (2nd dose)")
  }

  if (current_age_months == 2) {
    vaccines_due <- c(vaccines_due, "Rotavirus (RV) (1st dose)", "DTaP (1st dose)", "Hib (1st dose)", "PCV15/20 (1st dose)", "IPV (1st dose)")
  }

  if (current_age_months == 4) {
    vaccines_due <- c(vaccines_due, "Rotavirus (RV) (2nd dose)", "DTaP (2nd dose)", "Hib (2nd dose)", "PCV15/20 (2nd dose)", "IPV (2nd dose)")
  }

  if (current_age_months == 6) {
    vaccines_due <- c(vaccines_due, "Hepatitis B (3rd dose - 6-18 months)", "Rotavirus (RV) (3rd dose - if needed)", "DTaP (3rd dose)", "Hib (3rd dose - if needed)", "PCV15/20 (3rd dose)", "IPV (3rd dose - 6-18 months)", "Influenza (Annual)")
  }

  if (current_age_months >= 6 && current_age_months < 12) {
    if (!"Influenza (Annual)" %in% vaccines_due) {
      vaccines_due <- c(vaccines_due, "Influenza (Annual)")
    }
  }

  if (current_age_months >= 12 && current_age_months <= 15) {
    vaccines_due <- c(vaccines_due, "Hib (4th dose)", "PCV15/20 (4th dose)", "MMR (1st dose)", "Varicella (1st dose)")
  }

  if (current_age_months >= 12 && current_age_months <= 23) {
    vaccines_due <- c(vaccines_due, "Hepatitis A (2 dose series, 6 months apart)")
  }

  if (current_age_months >= 15 && current_age_months <= 18) {
    vaccines_due <- c(vaccines_due, "DTaP (4th dose)")
  }

  if (current_age_months >= 48 && current_age_months <= 72) {
    vaccines_due <- c(vaccines_due, "DTaP (5th dose)", "IPV (4th dose)", "MMR (2nd dose)", "Varicella (2nd dose)")
  }

  if (current_age_months >= 132 && current_age_months <= 144) {
    vaccines_due <- c(vaccines_due, "Tdap (1 dose)", "HPV (2 or 3 dose series)", "Meningococcal ACWY (1st dose)")
  }

  if (current_age_months >= 192 && current_age_months <= 216) {
    vaccines_due <- c(vaccines_due, "Meningococcal ACWY (Booster)", "Meningococcal B (Clinical discretion)")
  }

  if (length(vaccines_due) == 0) {
    vaccines_due <- c("No specific routine immunizations scheduled for this exact age. Check catch-up schedule if behind.")
  }

  return(list(
    Age_Months = current_age_months,
    Age_Years = round(current_age_months / 12, 1),
    Recommended_Vaccines = vaccines_due
  ))
}

impact_head_injury_prognosis <- function(age, motor_score, pupils_reactive, ct_marshall_class, tsah_present, epidural_mass_present, glucose_mg_dl = NULL, hemoglobin_g_dl = NULL) {

  age <- as.numeric(age)
  motor_score <- as.numeric(motor_score)
  pupils_reactive <- as.numeric(pupils_reactive)
  ct_marshall_class <- as.numeric(ct_marshall_class)
  tsah_present <- as.numeric(tsah_present)
  epidural_mass_present <- as.numeric(epidural_mass_present)

  if (any(is.na(c(age, motor_score, pupils_reactive, ct_marshall_class, tsah_present, epidural_mass_present)))) {
    stop("Error: Core and Extended inputs must be numeric.")
  }

  if (motor_score < 1 || motor_score > 6) stop("Error: Motor Score must be 1-6.")
  if (ct_marshall_class < 1 || ct_marshall_class > 6) stop("Error: Marshall Class must be 1-6.")

  logit_mort <- -3.55 + (0.046 * age)

  if (motor_score == 1) logit_mort <- logit_mort + 1.96
  else if (motor_score == 2) logit_mort <- logit_mort + 1.31
  else if (motor_score == 3) logit_mort <- logit_mort + 0.75
  else if (motor_score == 4) logit_mort <- logit_mort + 0.27
  else if (motor_score == 5) logit_mort <- logit_mort + 0.00
  else logit_mort <- logit_mort + 0.00

  if (pupils_reactive == 0) logit_mort <- logit_mort + 1.74
  else if (pupils_reactive == 1) logit_mort <- logit_mort + 0.80

  if (ct_marshall_class == 1) logit_mort <- logit_mort - 1.85
  else if (ct_marshall_class == 3) logit_mort <- logit_mort + 0.89
  else if (ct_marshall_class == 4) logit_mort <- logit_mort + 1.78
  else if (ct_marshall_class == 5) logit_mort <- logit_mort + 0.95
  else if (ct_marshall_class == 6) logit_mort <- logit_mort + 2.62

  if (tsah_present == 1) logit_mort <- logit_mort + 0.96

  if (epidural_mass_present == 1) logit_mort <- logit_mort - 0.76

  if (!is.null(glucose_mg_dl) && !is.null(hemoglobin_g_dl)) {
    glucose <- as.numeric(glucose_mg_dl)
    hgb <- as.numeric(hemoglobin_g_dl)
    logit_mort <- logit_mort + (0.003 * glucose) - (0.13 * hgb)
  }

  prob_mort <- 1 / (1 + exp(-logit_mort))

  logit_unfav <- -2.73 + (0.043 * age)

  if (motor_score == 1) logit_unfav <- logit_unfav + 1.80
  else if (motor_score == 2) logit_unfav <- logit_unfav + 1.29
  else if (motor_score == 3) logit_unfav <- logit_unfav + 0.83
  else if (motor_score == 4) logit_unfav <- logit_unfav + 0.46
  else if (motor_score == 5) logit_unfav <- logit_unfav + 0.15

  if (pupils_reactive == 0) logit_unfav <- logit_unfav + 1.58
  else if (pupils_reactive == 1) logit_unfav <- logit_unfav + 0.76

  if (ct_marshall_class == 1) logit_unfav <- logit_unfav - 1.44
  else if (ct_marshall_class == 3) logit_unfav <- logit_unfav + 0.88
  else if (ct_marshall_class == 4) logit_unfav <- logit_unfav + 1.71
  else if (ct_marshall_class == 5) logit_unfav <- logit_unfav + 1.04
  else if (ct_marshall_class == 6) logit_unfav <- logit_unfav + 2.84

  if (tsah_present == 1) logit_unfav <- logit_unfav + 1.15

  if (epidural_mass_present == 1) logit_unfav <- logit_unfav - 0.46

  if (!is.null(glucose_mg_dl) && !is.null(hemoglobin_g_dl)) {
    glucose <- as.numeric(glucose_mg_dl)
    hgb <- as.numeric(hemoglobin_g_dl)
    logit_unfav <- logit_unfav + (0.004 * glucose) - (0.11 * hgb)
  }

  prob_unfav <- 1 / (1 + exp(-logit_unfav))

  return(list(
    Predicted_6mo_Mortality = paste0(round(prob_mort * 100, 1), "%"),
    Predicted_6mo_Unfavorable_Outcome = paste0(round(prob_unfav * 100, 1), "%"),
    Definitions = list(
      Unfavorable_Outcome = "Death, Vegetative State, or Severe Disability (GOSE 1-4)",
      Mortality = "Death (GOSE 1)"
    ),
    Inputs = list(
      Age = age,
      Motor = motor_score,
      Pupils_Reactive = pupils_reactive,
      CT_Class = ct_marshall_class,
      tSAH = tsah_present == 1,
      Epidural = epidural_mass_present == 1,
      Labs = if(!is.null(glucose_mg_dl)) "Included" else "Not Included"
    )
  ))
}

impede_vte_score <- function(infection, bmi_lt_25, steroid_use, pe_history, dvt_history, erythropoietin_agent, doxorubicin_regimen, existing_central_catheter, thalidomide_analog, enzalutamide_abiraterone) {

  infection <- as.numeric(infection)
  bmi_lt_25 <- as.numeric(bmi_lt_25)
  steroid_use <- as.numeric(steroid_use)
  pe_history <- as.numeric(pe_history)
  dvt_history <- as.numeric(dvt_history)
  erythropoietin_agent <- as.numeric(erythropoietin_agent)
  doxorubicin_regimen <- as.numeric(doxorubicin_regimen)
  existing_central_catheter <- as.numeric(existing_central_catheter)
  thalidomide_analog <- as.numeric(thalidomide_analog)
  enzalutamide_abiraterone <- as.numeric(enzalutamide_abiraterone)

  if (any(is.na(c(infection, bmi_lt_25, steroid_use, pe_history, dvt_history, erythropoietin_agent, doxorubicin_regimen, existing_central_catheter, thalidomide_analog, enzalutamide_abiraterone)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (infection == 1) {
    score <- score - 3
  }

  if (bmi_lt_25 == 1) {
    score <- score - 1
  }

  if (steroid_use == 1) {
    score <- score + 1
  }

  if (pe_history == 1) {
    score <- score + 3
  }

  if (dvt_history == 1) {
    score <- score + 2
  }

  if (erythropoietin_agent == 1) {
    score <- score + 3
  }

  if (doxorubicin_regimen == 1) {
    score <- score + 2
  }

  if (existing_central_catheter == 1) {
    score <- score + 3
  }

  if (thalidomide_analog == 1) {
    score <- score + 3
  }

  if (enzalutamide_abiraterone == 1) {
    score <- score + 3
  }

  risk_group <- ""
  vte_risk_6mo <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    vte_risk_6mo <- "3.0%"
  } else if (score >= 2 && score <= 5) {
    risk_group <- "Intermediate Risk"
    vte_risk_6mo <- "7.2%"
  } else {
    risk_group <- "High Risk"
    vte_risk_6mo <- "15.2%"
  }

  return(list(
    IMPEDE_Score = score,
    Risk_Category = risk_group,
    Six_Month_VTE_Risk = vte_risk_6mo,
    Inputs = list(
      Infection_Neg3 = infection == 1,
      BMI_lt_25_Neg1 = bmi_lt_25 == 1,
      Steroids_Pos1 = steroid_use == 1,
      Hx_PE_Pos3 = pe_history == 1,
      Hx_DVT_Pos2 = dvt_history == 1,
      ESA_Pos3 = erythropoietin_agent == 1,
      Doxorubicin_Pos2 = doxorubicin_regimen == 1,
      CVC_Pos3 = existing_central_catheter == 1,
      IMiD_Pos3 = thalidomide_analog == 1,
      AntiAndrogen_Pos3 = enzalutamide_abiraterone == 1
    )
  ))
}

improve_bleeding_risk_score <- function(age, sex, gfr, active_cancer, rheumatic_disease, central_venous_catheter, icu_ccu_admission, hepatic_failure_inr_gt_1_5, platelet_count_lt_50, prior_bleeding_3mo, active_gastroduodenal_ulcer) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  gfr <- as.numeric(gfr)
  active_cancer <- as.numeric(active_cancer)
  rheumatic_disease <- as.numeric(rheumatic_disease)
  central_venous_catheter <- as.numeric(central_venous_catheter)
  icu_ccu_admission <- as.numeric(icu_ccu_admission)
  hepatic_failure_inr_gt_1_5 <- as.numeric(hepatic_failure_inr_gt_1_5)
  platelet_count_lt_50 <- as.numeric(platelet_count_lt_50)
  prior_bleeding_3mo <- as.numeric(prior_bleeding_3mo)
  active_gastroduodenal_ulcer <- as.numeric(active_gastroduodenal_ulcer)

  if (any(is.na(c(age, gfr, active_cancer, rheumatic_disease, central_venous_catheter, icu_ccu_admission, hepatic_failure_inr_gt_1_5, platelet_count_lt_50, prior_bleeding_3mo, active_gastroduodenal_ulcer)))) {
    stop("Error: All inputs must be numeric (0 or 1 for binary, numeric for age/gfr).")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (active_gastroduodenal_ulcer == 1) score <- score + 4.5
  if (prior_bleeding_3mo == 1) score <- score + 4.0
  if (platelet_count_lt_50 == 1) score <- score + 4.0
  if (age >= 85) score <- score + 3.5
  if (hepatic_failure_inr_gt_1_5 == 1) score <- score + 2.5
  if (gfr < 30) score <- score + 2.5
  if (icu_ccu_admission == 1) score <- score + 2.5
  if (central_venous_catheter == 1) score <- score + 2.0
  if (rheumatic_disease == 1) score <- score + 2.0
  if (active_cancer == 1) score <- score + 2.0
  if (age >= 40 && age < 85) score <- score + 1.5
  if (sex == "male") score <- score + 1.0
  if (gfr >= 30 && gfr < 60) score <- score + 1.0

  risk_category <- ""
  bleeding_prob <- ""
  recommendation <- ""

  if (score < 7) {
    risk_category <- "Low Bleeding Risk"
    bleeding_prob <- "2.1% (Any Bleed), 1.5% (Major Bleed)"
    recommendation <- "Proceed with anticoagulation if indicated."
  } else {
    risk_category <- "High Bleeding Risk"
    bleeding_prob <- "4.7% (Any Bleed), 3.2% (Major Bleed)"
    recommendation <- "Avoid anticoagulants if possible. Use mechanical prophylaxis or close monitoring."
  }

  return(list(
    IMPROVE_Bleeding_Score = score,
    Risk_Category = risk_category,
    Bleeding_Probability_14_Days = bleeding_prob,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Sex = sex,
      GFR = gfr,
      Active_Ulcer = active_gastroduodenal_ulcer == 1,
      Prior_Bleed = prior_bleeding_3mo == 1,
      Low_Platelets = platelet_count_lt_50 == 1,
      Hepatic_Failure = hepatic_failure_inr_gt_1_5 == 1,
      ICU_CCU = icu_ccu_admission == 1,
      CVC = central_venous_catheter == 1,
      Rheumatic = rheumatic_disease == 1,
      Cancer = active_cancer == 1
    )
  ))
}

improve_vte_risk_score <- function(previous_vte, known_thrombophilia, lower_limb_paralysis, current_cancer, immobilized_ge_7days, icu_ccu_stay, age_gt_60) {

  previous_vte <- as.numeric(previous_vte)
  known_thrombophilia <- as.numeric(known_thrombophilia)
  lower_limb_paralysis <- as.numeric(lower_limb_paralysis)
  current_cancer <- as.numeric(current_cancer)
  immobilized_ge_7days <- as.numeric(immobilized_ge_7days)
  icu_ccu_stay <- as.numeric(icu_ccu_stay)
  age_gt_60 <- as.numeric(age_gt_60)

  if (any(is.na(c(previous_vte, known_thrombophilia, lower_limb_paralysis, current_cancer, immobilized_ge_7days, icu_ccu_stay, age_gt_60)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (previous_vte == 1) score <- score + 3

  if (known_thrombophilia == 1) score <- score + 2

  if (lower_limb_paralysis == 1) score <- score + 2

  if (current_cancer == 1) score <- score + 2

  if (immobilized_ge_7days == 1) score <- score + 1

  if (icu_ccu_stay == 1) score <- score + 1

  if (age_gt_60 == 1) score <- score + 1

  vte_risk_3mo <- ""
  recommendation <- ""

  if (score <= 1) {
    vte_risk_3mo <- "0.7%"
    recommendation <- "Low Risk. Pharmacologic prophylaxis may not be warranted (consider early ambulation/mechanical)."
  } else if (score <= 3) {
    vte_risk_3mo <- "1.5%" # Wait, standard IMPROVE cutoffs:
    # 0-1: Low risk (~0.7%)
    # 2-3: Moderate risk (~1.5%)
    # >=4: High risk (~5.7%)
    recommendation <- "Moderate Risk. Pharmacologic or mechanical prophylaxis indicated."
  } else {
    vte_risk_3mo <- "5.7%"
    recommendation <- "High Risk. Pharmacologic prophylaxis strongly recommended."
  }

  return(list(
    IMPROVE_VTE_Score = score,
    Est_3_Month_VTE_Risk = vte_risk_3mo,
    Recommendation = recommendation,
    Inputs = list(
      Prior_VTE = previous_vte == 1,
      Thrombophilia = known_thrombophilia == 1,
      Paralysis = lower_limb_paralysis == 1,
      Cancer = current_cancer == 1,
      Immobile_7d = immobilized_ge_7days == 1,
      ICU_CCU = icu_ccu_stay == 1,
      Age_gt_60 = age_gt_60 == 1
    )
  ))
}

improvedd_vte_score <- function(previous_vte, known_thrombophilia, lower_limb_paralysis, current_cancer, immobilized_ge_7days, icu_ccu_stay, age_gt_60, d_dimer_ge_2x_uln) {

  previous_vte <- as.numeric(previous_vte)
  known_thrombophilia <- as.numeric(known_thrombophilia)
  lower_limb_paralysis <- as.numeric(lower_limb_paralysis)
  current_cancer <- as.numeric(current_cancer)
  immobilized_ge_7days <- as.numeric(immobilized_ge_7days)
  icu_ccu_stay <- as.numeric(icu_ccu_stay)
  age_gt_60 <- as.numeric(age_gt_60)
  d_dimer_ge_2x_uln <- as.numeric(d_dimer_ge_2x_uln)

  if (any(is.na(c(previous_vte, known_thrombophilia, lower_limb_paralysis, current_cancer, immobilized_ge_7days, icu_ccu_stay, age_gt_60, d_dimer_ge_2x_uln)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (previous_vte == 1) {
    score <- score + 3
  }

  if (known_thrombophilia == 1) {
    score <- score + 2
  }

  if (lower_limb_paralysis == 1) {
    score <- score + 2
  }

  if (current_cancer == 1) {
    score <- score + 2
  }

  if (immobilized_ge_7days == 1) {
    score <- score + 1
  }

  if (icu_ccu_stay == 1) {
    score <- score + 1
  }

  if (age_gt_60 == 1) {
    score <- score + 1
  }

  if (d_dimer_ge_2x_uln == 1) {
    score <- score + 2
  }

  risk_category <- ""
  vte_risk <- ""

  if (score <= 1) {
    risk_category <- "Low Risk"
    vte_risk <- "~0.5%"
  } else if (score <= 3) {
    risk_category <- "Moderate Risk"
    vte_risk <- "~1.5%"
  } else {
    risk_category <- "High Risk"
    vte_risk <- ">= 4.0%"
  }

  return(list(
    IMPROVEDD_Score = score,
    Risk_Category = risk_category,
    VTE_Risk_42_Day = vte_risk,
    Inputs = list(
      Prior_VTE = previous_vte == 1,
      Thrombophilia = known_thrombophilia == 1,
      Paralysis = lower_limb_paralysis == 1,
      Cancer = current_cancer == 1,
      Immobile_7d = immobilized_ge_7days == 1,
      ICU_CCU = icu_ccu_stay == 1,
      Age_gt_60 = age_gt_60 == 1,
      DDimer_2xULN = d_dimer_ge_2x_uln == 1
    )
  ))
}

isee_eoe_severity_index <- function(symptoms_frequency, food_impaction_code, hospitalization_eoe, esophageal_perforation, malnutrition_code, persistent_inflammation, endoscopic_inflammatory_grade, peak_eosinophil_count_grade, endoscopic_fibrostenotic_grade, histologic_fibrostenotic_grade) {

  symptoms_frequency <- as.numeric(symptoms_frequency)
  food_impaction_code <- as.numeric(food_impaction_code)
  hospitalization_eoe <- as.numeric(hospitalization_eoe)
  esophageal_perforation <- as.numeric(esophageal_perforation)
  malnutrition_code <- as.numeric(malnutrition_code)
  persistent_inflammation <- as.numeric(persistent_inflammation)
  endoscopic_inflammatory_grade <- as.numeric(endoscopic_inflammatory_grade)
  peak_eosinophil_count_grade <- as.numeric(peak_eosinophil_count_grade)
  endoscopic_fibrostenotic_grade <- as.numeric(endoscopic_fibrostenotic_grade)
  histologic_fibrostenotic_grade <- as.numeric(histologic_fibrostenotic_grade)

  if (any(is.na(c(symptoms_frequency, food_impaction_code, hospitalization_eoe, esophageal_perforation, malnutrition_code, persistent_inflammation, endoscopic_inflammatory_grade, peak_eosinophil_count_grade, endoscopic_fibrostenotic_grade, histologic_fibrostenotic_grade)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (symptoms_frequency == 1) score <- score + 1
  else if (symptoms_frequency == 2) score <- score + 2
  else if (symptoms_frequency == 3) score <- score + 4

  if (food_impaction_code == 1) score <- score + 2
  else if (food_impaction_code == 2) score <- score + 4

  if (hospitalization_eoe == 1) score <- score + 4

  if (esophageal_perforation == 1) score <- score + 15

  if (malnutrition_code == 1) score <- score + 15

  if (persistent_inflammation == 1) score <- score + 15

  if (endoscopic_inflammatory_grade == 1) score <- score + 1
  else if (endoscopic_inflammatory_grade == 2) score <- score + 2

  if (peak_eosinophil_count_grade == 1) score <- score + 1
  else if (peak_eosinophil_count_grade == 2) score <- score + 2

  if (endoscopic_fibrostenotic_grade == 1) score <- score + 1
  else if (endoscopic_fibrostenotic_grade == 2) score <- score + 2
  else if (endoscopic_fibrostenotic_grade == 3) score <- score + 15

  if (histologic_fibrostenotic_grade == 1) score <- score + 2

  severity_category <- ""
  if (score < 1) {
    severity_category <- "Inactive"
  } else if (score <= 6) {
    severity_category <- "Mild"
  } else if (score <= 14) {
    severity_category <- "Moderate"
  } else {
    severity_category <- "Severe"
  }

  return(list(
    ISEE_Score = score,
    Severity = severity_category,
    Inputs = list(
      Symptoms = symptoms_frequency,
      Impaction = food_impaction_code,
      Hospitalization = hospitalization_eoe == 1,
      Perforation = esophageal_perforation == 1,
      Malnutrition = malnutrition_code == 1,
      Refractory_Inflammation = persistent_inflammation == 1,
      Endoscopy_Inflammation = endoscopic_inflammatory_grade,
      Histology_Eosinophils = peak_eosinophil_count_grade,
      Endoscopy_Fibrostenosis = endoscopic_fibrostenotic_grade,
      Histology_Fibrosis = histologic_fibrostenotic_grade == 1
    )
  ))
}

itas2010_score <- function(systemic_symptoms, arthralgia, carotidynia, pulse_deficit, bruit, bp_discrepancy, claudication, hypertension, dyspnea, abdominal_pain, cutaneous_lesions, aortic_regurgitation, angina, heart_failure, syncope, tia, renal_failure, stroke, myocardial_infarction, bowel_ischemia, esr_crp_elevated = 0) {

  inputs <- c(systemic_symptoms, arthralgia, carotidynia, pulse_deficit, bruit, bp_discrepancy, claudication, hypertension, dyspnea, abdominal_pain, cutaneous_lesions, aortic_regurgitation, angina, heart_failure, syncope, tia, renal_failure, stroke, myocardial_infarction, bowel_ischemia, esr_crp_elevated)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (systemic_symptoms == 1) score <- score + 1
  if (arthralgia == 1) score <- score + 1
  if (carotidynia == 1) score <- score + 1
  if (pulse_deficit == 1) score <- score + 1
  if (bruit == 1) score <- score + 1
  if (bp_discrepancy == 1) score <- score + 1
  if (claudication == 1) score <- score + 1
  if (hypertension == 1) score <- score + 1
  if (dyspnea == 1) score <- score + 1
  if (abdominal_pain == 1) score <- score + 1
  if (cutaneous_lesions == 1) score <- score + 1

  if (aortic_regurgitation == 1) score <- score + 3
  if (angina == 1) score <- score + 3
  if (heart_failure == 1) score <- score + 3
  if (syncope == 1) score <- score + 3
  if (tia == 1) score <- score + 3
  if (renal_failure == 1) score <- score + 3

  if (stroke == 1) score <- score + 5
  if (myocardial_infarction == 1) score <- score + 5
  if (bowel_ischemia == 1) score <- score + 5

  itas_a_score <- score
  if (esr_crp_elevated == 1) {
    itas_a_score <- score + 1
  }

  disease_activity <- ""
  if (score == 0) {
    disease_activity <- "Inactive Disease"
  } else {
    disease_activity <- "Active Disease (Score >= 1)"
  }

  return(list(
    ITAS2010_Score = score,
    ITAS_A_Score = itas_a_score,
    Disease_Status = disease_activity,
    Inputs = list(
      Systemic = systemic_symptoms == 1,
      Cardiovascular_Signs = sum(carotidynia, pulse_deficit, bruit, bp_discrepancy, claudication, hypertension, aortic_regurgitation, angina, heart_failure, myocardial_infarction),
      Neurologic_Signs = sum(syncope, tia, stroke),
      Other_Signs = sum(arthralgia, dyspnea, abdominal_pain, cutaneous_lesions, renal_failure, bowel_ischemia),
      Acute_Phase_Reactants = esr_crp_elevated == 1
    )
  ))
}

paxlovid_eligibility_check <- function(covid_positive, symptom_onset_days, age_years, weight_kg, has_risk_factors, egfr_ml_min, severe_hepatic_impairment, significant_drug_interactions) {

  covid_positive <- as.numeric(covid_positive)
  symptom_onset_days <- as.numeric(symptom_onset_days)
  age_years <- as.numeric(age_years)
  weight_kg <- as.numeric(weight_kg)
  has_risk_factors <- as.numeric(has_risk_factors)
  egfr_ml_min <- as.numeric(egfr_ml_min)
  severe_hepatic_impairment <- as.numeric(severe_hepatic_impairment)
  significant_drug_interactions <- as.numeric(significant_drug_interactions)

  if (any(is.na(c(covid_positive, symptom_onset_days, age_years, weight_kg, has_risk_factors, egfr_ml_min, severe_hepatic_impairment, significant_drug_interactions)))) {
    stop("Error: All inputs must be numeric.")
  }

  eligibility_status <- ""
  dosing_recommendation <- ""
  reason <- ""

  is_eligible <- TRUE

  if (covid_positive == 0) {
    is_eligible <- FALSE
    reason <- "Negative COVID-19 test result."
  } else if (symptom_onset_days > 5) {
    is_eligible <- FALSE
    reason <- "Symptom onset > 5 days ago."
  } else if (age_years < 12) {
    is_eligible <- FALSE
    reason <- "Patient age < 12 years."
  } else if (weight_kg < 40) {
    is_eligible <- FALSE
    reason <- "Patient weight < 40 kg."
  } else if (has_risk_factors == 0) {
    is_eligible <- FALSE
    reason <- "No high-risk factors for severe COVID-19 progression identified."
  } else if (severe_hepatic_impairment == 1) {
    is_eligible <- FALSE
    reason <- "Severe hepatic impairment (Child-Pugh Class C)."
  } else if (significant_drug_interactions == 1) {
    is_eligible <- FALSE
    reason <- "Contraindicated drug-drug interactions present."
  } else if (egfr_ml_min < 30) {
    is_eligible <- FALSE
    reason <- "Severe renal impairment (eGFR < 30 mL/min)."
  }

  if (is_eligible) {
    eligibility_status <- "Eligible"
    if (egfr_ml_min >= 60) {
      dosing_recommendation <- "Standard Dose: 300 mg nirmatrelvir (two 150 mg tablets) + 100 mg ritonavir (one 100 mg tablet) PO BID for 5 days."
    } else if (egfr_ml_min >= 30) {
      dosing_recommendation <- "Renal Dose: 150 mg nirmatrelvir (one 150 mg tablet) + 100 mg ritonavir (one 100 mg tablet) PO BID for 5 days."
    }
  } else {
    eligibility_status <- "Not Eligible / Contraindicated"
    dosing_recommendation <- "None"
  }

  return(list(
    Status = eligibility_status,
    Recommendation = dosing_recommendation,
    Reason_for_Ineligibility = if (reason != "") reason else "N/A",
    Inputs = list(
      Covid_Positive = covid_positive == 1,
      Symptom_Days = symptom_onset_days,
      Age = age_years,
      Weight = weight_kg,
      High_Risk = has_risk_factors == 1,
      eGFR = egfr_ml_min,
      Severe_Liver_Disease = severe_hepatic_impairment == 1,
      Drug_Interactions = significant_drug_interactions == 1
    )
  ))
}

infant_scalp_score <- function(age_months, hematoma_size, hematoma_location) {

  age_months <- as.numeric(age_months)
  hematoma_size <- tolower(as.character(hematoma_size))
  hematoma_location <- tolower(as.character(hematoma_location))

  if (is.na(age_months)) {
    stop("Error: Age must be numeric.")
  }

  if (!hematoma_size %in% c("none", "small", "medium", "large")) {
    stop("Error: Hematoma size must be 'none', 'small', 'medium', or 'large'.")
  }

  if (!hematoma_location %in% c("frontal", "occipital", "parietal_temporal")) {
    stop("Error: Hematoma location must be 'frontal', 'occipital', or 'parietal_temporal'.")
  }

  score <- 0

  if (age_months < 3) {
    score <- score + 3
  } else if (age_months >= 3 && age_months < 6) {
    score <- score + 2
  } else if (age_months >= 6 && age_months < 12) {
    score <- score + 1
  }

  if (hematoma_size == "small") {
    score <- score + 1
  } else if (hematoma_size == "medium") {
    score <- score + 2
  } else if (hematoma_size == "large") {
    score <- score + 3
  }

  if (hematoma_location == "occipital") {
    score <- score + 1
  } else if (hematoma_location == "parietal_temporal") {
    score <- score + 2
  }

  risk_category <- ""
  recommendation <- ""

  if (score < 4) {
    risk_category <- "Low Risk"
    recommendation <- "Risk of ciTBI is very low. Observation is generally appropriate."
  } else {
    risk_category <- "High Risk"
    recommendation <- "Risk of ciTBI is elevated. Head CT should be considered."
  }

  return(list(
    Infant_Scalp_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age_Months = age_months,
      Size = hematoma_size,
      Location = hematoma_location
    )
  ))
}

ie_mortality_risk_score <- function(age, severe_heart_failure, microorganism, paravalvular_complication, dialysis, prosthetic_valve) {

  age <- as.numeric(age)
  severe_heart_failure <- as.numeric(severe_heart_failure)
  paravalvular_complication <- as.numeric(paravalvular_complication)
  dialysis <- as.numeric(dialysis)
  prosthetic_valve <- as.numeric(prosthetic_valve)
  microorganism <- tolower(as.character(microorganism))

  if (any(is.na(c(age, severe_heart_failure, paravalvular_complication, dialysis, prosthetic_valve)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!microorganism %in% c("viridans_streptococci", "staphylococcus_aureus", "other")) {
    stop("Error: Microorganism must be 'viridans_streptococci', 'staphylococcus_aureus', or 'other'.")
  }

  score <- 0

  if (age >= 45 && age <= 54) {
    score <- score + 3
  } else if (age >= 55 && age <= 64) {
    score <- score + 5
  } else if (age >= 65 && age <= 74) {
    score <- score + 7
  } else if (age >= 75) {
    score <- score + 9
  }

  if (severe_heart_failure == 1) {
    score <- score + 5
  }

  if (microorganism == "staphylococcus_aureus") {
    score <- score + 5
  } else if (microorganism == "other") {
    score <- score + 2
  }

  if (paravalvular_complication == 1) {
    score <- score + 4
  }

  if (dialysis == 1) {
    score <- score + 4
  }

  if (prosthetic_valve == 1) {
    score <- score + 4
  }

  mortality_6mo <- ""
  risk_category <- ""

  if (score <= 6) {
    risk_category <- "Low Risk"
    mortality_6mo <- "4.8%"
  } else if (score <= 14) {
    risk_category <- "Intermediate Risk"
    mortality_6mo <- "17.9%"
  } else {
    risk_category <- "High Risk"
    mortality_6mo <- "43.2%"
  }

  return(list(
    IE_Mortality_Score = score,
    Risk_Category = risk_category,
    Est_6_Month_Mortality = mortality_6mo,
    Inputs = list(
      Age = age,
      Severe_HF = severe_heart_failure == 1,
      Pathogen = microorganism,
      Paravalvular_Comp = paravalvular_complication == 1,
      Dialysis = dialysis == 1,
      Prosthetic_Valve = prosthetic_valve == 1
    )
  ))
}

injury_severity_score <- function(head_neck_ais, face_ais, chest_ais, abdomen_pelvis_ais, extremities_pelvic_girdle_ais, external_ais) {

  ais_scores <- c(head_neck_ais, face_ais, chest_ais, abdomen_pelvis_ais, extremities_pelvic_girdle_ais, external_ais)

  if (any(is.na(ais_scores))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(ais_scores < 0) || any(ais_scores > 6)) {
    stop("Error: AIS scores must be integers between 0 (None) and 6 (Unsurvivable).")
  }

  if (any(ais_scores == 6)) {
    iss <- 75
  } else {
    sorted_scores <- sort(ais_scores, decreasing = TRUE)
    top_three <- sorted_scores[1:3]
    iss <- sum(top_three^2)
  }

  trauma_classification <- ""
  if (iss < 9) {
    trauma_classification <- "Minor"
  } else if (iss <= 15) {
    trauma_classification <- "Moderate"
  } else if (iss <= 24) {
    trauma_classification <- "Serious"
  } else {
    trauma_classification <- "Severe / Critical (Major Trauma)"
  }

  return(list(
    ISS_Score = iss,
    Classification = trauma_classification,
    Inputs = list(
      Head_Neck = head_neck_ais,
      Face = face_ais,
      Chest = chest_ais,
      Abdomen = abdomen_pelvis_ais,
      Extremities = extremities_pelvic_girdle_ais,
      External = external_ais
    )
  ))
}

interchest_score <- function(history_cad, age_risk, effort_pain, palpation_pain, physician_suspicion, pressure_discomfort) {

  history_cad <- as.numeric(history_cad)
  age_risk <- as.numeric(age_risk)
  effort_pain <- as.numeric(effort_pain)
  palpation_pain <- as.numeric(palpation_pain)
  physician_suspicion <- as.numeric(physician_suspicion)
  pressure_discomfort <- as.numeric(pressure_discomfort)

  if (any(is.na(c(history_cad, age_risk, effort_pain, palpation_pain, physician_suspicion, pressure_discomfort)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (history_cad == 1) score <- score + 1
  if (age_risk == 1) score <- score + 1
  if (effort_pain == 1) score <- score + 1
  if (physician_suspicion == 1) score <- score + 1
  if (pressure_discomfort == 1) score <- score + 1

  if (palpation_pain == 1) score <- score - 1

  risk_category <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_category <- "Very Low Risk"
    recommendation <- "Unstable Coronary Artery Disease (UCAD) highly unlikely (NPV 98%). Outpatient non-urgent workup is safe."
  } else if (score == 2) {
    risk_category <- "Low Risk"
    recommendation <- "Further evaluation recommended; typically safe for outpatient referral."
  } else {
    risk_category <- "High Risk"
    recommendation <- "UCAD cannot be reliably excluded (Probability > 10%). Urgent/inpatient evaluation warranted."
  }

  return(list(
    INTERCHEST_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Hx_CAD = history_cad == 1,
      Age_Risk = age_risk == 1,
      Effort_Pain = effort_pain == 1,
      Palpation_Pain = palpation_pain == 1,
      MD_Suspicion = physician_suspicion == 1,
      Pressure_Discomfort = pressure_discomfort == 1
    )
  ))
}

karnofsky_performance_status <- function(score) {

  score <- as.numeric(score)

  if (is.na(score)) {
    stop("Error: Input must be numeric.")
  }

  valid_scores <- seq(0, 100, by = 10)
  if (!score %in% valid_scores) {
    stop("Error: Karnofsky score must be a multiple of 10 between 0 and 100.")
  }

  definition <- ""
  general_category <- ""

  if (score == 100) {
    definition <- "Normal no complaints; no evidence of disease."
    general_category <- "Able to carry on normal activity and to work; no special care needed."
  } else if (score == 90) {
    definition <- "Able to carry on normal activity; minor signs or symptoms of disease."
    general_category <- "Able to carry on normal activity and to work; no special care needed."
  } else if (score == 80) {
    definition <- "Normal activity with effort; some signs or symptoms of disease."
    general_category <- "Able to carry on normal activity and to work; no special care needed."
  } else if (score == 70) {
    definition <- "Cares for self; unable to carry on normal activity or to do active work."
    general_category <- "Unable to work; able to live at home and care for most personal needs; varying amount of assistance needed."
  } else if (score == 60) {
    definition <- "Requires occasional assistance, but is able to care for most of his personal needs."
    general_category <- "Unable to work; able to live at home and care for most personal needs; varying amount of assistance needed."
  } else if (score == 50) {
    definition <- "Requires considerable assistance and frequent medical care."
    general_category <- "Unable to work; able to live at home and care for most personal needs; varying amount of assistance needed."
  } else if (score == 40) {
    definition <- "Disabled; requires special care and assistance."
    general_category <- "Unable to care for self; requires equivalent of institutional or hospital care; disease may be progressing rapidly."
  } else if (score == 30) {
    definition <- "Severely disabled; hospital admission is indicated although death not imminent."
    general_category <- "Unable to care for self; requires equivalent of institutional or hospital care; disease may be progressing rapidly."
  } else if (score == 20) {
    definition <- "Very sick; hospital admission necessary; active supportive treatment necessary."
    general_category <- "Unable to care for self; requires equivalent of institutional or hospital care; disease may be progressing rapidly."
  } else if (score == 10) {
    definition <- "Moribund; fatal processes progressing rapidly."
    general_category <- "Unable to care for self; requires equivalent of institutional or hospital care; disease may be progressing rapidly."
  } else {
    definition <- "Dead."
    general_category <- "Dead."
  }

  return(list(
    KPS_Score = score,
    Definition = definition,
    Functional_Category = general_category
  ))
}

kawasaki_disease_criteria <- function(fever_ge_5_days, conjunctivitis, oral_mucosal_changes, extremity_changes, polymorphous_rash, cervical_lymphadenopathy) {

  fever_ge_5_days <- as.numeric(fever_ge_5_days)
  conjunctivitis <- as.numeric(conjunctivitis)
  oral_mucosal_changes <- as.numeric(oral_mucosal_changes)
  extremity_changes <- as.numeric(extremity_changes)
  polymorphous_rash <- as.numeric(polymorphous_rash)
  cervical_lymphadenopathy <- as.numeric(cervical_lymphadenopathy)

  if (any(is.na(c(fever_ge_5_days, conjunctivitis, oral_mucosal_changes, extremity_changes, polymorphous_rash, cervical_lymphadenopathy)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  clinical_signs_count <- conjunctivitis + oral_mucosal_changes + extremity_changes + polymorphous_rash + cervical_lymphadenopathy

  result <- ""
  recommendation <- ""

  if (fever_ge_5_days == 1) {
    if (clinical_signs_count >= 4) {
      result <- "Complete Kawasaki Disease"
      recommendation <- "Diagnostic criteria met. Initiate treatment (IVIG + Aspirin) and obtain Echocardiogram."
    } else {
      result <- "Incomplete Kawasaki Disease (Possible)"
      recommendation <- "Criteria for Complete KD not met. If clinical suspicion remains high, evaluate for Incomplete KD using supplemental laboratory criteria (CRP, ESR, etc.) per AHA guidelines."
    }
  } else {
    result <- "Criteria Not Met"
    recommendation <- "Fever duration < 5 days. Diagnostic criteria require fever for at least 5 days. Monitor or consider other diagnoses."
  }

  return(list(
    Diagnosis_Result = result,
    Criteria_Count = clinical_signs_count,
    Recommendation = recommendation,
    Inputs = list(
      Fever_5d = fever_ge_5_days == 1,
      Conjunctivitis = conjunctivitis == 1,
      Oral_Changes = oral_mucosal_changes == 1,
      Extremity_Changes = extremity_changes == 1,
      Rash = polymorphous_rash == 1,
      Lymphadenopathy = cervical_lymphadenopathy == 1
    )
  ))
}

khorana_vte_score <- function(cancer_site_risk, platelet_count, hemoglobin, wbc_count, bmi) {

  cancer_site_risk <- tolower(as.character(cancer_site_risk))
  platelet_count <- as.numeric(platelet_count)
  hemoglobin <- as.numeric(hemoglobin)
  wbc_count <- as.numeric(wbc_count)
  bmi <- as.numeric(bmi)

  if (any(is.na(c(platelet_count, hemoglobin, wbc_count, bmi)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  if (!cancer_site_risk %in% c("very_high", "high", "low")) {
    stop("Error: Cancer site risk must be 'very_high', 'high', or 'low'.")
  }

  score <- 0

  if (cancer_site_risk == "very_high") {
    score <- score + 2
  } else if (cancer_site_risk == "high") {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (platelet_count >= 350) {
    score <- score + 1
  }

  if (hemoglobin < 10) {
    score <- score + 1
  }

  if (wbc_count > 11) {
    score <- score + 1
  }

  if (bmi >= 35) {
    score <- score + 1
  }

  risk_class <- ""
  vte_risk <- ""

  if (score == 0) {
    risk_class <- "Low Risk"
    vte_risk <- "0.8 - 3%"
  } else if (score == 1 || score == 2) {
    risk_class <- "Intermediate Risk"
    vte_risk <- "1.8 - 8.4%"
  } else {
    risk_class <- "High Risk"
    vte_risk <- "7.1 - 41%"
  }

  return(list(
    Khorana_Score = score,
    Risk_Classification = risk_class,
    Estimated_VTE_Risk = vte_risk,
    Inputs = list(
      Site_Risk = cancer_site_risk,
      Plt_ge_350 = platelet_count >= 350,
      Hb_lt_10 = hemoglobin < 10,
      WBC_gt_11 = wbc_count > 11,
      BMI_ge_35 = bmi >= 35
    )
  ))
}

killip_classification <- function(class_grade) {

  class_grade <- as.numeric(class_grade)

  if (is.na(class_grade) || !class_grade %in% 1:4) {
    stop("Error: Killip Class must be an integer between 1 and 4.")
  }

  description <- ""
  mortality <- ""

  if (class_grade == 1) {
    description <- "No clinical signs of heart failure."
    mortality <- "6%"
  } else if (class_grade == 2) {
    description <- "Rales in the lungs, S3 gallop, and elevated jugular venous pressure."
    mortality <- "17%"
  } else if (class_grade == 3) {
    description <- "Frank pulmonary edema (acute)."
    mortality <- "38%"
  } else {
    description <- "Cardiogenic shock (systolic BP < 90 mmHg, oliguria, cyanosis, sweating)."
    mortality <- "81%"
  }

  return(list(
    Killip_Class = paste0("Class ", as.roman(class_grade)),
    Clinical_Findings = description,
    Estimated_30_Day_Mortality = mortality
  ))
}

kinetic_egfr <- function(initial_creatinine, final_creatinine, time_hours, baseline_creatinine, age, sex, weight_kg) {

  initial_creatinine <- as.numeric(initial_creatinine)
  final_creatinine <- as.numeric(final_creatinine)
  time_hours <- as.numeric(time_hours)
  baseline_creatinine <- as.numeric(baseline_creatinine)
  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)
  sex <- tolower(as.character(sex))

  if (any(is.na(c(initial_creatinine, final_creatinine, time_hours, baseline_creatinine, age, weight_kg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (time_hours <= 0) {
    stop("Error: Time interval must be positive.")
  }

  kappa <- if (sex == "female") 0.7 else 0.9
  alpha <- if (sex == "female") -0.241 else -0.302
  sex_factor <- if (sex == "female") 1.012 else 1.0

  scr_k_ratio <- baseline_creatinine / kappa

  baseline_egfr <- 142 * (min(scr_k_ratio, 1) ^ alpha) * (max(scr_k_ratio, 1) ^ -1.200) * (0.9938 ^ age) * sex_factor

  mean_creatinine <- (initial_creatinine + final_creatinine) / 2

  delta_creatinine <- final_creatinine - initial_creatinine

  vol_distribution_dl <- weight_kg * 0.6 * 10

  generation_rate_mg_min <- (baseline_egfr * baseline_creatinine) / 100

  accumulation_rate_mg_min <- (delta_creatinine * vol_distribution_dl) / (time_hours * 60)

  k_egfr <- ((generation_rate_mg_min - accumulation_rate_mg_min) / mean_creatinine) * 100

  return(list(
    Kinetic_eGFR = round(k_egfr, 1),
    Baseline_eGFR = round(baseline_egfr, 1),
    Inputs = list(
      Initial_Cr = initial_creatinine,
      Final_Cr = final_creatinine,
      Time_hr = time_hours,
      Baseline_Cr = baseline_creatinine,
      Age = age,
      Sex = sex,
      Weight = weight_kg
    )
  ))
}

kings_college_criteria_acetaminophen <- function(arterial_ph, inr, creatinine, encephalopathy_grade, creatinine_units = "mg/dL") {

  arterial_ph <- as.numeric(arterial_ph)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)
  encephalopathy_grade <- as.numeric(encephalopathy_grade)

  if (any(is.na(c(arterial_ph, inr, creatinine, encephalopathy_grade)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (encephalopathy_grade < 0 || encephalopathy_grade > 4) {
    stop("Error: Encephalopathy grade must be between 0 and 4.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  criteria_met <- FALSE
  met_criteria_list <- c()

  if (arterial_ph < 7.3) {
    criteria_met <- TRUE
    met_criteria_list <- c(met_criteria_list, "Arterial pH < 7.3")
  }

  triad_condition <- (inr > 6.5 && creat_mg_dl > 3.4 && encephalopathy_grade >= 3)

  if (triad_condition) {
    criteria_met <- TRUE
    met_criteria_list <- c(met_criteria_list, "Triad: INR > 6.5, Creatinine > 3.4 mg/dL, Grade III/IV Encephalopathy")
  }

  result <- ""
  recommendation <- ""

  if (criteria_met) {
    result <- "Criteria Met"
    recommendation <- "High mortality risk without liver transplantation. Early referral to a transplant center is strongly recommended."
  } else {
    result <- "Criteria Not Met"
    recommendation <- "Monitor closely. Does not currently meet criteria for urgent transplant listing based on KCC."
  }

  return(list(
    Result = result,
    Recommendation = recommendation,
    Criteria_Fulfilled = if (length(met_criteria_list) > 0) met_criteria_list else "None",
    Inputs = list(
      pH = arterial_ph,
      INR = inr,
      Creatinine_mg_dL = round(creat_mg_dl, 2),
      Encephalopathy_Grade = encephalopathy_grade
    )
  ))
}

kocher_septic_arthritis <- function(non_weight_bearing, temp_gt_38_5, esr_gt_40, wbc_gt_12000) {

  non_weight_bearing <- as.numeric(non_weight_bearing)
  temp_gt_38_5 <- as.numeric(temp_gt_38_5)
  esr_gt_40 <- as.numeric(esr_gt_40)
  wbc_gt_12000 <- as.numeric(wbc_gt_12000)

  if (any(is.na(c(non_weight_bearing, temp_gt_38_5, esr_gt_40, wbc_gt_12000)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (non_weight_bearing == 1) {
    score <- score + 1
  }

  if (temp_gt_38_5 == 1) {
    score <- score + 1
  }

  if (esr_gt_40 == 1) {
    score <- score + 1
  }

  if (wbc_gt_12000 == 1) {
    score <- score + 1
  }

  probability <- ""

  if (score == 0) {
    probability <- "< 0.2%"
  } else if (score == 1) {
    probability <- "3%"
  } else if (score == 2) {
    probability <- "40%"
  } else if (score == 3) {
    probability <- "93%"
  } else {
    probability <- "99%"
  }

  return(list(
    Kocher_Score = score,
    Septic_Arthritis_Probability = probability,
    Inputs = list(
      Non_Weight_Bearing = non_weight_bearing == 1,
      Fever_gt_38_5 = temp_gt_38_5 == 1,
      ESR_gt_40 = esr_gt_40 == 1,
      WBC_gt_12k = wbc_gt_12000 == 1
    )
  ))
}

kruis_ibs_score <- function(symptoms_triad, odd_pain_description, alternating_bowel_habits, duration_gt_2yr, abdominal_mass, leukocytosis, elevated_esr, positive_fob) {

  symptoms_triad <- as.numeric(symptoms_triad)
  odd_pain_description <- as.numeric(odd_pain_description)
  alternating_bowel_habits <- as.numeric(alternating_bowel_habits)
  duration_gt_2yr <- as.numeric(duration_gt_2yr)
  abdominal_mass <- as.numeric(abdominal_mass)
  leukocytosis <- as.numeric(leukocytosis)
  elevated_esr <- as.numeric(elevated_esr)
  positive_fob <- as.numeric(positive_fob)

  if (any(is.na(c(symptoms_triad, odd_pain_description, alternating_bowel_habits, duration_gt_2yr, abdominal_mass, leukocytosis, elevated_esr, positive_fob)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (symptoms_triad == 1) {
    score <- score + 34
  }

  if (odd_pain_description == 0) {
    score <- score + 23
  }

  if (alternating_bowel_habits == 1) {
    score <- score + 14
  }

  if (duration_gt_2yr == 1) {
    score <- score + 16
  }

  if (abdominal_mass == 0) {
    score <- score + 16
  }

  if (leukocytosis == 0) {
    score <- score + 16
  }

  if (elevated_esr == 0) {
    score <- score + 13
  }

  if (positive_fob == 0) {
    score <- score + 12
  }

  diagnosis <- ""
  if (score > 44) {
    diagnosis <- "Diagnosed as Irritable Bowel Syndrome (IBS)"
  } else {
    diagnosis <- "Possible Organic Pathology (IBS Diagnosis Not Supported)"
  }

  return(list(
    Kruis_Score = score,
    Result = diagnosis,
    Inputs = list(
      Symptoms_Triad = symptoms_triad == 1,
      Odd_Pain_Desc = odd_pain_description == 1,
      Alternating_Habits = alternating_bowel_habits == 1,
      Duration_gt_2yr = duration_gt_2yr == 1,
      Mass_Present = abdominal_mass == 1,
      Leukocytosis = leukocytosis == 1,
      High_ESR = elevated_esr == 1,
      Positive_FOBT = positive_fob == 1
    )
  ))
}

kt_v_dialysis <- function(pre_bun, post_bun, duration_hours, uf_volume_liters, post_weight_kg) {

  if (any(is.na(c(pre_bun, post_bun, duration_hours, uf_volume_liters, post_weight_kg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(pre_bun, post_bun, duration_hours, post_weight_kg) <= 0)) {
    stop("Error: BUN, Duration, and Weight must be positive.")
  }

  if (post_bun >= pre_bun) {
    stop("Error: Post-dialysis BUN must be lower than Pre-dialysis BUN.")
  }

  r_value <- post_bun / pre_bun

  kt_v <- -log(r_value - 0.008 * duration_hours) + (4 - 3.5 * r_value) * (uf_volume_liters / post_weight_kg)

  interpretation <- ""
  if (kt_v >= 1.2) {
    interpretation <- "Adequate Dialysis (Target >= 1.2 per session)"
  } else {
    interpretation <- "Inadequate Dialysis (Target < 1.2)"
  }

  return(list(
    Kt_V_Score = round(kt_v, 2),
    Interpretation = interpretation,
    Inputs = list(
      Pre_BUN = pre_bun,
      Post_BUN = post_bun,
      Duration_Hrs = duration_hours,
      UF_Volume_L = uf_volume_liters,
      Post_Weight_Kg = post_weight_kg
    )
  ))
}

lace_readmission_index <- function(length_of_stay_days, acute_admission, charlson_comorbidity_index, ed_visits_6months) {

  length_of_stay_days <- as.numeric(length_of_stay_days)
  acute_admission <- as.numeric(acute_admission)
  charlson_comorbidity_index <- as.numeric(charlson_comorbidity_index)
  ed_visits_6months <- as.numeric(ed_visits_6months)

  if (any(is.na(c(length_of_stay_days, acute_admission, charlson_comorbidity_index, ed_visits_6months)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (length_of_stay_days < 1) {
    score <- score + 0
  } else if (length_of_stay_days >= 1 && length_of_stay_days <= 3) {
    score <- score + length_of_stay_days
  } else if (length_of_stay_days >= 4 && length_of_stay_days <= 6) {
    score <- score + 4
  } else if (length_of_stay_days >= 7 && length_of_stay_days <= 13) {
    score <- score + 5
  } else {
    score <- score + 7
  }

  if (acute_admission == 1) {
    score <- score + 3
  }

  if (charlson_comorbidity_index <= 3) {
    score <- score + charlson_comorbidity_index
  } else {
    score <- score + 5
  }

  if (ed_visits_6months <= 3) {
    score <- score + ed_visits_6months
  } else {
    score <- score + 4
  }

  risk_category <- ""
  readmission_risk <- ""

  if (score <= 4) {
    risk_category <- "Low Risk"
    readmission_risk <- "< 5%"
  } else if (score <= 9) {
    risk_category <- "Moderate Risk"
    readmission_risk <- "5 - 12%"
  } else {
    risk_category <- "High Risk"
    readmission_risk <- "> 12%"
  }

  return(list(
    LACE_Score = score,
    Risk_Category = risk_category,
    Est_30_Day_Readmission_Risk = readmission_risk,
    Inputs = list(
      LOS_Days = length_of_stay_days,
      Acute_Admission = acute_admission == 1,
      Charlson_Score = charlson_comorbidity_index,
      ED_Visits_6mo = ed_visits_6months
    )
  ))
}

lansky_play_performance_scale <- function(score) {

  score <- as.numeric(score)

  if (is.na(score)) {
    stop("Error: Input must be numeric.")
  }

  valid_scores <- seq(0, 100, by = 10)
  if (!score %in% valid_scores) {
    stop("Error: Lansky score must be a multiple of 10 between 0 and 100.")
  }

  definition <- ""
  restriction_level <- ""

  if (score == 100) {
    definition <- "Fully active, normal."
    restriction_level <- "No Restriction"
  } else if (score == 90) {
    definition <- "Minor restrictions in physically strenuous activity."
    restriction_level <- "Mild Restriction"
  } else if (score == 80) {
    definition <- "Active, but tires more quickly."
    restriction_level <- "Mild Restriction"
  } else if (score == 70) {
    definition <- "Both greater restriction of, and less time spent in, active play."
    restriction_level <- "Moderate Restriction"
  } else if (score == 60) {
    definition <- "Up and around, but minimal active play; keeps busy with quieter activities."
    restriction_level <- "Moderate Restriction"
  } else if (score == 50) {
    definition <- "Gets dressed but lies around much of the day, no active play but able to participate in all quiet play and activities."
    restriction_level <- "Moderate Restriction"
  } else if (score == 40) {
    definition <- "Mostly in bed; participates in quiet activities."
    restriction_level <- "Severe Restriction"
  } else if (score == 30) {
    definition <- "In bed; needs assistance even for quiet play."
    restriction_level <- "Severe Restriction"
  } else if (score == 20) {
    definition <- "Often sleeping; play entirely limited to very passive activities."
    restriction_level <- "Severe Restriction"
  } else if (score == 10) {
    definition <- "No play; does not get out of bed."
    restriction_level <- "Very Severe Restriction"
  } else {
    definition <- "Unresponsive."
    restriction_level <- "Unresponsive / Moribund"
  }

  return(list(
    Lansky_Score = score,
    Definition = definition,
    Restriction_Level = restriction_level
  ))
}

calculate_ldl <- function(total_cholesterol, hdl_cholesterol, triglycerides, units = "mg/dL") {

  total_cholesterol <- as.numeric(total_cholesterol)
  hdl_cholesterol <- as.numeric(hdl_cholesterol)
  triglycerides <- as.numeric(triglycerides)

  if (any(is.na(c(total_cholesterol, hdl_cholesterol, triglycerides)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (units == "mmol/L") {
    # Friedewald equation for mmol/L: LDL = Total - HDL - (Triglycerides / 2.2)
    ldl <- total_cholesterol - hdl_cholesterol - (triglycerides / 2.2)
  } else {
    # Friedewald equation for mg/dL: LDL = Total - HDL - (Triglycerides / 5)
    ldl <- total_cholesterol - hdl_cholesterol - (triglycerides / 5)
  }

  validity_note <- ""
  if (units == "mg/dL" && triglycerides > 400) {
    validity_note <- "Note: Friedewald equation is not valid when Triglycerides > 400 mg/dL."
  } else if (units == "mmol/L" && triglycerides > 4.5) {
    validity_note <- "Note: Friedewald equation is not valid when Triglycerides > 4.5 mmol/L."
  }

  return(list(
    LDL_Cholesterol = round(ldl, 2),
    Validity = validity_note,
    Inputs = list(
      Total_Chol = total_cholesterol,
      HDL_Chol = hdl_cholesterol,
      Triglycerides = triglycerides,
      Units = units
    )
  ))
}

leiden_arthritis_prediction <- function(age_years, sex, distribution_small_hands_feet, distribution_symmetric, morning_stiffness_severity, tender_joint_count, swollen_joint_count, crp_mg_dl, rf_positive, acpa_positive) {

  age_years <- as.numeric(age_years)
  sex <- tolower(as.character(sex))
  distribution_small_hands_feet <- as.numeric(distribution_small_hands_feet)
  distribution_symmetric <- as.numeric(distribution_symmetric)
  morning_stiffness_severity <- as.numeric(morning_stiffness_severity)
  tender_joint_count <- as.numeric(tender_joint_count)
  swollen_joint_count <- as.numeric(swollen_joint_count)
  crp_mg_dl <- as.numeric(crp_mg_dl)
  rf_positive <- as.numeric(rf_positive)
  acpa_positive <- as.numeric(acpa_positive)

  if (any(is.na(c(age_years, distribution_small_hands_feet, distribution_symmetric, morning_stiffness_severity, tender_joint_count, swollen_joint_count, crp_mg_dl, rf_positive, acpa_positive)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  score <- score + (0.02 * age_years)

  if (sex == "female") {
    score <- score + 1
  }

  if (distribution_small_hands_feet == 1) {
    score <- score + 0.5
  }

  if (distribution_symmetric == 1) {
    score <- score + 0.5
  }

  if (morning_stiffness_severity >= 1) {
    score <- score + 1
  }

  if (tender_joint_count >= 4) {
    score <- score + 0.5
  } else if (tender_joint_count >= 1) {
    score <- score + 0.5 * (tender_joint_count / 3)
  }

  if (swollen_joint_count >= 4) {
    score <- score + 0.5
  } else if (swollen_joint_count >= 1) {
    score <- score + 0.5 * (swollen_joint_count / 3)
  }

  if (crp_mg_dl >= 5) {
    score <- score + 1.5
  }

  if (rf_positive == 1) {
    score <- score + 1
  }

  if (acpa_positive == 1) {
    score <- score + 2
  }

  risk_category <- ""
  ra_progression_prob <- ""

  if (score < 6) {
    risk_category <- "Low Risk"
    ra_progression_prob <- "< 30%"
  } else if (score >= 6 && score < 8) {
    risk_category <- "Intermediate Risk"
    ra_progression_prob <- "30 - 69%"
  } else {
    risk_category <- "High Risk"
    ra_progression_prob <- ">= 70%"
  }

  return(list(
    Leiden_Score = round(score, 1),
    Risk_Category = risk_category,
    Progression_Probability_1yr = ra_progression_prob,
    Inputs = list(
      Age = age_years,
      Sex = sex,
      Small_Joints = distribution_small_hands_feet == 1,
      Symmetric = distribution_symmetric == 1,
      Morning_Stiffness = morning_stiffness_severity >= 1,
      Tender_Count = tender_joint_count,
      Swollen_Count = swollen_joint_count,
      CRP_High = crp_mg_dl >= 5,
      RF_Pos = rf_positive == 1,
      ACPA_Pos = acpa_positive == 1
    )
  ))
}

lent_prognostic_score <- function(pleural_ldh, ecog_score, serum_nlr, tumor_type) {

  pleural_ldh <- as.numeric(pleural_ldh)
  ecog_score <- as.numeric(ecog_score)
  serum_nlr <- as.numeric(serum_nlr)
  tumor_type <- tolower(as.character(tumor_type))

  if (any(is.na(c(pleural_ldh, ecog_score, serum_nlr)))) {
    stop("Error: LDH, ECOG, and NLR must be numeric.")
  }

  if (ecog_score < 0 || ecog_score > 4) {
    stop("Error: ECOG score must be between 0 and 4.")
  }

  valid_tumors <- c("mesothelioma", "hematologic", "breast", "gynecologic", "renal", "sarcoma", "lung", "other")
  if (!tumor_type %in% valid_tumors) {
    stop("Error: Tumor type must be one of: 'mesothelioma', 'hematologic', 'breast', 'gynecologic', 'renal', 'sarcoma', 'lung', or 'other'.")
  }

  score <- 0

  if (pleural_ldh > 1500) {
    score <- score + 1
  }

  if (ecog_score == 0) {
    score <- score + 0
  } else if (ecog_score == 1) {
    score <- score + 1
  } else if (ecog_score == 2) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (serum_nlr > 9) {
    score <- score + 1
  }

  if (tumor_type == "mesothelioma" || tumor_type == "hematologic") {
    score <- score + 0
  } else if (tumor_type == "breast" || tumor_type == "gynecologic" || tumor_type == "renal" || tumor_type == "sarcoma") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  risk_group <- ""
  median_survival <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    median_survival <- "319 days (10.5 months)"
  } else if (score <= 4) {
    risk_group <- "Moderate Risk"
    median_survival <- "130 days (4.3 months)"
  } else {
    risk_group <- "High Risk"
    median_survival <- "44 days (1.5 months)"
  }

  return(list(
    LENT_Score = score,
    Risk_Category = risk_group,
    Median_Survival = median_survival,
    Inputs = list(
      Pleural_LDH_gt_1500 = pleural_ldh > 1500,
      ECOG = ecog_score,
      NLR_gt_9 = serum_nlr > 9,
      Tumor_Type = tumor_type
    )
  ))
}

licurse_renal_ultrasound_score <- function(age_ge_65, male_sex, history_hydronephrosis, history_chf) {

  age_ge_65 <- as.numeric(age_ge_65)
  male_sex <- as.numeric(male_sex)
  history_hydronephrosis <- as.numeric(history_hydronephrosis)
  history_chf <- as.numeric(history_chf)

  if (any(is.na(c(age_ge_65, male_sex, history_hydronephrosis, history_chf)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (age_ge_65 == 1) {
    score <- score + 1
  }

  if (male_sex == 1) {
    score <- score + 1
  }

  if (history_hydronephrosis == 1) {
    score <- score + 1
  }

  if (history_chf == 1) {
    score <- score - 1
  }

  risk_category <- ""
  hydronephrosis_risk <- ""
  recommendation <- ""

  if (score <= 0) {
    risk_category <- "Low Risk"
    hydronephrosis_risk <- "0.5%"
    recommendation <- "Renal ultrasound likely not indicated (Very low yield)."
  } else if (score == 1) {
    risk_category <- "Intermediate Risk"
    hydronephrosis_risk <- "6%"
    recommendation <- "Clinical judgment required."
  } else if (score == 2) {
    risk_category <- "High Risk"
    hydronephrosis_risk <- "16%"
    recommendation <- "Renal ultrasound indicated."
  } else {
    risk_category <- "Very High Risk"
    hydronephrosis_risk <- "26%"
    recommendation <- "Renal ultrasound indicated."
  }

  return(list(
    Licurse_Score = score,
    Risk_Category = risk_category,
    Prob_Hydronephrosis = hydronephrosis_risk,
    Recommendation = recommendation,
    Inputs = list(
      Age_ge_65 = age_ge_65 == 1,
      Male = male_sex == 1,
      Hx_Hydronephrosis = history_hydronephrosis == 1,
      Hx_CHF = history_chf == 1
    )
  ))
}

lights_criteria_effusion <- function(pleural_protein, serum_protein, pleural_ldh, serum_ldh, serum_ldh_uln) {

  pleural_protein <- as.numeric(pleural_protein)
  serum_protein <- as.numeric(serum_protein)
  pleural_ldh <- as.numeric(pleural_ldh)
  serum_ldh <- as.numeric(serum_ldh)
  serum_ldh_uln <- as.numeric(serum_ldh_uln)

  if (any(is.na(c(pleural_protein, serum_protein, pleural_ldh, serum_ldh, serum_ldh_uln)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (serum_protein == 0 || serum_ldh == 0) {
    stop("Error: Serum values cannot be zero.")
  }

  ratio_protein <- pleural_protein / serum_protein
  ratio_ldh <- pleural_ldh / serum_ldh
  ldh_limit <- (2/3) * serum_ldh_uln

  is_exudate <- FALSE
  reasons <- c()

  if (ratio_protein > 0.5) {
    is_exudate <- TRUE
    reasons <- c(reasons, paste0("Pleural/Serum Protein Ratio > 0.5 (", round(ratio_protein, 2), ")"))
  }

  if (ratio_ldh > 0.6) {
    is_exudate <- TRUE
    reasons <- c(reasons, paste0("Pleural/Serum LDH Ratio > 0.6 (", round(ratio_ldh, 2), ")"))
  }

  if (pleural_ldh > ldh_limit) {
    is_exudate <- TRUE
    reasons <- c(reasons, paste0("Pleural LDH > 2/3 Upper Limit of Normal (", pleural_ldh, " > ", round(ldh_limit, 1), ")"))
  }

  classification <- ""
  if (is_exudate) {
    classification <- "Exudative"
  } else {
    classification <- "Transudative"
    reasons <- c("None of the exudative criteria were met.")
  }

  return(list(
    Classification = classification,
    Criteria_Met = reasons,
    Calculated_Values = list(
      Protein_Ratio = round(ratio_protein, 2),
      LDH_Ratio = round(ratio_ldh, 2),
      Pleural_LDH_Threshold = round(ldh_limit, 1)
    )
  ))
}

lille_model_score <- function(age, renal_insufficiency, albumin, bilirubin_day0, bilirubin_day7, pt_seconds, bilirubin_units = "mg/dL", albumin_units = "g/dL") {

  age <- as.numeric(age)
  renal_insufficiency <- as.numeric(renal_insufficiency)
  albumin <- as.numeric(albumin)
  bilirubin_day0 <- as.numeric(bilirubin_day0)
  bilirubin_day7 <- as.numeric(bilirubin_day7)
  pt_seconds <- as.numeric(pt_seconds)

  if (any(is.na(c(age, renal_insufficiency, albumin, bilirubin_day0, bilirubin_day7, pt_seconds)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_day0_umol <- bilirubin_day0
  bili_day7_umol <- bilirubin_day7

  if (bilirubin_units == "mg/dL") {
    bili_day0_umol <- bilirubin_day0 * 17.1
    bili_day7_umol <- bilirubin_day7 * 17.1
  }

  albumin_g_l <- albumin
  if (albumin_units == "g/dL") {
    albumin_g_l <- albumin * 10
  }

  r_score <- 3.19 - (0.101 * age) + (0.147 * albumin_g_l) + (0.0165 * (bili_day0_umol - bili_day7_umol)) - (0.206 * renal_insufficiency) - (0.0065 * bili_day0_umol) - (0.0096 * pt_seconds)

  lille_prob <- exp(-r_score) / (1 + exp(-r_score))

  interpretation <- ""
  if (lille_prob >= 0.45) {
    interpretation <- "Non-responder. 6-month survival approx. 25%. Consider stopping corticosteroids."
  } else {
    interpretation <- "Responder. 6-month survival approx. 85%. Continue corticosteroids."
  }

  return(list(
    Lille_Model_Score = round(lille_prob, 4),
    Interpretation = interpretation,
    Inputs = list(
      Age = age,
      Renal_Insufficiency = renal_insufficiency == 1,
      Albumin_g_L = albumin_g_l,
      Bili_Day0_umol_L = round(bili_day0_umol, 1),
      Bili_Day7_umol_L = round(bili_day7_umol, 1),
      PT_Seconds = pt_seconds
    )
  ))
}

liver_decompensation_risk_hcc <- function(portal_hypertension, extent_hepatectomy, meld_score) {

  portal_hypertension <- as.numeric(portal_hypertension)
  extent_hepatectomy <- tolower(as.character(extent_hepatectomy))
  meld_score <- as.numeric(meld_score)

  if (any(is.na(c(portal_hypertension, meld_score)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!extent_hepatectomy %in% c("minor", "major")) {
    stop("Error: Extent of hepatectomy must be 'minor' (<=2 segments) or 'major' (>=3 segments).")
  }

  risk_category <- ""
  decompensation_rate <- ""
  mortality_rate <- ""

  if (portal_hypertension == 1 && extent_hepatectomy == "major") {
    risk_category <- "High Risk"
    decompensation_rate <- "High (> 30%)"
    mortality_rate <- "25.0%"
  } else if (portal_hypertension == 1 && extent_hepatectomy == "minor") {
    risk_category <- "Intermediate Risk"
    decompensation_rate <- "28.6%"
    mortality_rate <- "9.0%"
  } else if (portal_hypertension == 0 && extent_hepatectomy == "major") {
    risk_category <- "Intermediate Risk"
    decompensation_rate <- "28.6%"
    mortality_rate <- "9.0%"
  } else {
    if (meld_score > 9) {
      risk_category <- "Intermediate Risk"
      decompensation_rate <- "28.6%"
      mortality_rate <- "9.0%"
    } else {
      risk_category <- "Low Risk"
      decompensation_rate <- "4.9%"
      mortality_rate <- "4.4%"
    }
  }

  return(list(
    Risk_Category = risk_category,
    Est_Liver_Decompensation_Rate = decompensation_rate,
    Est_Liver_Related_Mortality = mortality_rate,
    Inputs = list(
      Portal_Hypertension = portal_hypertension == 1,
      Extent_Hepatectomy = extent_hepatectomy,
      MELD_Score = meld_score
    )
  ))
}

local_anesthetic_dosing <- function(weight_kg, anesthetic_type, concentration_percent, with_epinephrine = 0) {

  weight_kg <- as.numeric(weight_kg)
  concentration_percent <- as.numeric(concentration_percent)
  with_epinephrine <- as.numeric(with_epinephrine)
  anesthetic_type <- tolower(as.character(anesthetic_type))

  if (any(is.na(c(weight_kg, concentration_percent)))) {
    stop("Error: Weight and Concentration must be numeric.")
  }

  if (weight_kg <= 0 || concentration_percent <= 0) {
    stop("Error: Inputs must be positive.")
  }

  concentration_mg_ml <- concentration_percent * 10

  max_dose_mg_kg <- 0
  max_total_dose_mg <- Inf

  if (anesthetic_type == "lidocaine") {
    if (with_epinephrine == 1) {
      max_dose_mg_kg <- 7.0
      max_total_dose_mg <- 500
    } else {
      max_dose_mg_kg <- 4.5
      max_total_dose_mg <- 300
    }
  } else if (anesthetic_type == "bupivacaine") {
    max_dose_mg_kg <- 2.5
    max_total_dose_mg <- 175
    if (with_epinephrine == 1) {
      max_total_dose_mg <- 225
    }
  } else if (anesthetic_type == "mepivacaine") {
    if (with_epinephrine == 1) {
      max_dose_mg_kg <- 7.0
      max_total_dose_mg <- 550
    } else {
      max_dose_mg_kg <- 4.5
      max_total_dose_mg <- 400
    }
  } else if (anesthetic_type == "ropivacaine") {
    max_dose_mg_kg <- 3.0
    max_total_dose_mg <- 200
  } else if (anesthetic_type == "procaine") {
    max_dose_mg_kg <- 7.0
    max_total_dose_mg <- 500
  } else if (anesthetic_type == "chloroprocaine") {
    if (with_epinephrine == 1) {
      max_dose_mg_kg <- 14.0
      max_total_dose_mg <- 1000
    } else {
      max_dose_mg_kg <- 11.0
      max_total_dose_mg <- 800
    }
  } else if (anesthetic_type == "tetracaine") {
    max_dose_mg_kg <- 1.5
    max_total_dose_mg <- 20
  } else {
    stop("Error: Anesthetic type not supported. Try 'lidocaine', 'bupivacaine', 'mepivacaine', 'ropivacaine', 'procaine', 'chloroprocaine', or 'tetracaine'.")
  }

  calculated_max_mg <- weight_kg * max_dose_mg_kg

  final_max_mg <- min(calculated_max_mg, max_total_dose_mg)

  max_volume_ml <- final_max_mg / concentration_mg_ml

  return(list(
    Anesthetic = anesthetic_type,
    Max_Dose_mg = round(final_max_mg, 1),
    Max_Volume_mL = round(max_volume_ml, 1),
    Constraint = if(calculated_max_mg > max_total_dose_mg) "Capped by absolute max dose" else "Based on weight",
    Inputs = list(
      Weight_kg = weight_kg,
      Concentration_Pct = concentration_percent,
      Epinephrine = with_epinephrine == 1
    )
  ))
}

los_angeles_esophagitis_grade <- function(grade) {

  grade <- tolower(as.character(grade))

  valid_grades <- c("grade_a", "grade_b", "grade_c", "grade_d")

  if (!grade %in% valid_grades) {
    stop("Error: Grade must be 'grade_a', 'grade_b', 'grade_c', or 'grade_d'.")
  }

  definition <- ""

  if (grade == "grade_a") {
    definition <- "One or more mucosal breaks confined to the mucosal folds, each <= 5 mm in maximum length."
  } else if (grade == "grade_b") {
    definition <- "One or more mucosal breaks > 5 mm in maximum length, but not continuous between the tops of two mucosal folds."
  } else if (grade == "grade_c") {
    definition <- "One or more mucosal breaks that are continuous between the tops of two or more mucosal folds, but which involve < 75% of the esophageal circumference."
  } else {
    definition <- "One or more mucosal breaks which involve >= 75% of the esophageal circumference."
  }

  return(list(
    LA_Grade = toupper(sub("grade_", "Grade ", grade)),
    Definition = definition
  ))
}

lams_stroke_score <- function(facial_droop, arm_drift, grip_strength) {

  facial_droop <- as.numeric(facial_droop)
  arm_drift <- as.numeric(arm_drift)
  grip_strength <- as.numeric(grip_strength)

  if (any(is.na(c(facial_droop, arm_drift, grip_strength)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!facial_droop %in% c(0, 1)) stop("Error: Facial Droop must be 0 or 1.")
  if (!arm_drift %in% c(0, 1, 2)) stop("Error: Arm Drift must be 0, 1, or 2.")
  if (!grip_strength %in% c(0, 1, 2)) stop("Error: Grip Strength must be 0, 1, or 2.")

  score <- facial_droop + arm_drift + grip_strength

  interpretation <- ""
  recommendation <- ""

  if (score >= 4) {
    interpretation <- "High Likelihood of Large Vessel Occlusion (LVO)"
    recommendation <- "Transport to Comprehensive Stroke Center (CSC) recommended."
  } else {
    interpretation <- "Lower Likelihood of LVO"
    recommendation <- "Transport to nearest Stroke Center (Primary or Comprehensive)."
  }

  return(list(
    LAMS_Score = score,
    Likelihood_LVO = interpretation,
    Recommendation = recommendation,
    Inputs = list(
      Face = facial_droop,
      Arm = arm_drift,
      Grip = grip_strength
    )
  ))
}

lrinec_score_necrotizing_infection <- function(crp_mg_dl, wbc_count, hemoglobin_g_dl, sodium, creatinine, glucose_mg_dl, creatinine_units = "mg/dL", glucose_units = "mg/dL") {

  crp_mg_dl <- as.numeric(crp_mg_dl)
  wbc_count <- as.numeric(wbc_count)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  sodium <- as.numeric(sodium)
  creatinine <- as.numeric(creatinine)
  glucose_mg_dl <- as.numeric(glucose_mg_dl)

  if (any(is.na(c(crp_mg_dl, wbc_count, hemoglobin_g_dl, sodium, creatinine, glucose_mg_dl)))) {
    stop("Error: All inputs must be numeric.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "micromol/L" || creatinine_units == "umol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  gluc_mg_dl <- glucose_mg_dl
  if (glucose_units == "mmol/L") {
    gluc_mg_dl <- glucose_mg_dl * 18.018
  }

  score <- 0

  if (crp_mg_dl >= 15) {
    score <- score + 4
  }

  if (wbc_count < 15) { # x10^9/L or x10^3/uL
    score <- score + 0
  } else if (wbc_count >= 15 && wbc_count <= 25) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (hemoglobin_g_dl > 13.5) {
    score <- score + 0
  } else if (hemoglobin_g_dl >= 11) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (sodium < 135) {
    score <- score + 2
  }

  if (creat_mg_dl > 1.6) { # > 141 umol/L
    score <- score + 2
  }

  if (gluc_mg_dl > 180) { # > 10 mmol/L
    score <- score + 1
  }

  risk_category <- ""
  probability <- ""

  if (score <= 5) {
    risk_category <- "Low Risk"
    probability <- "< 50%"
  } else if (score <= 7) {
    risk_category <- "Intermediate Risk"
    probability <- "50-75%"
  } else {
    risk_category <- "High Risk"
    probability <- "> 75%"
  }

  return(list(
    LRINEC_Score = score,
    Risk_Category = risk_category,
    Est_Probability_NSTI = probability,
    Inputs = list(
      CRP_ge_15mgdL = crp_mg_dl >= 15,
      WBC = wbc_count,
      Hb = hemoglobin_g_dl,
      Na_lt_135 = sodium < 135,
      Cr_gt_1_6 = creat_mg_dl > 1.6,
      Glucose_gt_180 = gluc_mg_dl > 180
    )
  ))
}

lips_score <- function(shock, aspiration, sepsis, pneumonia, high_risk_surgery, high_risk_trauma, alcohol_abuse, obesity_bmi_gt_30, hypoalbuminemia, chemotherapy, fio2_gt_0_35_or_4l, tachypnea_rr_gt_30, spo2_lt_95, acidosis_ph_lt_7_35, diabetes_mellitus) {

  shock <- as.numeric(shock)
  aspiration <- as.numeric(aspiration)
  sepsis <- as.numeric(sepsis)
  pneumonia <- as.numeric(pneumonia)
  high_risk_surgery <- as.numeric(high_risk_surgery)
  high_risk_trauma <- as.numeric(high_risk_trauma)
  alcohol_abuse <- as.numeric(alcohol_abuse)
  obesity_bmi_gt_30 <- as.numeric(obesity_bmi_gt_30)
  hypoalbuminemia <- as.numeric(hypoalbuminemia)
  chemotherapy <- as.numeric(chemotherapy)
  fio2_gt_0_35_or_4l <- as.numeric(fio2_gt_0_35_or_4l)
  tachypnea_rr_gt_30 <- as.numeric(tachypnea_rr_gt_30)
  spo2_lt_95 <- as.numeric(spo2_lt_95)
  acidosis_ph_lt_7_35 <- as.numeric(acidosis_ph_lt_7_35)
  diabetes_mellitus <- as.numeric(diabetes_mellitus)

  if (any(is.na(c(shock, aspiration, sepsis, pneumonia, high_risk_surgery, high_risk_trauma, alcohol_abuse, obesity_bmi_gt_30, hypoalbuminemia, chemotherapy, fio2_gt_0_35_or_4l, tachypnea_rr_gt_30, spo2_lt_95, acidosis_ph_lt_7_35, diabetes_mellitus)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (shock == 1) score <- score + 2
  if (aspiration == 1) score <- score + 2
  if (high_risk_surgery == 1) score <- score + 2
  if (high_risk_trauma == 1) score <- score + 2
  if (pneumonia == 1) score <- score + 1.5
  if (sepsis == 1) score <- score + 1

  if (alcohol_abuse == 1) score <- score + 1
  if (obesity_bmi_gt_30 == 1) score <- score + 1
  if (hypoalbuminemia == 1) score <- score + 1
  if (chemotherapy == 1) score <- score + 1

  if (fio2_gt_0_35_or_4l == 1) score <- score + 2
  if (tachypnea_rr_gt_30 == 1) score <- score + 1.5
  if (spo2_lt_95 == 1) score <- score + 1
  if (acidosis_ph_lt_7_35 == 1) score <- score + 1

  if (diabetes_mellitus == 1) score <- score - 1

  risk_category <- ""
  recommendation <- ""

  if (score < 4) {
    risk_category <- "Low Risk"
    recommendation <- "Standard care."
  } else {
    risk_category <- "High Risk"
    recommendation <- "Close monitoring for ARDS development. Avoid 'second hit' exposures (e.g., fluid overload, high tidal volumes, aspiration)."
  }

  return(list(
    LIPS_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Shock = shock == 1,
      Aspiration = aspiration == 1,
      HighRisk_Sx = high_risk_surgery == 1,
      HighRisk_Trauma = high_risk_trauma == 1,
      Pneumonia = pneumonia == 1,
      Sepsis = sepsis == 1,
      Alcohol = alcohol_abuse == 1,
      Obesity = obesity_bmi_gt_30 == 1,
      Hypoalbuminemia = hypoalbuminemia == 1,
      Chemo = chemotherapy == 1,
      High_FiO2 = fio2_gt_0_35_or_4l == 1,
      Tachypnea = tachypnea_rr_gt_30 == 1,
      Low_SpO2 = spo2_lt_95 == 1,
      Acidosis = acidosis_ph_lt_7_35 == 1,
      Diabetes = diabetes_mellitus == 1
    )
  ))
}

icc_systemic_mastocytosis_criteria <- function(major_criterion_multifocal_infiltrates, minor_criterion_spindle_mast_cells, minor_criterion_kit_mutation, minor_criterion_cd25_expression, minor_criterion_tryptase_gt_20, minor_criterion_cd30_expression_bmm) {

  major_criterion_multifocal_infiltrates <- as.numeric(major_criterion_multifocal_infiltrates)
  minor_criterion_spindle_mast_cells <- as.numeric(minor_criterion_spindle_mast_cells)
  minor_criterion_kit_mutation <- as.numeric(minor_criterion_kit_mutation)
  minor_criterion_cd25_expression <- as.numeric(minor_criterion_cd25_expression)
  minor_criterion_tryptase_gt_20 <- as.numeric(minor_criterion_tryptase_gt_20)
  minor_criterion_cd30_expression_bmm <- as.numeric(minor_criterion_cd30_expression_bmm)

  if (any(is.na(c(major_criterion_multifocal_infiltrates, minor_criterion_spindle_mast_cells, minor_criterion_kit_mutation, minor_criterion_cd25_expression, minor_criterion_tryptase_gt_20, minor_criterion_cd30_expression_bmm)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  minor_count <- 0

  if (minor_criterion_spindle_mast_cells == 1) minor_count <- minor_count + 1
  if (minor_criterion_kit_mutation == 1) minor_count <- minor_count + 1
  if (minor_criterion_cd25_expression == 1) minor_count <- minor_count + 1
  if (minor_criterion_tryptase_gt_20 == 1) minor_count <- minor_count + 1
  if (minor_criterion_cd30_expression_bmm == 1) minor_count <- minor_count + 1

  diagnosis_met <- FALSE

  if (major_criterion_multifocal_infiltrates == 1) {
    if (minor_count >= 1) {
      diagnosis_met <- TRUE
    }
  } else {
    if (minor_count >= 3) {
      diagnosis_met <- TRUE
    }
  }

  result <- ""
  if (diagnosis_met) {
    result <- "Criteria Met for Systemic Mastocytosis"
  } else {
    result <- "Criteria Not Met for Systemic Mastocytosis"
  }

  return(list(
    Diagnosis = result,
    Major_Criteria_Met = major_criterion_multifocal_infiltrates == 1,
    Minor_Criteria_Count = minor_count,
    Inputs = list(
      Major_Infiltrates = major_criterion_multifocal_infiltrates == 1,
      Minor_Spindle_Cells = minor_criterion_spindle_mast_cells == 1,
      Minor_KIT_Mutation = minor_criterion_kit_mutation == 1,
      Minor_CD25 = minor_criterion_cd25_expression == 1,
      Minor_Tryptase = minor_criterion_tryptase_gt_20 == 1,
      Minor_CD30_BMM = minor_criterion_cd30_expression_bmm == 1
    )
  ))
}

icc_pmf_criteria <- function(histology_consistent, exclusion_of_other_neoplasms, clonal_marker_present, anemia, leukocytosis, splenomegaly, ldh_elevated, leukoerythroblastosis) {

  histology_consistent <- as.numeric(histology_consistent)
  exclusion_of_other_neoplasms <- as.numeric(exclusion_of_other_neoplasms)
  clonal_marker_present <- as.numeric(clonal_marker_present)
  anemia <- as.numeric(anemia)
  leukocytosis <- as.numeric(leukocytosis)
  splenomegaly <- as.numeric(splenomegaly)
  ldh_elevated <- as.numeric(ldh_elevated)
  leukoerythroblastosis <- as.numeric(leukoerythroblastosis)

  if (any(is.na(c(histology_consistent, exclusion_of_other_neoplasms, clonal_marker_present, anemia, leukocytosis, splenomegaly, ldh_elevated, leukoerythroblastosis)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  major_count <- 0
  if (histology_consistent == 1) major_count <- major_count + 1
  if (exclusion_of_other_neoplasms == 1) major_count <- major_count + 1
  if (clonal_marker_present == 1) major_count <- major_count + 1

  minor_count <- 0
  if (anemia == 1) minor_count <- minor_count + 1
  if (leukocytosis == 1) minor_count <- minor_count + 1
  if (splenomegaly == 1) minor_count <- minor_count + 1
  if (ldh_elevated == 1) minor_count <- minor_count + 1
  if (leukoerythroblastosis == 1) minor_count <- minor_count + 1

  diagnosis <- ""
  if (major_count == 3 && minor_count >= 1) {
    diagnosis <- "Criteria Met for Primary Myelofibrosis (PMF)"
  } else {
    diagnosis <- "Criteria Not Met for PMF"
  }

  return(list(
    Diagnosis = diagnosis,
    Major_Criteria_Met = paste(major_count, "/ 3"),
    Minor_Criteria_Met = paste(minor_count, "(Minimum 1 required)"),
    Inputs = list(
      Histology = histology_consistent == 1,
      Exclusions = exclusion_of_other_neoplasms == 1,
      Clonal_Marker = clonal_marker_present == 1,
      Anemia = anemia == 1,
      Leukocytosis = leukocytosis == 1,
      Splenomegaly = splenomegaly == 1,
      LDH = ldh_elevated == 1,
      Leukoerythroblastosis = leukoerythroblastosis == 1
    )
  ))
}

idf_dar_fasting_risk <- function(type_1_diabetes, duration_gt_10_years, hba1c_value, hypoglycemia_points, complication_points, medication_points, smbg_not_performed, pregnancy, frailty_cognitive, intense_labor, previous_ramadan_bad) {

  type_1_diabetes <- as.numeric(type_1_diabetes)
  duration_gt_10_years <- as.numeric(duration_gt_10_years)
  hba1c_value <- as.numeric(hba1c_value)
  hypoglycemia_points <- as.numeric(hypoglycemia_points)
  complication_points <- as.numeric(complication_points)
  medication_points <- as.numeric(medication_points)
  smbg_not_performed <- as.numeric(smbg_not_performed)
  pregnancy <- as.numeric(pregnancy)
  frailty_cognitive <- as.numeric(frailty_cognitive)
  intense_labor <- as.numeric(intense_labor)
  previous_ramadan_bad <- as.numeric(previous_ramadan_bad)

  if (any(is.na(c(type_1_diabetes, duration_gt_10_years, hba1c_value, hypoglycemia_points, complication_points, medication_points, smbg_not_performed, pregnancy, frailty_cognitive, intense_labor, previous_ramadan_bad)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (type_1_diabetes == 1) {
    score <- score + 1
  }

  if (duration_gt_10_years == 1) {
    score <- score + 1
  }

  if (hba1c_value >= 7.5 && hba1c_value <= 9.0) {
    score <- score + 1
  } else if (hba1c_value > 9.0) {
    score <- score + 2
  }

  score <- score + hypoglycemia_points

  score <- score + complication_points

  score <- score + medication_points

  if (smbg_not_performed == 1) {
    score <- score + 2
  }

  if (pregnancy == 1) {
    score <- score + 6.5
  }

  if (frailty_cognitive == 1) {
    score <- score + 5.5
  }

  if (intense_labor == 1) {
    score <- score + 1
  }

  if (previous_ramadan_bad == 1) {
    score <- score + 1
  }

  risk_category <- ""
  recommendation <- ""

  if (score <= 3) {
    risk_category <- "Low Risk (Score 0-3)"
    recommendation <- "Fasting is likely safe with medical advice."
  } else if (score <= 6) {
    risk_category <- "Moderate Risk (Score 3.5-6)"
    recommendation <- "Fasting is not recommended. If patient insists, requires expert guidance and structured education."
  } else {
    risk_category <- "High Risk (Score > 6)"
    recommendation <- "Fasting is strictly not recommended (Very High Risk). Patient should be strongly advised against fasting."
  }

  return(list(
    Total_Risk_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      T1DM = type_1_diabetes == 1,
      Duration_gt_10y = duration_gt_10_years == 1,
      HbA1c = hba1c_value,
      Hypo_Points = hypoglycemia_points,
      Complication_Points = complication_points,
      Med_Points = medication_points,
      No_SMBG = smbg_not_performed == 1,
      Pregnancy = pregnancy == 1,
      Frailty = frailty_cognitive == 1,
      Labor = intense_labor == 1,
      Bad_Experience = previous_ramadan_bad == 1
    )
  ))
}

international_iga_nephropathy_prediction <- function(age_at_biopsy, egfr_at_biopsy, proteinuria_g_day, map_mmhg, race, mest_m, mest_e, mest_s, mest_t, use_rasb, use_immunosuppression) {

  age_at_biopsy <- as.numeric(age_at_biopsy)
  egfr_at_biopsy <- as.numeric(egfr_at_biopsy)
  proteinuria_g_day <- as.numeric(proteinuria_g_day)
  map_mmhg <- as.numeric(map_mmhg)
  race <- tolower(as.character(race))
  mest_m <- as.numeric(mest_m)
  mest_e <- as.numeric(mest_e)
  mest_s <- as.numeric(mest_s)
  mest_t <- as.numeric(mest_t)
  use_rasb <- as.numeric(use_rasb)
  use_immunosuppression <- as.numeric(use_immunosuppression)

  if (any(is.na(c(age_at_biopsy, egfr_at_biopsy, proteinuria_g_day, map_mmhg, mest_m, mest_e, mest_s, mest_t, use_rasb, use_immunosuppression)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!race %in% c("white", "chinese", "japanese", "other")) {
    stop("Error: Race must be 'white', 'chinese', 'japanese', or 'other'.")
  }

  coeff_race <- 0
  if (race == "chinese") coeff_race <- 0.197
  if (race == "japanese") coeff_race <- -0.291

  coeff_t <- 0
  if (mest_t == 1) coeff_t <- 0.277
  if (mest_t == 2) coeff_t <- 0.625

  log_proteinuria <- log(proteinuria_g_day)

  lp <- ( -0.055 * egfr_at_biopsy ) +
    ( 0.00033 * egfr_at_biopsy^2 ) +
    ( 0.016 * map_mmhg ) +
    ( 0.562 * log_proteinuria ) +
    ( -0.016 * age_at_biopsy ) +
    ( 0.239 * mest_m ) +
    ( 0.168 * mest_e ) +
    ( 0.295 * mest_s ) +
    coeff_t +
    coeff_race +
    ( -0.232 * use_rasb ) +
    ( -0.397 * use_immunosuppression )

  return(list(
    Linear_Predictor = round(lp, 3),
    Inputs = list(
      Age = age_at_biopsy,
      eGFR = egfr_at_biopsy,
      Proteinuria = proteinuria_g_day,
      MAP = map_mmhg,
      Race = race,
      MEST = paste0("M", mest_m, " E", mest_e, " S", mest_s, " T", mest_t),
      RASB = use_rasb == 1,
      Immuno = use_immunosuppression == 1
    )
  ))
}

iota_simple_rules_risk <- function(b1_unilocular, b2_solid_lt_7mm, b3_acoustic_shadows, b4_smooth_multi_lt_100mm, b5_no_blood_flow, m1_irregular_solid, m2_ascites, m3_four_papillary_structures, m4_irregular_multi_gt_100mm, m5_strong_blood_flow) {

  b1_unilocular <- as.numeric(b1_unilocular)
  b2_solid_lt_7mm <- as.numeric(b2_solid_lt_7mm)
  b3_acoustic_shadows <- as.numeric(b3_acoustic_shadows)
  b4_smooth_multi_lt_100mm <- as.numeric(b4_smooth_multi_lt_100mm)
  b5_no_blood_flow <- as.numeric(b5_no_blood_flow)
  m1_irregular_solid <- as.numeric(m1_irregular_solid)
  m2_ascites <- as.numeric(m2_ascites)
  m3_four_papillary_structures <- as.numeric(m3_four_papillary_structures)
  m4_irregular_multi_gt_100mm <- as.numeric(m4_irregular_multi_gt_100mm)
  m5_strong_blood_flow <- as.numeric(m5_strong_blood_flow)

  if (any(is.na(c(b1_unilocular, b2_solid_lt_7mm, b3_acoustic_shadows, b4_smooth_multi_lt_100mm, b5_no_blood_flow, m1_irregular_solid, m2_ascites, m3_four_papillary_structures, m4_irregular_multi_gt_100mm, m5_strong_blood_flow)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  b_count <- b1_unilocular + b2_solid_lt_7mm + b3_acoustic_shadows + b4_smooth_multi_lt_100mm + b5_no_blood_flow
  m_count <- m1_irregular_solid + m2_ascites + m3_four_papillary_structures + m4_irregular_multi_gt_100mm + m5_strong_blood_flow

  classification <- ""
  if (b_count >= 1 && m_count == 0) {
    classification <- "Benign"
  } else if (m_count >= 1 && b_count == 0) {
    classification <- "Malignant"
  } else {
    classification <- "Inconclusive"
  }

  logit <- -2.3548 +
    (1.4631 * m1_irregular_solid) +
    (1.8221 * m2_ascites) +
    (0.9106 * m3_four_papillary_structures) +
    (1.1781 * m4_irregular_multi_gt_100mm) +
    (1.4553 * m5_strong_blood_flow) -
    (1.5985 * b1_unilocular) -
    (1.9492 * b2_solid_lt_7mm) -
    (2.5153 * b3_acoustic_shadows) -
    (1.8556 * b4_smooth_multi_lt_100mm) -
    (0.7939 * b5_no_blood_flow)

  risk_probability <- 1 / (1 + exp(-logit))
  risk_percent <- risk_probability * 100

  return(list(
    Risk_of_Malignancy_Percent = round(risk_percent, 1),
    Original_Classification = classification,
    Inputs = list(
      B_Features = list(B1=b1_unilocular, B2=b2_solid_lt_7mm, B3=b3_acoustic_shadows, B4=b4_smooth_multi_lt_100mm, B5=b5_no_blood_flow),
      M_Features = list(M1=m1_irregular_solid, M2=m2_ascites, M3=m3_four_papillary_structures, M4=m4_irregular_multi_gt_100mm, M5=m5_strong_blood_flow)
    )
  ))
}

cll_ipi_score <- function(tp53_deleted_or_mutated, ighv_unmutated, beta2_microglobulin_gt_3_5, clinical_stage_advanced, age_gt_65) {

  tp53_deleted_or_mutated <- as.numeric(tp53_deleted_or_mutated)
  ighv_unmutated <- as.numeric(ighv_unmutated)
  beta2_microglobulin_gt_3_5 <- as.numeric(beta2_microglobulin_gt_3_5)
  clinical_stage_advanced <- as.numeric(clinical_stage_advanced)
  age_gt_65 <- as.numeric(age_gt_65)

  if (any(is.na(c(tp53_deleted_or_mutated, ighv_unmutated, beta2_microglobulin_gt_3_5, clinical_stage_advanced, age_gt_65)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (tp53_deleted_or_mutated == 1) {
    score <- score + 4
  }

  if (ighv_unmutated == 1) {
    score <- score + 2
  }

  if (beta2_microglobulin_gt_3_5 == 1) {
    score <- score + 2
  }

  if (clinical_stage_advanced == 1) {
    score <- score + 1
  }

  if (age_gt_65 == 1) {
    score <- score + 1
  }

  risk_group <- ""
  survival_5yr <- ""
  recommendation <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    survival_5yr <- "93.2%"
    recommendation <- "Do not treat."
  } else if (score <= 3) {
    risk_group <- "Intermediate Risk"
    survival_5yr <- "79.3%"
    recommendation <- "Do not treat unless symptomatic."
  } else if (score <= 6) {
    risk_group <- "High Risk"
    survival_5yr <- "63.3%"
    recommendation <- "Treatment indicated (if symptomatic)."
  } else {
    risk_group <- "Very High Risk"
    survival_5yr <- "23.3%"
    recommendation <- "Treatment indicated (clinical trials or novel agents recommended)."
  }

  return(list(
    CLL_IPI_Score = score,
    Risk_Group = risk_group,
    Est_5_Year_Overall_Survival = survival_5yr,
    Recommendation = recommendation,
    Inputs = list(
      TP53_Status = tp53_deleted_or_mutated == 1,
      IGHV_Unmutated = ighv_unmutated == 1,
      B2M_High = beta2_microglobulin_gt_3_5 == 1,
      Stage_Advanced = clinical_stage_advanced == 1,
      Age_gt_65 = age_gt_65 == 1
    )
  ))
}

ipi_score_dlbcl <- function(age, stage, ldh_ratio, performance_status, extranodal_sites) {

  age <- as.numeric(age)
  stage <- as.numeric(stage)
  ldh_ratio <- as.numeric(ldh_ratio)
  performance_status <- as.numeric(performance_status)
  extranodal_sites <- as.numeric(extranodal_sites)

  if (any(is.na(c(age, stage, ldh_ratio, performance_status, extranodal_sites)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age > 60) {
    score <- score + 1
  }

  if (stage == 3 || stage == 4) {
    score <- score + 1
  }

  if (ldh_ratio > 1.0) {
    score <- score + 1
  }

  if (performance_status >= 2) {
    score <- score + 1
  }

  if (extranodal_sites > 1) {
    score <- score + 1
  }

  standard_ipi_risk <- ""
  standard_ipi_survival <- ""

  if (score == 0 || score == 1) {
    standard_ipi_risk <- "Low Risk"
    standard_ipi_survival <- "73% (5-year OS)"
  } else if (score == 2) {
    standard_ipi_risk <- "Low-Intermediate Risk"
    standard_ipi_survival <- "51% (5-year OS)"
  } else if (score == 3) {
    standard_ipi_risk <- "High-Intermediate Risk"
    standard_ipi_survival <- "43% (5-year OS)"
  } else {
    standard_ipi_risk <- "High Risk"
    standard_ipi_survival <- "26% (5-year OS)"
  }

  r_ipi_risk <- ""
  r_ipi_survival <- ""

  if (score == 0) {
    r_ipi_risk <- "Very Good Risk"
    r_ipi_survival <- "94% (4-year OS)"
  } else if (score == 1 || score == 2) {
    r_ipi_risk <- "Good Risk"
    r_ipi_survival <- "79% (4-year OS)"
  } else {
    r_ipi_risk <- "Poor Risk"
    r_ipi_survival <- "55% (4-year OS)"
  }

  return(list(
    IPI_Score = score,
    Standard_IPI_Classification = list(
      Risk_Group = standard_ipi_risk,
      Survival_Estimate = standard_ipi_survival
    ),
    Revised_IPI_Classification = list(
      Risk_Group = r_ipi_risk,
      Survival_Estimate = r_ipi_survival
    ),
    Inputs = list(
      Age_gt_60 = age > 60,
      Stage_III_IV = stage >= 3,
      LDH_Elevated = ldh_ratio > 1,
      ECOG_ge_2 = performance_status >= 2,
      Extranodal_gt_1 = extranodal_sites > 1
    )
  ))
}

ips_e_score <- function(ighv_unmutated, lymphocyte_count, palpable_nodes) {

  ighv_unmutated <- as.numeric(ighv_unmutated)
  lymphocyte_count <- as.numeric(lymphocyte_count)
  palpable_nodes <- as.numeric(palpable_nodes)

  if (any(is.na(c(ighv_unmutated, lymphocyte_count, palpable_nodes)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (ighv_unmutated == 1) {
    score <- score + 1
  }

  if (lymphocyte_count > 15) {
    score <- score + 1
  }

  if (palpable_nodes == 1) {
    score <- score + 1
  }

  risk_group <- ""
  treatment_risk_5yr <- ""

  if (score == 0) {
    risk_group <- "Low Risk"
    treatment_risk_5yr <- "8.4%"
  } else if (score == 1) {
    risk_group <- "Intermediate Risk"
    treatment_risk_5yr <- "28.4%"
  } else {
    risk_group <- "High Risk"
    treatment_risk_5yr <- "61.2%"
  }

  return(list(
    IPS_E_Score = score,
    Risk_Group = risk_group,
    Est_5_Year_Treatment_Risk = treatment_risk_5yr,
    Inputs = list(
      IGHV_Unmutated = ighv_unmutated == 1,
      ALC_gt_15 = lymphocyte_count > 15,
      Palpable_Nodes = palpable_nodes == 1
    )
  ))
}

iwg2_alzheimers_criteria <- function(clinical_phenotype_present, pathophysiological_marker_positive, autosomal_dominant_mutation_present) {

  clinical_phenotype_present <- as.numeric(clinical_phenotype_present)
  pathophysiological_marker_positive <- as.numeric(pathophysiological_marker_positive)
  autosomal_dominant_mutation_present <- as.numeric(autosomal_dominant_mutation_present)

  if (any(is.na(c(clinical_phenotype_present, pathophysiological_marker_positive, autosomal_dominant_mutation_present)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  diagnosis <- ""
  classification <- ""

  if (clinical_phenotype_present == 1) {
    if (autosomal_dominant_mutation_present == 1) {
      diagnosis <- "Autosomal Dominant Alzheimer's Disease"
      classification <- "Definite AD"
    } else if (pathophysiological_marker_positive == 1) {
      diagnosis <- "Alzheimer's Disease (Typical or Atypical)"
      classification <- "Diagnosis Supported"
    } else {
      diagnosis <- "Clinical AD Phenotype (Biomarker Unconfirmed/Negative)"
      classification <- "Criteria for AD Not Met"
    }
  } else {
    if (autosomal_dominant_mutation_present == 1) {
      diagnosis <- "Preclinical Autosomal Dominant AD"
      classification <- "At Risk"
    } else if (pathophysiological_marker_positive == 1) {
      diagnosis <- "Preclinical Alzheimer's Disease (Asymptomatic at risk)"
      classification <- "At Risk"
    } else {
      diagnosis <- "Normal / Non-AD"
      classification <- "Criteria Not Met"
    }
  }

  return(list(
    IWG2_Diagnosis = diagnosis,
    Classification = classification,
    Inputs = list(
      Phenotype = clinical_phenotype_present == 1,
      Biomarker = pathophysiological_marker_positive == 1,
      Mutation = autosomal_dominant_mutation_present == 1
    )
  ))
}

ich_score <- function(gcs_score, ich_volume_cm3, intraventricular_hemorrhage, infratentorial_origin, age_years) {

  gcs_score <- as.numeric(gcs_score)
  ich_volume_cm3 <- as.numeric(ich_volume_cm3)
  intraventricular_hemorrhage <- as.numeric(intraventricular_hemorrhage)
  infratentorial_origin <- as.numeric(infratentorial_origin)
  age_years <- as.numeric(age_years)

  if (any(is.na(c(gcs_score, ich_volume_cm3, intraventricular_hemorrhage, infratentorial_origin, age_years)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (gcs_score >= 3 && gcs_score <= 4) {
    score <- score + 2
  } else if (gcs_score >= 5 && gcs_score <= 12) {
    score <- score + 1
  } else if (gcs_score >= 13 && gcs_score <= 15) {
    score <- score + 0
  } else {
    stop("Error: GCS score must be between 3 and 15.")
  }

  if (ich_volume_cm3 >= 30) {
    score <- score + 1
  }

  if (intraventricular_hemorrhage == 1) {
    score <- score + 1
  }

  if (infratentorial_origin == 1) {
    score <- score + 1
  }

  if (age_years >= 80) {
    score <- score + 1
  }

  mortality_30_day <- ""

  if (score == 0) {
    mortality_30_day <- "0%"
  } else if (score == 1) {
    mortality_30_day <- "13%"
  } else if (score == 2) {
    mortality_30_day <- "26%"
  } else if (score == 3) {
    mortality_30_day <- "72%"
  } else if (score == 4) {
    mortality_30_day <- "97%"
  } else {
    mortality_30_day <- "100%"
  }

  return(list(
    ICH_Score = score,
    Est_30_Day_Mortality = mortality_30_day,
    Inputs = list(
      GCS = gcs_score,
      Volume_ge_30 = ich_volume_cm3 >= 30,
      IVH = intraventricular_hemorrhage == 1,
      Infratentorial = infratentorial_origin == 1,
      Age_ge_80 = age_years >= 80
    )
  ))
}

intraoperative_fluid_dosing <- function(weight_kg, npo_time_hours, trauma_severity) {

  weight_kg <- as.numeric(weight_kg)
  npo_time_hours <- as.numeric(npo_time_hours)
  trauma_severity <- tolower(as.character(trauma_severity))

  if (any(is.na(c(weight_kg, npo_time_hours)))) {
    stop("Error: Weight and NPO time must be numeric.")
  }

  if (!trauma_severity %in% c("minimal", "moderate", "severe")) {
    stop("Error: Trauma severity must be 'minimal', 'moderate', or 'severe'.")
  }

  if (weight_kg < 20) {
    warning("Warning: This calculator is designed for patients >= 20 kg.")
  }

  maintenance_rate <- 0
  if (weight_kg <= 10) {
    maintenance_rate <- 4 * weight_kg
  } else if (weight_kg <= 20) {
    maintenance_rate <- 40 + (2 * (weight_kg - 10))
  } else {
    maintenance_rate <- 60 + (1 * (weight_kg - 20))
  }

  fluid_deficit <- maintenance_rate * npo_time_hours

  third_space_loss_rate <- 0
  if (trauma_severity == "minimal") {
    third_space_loss_rate <- 2 * weight_kg
  } else if (trauma_severity == "moderate") {
    third_space_loss_rate <- 4 * weight_kg
  } else {
    third_space_loss_rate <- 6 * weight_kg
  }

  rate_1st_hour <- maintenance_rate + (0.5 * fluid_deficit) + third_space_loss_rate
  rate_2nd_hour <- maintenance_rate + (0.25 * fluid_deficit) + third_space_loss_rate
  rate_3rd_hour <- maintenance_rate + (0.25 * fluid_deficit) + third_space_loss_rate
  rate_4th_hour <- maintenance_rate + third_space_loss_rate

  return(list(
    Maintenance_Rate_mL_hr = round(maintenance_rate, 1),
    Fluid_Deficit_Total_mL = round(fluid_deficit, 1),
    Third_Space_Loss_mL_hr = round(third_space_loss_rate, 1),
    Hourly_Rates = list(
      Hour_1 = round(rate_1st_hour, 1),
      Hour_2 = round(rate_2nd_hour, 1),
      Hour_3 = round(rate_3rd_hour, 1),
      Hour_4_Plus = round(rate_4th_hour, 1)
    ),
    Inputs = list(
      Weight = weight_kg,
      NPO_Hours = npo_time_hours,
      Trauma = trauma_severity
    )
  ))
}

intrauterine_rbc_transfusion <- function(fetal_weight_grams, current_hematocrit, target_hematocrit, donor_hematocrit = 80) {

  fetal_weight_grams <- as.numeric(fetal_weight_grams)
  current_hematocrit <- as.numeric(current_hematocrit)
  target_hematocrit <- as.numeric(target_hematocrit)
  donor_hematocrit <- as.numeric(donor_hematocrit)

  if (any(is.na(c(fetal_weight_grams, current_hematocrit, target_hematocrit, donor_hematocrit)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (target_hematocrit <= current_hematocrit) {
    return(list(
      Volume_Needed_mL = 0,
      Message = "Current hematocrit is already at or above target."
    ))
  }

  feto_placental_volume_ml <- fetal_weight_grams * 0.14

  volume_to_transfuse <- feto_placental_volume_ml * ((target_hematocrit - current_hematocrit) / (donor_hematocrit - current_hematocrit))

  return(list(
    Volume_to_Transfuse_mL = round(volume_to_transfuse, 1),
    Estimated_FetoPlacental_Volume_mL = round(feto_placental_volume_ml, 1),
    Inputs = list(
      Fetal_Weight_g = fetal_weight_grams,
      Current_Hct = current_hematocrit,
      Target_Hct = target_hematocrit,
      Donor_Hct = donor_hematocrit
    )
  ))
}

isth_dic_score <- function(platelet_count, pt_prolongation_seconds, fibrinogen_g_l, fibrin_marker_increase) {

  platelet_count <- as.numeric(platelet_count)
  pt_prolongation_seconds <- as.numeric(pt_prolongation_seconds)
  fibrinogen_g_l <- as.numeric(fibrinogen_g_l)
  fibrin_marker_increase <- tolower(as.character(fibrin_marker_increase))

  if (any(is.na(c(platelet_count, pt_prolongation_seconds, fibrinogen_g_l)))) {
    stop("Error: Numeric inputs required for Platelets, PT Prolongation, and Fibrinogen.")
  }

  if (!fibrin_marker_increase %in% c("none", "moderate", "strong")) {
    stop("Error: Fibrin marker increase must be 'none', 'moderate', or 'strong'.")
  }

  score <- 0

  if (platelet_count < 50) {
    score <- score + 2
  } else if (platelet_count < 100) {
    score <- score + 1
  }

  if (fibrin_marker_increase == "strong") {
    score <- score + 3
  } else if (fibrin_marker_increase == "moderate") {
    score <- score + 2
  }

  if (pt_prolongation_seconds >= 6) {
    score <- score + 2
  } else if (pt_prolongation_seconds >= 3) {
    score <- score + 1
  }

  if (fibrinogen_g_l < 1.0) {
    score <- score + 1
  }

  interpretation <- ""
  if (score >= 5) {
    interpretation <- "Compatible with overt DIC."
  } else {
    interpretation <- "Suggestive of non-overt DIC."
  }

  return(list(
    ISTH_DIC_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Platelets = platelet_count,
      PT_Prolongation_sec = pt_prolongation_seconds,
      Fibrinogen_g_L = fibrinogen_g_l,
      Fibrin_Marker = fibrin_marker_increase
    )
  ))
}

isth_scc_bleeding_tool <- function(epistaxis_score, cutaneous_score, minor_wounds_score, oral_cavity_score, gi_bleeding_score, pulmonary_score, hematuria_score, menorrhagia_score, postpartum_hemorrhage_score, surgery_bleeding_score, dental_bleeding_score, cns_bleeding_score, other_bleeding_score, age, sex) {

  inputs <- c(epistaxis_score, cutaneous_score, minor_wounds_score, oral_cavity_score, gi_bleeding_score, pulmonary_score, hematuria_score, menorrhagia_score, postpartum_hemorrhage_score, surgery_bleeding_score, dental_bleeding_score, cns_bleeding_score, other_bleeding_score)

  if (any(is.na(inputs))) {
    stop("Error: All domain scores must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: Domain scores must be between 0 and 4.")
  }

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  total_score <- sum(inputs)

  is_abnormal <- FALSE
  cutoff_used <- 0

  if (age < 18) {
    cutoff_used <- 3
    if (total_score >= 3) is_abnormal <- TRUE
  } else {
    if (sex == "male") {
      cutoff_used <- 4
      if (total_score >= 4) is_abnormal <- TRUE
    } else {
      cutoff_used <- 6
      if (total_score >= 6) is_abnormal <- TRUE
    }
  }

  interpretation <- ""
  if (is_abnormal) {
    interpretation <- "Abnormal Bleeding Score (Positive Screen for Bleeding Disorder)."
  } else {
    interpretation <- "Normal Bleeding Score."
  }

  return(list(
    ISTH_BAT_Score = total_score,
    Interpretation = interpretation,
    Cutoff_Applied = paste("Score >=", cutoff_used),
    Inputs = list(
      Epistaxis = epistaxis_score,
      Cutaneous = cutaneous_score,
      Minor_Wounds = minor_wounds_score,
      Oral = oral_cavity_score,
      GI = gi_bleeding_score,
      Pulmonary = pulmonary_score,
      Hematuria = hematuria_score,
      Menorrhagia = menorrhagia_score,
      Postpartum = postpartum_hemorrhage_score,
      Surgery = surgery_bleeding_score,
      Dental = dental_bleeding_score,
      CNS = cns_bleeding_score,
      Other = other_bleeding_score
    )
  ))
}

iv_drip_rate <- function(volume_ml, time_min, drop_factor_gtts_ml) {

  volume_ml <- as.numeric(volume_ml)
  time_min <- as.numeric(time_min)
  drop_factor_gtts_ml <- as.numeric(drop_factor_gtts_ml)

  if (any(is.na(c(volume_ml, time_min, drop_factor_gtts_ml)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (time_min <= 0) {
    stop("Error: Time must be positive.")
  }

  drip_rate <- (volume_ml * drop_factor_gtts_ml) / time_min

  return(list(
    Drip_Rate_gtts_min = round(drip_rate, 0),
    Inputs = list(
      Volume_mL = volume_ml,
      Time_min = time_min,
      Drop_Factor = drop_factor_gtts_ml
    )
  ))
}

jones_criteria_diagnosis <- function(risk_population, carditis, chorea, erythema_marginatum, subcutaneous_nodules, joint_manifestation, fever_c, esr, crp, prolonged_pr, evidence_gas, history_recurrence) {

  risk_population <- tolower(as.character(risk_population))
  joint_manifestation <- tolower(as.character(joint_manifestation))
  carditis <- as.numeric(carditis)
  chorea <- as.numeric(chorea)
  erythema_marginatum <- as.numeric(erythema_marginatum)
  subcutaneous_nodules <- as.numeric(subcutaneous_nodules)
  fever_c <- as.numeric(fever_c)
  esr <- as.numeric(esr)
  crp <- as.numeric(crp)
  prolonged_pr <- as.numeric(prolonged_pr)
  evidence_gas <- as.numeric(evidence_gas)
  history_recurrence <- as.numeric(history_recurrence)

  if (!risk_population %in% c("low", "moderate_high")) {
    stop("Error: Risk population must be 'low' or 'moderate_high'.")
  }

  valid_joints <- c("none", "monoarthralgia", "polyarthralgia", "monoarthritis", "polyarthritis")
  if (!joint_manifestation %in% valid_joints) {
    stop("Error: Joint manifestation must be valid category.")
  }

  major_count <- 0
  minor_count <- 0

  if (carditis == 1) major_count <- major_count + 1
  if (chorea == 1) major_count <- major_count + 1
  if (erythema_marginatum == 1) major_count <- major_count + 1
  if (subcutaneous_nodules == 1) major_count <- major_count + 1

  joint_is_major <- FALSE
  joint_is_minor <- FALSE

  if (risk_population == "low") {
    if (joint_manifestation == "polyarthritis") {
      major_count <- major_count + 1
      joint_is_major <- TRUE
    } else if (joint_manifestation == "polyarthralgia") {
      minor_count <- minor_count + 1
      joint_is_minor <- TRUE
    }
  } else {
    if (joint_manifestation == "monoarthritis" || joint_manifestation == "polyarthritis") {
      major_count <- major_count + 1
      joint_is_major <- TRUE
    } else if (joint_manifestation == "monoarthralgia" || joint_manifestation == "polyarthralgia") {
      minor_count <- minor_count + 1
      joint_is_minor <- TRUE
    }
  }

  fever_threshold <- if (risk_population == "low") 38.5 else 38.0
  if (fever_c >= fever_threshold) {
    minor_count <- minor_count + 1
  }

  esr_threshold <- if (risk_population == "low") 60 else 30
  crp_threshold <- 3.0

  if (esr >= esr_threshold || crp >= crp_threshold) {
    minor_count <- minor_count + 1
  }

  if (prolonged_pr == 1 && carditis == 0) {
    minor_count <- minor_count + 1
  }

  diagnosis <- "Criteria Not Met"

  if (evidence_gas == 1) {
    if (history_recurrence == 1) {
      if (major_count >= 2 || (major_count == 1 && minor_count >= 2) || minor_count >= 3) {
        diagnosis <- "Positive for Recurrent Rheumatic Fever"
      }
    } else {
      if (major_count >= 2 || (major_count == 1 && minor_count >= 2)) {
        diagnosis <- "Positive for Acute Rheumatic Fever"
      }
    }
  } else {
    if (chorea == 1 || carditis == 1) {
      diagnosis <- "Positive (Chorea/Carditis may not require acute GAS evidence)"
    } else {
      diagnosis <- "Negative (No Evidence of GAS Infection)"
    }
  }

  return(list(
    Diagnosis = diagnosis,
    Major_Criteria_Met = major_count,
    Minor_Criteria_Met = minor_count,
    Inputs = list(
      Risk = risk_population,
      Carditis = carditis == 1,
      Chorea = chorea == 1,
      Rash = erythema_marginatum == 1,
      Nodules = subcutaneous_nodules == 1,
      Joints = joint_manifestation,
      Fever = fever_c,
      Labs_Elevated = (esr >= esr_threshold || crp >= crp_threshold),
      PR_Prolonged = prolonged_pr == 1,
      GAS_Evidence = evidence_gas == 1
    )
  ))
}

macocha_score <- function(mallampati_iii_iv, apnea_syndrome, cervical_spine_limited, mouth_opening_lt_3cm, coma_gcs_lt_8, hypoxia_severe, non_anesthesiologist) {

  mallampati_iii_iv <- as.numeric(mallampati_iii_iv)
  apnea_syndrome <- as.numeric(apnea_syndrome)
  cervical_spine_limited <- as.numeric(cervical_spine_limited)
  mouth_opening_lt_3cm <- as.numeric(mouth_opening_lt_3cm)
  coma_gcs_lt_8 <- as.numeric(coma_gcs_lt_8)
  hypoxia_severe <- as.numeric(hypoxia_severe)
  non_anesthesiologist <- as.numeric(non_anesthesiologist)

  if (any(is.na(c(mallampati_iii_iv, apnea_syndrome, cervical_spine_limited, mouth_opening_lt_3cm, coma_gcs_lt_8, hypoxia_severe, non_anesthesiologist)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (mallampati_iii_iv == 1) {
    score <- score + 5
  }

  if (apnea_syndrome == 1) {
    score <- score + 2
  }

  if (cervical_spine_limited == 1) {
    score <- score + 1
  }

  if (mouth_opening_lt_3cm == 1) {
    score <- score + 1
  }

  if (coma_gcs_lt_8 == 1) {
    score <- score + 1
  }

  if (hypoxia_severe == 1) {
    score <- score + 1
  }

  if (non_anesthesiologist == 1) {
    score <- score + 1
  }

  risk_category <- ""
  recommendation <- ""

  if (score < 3) {
    risk_category <- "Low Risk"
    recommendation <- "Standard intubation approach likely sufficient."
  } else {
    risk_category <- "High Risk"
    recommendation <- "Anticipate difficult intubation. Prepare advanced airway equipment and experienced personnel."
  }

  return(list(
    MACOCHA_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Mallampati_III_IV = mallampati_iii_iv == 1,
      OSAS = apnea_syndrome == 1,
      C_Spine_Limited = cervical_spine_limited == 1,
      Mouth_Opening_lt_3cm = mouth_opening_lt_3cm == 1,
      Coma = coma_gcs_lt_8 == 1,
      Hypoxia = hypoxia_severe == 1,
      Non_Anesthesiologist = non_anesthesiologist == 1
    )
  ))
}

milan_criteria_liver_transplant <- function(num_tumors, max_tumor_size_cm, extra_hepatic_spread, vascular_invasion) {

  num_tumors <- as.numeric(num_tumors)
  max_tumor_size_cm <- as.numeric(max_tumor_size_cm)
  extra_hepatic_spread <- as.numeric(extra_hepatic_spread)
  vascular_invasion <- as.numeric(vascular_invasion)

  if (any(is.na(c(num_tumors, max_tumor_size_cm, extra_hepatic_spread, vascular_invasion)))) {
    stop("Error: All inputs must be numeric.")
  }

  eligible <- FALSE
  reason <- ""

  if (extra_hepatic_spread == 1) {
    eligible <- FALSE
    reason <- "Extra-hepatic spread present."
  } else if (vascular_invasion == 1) {
    eligible <- FALSE
    reason <- "Vascular invasion present."
  } else {
    if (num_tumors == 1) {
      if (max_tumor_size_cm <= 5) {
        eligible <- TRUE
        reason <- "Single tumor <= 5 cm."
      } else {
        eligible <- FALSE
        reason <- "Single tumor > 5 cm."
      }
    } else if (num_tumors >= 2 && num_tumors <= 3) {
      if (max_tumor_size_cm <= 3) {
        eligible <- TRUE
        reason <- "Up to 3 tumors, none > 3 cm."
      } else {
        eligible <- FALSE
        reason <- "Multiple tumors with at least one > 3 cm."
      }
    } else {
      eligible <- FALSE
      reason <- "More than 3 tumors present."
    }
  }

  return(list(
    Result = if(eligible) "Eligible for Transplant (Milan Criteria Met)" else "Not Eligible (Milan Criteria Not Met)",
    Explanation = reason,
    Inputs = list(
      Tumor_Count = num_tumors,
      Max_Size_cm = max_tumor_size_cm,
      Metastasis = extra_hepatic_spread == 1,
      Invasion = vascular_invasion == 1
    )
  ))
}

mirels_fracture_score <- function(site, pain, lesion_type, size_ratio) {

  site <- tolower(as.character(site))
  pain <- tolower(as.character(pain))
  lesion_type <- tolower(as.character(lesion_type))
  size_ratio <- tolower(as.character(size_ratio))

  if (!site %in% c("upper_limb", "lower_limb", "peritrochanteric")) {
    stop("Error: Site must be 'upper_limb', 'lower_limb', or 'peritrochanteric'.")
  }

  if (!pain %in% c("mild", "moderate", "functional")) {
    stop("Error: Pain must be 'mild', 'moderate', or 'functional'.")
  }

  if (!lesion_type %in% c("blastic", "mixed", "lytic")) {
    stop("Error: Lesion type must be 'blastic', 'mixed', or 'lytic'.")
  }

  if (!size_ratio %in% c("lt_one_third", "one_third_to_two_thirds", "gt_two_thirds")) {
    stop("Error: Size ratio must be 'lt_one_third', 'one_third_to_two_thirds', or 'gt_two_thirds'.")
  }

  score <- 0

  if (site == "upper_limb") {
    score <- score + 1
  } else if (site == "lower_limb") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (pain == "mild") {
    score <- score + 1
  } else if (pain == "moderate") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (lesion_type == "blastic") {
    score <- score + 1
  } else if (lesion_type == "mixed") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (size_ratio == "lt_one_third") {
    score <- score + 1
  } else if (size_ratio == "one_third_to_two_thirds") {
    score <- score + 2
  } else {
    score <- score + 3
  }

  recommendation <- ""
  fracture_risk <- ""

  if (score <= 7) {
    recommendation <- "Prophylactic fixation usually NOT indicated. Radiotherapy and observation."
    fracture_risk <- "< 5%"
  } else if (score == 8) {
    recommendation <- "Borderline. Use clinical judgment."
    fracture_risk <- "15%"
  } else {
    recommendation <- "Prophylactic fixation indicated."
    fracture_risk > "33%"
  }

  return(list(
    Mirels_Score = score,
    Recommendation = recommendation,
    Est_Fracture_Risk = fracture_risk,
    Inputs = list(
      Site = site,
      Pain = pain,
      Lesion = lesion_type,
      Size = size_ratio
    )
  ))
}

mmrc_dyspnea_scale <- function(grade) {

  grade <- as.numeric(grade)

  if (is.na(grade) || !grade %in% 0:4) {
    stop("Error: Grade must be an integer between 0 and 4.")
  }

  description <- ""

  if (grade == 0) {
    description <- "I only get breathless with strenuous exercise."
  } else if (grade == 1) {
    description <- "I get short of breath when hurrying on the level or walking up a slight hill."
  } else if (grade == 2) {
    description <- "I walk slower than people of the same age on the level because of breathlessness, or I have to stop for breath when walking on my own pace on the level."
  } else if (grade == 3) {
    description <- "I stop for breath after walking about 100 meters or after a few minutes on the level."
  } else {
    description <- "I am too breathless to leave the house or I am breathless when dressing or undressing."
  }

  return(list(
    mMRC_Grade = grade,
    Description = description
  ))
}

calculate_meld_score <- function(bilirubin, inr, creatinine, sodium, on_dialysis = 0, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  bilirubin <- as.numeric(bilirubin)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)
  sodium <- as.numeric(sodium)
  on_dialysis <- as.numeric(on_dialysis)

  if (any(is.na(c(bilirubin, inr, creatinine, sodium)))) {
    stop("Error: Bilirubin, INR, Creatinine, and Sodium must be numeric.")
  }

  bili_val <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_val <- bilirubin / 17.1
  }

  creat_val <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    creat_val <- creatinine / 88.4
  }

  if (bili_val < 1.0) bili_val <- 1.0
  if (inr < 1.0) inr <- 1.0

  if (on_dialysis == 1 || creat_val > 4.0) {
    creat_val <- 4.0
  } else if (creat_val < 1.0) {
    creat_val <- 1.0
  }

  meld_i <- 0.957 * log(creat_val) + 0.378 * log(bili_val) + 1.120 * log(inr) + 0.643
  meld_initial <- meld_i * 10

  meld_initial <- round(meld_initial, 1)

  meld_final <- meld_initial

  if (meld_initial > 11) {
    na_val <- sodium
    if (na_val < 125) na_val <- 125
    if (na_val > 137) na_val <- 137

    meld_final <- meld_initial + 1.32 * (137 - na_val) - (0.033 * meld_initial * (137 - na_val))
  }

  meld_final <- round(meld_final, 0)
  if (meld_final > 40) meld_final <- 40

  mortality <- ""
  if (meld_final <= 9) {
    mortality <- "1.9%"
  } else if (meld_final <= 19) {
    mortality <- "6.0%"
  } else if (meld_final <= 29) {
    mortality <- "19.6%"
  } else if (meld_final <= 39) {
    mortality <- "52.6%"
  } else {
    mortality <- "71.3%"
  }

  return(list(
    MELD_Na_Score = meld_final,
    Est_3_Month_Mortality = mortality,
    Inputs = list(
      Bilirubin = round(bili_val, 2),
      INR = inr,
      Creatinine = round(creat_val, 2),
      Sodium = sodium,
      Dialysis = on_dialysis == 1
    )
  ))
}

modified_asthma_predictive_index <- function(age, wheezing_episodes_gt_3_per_year, parental_asthma, eczema, aeroallergen_sensitization, food_sensitization, wheezing_apart_from_colds, eosinophilia_gt_4_percent) {

  age <- as.numeric(age)
  wheezing_episodes_gt_3_per_year <- as.numeric(wheezing_episodes_gt_3_per_year)
  parental_asthma <- as.numeric(parental_asthma)
  eczema <- as.numeric(eczema)
  aeroallergen_sensitization <- as.numeric(aeroallergen_sensitization)
  food_sensitization <- as.numeric(food_sensitization)
  wheezing_apart_from_colds <- as.numeric(wheezing_apart_from_colds)
  eosinophilia_gt_4_percent <- as.numeric(eosinophilia_gt_4_percent)

  if (any(is.na(c(age, wheezing_episodes_gt_3_per_year, parental_asthma, eczema, aeroallergen_sensitization, food_sensitization, wheezing_apart_from_colds, eosinophilia_gt_4_percent)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (age > 3) {
    warning("The mAPI is primarily validated for children <= 3 years to predict asthma at school age.")
  }

  major_criteria <- 0
  if (parental_asthma == 1) major_criteria <- major_criteria + 1
  if (eczema == 1) major_criteria <- major_criteria + 1
  if (aeroallergen_sensitization == 1) major_criteria <- major_criteria + 1

  minor_criteria <- 0
  if (food_sensitization == 1) minor_criteria <- minor_criteria + 1
  if (wheezing_apart_from_colds == 1) minor_criteria <- minor_criteria + 1
  if (eosinophilia_gt_4_percent == 1) minor_criteria <- minor_criteria + 1

  is_positive <- FALSE

  if (wheezing_episodes_gt_3_per_year == 1) {
    if (major_criteria >= 1 || minor_criteria >= 2) {
      is_positive <- TRUE
    }
  }

  result <- ""
  prediction <- ""

  if (is_positive) {
    result <- "Positive mAPI"
    prediction <- "High likelihood of developing asthma by school age (6-13 years)."
  } else {
    result <- "Negative mAPI"
    prediction <- "Low likelihood of developing asthma by school age."
  }

  return(list(
    mAPI_Result = result,
    Prediction = prediction,
    Criteria_Met = list(
      Major_Count = major_criteria,
      Minor_Count = minor_criteria,
      Frequent_Wheezing = wheezing_episodes_gt_3_per_year == 1
    )
  ))
}

modified_bishop_score <- function(dilation_cm, effacement_percent, station, consistency, position) {

  dilation_cm <- as.numeric(dilation_cm)
  effacement_percent <- as.numeric(effacement_percent)
  station <- as.numeric(station)
  consistency <- tolower(as.character(consistency))
  position <- tolower(as.character(position))

  if (any(is.na(c(dilation_cm, effacement_percent, station)))) {
    stop("Error: Dilation, Effacement, and Station must be numeric.")
  }

  if (!consistency %in% c("firm", "medium", "soft")) {
    stop("Error: Consistency must be 'firm', 'medium', or 'soft'.")
  }

  if (!position %in% c("posterior", "midposition", "anterior")) {
    stop("Error: Position must be 'posterior', 'midposition', or 'anterior'.")
  }

  score <- 0

  if (dilation_cm == 0) {
    score <- score + 0
  } else if (dilation_cm >= 1 && dilation_cm <= 2) {
    score <- score + 1
  } else if (dilation_cm >= 3 && dilation_cm <= 4) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (effacement_percent <= 30) {
    score <- score + 0
  } else if (effacement_percent >= 40 && effacement_percent <= 50) {
    score <- score + 1
  } else if (effacement_percent >= 60 && effacement_percent <= 70) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (station <= -3) {
    score <- score + 0
  } else if (station == -2) {
    score <- score + 1
  } else if (station == -1 || station == 0) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (consistency == "firm") {
    score <- score + 0
  } else if (consistency == "medium") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (position == "posterior") {
    score <- score + 0
  } else if (position == "midposition") {
    score <- score + 1
  } else {
    score <- score + 2
  }

  interpretation <- ""
  if (score > 8) {
    interpretation <- "Favorable Cervix. Probability of vaginal delivery is high (similar to spontaneous labor)."
  } else if (score < 6) {
    interpretation <- "Unfavorable Cervix. Cervical ripening agents recommended before induction."
  } else {
    interpretation <- "Indeterminate / Intermediate Favorability."
  }

  return(list(
    Bishop_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Dilation = dilation_cm,
      Effacement = effacement_percent,
      Station = station,
      Consistency = consistency,
      Position = position
    )
  ))
}

mbig_score <- function(gcs, focal_neuro_deficit, anti_coagulation, intoxication, skull_fracture_type, epidural_hematoma, subdural_hematoma_mm, intraparenchymal_hemorrhage_mm, subarachnoid_hemorrhage_type, intraventricular_hemorrhage) {

  gcs <- as.numeric(gcs)
  focal_neuro_deficit <- as.numeric(focal_neuro_deficit)
  anti_coagulation <- as.numeric(anti_coagulation)
  intoxication <- as.numeric(intoxication)
  epidural_hematoma <- as.numeric(epidural_hematoma)
  subdural_hematoma_mm <- as.numeric(subdural_hematoma_mm)
  intraparenchymal_hemorrhage_mm <- as.numeric(intraparenchymal_hemorrhage_mm)
  intraventricular_hemorrhage <- as.numeric(intraventricular_hemorrhage)
  skull_fracture_type <- tolower(as.character(skull_fracture_type))
  subarachnoid_hemorrhage_type <- tolower(as.character(subarachnoid_hemorrhage_type))

  if (any(is.na(c(gcs, focal_neuro_deficit, anti_coagulation, intoxication, epidural_hematoma, subdural_hematoma_mm, intraparenchymal_hemorrhage_mm, intraventricular_hemorrhage)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!skull_fracture_type %in% c("none", "non_displaced", "displaced")) {
    stop("Error: Skull fracture type must be 'none', 'non_displaced', or 'displaced'.")
  }

  if (!subarachnoid_hemorrhage_type %in% c("none", "trace", "localized", "extensive")) {
    stop("Error: SAH type must be 'none', 'trace' (<=3 sulci, <1mm), 'localized' (1 hemisphere, 1-3mm), or 'extensive' (bi-hemispheric, >3mm).")
  }

  is_mbig3 <- FALSE
  is_mbig2 <- FALSE

  if (gcs <= 12 || focal_neuro_deficit == 1) is_mbig3 <- TRUE
  if (anti_coagulation == 1) is_mbig3 <- TRUE
  if (intoxication == 1) is_mbig3 <- TRUE
  if (skull_fracture_type == "displaced") is_mbig3 <- TRUE
  if (epidural_hematoma == 1) is_mbig3 <- TRUE
  if (subdural_hematoma_mm >= 8) is_mbig3 <- TRUE
  if (intraparenchymal_hemorrhage_mm >= 8) is_mbig3 <- TRUE
  if (subarachnoid_hemorrhage_type == "extensive") is_mbig3 <- TRUE
  if (intraventricular_hemorrhage == 1) is_mbig3 <- TRUE

  if (!is_mbig3) {
    if (gcs == 13) is_mbig2 <- TRUE
    if (skull_fracture_type == "non_displaced") is_mbig2 <- TRUE
    if (subdural_hematoma_mm >= 4 && subdural_hematoma_mm < 8) is_mbig2 <- TRUE
    if (intraparenchymal_hemorrhage_mm >= 4 && intraparenchymal_hemorrhage_mm < 8) is_mbig2 <- TRUE
    if (subarachnoid_hemorrhage_type == "localized") is_mbig2 <- TRUE
  }

  category <- ""
  management <- ""

  if (is_mbig3) {
    category <- "mBIG 3 (High Risk)"
    management <- "Admit to ICU. Neurosurgery Consult. Repeat Head CT at 6 hours."
  } else if (is_mbig2) {
    category <- "mBIG 2 (Moderate Risk)"
    management <- "Admit to floor/observation (24-48 hrs). No routine repeat Head CT. No Neurosurgery Consult needed (unless deterioration)."
  } else {
    category <- "mBIG 1 (Low Risk)"
    management <- "Observe in ED for 6 hours. No repeat Head CT. No Neurosurgery Consult. Discharge if GCS 15 and stable."
  }

  return(list(
    mBIG_Category = category,
    Management_Plan = management,
    Inputs = list(
      GCS = gcs,
      Focal_Deficits = focal_neuro_deficit == 1,
      AntiCoagulants = anti_coagulation == 1,
      Intoxication = intoxication == 1,
      Skull_Fracture = skull_fracture_type,
      EDH = epidural_hematoma == 1,
      SDH_Size_mm = subdural_hematoma_mm,
      IPH_Size_mm = intraparenchymal_hemorrhage_mm,
      SAH_Type = subarachnoid_hemorrhage_type,
      IVH = intraventricular_hemorrhage == 1
    )
  ))
}

mews_score <- function(systolic_bp, heart_rate, respiratory_rate, temp_c, avpu_score) {

  systolic_bp <- as.numeric(systolic_bp)
  heart_rate <- as.numeric(heart_rate)
  respiratory_rate <- as.numeric(respiratory_rate)
  temp_c <- as.numeric(temp_c)
  avpu_score <- tolower(as.character(avpu_score))

  if (any(is.na(c(systolic_bp, heart_rate, respiratory_rate, temp_c)))) {
    stop("Error: Vital signs must be numeric.")
  }

  score <- 0

  if (systolic_bp <= 70) {
    score <- score + 3
  } else if (systolic_bp >= 71 && systolic_bp <= 80) {
    score <- score + 2
  } else if (systolic_bp >= 81 && systolic_bp <= 100) {
    score <- score + 1
  } else if (systolic_bp >= 200) {
    score <- score + 2
  }

  if (heart_rate <= 40) {
    score <- score + 2
  } else if (heart_rate >= 41 && heart_rate <= 50) {
    score <- score + 1
  } else if (heart_rate >= 101 && heart_rate <= 110) {
    score <- score + 1
  } else if (heart_rate >= 111 && heart_rate <= 129) {
    score <- score + 2
  } else if (heart_rate >= 130) {
    score <- score + 3
  }

  if (respiratory_rate < 9) {
    score <- score + 2
  } else if (respiratory_rate >= 15 && respiratory_rate <= 20) {
    score <- score + 1
  } else if (respiratory_rate >= 21 && respiratory_rate <= 29) {
    score <- score + 2
  } else if (respiratory_rate >= 30) {
    score <- score + 3
  }

  if (temp_c < 35) {
    score <- score + 2
  } else if (temp_c >= 38.5) {
    score <- score + 2
  }

  if (avpu_score == "alert") {
    score <- score + 0
  } else if (avpu_score == "verbal") {
    score <- score + 1
  } else if (avpu_score == "pain") {
    score <- score + 2
  } else if (avpu_score == "unresponsive") {
    score <- score + 3
  } else {
    stop("Error: AVPU score must be 'alert', 'verbal', 'pain', or 'unresponsive'.")
  }

  risk_level <- ""
  action <- ""

  if (score <= 2) {
    risk_level <- "Low Risk"
    action <- "Continue routine monitoring."
  } else if (score <= 4) {
    risk_level <- "Medium Risk"
    action <- "Inform charge nurse or ward doctor. Increase monitoring frequency."
  } else {
    risk_level <- "High Risk (Critical)"
    action <- "Rapid Response Team (RRT) or Critical Care evaluation required immediately."
  }

  return(list(
    MEWS_Score = score,
    Risk_Level = risk_level,
    Action = action,
    Inputs = list(
      SBP = systolic_bp,
      HR = heart_rate,
      RR = respiratory_rate,
      Temp = temp_c,
      AVPU = avpu_score
    )
  ))
}

modified_fatigue_impact_scale <- function(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17, q18, q19, q20, q21) {

  inputs <- c(q1, q2, q3, q4, q5, q6, q7, q8, q9, q10, q11, q12, q13, q14, q15, q16, q17, q18, q19, q20, q21)

  if (any(is.na(inputs))) {
    stop("Error: All 21 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 4)) {
    stop("Error: All item scores must be integers between 0 (Never) and 4 (Almost always).")
  }

  physical_indices <- c(4, 6, 7, 10, 13, 14, 17, 20, 21)
  cognitive_indices <- c(1, 2, 3, 5, 11, 12, 15, 16, 18, 19)
  psychosocial_indices <- c(8, 9)

  physical_score <- sum(inputs[physical_indices])
  cognitive_score <- sum(inputs[cognitive_indices])
  psychosocial_score <- sum(inputs[psychosocial_indices])

  total_score <- sum(inputs)

  impact_level <- ""
  if (total_score < 38) {
    impact_level <- "Low Impact (Less likely to have clinically significant fatigue)"
  } else {
    impact_level <- "High Impact (Clinically significant fatigue)"
  }

  return(list(
    MFIS_Total_Score = total_score,
    Impact_Interpretation = impact_level,
    Subscales = list(
      Physical_Score = physical_score,
      Cognitive_Score = cognitive_score,
      Psychosocial_Score = psychosocial_score
    ),
    Inputs = inputs
  ))
}

modified_finnegan_nas_score <- function(cry_score, sleep_score, moro_reflex_score, tremors_score, increased_muscle_tone, excoriation, myoclonic_jerks, generalized_convulsions, sweating, fever_score, frequent_yawning, mottling, nasal_stuffiness, frequent_sneezing, nasal_flaring, respiration_score, excessive_sucking, poor_feeding, projectile_vomiting, stool_score) {

  cry_score <- as.numeric(cry_score)
  sleep_score <- as.numeric(sleep_score)
  moro_reflex_score <- as.numeric(moro_reflex_score)
  tremors_score <- as.numeric(tremors_score)
  increased_muscle_tone <- as.numeric(increased_muscle_tone)
  excoriation <- as.numeric(excoriation)
  myoclonic_jerks <- as.numeric(myoclonic_jerks)
  generalized_convulsions <- as.numeric(generalized_convulsions)
  sweating <- as.numeric(sweating)
  fever_score <- as.numeric(fever_score)
  frequent_yawning <- as.numeric(frequent_yawning)
  mottling <- as.numeric(mottling)
  nasal_stuffiness <- as.numeric(nasal_stuffiness)
  frequent_sneezing <- as.numeric(frequent_sneezing)
  nasal_flaring <- as.numeric(nasal_flaring)
  respiration_score <- as.numeric(respiration_score)
  excessive_sucking <- as.numeric(excessive_sucking)
  poor_feeding <- as.numeric(poor_feeding)
  projectile_vomiting <- as.numeric(projectile_vomiting)
  stool_score <- as.numeric(stool_score)

  inputs <- c(cry_score, sleep_score, moro_reflex_score, tremors_score, increased_muscle_tone, excoriation, myoclonic_jerks, generalized_convulsions, sweating, fever_score, frequent_yawning, mottling, nasal_stuffiness, frequent_sneezing, nasal_flaring, respiration_score, excessive_sucking, poor_feeding, projectile_vomiting, stool_score)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!cry_score %in% c(0, 2, 3)) stop("Error: Cry score must be 0, 2, or 3.")
  if (!sleep_score %in% c(0, 1, 2, 3)) stop("Error: Sleep score must be 0, 1, 2, or 3.")
  if (!moro_reflex_score %in% c(0, 2, 3)) stop("Error: Moro reflex score must be 0, 2, or 3.")
  if (!tremors_score %in% c(0, 1, 2, 3, 4)) stop("Error: Tremors score must be 0, 1, 2, 3, or 4.")
  if (!fever_score %in% c(0, 1, 2)) stop("Error: Fever score must be 0, 1, or 2.")
  if (!respiration_score %in% c(0, 1, 2)) stop("Error: Respiration score must be 0, 1, or 2.")
  if (!stool_score %in% c(0, 2, 3)) stop("Error: Stool score must be 0, 2, or 3.")

  score <- 0

  score <- score + cry_score
  score <- score + sleep_score
  score <- score + moro_reflex_score
  score <- score + tremors_score
  score <- score + fever_score
  score <- score + respiration_score
  score <- score + stool_score

  if (increased_muscle_tone == 1) score <- score + 2
  if (excoriation == 1) score <- score + 1
  if (myoclonic_jerks == 1) score <- score + 3
  if (generalized_convulsions == 1) score <- score + 5

  if (sweating == 1) score <- score + 1
  if (frequent_yawning == 1) score <- score + 1
  if (mottling == 1) score <- score + 1
  if (nasal_stuffiness == 1) score <- score + 1
  if (frequent_sneezing == 1) score <- score + 1
  if (nasal_flaring == 1) score <- score + 2

  if (excessive_sucking == 1) score <- score + 1
  if (poor_feeding == 1) score <- score + 2
  if (projectile_vomiting == 1) score <- score + 2

  action <- ""
  if (score >= 8) {
    action <- "Score >= 8. Suggests significant withdrawal. Pharmacologic intervention may be indicated if score remains >= 8 on subsequent assessments (e.g., average of 3 scores >= 8 or 2 scores >= 12)."
  } else {
    action <- "Score < 8. Suggests mild withdrawal. Continue supportive care and monitoring."
  }

  return(list(
    NAS_Score = score,
    Recommendation = action,
    Inputs = list(
      CNS = list(Cry=cry_score, Sleep=sleep_score, Moro=moro_reflex_score, Tremors=tremors_score, Tone=increased_muscle_tone, Excoriation=excoriation, Myoclonus=myoclonic_jerks, Convulsions=generalized_convulsions),
      Metabolic_Respiratory = list(Sweating=sweating, Fever=fever_score, Yawning=frequent_yawning, Mottling=mottling, Stuffiness=nasal_stuffiness, Sneezing=frequent_sneezing, Flaring=nasal_flaring, Respiration=respiration_score),
      GI = list(Sucking=excessive_sucking, Feeding=poor_feeding, Vomiting=projectile_vomiting, Stool=stool_score)
    )
  ))
}

modified_fisher_grade <- function(sah_present, ich_ivh_present, clot_thickness_ge_1mm) {

  sah_present <- as.numeric(sah_present)
  ich_ivh_present <- as.numeric(ich_ivh_present)
  clot_thickness_ge_1mm <- as.numeric(clot_thickness_ge_1mm)

  if (any(is.na(c(sah_present, ich_ivh_present, clot_thickness_ge_1mm)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  grade <- 0
  description <- ""
  vasospasm_risk <- ""

  if (sah_present == 0) {
    if (ich_ivh_present == 0) {
      grade <- 0
      description <- "No subarachnoid hemorrhage or intraventricular hemorrhage."
      vasospasm_risk <- "0% (Minimal)"
    } else {
      grade <- 0
      description <- "No SAH detected (IVH only does not fit standard Modified Fisher definition for vasospasm risk stratification)."
      vasospasm_risk <- "Unknown"
    }
  } else {
    if (clot_thickness_ge_1mm == 0 && ich_ivh_present == 0) {
      grade <- 1
      description <- "Focal or diffuse thin (<1mm) SAH, no IVH."
      vasospasm_risk <- "24%"
    } else if (clot_thickness_ge_1mm == 0 && ich_ivh_present == 1) {
      grade <- 2
      description <- "Focal or diffuse thin (<1mm) SAH, with IVH."
      vasospasm_risk <- "33%"
    } else if (clot_thickness_ge_1mm == 1 && ich_ivh_present == 0) {
      grade <- 3
      description <- "Thick (>=1mm) SAH, no IVH."
      vasospasm_risk <- "33%"
    } else {
      grade <- 4
      description <- "Thick (>=1mm) SAH, with IVH."
      vasospasm_risk <- "40%"
    }
  }

  return(list(
    Modified_Fisher_Grade = grade,
    CT_Findings = description,
    Symptomatic_Vasospasm_Risk = vasospasm_risk,
    Inputs = list(
      SAH = sah_present == 1,
      Thick_SAH = clot_thickness_ge_1mm == 1,
      IVH = ich_ivh_present == 1
    )
  ))
}

modified_gps_score <- function(crp_mg_l, albumin_g_l) {

  crp_mg_l <- as.numeric(crp_mg_l)
  albumin_g_l <- as.numeric(albumin_g_l)

  if (any(is.na(c(crp_mg_l, albumin_g_l)))) {
    stop("Error: CRP and Albumin must be numeric.")
  }

  score <- 0

  if (crp_mg_l <= 10) {
    score <- 0
  } else {
    if (albumin_g_l >= 35) {
      score <- 1
    } else {
      score <- 2
    }
  }

  prognosis <- ""
  if (score == 0) {
    prognosis <- "Good Prognosis"
  } else if (score == 1) {
    prognosis <- "Intermediate Prognosis"
  } else {
    prognosis <- "Poor Prognosis"
  }

  return(list(
    mGPS_Score = score,
    Prognosis = prognosis,
    Inputs = list(
      CRP_mg_L = crp_mg_l,
      Albumin_g_L = albumin_g_l
    )
  ))
}

modified_hoehn_yahr_scale <- function(stage) {

  stage <- as.numeric(stage)

  valid_stages <- c(1, 1.5, 2, 2.5, 3, 4, 5)

  if (!stage %in% valid_stages) {
    stop("Error: Stage must be 1, 1.5, 2, 2.5, 3, 4, or 5.")
  }

  description <- ""

  if (stage == 1) {
    description <- "Unilateral involvement only."
  } else if (stage == 1.5) {
    description <- "Unilateral and axial involvement."
  } else if (stage == 2) {
    description <- "Bilateral involvement without impairment of balance."
  } else if (stage == 2.5) {
    description <- "Mild bilateral disease with recovery on pull test."
  } else if (stage == 3) {
    description <- "Mild to moderate bilateral disease; some postural instability; physically independent."
  } else if (stage == 4) {
    description <- "Severe disability; still able to walk or stand unassisted."
  } else {
    description <- "Wheelchair bound or bedridden unless aided."
  }

  return(list(
    Hoehn_Yahr_Stage = stage,
    Description = description
  ))
}

modified_mallampati_classification <- function(class_grade) {

  class_grade <- as.numeric(class_grade)

  if (is.na(class_grade) || !class_grade %in% 1:4) {
    stop("Error: Mallampati Class must be an integer between 1 and 4.")
  }

  visibility <- ""
  difficulty <- ""

  if (class_grade == 1) {
    visibility <- "Soft palate, fauces, uvula, and pillars visible."
    difficulty <- "No difficulty."
  } else if (class_grade == 2) {
    visibility <- "Soft palate, fauces, and uvula visible."
    difficulty <- "No difficulty."
  } else if (class_grade == 3) {
    visibility <- "Soft palate and base of uvula visible."
    difficulty <- "Moderate difficulty."
  } else {
    visibility <- "Soft palate not visible at all."
    difficulty <- "Severe difficulty."
  }

  return(list(
    Mallampati_Class = paste0("Class ", as.roman(class_grade)),
    Visibility = visibility,
    Intubation_Difficulty = difficulty
  ))
}

modified_minnesota_detox_scale <- function(pulse_rate, diastolic_bp, tremor, sweat, hallucinations, agitation, orientation, seizures) {

  pulse_rate <- as.numeric(pulse_rate)
  diastolic_bp <- as.numeric(diastolic_bp)
  tremor <- as.numeric(tremor)
  sweat <- as.numeric(sweat)
  hallucinations <- as.numeric(hallucinations)
  agitation <- as.numeric(agitation)
  orientation <- as.numeric(orientation)
  seizures <- as.numeric(seizures)

  if (any(is.na(c(pulse_rate, diastolic_bp, tremor, sweat, hallucinations, agitation, orientation, seizures)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (pulse_rate < 80) score <- score + 0
  else if (pulse_rate < 100) score <- score + 1
  else if (pulse_rate < 120) score <- score + 2
  else score <- score + 3

  if (diastolic_bp < 90) score <- score + 0
  else if (diastolic_bp < 100) score <- score + 1
  else if (diastolic_bp < 110) score <- score + 2
  else score <- score + 3

  if (tremor < 0 || tremor > 3) stop("Error: Tremor score must be between 0 and 3.")
  score <- score + tremor

  if (sweat < 0 || sweat > 3) stop("Error: Sweat score must be between 0 and 3.")
  score <- score + sweat

  if (hallucinations < 0 || hallucinations > 3) stop("Error: Hallucinations score must be between 0 and 3.")
  score <- score + hallucinations

  if (agitation < 0 || agitation > 3) stop("Error: Agitation score must be between 0 and 3.")
  score <- score + agitation

  if (orientation < 0 || orientation > 4) stop("Error: Orientation score must be between 0 and 4.")
  score <- score + orientation

  if (seizures < 0 || seizures > 2) stop("Error: Seizures score must be between 0 and 2.")
  score <- score + seizures

  severity <- ""
  if (score <= 9) {
    severity <- "Mild/Moderate Withdrawal"
  } else {
    severity <- "Severe Withdrawal (Risk of Delirium Tremens)"
  }

  return(list(
    MMDS_Score = score,
    Severity_Interpretation = severity,
    Inputs = list(
      Pulse = pulse_rate,
      DBP = diastolic_bp,
      Tremor = tremor,
      Sweat = sweat,
      Hallucinations = hallucinations,
      Agitation = agitation,
      Orientation = orientation,
      Seizures = seizures
    )
  ))
}

modified_nih_stroke_scale <- function(loc_questions, loc_commands, gaze, visual_fields, motor_arm_left, motor_arm_right, motor_leg_left, motor_leg_right, sensory, best_language, extinction_inattention) {

  loc_questions <- as.numeric(loc_questions)
  loc_commands <- as.numeric(loc_commands)
  gaze <- as.numeric(gaze)
  visual_fields <- as.numeric(visual_fields)
  motor_arm_left <- as.numeric(motor_arm_left)
  motor_arm_right <- as.numeric(motor_arm_right)
  motor_leg_left <- as.numeric(motor_leg_left)
  motor_leg_right <- as.numeric(motor_leg_right)
  sensory <- as.numeric(sensory)
  best_language <- as.numeric(best_language)
  extinction_inattention <- as.numeric(extinction_inattention)

  if (any(is.na(c(loc_questions, loc_commands, gaze, visual_fields, motor_arm_left, motor_arm_right, motor_leg_left, motor_leg_right, sensory, best_language, extinction_inattention)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- loc_questions + loc_commands + gaze + visual_fields + motor_arm_left + motor_arm_right + motor_leg_left + motor_leg_right + sensory + best_language + extinction_inattention

  severity <- ""
  if (score == 0) {
    severity <- "No Stroke Symptoms"
  } else if (score >= 1 && score <= 4) {
    severity <- "Minor Stroke"
  } else if (score >= 5 && score <= 15) {
    severity <- "Moderate Stroke"
  } else if (score >= 16 && score <= 20) {
    severity <- "Moderate to Severe Stroke"
  } else {
    severity <- "Severe Stroke"
  }

  return(list(
    mNIHSS_Score = score,
    Severity = severity,
    Inputs = list(
      LOC_Questions = loc_questions,
      LOC_Commands = loc_commands,
      Gaze = gaze,
      Visual = visual_fields,
      Motor_Arm_L = motor_arm_left,
      Motor_Arm_R = motor_arm_right,
      Motor_Leg_L = motor_leg_left,
      Motor_Leg_R = motor_leg_right,
      Sensory = sensory,
      Language = best_language,
      Extinction = extinction_inattention
    )
  ))
}

modified_rankin_scale <- function(score) {

  score <- as.numeric(score)

  if (is.na(score) || !score %in% 0:6) {
    stop("Error: Score must be an integer between 0 and 6.")
  }

  description <- ""

  if (score == 0) {
    description <- "No symptoms at all."
  } else if (score == 1) {
    description <- "No significant disability despite symptoms; able to carry out all usual duties and activities."
  } else if (score == 2) {
    description <- "Slight disability; unable to carry out all previous activities, but able to look after own affairs without assistance."
  } else if (score == 3) {
    description <- "Moderate disability; requiring some help, but able to walk without assistance."
  } else if (score == 4) {
    description <- "Moderately severe disability; unable to walk without assistance and unable to attend to own bodily needs without assistance."
  } else if (score == 5) {
    description <- "Severe disability; bedridden, incontinent and requiring constant nursing care and attention."
  } else {
    description <- "Dead."
  }

  return(list(
    mRS_Score = score,
    Description = description
  ))
}

mrs_9q_score <- function(symptoms_present, able_usual_activities, able_own_affairs, able_walk_unassisted, able_bodily_needs) {

  symptoms_present <- as.numeric(symptoms_present)
  able_usual_activities <- as.numeric(able_usual_activities)
  able_own_affairs <- as.numeric(able_own_affairs)
  able_walk_unassisted <- as.numeric(able_walk_unassisted)
  able_bodily_needs <- as.numeric(able_bodily_needs)

  if (any(is.na(c(symptoms_present, able_usual_activities, able_own_affairs, able_walk_unassisted, able_bodily_needs)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0
  description <- ""

  if (symptoms_present == 0) {
    score <- 0
    description <- "No symptoms at all."
  } else if (able_usual_activities == 1) {
    score <- 1
    description <- "No significant disability despite symptoms; able to carry out all usual duties and activities."
  } else if (able_own_affairs == 1) {
    score <- 2
    description <- "Slight disability; unable to carry out all previous activities, but able to look after own affairs without assistance."
  } else if (able_walk_unassisted == 1) {
    score <- 3
    description <- "Moderate disability; requiring some help, but able to walk without assistance."
  } else if (able_bodily_needs == 1) {
    score <- 4
    description <- "Moderately severe disability; unable to walk without assistance and unable to attend to own bodily needs without assistance."
  } else {
    score <- 5
    description <- "Severe disability; bedridden, incontinent and requiring constant nursing care and attention."
  }

  return(list(
    mRS_Score = score,
    Description = description,
    Inputs = list(
      Symptoms = symptoms_present == 1,
      Usual_Activities = able_usual_activities == 1,
      Independent_Affairs = able_own_affairs == 1,
      Walk_Unassisted = able_walk_unassisted == 1,
      Bodily_Needs_Met = able_bodily_needs == 1
    )
  ))
}

mrecist_assessment <- function(target_lesions_sum_change, new_lesions_present, nontarget_lesions_progression, arterial_enhancement_present) {

  target_lesions_sum_change <- as.numeric(target_lesions_sum_change)
  new_lesions_present <- as.numeric(new_lesions_present)
  nontarget_lesions_progression <- as.numeric(nontarget_lesions_progression)
  arterial_enhancement_present <- as.numeric(arterial_enhancement_present)

  if (any(is.na(c(target_lesions_sum_change, new_lesions_present, nontarget_lesions_progression, arterial_enhancement_present)))) {
    stop("Error: All inputs must be numeric (0 or 1 for binary).")
  }

  response_category <- ""
  description <- ""

  if (new_lesions_present == 1) {
    response_category <- "PD (Progressive Disease)"
    description <- "New lesions detected."
  } else if (nontarget_lesions_progression == 1) {
    response_category <- "PD (Progressive Disease)"
    description <- "Unequivocal progression of non-target lesions."
  } else if (target_lesions_sum_change >= 20) {
    response_category <- "PD (Progressive Disease)"
    description <- "Increase in sum of diameters of viable (enhancing) target lesions by >= 20%."
  } else if (arterial_enhancement_present == 0 && target_lesions_sum_change == -100) {
    response_category <- "CR (Complete Response)"
    description <- "Disappearance of any intratumoral arterial enhancement in all target lesions."
  } else if (target_lesions_sum_change <= -30) {
    response_category <- "PR (Partial Response)"
    description <- "Decrease in sum of diameters of viable (enhancing) target lesions by >= 30%."
  } else {
    response_category <- "SD (Stable Disease)"
    description <- "Any cases that do not qualify for either partial response or progressive disease."
  }

  return(list(
    mRECIST_Response = response_category,
    Description = description,
    Inputs = list(
      Target_Change_Percent = target_lesions_sum_change,
      New_Lesions = new_lesions_present == 1,
      NonTarget_Progression = nontarget_lesions_progression == 1,
      Enhancement_Present = arterial_enhancement_present == 1
    )
  ))
}

modified_sofa_score <- function(spo2_percent, fio2_percent, platelet_count, bilirubin_mg_dl, hypotension_level, gcs_score, creatinine_mg_dl) {

  spo2_percent <- as.numeric(spo2_percent)
  fio2_percent <- as.numeric(fio2_percent)
  platelet_count <- as.numeric(platelet_count)
  bilirubin_mg_dl <- as.numeric(bilirubin_mg_dl)
  hypotension_level <- as.numeric(hypotension_level)
  gcs_score <- as.numeric(gcs_score)
  creatinine_mg_dl <- as.numeric(creatinine_mg_dl)

  if (any(is.na(c(spo2_percent, fio2_percent, platelet_count, bilirubin_mg_dl, hypotension_level, gcs_score, creatinine_mg_dl)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (fio2_percent > 1) {
    fio2_fraction <- fio2_percent / 100
  } else {
    fio2_fraction <- fio2_percent
  }

  sf_ratio <- spo2_percent / fio2_fraction

  resp_score <- 0
  if (sf_ratio < 150) {
    resp_score <- 4
  } else if (sf_ratio < 235) {
    resp_score <- 3
  } else if (sf_ratio < 315) {
    resp_score <- 2
  } else if (sf_ratio < 400) {
    resp_score <- 1
  } else {
    resp_score <- 0
  }

  coag_score <- 0
  if (platelet_count < 20) {
    coag_score <- 4
  } else if (platelet_count < 50) {
    coag_score <- 3
  } else if (platelet_count < 100) {
    coag_score <- 2
  } else if (platelet_count < 150) {
    coag_score <- 1
  } else {
    coag_score <- 0
  }

  liver_score <- 0
  if (bilirubin_mg_dl > 12.0) {
    liver_score <- 4
  } else if (bilirubin_mg_dl >= 6.0) {
    liver_score <- 3
  } else if (bilirubin_mg_dl >= 2.0) {
    liver_score <- 2
  } else if (bilirubin_mg_dl >= 1.2) {
    liver_score <- 1
  } else {
    liver_score <- 0
  }

  cv_score <- hypotension_level

  cns_score <- 0
  if (gcs_score < 6) {
    cns_score <- 4
  } else if (gcs_score >= 6 && gcs_score <= 9) {
    cns_score <- 3
  } else if (gcs_score >= 10 && gcs_score <= 12) {
    cns_score <- 2
  } else if (gcs_score >= 13 && gcs_score <= 14) {
    cns_score <- 1
  } else {
    cns_score <- 0
  }

  renal_score <- 0
  if (creatinine_mg_dl > 5.0) {
    renal_score <- 4
  } else if (creatinine_mg_dl >= 3.5) {
    renal_score <- 3
  } else if (creatinine_mg_dl >= 2.0) {
    renal_score <- 2
  } else if (creatinine_mg_dl >= 1.2) {
    renal_score <- 1
  } else {
    renal_score <- 0
  }

  total_score <- resp_score + coag_score + liver_score + cv_score + cns_score + renal_score

  mortality_est <- ""
  if (total_score <= 6) {
    mortality_est <- "< 10%"
  } else if (total_score <= 9) {
    mortality_est <- "15-20%"
  } else if (total_score <= 12) {
    mortality_est <- "40-50%"
  } else if (total_score <= 14) {
    mortality_est <- "50-60%"
  } else {
    mortality_est <- "> 80%"
  }

  return(list(
    mSOFA_Score = total_score,
    Mortality_Estimate = mortality_est,
    Subscores = list(
      Respiratory = resp_score,
      Coagulation = coag_score,
      Liver = liver_score,
      Cardiovascular = cv_score,
      CNS = cns_score,
      Renal = renal_score
    )
  ))
}

modified_sgarbossa_criteria <- function(st_elevation_concordant_mm, st_depression_v1_v3_mm, st_elevation_discordant_mm, s_wave_amplitude_mm) {

  st_elevation_concordant_mm <- as.numeric(st_elevation_concordant_mm)
  st_depression_v1_v3_mm <- as.numeric(st_depression_v1_v3_mm)
  st_elevation_discordant_mm <- as.numeric(st_elevation_discordant_mm)
  s_wave_amplitude_mm <- as.numeric(s_wave_amplitude_mm)

  if (any(is.na(c(st_elevation_concordant_mm, st_depression_v1_v3_mm, st_elevation_discordant_mm, s_wave_amplitude_mm)))) {
    stop("Error: All inputs must be numeric (in mm).")
  }

  criteria_met <- FALSE
  positive_criteria <- c()

  if (st_elevation_concordant_mm >= 1) {
    criteria_met <- TRUE
    positive_criteria <- c(positive_criteria, "Concordant ST elevation >= 1 mm")
  }

  if (st_depression_v1_v3_mm >= 1) {
    criteria_met <- TRUE
    positive_criteria <- c(positive_criteria, "Concordant ST depression >= 1 mm in V1-V3")
  }

  ratio <- 0
  if (s_wave_amplitude_mm != 0) {
    ratio <- abs(st_elevation_discordant_mm) / abs(s_wave_amplitude_mm)
  } else {
    if (st_elevation_discordant_mm > 0) ratio <- 1
  }

  if (ratio >= 0.25) {
    criteria_met <- TRUE
    positive_criteria <- c(positive_criteria, paste0("Discordant ST elevation / S-wave ratio >= 0.25 (Ratio: ", round(ratio, 2), ")"))
  }

  result <- ""
  if (criteria_met) {
    result <- "Positive for Acute MI (STEMI equivalent)"
  } else {
    result <- "Negative by Modified Sgarbossa Criteria"
  }

  return(list(
    Result = result,
    Criteria_Met = if (length(positive_criteria) > 0) positive_criteria else "None",
    Calculated_Ratio = round(ratio, 2),
    Inputs = list(
      STE_Concordant = st_elevation_concordant_mm,
      STD_Concordant_V1_V3 = st_depression_v1_v3_mm,
      STE_Discordant = st_elevation_discordant_mm,
      S_Wave_Amp = s_wave_amplitude_mm
    )
  ))
}

modified_soar_stroke_score <- function(stroke_type, ocsp_classification, age, pre_stroke_mrs, nihss_score) {

  stroke_type <- tolower(as.character(stroke_type))
  ocsp_classification <- tolower(as.character(ocsp_classification))
  age <- as.numeric(age)
  pre_stroke_mrs <- as.numeric(pre_stroke_mrs)
  nihss_score <- as.numeric(nihss_score)

  if (any(is.na(c(age, pre_stroke_mrs, nihss_score)))) {
    stop("Error: Age, mRS, and NIHSS must be numeric.")
  }

  if (!stroke_type %in% c("infarct", "hemorrhage")) {
    stop("Error: Stroke type must be 'infarct' or 'hemorrhage'.")
  }

  valid_ocsp <- c("lacs", "pacs", "pocs", "tacs")
  if (!ocsp_classification %in% valid_ocsp) {
    stop("Error: OCSP classification must be 'lacs', 'pacs', 'pocs', or 'tacs'.")
  }

  score <- 0

  if (stroke_type == "hemorrhage") {
    score <- score + 1
  }

  if (ocsp_classification == "pocs") {
    score <- score + 1
  } else if (ocsp_classification == "tacs") {
    score <- score + 2
  }

  if (age > 85) {
    score <- score + 2
  } else if (age >= 66) {
    score <- score + 1
  }

  if (pre_stroke_mrs == 5) {
    score <- score + 2
  } else if (pre_stroke_mrs >= 3) {
    score <- score + 1
  }

  if (nihss_score >= 11) {
    score <- score + 2
  } else if (nihss_score >= 5) {
    score <- score + 1
  }

  mortality_risk <- ""

  if (score == 0) {
    mortality_risk <- "1.0%"
  } else if (score == 1) {
    mortality_risk <- "1.5%"
  } else if (score == 2) {
    mortality_risk <- "6.5%"
  } else if (score == 3) {
    mortality_risk <- "9.2%"
  } else if (score == 4) {
    mortality_risk <- "19.5%"
  } else if (score == 5) {
    mortality_risk <- "26.2%"
  } else {
    mortality_risk <- "~50% (High Risk)"
  }

  return(list(
    mSOAR_Score = score,
    Est_Inpatient_Mortality = mortality_risk,
    Inputs = list(
      Stroke_Type = stroke_type,
      OCSP = ocsp_classification,
      Age = age,
      Pre_Stroke_mRS = pre_stroke_mrs,
      NIHSS = nihss_score
    )
  ))
}

madrs_depression_score <- function(apparent_sadness, reported_sadness, inner_tension, reduced_sleep, reduced_appetite, concentration_difficulties, lassitude, inability_to_feel, pessimistic_thoughts, suicidal_thoughts) {

  inputs <- c(apparent_sadness, reported_sadness, inner_tension, reduced_sleep, reduced_appetite, concentration_difficulties, lassitude, inability_to_feel, pessimistic_thoughts, suicidal_thoughts)

  if (any(is.na(inputs))) {
    stop("Error: All 10 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 6)) {
    stop("Error: All item scores must be integers between 0 and 6.")
  }

  total_score <- sum(inputs)

  severity <- ""
  if (total_score <= 6) {
    severity <- "Normal / Symptom Absent"
  } else if (total_score <= 19) {
    severity <- "Mild Depression"
  } else if (total_score <= 34) {
    severity <- "Moderate Depression"
  } else {
    severity <- "Severe Depression"
  }

  return(list(
    MADRS_Score = total_score,
    Severity_Category = severity,
    Inputs = list(
      Q1_Apparent_Sadness = apparent_sadness,
      Q2_Reported_Sadness = reported_sadness,
      Q3_Inner_Tension = inner_tension,
      Q4_Reduced_Sleep = reduced_sleep,
      Q5_Reduced_Appetite = reduced_appetite,
      Q6_Concentration = concentration_difficulties,
      Q7_Lassitude = lassitude,
      Q8_Inability_Feel = inability_to_feel,
      Q9_Pessimism = pessimistic_thoughts,
      Q10_Suicidal_Thoughts = suicidal_thoughts
    )
  ))
}

montreal_ibd_classification <- function(disease_type, age_years = NULL, cd_location = NULL, cd_behavior = NULL, cd_upper_gi = 0, cd_perianal = 0, uc_extent = NULL, uc_severity = NULL) {

  disease_type <- tolower(as.character(disease_type))

  if (!disease_type %in% c("crohns", "uc")) {
    stop("Error: Disease type must be 'crohns' or 'uc'.")
  }

  classification_code <- ""
  description <- ""

  if (disease_type == "crohns") {

    if (is.null(age_years) || !is.numeric(age_years)) stop("Error: Age required for Crohn's classification.")
    if (is.null(cd_location)) stop("Error: Location required for Crohn's classification.")
    if (is.null(cd_behavior)) stop("Error: Behavior required for Crohn's classification.")

    a_code <- ""
    if (age_years < 17) a_code <- "A1"
    else if (age_years <= 40) a_code <- "A2"
    else a_code <- "A3"

    l_code <- ""
    if (cd_location == "ileal") l_code <- "L1"
    else if (cd_location == "colonic") l_code <- "L2"
    else if (cd_location == "ileocolonic") l_code <- "L3"
    else l_code <- "L?"

    if (cd_upper_gi == 1) l_code <- paste0(l_code, "+L4")

    b_code <- ""
    if (cd_behavior == "non_stricturing_non_penetrating") b_code <- "B1"
    else if (cd_behavior == "stricturing") b_code <- "B2"
    else if (cd_behavior == "penetrating") b_code <- "B3"

    if (cd_perianal == 1) b_code <- paste0(b_code, "p")

    classification_code <- paste(a_code, l_code, b_code)

  } else {

    if (is.null(uc_extent)) stop("Error: Extent required for UC classification.")
    if (is.null(uc_severity)) stop("Error: Severity required for UC classification.")

    e_code <- ""
    if (uc_extent == "proctitis") e_code <- "E1"
    else if (uc_extent == "left_sided") e_code <- "E2"
    else if (uc_extent == "extensive") e_code <- "E3"

    s_code <- ""
    if (uc_severity == "remission") s_code <- "S0"
    else if (uc_severity == "mild") s_code <- "S1"
    else if (uc_severity == "moderate") s_code <- "S2"
    else if (uc_severity == "severe") s_code <- "S3"

    classification_code <- paste(e_code, s_code)
  }

  return(list(
    Classification_Code = classification_code,
    Inputs = list(
      Type = disease_type,
      Age = age_years,
      CD_Location = cd_location,
      CD_Behavior = cd_behavior,
      CD_Modifiers = paste("L4:", cd_upper_gi, "p:", cd_perianal),
      UC_Extent = uc_extent,
      UC_Severity = uc_severity
    )
  ))
}

moca_cognitive_assessment <- function(visuospatial, naming, attention, language, abstraction, delayed_recall, orientation, education_years) {

  visuospatial <- as.numeric(visuospatial)
  naming <- as.numeric(naming)
  attention <- as.numeric(attention)
  language <- as.numeric(language)
  abstraction <- as.numeric(abstraction)
  delayed_recall <- as.numeric(delayed_recall)
  orientation <- as.numeric(orientation)
  education_years <- as.numeric(education_years)

  if (any(is.na(c(visuospatial, naming, attention, language, abstraction, delayed_recall, orientation, education_years)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (visuospatial < 0 || visuospatial > 5) stop("Error: Visuospatial score must be 0-5.")
  if (naming < 0 || naming > 3) stop("Error: Naming score must be 0-3.")
  if (attention < 0 || attention > 6) stop("Error: Attention score must be 0-6.")
  if (language < 0 || language > 3) stop("Error: Language score must be 0-3.")
  if (abstraction < 0 || abstraction > 2) stop("Error: Abstraction score must be 0-2.")
  if (delayed_recall < 0 || delayed_recall > 5) stop("Error: Delayed recall score must be 0-5.")
  if (orientation < 0 || orientation > 6) stop("Error: Orientation score must be 0-6.")

  raw_score <- visuospatial + naming + attention + language + abstraction + delayed_recall + orientation

  final_score <- raw_score
  education_bonus <- 0

  if (education_years <= 12 && raw_score < 30) {
    final_score <- final_score + 1
    education_bonus <- 1
  }

  if (final_score > 30) {
    final_score <- 30
  }

  interpretation <- ""
  if (final_score >= 26) {
    interpretation <- "Normal Cognitive Function"
  } else if (final_score >= 18) {
    interpretation <- "Mild Cognitive Impairment"
  } else if (final_score >= 10) {
    interpretation <- "Moderate Cognitive Impairment"
  } else {
    interpretation <- "Severe Cognitive Impairment"
  }

  return(list(
    MoCA_Score = final_score,
    Interpretation = interpretation,
    Education_Adjustment = if(education_bonus == 1) "+1 point applied (<= 12 years education)" else "None",
    Inputs = list(
      Visuospatial_Executive = visuospatial,
      Naming = naming,
      Attention = attention,
      Language = language,
      Abstraction = abstraction,
      Delayed_Recall = delayed_recall,
      Orientation = orientation,
      Education_Years = education_years
    )
  ))
}

calculate_mme <- function(codeine_mg_day = 0, fentanyl_transdermal_mcg_hr = 0, hydrocodone_mg_day = 0, hydromorphone_mg_day = 0, methadone_mg_day = 0, morphine_mg_day = 0, oxycodone_mg_day = 0, oxymorphone_mg_day = 0, tapentadol_mg_day = 0, tramadol_mg_day = 0) {

  codeine_mg_day <- as.numeric(codeine_mg_day)
  fentanyl_transdermal_mcg_hr <- as.numeric(fentanyl_transdermal_mcg_hr)
  hydrocodone_mg_day <- as.numeric(hydrocodone_mg_day)
  hydromorphone_mg_day <- as.numeric(hydromorphone_mg_day)
  methadone_mg_day <- as.numeric(methadone_mg_day)
  morphine_mg_day <- as.numeric(morphine_mg_day)
  oxycodone_mg_day <- as.numeric(oxycodone_mg_day)
  oxymorphone_mg_day <- as.numeric(oxymorphone_mg_day)
  tapentadol_mg_day <- as.numeric(tapentadol_mg_day)
  tramadol_mg_day <- as.numeric(tramadol_mg_day)

  if (any(is.na(c(codeine_mg_day, fentanyl_transdermal_mcg_hr, hydrocodone_mg_day, hydromorphone_mg_day, methadone_mg_day, morphine_mg_day, oxycodone_mg_day, oxymorphone_mg_day, tapentadol_mg_day, tramadol_mg_day)))) {
    stop("Error: All inputs must be numeric.")
  }

  mme_total <- 0

  mme_total <- mme_total + (codeine_mg_day * 0.15)

  mme_total <- mme_total + (fentanyl_transdermal_mcg_hr * 2.4)

  mme_total <- mme_total + (hydrocodone_mg_day * 1)

  mme_total <- mme_total + (hydromorphone_mg_day * 4)

  mme_total <- mme_total + (morphine_mg_day * 1)

  mme_total <- mme_total + (oxycodone_mg_day * 1.5)

  mme_total <- mme_total + (oxymorphone_mg_day * 3)

  mme_total <- mme_total + (tapentadol_mg_day * 0.4)

  mme_total <- mme_total + (tramadol_mg_day * 0.1)

  methadone_factor <- 0
  if (methadone_mg_day > 0) {
    if (methadone_mg_day <= 20) {
      methadone_factor <- 4
    } else if (methadone_mg_day <= 40) {
      methadone_factor <- 8
    } else if (methadone_mg_day <= 60) {
      methadone_factor <- 10
    } else {
      methadone_factor <- 12
    }
  }
  mme_total <- mme_total + (methadone_mg_day * methadone_factor)

  risk_category <- ""
  if (mme_total < 50) {
    risk_category <- "Low Risk (< 50 MME/day)"
  } else if (mme_total < 90) {
    risk_category <- "Increased Risk (50-90 MME/day). Reassess benefits/risks, consider prescribing naloxone."
  } else {
    risk_category <- "High Risk (>= 90 MME/day). Avoid increasing dosage; consider tapering."
  }

  return(list(
    Total_MME_Day = round(mme_total, 1),
    Risk_Interpretation = risk_category,
    Inputs = list(
      Codeine = codeine_mg_day,
      Fentanyl_patch = fentanyl_transdermal_mcg_hr,
      Hydrocodone = hydrocodone_mg_day,
      Hydromorphone = hydromorphone_mg_day,
      Methadone = methadone_mg_day,
      Morphine = morphine_mg_day,
      Oxycodone = oxycodone_mg_day,
      Oxymorphone = oxymorphone_mg_day,
      Tapentadol = tapentadol_mg_day,
      Tramadol = tramadol_mg_day
    )
  ))
}

mulbsta_score <- function(multilobular_infiltration, lymphocyte_count, bacterial_coinfection, smoking_history, hypertension_history, age) {

  multilobular_infiltration <- as.numeric(multilobular_infiltration)
  lymphocyte_count <- as.numeric(lymphocyte_count)
  bacterial_coinfection <- as.numeric(bacterial_coinfection)
  smoking_history <- as.numeric(smoking_history)
  hypertension_history <- as.numeric(hypertension_history)
  age <- as.numeric(age)

  if (any(is.na(c(multilobular_infiltration, lymphocyte_count, bacterial_coinfection, smoking_history, hypertension_history, age)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (multilobular_infiltration == 1) {
    score <- score + 5
  }

  if (lymphocyte_count <= 0.8) {
    score <- score + 4
  }

  if (bacterial_coinfection == 1) {
    score <- score + 4
  }

  if (smoking_history == 1) {
    score <- score + 3
  }

  if (hypertension_history == 1) {
    score <- score + 2
  }

  if (age >= 60) {
    score <- score + 2
  }

  risk_category <- ""
  mortality_risk_90d <- ""

  if (score <= 11) {
    risk_category <- "Low Risk"
    mortality_risk_90d <- "Low (< 5%)"
  } else {
    risk_category <- "High Risk"
    mortality_risk_90d <- "High (> 16%)"
  }

  return(list(
    MuLBSTA_Score = score,
    Risk_Category = risk_category,
    Mortality_Risk_90_Day = mortality_risk_90d,
    Inputs = list(
      Multilobular_Infiltrates = multilobular_infiltration == 1,
      Lymphocytes_le_0_8 = lymphocyte_count <= 0.8,
      Bacterial_Coinfection = bacterial_coinfection == 1,
      Smoking_History = smoking_history == 1,
      Hypertension = hypertension_history == 1,
      Age_ge_60 = age >= 60
    )
  ))
}

multiple_myeloma_diagnosis <- function(bmpc_percent, plasmacytoma, calcium_elevated, renal_insufficiency, anemia, bone_lesions, flc_ratio_gt_100, mri_focal_lesions_gt_1, m_protein_g_dl = 0) {

  bmpc_percent <- as.numeric(bmpc_percent)
  plasmacytoma <- as.numeric(plasmacytoma)
  calcium_elevated <- as.numeric(calcium_elevated)
  renal_insufficiency <- as.numeric(renal_insufficiency)
  anemia <- as.numeric(anemia)
  bone_lesions <- as.numeric(bone_lesions)
  flc_ratio_gt_100 <- as.numeric(flc_ratio_gt_100)
  mri_focal_lesions_gt_1 <- as.numeric(mri_focal_lesions_gt_1)
  m_protein_g_dl <- as.numeric(m_protein_g_dl)

  if (any(is.na(c(bmpc_percent, plasmacytoma, calcium_elevated, renal_insufficiency, anemia, bone_lesions, flc_ratio_gt_100, mri_focal_lesions_gt_1)))) {
    stop("Error: All inputs must be numeric.")
  }

  crab_present <- (calcium_elevated == 1 || renal_insufficiency == 1 || anemia == 1 || bone_lesions == 1)

  slim_present <- (bmpc_percent >= 60 || flc_ratio_gt_100 == 1 || mri_focal_lesions_gt_1 == 1)

  diagnosis <- ""

  if (bmpc_percent >= 10 || plasmacytoma == 1) {
    if (crab_present || slim_present) {
      diagnosis <- "Active Multiple Myeloma"
    } else {
      if (bmpc_percent >= 10 || m_protein_g_dl >= 3) {
        diagnosis <- "Smoldering Multiple Myeloma"
      } else {
        diagnosis <- "MGUS (Monoclonal Gammopathy of Undetermined Significance)"
      }
    }
  } else {
    if (m_protein_g_dl >= 3) {
      diagnosis <- "Smoldering Multiple Myeloma"
    } else {
      diagnosis <- "MGUS (Monoclonal Gammopathy of Undetermined Significance)"
    }
  }

  return(list(
    Diagnosis = diagnosis,
    Criteria_Met = list(
      BMPC_Threshold = if(bmpc_percent >= 10) ">=10%" else "<10%",
      CRAB_Features = crab_present,
      SLiM_Biomarkers = slim_present
    ),
    Inputs = list(
      BMPC = bmpc_percent,
      Plasmacytoma = plasmacytoma == 1,
      CRAB = list(C=calcium_elevated==1, R=renal_insufficiency==1, A=anemia==1, B=bone_lesions==1),
      SLiM = list(S_BMPC60=bmpc_percent>=60, Li_FLC=flc_ratio_gt_100==1, M_MRI=mri_focal_lesions_gt_1==1),
      M_Protein = m_protein_g_dl
    )
  ))
}

iss_multiple_myeloma <- function(beta2_microglobulin, albumin) {

  beta2_microglobulin <- as.numeric(beta2_microglobulin)
  albumin <- as.numeric(albumin)

  if (any(is.na(c(beta2_microglobulin, albumin)))) {
    stop("Error: All inputs must be numeric.")
  }

  stage <- ""
  median_survival <- ""

  if (beta2_microglobulin >= 5.5) {
    stage <- "Stage III"
    median_survival <- "29 months"
  } else if (beta2_microglobulin < 3.5 && albumin >= 3.5) {
    stage <- "Stage I"
    median_survival <- "62 months"
  } else {
    stage <- "Stage II"
    median_survival <- "44 months"
  }

  return(list(
    ISS_Stage = stage,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      B2M_mg_L = beta2_microglobulin,
      Albumin_g_dL = albumin
    )
  ))
}

multiple_myeloma_response_criteria <- function(serum_m_baseline, serum_m_nadir, serum_m_current, urine_m_baseline, urine_m_nadir, urine_m_current, flc_diff_baseline, flc_diff_nadir, flc_diff_current, bone_marrow_pct, immunofixation_status, flc_ratio, plasmacytoma_status, bone_lesions_new_increased, hypercalcemia_present) {

  serum_m_baseline <- as.numeric(serum_m_baseline)
  serum_m_nadir <- as.numeric(serum_m_nadir)
  serum_m_current <- as.numeric(serum_m_current)

  urine_m_baseline <- as.numeric(urine_m_baseline)
  urine_m_nadir <- as.numeric(urine_m_nadir)
  urine_m_current <- as.numeric(urine_m_current)

  flc_diff_baseline <- as.numeric(flc_diff_baseline)
  flc_diff_nadir <- as.numeric(flc_diff_nadir)
  flc_diff_current <- as.numeric(flc_diff_current)

  bone_marrow_pct <- as.numeric(bone_marrow_pct)
  flc_ratio <- as.numeric(flc_ratio)

  immunofixation_status <- tolower(as.character(immunofixation_status))
  plasmacytoma_status <- tolower(as.character(plasmacytoma_status))

  bone_lesions_new_increased <- as.numeric(bone_lesions_new_increased)
  hypercalcemia_present <- as.numeric(hypercalcemia_present)

  if (any(is.na(c(serum_m_current, urine_m_current)))) {
    stop("Error: Current M-protein levels are required.")
  }

  is_pd <- FALSE

  if (!is.na(serum_m_nadir)) {
    serum_abs_increase <- serum_m_current - serum_m_nadir
    if (serum_m_nadir > 0) {
      serum_pct_increase <- (serum_abs_increase / serum_m_nadir) * 100
      if (serum_pct_increase >= 25 && serum_abs_increase >= 0.5) is_pd <- TRUE
    } else if (serum_abs_increase >= 0.5) {
      is_pd <- TRUE
    }
  }

  if (!is.na(urine_m_nadir)) {
    urine_abs_increase <- urine_m_current - urine_m_nadir
    if (urine_m_nadir > 0) {
      urine_pct_increase <- (urine_abs_increase / urine_m_nadir) * 100
      if (urine_pct_increase >= 25 && urine_abs_increase >= 200) is_pd <- TRUE
    } else if (urine_abs_increase >= 200) {
      is_pd <- TRUE
    }
  }

  if (!is.na(flc_diff_nadir)) {
    flc_abs_increase <- flc_diff_current - flc_diff_nadir
    if (flc_diff_nadir > 0) {
      flc_pct_increase <- (flc_abs_increase / flc_diff_nadir) * 100
      if (flc_pct_increase >= 25 && flc_abs_increase >= 10) is_pd <- TRUE
    } else if (flc_abs_increase >= 10) {
      is_pd <- TRUE
    }
  }

  if (bone_lesions_new_increased == 1) is_pd <- TRUE
  if (hypercalcemia_present == 1) is_pd <- TRUE
  if (plasmacytoma_status == "new" || plasmacytoma_status == "increased") is_pd <- TRUE

  response <- "Stable Disease (SD)"

  if (is_pd) {
    response <- "Progressive Disease (PD)"
  } else {

    has_pr_features <- FALSE
    if (!is.na(serum_m_baseline) && serum_m_baseline > 0) {
      if ((serum_m_current / serum_m_baseline) <= 0.5) has_pr_features <- TRUE
    }
    if (!is.na(urine_m_baseline) && urine_m_baseline > 0) {
      if ((urine_m_current / urine_m_baseline) <= 0.1) has_pr_features <- TRUE
      if (urine_m_current < 200) has_pr_features <- TRUE
    }
    if (plasmacytoma_status == "decreased_50" || plasmacytoma_status == "disappeared") has_pr_features <- TRUE

    has_vgpr_features <- FALSE
    if (immunofixation_status == "positive") {
      serum_90_red <- FALSE
      if (!is.na(serum_m_baseline) && serum_m_baseline > 0) {
        if ((serum_m_current / serum_m_baseline) <= 0.1) serum_90_red <- TRUE
      }
      if (serum_90_red && urine_m_current < 100) has_vgpr_features <- TRUE
    } else if (immunofixation_status == "negative") {
      has_vgpr_features <- TRUE
    }

    has_cr_features <- FALSE
    if (immunofixation_status == "negative") {
      if ((is.na(bone_marrow_pct) || bone_marrow_pct < 5) &&
          (plasmacytoma_status == "disappeared" || plasmacytoma_status == "none")) {
        has_cr_features <- TRUE
      }
    }

    has_scr_features <- FALSE
    if (has_cr_features) {
      if (!is.na(flc_ratio) && flc_ratio >= 0.26 && flc_ratio <= 1.65) {
        has_scr_features <- TRUE
      }
    }

    if (has_scr_features) {
      response <- "Stringent Complete Response (sCR)"
    } else if (has_cr_features) {
      response <- "Complete Response (CR)"
    } else if (has_vgpr_features) {
      response <- "Very Good Partial Response (VGPR)"
    } else if (has_pr_features) {
      response <- "Partial Response (PR)"
    }
  }

  return(list(
    Response_Category = response,
    Inputs = list(
      Serum_M = list(Baseline=serum_m_baseline, Nadir=serum_m_nadir, Current=serum_m_current),
      Urine_M = list(Baseline=urine_m_baseline, Nadir=urine_m_nadir, Current=urine_m_current),
      FLC_Diff = list(Baseline=flc_diff_baseline, Nadir=flc_diff_nadir, Current=flc_diff_current),
      Status = list(Immunofixation=immunofixation_status, Plasmacytoma=plasmacytoma_status, Bone_Lesions_New=bone_lesions_new_increased==1, Hypercalcemia=hypercalcemia_present==1)
    )
  ))
}

mumtaz_cirrhosis_readmission_score <- function(age, primary_payer, elixhauser_comorbidity_score, non_alcoholic_etiology, ascites_present, hepatic_encephalopathy_present, variceal_bleeding_present, hcc_present, paracentesis_performed, hemodialysis_performed, disposition) {

  age <- as.numeric(age)
  elixhauser_comorbidity_score <- as.numeric(elixhauser_comorbidity_score)
  non_alcoholic_etiology <- as.numeric(non_alcoholic_etiology)
  ascites_present <- as.numeric(ascites_present)
  hepatic_encephalopathy_present <- as.numeric(hepatic_encephalopathy_present)
  variceal_bleeding_present <- as.numeric(variceal_bleeding_present)
  hcc_present <- as.numeric(hcc_present)
  paracentesis_performed <- as.numeric(paracentesis_performed)
  hemodialysis_performed <- as.numeric(hemodialysis_performed)
  primary_payer <- tolower(as.character(primary_payer))
  disposition <- tolower(as.character(disposition))

  if (any(is.na(c(age, elixhauser_comorbidity_score, non_alcoholic_etiology, ascites_present, hepatic_encephalopathy_present, variceal_bleeding_present, hcc_present, paracentesis_performed, hemodialysis_performed)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!primary_payer %in% c("medicare", "medicaid", "private", "self_pay", "no_charge_other")) {
    stop("Error: Primary payer must be 'medicare', 'medicaid', 'private', 'self_pay', or 'no_charge_other'.")
  }

  if (!disposition %in% c("routine", "transfer", "home_health", "ama")) {
    stop("Error: Disposition must be 'routine', 'transfer', 'home_health', or 'ama' (Against Medical Advice).")
  }

  score <- 0

  if (age >= 18 && age <= 39) {
    score <- score + 7
  } else if (age >= 40 && age <= 64) {
    score <- score + 4
  } else { # >= 65
    score <- score + 0
  }

  if (primary_payer == "medicare") {
    score <- score + 4
  } else if (primary_payer == "medicaid") {
    score <- score + 5
  } else if (primary_payer == "private") {
    score <- score + 0
  } else if (primary_payer == "self_pay") {
    score <- score - 3
  } else {
    score <- score - 1
  }

  if (elixhauser_comorbidity_score >= 3) {
    score <- score + 2
  }

  if (non_alcoholic_etiology == 1) {
    score <- score + 2
  }

  if (ascites_present == 1) {
    score <- score + 5
  }

  if (hepatic_encephalopathy_present == 1) {
    score <- score + 4
  }

  if (variceal_bleeding_present == 1) {
    score <- score - 7
  }

  if (hcc_present == 1) {
    score <- score + 3
  }

  if (paracentesis_performed == 1) {
    score <- score + 4
  }

  if (hemodialysis_performed == 1) {
    score <- score + 7
  }

  if (disposition == "routine") {
    score <- score + 0
  } else if (disposition == "transfer") {
    score <- score - 1
  } else if (disposition == "home_health") {
    score <- score + 3
  } else { # ama
    score <- score + 10
  }

  risk_probability <- ""
  risk_category <- ""

  if (score <= 15) {
    risk_category <- "Low Risk (<20% Readmission Risk)"
    risk_probability <- "< 20%"
  } else if (score >= 16 && score <= 20) {
    risk_category <- "Medium Risk (20-30% Readmission Risk)"
    risk_probability <- "20% - 30%"
  } else {
    risk_category <- "High Risk (>30% Readmission Risk)"
    risk_probability <- "> 30%"
  }

  return(list(
    Mumtaz_Score = score,
    Risk_Category = risk_category,
    Est_30_Day_Readmission_Risk = risk_probability,
    Inputs = list(
      Age = age,
      Payer = primary_payer,
      Elixhauser_ge_3 = elixhauser_comorbidity_score >= 3,
      Non_Alcoholic = non_alcoholic_etiology == 1,
      Ascites = ascites_present == 1,
      HE = hepatic_encephalopathy_present == 1,
      Variceal_Bleed = variceal_bleeding_present == 1,
      HCC = hcc_present == 1,
      Paracentesis = paracentesis_performed == 1,
      Hemodialysis = hemodialysis_performed == 1,
      Disposition = disposition
    )
  ))
}

murray_lung_injury_score <- function(pao2_fio2_ratio, peep, respiratory_compliance, cxr_quadrants) {

  pao2_fio2_ratio <- as.numeric(pao2_fio2_ratio)
  peep <- as.numeric(peep)
  respiratory_compliance <- as.numeric(respiratory_compliance)
  cxr_quadrants <- as.numeric(cxr_quadrants)

  if (any(is.na(c(pao2_fio2_ratio, peep, respiratory_compliance, cxr_quadrants)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (cxr_quadrants < 0 || cxr_quadrants > 4) {
    stop("Error: CXR quadrants must be between 0 and 4.")
  }

  hypoxemia_score <- 0
  if (pao2_fio2_ratio >= 300) {
    hypoxemia_score <- 0
  } else if (pao2_fio2_ratio >= 225) {
    hypoxemia_score <- 1
  } else if (pao2_fio2_ratio >= 175) {
    hypoxemia_score <- 2
  } else if (pao2_fio2_ratio >= 100) {
    hypoxemia_score <- 3
  } else {
    hypoxemia_score <- 4
  }

  peep_score <- 0
  if (peep <= 5) {
    peep_score <- 0
  } else if (peep >= 6 && peep <= 8) {
    peep_score <- 1
  } else if (peep >= 9 && peep <= 11) {
    peep_score <- 2
  } else if (peep >= 12 && peep <= 14) {
    peep_score <- 3
  } else {
    peep_score <- 4
  }

  compliance_score <- 0
  if (respiratory_compliance >= 80) {
    compliance_score <- 0
  } else if (respiratory_compliance >= 60) {
    compliance_score <- 1
  } else if (respiratory_compliance >= 40) {
    compliance_score <- 2
  } else if (respiratory_compliance >= 20) {
    compliance_score <- 3
  } else {
    compliance_score <- 4
  }

  cxr_score <- cxr_quadrants

  total_points <- hypoxemia_score + peep_score + compliance_score + cxr_score
  final_score <- total_points / 4

  severity <- ""
  if (final_score == 0) {
    severity <- "No Lung Injury"
  } else if (final_score <= 2.5) {
    severity <- "Mild to Moderate Lung Injury"
  } else {
    severity <- "Severe Lung Injury (ARDS)"
  }

  return(list(
    Murray_Score = round(final_score, 2),
    Severity_Assessment = severity,
    Subscores = list(
      Hypoxemia_Points = hypoxemia_score,
      PEEP_Points = peep_score,
      Compliance_Points = compliance_score,
      CXR_Points = cxr_score
    )
  ))
}

mipss70_score <- function(hemoglobin_g_l, wbc_x10_9_l, platelets_x10_9_l, blasts_percent, fibrosis_grade_ge_2, constitutional_symptoms, calr_type1_absent, hmr_mutation_count, karyotype_unfavorable = NULL) {

  hemoglobin_g_l <- as.numeric(hemoglobin_g_l)
  wbc_x10_9_l <- as.numeric(wbc_x10_9_l)
  platelets_x10_9_l <- as.numeric(platelets_x10_9_l)
  blasts_percent <- as.numeric(blasts_percent)
  fibrosis_grade_ge_2 <- as.numeric(fibrosis_grade_ge_2)
  constitutional_symptoms <- as.numeric(constitutional_symptoms)
  calr_type1_absent <- as.numeric(calr_type1_absent)
  hmr_mutation_count <- as.numeric(hmr_mutation_count)

  if (any(is.na(c(hemoglobin_g_l, wbc_x10_9_l, platelets_x10_9_l, blasts_percent, fibrosis_grade_ge_2, constitutional_symptoms, calr_type1_absent, hmr_mutation_count)))) {
    stop("Error: All core inputs must be numeric.")
  }

  score <- 0

  if (hemoglobin_g_l < 100) {
    score <- score + 1
  }

  if (wbc_x10_9_l > 25) {
    score <- score + 2
  }

  if (platelets_x10_9_l < 100) {
    score <- score + 2
  }

  if (blasts_percent >= 2) {
    score <- score + 1
  }

  if (fibrosis_grade_ge_2 == 1) {
    score <- score + 1
  }

  if (constitutional_symptoms == 1) {
    score <- score + 1
  }

  if (calr_type1_absent == 1) {
    score <- score + 1
  }

  if (hmr_mutation_count == 1) {
    score <- score + 1
  } else if (hmr_mutation_count >= 2) {
    score <- score + 2
  }

  mipss70_risk <- ""
  if (score <= 1) {
    mipss70_risk <- "Low Risk"
  } else if (score <= 4) {
    mipss70_risk <- "Intermediate Risk"
  } else {
    mipss70_risk <- "High Risk"
  }

  mipss70_plus_result <- NULL

  if (!is.null(karyotype_unfavorable)) {
    karyotype_unfavorable <- as.numeric(karyotype_unfavorable)
    if (!is.na(karyotype_unfavorable)) {

      plus_score <- score
      if (karyotype_unfavorable == 1) {
        plus_score <- plus_score + 3
      }

      plus_risk <- ""
      if (plus_score <= 2) {
        plus_risk <- "Low Risk"
      } else if (plus_score <= 4) {
        plus_risk <- "Intermediate Risk"
      } else if (plus_score <= 8) {
        plus_risk <- "High Risk"
      } else {
        plus_risk <- "Very High Risk"
      }

      mipss70_plus_result <- list(
        MIPSS70_Plus_Score = plus_score,
        MIPSS70_Plus_Risk_Category = plus_risk
      )
    }
  }

  return(list(
    MIPSS70_Score = score,
    MIPSS70_Risk_Category = mipss70_risk,
    MIPSS70_Plus = mipss70_plus_result,
    Inputs = list(
      Hb_lt_100 = hemoglobin_g_l < 100,
      WBC_gt_25 = wbc_x10_9_l > 25,
      Plt_lt_100 = platelets_x10_9_l < 100,
      Blasts_ge_2 = blasts_percent >= 2,
      Fibrosis_ge_2 = fibrosis_grade_ge_2 == 1,
      Symptoms = constitutional_symptoms == 1,
      CALR_Type1_Absent = calr_type1_absent == 1,
      HMR_Mutations = hmr_mutation_count,
      Unfavorable_Karyotype = if(!is.null(karyotype_unfavorable)) (karyotype_unfavorable == 1) else "Not Provided"
    )
  ))
}

mg_adl_score <- function(talking, chewing, swallowing, breathing, tooth_brushing, combing_hair, standing_from_chair, double_vision, eyelid_droop) {

  inputs <- c(talking, chewing, swallowing, breathing, tooth_brushing, combing_hair, standing_from_chair, double_vision, eyelid_droop)

  if (any(is.na(inputs))) {
    stop("Error: All 8 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 3)) {
    stop("Error: All item scores must be integers between 0 and 3.")
  }

  total_score <- sum(inputs)

  return(list(
    MG_ADL_Score = total_score,
    Interpretation = "Higher scores indicate more severe disease impact. A 2-point improvement is considered clinically significant.",
    Inputs = list(
      Talking = talking,
      Chewing = chewing,
      Swallowing = swallowing,
      Breathing = breathing,
      Tooth_Brushing = tooth_brushing,
      Combing_Hair = combing_hair,
      Standing = standing_from_chair,
      Double_Vision = double_vision,
      Eyelid_Droop = eyelid_droop
    )
  ))
}

mgfa_classification <- function(intubated, ocular_weakness_only, generalized_severity, predominant_symptoms) {

  intubated <- as.numeric(intubated)
  ocular_weakness_only <- as.numeric(ocular_weakness_only)
  generalized_severity <- tolower(as.character(generalized_severity))
  predominant_symptoms <- tolower(as.character(predominant_symptoms))

  if (any(is.na(c(intubated, ocular_weakness_only)))) {
    stop("Error: Intubation and Ocular status must be numeric (0 or 1).")
  }

  if (intubated == 1) {
    return(list(
      MGFA_Class = "Class V",
      Description = "Defined by intubation, with or without mechanical ventilation, except when used during routine postoperative management. Use of feeding tube without intubation places patient in Class IVb."
    ))
  }

  if (ocular_weakness_only == 1) {
    return(list(
      MGFA_Class = "Class I",
      Description = "Any ocular muscle weakness; may have weakness of eye closure. All other muscle strength is normal."
    ))
  }

  base_class <- ""
  base_desc <- ""

  if (generalized_severity == "mild") {
    base_class <- "Class II"
    base_desc <- "Mild weakness affecting other than ocular muscles; may also have ocular muscle weakness of any severity."
  } else if (generalized_severity == "moderate") {
    base_class <- "Class III"
    base_desc <- "Moderate weakness affecting other than ocular muscles; may also have ocular muscle weakness of any severity."
  } else if (generalized_severity == "severe") {
    base_class <- "Class IV"
    base_desc <- "Severe weakness affecting other than ocular muscles; may also have ocular muscle weakness of any severity."
  } else {
    stop("Error: Severity must be 'mild', 'moderate', or 'severe' for generalized weakness.")
  }

  subtype <- ""
  subtype_desc <- ""

  if (predominant_symptoms == "limb_axial") {
    subtype <- "a"
    subtype_desc <- " Predominantly affecting limb, axial muscles, or both. May also have lesser involvement of oropharyngeal muscles."
  } else if (predominant_symptoms == "oropharyngeal_respiratory") {
    subtype <- "b"
    subtype_desc <- " Predominantly affecting oropharyngeal, respiratory muscles, or both. May also have lesser or equal involvement of limb, axial muscles, or both."
  } else {
    stop("Error: Predominant symptoms must be 'limb_axial' or 'oropharyngeal_respiratory'.")
  }

  return(list(
    MGFA_Class = paste0(base_class, subtype),
    Description = paste0(base_desc, subtype_desc)
  ))
}

mysec_pm_score <- function(age, hemoglobin_g_dl, platelet_count, circulating_blasts_percent, calr_unmutated, constitutional_symptoms) {

  age <- as.numeric(age)
  hemoglobin_g_dl <- as.numeric(hemoglobin_g_dl)
  platelet_count <- as.numeric(platelet_count)
  circulating_blasts_percent <- as.numeric(circulating_blasts_percent)
  calr_unmutated <- as.numeric(calr_unmutated)
  constitutional_symptoms <- as.numeric(constitutional_symptoms)

  if (any(is.na(c(age, hemoglobin_g_dl, platelet_count, circulating_blasts_percent, calr_unmutated, constitutional_symptoms)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0.23 * age

  if (hemoglobin_g_dl < 11) {
    score <- score + 17.5
  }

  if (platelet_count < 150) {
    score <- score + 7.5
  }

  if (circulating_blasts_percent >= 3) {
    score <- score + 13
  }

  if (calr_unmutated == 1) {
    score <- score + 16.3
  }

  if (constitutional_symptoms == 1) {
    score <- score + 7.4
  }

  risk_category <- ""
  median_survival <- ""

  if (score < 11) {
    risk_category <- "Low Risk"
    median_survival <- "Not Reached"
  } else if (score >= 11 && score < 14) {
    risk_category <- "Intermediate-1 Risk"
    median_survival <- "9.3 years"
  } else if (score >= 14 && score < 16) {
    risk_category <- "Intermediate-2 Risk"
    median_survival <- "4.4 years"
  } else {
    risk_category <- "High Risk"
    median_survival <- "2.0 years"
  }

  return(list(
    MYSEC_PM_Score = round(score, 1),
    Risk_Category = risk_category,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Age = age,
      Hb_lt_11 = hemoglobin_g_dl < 11,
      Plt_lt_150 = platelet_count < 150,
      Blasts_ge_3 = circulating_blasts_percent >= 3,
      CALR_Unmutated = calr_unmutated == 1,
      Symptoms = constitutional_symptoms == 1
    )
  ))
}

myxedema_coma_score <- function(temp_c, cns_status, gi_symptoms, precipitating_event, bradycardia, ecg_changes, hypotension, hyponatremia, hypoglycemia, hypoxemia_hypercapnia) {

  temp_c <- as.numeric(temp_c)
  cns_status <- tolower(as.character(cns_status))
  gi_symptoms <- tolower(as.character(gi_symptoms))
  precipitating_event <- as.numeric(precipitating_event)
  bradycardia <- as.numeric(bradycardia)
  ecg_changes <- as.numeric(ecg_changes)
  hypotension <- as.numeric(hypotension)
  hyponatremia <- as.numeric(hyponatremia)
  hypoglycemia <- as.numeric(hypoglycemia)
  hypoxemia_hypercapnia <- as.numeric(hypoxemia_hypercapnia)

  if (any(is.na(c(temp_c, precipitating_event, bradycardia, ecg_changes, hypotension, hyponatremia, hypoglycemia, hypoxemia_hypercapnia)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  score <- 0

  if (temp_c > 35) {
    score <- score + 0
  } else if (temp_c >= 32 && temp_c <= 35) {
    score <- score + 10
  } else {
    score <- score + 20
  }

  if (cns_status == "absent") {
    score <- score + 0
  } else if (cns_status == "drowsy_lethargic") {
    score <- score + 10
  } else if (cns_status == "obtunded") {
    score <- score + 15
  } else if (cns_status == "stupor_coma") {
    score <- score + 20
  } else {
    stop("Error: CNS status must be 'absent', 'drowsy_lethargic', 'obtunded', or 'stupor_coma'.")
  }

  if (gi_symptoms == "absent") {
    score <- score + 0
  } else if (gi_symptoms == "anorexia_pain_constipation") {
    score <- score + 5
  } else if (gi_symptoms == "decreased_motility_ileus") {
    score <- score + 10
  } else {
    stop("Error: GI symptoms must be 'absent', 'anorexia_pain_constipation', or 'decreased_motility_ileus'.")
  }

  if (precipitating_event == 1) score <- score + 10

  if (bradycardia == 1) score <- score + 10
  if (ecg_changes == 1) score <- score + 10
  if (hypotension == 1) score <- score + 10

  if (hyponatremia == 1) score <- score + 10
  if (hypoglycemia == 1) score <- score + 10
  if (hypoxemia_hypercapnia == 1) score <- score + 10

  interpretation <- ""
  if (score < 25) {
    interpretation <- "Unlikely Myxedema Coma"
  } else if (score >= 25 && score <= 59) {
    interpretation <- "Suggestive of Myxedema Coma (Risk present)"
  } else {
    interpretation <- "Diagnostic of Myxedema Coma (Highly Likely)"
  }

  return(list(
    Diagnostic_Score = score,
    Interpretation = interpretation,
    Inputs = list(
      Temp = temp_c,
      CNS = cns_status,
      GI = gi_symptoms,
      Precipitant = precipitating_event == 1,
      CV_Signs = sum(bradycardia, ecg_changes, hypotension),
      Metabolic_Signs = sum(hyponatremia, hypoglycemia, hypoxemia_hypercapnia)
    )
  ))
}

maddreys_discriminant_function <- function(pt_patient_seconds, pt_control_seconds, bilirubin, bilirubin_units = "mg/dL") {

  pt_patient_seconds <- as.numeric(pt_patient_seconds)
  pt_control_seconds <- as.numeric(pt_control_seconds)
  bilirubin <- as.numeric(bilirubin)

  if (any(is.na(c(pt_patient_seconds, pt_control_seconds, bilirubin)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_mg_dl <- bilirubin
  if (bilirubin_units == "umol/L" || bilirubin_units == "micromol/L") {
    bili_mg_dl <- bilirubin / 17.1
  }

  mdf_score <- 4.6 * (pt_patient_seconds - pt_control_seconds) + bili_mg_dl

  interpretation <- ""
  mortality_risk <- ""
  recommendation <- ""

  if (mdf_score >= 32) {
    interpretation <- "Poor Prognosis (Severe Alcoholic Hepatitis)"
    mortality_risk <- "High (30-50% 1-month mortality)"
    recommendation <- "Consider corticosteroids (e.g., Prednisolone) or Pentoxifylline if no contraindications."
  } else {
    interpretation <- "Good Prognosis (Mild/Moderate Alcoholic Hepatitis)"
    mortality_risk <- "Low"
    recommendation <- "Supportive care."
  }

  return(list(
    Maddrey_DF_Score = round(mdf_score, 1),
    Prognosis = interpretation,
    Mortality_Risk = mortality_risk,
    Recommendation = recommendation,
    Inputs = list(
      PT_Patient = pt_patient_seconds,
      PT_Control = pt_control_seconds,
      Bilirubin_mg_dL = round(bili_mg_dl, 2)
    )
  ))
}

maggic_risk_score <- function(age, lvef, sbp, bmi, creatinine, nyha_class, sex, smoker, diabetes, copd, heart_failure_duration_gt_18mo, beta_blocker_use, acei_arb_use, creatinine_units = "mg/dL") {

  age <- as.numeric(age)
  lvef <- as.numeric(lvef)
  sbp <- as.numeric(sbp)
  bmi <- as.numeric(bmi)
  creatinine <- as.numeric(creatinine)
  nyha_class <- as.numeric(nyha_class)

  sex <- tolower(as.character(sex))
  smoker <- as.numeric(smoker)
  diabetes <- as.numeric(diabetes)
  copd <- as.numeric(copd)
  heart_failure_duration_gt_18mo <- as.numeric(heart_failure_duration_gt_18mo)
  beta_blocker_use <- as.numeric(beta_blocker_use)
  acei_arb_use <- as.numeric(acei_arb_use)

  if (any(is.na(c(age, lvef, sbp, bmi, creatinine, nyha_class, smoker, diabetes, copd, heart_failure_duration_gt_18mo, beta_blocker_use, acei_arb_use)))) {
    stop("Error: All inputs must be numeric.")
  }

  creat_umol <- creatinine
  if (creatinine_units == "mg/dL") {
    creat_umol <- creatinine * 88.4
  }

  score <- 0

  if (age < 55) score <- score + 0
  else if (age >= 55 && age <= 59) score <- score + 1
  else if (age >= 60 && age <= 64) score <- score + 2
  else if (age >= 65 && age <= 69) score <- score + 4
  else if (age >= 70 && age <= 74) score <- score + 6
  else if (age >= 75 && age <= 79) score <- score + 8
  else score <- score + 10

  if (lvef < 20) score <- score + 7
  else if (lvef >= 20 && lvef <= 24) score <- score + 6
  else if (lvef >= 25 && lvef <= 29) score <- score + 5
  else if (lvef >= 30 && lvef <= 34) score <- score + 3
  else if (lvef >= 35 && lvef <= 39) score <- score + 2
  else score <- score + 0

  if (sbp < 110) score <- score + 5
  else if (sbp >= 110 && sbp <= 119) score <- score + 4
  else if (sbp >= 120 && sbp <= 129) score <- score + 3
  else if (sbp >= 130 && sbp <= 139) score <- score + 2
  else if (sbp >= 140 && sbp <= 149) score <- score + 1
  else score <- score + 0

  if (bmi < 15) score <- score + 6
  else if (bmi >= 15 && bmi <= 19) score <- score + 5
  else if (bmi >= 20 && bmi <= 24) score <- score + 3
  else if (bmi >= 25 && bmi <= 29) score <- score + 2
  else score <- score + 0

  if (creat_umol < 90) score <- score + 0
  else if (creat_umol >= 90 && creat_umol <= 109) score <- score + 1
  else if (creat_umol >= 110 && creat_umol <= 129) score <- score + 2
  else if (creat_umol >= 130 && creat_umol <= 149) score <- score + 3
  else if (creat_umol >= 150 && creat_umol <= 169) score <- score + 4
  else if (creat_umol >= 170 && creat_umol <= 209) score <- score + 5
  else if (creat_umol >= 210 && creat_umol <= 249) score <- score + 6
  else score <- score + 8

  if (nyha_class == 1) score <- score + 0
  else if (nyha_class == 2) score <- score + 2
  else if (nyha_class == 3) score <- score + 6
  else if (nyha_class == 4) score <- score + 8

  if (sex == "male") score <- score + 1

  if (smoker == 1) score <- score + 1

  if (diabetes == 1) score <- score + 3

  if (copd == 1) score <- score + 2

  if (heart_failure_duration_gt_18mo == 1) score <- score + 2

  if (beta_blocker_use == 0) score <- score + 5

  if (acei_arb_use == 0) score <- score + 1

  risk_1yr <- NA
  risk_3yr <- NA

  if (score <= 6) { risk_1yr <- 2; risk_3yr <- 5 }
  else if (score <= 10) { risk_1yr <- 3; risk_3yr <- 7 }
  else if (score <= 14) { risk_1yr <- 4; risk_3yr <- 11 }
  else if (score <= 18) { risk_1yr <- 6; risk_3yr <- 16 }
  else if (score <= 22) { risk_1yr <- 9; risk_3yr <- 22 }
  else if (score <= 26) { risk_1yr <- 13; risk_3yr <- 31 }
  else if (score <= 30) { risk_1yr <- 18; risk_3yr <- 42 }
  else if (score <= 34) { risk_1yr <- 26; risk_3yr <- 54 }
  else if (score <= 38) { risk_1yr <- 35; risk_3yr <- 67 }
  else { risk_1yr <- 47; risk_3yr <- 78 }

  return(list(
    MAGGIC_Risk_Score = score,
    One_Year_Mortality_Risk = paste0(risk_1yr, "%"),
    Three_Year_Mortality_Risk = paste0(risk_3yr, "%"),
    Inputs = list(
      Age = age, LVEF = lvef, SBP = sbp, BMI = bmi, Creatinine_umol = round(creat_umol, 1),
      NYHA = nyha_class, Sex = sex, Smoker = smoker == 1, Diabetes = diabetes == 1,
      COPD = copd == 1, Diagnosis_gt_18mo = heart_failure_duration_gt_18mo == 1,
      Taking_BetaBlocker = beta_blocker_use == 1, Taking_ACEI_ARB = acei_arb_use == 1
    )
  ))
}

maintenance_fluids_dosing <- function(weight_kg) {

  weight_kg <- as.numeric(weight_kg)

  if (is.na(weight_kg) || weight_kg <= 0) {
    stop("Error: Weight must be a positive number.")
  }

  hourly_rate <- 0

  if (weight_kg <= 10) {
    hourly_rate <- 4 * weight_kg
  } else if (weight_kg <= 20) {
    hourly_rate <- 40 + (2 * (weight_kg - 10))
  } else {
    hourly_rate <- 60 + (1 * (weight_kg - 20))
  }

  daily_volume <- hourly_rate * 24

  return(list(
    Hourly_Rate_mL_hr = hourly_rate,
    Daily_Volume_mL_day = daily_volume,
    Method = "4-2-1 Rule (Holliday-Segar)",
    Inputs = list(
      Weight_kg = weight_kg
    )
  ))
}

major_depression_index <- function(low_spirits, loss_of_interest, lacking_energy, self_confidence, guilt, suicidal_ideation, concentration, restlessness, slowed_down, sleep_trouble, reduced_appetite, increased_appetite) {

  low_spirits <- as.numeric(low_spirits)
  loss_of_interest <- as.numeric(loss_of_interest)
  lacking_energy <- as.numeric(lacking_energy)
  self_confidence <- as.numeric(self_confidence)
  guilt <- as.numeric(guilt)
  suicidal_ideation <- as.numeric(suicidal_ideation)
  concentration <- as.numeric(concentration)
  restlessness <- as.numeric(restlessness)
  slowed_down <- as.numeric(slowed_down)
  sleep_trouble <- as.numeric(sleep_trouble)
  reduced_appetite <- as.numeric(reduced_appetite)
  increased_appetite <- as.numeric(increased_appetite)

  if (any(is.na(c(low_spirits, loss_of_interest, lacking_energy, self_confidence, guilt, suicidal_ideation, concentration, restlessness, slowed_down, sleep_trouble, reduced_appetite, increased_appetite)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(low_spirits, loss_of_interest, lacking_energy, self_confidence, guilt, suicidal_ideation, concentration, restlessness, slowed_down, sleep_trouble, reduced_appetite, increased_appetite) < 0) ||
      any(c(low_spirits, loss_of_interest, lacking_energy, self_confidence, guilt, suicidal_ideation, concentration, restlessness, slowed_down, sleep_trouble, reduced_appetite, increased_appetite) > 5)) {
    stop("Error: All item scores must be integers between 0 (At no time) and 5 (All the time).")
  }

  item8_score <- max(restlessness, slowed_down)
  item10_score <- max(reduced_appetite, increased_appetite)

  total_score <- low_spirits + loss_of_interest + lacking_energy + self_confidence + guilt + suicidal_ideation + concentration + item8_score + sleep_trouble + item10_score

  severity <- ""
  if (total_score < 20) {
    severity <- "No Depression / Remission"
  } else if (total_score <= 24) {
    severity <- "Mild Depression"
  } else if (total_score <= 29) {
    severity <- "Moderate Depression"
  } else {
    severity <- "Severe Depression"
  }

  return(list(
    MDI_Score = total_score,
    Severity_Category = severity,
    Inputs = list(
      Q1_Low_Spirits = low_spirits,
      Q2_Interest = loss_of_interest,
      Q3_Energy = lacking_energy,
      Q4_Confidence = self_confidence,
      Q5_Guilt = guilt,
      Q6_Suicide = suicidal_ideation,
      Q7_Concentration = concentration,
      Q8_Psychomotor = item8_score,
      Q9_Sleep = sleep_trouble,
      Q10_Appetite = item10_score
    )
  ))
}

must_screening_tool <- function(bmi, weight_loss_percent, acute_disease_effect) {

  bmi <- as.numeric(bmi)
  weight_loss_percent <- as.numeric(weight_loss_percent)
  acute_disease_effect <- as.numeric(acute_disease_effect)

  if (any(is.na(c(bmi, weight_loss_percent, acute_disease_effect)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (bmi > 20.0) {
    score <- score + 0
  } else if (bmi >= 18.5 && bmi <= 20.0) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (weight_loss_percent < 5) {
    score <- score + 0
  } else if (weight_loss_percent >= 5 && weight_loss_percent <= 10) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (acute_disease_effect == 1) {
    score <- score + 2
  }

  risk_category <- ""
  management <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    management <- "Routine clinical care. Repeat screening (e.g., weekly in hospital, monthly in care homes)."
  } else if (score == 1) {
    risk_category <- "Medium Risk"
    management <- "Observe. Document dietary intake for 3 days. If improving/adequate, little concern. If inadequate, clinical concern."
  } else {
    risk_category <- "High Risk"
    management <- "Treat. Refer to dietitian, nutritional support team, or implement local policy."
  }

  return(list(
    MUST_Score = score,
    Risk_Category = risk_category,
    Management_Guideline = management,
    Inputs = list(
      BMI = bmi,
      Weight_Loss_Pct = weight_loss_percent,
      Acute_Disease = acute_disease_effect == 1
    )
  ))
}

malt_ipi_score <- function(age, ann_arbor_stage, ldh_elevated) {

  age <- as.numeric(age)
  ann_arbor_stage <- as.numeric(ann_arbor_stage)
  ldh_elevated <- as.numeric(ldh_elevated)

  if (any(is.na(c(age, ann_arbor_stage, ldh_elevated)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age >= 70) {
    score <- score + 1
  }

  if (ann_arbor_stage >= 3) {
    score <- score + 1
  }

  if (ldh_elevated == 1) {
    score <- score + 1
  }

  risk_group <- ""
  efs_5yr <- ""

  if (score == 0) {
    risk_group <- "Low Risk"
    efs_5yr <- "70%"
  } else if (score == 1) {
    risk_group <- "Intermediate Risk"
    efs_5yr <- "56%"
  } else {
    risk_group <- "High Risk"
    efs_5yr <- "29%"
  }

  return(list(
    MALT_IPI_Score = score,
    Risk_Group = risk_group,
    Est_5_Year_Event_Free_Survival = efs_5yr,
    Inputs = list(
      Age_ge_70 = age >= 70,
      Stage_III_IV = ann_arbor_stage >= 3,
      LDH_Elevated = ldh_elevated == 1
    )
  ))
}

manchester_sclc_score <- function(ldh_level, ldh_uln, sodium, alk_phos_level, alk_phos_uln, kps, bicarbonate, stage_extensive) {

  ldh_level <- as.numeric(ldh_level)
  ldh_uln <- as.numeric(ldh_uln)
  sodium <- as.numeric(sodium)
  alk_phos_level <- as.numeric(alk_phos_level)
  alk_phos_uln <- as.numeric(alk_phos_uln)
  kps <- as.numeric(kps)
  bicarbonate <- as.numeric(bicarbonate)
  stage_extensive <- as.numeric(stage_extensive)

  if (any(is.na(c(ldh_level, ldh_uln, sodium, alk_phos_level, alk_phos_uln, kps, bicarbonate, stage_extensive)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (ldh_uln <= 0 || alk_phos_uln <= 0) {
    stop("Error: Upper limits of normal (ULN) must be positive.")
  }

  score <- 0

  if (ldh_level >= ldh_uln) {
    score <- score + 1
  }

  if (sodium < 132) {
    score <- score + 1
  }

  if (alk_phos_level >= (1.5 * alk_phos_uln)) {
    score <- score + 1
  }

  if (kps < 60) {
    score <- score + 1
  }

  if (bicarbonate < 24) {
    score <- score + 1
  }

  if (stage_extensive == 1) {
    score <- score + 1
  }

  prognosis <- ""
  survival_2yr <- ""
  median_survival <- ""

  if (score <= 1) {
    prognosis <- "Good Prognosis"
    median_survival <- "16.3 months"
    survival_2yr <- "28%"
  } else if (score <= 3) {
    prognosis <- "Intermediate Prognosis"
    median_survival <- "10.7 months"
    survival_2yr <- "9%"
  } else {
    prognosis <- "Poor Prognosis"
    median_survival <- "6.5 months"
    survival_2yr <- "0%"
  }

  return(list(
    Manchester_Score = score,
    Prognosis_Group = prognosis,
    Median_Survival = median_survival,
    Two_Year_Survival = survival_2yr,
    Inputs = list(
      LDH_Elevated = ldh_level >= ldh_uln,
      Na_lt_132 = sodium < 132,
      ALP_Elevated_1_5x = alk_phos_level >= (1.5 * alk_phos_uln),
      KPS_lt_60 = kps < 60,
      HCO3_lt_24 = bicarbonate < 24,
      Extensive_Stage = stage_extensive == 1
    )
  ))
}

mess_score <- function(skeletal_soft_tissue, limb_ischemia, shock, age, ischemia_duration_gt_6h = 0) {

  skeletal_soft_tissue <- as.numeric(skeletal_soft_tissue)
  limb_ischemia <- as.numeric(limb_ischemia)
  shock <- as.numeric(shock)
  age <- as.numeric(age)
  ischemia_duration_gt_6h <- as.numeric(ischemia_duration_gt_6h)

  if (any(is.na(c(skeletal_soft_tissue, limb_ischemia, shock, age, ischemia_duration_gt_6h)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (skeletal_soft_tissue < 1 || skeletal_soft_tissue > 4) {
    stop("Error: Skeletal/Soft Tissue score must be between 1 and 4.")
  }

  if (limb_ischemia < 0 || limb_ischemia > 3) {
    stop("Error: Limb Ischemia score must be between 0 and 3.")
  }

  if (shock < 0 || shock > 2) {
    stop("Error: Shock score must be between 0 and 2.")
  }

  if (age < 0 || age > 2) {
    stop("Error: Age score must be 0, 1, or 2.")
  }

  ischemia_points <- limb_ischemia
  if (ischemia_duration_gt_6h == 1) {
    ischemia_points <- ischemia_points * 2
  }

  total_score <- skeletal_soft_tissue + ischemia_points + shock + age

  risk_category <- ""
  if (total_score >= 7) {
    risk_category <- "High Risk: Amputation likely. Score >= 7 predicts amputation with high specificity."
  } else {
    risk_category <- "Low Risk: Limb salvage likely viable."
  }

  return(list(
    MESS_Score = total_score,
    Interpretation = risk_category,
    Inputs = list(
      Injury_Energy = skeletal_soft_tissue,
      Ischemia_Points = ischemia_points,
      Shock_Points = shock,
      Age_Points = age
    )
  ))
}

manning_ibs_criteria <- function(pain_relieved_by_defecation, incomplete_bowel_movement, mucus_in_stool, change_in_stool_consistency, more_frequent_stool_with_pain, abdominal_distension) {

  pain_relieved_by_defecation <- as.numeric(pain_relieved_by_defecation)
  incomplete_bowel_movement <- as.numeric(incomplete_bowel_movement)
  mucus_in_stool <- as.numeric(mucus_in_stool)
  change_in_stool_consistency <- as.numeric(change_in_stool_consistency)
  more_frequent_stool_with_pain <- as.numeric(more_frequent_stool_with_pain)
  abdominal_distension <- as.numeric(abdominal_distension)

  if (any(is.na(c(pain_relieved_by_defecation, incomplete_bowel_movement, mucus_in_stool, change_in_stool_consistency, more_frequent_stool_with_pain, abdominal_distension)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- pain_relieved_by_defecation + incomplete_bowel_movement + mucus_in_stool + change_in_stool_consistency + more_frequent_stool_with_pain + abdominal_distension

  likelihood <- ""
  if (score <= 1) {
    likelihood <- "IBS Less Likely"
  } else if (score == 2) {
    likelihood <- "Possible IBS (Sensitivity 58%, Specificity 74%)"
  } else if (score == 3) {
    likelihood <- "Probable IBS (Sensitivity 42%, Specificity 85%)"
  } else {
    likelihood <- "Highly Probable IBS (Specificity > 90%)"
  }

  return(list(
    Manning_Score = score,
    Interpretation = likelihood,
    Inputs = list(
      Pain_Relief = pain_relieved_by_defecation == 1,
      Incomplete_Evacuation = incomplete_bowel_movement == 1,
      Mucus = mucus_in_stool == 1,
      Consistency_Change = change_in_stool_consistency == 1,
      Frequency_Change = more_frequent_stool_with_pain == 1,
      Distension = abdominal_distension == 1
    )
  ))
}

mipi_lymphoma_score <- function(age, ecog_score, ldh_level, ldh_uln, wbc_count) {

  age <- as.numeric(age)
  ecog_score <- as.numeric(ecog_score)
  ldh_level <- as.numeric(ldh_level)
  ldh_uln <- as.numeric(ldh_uln)
  wbc_count <- as.numeric(wbc_count)

  if (any(is.na(c(age, ecog_score, ldh_level, ldh_uln, wbc_count)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (ldh_uln <= 0) stop("Error: LDH ULN must be positive.")
  if (ldh_level <= 0) stop("Error: LDH level must be positive.")
  if (wbc_count <= 0) stop("Error: WBC count must be positive.")

  ecog_binary <- if (ecog_score >= 2) 1 else 0

  ldh_ratio <- ldh_level / ldh_uln

  score <- (0.03535 * age) + (0.6978 * ecog_binary) + (1.367 * log10(ldh_ratio)) + (0.9393 * log10(wbc_count))

  risk_group <- ""
  median_survival <- ""

  if (score < 5.7) {
    risk_group <- "Low Risk"
    median_survival <- "Not Reached (5-year OS 60%)"
  } else if (score >= 5.7 && score < 6.2) {
    risk_group <- "Intermediate Risk"
    median_survival <- "51 months"
  } else {
    risk_group <- "High Risk"
    median_survival <- "29 months"
  }

  return(list(
    MIPI_Score = round(score, 2),
    Risk_Group = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Age = age,
      ECOG_Binary = ecog_binary,
      LDH_Ratio = round(ldh_ratio, 2),
      WBC = wbc_count
    )
  ))
}

marburg_heart_score <- function(age, sex, history_vascular_disease, patient_suspects_cardiac, pain_worsens_exercise, pain_not_reproducible) {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  history_vascular_disease <- as.numeric(history_vascular_disease)
  patient_suspects_cardiac <- as.numeric(patient_suspects_cardiac)
  pain_worsens_exercise <- as.numeric(pain_worsens_exercise)
  pain_not_reproducible <- as.numeric(pain_not_reproducible)

  if (any(is.na(c(age, history_vascular_disease, patient_suspects_cardiac, pain_worsens_exercise, pain_not_reproducible)))) {
    stop("Error: All inputs must be numeric (0 or 1, except Age).")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  score <- 0

  if (sex == "female") {
    if (age >= 65) {
      score <- score + 1
    }
  } else {
    if (age >= 55) {
      score <- score + 1
    }
  }

  if (history_vascular_disease == 1) {
    score <- score + 1
  }

  if (patient_suspects_cardiac == 1) {
    score <- score + 1
  }

  if (pain_worsens_exercise == 1) {
    score <- score + 1
  }

  if (pain_not_reproducible == 1) {
    score <- score + 1
  }

  risk_category <- ""
  cad_probability <- ""

  if (score <= 2) {
    risk_category <- "Low Risk"
    cad_probability <- "< 2.5%"
  } else if (score == 3) {
    risk_category <- "Intermediate Risk"
    cad_probability <- "~17%"
  } else {
    risk_category <- "High Risk"
    cad_probability <- "~50%"
  }

  return(list(
    MHS_Score = score,
    Risk_Category = risk_category,
    CAD_Probability = cad_probability,
    Inputs = list(
      Age = age,
      Sex = sex,
      Hx_Vascular_Disease = history_vascular_disease == 1,
      Patient_Suspicion = patient_suspects_cardiac == 1,
      Exertional_Pain = pain_worsens_exercise == 1,
      Pain_Not_Reproducible = pain_not_reproducible == 1
    )
  ))
}

mascc_risk_index <- function(burden_severity, hypotension_present, copd_history, tumor_type, previous_fungal_infection, dehydration_present, outpatient_status, age) {

  burden_severity <- tolower(as.character(burden_severity))
  hypotension_present <- as.numeric(hypotension_present)
  copd_history <- as.numeric(copd_history)
  tumor_type <- tolower(as.character(tumor_type))
  previous_fungal_infection <- as.numeric(previous_fungal_infection)
  dehydration_present <- as.numeric(dehydration_present)
  outpatient_status <- as.numeric(outpatient_status)
  age <- as.numeric(age)

  if (any(is.na(c(hypotension_present, copd_history, previous_fungal_infection, dehydration_present, outpatient_status, age)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!burden_severity %in% c("none_mild", "moderate", "severe")) {
    stop("Error: Burden severity must be 'none_mild', 'moderate', or 'severe'.")
  }

  if (!tumor_type %in% c("solid", "hematologic")) {
    stop("Error: Tumor type must be 'solid' or 'hematologic'.")
  }

  score <- 0

  if (burden_severity == "none_mild") {
    score <- score + 5
  } else if (burden_severity == "moderate") {
    score <- score + 3
  }

  if (hypotension_present == 0) {
    score <- score + 5
  }

  if (copd_history == 0) {
    score <- score + 4
  }

  if (tumor_type == "solid") {
    score <- score + 4
  } else {
    if (previous_fungal_infection == 0) {
      score <- score + 4
    }
  }

  if (dehydration_present == 0) {
    score <- score + 3
  }

  if (outpatient_status == 1) {
    score <- score + 3
  }

  if (age < 60) {
    score <- score + 2
  }

  risk_category <- ""
  recommendation <- ""

  if (score >= 21) {
    risk_category <- "Low Risk"
    recommendation <- "Patient may be eligible for outpatient management and/or oral antibiotics."
  } else {
    risk_category <- "High Risk"
    recommendation <- "Inpatient admission and IV antibiotics generally required."
  }

  return(list(
    MASCC_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Burden = burden_severity,
      Hypotension = hypotension_present == 1,
      COPD = copd_history == 1,
      Tumor = tumor_type,
      Prev_Fungal = previous_fungal_infection == 1,
      Dehydration = dehydration_present == 1,
      Outpatient = outpatient_status == 1,
      Age = age
    )
  ))
}

rhd_immune_globulin_dosage <- function(kbs_fetal_cells_per_5000, maternal_blood_volume_ml = 5000) {

  kbs_fetal_cells_per_5000 <- as.numeric(kbs_fetal_cells_per_5000)
  maternal_blood_volume_ml <- as.numeric(maternal_blood_volume_ml)

  if (any(is.na(c(kbs_fetal_cells_per_5000, maternal_blood_volume_ml)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (maternal_blood_volume_ml <= 0) {
    stop("Error: Maternal blood volume must be positive.")
  }

  fetal_blood_volume_ml <- (kbs_fetal_cells_per_5000 / 5000) * maternal_blood_volume_ml

  rhogam_vials_raw <- fetal_blood_volume_ml / 30

  rhogam_vials_needed <- ceiling(rhogam_vials_raw) + 1

  return(list(
    Fetal_Blood_Volume_mL = round(fetal_blood_volume_ml, 1),
    RhoGAM_Vials_Calculated = rhogam_vials_needed,
    Inputs = list(
      KBS_Count = kbs_fetal_cells_per_5000,
      Maternal_Volume_mL = maternal_blood_volume_ml
    )
  ))
}

maximum_allowable_blood_loss <- function(weight_kg, initial_hct, target_hct, patient_type) {

  weight_kg <- as.numeric(weight_kg)
  initial_hct <- as.numeric(initial_hct)
  target_hct <- as.numeric(target_hct)
  patient_type <- tolower(as.character(patient_type))

  if (any(is.na(c(weight_kg, initial_hct, target_hct)))) {
    stop("Error: Weight and Hematocrit values must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  if (target_hct >= initial_hct) {
    return(list(
      Allowable_Blood_Loss_mL = 0,
      Message = "Target Hct is higher than or equal to Initial Hct. No blood loss is allowable to maintain target."
    ))
  }

  ebv_factor <- 0
  if (patient_type == "premature") {
    ebv_factor <- 95
  } else if (patient_type == "neonate") {
    ebv_factor <- 85
  } else if (patient_type == "infant") {
    ebv_factor <- 80
  } else if (patient_type == "child") {
    ebv_factor <- 75
  } else if (patient_type == "adult_male") {
    ebv_factor <- 75
  } else if (patient_type == "adult_female") {
    ebv_factor <- 65
  } else {
    stop("Error: Patient type must be 'premature', 'neonate', 'infant', 'child', 'adult_male', or 'adult_female'.")
  }

  estimated_blood_volume <- weight_kg * ebv_factor

  hct_average <- (initial_hct + target_hct) / 2

  abl <- estimated_blood_volume * (initial_hct - target_hct) / hct_average

  return(list(
    Allowable_Blood_Loss_mL = round(abl, 0),
    Estimated_Blood_Volume_mL = round(estimated_blood_volume, 0),
    Inputs = list(
      Weight_kg = weight_kg,
      Initial_Hct = initial_hct,
      Target_Hct = target_hct,
      Patient_Type = patient_type,
      EBV_Factor_Used = ebv_factor
    )
  ))
}

maps_systemic_mastocytosis_score <- function(sm_category, age, platelet_count, alp_elevated, adverse_mutations) {

  sm_category <- tolower(as.character(sm_category))
  age <- as.numeric(age)
  platelet_count <- as.numeric(platelet_count)
  alp_elevated <- as.numeric(alp_elevated)
  adverse_mutations <- as.numeric(adverse_mutations)

  if (any(is.na(c(age, platelet_count, alp_elevated, adverse_mutations)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sm_category %in% c("indolent", "advanced")) {
    stop("Error: SM Category must be 'indolent' (Indolent/Smoldering) or 'advanced' (Aggressive/MCL/SM-AHN).")
  }

  score <- 0

  if (sm_category == "advanced") {
    score <- score + 2
  }

  if (age > 60) {
    score <- score + 1
  }

  if (platelet_count < 150) {
    score <- score + 1
  }

  if (alp_elevated == 1) {
    score <- score + 1
  }

  if (adverse_mutations == 1) {
    score <- score + 1
  }

  risk_group <- ""
  if (score == 0) {
    risk_group <- "Low Risk"
  } else if (score <= 2) {
    risk_group <- "Intermediate Risk"
  } else {
    risk_group <- "High Risk"
  }

  return(list(
    MAPS_Score = score,
    Risk_Category = risk_group,
    Inputs = list(
      SM_Type = sm_category,
      Age_gt_60 = age > 60,
      Plt_lt_150 = platelet_count < 150,
      ALP_Elevated = alp_elevated == 1,
      Mutations = adverse_mutations == 1
    )
  ))
}

mayo_uc_score <- function(stool_frequency, rectal_bleeding, endoscopic_findings, physician_global_assessment) {

  inputs <- c(stool_frequency, rectal_bleeding, endoscopic_findings, physician_global_assessment)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 3)) {
    stop("Error: All subscores must be integers between 0 and 3.")
  }

  total_score <- sum(inputs)

  severity <- ""
  if (total_score <= 2) {
    severity <- "Remission or Inactive Disease"
  } else if (total_score <= 5) {
    severity <- "Mild Activity"
  } else if (total_score <= 10) {
    severity <- "Moderate Activity"
  } else {
    severity <- "Severe Activity"
  }

  return(list(
    Mayo_Score = total_score,
    Disease_Activity = severity,
    Inputs = list(
      Stool_Frequency = stool_frequency,
      Rectal_Bleeding = rectal_bleeding,
      Endoscopy = endoscopic_findings,
      Physician_Rating = physician_global_assessment
    )
  ))
}

mcdonald_ms_criteria <- function(clinical_attacks, clinical_lesions, dis_demonstrated, dit_demonstrated, csf_oligoclonal_bands) {

  clinical_attacks <- as.numeric(clinical_attacks)
  clinical_lesions <- as.numeric(clinical_lesions)
  dis_demonstrated <- as.numeric(dis_demonstrated)
  dit_demonstrated <- as.numeric(dit_demonstrated)
  csf_oligoclonal_bands <- as.numeric(csf_oligoclonal_bands)

  if (any(is.na(c(clinical_attacks, clinical_lesions, dis_demonstrated, dit_demonstrated, csf_oligoclonal_bands)))) {
    stop("Error: All inputs must be numeric (0 or 1 for booleans, integers for counts).")
  }

  diagnosis <- "Possible MS / Clinically Isolated Syndrome (CIS)"
  missing_criteria <- c()

  if (clinical_attacks >= 2) {
    if (clinical_lesions >= 2) {
      diagnosis <- "Multiple Sclerosis (MS)"
    } else {
      if (dis_demonstrated == 1) {
        diagnosis <- "Multiple Sclerosis (MS)"
      } else {
        missing_criteria <- c(missing_criteria, "Dissemination in Space (DIS)")
      }
    }
  } else if (clinical_attacks == 1) {
    if (clinical_lesions >= 2) {
      if (dit_demonstrated == 1 || csf_oligoclonal_bands == 1) {
        diagnosis <- "Multiple Sclerosis (MS)"
      } else {
        missing_criteria <- c(missing_criteria, "Dissemination in Time (DIT) OR CSF Oligoclonal Bands")
      }
    } else {
      req_dis <- FALSE
      req_dit_or_csf <- FALSE

      if (dis_demonstrated == 1) {
        req_dis <- TRUE
      } else {
        missing_criteria <- c(missing_criteria, "Dissemination in Space (DIS)")
      }

      if (dit_demonstrated == 1 || csf_oligoclonal_bands == 1) {
        req_dit_or_csf <- TRUE
      } else {
        missing_criteria <- c(missing_criteria, "Dissemination in Time (DIT) OR CSF Oligoclonal Bands")
      }

      if (req_dis && req_dit_or_csf) {
        diagnosis <- "Multiple Sclerosis (MS)"
      }
    }
  } else {
    diagnosis <- "No Clinical Attacks (Criteria Not Applicable)"
  }

  return(list(
    Diagnosis = diagnosis,
    Missing_Requirements = if (length(missing_criteria) > 0) missing_criteria else "None",
    Inputs = list(
      Attacks = clinical_attacks,
      Lesions = clinical_lesions,
      DIS = dis_demonstrated == 1,
      DIT = dit_demonstrated == 1,
      CSF_Bands = csf_oligoclonal_bands == 1
    )
  ))
}

mcmahon_rhabdo_score <- function(age, sex, etiology_high_risk, cpk, bicarbonate, creatinine, calcium, phosphate, creatinine_units = "mg/dL", calcium_units = "mg/dL", phosphate_units = "mg/dL") {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  etiology_high_risk <- as.numeric(etiology_high_risk)
  cpk <- as.numeric(cpk)
  bicarbonate <- as.numeric(bicarbonate)
  creatinine <- as.numeric(creatinine)
  calcium <- as.numeric(calcium)
  phosphate <- as.numeric(phosphate)

  if (any(is.na(c(age, etiology_high_risk, cpk, bicarbonate, creatinine, calcium, phosphate)))) {
    stop("Error: All clinical inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  creat_mg_dl <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    creat_mg_dl <- creatinine / 88.4
  }

  calc_mg_dl <- calcium
  if (calcium_units == "mmol/L") {
    calc_mg_dl <- calcium / 0.25
  }

  phos_mg_dl <- phosphate
  if (phosphate_units == "mmol/L") {
    phos_mg_dl <- phosphate / 0.3229
  }

  score <- 0

  if (age < 50) {
    score <- score + 0
  } else if (age >= 50 && age <= 70) {
    score <- score + 1.5
  } else if (age > 70 && age <= 80) {
    score <- score + 3
  } else {
    score <- score + 4.5
  }

  if (sex == "female") {
    score <- score + 1
  }

  if (etiology_high_risk == 1) {
    score <- score + 3
  }

  if (cpk > 40000) {
    score <- score + 2
  }

  if (bicarbonate < 19) {
    score <- score + 2
  }

  if (creat_mg_dl < 1.4) {
    score <- score + 0
  } else if (creat_mg_dl >= 1.4 && creat_mg_dl <= 2.2) {
    score <- score + 1.5
  } else if (creat_mg_dl > 2.2 && creat_mg_dl <= 4.5) {
    score <- score + 3
  } else {
    score <- score + 4.5
  }

  if (calc_mg_dl < 7.5) {
    score <- score + 2
  }

  if (phos_mg_dl < 4.0) {
    score <- score + 0
  } else if (phos_mg_dl >= 4.0 && phos_mg_dl <= 5.4) {
    score <- score + 1.5
  } else {
    score <- score + 3
  }

  risk_category <- ""
  probability <- ""

  if (score < 5) {
    risk_category <- "Low Risk"
    probability <- "~2.3%"
  } else {
    risk_category <- "High Risk (Protective measures indicated)"
    probability <- "~52%"
  }

  return(list(
    McMahon_Score = score,
    Risk_Category = risk_category,
    Risk_Renal_Failure_or_Death = probability,
    Inputs = list(
      Age = age,
      Sex = sex,
      High_Risk_Etiology = etiology_high_risk == 1,
      CPK = cpk,
      Bicarbonate = bicarbonate,
      Creatinine_mg_dL = round(creat_mg_dl, 2),
      Calcium_mg_dL = round(calc_mg_dl, 2),
      Phosphate_mg_dL = round(phos_mg_dl, 2)
    )
  ))
}

mdrd_gfr_equation <- function(creatinine, age, sex, race, creatinine_units = "mg/dL") {

  creatinine <- as.numeric(creatinine)
  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  race <- tolower(as.character(race))

  if (any(is.na(c(creatinine, age)))) {
    stop("Error: Creatinine and Age must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (!race %in% c("black", "african_american", "other", "non_black")) {
    stop("Error: Race must be 'black'/'african_american' or 'other'/'non_black'.")
  }

  scr_mg_dl <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    scr_mg_dl <- creatinine / 88.4
  }

  gfr <- 175 * (scr_mg_dl ^ -1.154) * (age ^ -0.203)

  if (sex == "female") {
    gfr <- gfr * 0.742
  }

  if (race == "black" || race == "african_american") {
    gfr <- gfr * 1.212
  }

  interpretation <- ""
  if (gfr >= 90) {
    interpretation <- "Stage 1 (Normal or High GFR)"
  } else if (gfr >= 60) {
    interpretation <- "Stage 2 (Mildly Decreased)"
  } else if (gfr >= 45) {
    interpretation <- "Stage 3a (Mild-Moderately Decreased)"
  } else if (gfr >= 30) {
    interpretation <- "Stage 3b (Moderate-Severely Decreased)"
  } else if (gfr >= 15) {
    interpretation <- "Stage 4 (Severely Decreased)"
  } else {
    interpretation <- "Stage 5 (Kidney Failure)"
  }

  return(list(
    MDRD_GFR = round(gfr, 1),
    CKD_Stage = interpretation,
    Inputs = list(
      Creatinine_mg_dL = round(scr_mg_dl, 2),
      Age = age,
      Sex = sex,
      Race = race
    )
  ))
}

calculate_map <- function(systolic_bp, diastolic_bp) {

  systolic_bp <- as.numeric(systolic_bp)
  diastolic_bp <- as.numeric(diastolic_bp)

  if (any(is.na(c(systolic_bp, diastolic_bp)))) {
    stop("Error: Blood pressure values must be numeric.")
  }

  map <- (systolic_bp + (2 * diastolic_bp)) / 3

  return(list(
    Mean_Arterial_Pressure = round(map, 1),
    Inputs = list(
      SBP = systolic_bp,
      DBP = diastolic_bp
    )
  ))
}

mrc_icu_score <- function(aminoglycosides = 0, amphotericin_b = 0, antiarrhythmics = 0, anticoagulants_oral = 0, anticonvulsants = 0, argatroban = 0, azole_antifungals = 0, blood_products = 0, chemotherapy = 0, clozapine = 0, digoxin = 0, ganciclovir_valganciclovir = 0, hyperosmolar_fluids = 0, immunosuppressants = 0, lidocaine_infusion = 0, lithium = 0, prostacyclins = 0, theophylline = 0, therapeutic_heparins = 0, vancomycin_iv = 0, warfarin = 0, thromboembolic_prophylaxis = 0, stress_ulcer_prophylaxis = 0, glycemic_control = 0, bowel_regimen = 0, chlorhexidine = 0, inhalers_nebulizers = 0, continuous_sedation = 0, continuous_vasopressors = 0, continuous_inotropes = 0, continuous_nmba = 0, continuous_iv_fluids = 0, parenteral_nutrition = 0, fluid_bolus = 0, renal_replacement_therapy = 0, mechanical_ventilation = 0) {

  score <- 0

  score <- score + (3 * aminoglycosides)
  score <- score + (1 * amphotericin_b)
  score <- score + (1 * antiarrhythmics)
  score <- score + (1 * anticoagulants_oral)
  score <- score + (3 * anticonvulsants)
  score <- score + (2 * argatroban)
  score <- score + (2 * azole_antifungals)
  score <- score + (2 * blood_products)
  score <- score + (3 * chemotherapy)
  score <- score + (3 * clozapine)
  score <- score + (3 * digoxin)
  score <- score + (1 * ganciclovir_valganciclovir)
  score <- score + (1 * hyperosmolar_fluids)
  score <- score + (3 * immunosuppressants)
  score <- score + (2 * lidocaine_infusion)
  score <- score + (3 * lithium)
  score <- score + (2 * prostacyclins)
  score <- score + (3 * theophylline)
  score <- score + (2 * therapeutic_heparins)
  score <- score + (3 * vancomycin_iv)
  score <- score + (3 * warfarin)

  score <- score + (1 * thromboembolic_prophylaxis)
  score <- score + (1 * stress_ulcer_prophylaxis)
  score <- score + (1 * glycemic_control)
  score <- score + (1 * bowel_regimen)
  score <- score + (1 * chlorhexidine)
  score <- score + (1 * inhalers_nebulizers)
  score <- score + (1 * continuous_iv_fluids)

  score <- score + (3 * continuous_vasopressors)
  score <- score + (3 * continuous_inotropes)
  score <- score + (3 * continuous_nmba)
  score <- score + (2 * continuous_sedation)

  score <- score + (3 * parenteral_nutrition)
  score <- score + (1 * fluid_bolus)
  score <- score + (3 * renal_replacement_therapy)
  score <- score + (3 * mechanical_ventilation)

  return(list(
    MRC_ICU_Score = score,
    Interpretation = "Higher scores correlate with increased ICU mortality, length of stay, and pharmacist workload."
  ))
}

mehran_score_cin <- function(hypotension, intraaortic_balloon_pump, congestive_heart_failure, age_gt_75, diabetes, anemia, contrast_volume_cc, serum_creatinine_mg_dl = NULL, egfr_ml_min = NULL) {

  hypotension <- as.numeric(hypotension)
  intraaortic_balloon_pump <- as.numeric(intraaortic_balloon_pump)
  congestive_heart_failure <- as.numeric(congestive_heart_failure)
  age_gt_75 <- as.numeric(age_gt_75)
  diabetes <- as.numeric(diabetes)
  anemia <- as.numeric(anemia)
  contrast_volume_cc <- as.numeric(contrast_volume_cc)

  if (any(is.na(c(hypotension, intraaortic_balloon_pump, congestive_heart_failure, age_gt_75, diabetes, anemia, contrast_volume_cc)))) {
    stop("Error: All non-renal inputs must be numeric.")
  }

  if (is.null(serum_creatinine_mg_dl) && is.null(egfr_ml_min)) {
    stop("Error: Either Serum Creatinine or eGFR must be provided.")
  }

  score <- 0

  if (hypotension == 1) score <- score + 5
  if (intraaortic_balloon_pump == 1) score <- score + 5
  if (congestive_heart_failure == 1) score <- score + 5
  if (age_gt_75 == 1) score <- score + 4
  if (diabetes == 1) score <- score + 3
  if (anemia == 1) score <- score + 3

  contrast_points <- floor(contrast_volume_cc / 100)
  score <- score + contrast_points

  if (!is.null(serum_creatinine_mg_dl)) {
    scr <- as.numeric(serum_creatinine_mg_dl)
    if (scr > 1.5) score <- score + 4
  } else if (!is.null(egfr_ml_min)) {
    egfr <- as.numeric(egfr_ml_min)
    if (egfr < 20) score <- score + 6
    else if (egfr >= 20 && egfr < 40) score <- score + 4
    else if (egfr >= 40 && egfr < 60) score <- score + 2
    else score <- score + 0
  }

  risk_class <- ""
  cin_risk <- ""
  dialysis_risk <- ""

  if (score <= 5) {
    risk_class <- "Low Risk"
    cin_risk <- "7.5%"
    dialysis_risk <- "0.04%"
  } else if (score <= 10) {
    risk_class <- "Moderate Risk"
    cin_risk <- "14.0%"
    dialysis_risk <- "0.12%"
  } else if (score <= 15) {
    risk_class <- "High Risk"
    cin_risk <- "26.1%"
    dialysis_risk <- "1.09%"
  } else {
    risk_class <- "Very High Risk"
    cin_risk <- "57.3%"
    dialysis_risk <- "12.6%"
  }

  return(list(
    Mehran_Score = score,
    Risk_Class = risk_class,
    Risk_CIN = cin_risk,
    Risk_Dialysis = dialysis_risk,
    Inputs = list(
      Hypotension = hypotension == 1,
      IABP = intraaortic_balloon_pump == 1,
      CHF = congestive_heart_failure == 1,
      Age_gt_75 = age_gt_75 == 1,
      Diabetes = diabetes == 1,
      Anemia = anemia == 1,
      Contrast_Vol = contrast_volume_cc,
      Renal_Input = if(!is.null(serum_creatinine_mg_dl)) paste("SCr:", serum_creatinine_mg_dl) else paste("eGFR:", egfr_ml_min)
    )
  ))
}

mekhail_motzer_score <- function(time_to_treatment_lt_1y, kps_lt_80, hemoglobin_low, calcium_high, ldh_high, metastasis_sites_ge_2) {

  time_to_treatment_lt_1y <- as.numeric(time_to_treatment_lt_1y)
  kps_lt_80 <- as.numeric(kps_lt_80)
  hemoglobin_low <- as.numeric(hemoglobin_low)
  calcium_high <- as.numeric(calcium_high)
  ldh_high <- as.numeric(ldh_high)
  metastasis_sites_ge_2 <- as.numeric(metastasis_sites_ge_2)

  if (any(is.na(c(time_to_treatment_lt_1y, kps_lt_80, hemoglobin_low, calcium_high, ldh_high, metastasis_sites_ge_2)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (time_to_treatment_lt_1y == 1) score <- score + 1
  if (kps_lt_80 == 1) score <- score + 1
  if (hemoglobin_low == 1) score <- score + 1
  if (calcium_high == 1) score <- score + 1
  if (ldh_high == 1) score <- score + 1
  if (metastasis_sites_ge_2 == 1) score <- score + 1

  risk_group <- ""
  median_survival <- ""

  if (score == 0) {
    risk_group <- "Favorable Risk"
    median_survival <- "24 months"
  } else if (score >= 1 && score <= 2) {
    risk_group <- "Intermediate Risk"
    median_survival <- "12 months"
  } else {
    risk_group <- "Poor Risk"
    median_survival <- "5 months"
  }

  return(list(
    Mekhail_Score = score,
    Risk_Group = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Time_to_Tx_lt_1yr = time_to_treatment_lt_1y == 1,
      KPS_lt_80 = kps_lt_80 == 1,
      Low_Hemoglobin = hemoglobin_low == 1,
      High_Calcium = calcium_high == 1,
      High_LDH = ldh_high == 1,
      Metastasis_Sites_ge_2 = metastasis_sites_ge_2 == 1
    )
  ))
}

meld_score_original <- function(bilirubin, inr, creatinine, on_dialysis = 0, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  bilirubin <- as.numeric(bilirubin)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)
  on_dialysis <- as.numeric(on_dialysis)

  if (any(is.na(c(bilirubin, inr, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_val <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_val <- bilirubin / 17.1
  }

  creat_val <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    creat_val <- creatinine / 88.4
  }

  if (bili_val < 1.0) bili_val <- 1.0
  if (inr < 1.0) inr <- 1.0

  if (on_dialysis == 1 || creat_val > 4.0) {
    creat_val <- 4.0
  } else if (creat_val < 1.0) {
    creat_val <- 1.0
  }

  meld_score <- 0.957 * log(creat_val) + 0.378 * log(bili_val) + 1.120 * log(inr) + 0.643
  meld_score <- meld_score * 10
  meld_score <- round(meld_score, 0)

  if (meld_score > 40) meld_score <- 40

  mortality_3mo <- ""
  if (meld_score <= 9) {
    mortality_3mo <- "1.9%"
  } else if (meld_score <= 19) {
    mortality_3mo <- "6.0%"
  } else if (meld_score <= 29) {
    mortality_3mo <- "19.6%"
  } else if (meld_score <= 39) {
    mortality_3mo <- "52.6%"
  } else {
    mortality_3mo <- "71.3%"
  }

  return(list(
    MELD_Score = meld_score,
    Est_3_Month_Mortality = mortality_3mo,
    Inputs = list(
      Bilirubin_mg_dL = round(bili_val, 2),
      INR = inr,
      Creatinine_mg_dL = round(creat_val, 2),
      Dialysis = on_dialysis == 1
    )
  ))
}

meld_score_pre_2016 <- function(bilirubin, inr, creatinine, on_dialysis = 0, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  bilirubin <- as.numeric(bilirubin)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)
  on_dialysis <- as.numeric(on_dialysis)

  if (any(is.na(c(bilirubin, inr, creatinine)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_val <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_val <- bilirubin / 17.1
  }

  creat_val <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    creat_val <- creatinine / 88.4
  }

  if (bili_val < 1.0) bili_val <- 1.0
  if (inr < 1.0) inr <- 1.0

  if (on_dialysis == 1 || creat_val > 4.0) {
    creat_val <- 4.0
  } else if (creat_val < 1.0) {
    creat_val <- 1.0
  }

  meld_score <- 0.957 * log(creat_val) + 0.378 * log(bili_val) + 1.120 * log(inr) + 0.643
  meld_score <- round(meld_score * 10, 0)

  if (meld_score > 40) meld_score <- 40

  mortality_3mo <- ""
  if (meld_score <= 9) {
    mortality_3mo <- "1.9%"
  } else if (meld_score <= 19) {
    mortality_3mo <- "6.0%"
  } else if (meld_score <= 29) {
    mortality_3mo <- "19.6%"
  } else if (meld_score <= 39) {
    mortality_3mo <- "52.6%"
  } else {
    mortality_3mo <- "71.3%"
  }

  return(list(
    MELD_Score_Original = meld_score,
    Est_3_Month_Mortality = mortality_3mo,
    Inputs = list(
      Bilirubin_mg_dL = round(bili_val, 2),
      INR = inr,
      Creatinine_mg_dL = round(creat_val, 2),
      Dialysis = on_dialysis == 1
    )
  ))
}

meld_na_score_liver <- function(bilirubin, inr, creatinine, sodium, on_dialysis = 0, bilirubin_units = "mg/dL", creatinine_units = "mg/dL") {

  bilirubin <- as.numeric(bilirubin)
  inr <- as.numeric(inr)
  creatinine <- as.numeric(creatinine)
  sodium <- as.numeric(sodium)
  on_dialysis <- as.numeric(on_dialysis)

  if (any(is.na(c(bilirubin, inr, creatinine, sodium)))) {
    stop("Error: All inputs must be numeric.")
  }

  bili_val <- bilirubin
  if (bilirubin_units == "umol/L") {
    bili_val <- bilirubin / 17.1
  }

  creat_val <- creatinine
  if (creatinine_units == "umol/L" || creatinine_units == "micromol/L") {
    creat_val <- creatinine / 88.4
  }

  if (bili_val < 1.0) bili_val <- 1.0
  if (inr < 1.0) inr <- 1.0

  if (on_dialysis == 1 || creat_val > 4.0) {
    creat_val <- 4.0
  } else if (creat_val < 1.0) {
    creat_val <- 1.0
  }

  meld_i <- 0.957 * log(creat_val) + 0.378 * log(bili_val) + 1.120 * log(inr) + 0.643
  meld_initial <- meld_i * 10
  meld_initial <- round(meld_initial, 1)

  meld_final <- meld_initial

  if (meld_initial > 11) {
    na_val <- sodium
    if (na_val < 125) na_val <- 125
    if (na_val > 137) na_val <- 137

    meld_final <- meld_initial + 1.32 * (137 - na_val) - (0.033 * meld_initial * (137 - na_val))
  }

  meld_final <- round(meld_final, 0)
  if (meld_final > 40) meld_final <- 40

  mortality <- ""
  if (meld_final <= 9) {
    mortality <- "1.9%"
  } else if (meld_final <= 19) {
    mortality <- "6.0%"
  } else if (meld_final <= 29) {
    mortality <- "19.6%"
  } else if (meld_final <= 39) {
    mortality <- "52.6%"
  } else {
    mortality <- "71.3%"
  }

  return(list(
    MELD_Na_Score = meld_final,
    Est_3_Month_Mortality = mortality,
    Inputs = list(
      Bilirubin_mg_dL = round(bili_val, 2),
      INR = inr,
      Creatinine_mg_dL = round(creat_val, 2),
      Sodium = sodium,
      Dialysis = on_dialysis == 1
    )
  ))
}

mskcc_gist_recurrence_risk <- function(tumor_size_cm, tumor_site, mitotic_index_ge_5) {

  tumor_size_cm <- as.numeric(tumor_size_cm)
  mitotic_index_ge_5 <- as.numeric(mitotic_index_ge_5)
  tumor_site <- tolower(as.character(tumor_site))

  if (is.na(tumor_size_cm)) {
    stop("Error: Tumor size must be numeric.")
  }

  if (tumor_size_cm <= 0) {
    stop("Error: Tumor size must be positive.")
  }

  if (!tumor_site %in% c("stomach", "small_intestine", "colon_rectum", "other")) {
    stop("Error: Tumor site must be 'stomach', 'small_intestine', 'colon_rectum', or 'other'.")
  }

  size_points <- 0
  if (tumor_size_cm <= 2) {
    size_points <- 0
  } else if (tumor_size_cm <= 5) {
    size_points <- 6
  } else if (tumor_size_cm <= 10) {
    size_points <- 36
  } else {
    size_points <- 88
  }

  site_points <- 0
  if (tumor_site == "stomach") {
    site_points <- 0
  } else if (tumor_site == "small_intestine") {
    site_points <- 26
  } else if (tumor_site == "colon_rectum") {
    site_points <- 15
  } else {
    site_points <- 16
  }

  mitosis_points <- 0
  if (mitotic_index_ge_5 == 1) {
    mitosis_points <- 62
  } else {
    mitosis_points <- 0
  }

  total_points <- size_points + site_points + mitosis_points

  rfs_2yr <- NA
  rfs_5yr <- NA

  if (total_points <= 0) {
    rfs_2yr <- 100
    rfs_5yr <- 100
  } else if (total_points <= 20) {
    rfs_2yr <- 99
    rfs_5yr <- 98
  } else if (total_points <= 40) {
    rfs_2yr <- 97
    rfs_5yr <- 95
  } else if (total_points <= 60) {
    rfs_2yr <- 94
    rfs_5yr <- 88
  } else if (total_points <= 80) {
    rfs_2yr <- 88
    rfs_5yr <- 78
  } else if (total_points <= 100) {
    rfs_2yr <- 81
    rfs_5yr <- 66
  } else if (total_points <= 120) {
    rfs_2yr <- 70
    rfs_5yr <- 52
  } else if (total_points <= 140) {
    rfs_2yr <- 57
    rfs_5yr <- 38
  } else if (total_points <= 160) {
    rfs_2yr <- 42
    rfs_5yr <- 26
  } else if (total_points <= 180) {
    rfs_2yr <- 28
    rfs_5yr <- 16
  } else if (total_points <= 200) {
    rfs_2yr <- 16
    rfs_5yr <- 9
  } else if (total_points <= 220) {
    rfs_2yr <- 8
    rfs_5yr <- 4
  } else {
    rfs_2yr <- 3
    rfs_5yr <- 1
  }

  recurrence_risk_2yr <- 100 - rfs_2yr
  recurrence_risk_5yr <- 100 - rfs_5yr

  return(list(
    Recurrence_Free_Survival_2yr = paste0(rfs_2yr, "%"),
    Recurrence_Free_Survival_5yr = paste0(rfs_5yr, "%"),
    Risk_of_Recurrence_2yr = paste0(recurrence_risk_2yr, "%"),
    Risk_of_Recurrence_5yr = paste0(recurrence_risk_5yr, "%"),
    Inputs = list(
      Size_cm = tumor_size_cm,
      Site = tumor_site,
      Mitosis_ge_5 = mitotic_index_ge_5 == 1
    )
  ))
}

mskcc_motzer_score <- function(time_to_systemic_tx_lt_1yr, karnofsky_performance_status_lt_80, serum_calcium_gt_10_mg_dl, hemoglobin_lt_lln, ldh_gt_1_5_uln) {

  time_to_systemic_tx_lt_1yr <- as.numeric(time_to_systemic_tx_lt_1yr)
  karnofsky_performance_status_lt_80 <- as.numeric(karnofsky_performance_status_lt_80)
  serum_calcium_gt_10_mg_dl <- as.numeric(serum_calcium_gt_10_mg_dl)
  hemoglobin_lt_lln <- as.numeric(hemoglobin_lt_lln)
  ldh_gt_1_5_uln <- as.numeric(ldh_gt_1_5_uln)

  if (any(is.na(c(time_to_systemic_tx_lt_1yr, karnofsky_performance_status_lt_80, serum_calcium_gt_10_mg_dl, hemoglobin_lt_lln, ldh_gt_1_5_uln)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (time_to_systemic_tx_lt_1yr == 1) score <- score + 1
  if (karnofsky_performance_status_lt_80 == 1) score <- score + 1
  if (serum_calcium_gt_10_mg_dl == 1) score <- score + 1
  if (hemoglobin_lt_lln == 1) score <- score + 1
  if (ldh_gt_1_5_uln == 1) score <- score + 1

  risk_group <- ""
  median_survival <- ""

  if (score == 0) {
    risk_group <- "Favorable Risk"
    median_survival <- "30 months"
  } else if (score >= 1 && score <= 2) {
    risk_group <- "Intermediate Risk"
    median_survival <- "14 months"
  } else {
    risk_group <- "Poor Risk"
    median_survival <- "5 months"
  }

  return(list(
    MSKCC_Motzer_Score = score,
    Risk_Group = risk_group,
    Median_Overall_Survival = median_survival,
    Inputs = list(
      Time_to_Tx_lt_1yr = time_to_systemic_tx_lt_1yr == 1,
      KPS_lt_80 = karnofsky_performance_status_lt_80 == 1,
      High_Calcium = serum_calcium_gt_10_mg_dl == 1,
      Low_Hemoglobin = hemoglobin_lt_lln == 1,
      High_LDH = ldh_gt_1_5_uln == 1
    )
  ))
}

mentzer_index <- function(mcv, rbc_count) {

  mcv <- as.numeric(mcv)
  rbc_count <- as.numeric(rbc_count)

  if (any(is.na(c(mcv, rbc_count)))) {
    stop("Error: MCV and RBC count must be numeric.")
  }

  if (rbc_count == 0) {
    stop("Error: RBC count cannot be zero.")
  }

  mentzer_score <- mcv / rbc_count

  interpretation <- ""
  if (mentzer_score < 13) {
    interpretation <- "Thalassemia Trait More Likely"
  } else {
    interpretation <- "Iron Deficiency Anemia More Likely"
  }

  return(list(
    Mentzer_Index = round(mentzer_score, 1),
    Interpretation = interpretation,
    Inputs = list(
      MCV = mcv,
      RBC_Count_M_uL = rbc_count
    )
  ))
}

menza_hiv_risk_score <- function(history_sti, drug_use_meth_poppers, unprotected_anal_intercourse, partners_gt_10) {

  history_sti <- as.numeric(history_sti)
  drug_use_meth_poppers <- as.numeric(drug_use_meth_poppers)
  unprotected_anal_intercourse <- as.numeric(unprotected_anal_intercourse)
  partners_gt_10 <- as.numeric(partners_gt_10)

  if (any(is.na(c(history_sti, drug_use_meth_poppers, unprotected_anal_intercourse, partners_gt_10)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (history_sti == 1) {
    score <- score + 4
  }

  if (drug_use_meth_poppers == 1) {
    score <- score + 11
  }

  if (unprotected_anal_intercourse == 1) {
    score <- score + 1
  }

  if (partners_gt_10 == 1) {
    score <- score + 3
  }

  risk_category <- ""
  recommendation <- ""

  if (score >= 1) {
    risk_category <- "High Risk"
    recommendation <- "Targeted prevention strategies indicated (PrEP, frequent screening, counseling)."
  } else {
    risk_category <- "Low Risk"
    recommendation <- "Standard prevention counseling."
  }

  return(list(
    Menza_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      History_STI = history_sti == 1,
      Meth_Poppers_Use = drug_use_meth_poppers == 1,
      Unprotected_Anal_Sex = unprotected_anal_intercourse == 1,
      Partners_gt_10 = partners_gt_10 == 1
    )
  ))
}

mets_ir_score <- function(glucose_mg_dl, triglycerides_mg_dl, hdl_mg_dl, bmi) {

  glucose_mg_dl <- as.numeric(glucose_mg_dl)
  triglycerides_mg_dl <- as.numeric(triglycerides_mg_dl)
  hdl_mg_dl <- as.numeric(hdl_mg_dl)
  bmi <- as.numeric(bmi)

  if (any(is.na(c(glucose_mg_dl, triglycerides_mg_dl, hdl_mg_dl, bmi)))) {
    stop("Error: All inputs must be numeric.")
  }

  numerator_val <- (2 * glucose_mg_dl) + triglycerides_mg_dl
  mets_ir <- (log(numerator_val) * bmi) / log(hdl_mg_dl)

  return(list(
    METS_IR_Score = round(mets_ir, 2),
    Inputs = list(
      Glucose_mg_dL = glucose_mg_dl,
      Triglycerides_mg_dL = triglycerides_mg_dl,
      HDL_mg_dL = hdl_mg_dl,
      BMI = bmi
    )
  ))
}

metroticket_hcc_survival <- function(largest_nodule_size_mm, number_of_nodules, vascular_invasion) {

  largest_nodule_size_mm <- as.numeric(largest_nodule_size_mm)
  number_of_nodules <- as.numeric(number_of_nodules)
  vascular_invasion <- tolower(as.character(vascular_invasion))

  if (any(is.na(c(largest_nodule_size_mm, number_of_nodules)))) {
    stop("Error: Tumor size and number must be numeric.")
  }

  if (!vascular_invasion %in% c("absent", "present", "unknown")) {
    stop("Error: Vascular invasion must be 'absent', 'present', or 'unknown'.")
  }

  size_cm <- largest_nodule_size_mm / 10

  up_to_seven_score <- size_cm + number_of_nodules

  survival_prediction <- ""
  criteria_met <- ""

  if (vascular_invasion == "present") {
    survival_prediction <- "Significant reduction in survival due to microvascular invasion. 5-year survival estimated < 50%."
    criteria_met <- "Does not meet Up-to-7 Criteria (Vascular Invasion)."
  } else if (vascular_invasion == "unknown") {
    survival_prediction <- "Survival prediction limited by unknown vascular invasion status. If invasion is absent, prognosis follows Up-to-7 rule."
    criteria_met <- "Unknown"
  } else {
    if (up_to_seven_score <= 7) {
      survival_prediction <- "Estimated 5-year Overall Survival: 71.2%"
      criteria_met <- "Meets Up-to-7 Criteria"
    } else {
      survival_prediction <- "Estimated 5-year Overall Survival: ~48% (Decreases as score increases)"
      criteria_met <- "Does not meet Up-to-7 Criteria"
    }
  }

  return(list(
    Up_To_Seven_Score = round(up_to_seven_score, 1),
    Predicted_5_Year_Survival = survival_prediction,
    Criteria_Status = criteria_met,
    Inputs = list(
      Size_mm = largest_nodule_size_mm,
      Count = number_of_nodules,
      Invasion = vascular_invasion
    )
  ))
}

michigan_picc_risk_score <- function(history_thrombosis, multilumen_catheter, active_cancer, another_cvc_present, wbc_gt_12000) {

  history_thrombosis <- as.numeric(history_thrombosis)
  multilumen_catheter <- as.numeric(multilumen_catheter)
  active_cancer <- as.numeric(active_cancer)
  another_cvc_present <- as.numeric(another_cvc_present)
  wbc_gt_12000 <- as.numeric(wbc_gt_12000)

  if (any(is.na(c(history_thrombosis, multilumen_catheter, active_cancer, another_cvc_present, wbc_gt_12000)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (history_thrombosis == 1) {
    score <- score + 2
  }

  if (multilumen_catheter == 1) {
    score <- score + 1
  }

  if (active_cancer == 1) {
    score <- score + 3
  }

  if (another_cvc_present == 1) {
    score <- score + 1
  }

  if (wbc_gt_12000 == 1) {
    score <- score + 1
  }

  risk_class <- ""
  thrombosis_risk <- ""

  if (score == 0) {
    risk_class <- "Class I (Low Risk)"
    thrombosis_risk <- "0.9%"
  } else if (score <= 2) {
    risk_class <- "Class II (Moderate Risk)"
    thrombosis_risk <- "1.6%"
  } else if (score == 3) {
    risk_class <- "Class III (High Risk)"
    thrombosis_risk <- "2.7%"
  } else {
    risk_class <- "Class IV (Very High Risk)"
    thrombosis_risk <- "4.5%"
  }

  return(list(
    MRS_Score = score,
    Risk_Class = risk_class,
    Est_Thrombosis_Risk = thrombosis_risk,
    Inputs = list(
      Hx_Thrombosis = history_thrombosis == 1,
      MultiLumen = multilumen_catheter == 1,
      Active_Cancer = active_cancer == 1,
      Another_CVC = another_cvc_present == 1,
      WBC_High = wbc_gt_12000 == 1
    )
  ))
}

midas_migraine_score <- function(missed_work_school, reduced_work_school, missed_housework, reduced_housework, missed_social_leisure) {

  inputs <- c(missed_work_school, reduced_work_school, missed_housework, reduced_housework, missed_social_leisure)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric days.")
  }

  if (any(inputs < 0)) {
    stop("Error: Days cannot be negative.")
  }

  total_days <- sum(inputs)

  grade <- ""
  severity <- ""

  if (total_days <= 5) {
    grade <- "Grade I"
    severity <- "Little or No Disability"
  } else if (total_days <= 10) {
    grade <- "Grade II"
    severity <- "Mild Disability"
  } else if (total_days <= 20) {
    grade <- "Grade III"
    severity <- "Moderate Disability"
  } else {
    grade <- "Grade IV"
    severity <- "Severe Disability"
  }

  return(list(
    MIDAS_Score = total_days,
    Grade = grade,
    Interpretation = severity,
    Inputs = list(
      Q1_Missed_Work = missed_work_school,
      Q2_Reduced_Work = reduced_work_school,
      Q3_Missed_Home = missed_housework,
      Q4_Reduced_Home = reduced_housework,
      Q5_Missed_Social = missed_social_leisure
    )
  ))
}

mtoq_4_score <- function(q1_pain_free_2h, q2_function_2h, q3_comfortable_2h, q4_single_dose) {

  q1_pain_free_2h <- as.numeric(q1_pain_free_2h)
  q2_function_2h <- as.numeric(q2_function_2h)
  q3_comfortable_2h <- as.numeric(q3_comfortable_2h)
  q4_single_dose <- as.numeric(q4_single_dose)

  if (any(is.na(c(q1_pain_free_2h, q2_function_2h, q3_comfortable_2h, q4_single_dose)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(c(q1_pain_free_2h, q2_function_2h, q3_comfortable_2h, q4_single_dose) < 0) ||
      any(c(q1_pain_free_2h, q2_function_2h, q3_comfortable_2h, q4_single_dose) > 4)) {
    stop("Error: All item scores must be integers between 0 (Never) and 4 (All the time).")
  }

  total_score <- q1_pain_free_2h + q2_function_2h + q3_comfortable_2h + q4_single_dose

  optimization <- ""
  if (total_score == 0) {
    optimization <- "Poor Optimization"
  } else if (total_score <= 5) {
    optimization <- "Low Optimization"
  } else if (total_score <= 8) {
    optimization <- "Moderate Optimization"
  } else {
    optimization <- "Maximum Optimization (Goal)"
  }

  return(list(
    mTOQ_4_Score = total_score,
    Optimization_Status = optimization,
    Inputs = list(
      Q1_PainFree = q1_pain_free_2h,
      Q2_Function = q2_function_2h,
      Q3_Comfortable = q3_comfortable_2h,
      Q4_SingleDose = q4_single_dose
    )
  ))
}

nafld_activity_score <- function(steatosis_grade, lobular_inflammation, hepatocyte_ballooning) {

  steatosis_grade <- as.numeric(steatosis_grade)
  lobular_inflammation <- as.numeric(lobular_inflammation)
  hepatocyte_ballooning <- as.numeric(hepatocyte_ballooning)

  if (any(is.na(c(steatosis_grade, lobular_inflammation, hepatocyte_ballooning)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (steatosis_grade < 0 || steatosis_grade > 3) stop("Error: Steatosis grade must be 0-3.")
  if (lobular_inflammation < 0 || lobular_inflammation > 3) stop("Error: Lobular inflammation must be 0-3.")
  if (hepatocyte_ballooning < 0 || hepatocyte_ballooning > 2) stop("Error: Ballooning score must be 0-2.")

  nas_score <- steatosis_grade + lobular_inflammation + hepatocyte_ballooning

  interpretation <- ""
  if (nas_score >= 5) {
    interpretation <- "NASH Likely (Score >= 5)"
  } else if (nas_score >= 3 && nas_score <= 4) {
    interpretation <- "Borderline NASH (Score 3-4)"
  } else {
    interpretation <- "NASH Unlikely (Score < 3)"
  }

  return(list(
    NAS_Score = nas_score,
    Interpretation = interpretation,
    Inputs = list(
      Steatosis = steatosis_grade,
      Inflammation = lobular_inflammation,
      Ballooning = hepatocyte_ballooning
    )
  ))
}

naloxone_drip_dosing <- function(wake_up_bolus_mg) {

  wake_up_bolus_mg <- as.numeric(wake_up_bolus_mg)

  if (is.na(wake_up_bolus_mg) || wake_up_bolus_mg <= 0) {
    stop("Error: Wake-up bolus dose must be a positive number.")
  }

  hourly_rate <- wake_up_bolus_mg * (2/3)

  return(list(
    Calculated_Hourly_Rate_mg_hr = round(hourly_rate, 2),
    Protocol_Note = "Set continuous infusion rate at 2/3 of the effective wake-up bolus dose per hour.",
    Inputs = list(
      Wake_Up_Bolus_mg = wake_up_bolus_mg
    )
  ))
}

nccn_ipi_score <- function(age, ldh_ratio, extranodal_sites_disease, stage, performance_status) {

  age <- as.numeric(age)
  ldh_ratio <- as.numeric(ldh_ratio)
  extranodal_sites_disease <- tolower(as.character(extranodal_sites_disease))
  stage <- as.numeric(stage)
  performance_status <- as.numeric(performance_status)

  if (any(is.na(c(age, ldh_ratio, stage, performance_status)))) {
    stop("Error: All numeric inputs must be valid.")
  }

  if (!extranodal_sites_disease %in% c("bone_marrow", "cns", "liver", "gi_tract", "lung", "none_or_other")) {
    stop("Error: Extranodal disease must be 'bone_marrow', 'cns', 'liver', 'gi_tract', 'lung', or 'none_or_other'.")
  }

  score <- 0

  if (age > 40 && age <= 60) {
    score <- score + 1
  } else if (age > 60 && age <= 75) {
    score <- score + 2
  } else if (age > 75) {
    score <- score + 3
  }

  if (ldh_ratio > 1 && ldh_ratio <= 3) {
    score <- score + 1
  } else if (ldh_ratio > 3) {
    score <- score + 2
  }

  if (extranodal_sites_disease %in% c("bone_marrow", "cns", "liver", "gi_tract", "lung")) {
    score <- score + 1
  }

  if (stage == 3 || stage == 4) {
    score <- score + 1
  }

  if (performance_status >= 2) {
    score <- score + 1
  }

  risk_group <- ""
  survival_5yr <- ""

  if (score <= 1) {
    risk_group <- "Low Risk"
    survival_5yr <- "96%"
  } else if (score <= 3) {
    risk_group <- "Low-Intermediate Risk"
    survival_5yr <- "82%"
  } else if (score <= 5) {
    risk_group <- "High-Intermediate Risk"
    survival_5yr <- "64%"
  } else {
    risk_group <- "High Risk"
    survival_5yr <- "33%"
  }

  return(list(
    NCCN_IPI_Score = score,
    Risk_Group = risk_group,
    Est_5_Year_Overall_Survival = survival_5yr,
    Inputs = list(
      Age = age,
      LDH_Ratio = ldh_ratio,
      Extranodal_Disease = extranodal_sites_disease,
      Stage = stage,
      ECOG_PS = performance_status
    )
  ))
}

news_score <- function(respiratory_rate, oxygen_saturation, any_supplemental_oxygen, temperature_c, systolic_bp, heart_rate, avpu_score) {

  respiratory_rate <- as.numeric(respiratory_rate)
  oxygen_saturation <- as.numeric(oxygen_saturation)
  any_supplemental_oxygen <- as.numeric(any_supplemental_oxygen)
  temperature_c <- as.numeric(temperature_c)
  systolic_bp <- as.numeric(systolic_bp)
  heart_rate <- as.numeric(heart_rate)
  avpu_score <- tolower(as.character(avpu_score))

  if (any(is.na(c(respiratory_rate, oxygen_saturation, any_supplemental_oxygen, temperature_c, systolic_bp, heart_rate)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (respiratory_rate <= 8) {
    score <- score + 3
  } else if (respiratory_rate >= 9 && respiratory_rate <= 11) {
    score <- score + 1
  } else if (respiratory_rate >= 12 && respiratory_rate <= 20) {
    score <- score + 0
  } else if (respiratory_rate >= 21 && respiratory_rate <= 24) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (oxygen_saturation <= 91) {
    score <- score + 3
  } else if (oxygen_saturation >= 92 && oxygen_saturation <= 93) {
    score <- score + 2
  } else if (oxygen_saturation >= 94 && oxygen_saturation <= 95) {
    score <- score + 1
  } else {
    score <- score + 0
  }

  if (any_supplemental_oxygen == 1) {
    score <- score + 2
  } else {
    score <- score + 0
  }

  if (temperature_c <= 35.0) {
    score <- score + 3
  } else if (temperature_c >= 35.1 && temperature_c <= 36.0) {
    score <- score + 1
  } else if (temperature_c >= 36.1 && temperature_c <= 38.0) {
    score <- score + 0
  } else if (temperature_c >= 38.1 && temperature_c <= 39.0) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  if (systolic_bp <= 90) {
    score <- score + 3
  } else if (systolic_bp >= 91 && systolic_bp <= 100) {
    score <- score + 2
  } else if (systolic_bp >= 101 && systolic_bp <= 110) {
    score <- score + 1
  } else if (systolic_bp >= 111 && systolic_bp <= 219) {
    score <- score + 0
  } else {
    score <- score + 3
  }

  if (heart_rate <= 40) {
    score <- score + 3
  } else if (heart_rate >= 41 && heart_rate <= 50) {
    score <- score + 1
  } else if (heart_rate >= 51 && heart_rate <= 90) {
    score <- score + 0
  } else if (heart_rate >= 91 && heart_rate <= 110) {
    score <- score + 1
  } else if (heart_rate >= 111 && heart_rate <= 130) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (avpu_score == "alert") {
    score <- score + 0
  } else {
    score <- score + 3
  }

  risk_level <- ""
  clinical_risk <- ""

  if (score <= 4) {
    if (any(c(respiratory_rate <= 8, respiratory_rate >= 25, oxygen_saturation <= 91, temperature_c <= 35.0, systolic_bp <= 90, systolic_bp >= 220, heart_rate <= 40, heart_rate >= 131, avpu_score != "alert"))) {
      risk_level <- "Low-Medium Risk (Red Score)"
    } else {
      risk_level <- "Low Risk"
    }
    clinical_risk <- "Ward-based response"
  } else if (score <= 6) {
    risk_level <- "Medium Risk"
    clinical_risk <- "Urgent response threshold"
  } else {
    risk_level <- "High Risk"
    clinical_risk <- "Urgent or Emergency response threshold"
  }

  return(list(
    NEWS_Score = score,
    Risk_Category = risk_level,
    Response_Level = clinical_risk,
    Inputs = list(
      RR = respiratory_rate,
      SpO2 = oxygen_saturation,
      O2_Support = any_supplemental_oxygen == 1,
      Temp = temperature_c,
      SBP = systolic_bp,
      HR = heart_rate,
      AVPU = avpu_score
    )
  ))
}

news2_score <- function(respiratory_rate, oxygen_saturation, supplemental_oxygen, systolic_bp, heart_rate, consciousness, temperature, hypercapnic_respiratory_failure = FALSE) {

  respiratory_rate <- as.numeric(respiratory_rate)
  oxygen_saturation <- as.numeric(oxygen_saturation)
  supplemental_oxygen <- as.numeric(supplemental_oxygen)
  systolic_bp <- as.numeric(systolic_bp)
  heart_rate <- as.numeric(heart_rate)
  temperature <- as.numeric(temperature)
  hypercapnic_respiratory_failure <- as.numeric(hypercapnic_respiratory_failure)
  consciousness <- tolower(as.character(consciousness))

  if (any(is.na(c(respiratory_rate, oxygen_saturation, supplemental_oxygen, systolic_bp, heart_rate, temperature)))) {
    stop("Error: All physiological inputs must be numeric.")
  }

  score <- 0

  if (respiratory_rate <= 8) {
    score <- score + 3
  } else if (respiratory_rate >= 9 && respiratory_rate <= 11) {
    score <- score + 1
  } else if (respiratory_rate >= 12 && respiratory_rate <= 20) {
    score <- score + 0
  } else if (respiratory_rate >= 21 && respiratory_rate <= 24) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (hypercapnic_respiratory_failure == 1) {
    if (oxygen_saturation <= 83) {
      score <- score + 3
    } else if (oxygen_saturation >= 84 && oxygen_saturation <= 85) {
      score <- score + 2
    } else if (oxygen_saturation >= 86 && oxygen_saturation <= 87) {
      score <- score + 1
    } else if (oxygen_saturation >= 88 && oxygen_saturation <= 92) {
      score <- score + 0
    } else if (oxygen_saturation >= 93 && oxygen_saturation <= 94) {
      score <- score + 1
    } else if (oxygen_saturation >= 95 && oxygen_saturation <= 96) {
      score <- score + 2
    } else {
      score <- score + 3
    }
  } else {
    if (oxygen_saturation <= 91) {
      score <- score + 3
    } else if (oxygen_saturation >= 92 && oxygen_saturation <= 93) {
      score <- score + 2
    } else if (oxygen_saturation >= 94 && oxygen_saturation <= 95) {
      score <- score + 1
    } else {
      score <- score + 0
    }
  }

  if (supplemental_oxygen == 1) {
    score <- score + 2
  }

  if (systolic_bp <= 90) {
    score <- score + 3
  } else if (systolic_bp >= 91 && systolic_bp <= 100) {
    score <- score + 2
  } else if (systolic_bp >= 101 && systolic_bp <= 110) {
    score <- score + 1
  } else if (systolic_bp >= 111 && systolic_bp <= 219) {
    score <- score + 0
  } else {
    score <- score + 3
  }

  if (heart_rate <= 40) {
    score <- score + 3
  } else if (heart_rate >= 41 && heart_rate <= 50) {
    score <- score + 1
  } else if (heart_rate >= 51 && heart_rate <= 90) {
    score <- score + 0
  } else if (heart_rate >= 91 && heart_rate <= 110) {
    score <- score + 1
  } else if (heart_rate >= 111 && heart_rate <= 130) {
    score <- score + 2
  } else {
    score <- score + 3
  }

  if (consciousness == "alert") {
    score <- score + 0
  } else {
    score <- score + 3
  }

  if (temperature <= 35.0) {
    score <- score + 3
  } else if (temperature >= 35.1 && temperature <= 36.0) {
    score <- score + 1
  } else if (temperature >= 36.1 && temperature <= 38.0) {
    score <- score + 0
  } else if (temperature >= 38.1 && temperature <= 39.0) {
    score <- score + 1
  } else {
    score <- score + 2
  }

  clinical_risk <- ""
  response <- ""

  if (score <= 4) {
    if (score >= 1 && score <= 4) {
      clinical_risk <- "Low"
      response <- "Ward-based response"
    } else {
      clinical_risk <- "Low"
      response <- "Standard frequency of monitoring"
    }
  } else if (score <= 6) {
    if (score == 3 && (respiratory_rate <= 8 || respiratory_rate >= 25 || oxygen_saturation <= 91 || systolic_bp <= 90 || systolic_bp >= 220 || heart_rate <= 40 || heart_rate >= 131 || consciousness != "alert")) {
      clinical_risk <- "Low-Medium"
      response <- "Urgent ward-based response"
    } else {
      clinical_risk <- "Medium"
      response <- "Urgent response threshold"
    }
  } else {
    clinical_risk <- "High"
    response <- "Urgent or Emergency response threshold"
  }

  return(list(
    NEWS2_Score = score,
    Clinical_Risk = clinical_risk,
    Response_Urgency = response,
    Inputs = list(
      RR = respiratory_rate,
      SpO2 = oxygen_saturation,
      Scale_Used = if(hypercapnic_respiratory_failure == 1) "Scale 2 (Hypercapnic)" else "Scale 1 (Standard)",
      O2_Support = supplemental_oxygen == 1,
      SBP = systolic_bp,
      HR = heart_rate,
      Consciousness = consciousness,
      Temp = temperature
    )
  ))
}

nrpe_diuretic_response <- function(spot_urine_sodium) {

  spot_urine_sodium <- as.numeric(spot_urine_sodium)

  if (is.na(spot_urine_sodium)) {
    stop("Error: Spot urine sodium must be numeric.")
  }

  if (spot_urine_sodium < 0) {
    stop("Error: Urine sodium cannot be negative.")
  }

  predicted_6h_sodium_output <- (4.65 * spot_urine_sodium) - 22.6

  if (predicted_6h_sodium_output < 0) {
    predicted_6h_sodium_output <- 0
  }

  clinical_utility <- ""
  if (spot_urine_sodium < 50) {
    clinical_utility <- "Poor Natriuretic Response (<50 mmol/L). Likely insufficient decongestion."
  } else {
    clinical_utility <- "Adequate Natriuretic Response (>=50 mmol/L)."
  }

  return(list(
    Predicted_6hr_Sodium_Output_mmol = round(predicted_6h_sodium_output, 1),
    Interpretation = clinical_utility,
    Inputs = list(
      Spot_Urine_Na_mmol_L = spot_urine_sodium
    )
  ))
}

oakland_score_lgib <- function(age, sex, previous_lgib_admission, dre_blood, heart_rate, systolic_bp, hemoglobin, hemoglobin_units = "g/dL") {

  age <- as.numeric(age)
  sex <- tolower(as.character(sex))
  previous_lgib_admission <- as.numeric(previous_lgib_admission)
  dre_blood <- as.numeric(dre_blood)
  heart_rate <- as.numeric(heart_rate)
  systolic_bp <- as.numeric(systolic_bp)
  hemoglobin <- as.numeric(hemoglobin)

  if (any(is.na(c(age, previous_lgib_admission, dre_blood, heart_rate, systolic_bp, hemoglobin)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  hgb_g_l <- hemoglobin
  if (hemoglobin_units == "g/dL") {
    hgb_g_l <- hemoglobin * 10
  } else if (hemoglobin_units == "mmol/L") {
    hgb_g_l <- hemoglobin * 16.115
  }

  score <- 0

  if (age >= 80) {
    score <- score + 2
  } else if (age >= 70) {
    score <- score + 1
  }

  if (sex == "male") {
    score <- score + 1
  }

  if (previous_lgib_admission == 1) {
    score <- score + 1
  }

  if (dre_blood == 1) {
    score <- score + 1
  }

  if (heart_rate >= 110) {
    score <- score + 3
  } else if (heart_rate >= 90) {
    score <- score + 2
  } else if (heart_rate >= 70) {
    score <- score + 1
  }

  if (systolic_bp < 90) {
    score <- score + 5
  } else if (systolic_bp >= 90 && systolic_bp <= 119) {
    score <- score + 4
  } else if (systolic_bp >= 120 && systolic_bp <= 129) {
    score <- score + 3
  } else if (systolic_bp >= 130 && systolic_bp <= 159) {
    score <- score + 2
  }

  if (hgb_g_l < 70) {
    score <- score + 22
  } else if (hgb_g_l >= 70 && hgb_g_l <= 89) {
    score <- score + 17
  } else if (hgb_g_l >= 90 && hgb_g_l <= 109) {
    score <- score + 13
  } else if (hgb_g_l >= 110 && hgb_g_l <= 129) {
    score <- score + 8
  } else if (hgb_g_l >= 130 && hgb_g_l <= 159) {
    score <- score + 4
  }

  recommendation <- ""
  safe_discharge_prob <- ""

  if (score <= 8) {
    recommendation <- "Safe Discharge (Low Risk of Adverse Events)"
    safe_discharge_prob <- "95%"
  } else {
    recommendation <- "Admission Recommended (Not Low Risk)"
    safe_discharge_prob <- "< 95%"
  }

  return(list(
    Oakland_Score = score,
    Recommendation = recommendation,
    Probability_Safe_Discharge = safe_discharge_prob,
    Inputs = list(
      Age = age,
      Sex = sex,
      Prev_Admit = previous_lgib_admission == 1,
      DRE_Blood = dre_blood == 1,
      HR = heart_rate,
      SBP = systolic_bp,
      Hemoglobin_g_L = round(hgb_g_l, 1)
    )
  ))
}

obesity_surgery_mortality_risk_score <- function(bmi_ge_50, male_sex, hypertension, risk_factors_dvt_pe, age_ge_45) {

  bmi_ge_50 <- as.numeric(bmi_ge_50)
  male_sex <- as.numeric(male_sex)
  hypertension <- as.numeric(hypertension)
  risk_factors_dvt_pe <- as.numeric(risk_factors_dvt_pe)
  age_ge_45 <- as.numeric(age_ge_45)

  if (any(is.na(c(bmi_ge_50, male_sex, hypertension, risk_factors_dvt_pe, age_ge_45)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (bmi_ge_50 == 1) {
    score <- score + 1
  }

  if (male_sex == 1) {
    score <- score + 1
  }

  if (hypertension == 1) {
    score <- score + 1
  }

  if (risk_factors_dvt_pe == 1) {
    score <- score + 1
  }

  if (age_ge_45 == 1) {
    score <- score + 1
  }

  risk_class <- ""
  mortality_rate <- ""

  if (score <= 1) {
    risk_class <- "Class A (Low Risk)"
    mortality_rate <- "0.2 - 0.3%"
  } else if (score >= 2 && score <= 3) {
    risk_class <- "Class B (Intermediate Risk)"
    mortality_rate <- "1.1 - 1.2%"
  } else {
    risk_class <- "Class C (High Risk)"
    mortality_rate <- "2.4 - 3.0%"
  }

  return(list(
    OS_MRS_Score = score,
    Risk_Class = risk_class,
    Est_Mortality_90_Day = mortality_rate,
    Inputs = list(
      BMI_ge_50 = bmi_ge_50 == 1,
      Male = male_sex == 1,
      Hypertension = hypertension == 1,
      Risk_Pulmonary_Embolism = risk_factors_dvt_pe == 1,
      Age_ge_45 = age_ge_45 == 1
    )
  ))
}

ohts_risk_score <- function(age, iop, cct, vcdr, psd) {

  age <- as.numeric(age)
  iop <- as.numeric(iop)
  cct <- as.numeric(cct)
  vcdr <- as.numeric(vcdr)
  psd <- as.numeric(psd)

  if (any(is.na(c(age, iop, cct, vcdr, psd)))) {
    stop("Error: All inputs must be numeric.")
  }

  points <- 0

  if (age >= 50 && age <= 59) {
    points <- points + 1
  } else if (age >= 60 && age <= 69) {
    points <- points + 2
  } else if (age >= 70) {
    points <- points + 3
  }

  if (iop >= 22 && iop <= 23.9) {
    points <- points + 1
  } else if (iop >= 24 && iop <= 26.9) {
    points <- points + 2
  } else if (iop >= 27 && iop <= 30) {
    points <- points + 3
  } else if (iop > 30) {
    points <- points + 4
  }

  if (cct <= 555) {
    points <- points + 2
  } else if (cct >= 556 && cct <= 588) {
    points <- points + 1
  }

  if (vcdr >= 0.4 && vcdr <= 0.6) {
    points <- points + 1
  } else if (vcdr > 0.6) {
    points <- points + 2
  }

  if (psd >= 1.8 && psd <= 2.5) {
    points <- points + 1
  } else if (psd > 2.5) {
    points <- points + 2
  }

  risk_percent <- ""

  if (points <= 1) {
    risk_percent <- "< 2%"
  } else if (points == 2) {
    risk_percent <- "2 - 3%"
  } else if (points == 3) {
    risk_percent <- "4 - 5%"
  } else if (points == 4) {
    risk_percent <- "6 - 8%"
  } else if (points == 5) {
    risk_percent <- "10 - 12%"
  } else if (points == 6) {
    risk_percent <- "16 - 19%"
  } else if (points == 7) {
    risk_percent <- "23 - 27%"
  } else if (points == 8) {
    risk_percent <- "32 - 37%"
  } else {
    risk_percent <- "> 40%"
  }

  risk_group <- ""
  if (points <= 3) {
    risk_group <- "Low Risk"
  } else if (points <= 5) {
    risk_group <- "Intermediate Risk"
  } else {
    risk_group <- "High Risk"
  }

  return(list(
    OHTS_Points = points,
    Five_Year_POAG_Risk = risk_percent,
    Risk_Category = risk_group,
    Inputs = list(
      Age = age,
      IOP_mmHg = iop,
      CCT_microns = cct,
      VCDR = vcdr,
      PSD_dB = psd
    )
  ))
}

oesil_syncope_score <- function(age, cv_history, no_prodrome, abnormal_ecg) {

  age <- as.numeric(age)
  cv_history <- as.numeric(cv_history)
  no_prodrome <- as.numeric(no_prodrome)
  abnormal_ecg <- as.numeric(abnormal_ecg)

  if (any(is.na(c(age, cv_history, no_prodrome, abnormal_ecg)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (age > 65) {
    score <- score + 1
  }

  if (cv_history == 1) {
    score <- score + 1
  }

  if (no_prodrome == 1) {
    score <- score + 1
  }

  if (abnormal_ecg == 1) {
    score <- score + 1
  }

  mortality_1yr <- ""

  if (score == 0) {
    mortality_1yr <- "0%"
  } else if (score == 1) {
    mortality_1yr <- "0.8%"
  } else if (score == 2) {
    mortality_1yr <- "19.6%"
  } else if (score == 3) {
    mortality_1yr <- "34.7%"
  } else {
    mortality_1yr <- "57.1%"
  }

  return(list(
    OESIL_Score = score,
    Est_1_Year_Mortality = mortality_1yr,
    Inputs = list(
      Age_gt_65 = age > 65,
      CV_History = cv_history == 1,
      No_Prodrome = no_prodrome == 1,
      Abnormal_ECG = abnormal_ecg == 1
    )
  ))
}

ovat_screening_tool <- function(physically_hurt, threaten_harm, scream_curse, insult_talk_down) {

  inputs <- c(physically_hurt, threaten_harm, scream_curse, insult_talk_down)

  if (any(is.na(inputs))) {
    stop("Error: All inputs must be numeric.")
  }

  if (any(!inputs %in% c(0, 1))) {
    stop("Error: Inputs must be binary (0 for No, 1 for Yes).")
  }

  score <- sum(inputs)

  result <- ""
  recommendation <- ""

  if (score >= 1) {
    result <- "Positive Screen"
    recommendation <- "The patient is likely experiencing ongoing intimate partner violence. Safety assessment and referral indicated."
  } else {
    result <- "Negative Screen"
    recommendation <- "No ongoing violence disclosed. Monitor and rescreen if suspicion remains."
  }

  return(list(
    OVAT_Score = score,
    Screening_Result = result,
    Recommendation = recommendation,
    Inputs = list(
      Hurt = physically_hurt == 1,
      Threaten = threaten_harm == 1,
      Scream = scream_curse == 1,
      Insult = insult_talk_down == 1
    )
  ))
}

opioid_conversion_calculator <- function(current_drug, current_route, current_dose_mg, target_drug, target_route, reduction_percent = 0) {

  current_drug <- tolower(as.character(current_drug))
  current_route <- tolower(as.character(current_route))
  target_drug <- tolower(as.character(target_drug))
  target_route <- tolower(as.character(target_route))
  current_dose_mg <- as.numeric(current_dose_mg)
  reduction_percent <- as.numeric(reduction_percent)

  if (any(is.na(c(current_dose_mg, reduction_percent)))) {
    stop("Error: Dose and reduction percentage must be numeric.")
  }

  key_in <- paste(current_drug, current_route, sep = "_")
  key_out <- paste(target_drug, target_route, sep = "_")

  factors <- c(
    "morphine_po" = 1.0,
    "morphine_iv" = 3.0,
    "oxycodone_po" = 1.5,
    "hydrocodone_po" = 1.0,
    "hydromorphone_po" = 4.0,
    "hydromorphone_iv" = 20.0,
    "codeine_po" = 0.15,
    "tramadol_po" = 0.1,
    "oxymorphone_po" = 3.0,
    "oxymorphone_iv" = 30.0,
    "fentanyl_iv" = 300.0
  )

  if (!key_in %in% names(factors)) {
    stop(paste("Error: Current drug/route combination not supported or invalid:", key_in))
  }

  if (!key_out %in% names(factors)) {
    stop(paste("Error: Target drug/route combination not supported or invalid:", key_out))
  }

  mme_conversion_factor <- factors[[key_in]]

  total_mme <- current_dose_mg * mme_conversion_factor

  reduced_mme <- total_mme * (1 - (reduction_percent / 100))

  target_factor <- factors[[key_out]]
  target_dose <- reduced_mme / target_factor

  return(list(
    Input_Dose = paste(current_dose_mg, "mg", current_drug, current_route),
    Morphine_Milligram_Equivalents = round(total_mme, 1),
    Target_Dose_Calculated = round(target_dose, 2),
    Target_Drug = paste(target_drug, target_route),
    Reduction_Applied = paste0(reduction_percent, "%")
  ))
}

opioid_risk_tool <- function(sex, age, family_hx_alcohol, family_hx_illegal, family_hx_rx, personal_hx_alcohol, personal_hx_illegal, personal_hx_rx, history_sexual_abuse, psych_disease_adhd_ocd_bipolar_schizo, psych_disease_depression) {

  sex <- tolower(as.character(sex))
  age <- as.numeric(age)
  family_hx_alcohol <- as.numeric(family_hx_alcohol)
  family_hx_illegal <- as.numeric(family_hx_illegal)
  family_hx_rx <- as.numeric(family_hx_rx)
  personal_hx_alcohol <- as.numeric(personal_hx_alcohol)
  personal_hx_illegal <- as.numeric(personal_hx_illegal)
  personal_hx_rx <- as.numeric(personal_hx_rx)
  history_sexual_abuse <- as.numeric(history_sexual_abuse)
  psych_disease_adhd_ocd_bipolar_schizo <- as.numeric(psych_disease_adhd_ocd_bipolar_schizo)
  psych_disease_depression <- as.numeric(psych_disease_depression)

  if (!sex %in% c("male", "female")) {
    stop("Error: Sex must be 'male' or 'female'.")
  }

  if (any(is.na(c(age, family_hx_alcohol, family_hx_illegal, family_hx_rx, personal_hx_alcohol, personal_hx_illegal, personal_hx_rx, history_sexual_abuse, psych_disease_adhd_ocd_bipolar_schizo, psych_disease_depression)))) {
    stop("Error: All inputs must be numeric (0 or 1, except Age).")
  }

  score <- 0

  if (sex == "female") {
    if (family_hx_alcohol == 1) score <- score + 1
    if (family_hx_illegal == 1) score <- score + 2
    if (family_hx_rx == 1) score <- score + 4
    if (personal_hx_alcohol == 1) score <- score + 3
    if (personal_hx_illegal == 1) score <- score + 4
    if (personal_hx_rx == 1) score <- score + 5
    if (age >= 16 && age <= 45) score <- score + 1
    if (history_sexual_abuse == 1) score <- score + 3
    if (psych_disease_adhd_ocd_bipolar_schizo == 1) score <- score + 2
    if (psych_disease_depression == 1) score <- score + 1
  } else {
    if (family_hx_alcohol == 1) score <- score + 3
    if (family_hx_illegal == 1) score <- score + 3
    if (family_hx_rx == 1) score <- score + 4
    if (personal_hx_alcohol == 1) score <- score + 3
    if (personal_hx_illegal == 1) score <- score + 4
    if (personal_hx_rx == 1) score <- score + 5
    if (age >= 16 && age <= 45) score <- score + 1
    if (psych_disease_adhd_ocd_bipolar_schizo == 1) score <- score + 2
    if (psych_disease_depression == 1) score <- score + 1
  }

  risk_category <- ""
  if (score <= 3) {
    risk_category <- "Low Risk"
  } else if (score <= 7) {
    risk_category <- "Moderate Risk"
  } else {
    risk_category <- "High Risk"
  }

  return(list(
    ORT_Score = score,
    Risk_Category = risk_category,
    Inputs = list(
      Sex = sex,
      Age = age,
      Family_Hx = c(Alcohol=family_hx_alcohol, Illegal=family_hx_illegal, Rx=family_hx_rx),
      Personal_Hx = c(Alcohol=personal_hx_alcohol, Illegal=personal_hx_illegal, Rx=personal_hx_rx),
      Sexual_Abuse = history_sexual_abuse == 1,
      Psych = c(Severe=psych_disease_adhd_ocd_bipolar_schizo, Depression=psych_disease_depression)
    )
  ))
}

orbit_bleeding_risk_score <- function(age_gt_74, reduced_hemoglobin_or_hct_anemia, bleeding_history, insufficient_kidney_function, treatment_with_antiplatelet) {

  age_gt_74 <- as.numeric(age_gt_74)
  reduced_hemoglobin_or_hct_anemia <- as.numeric(reduced_hemoglobin_or_hct_anemia)
  bleeding_history <- as.numeric(bleeding_history)
  insufficient_kidney_function <- as.numeric(insufficient_kidney_function)
  treatment_with_antiplatelet <- as.numeric(treatment_with_antiplatelet)

  if (any(is.na(c(age_gt_74, reduced_hemoglobin_or_hct_anemia, bleeding_history, insufficient_kidney_function, treatment_with_antiplatelet)))) {
    stop("Error: All inputs must be numeric (0 or 1).")
  }

  score <- 0

  if (age_gt_74 == 1) {
    score <- score + 1
  }

  if (reduced_hemoglobin_or_hct_anemia == 1) {
    score <- score + 2
  }

  if (bleeding_history == 1) {
    score <- score + 2
  }

  if (insufficient_kidney_function == 1) {
    score <- score + 1
  }

  if (treatment_with_antiplatelet == 1) {
    score <- score + 1
  }

  risk_group <- ""
  bleeds_per_100_py <- ""

  if (score <= 2) {
    risk_group <- "Low Risk"
    bleeds_per_100_py <- "2.4 bleeds per 100 patient-years"
  } else if (score == 3) {
    risk_group <- "Medium Risk"
    bleeds_per_100_py <- "4.7 bleeds per 100 patient-years"
  } else {
    risk_group <- "High Risk"
    bleeds_per_100_py <- "8.1 bleeds per 100 patient-years"
  }

  return(list(
    ORBIT_Score = score,
    Risk_Group = risk_group,
    Bleeding_Risk = bleeds_per_100_py,
    Inputs = list(
      Older_Age = age_gt_74 == 1,
      Reduced_Hb_Hct_Anemia = reduced_hemoglobin_or_hct_anemia == 1,
      Bleeding_Hx = bleeding_history == 1,
      Insufficient_Kidney_Fx = insufficient_kidney_function == 1,
      Treatment_Antiplatelet = treatment_with_antiplatelet == 1
    )
  ))
}

calculate_tir <- function(data_vector, target_min, target_max) {

  if (!is.numeric(data_vector)) {
    return("Error: 'data_vector' must be a numeric vector.")
  }

  if (!is.numeric(target_min) || length(target_min) != 1) {
    return("Error: 'target_min' must be a single numeric value.")
  }
  if (!is.numeric(target_max) || length(target_max) != 1) {
    return("Error: 'target_max' must be a single numeric value.")
  }

  if (target_min > target_max) {
    return("Error: 'target_min' cannot be greater than 'target_max'.")
  }

  if (length(data_vector) == 0) {
    return("Warning: 'data_vector' is empty. TIR is 0%.")
  }

  result <- tryCatch({

    clean_data <- data_vector[!is.na(data_vector)]

    if (length(clean_data) == 0) {
      return("Warning: All values in 'data_vector' were NA. TIR is 0%.")
    }

    in_range_count <- sum(clean_data >= target_min & clean_data <= target_max)

    total_count <- length(clean_data)

    tir_percentage <- (in_range_count / total_count) * 100

    return(tir_percentage)

  }, error = function(e) {
    return(paste("A critical calculation error occurred:", conditionMessage(e)))
  })

  return(result)
}

osiris_osteoporosis_score <- function(age, weight_kg, history_fracture, estrogen_therapy) {

  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)
  history_fracture <- as.numeric(history_fracture)
  estrogen_therapy <- as.numeric(estrogen_therapy)

  if (any(is.na(c(age, weight_kg, history_fracture, estrogen_therapy)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  score <- (weight_kg - age) * 0.2

  if (history_fracture == 1) {
    score <- score - 2
  }

  if (estrogen_therapy == 1) {
    score <- score + 2
  }

  score <- floor(score)

  risk_level <- ""
  recommendation <- ""

  if (score < 1) {
    risk_level <- "High Risk"
    recommendation <- "Bone Mineral Density (BMD) testing strongly recommended."
  } else if (score >= 1 && score <= 3) {
    risk_level <- "Medium Risk"
    recommendation <- "Consider BMD testing based on clinical judgment."
  } else {
    risk_level <- "Low Risk"
    recommendation <- "BMD testing generally not indicated unless other risk factors present."
  }

  return(list(
    OSIRIS_Score = score,
    Risk_Category = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Weight_kg = weight_kg,
      Hx_Fracture = history_fracture == 1,
      Estrogen_Therapy = estrogen_therapy == 1
    )
  ))
}

orai_osteoporosis_score <- function(age, weight_kg, no_estrogen_therapy) {

  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)
  no_estrogen_therapy <- as.numeric(no_estrogen_therapy)

  if (any(is.na(c(age, weight_kg, no_estrogen_therapy)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  score <- 0

  if (age >= 75) {
    score <- score + 15
  } else if (age >= 65 && age <= 74) {
    score <- score + 9
  } else if (age >= 55 && age <= 64) {
    score <- score + 5
  } else {
    score <- score + 0
  }

  if (weight_kg < 60) {
    score <- score + 9
  } else if (weight_kg >= 60 && weight_kg < 70) {
    score <- score + 3
  } else {
    score <- score + 0
  }

  if (no_estrogen_therapy == 1) {
    score <- score + 2
  }

  risk_level <- ""
  recommendation <- ""

  if (score >= 9) {
    risk_level <- "High Risk"
    recommendation <- "Bone Mineral Density (BMD) testing recommended."
  } else {
    risk_level <- "Low Risk"
    recommendation <- "BMD testing generally not indicated based on ORAI alone."
  }

  return(list(
    ORAI_Score = score,
    Risk_Category = risk_level,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Weight_kg = weight_kg,
      No_Estrogen_Use = no_estrogen_therapy == 1
    )
  ))
}

osteoporosis_self_assessment_tool <- function(age, weight_kg) {

  age <- as.numeric(age)
  weight_kg <- as.numeric(weight_kg)

  if (any(is.na(c(age, weight_kg)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (weight_kg <= 0) {
    stop("Error: Weight must be positive.")
  }

  score_raw <- (weight_kg - age) * 0.2
  score <- floor(score_raw)

  risk_category <- ""
  recommendation <- ""

  if (score < -3) {
    risk_category <- "High Risk"
    recommendation <- "Bone Mineral Density (BMD) testing strongly recommended."
  } else if (score >= -3 && score <= 1) {
    risk_category <- "Moderate Risk"
    recommendation <- "Measure BMD."
  } else {
    risk_category <- "Low Risk"
    recommendation <- "BMD testing likely not indicated unless other clinical risk factors present."
  }

  return(list(
    OST_Score = score,
    Risk_Category = risk_category,
    Recommendation = recommendation,
    Inputs = list(
      Age = age,
      Weight_kg = weight_kg
    )
  ))
}

calculate_tir_rosendaal <- function(data_vector, time_interval_minutes, target_min, target_max) {

  if (!is.numeric(data_vector) || any(is.infinite(data_vector))) {
    return("Error: 'data_vector' must be a finite numeric vector.")
  }
  if (!is.numeric(time_interval_minutes) || length(time_interval_minutes) != 1 || time_interval_minutes <= 0) {
    return("Error: 'time_interval_minutes' must be a single positive numeric value.")
  }
  if (!is.numeric(target_min) || !is.numeric(target_max) || length(target_min) != 1 || length(target_max) != 1) {
    return("Error: 'target_min' and 'target_max' must be single numeric values.")
  }
  if (target_min >= target_max) {
    return("Error: 'target_min' must be strictly less than 'target_max'.")
  }

  clean_data <- na.omit(data_vector)
  N <- length(clean_data)

  if (N < 2) {
    return("Warning: Need at least two non-NA data points for interpolation. TIR is 0%.")
  }

  total_time_observed <- (N - 1) * time_interval_minutes
  time_in_range <- 0

  L <- target_min
  H <- target_max

  for (i in 1:(N - 1)) {
    x1 <- clean_data[i]
    x2 <- clean_data[i + 1]
    T_interval <- time_interval_minutes

    min_x <- min(x1, x2)
    max_x <- max(x1, x2)

    if (min_x >= L && max_x <= H) {
      time_in_range <- time_in_range + T_interval
      next
    }

    if (max_x < L || min_x > H) {
      time_in_range <- time_in_range + 0
      next
    }

    total_change <- abs(x2 - x1)

    if (total_change == 0) {
      if (x1 >= L && x1 <= H) {
        time_in_range <- time_in_range + T_interval
      }
      next
    }

    t_out_low <- 0
    t_out_high <- 0

    if (min_x < L) {
      t_out_low <- T_interval * (L - min_x) / total_change
    }

    if (max_x > H) {
      t_out_high <- T_interval * (max_x - H) / total_change
    }

    t_in <- T_interval - t_out_low - t_out_high

    time_in_range <- time_in_range + max(0, t_in)
  }

  tir_percentage <- (time_in_range / total_time_observed) * 100

  return(tir_percentage)
}

ottawa_ankle_rules <- function(age, pain_malleolar_zone, pain_midfoot_zone, tenderness_distal_tibia_fibula, tenderness_medial_malleolus, tenderness_lateral_malleolus, tenderness_base_5th_metatarsal, tenderness_navicular, inability_to_bear_weight) {

  age <- as.numeric(age)
  pain_malleolar_zone <- as.numeric(pain_malleolar_zone)
  pain_midfoot_zone <- as.numeric(pain_midfoot_zone)
  tenderness_distal_tibia_fibula <- as.numeric(tenderness_distal_tibia_fibula)
  tenderness_medial_malleolus <- as.numeric(tenderness_medial_malleolus)
  tenderness_lateral_malleolus <- as.numeric(tenderness_lateral_malleolus)
  tenderness_base_5th_metatarsal <- as.numeric(tenderness_base_5th_metatarsal)
  tenderness_navicular <- as.numeric(tenderness_navicular)
  inability_to_bear_weight <- as.numeric(inability_to_bear_weight)

  if (any(is.na(c(age, pain_malleolar_zone, pain_midfoot_zone, tenderness_distal_tibia_fibula, tenderness_medial_malleolus, tenderness_lateral_malleolus, tenderness_base_5th_metatarsal, tenderness_navicular, inability_to_bear_weight)))) {
    stop("Error: All inputs must be numeric.")
  }

  ankle_xray_indicated <- FALSE
  foot_xray_indicated <- FALSE

  if (pain_malleolar_zone == 1) {
    if (tenderness_distal_tibia_fibula == 1 || tenderness_medial_malleolus == 1 || tenderness_lateral_malleolus == 1 || inability_to_bear_weight == 1) {
      ankle_xray_indicated <- TRUE
    }
  }

  if (pain_midfoot_zone == 1) {
    if (tenderness_base_5th_metatarsal == 1 || tenderness_navicular == 1 || inability_to_bear_weight == 1) {
      foot_xray_indicated <- TRUE
    }
  }

  recommendation <- ""
  if (ankle_xray_indicated && foot_xray_indicated) {
    recommendation <- "Ankle and Foot X-ray Series Indicated"
  } else if (ankle_xray_indicated) {
    recommendation <- "Ankle X-ray Series Indicated"
  } else if (foot_xray_indicated) {
    recommendation <- "Foot X-ray Series Indicated"
  } else {
    recommendation <- "X-rays Not Indicated (99% sensitivity for ruling out fracture)"
  }

  return(list(
    Result = recommendation,
    Imaging_Indicated = list(
      Ankle_Xray = ankle_xray_indicated,
      Foot_Xray = foot_xray_indicated
    ),
    Inputs = list(
      Age = age,
      Malleolar_Pain = pain_malleolar_zone == 1,
      Midfoot_Pain = pain_midfoot_zone == 1,
      Tenderness_Medial = tenderness_medial_malleolus == 1,
      Tenderness_Lateral = tenderness_lateral_malleolus == 1,
      Tenderness_5th_Metatarsal = tenderness_base_5th_metatarsal == 1,
      Tenderness_Navicular = tenderness_navicular == 1,
      Cant_Bear_Weight = inability_to_bear_weight == 1
    )
  ))
}

ottawa_copd_risk_scale <- function(history_copd, history_intubation, history_revascularization, sa02_on_arrival, ecg_ischemia, antianginal_medication, heart_rate_on_arrival, cxr_pulmonary_venous_congestion, hemoglobin_on_arrival, bun_on_arrival) {

  history_copd <- as.numeric(history_copd)
  history_intubation <- as.numeric(history_intubation)
  history_revascularization <- as.numeric(history_revascularization)
  sa02_on_arrival <- as.numeric(sa02_on_arrival)
  ecg_ischemia <- as.numeric(ecg_ischemia)
  antianginal_medication <- as.numeric(antianginal_medication)
  heart_rate_on_arrival <- as.numeric(heart_rate_on_arrival)
  cxr_pulmonary_venous_congestion <- as.numeric(cxr_pulmonary_venous_congestion)
  hemoglobin_on_arrival <- as.numeric(hemoglobin_on_arrival)
  bun_on_arrival <- as.numeric(bun_on_arrival)

  if (any(is.na(c(history_copd, history_intubation, history_revascularization, sa02_on_arrival, ecg_ischemia, antianginal_medication, heart_rate_on_arrival, cxr_pulmonary_venous_congestion, hemoglobin_on_arrival, bun_on_arrival)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (history_copd == 0) score <- score + 0
  else score <- score + 0

  if (history_intubation == 1) {
    score <- score + 2
  }

  if (history_revascularization == 1) {
    score <- score + 1
  }

  if (sa02_on_arrival < 92) {
    score <- score + 2
  }

  if (ecg_ischemia == 1) {
    score <- score + 2
  }

  if (antianginal_medication == 1) {
    score <- score + 1
  }

  if (heart_rate_on_arrival >= 110) {
    score <- score + 2
  }

  if (cxr_pulmonary_venous_congestion == 1) {
    score <- score + 2
  }

  if (hemoglobin_on_arrival < 100) { # g/L
    score <- score + 3
  }

  if (bun_on_arrival > 12) { # mmol/L
    score <- score + 1
  }

  risk_percent <- ""
  if (score <= 0) risk_percent <- "0.0%"
  else if (score == 1) risk_percent <- "1.8%"
  else if (score == 2) risk_percent <- "2.9%"
  else if (score == 3) risk_percent <- "4.3%"
  else if (score == 4) risk_percent <- "6.3%"
  else if (score == 5) risk_percent <- "9.4%"
  else if (score == 6) risk_percent <- "13.9%"
  else if (score == 7) risk_percent <- "21.9%"
  else if (score == 8) risk_percent <- "29.9%"
  else if (score == 9) risk_percent <- "39.5%"
  else risk_percent <- "50.0%"

  return(list(
    Ottawa_COPD_Risk_Score = score,
    Risk_Short_Term_Adverse_Event = risk_percent,
    Inputs = list(
      Hx_Intubation = history_intubation == 1,
      Hx_Revascularization = history_revascularization == 1,
      Low_SaO2 = sa02_on_arrival < 92,
      ECG_Ischemia = ecg_ischemia == 1,
      AntiAnginals = antianginal_medication == 1,
      Tachycardia = heart_rate_on_arrival >= 110,
      CXR_Congestion = cxr_pulmonary_venous_congestion == 1,
      Low_Hemoglobin = hemoglobin_on_arrival < 100,
      High_BUN = bun_on_arrival > 12
    )
  ))
}

ottawa_heart_failure_risk_scale <- function(stroke_tia_history, intubation_history, heart_rate_arrival, oxygen_sat_arrival, heart_rate_walk_test, ecg_ischemia, urea_mmol_l, co2_mmol_l, troponin_elevated, nt_probnp) {

  stroke_tia_history <- as.numeric(stroke_tia_history)
  intubation_history <- as.numeric(intubation_history)
  heart_rate_arrival <- as.numeric(heart_rate_arrival)
  oxygen_sat_arrival <- as.numeric(oxygen_sat_arrival)
  heart_rate_walk_test <- as.numeric(heart_rate_walk_test)
  ecg_ischemia <- as.numeric(ecg_ischemia)
  urea_mmol_l <- as.numeric(urea_mmol_l)
  co2_mmol_l <- as.numeric(co2_mmol_l)
  troponin_elevated <- as.numeric(troponin_elevated)
  nt_probnp <- as.numeric(nt_probnp)

  if (any(is.na(c(stroke_tia_history, intubation_history, heart_rate_arrival, oxygen_sat_arrival, heart_rate_walk_test, ecg_ischemia, urea_mmol_l, co2_mmol_l, troponin_elevated, nt_probnp)))) {
    stop("Error: All inputs must be numeric.")
  }

  score <- 0

  if (stroke_tia_history == 1) score <- score + 1

  if (intubation_history == 1) score <- score + 2

  if (heart_rate_arrival >= 110) score <- score + 2

  if (oxygen_sat_arrival < 90) score <- score + 1

  if (heart_rate_walk_test >= 110) score <- score + 1 # Or if unable to perform walk test

  if (ecg_ischemia == 1) score <- score + 2

  if (urea_mmol_l >= 12) score <- score + 1

  if (co2_mmol_l >= 35) score <- score + 2

  if (troponin_elevated == 1) score <- score + 2

  if (nt_probnp >= 5000) score <- score + 1

  risk_category <- ""
  sae_risk_14d <- ""
  recommendation <- ""

  if (score == 0) {
    risk_category <- "Low Risk"
    sae_risk_14d <- "2.8%"
    recommendation <- "Consider discharge with close follow-up."
  } else if (score <= 2) {
    risk_category <- "Moderate Risk"
    sae_risk_14d <- "5.1 - 9.2%"
    recommendation <- "Consider admission or close observation (Score > 1 often used as admission threshold)."
  } else if (score <= 4) {
    risk_category <- "High Risk"
    sae_risk_14d <- "15.9 - 26.1%"
    recommendation <- "Admission recommended."
  } else {
    risk_category <- "Very High Risk"
    sae_risk_14d <- "39.8 - 89.0%"
    recommendation <- "Admission to monitored setting recommended."
  }

  return(list(
    OHFRS_Score = score,
    Risk_Category = risk_category,
    Risk_Serious_Adverse_Event_14d = sae_risk_14d,
    Recommendation = recommendation,
    Inputs = list(
      Hx_Stroke_TIA = stroke_tia_history == 1,
      Hx_Intubation = intubation_history == 1,
      HR_Arrival_ge_110 = heart_rate_arrival >= 110,
      SaO2_Arrival_lt_90 = oxygen_sat_arrival < 90,
      HR_Walk_ge_110_or_Unable = heart_rate_walk_test >= 110,
      ECG_Ischemia = ecg_ischemia == 1,
      Urea_ge_12 = urea_mmol_l >= 12,
      CO2_ge_35 = co2_mmol_l >= 35,
      Trop_Elevated = troponin_elevated == 1,
      NT_proBNP_ge_5000 = nt_probnp >= 5000
    )
  ))
}

ottawa_knee_rule <- function(age, isolated_patella_tenderness, fibula_head_tenderness, inability_to_flex_90, inability_to_bear_weight) {

  age <- as.numeric(age)
  isolated_patella_tenderness <- as.numeric(isolated_patella_tenderness)
  fibula_head_tenderness <- as.numeric(fibula_head_tenderness)
  inability_to_flex_90 <- as.numeric(inability_to_flex_90)
  inability_to_bear_weight <- as.numeric(inability_to_bear_weight)

  if (any(is.na(c(age, isolated_patella_tenderness, fibula_head_tenderness, inability_to_flex_90, inability_to_bear_weight)))) {
    stop("Error: All inputs must be numeric.")
  }

  xray_indicated <- FALSE
  reasons <- c()

  if (age >= 55) {
    xray_indicated <- TRUE
    reasons <- c(reasons, "Age >= 55")
  }

  if (isolated_patella_tenderness == 1) {
    xray_indicated <- TRUE
    reasons <- c(reasons, "Isolated tenderness of the patella")
  }

  if (fibula_head_tenderness == 1) {
    xray_indicated <- TRUE
    reasons <- c(reasons, "Tenderness at the head of the fibula")
  }

  if (inability_to_flex_90 == 1) {
    xray_indicated <- TRUE
    reasons <- c(reasons, "Inability to flex knee to 90 degrees")
  }

  if (inability_to_bear_weight == 1) {
    xray_indicated <- TRUE
    reasons <- c(reasons, "Inability to bear weight (4 steps)")
  }

  recommendation <- ""
  sensitivity <- ""

  if (xray_indicated) {
    recommendation <- "Knee X-ray Series Indicated"
    sensitivity <- "98.5% Sensitivity for fracture"
  } else {
    recommendation <- "Knee X-ray Not Indicated"
    sensitivity <- "High sensitivity for ruling out fracture (98.5%)"
  }

  return(list(
    Result = recommendation,
    Reasoning = if (length(reasons) > 0) reasons else "No high-risk criteria met",
    Test_Characteristics = sensitivity,
    Inputs = list(
      Age = age,
      Patella_Tenderness = isolated_patella_tenderness == 1,
      Fibula_Tenderness = fibula_head_tenderness == 1,
      Flexion_Limit = inability_to_flex_90 == 1,
      Weight_Bearing_Issue = inability_to_bear_weight == 1
    )
  ))
}

ottawa_sah_rule <- function(age, neck_pain_stiffness, witnessed_loc, onset_exertion, thunderclap_headache, limited_neck_flexion) {

  age <- as.numeric(age)
  neck_pain_stiffness <- as.numeric(neck_pain_stiffness)
  witnessed_loc <- as.numeric(witnessed_loc)
  onset_exertion <- as.numeric(onset_exertion)
  thunderclap_headache <- as.numeric(thunderclap_headache)
  limited_neck_flexion <- as.numeric(limited_neck_flexion)

  if (any(is.na(c(age, neck_pain_stiffness, witnessed_loc, onset_exertion, thunderclap_headache, limited_neck_flexion)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (age < 15) {
    warning("Ottawa SAH Rule is not validated for patients under 15 years of age.")
  }

  high_risk_criteria <- FALSE
  reasons <- c()

  if (age >= 40) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Age >= 40")
  }

  if (neck_pain_stiffness == 1) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Neck pain or stiffness")
  }

  if (witnessed_loc == 1) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Witnessed loss of consciousness")
  }

  if (onset_exertion == 1) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Onset during exertion")
  }

  if (thunderclap_headache == 1) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Thunderclap headache (peaking instantly)")
  }

  if (limited_neck_flexion == 1) {
    high_risk_criteria <- TRUE
    reasons <- c(reasons, "Limited neck flexion on exam")
  }

  recommendation <- ""
  sensitivity <- ""

  if (high_risk_criteria) {
    recommendation <- "Investigate for Subarachnoid Hemorrhage (CT Head usually indicated)."
    sensitivity <- "100% Sensitivity (Rules IN need for workup, not diagnosis)"
  } else {
    recommendation <- "Rule Out SAH (High Sensitivity: 100%). Investigation likely not needed if all criteria absent."
    sensitivity <- "100% Sensitivity"
  }

  return(list(
    Result = recommendation,
    Criteria_Met = if (length(reasons) > 0) reasons else "None (All criteria negative)",
    Test_Sensitivity = sensitivity,
    Inputs = list(
      Age = age,
      Neck_Pain = neck_pain_stiffness == 1,
      Witnessed_LOC = witnessed_loc == 1,
      Exertion_Onset = onset_exertion == 1,
      Thunderclap = thunderclap_headache == 1,
      Limited_Neck_Flexion = limited_neck_flexion == 1
    )
  ))
}

convert_lab_units <- function(value, analyte, from_unit, to_unit) {

  conversion_table <- list(
    "Glucose" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.0555),
    "Creatinine" = list(us_unit = "mg/dL", si_unit = "umol/L", factor = 88.4),
    "BUN" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.357),
    "Cholesterol" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.0259),
    "Triglycerides" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.0113),
    "Calcium" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.25),
    "Magnesium" = list(us_unit = "mg/dL", si_unit = "mmol/L", factor = 0.4114),
    "Bilirubin" = list(us_unit = "mg/dL", si_unit = "umol/L", factor = 17.1),
    "Sodium" = list(us_unit = "mEq/L", si_unit = "mmol/L", factor = 1.0),
    "Potassium" = list(us_unit = "mEq/L", si_unit = "mmol/L", factor = 1.0),
    "Hemoglobin" = list(us_unit = "g/dL", si_unit = "g/L", factor = 10.0),
    "Albumin" = list(us_unit = "g/dL", si_unit = "g/L", factor = 10.0),
    "Total Protein" = list(us_unit = "g/dL", si_unit = "g/L", factor = 10.0),
    "Iron" = list(us_unit = "ug/dL", si_unit = "umol/L", factor = 0.1791),
    "Ferritin" = list(us_unit = "ng/mL", si_unit = "pmol/L", factor = 2.247)
  )

  if (!is.numeric(value) || length(value) != 1 || is.na(value) || is.infinite(value)) {
    return("Error: 'value' must be a single finite numeric value.")
  }

  analyte_norm <- tolower(trimws(analyte))
  from_unit_norm <- tolower(trimws(from_unit))
  to_unit_norm <- tolower(trimws(to_unit))

  if (!is.character(analyte) || length(analyte) != 1) {
    return("Error: 'analyte' must be a single character string.")
  }
  if (!is.character(from_unit) || length(from_unit) != 1) {
    return("Error: 'from_unit' must be a single character string.")
  }
  if (!is.character(to_unit) || length(to_unit) != 1) {
    return("Error: 'to_unit' must be a single character string.")
  }

  analyte_names <- tolower(names(conversion_table))
  match_index <- match(analyte_norm, analyte_names)

  if (is.na(match_index)) {
    return(paste0("Error: Analyte '", analyte, "' not supported. Supported: ",
                  paste(names(conversion_table), collapse = ", ")))
  }

  analyte_key <- names(conversion_table)[match_index]
  conversion <- conversion_table[[analyte_key]]

  valid_systems <- c("us", "si")
  if (!(from_unit_norm %in% valid_systems) || !(to_unit_norm %in% valid_systems)) {
    return("Error: Unit systems must be 'US' or 'SI'.")
  }

  if (from_unit_norm == to_unit_norm) {
    return(c(value, switch(from_unit_norm, "us" = conversion$us_unit, "si" = conversion$si_unit)))
  }

  factor <- conversion$factor

  if (from_unit_norm == "us" && to_unit_norm == "si") {
    converted_value <- value * factor
    target_unit <- conversion$si_unit

  } else if (from_unit_norm == "si" && to_unit_norm == "us") {
    converted_value <- value / factor
    target_unit <- conversion$us_unit

  } else {
    return("Error: Internal logic failure during unit conversion.")
  }

  return(c(converted_value, target_unit))
}

overall_neuropathy_limitations_scale <- function(arm_symptoms, wash_hair, turn_key, use_knife, fasten_buttons, leg_symptoms, walking_status) {

  arm_symptoms <- as.numeric(arm_symptoms)
  wash_hair <- as.numeric(wash_hair)
  turn_key <- as.numeric(turn_key)
  use_knife <- as.numeric(use_knife)
  fasten_buttons <- as.numeric(fasten_buttons)
  leg_symptoms <- as.numeric(leg_symptoms)
  walking_status <- tolower(as.character(walking_status))

  if (any(is.na(c(arm_symptoms, wash_hair, turn_key, use_knife, fasten_buttons, leg_symptoms)))) {
    stop("Error: All numeric/binary inputs must be valid.")
  }

  if (!walking_status %in% c("normal", "symptoms_only", "difficulty_running", "difficulty_walking", "unilateral_support", "bilateral_support", "wheelchair", "bedbound")) {
    stop("Error: Walking status must be one of: 'normal', 'symptoms_only', 'difficulty_running', 'difficulty_walking', 'unilateral_support', 'bilateral_support', 'wheelchair', 'bedbound'.")
  }

  arm_score <- 0

  if (arm_symptoms == 1) {
    impossible_count <- 0
    difficulty_count <- 0

    task_scores <- c(wash_hair, turn_key, use_knife, fasten_buttons)

    for (score in task_scores) {
      if (score == 2) impossible_count <- impossible_count + 1
      if (score == 1) difficulty_count <- difficulty_count + 1
    }

    if (impossible_count >= 3) {
      arm_score <- 5
    } else if (impossible_count == 2) {
      arm_score <- 4
    } else if (impossible_count == 1) {
      arm_score <- 3
    } else if (difficulty_count > 0) {
      arm_score <- 2
    } else {
      arm_score <- 1
    }
  }

  leg_score <- 0

  if (walking_status == "bedbound") {
    leg_score <- 7
  } else if (walking_status == "wheelchair") {
    leg_score <- 6
  } else if (walking_status == "bilateral_support") {
    leg_score <- 5
  } else if (walking_status == "unilateral_support") {
    leg_score <- 4
  } else if (walking_status == "difficulty_walking") {
    leg_score <- 3
  } else if (walking_status == "difficulty_running") {
    leg_score <- 2
  } else if (walking_status == "symptoms_only") {
    leg_score <- 1
  } else {
    if (leg_symptoms == 1) {
      leg_score <- 1
    } else {
      leg_score <- 0
    }
  }

  total_score <- arm_score + leg_score

  return(list(
    ONLS_Total_Score = total_score,
    Arm_Scale_Score = arm_score,
    Leg_Scale_Score = leg_score,
    Interpretation = "Higher scores indicate greater disability (Max 12).",
    Inputs = list(
      Arm_Symptoms = arm_symptoms == 1,
      Arm_Tasks = list(Hair=wash_hair, Key=turn_key, Knife=use_knife, Buttons=fasten_buttons),
      Leg_Status = walking_status
    )
  ))
}

oxygenation_index <- function(mean_airway_pressure, fio2_percent, pao2) {

  mean_airway_pressure <- as.numeric(mean_airway_pressure)
  fio2_percent <- as.numeric(fio2_percent)
  pao2 <- as.numeric(pao2)

  if (any(is.na(c(mean_airway_pressure, fio2_percent, pao2)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (pao2 <= 0) {
    stop("Error: PaO2 must be positive.")
  }

  if (fio2_percent <= 1) {
    warning("FiO2 appears to be a fraction (0-1). Converting to percent (0-100).")
    fio2_percent <- fio2_percent * 100
  }

  oi <- (fio2_percent * mean_airway_pressure) / pao2

  interpretation <- ""
  if (oi <= 25) {
    interpretation <- "Good Outcome Likely (if maintained). Continue conventional management."
  } else if (oi > 25 && oi < 40) {
    interpretation <- "High Risk of Mortality (>50%). Consider High Frequency Oscillatory Ventilation (HFOV) or Inhaled Nitric Oxide (iNO)."
  } else {
    interpretation <- "Very High Risk of Mortality (>80%). Consider Extracorporeal Membrane Oxygenation (ECMO)."
  }

  return(list(
    Oxygenation_Index = round(oi, 2),
    Interpretation = interpretation,
    Inputs = list(
      MAP = mean_airway_pressure,
      FiO2_Percent = fio2_percent,
      PaO2 = pao2
    )
  ))
}

neck_disability_index <- function(pain_intensity, personal_care, lifting, reading, headaches, concentration, work, driving, sleeping, recreation) {

  inputs <- c(pain_intensity, personal_care, lifting, reading, headaches, concentration, work, driving, sleeping, recreation)

  if (any(is.na(inputs))) {
    stop("Error: All 10 inputs must be numeric.")
  }

  if (any(inputs < 0) || any(inputs > 5)) {
    stop("Error: All item scores must be integers between 0 and 5.")
  }

  total_score <- sum(inputs)
  percentage <- total_score * 2

  disability_level <- ""

  if (total_score <= 4) {
    disability_level <- "No Disability"
  } else if (total_score <= 14) {
    disability_level <- "Mild Disability"
  } else if (total_score <= 24) {
    disability_level <- "Moderate Disability"
  } else if (total_score <= 34) {
    disability_level <- "Severe Disability"
  } else {
    disability_level <- "Complete Disability"
  }

  return(list(
    NDI_Score_Raw = total_score,
    NDI_Percentage = paste0(percentage, "%"),
    Interpretation = disability_level,
    Inputs = list(
      Q1_Pain = pain_intensity,
      Q2_Personal_Care = personal_care,
      Q3_Lifting = lifting,
      Q4_Reading = reading,
      Q5_Headaches = headaches,
      Q6_Concentration = concentration,
      Q7_Work = work,
      Q8_Driving = driving,
      Q9_Sleeping = sleeping,
      Q10_Recreation = recreation
    )
  ))
}

nedocs_score <- function(total_patients_ed, total_ed_beds, admits_in_ed, total_hospital_beds, ventilated_patients_ed, longest_admit_time_hrs, wait_time_last_patient_hrs) {

  total_patients_ed <- as.numeric(total_patients_ed)
  total_ed_beds <- as.numeric(total_ed_beds)
  admits_in_ed <- as.numeric(admits_in_ed)
  total_hospital_beds <- as.numeric(total_hospital_beds)
  ventilated_patients_ed <- as.numeric(ventilated_patients_ed)
  longest_admit_time_hrs <- as.numeric(longest_admit_time_hrs)
  wait_time_last_patient_hrs <- as.numeric(wait_time_last_patient_hrs)

  if (any(is.na(c(total_patients_ed, total_ed_beds, admits_in_ed, total_hospital_beds, ventilated_patients_ed, longest_admit_time_hrs, wait_time_last_patient_hrs)))) {
    stop("Error: All inputs must be numeric.")
  }

  if (total_ed_beds == 0 || total_hospital_beds == 0) {
    stop("Error: Bed counts cannot be zero.")
  }

  ratio_ed_beds <- total_patients_ed / total_ed_beds
  ratio_hospital_beds <- admits_in_ed / total_hospital_beds

  score <- -20 + (85.8 * ratio_ed_beds) + (600 * ratio_hospital_beds) + (13.4 * ventilated_patients_ed) + (0.93 * longest_admit_time_hrs) + (5.64 * wait_time_last_patient_hrs)

  if (score > 200) score <- 200
  if (score < 0) score <- 0

  status <- ""
  if (score <= 20) {
    status <- "Not Busy"
  } else if (score <= 60) {
    status <- "Busy"
  } else if (score <= 100) {
    status <- "Overcrowded"
  } else if (score <= 140) {
    status <- "Severely Overcrowded"
  } else if (score <= 180) {
    status <- "Dangerous Overcrowding"
  } else {
    status <- "Disaster"
  }

  return(list(
    NEDOCS_Score = round(score, 0),
    Status = status,
    Inputs = list(
      Total_Patients = total_patients_ed,
      Total_ED_Beds = total_ed_beds,
      Admits_Waiting = admits_in_ed,
      Total_Hospital_Beds = total_hospital_beds,
      Vents = ventilated_patients_ed,
      Longest_Admit_Wait = longest_admit_time_hrs,
      Last_Patient_Wait = wait_time_last_patient_hrs
    )
  ))
}

neonatal_eos_calculator <- function(gestational_age_weeks, highest_maternal_temp_c, rom_hours, gbs_status, intrapartum_antibiotics, neonatal_appearance, baseline_incidence_per_1000 = 0.5) {

  gestational_age_weeks <- as.numeric(gestational_age_weeks)
  highest_maternal_temp_c <- as.numeric(highest_maternal_temp_c)
  rom_hours <- as.numeric(rom_hours)
  baseline_incidence_per_1000 <- as.numeric(baseline_incidence_per_1000)

  gbs_status <- tolower(as.character(gbs_status))
  intrapartum_antibiotics <- tolower(as.character(intrapartum_antibiotics))
  neonatal_appearance <- tolower(as.character(neonatal_appearance))

  if (any(is.na(c(gestational_age_weeks, highest_maternal_temp_c, rom_hours, baseline_incidence_per_1000)))) {
    stop("Error: Numeric inputs are required for age, temperature, ROM durati